unit PegasusProtocol;

{$undef USE_CODESITE}
{$define RANGE_CHECK}

interface

uses
  SerialPort,
  classes, messages, sysutils, windows;

const
  // messages                        // WParam           LParam
  PM_DSPSTART   = WM_USER + 1;       //   n/a              n/a
  PM_RADIOSTART = WM_USER + 2;       //   n/a              n/a
  PM_ENCODER    = WM_USER + 3;       // movement         key pressed
  PM_FORWARD    = WM_USER + 4;       // forward power      n/a
  PM_REFLECTED  = WM_USER + 5;       // reflected power    n/a
  PM_RXMETER    = WM_USER + 6;       // s-meter            n/a
  PM_TXMETER    = WM_USER + 7;       // forward power    reflected power
  PM_PODKEY     = WM_USER + 8;       // key pressed        n/a
  PM_VERSION    = WM_USER + 9;       // version * 1000     n/a
  PM_DIGLEVEL   = WM_USER + 10;      // DSP level          n/a
  PM_ANALEVEL   = WM_USER + 11;      // AGC level          n/a
  PM_BADSYNTAX  = WM_USER + 12;      //   n/a              n/a


const
  CMD_BUFFER_LEN = 256;

type
  // Exceptions
  EPegasusProtocol = class( Exception )
  public
    constructor createMsg;
  end;

  EBadMessage = class( EPegasusProtocol )
  public
    constructor createMsg;
  end;

  EParmRange = class( EPegasusProtocol )
  public
    constructor createMsg;
  end;


type
  Str2 = string[2];
  Str7 = string[7];

  TPegasusProtocol = class;
  TPegPNotifyEvent = procedure( const sender : TPegasusProtocol ) of object;

  TPegPSetByte = procedure( const aValue : byte ) of object;

  TPegasusProtocol = class( TObject )
  private
    fComPort   : TmeComPort;
    fCmdLen    : integer;
    fCmdStr    : array [0 .. CMD_BUFFER_LEN - 1] of char;
    fPostTo    : HWND;

    procedure ReadRadioString( aSender : TObject );
    procedure WriteRadioString( const aString : str7 );
    procedure ProcessRadioString;
    function GetPortNum : byte;

  // TPegasusProtocol
  public
    constructor create( const aPostTo : HWND );
    destructor destroy; override;

    procedure OpenSerialPort( const aWhich : byte );
    procedure CloseSerialPort;

    // commands
    procedure RestartAndNotify;
    procedure DSPProgramExecute;

    procedure AGCMode( const aWhich : char );
    procedure AMCarrier( const aLevel : byte );
    procedure AudioSourceAndGain( const aAux : Boolean; const aLevel : byte );
    procedure CWQSK( const aLevel : byte );
    procedure CWSidetoneVolume( const aLevel : byte );
    procedure CWSpotLevel( const aLevel : byte );
    procedure KeepAlive( const aEnable : Boolean );
    procedure KeyDown( const aEnable : Boolean );
    procedure Keyer( const aEnable : Boolean );
    procedure KeyerTiming( const aDit, aDah, aSpace : word );
    procedure LineLevelAudioOutput( const aLevel : byte );
    procedure NoiseBlankerLevel( const aLevel : byte );
    procedure NoiseReductionAutoNotch( const aNoiseReduction, aAutoNotch : Boolean );
    procedure OutputPower( const aLevel : byte );
    procedure PT11Tuner( const aWhat : char );
    procedure QueryAnalogAGCLevel;
    procedure QueryDSPSignalLevel;
    procedure QueryForwardPowerReading;
    procedure QueryRAWAuxStatus;
    procedure QueryReflectedPowerReading;
    procedure QuerySignalLevel;
    procedure QueryVersion;
    procedure RFAttenuator( const aEnable : Boolean );
    procedure RFGain( const aLevel : byte );
    procedure RxFilter( const aWhich : byte );
    procedure RxTuningFactors( const aCoarse, aFine, aBFO : word );
    procedure RxTxModes( const aRxMode, aTxMode : char );
    procedure SpeakerOutput( const aLevel : byte );
    procedure SpeechProcessor( const aLevel : byte );
    procedure Squelch( const aLevel : byte );
    procedure TLoop( const aEnable : Boolean );
    procedure Transmitter( const aEnable : Boolean );
    procedure TxAudioMonitorVolume( const aLevel : byte );
    procedure TxFilter( const aWhich : byte );
    procedure TXHang( const aLevel : byte );
    procedure TxTuningFactors( const aCoarse, aFine, aBFO : word );
    procedure VOX( const aEnable : Boolean );
    procedure VOXAnti( const aLevel : byte );
    procedure VOXGain( const aLevel : byte );
    procedure VOXHang( const aLevel : byte );
    procedure WideBandTx( const aEnable : Boolean );

    procedure SetDTR( const aOn : Boolean );

    property PortNum : byte read GetPortNum;
  end;


implementation

{$ifdef USE_CODESITE}
uses
  rzCSIntf;
{$endif}

{ --------------------------------------------------------------------------- }
{ ----- constructor / destructor -------------------------------------------- }
{ --------------------------------------------------------------------------- }

const
  PEG_BAUD   = 57600;
  PEG_PARITY = parNone;
  PEG_BITS   = 8;
  PEG_STOP   = sbOne;

  WRITE_TO_MULT  = 3;
  WRITE_TO_CONST = 1000;

  CTS_FLOW = true;
  RTS_CTRL = ctrlHandshake;
  EVT_CHAR = #13;

  ZeroOne : array[Boolean] of char = ( '0', '1' );


constructor TPegasusProtocol.create( const aPostTo : HWND );
begin
  fPostTo := aPostTo;
  fComPort := TmeComPort.create( nil );
  with fComPort do
  begin
    BaudRate := PEG_BAUD;
    Parity   := PEG_PARITY;
    ByteSize := PEG_BITS;
    StopBits := PEG_STOP;
    WriteTotalTimeoutMultiplier := WRITE_TO_MULT;
    WriteTotalTimeoutConstant   := WRITE_TO_CONST;
    OutxCTSFlow := CTS_FLOW;
    RTSControl  := RTS_CTRL;
    EvtChar     := EVT_CHAR;
    OnRxFlag    := ReadRadioString;
  end;
end;

destructor TPegasusProtocol.destroy;
begin
  fComPort.Free;
  inherited;
end;


procedure TPegasusProtocol.OpenSerialPort( const aWhich : byte );
begin
  with fComPort do
  begin
    Open( aWhich );
    PurgeIn;
    PurgeOut;
  end;
end;

procedure TPegasusProtocol.CloseSerialPort;
begin
  fComPort.Close;
end;

{ --------------------------------------------------------------------------- }
{ ----- com port event handlers --------------------------------------------- }
{ --------------------------------------------------------------------------- }


// handler for serial port event char read

procedure TPegasusProtocol.ReadRadioString( aSender : TObject );
var
  BytesRead : integer;
begin
  BytesRead := fComPort.ReadBuffer(fCmdStr[fCmdLen], fComPort.InQue);
  if BytesRead > 0 then
  begin
    inc( fCmdLen, BytesRead );
    ProcessRadioString;
  end;
end;

procedure PostMessage1( const h : HWND; i, j, k : integer );
begin
  Win32Check( PostMessage( h, i, j, k ));
end;

procedure TPegasusProtocol.ProcessRadioString;
var
  i, err : integer;

  procedure AdjustLen( const aAmount : integer );
  begin
    dec( fCmdLen, aAmount);
    if fCmdLen > 1
      then move( fCmdStr[aAmount], fCmdStr[0], fCmdLen )
      else fCmdLen := 0;
  end;

begin
  while fCmdLen > 0 do
  begin
    case fCmdStr[0] of
      ' ' : begin
              if fCmdLen < 5 then break;
              if fCmdStr[4] = 'D' then
              begin
                if fCmdLen < 14 then break;
                AdjustLen(14);
                PostMessage1( fPostTo, PM_DSPSTART, 0, 0 );
              end
              else if fCmdStr[3] = 'R' then
              begin
                if fCmdLen < 15 then break;
                AdjustLen(15);
                PostMessage1( fPostTo, PM_RADIOSTART, 0, 0 );
              end
              else
                AdjustLen(1);
            end;

      '!' : begin
              if fCmdLen < 5 then break;
              PostMessage1( fPostTo, PM_ENCODER,
                ord( fCmdStr[1] ) shl 8 or ord( fCmdStr[2] ), ord( fCmdStr[3] ));
              AdjustLen(5);
            end;

      'F' : begin
              if fCmdLen < 3 then break;
              PostMessage1( fPostTo, PM_FORWARD, ord( fCmdStr[1] ), 0 );
              AdjustLen( 3 );
            end;

      'R' : begin
              if fCmdLen < 3 then break;
              PostMessage1( fPostTo, PM_REFLECTED, ord( fCmdStr[1] ), 0 );
              AdjustLen( 3 );
            end;

      'S' : begin
              if fCmdLen < 6 then break;
              val( '$' + fCmdStr[1] + fCmdStr[2] + fCmdStr[3] + fCmdStr[4], i, err );
              AdjustLen(6);
              if err = 0 then
              begin
                if i >= $8000 then i := 0;
                PostMessage1( fPostTo, PM_RXMETER, i, 0 );
              end;
            end;

      'T' : begin
              if fCmdLen < 4 then break;
              PostMessage1( fPostTo, PM_TXMETER, ord( fCmdStr[1] ), ord( fCmdStr[2] ));
              AdjustLen( 4 );
            end;

      'U' : begin
              if fCmdLen < 3 then break;
              PostMessage1( fPostTo, PM_PODKEY, ord( fCmdStr[1] ), 0 );
              AdjustLen( 3 );
            end;

      'V' : begin
              if fCmdLen < 9 then break;
              val( copy( fCmdStr, 5, 4 ), i, err );
              AdjustLen ( 9 );
              if err = 0 then
                PostMessage1( fPostTo, PM_VERSION, i, 0 );
            end;

      'X' : begin
              if fCmdLen < 4 then break;
              PostMessage1( fPostTo, PM_DIGLEVEL, ord( fCmdStr[1] ) shl 8 or ord( fCmdStr[2] ), 0 );
              AdjustLen( 4 );
            end;

      'Y' : begin
              if fCmdLen < 3 then break;
              PostMessage1( fPostTo, PM_ANALEVEL, ord( fCmdStr[1] ), 0 );
              AdjustLen( 3 );
            end;

      'Z' : begin
              if fCmdLen < 2 then break;
              PostMessage1( fPostTo, PM_BADSYNTAX, 0, 0 );
              AdjustLen( 2 );
            end;
    else
      AdjustLen( 1 );
    end;
  end;
end;


{ --------------------------------------------------------------------------- }
{ ----- utility routines ---------------------------------------------------- }
{ --------------------------------------------------------------------------- }

function IntToStr2( const aWhich : integer ) : str2;
begin
  result := chr( aWhich shr 8 ) + chr( aWhich and 255 );
end;

procedure TPegasusProtocol.WriteRadioString( const aString: str7 );
begin
  if fComPort.Connected then
    fComPort.WriteString( aString + #13 );
end;


{ --------------------------------------------------------------------------- }
{ ----- radio commands ------------------------------------------------------ }
{ --------------------------------------------------------------------------- }

procedure TPegasusProtocol.RestartAndNotify;
begin
  WriteRadioString( 'XX' );
end;

procedure TPegasusProtocol.DSPProgramExecute;
begin
  WriteRadioString( 'P1' );
end;

procedure TPegasusProtocol.RxTuningFactors( const aCoarse, aFine, aBFO : word );
begin
  WriteRadioString( 'N' + IntToStr2( aCoarse ) + IntToStr2( aFine )
    + IntToStr2( aBFO ));
end;

procedure TPegasusProtocol.TxTuningFactors( const aCoarse, aFine, aBFO : word );
begin
  WriteRadioString( 'T' + IntToStr2( aCoarse ) + IntToStr2( aFine )
    + IntToStr2( aBFO ));
end;

procedure TPegasusProtocol.RxTxModes( const aRxMode, aTxMode : char );
begin
  {$ifdef RANGE_CHECK}
  if not (( aRXMode in ['0' .. '4'] ) and ( aTXMode in ['0' .. '4'] )) then
    raise EParmRange.createMsg;
  {$endif}
  WriteRadioString( 'M' + aRxMode + aTxMode );
end;

procedure TPegasusProtocol.RxFilter( const aWhich : byte );
begin
  {$ifdef RANGE_CHECK}
  if aWhich > 33 then
    raise EParmRange.createMsg;
  {$endif}
  WriteRadioString( 'W' + chr( aWhich ));
end;

procedure TPegasusProtocol.TxFilter( const aWhich : byte );
begin
  {$ifdef RANGE_CHECK}
  if not aWhich in [7 .. 24] then
    raise EParmRange.createMsg;
  {$endif}
  WriteRadioString( 'C' + chr( aWhich ));
end;

procedure TPegasusProtocol.LineLevelAudioOutput( const aLevel : byte );
begin
  {$ifdef RANGE_CHECK}
  if aLevel > 63 then
    raise EParmRange.createMsg;
  {$endif}
  WriteRadioString( 'L' + chr( aLevel ));
end;

procedure TPegasusProtocol.SpeakerOutput( const aLevel : byte );
begin
  WriteRadioString( 'V' + chr( aLevel ));
end;

procedure TPegasusProtocol.AGCMode( const aWhich : char );
begin
  {$ifdef RANGE_CHECK}
  if not ( aWhich in ['1', '2', '3'] ) then
    raise EparmRange.createMsg;
  {$endif}
  WriteRadioString( 'G' + aWhich );
end;

procedure TPegasusProtocol.RFGain( const aLevel : byte );
begin
  WriteRadioString( 'A'+ chr( aLevel ));
end;

procedure TPegasusProtocol.RFAttenuator( const aEnable : Boolean );
begin
  WriteRadioString( 'B' + ZeroOne[aEnable] );
end;

procedure TPegasusProtocol.KeyerTiming( const aDit, aDah, aSpace : word );
begin
  WriteRadioString( 'E' + IntToStr2( aDit ) + IntToStr2( aDah )
    + IntToStr2( aSpace ));
end;

procedure TPegasusProtocol.TxAudioMonitorVolume( const aLevel : byte );
begin
  {$ifdef RANGE_CHECK}
  if aLevel > 63 then
    raise EParmRange.createMsg;
  {$endif}
  WriteRadioString( 'H' + chr( aLevel ));
end;

procedure TPegasusProtocol.CWSidetoneVolume( const aLevel : byte );
begin
  WriteRadioString( 'J' + chr( aLevel ));
end;

procedure TPegasusProtocol.NoiseReductionAutoNotch( const aNoiseReduction, aAutoNotch : Boolean );
begin
  WriteRadioString( 'K' + ZeroOne[aNoiseReduction] + ZeroOne[aAutoNotch] );
end;

procedure TPegasusProtocol.AudioSourceAndGain( const aAux : Boolean; const aLevel : byte );
begin
  {$ifdef RANGE_CHECK}
  if aLevel > 15 then
    raise EParmRange.createMsg;
  {$endif}
  WriteRadioString( 'O1' + ZeroOne[aAux] + chr( aLevel ));
end;

procedure TPegasusProtocol.OutputPower( const aLevel : byte );
begin
  WriteRadioString( 'P' + chr( aLevel ));
end;

procedure TPegasusProtocol.KeyDown( const aEnable : Boolean );
begin
  WriteRadioString( 'Q' + ZeroOne[aEnable] );
end;

procedure TPegasusProtocol.Squelch( const aLevel : byte );
begin
  WriteRadioString( 'S' + chr( aLevel ));
end;

procedure TPegasusProtocol.VOX( const aEnable : Boolean );
begin
  WriteRadioString( 'U' + ZeroOne[aEnable] );
end;

procedure TPegasusProtocol.VOXGain( const aLevel : byte );
begin
  WriteRadioString( 'UG' + chr( aLevel ));
end;

procedure TPegasusProtocol.VOXAnti( const aLevel : byte );
begin
  WriteRadioString( 'UA' + chr( aLevel ));
end;

procedure TPegasusProtocol.VOXHang( const aLevel : byte );
begin
  WriteRadioString( 'UH' + chr( aLevel ));
end;

procedure TPegasusProtocol.CWSpotLevel( const aLevel : byte );
begin
  WriteRadioString( 'F' + chr( aLevel ));
end;

procedure TPegasusProtocol.Transmitter( const aEnable : Boolean );
const
  EnableTx : array[Boolean] of char = ( '0', '1' );
begin
  WriteRadioString( '#' + EnableTx[aEnable] );
end;

procedure TPegasusProtocol.TLoop;
const
  EnableTLoop : array[Boolean] of char = ( '2', '3' );
begin
  WriteRadioString( '#' + EnableTLoop[aEnable] );
end;

procedure TPegasusProtocol.WidebandTx( const aEnable : Boolean );
const
  EnableWidebandTx : array[Boolean] of char = ( '4', '5' );
begin
  WriteRadioString( '#' + EnableWidebandTx[aEnable] );
end;

procedure TPegasusProtocol.Keyer( const aEnable : Boolean );
const
  EnableKeyer : array[Boolean] of char = ( '6', '7' );
begin
  WriteRadioString( '#' + EnableKeyer[aEnable] );
end;

procedure TPegasusProtocol.KeepAlive( const aEnable : Boolean );
const
  EnableKeepAlive : array[Boolean] of char = ( '8', '9' );
begin
  WriteRadioString( '#' + EnableKeepAlive[aEnable] );
end;

procedure TPegasusProtocol.QueryVersion;
begin
  WriteRadioString( '?V' );
end;

procedure TPegasusProtocol.QueryDSPSignalLevel;
begin
  WriteRadioString( '?X' );
end;

procedure TPegasusProtocol.QueryAnalogAGCLevel;
begin
  WriteRadioString( '?Y' );
end;

procedure TPegasusProtocol.QueryRAWAuxStatus;
begin
  WriteRadioString( '?!' );
end;

procedure TPegasusProtocol.QueryForwardPowerReading;
begin
  WriteRadioString( '?F' );
end;

procedure TPegasusProtocol.QueryReflectedPowerReading;
begin
  WriteRadioString( '?R' );
end;

procedure TPegasusProtocol.QuerySignalLevel;
begin
  WriteRadioString( '?S' );
end;

procedure TPegasusProtocol.NoiseBlankerLevel( const aLevel : byte );
begin
  {$ifdef RANGE_CHECK}
  if aLevel > 7 then
    raise EParmRange.createMsg;
  {$endif}
  WriteRadioString( 'D' + chr( aLevel ));
end;

procedure TPegasusProtocol.SpeechProcessor( const aLevel : byte );
begin
  {$ifdef RANGE_CHECK}
  if aLevel > 127 then
    raise EParmRange.createMsg;
  {$endif}
  WriteRadioString( 'Y' + chr( aLevel ));
end;

procedure TPegasusProtocol.TXHang( const aLevel : byte );
begin
  WriteRadioString( 'UT' + chr( aLevel ));
end;

procedure TPegasusProtocol.CWQSK( const aLevel : byte );
begin
  WriteRadioString( 'UQ' + chr( aLevel ));
end;

procedure TPegasusProtocol.AMCarrier( const aLevel : byte );
begin
  WriteRadioString( 'R' + chr( aLevel ));
end;

procedure TPegasusProtocol.PT11Tuner( const aWhat : char );
begin
  WriteRadioString( '$' + aWhat );
end;

function TPegasusProtocol.GetPortNum : byte;
begin
  result := fComPort.PortNum;
end;

procedure TPegasusProtocol.SetDTR( const aOn : Boolean );
begin
  fComPort.SetDTR( aOn );
end;

// -----------------------------------------------------------------------------
// ----- Exceptions ------------------------------------------------------------
// -----------------------------------------------------------------------------

const
  HLP_EPEGASUSPROTOCOL = 0;
  HLP_EBADMESSAGE = 0;
  HLP_EPARMRANGE = 0;

resourcestring
  SEPegasusProtocol = 'unknown error';
  SEBadMessage = 'unknown message from radio';
  SEParmRange = 'parameter out of range';

constructor EPegasusProtocol.createMsg;
begin
  createResHelp( @SEPegasusProtocol, HLP_EPEGASUSPROTOCOL )
end;

constructor EBadMessage.createMsg;
begin
  createResHelp( @SEBadMessage, HLP_EBADMESSAGE )
end;

constructor EParmRange.createMsg;
begin
  createResHelp( @SEParmRange, HLP_EPARMRANGE )
end;

end.
