/*
 * locatedir.c
 *
 * This file is part of "locatedir"
 * by Alexander Kourakos <awk@vt.edu>
 *
 * Copyright (C) 1994 Alexander Kourakos
 * All rights reserved.
 *
 * This software may be freely copied, modified, and redistributed
 * provided that this copyright notice remains unchanged on all copies.
 *
 * You may not distribute this software, in whole or in part, as part of
 * any commercial product without the express consent of the author.
 *
 * There is no warranty or other guarantee of this software fitness
 * for any purpose. It is provided solely "as is".
 */

#define GLOBALS
#include "locatedir.h"
#include <unistd.h>
#include <getopt.h>
#include <sys/stat.h>

/*
 * Gets the user's home directory from the environment.
 */

static char *
gethome (void)
{
  char   *homep;

  if ((homep = getenv ("HOME")) == (char *) NULL)
    error ("can't find your home directory");

  return homep;
}

/*
 * Finds the basename of a path, like the shell command of the same
 * name.
 */

static void
basename (char *result, const char *path)
{
  char   *basep;

  if ((basep = strrchr (path, (int) DIR_SEP_CHAR)) == (char *) NULL)
    strcpy (result, path);
  else
    strcpy (result, ++basep);
}

/*
 * Main.
 */

int
main (int ac, char *av[])
{
  char    listfilepath[MAX_PATH_LENGTH];
  char    find_this[MAX_NAME_LENGTH];
  char    nwdpath[MAX_PATH_LENGTH];
  int     result;
  int     building;
  FILE   *listfile;

  basename (progname, av[0]);

  /*
   * Get the home and current directory, place them in global variables.
   */

  strcpy (homepath, gethome ());

  if (getcwd (cwdpath, MAX_PATH_LENGTH) == (char *) NULL)
    error ("unable to get current working directory");

  building = 0;

  quiet = 0;
  rebuild_if_old = 1;
  echo_dots = 1;

  if (!strcmp (av[1], "-r"))
  {
    building = 1;

    if (ac == 2)
      strcpy (startpath, homepath);
    else if (ac == 3)
    {
      if (av[2][0] != DIR_SEP_CHAR)
      {
	strcpy (startpath, cwdpath);
	strcat (startpath, DIR_SEP);
      }
      else
      {
	startpath[0] = '\0';
      }
      strcat (startpath, av[2]);
    }
    else
      usage ();
  }
  else
  {
    char    opt;

    while ((opt = getopt (ac, av, "qnd")) != -1)
    {
      switch (opt)
      {
      case 'q':
	quiet = !quiet;
	break;
      case 'n':
	rebuild_if_old = !rebuild_if_old;
	break;
      case 'd':
	echo_dots = !echo_dots;
	break;
      default:
	usage ();
      }
    }

    ac -= optind;

    if (ac == 0)
      find_this[0] = '\0';
    else if (ac == 1)
      strcpy (find_this, av[optind]);
    else
      usage ();
  }

  if (!quiet)
    fprintf (stderr, "locatedir v%s by Alexander Kourakos <awk@vt.edu>\n\n", VERSION);

  /*
   * Build the pathname of the ".dirlist" file.
   */

  strcpy (listfilepath, homepath);
  strcat (listfilepath, DIR_SEP);
  strcat (listfilepath, DIRLIST);

  if (building)
  {
    struct stat statbuf;

    if (stat (startpath, &statbuf))
    {
      fprintf (stderr, "unable to stat \"%s\", using home directory\n", startpath);
      strcpy (startpath, homepath);
    }
    else if (!S_ISDIR (statbuf.st_mode))
    {
      fprintf (stderr, "\"%s\" is not a directory, using home directory\n", startpath);
      strcpy (startpath, homepath);
    }

    build_dirlistfile (listfilepath);
    printf ("%s\n", HERE);
    goto exit;
  }

  /*
   * Did they type "." or ".." and want them echoed?
   */

  if (!(strcmp (find_this, ".") && strcmp (find_this, "..")) && echo_dots)
  {
    if (!quiet)
      fprintf (stderr, "%s\n", find_this);
    printf (find_this);
    return EXIT_SUCCESS;
  }

  /*
   * Perform the search, placing result in "nwdpath".
   */

  listfile = open_dirlistfile (listfilepath);

  if (*find_this == '\0')
  {
    strcpy (nwdpath, startpath);
    result = 0;
  }
  else
    result = find_best_match (listfile, find_this, nwdpath);

  if (!quiet)
  {
    if (result)
      fprintf (stderr, "(distinct match not found)\n\n");
    else
      fprintf (stderr, "matched: %s\n\n", nwdpath);
  }

  if (result)
    printf (cwdpath);
  else
    printf (nwdpath);

  if (fclose (listfile) == EOF)
    error ("unable to close directory list file");

exit:
  return EXIT_SUCCESS;
}

/*
 * end of file 
 */
