/*
 * slip.c - Slip specific code in diald.
 *
 * Copyright (c) 1994 Eric Schenk.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL ERIC SCHENK BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ERIC
 * SCHENK HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * ERIC SCHENK SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND ERIC SCHENK HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include "diald.h"

static int dead = 1;
static int waiting_for_bootp = 0;
static int have_local, have_remote;
static int start_disc;
static FILE *bootpfp;
static PIPE bootp_pipe;


static void start_bootp()
{
    char buf[128];

    idle_filter_init();

    sprintf(buf,"%s --dev sl%d",PATH_BOOTPC,link_iface);
    if ((bootpfp = popen(buf,"r"))==NULL) {
	syslog(LOG_ERR,"Could not run command '%s': %m",buf);
	die(1);
    }
    pipe_init(fileno(bootpfp),&bootp_pipe);
    have_local = 0;
    have_remote = 0;
    waiting_for_bootp = 1;
}

static int grab_addr(char **var)
{
    int len, i = 0;
    int state = 0;
    unsigned char buffer[128];
    unsigned char c;

    while (1) {
	if ((len = read(modem_fd,&c,1)) < 0) {
	    if (errno != EINTR || terminate) {
	        syslog(LOG_ERR,"Error reading from modem: %m");
	        return 0;
            }
        }
	if (len == 1) {
	    switch (state) {
	    case 0:	/* wait for a number to come up */
		if (isdigit(c)) buffer[i++] = c, state = 1;
		break;
	    case 1:	/* wait for at least one '.' */
		if (isdigit(c)) buffer[i++] = c;
		else if (c == '.') buffer[i++] = c, state = 2;
		else i = 0, state = 0;
		break;
	    case 2:	/* we saw at least one '.', assume its an ip address */
		if (isdigit(c) || c == '.') buffer[i++] = c;
		else {
		    if (buffer[i-1] == '.') i--;   /* trim off trailing "." */
		    buffer[i++] = 0;
		    goto done;
		}
		break;
	    }
	    if (i >= 128)
		syslog(LOG_ERR,"Buffer overflow when reading IP address"),
		die(1);
	}
    }
done:
    *var = strdup(buffer);
    return 1;
}

/*
 * The SLIP configuration is essentially what slattach
 * does, but we do it here so we know what interface (sl*)
 * gets opened as a result. (slattach doesn't return this)
 */

void slip_start(void)
{
    char buf[128];
    int disc, sencap;

    if (dynamic_addrs) {
	if (debug&DEBUG_VERBOSE)
	    syslog(LOG_INFO,"Fetching IP addresses from SLIP server");
	if (dynamic_mode != DMODE_BOOTP) {
	    if (dynamic_mode == DMODE_REMOTE || dynamic_mode == DMODE_REMOTE_LOCAL)
		if (!grab_addr(&remote_ip)) return;
	    if (dynamic_mode != DMODE_REMOTE)
		if (!grab_addr(&local_ip)) return;
	    if (dynamic_mode == DMODE_LOCAL_REMOTE)
		if (!grab_addr(&remote_ip)) return;
	    syslog(LOG_INFO,"New addresses: local %s, remote %s.",
		local_ip,remote_ip);
	}
    }

    if (ioctl(modem_fd, TIOCGETD, &start_disc) < 0)
	syslog(LOG_ERR,"Can't get line discipline on proxy device: %m"), die(1);

    /* change line disciple to SLIP and set the SLIP encapsulation */
    disc = N_SLIP;
    if ((link_iface = ioctl(modem_fd, TIOCSETD, &disc)) < 0) {
	if (errno == ENFILE) {
	   syslog(LOG_ERR,"No free slip device available for link."), die(1);
	} else if (errno == EEXIST) {
	    syslog(LOG_ERR,"Link device already in slip mode!?");
	} else if (errno == EINVAL) {
	    syslog(LOG_ERR,"SLIP not supported by kernel, can't build link.");
	    die(1);
	} else
	   syslog(LOG_ERR,"Can't set line discipline: %m"), die(1);
    }

    if (ioctl(modem_fd, SIOCSIFENCAP, &slip_encap) < 0)
	syslog(LOG_ERR,"Can't set encapsulation: %m"), die(1);

    /* verify that it worked */
    if (ioctl(modem_fd, TIOCGETD, &disc) < 0)
	syslog(LOG_ERR,"Can't get line discipline: %m"), die(1);
    if (ioctl(modem_fd, SIOCGIFENCAP, &sencap) < 0)
	syslog(LOG_ERR,"Can't get encapsulation: %m"), die(1);

    if (disc != N_SLIP || sencap != slip_encap)
        syslog(LOG_ERR,"Couldn't set up the slip link correctly!"), die(1);

    if (debug&DEBUG_VERBOSE)
        syslog(LOG_INFO,"Slip link established on interface sl%d",
	    link_iface);

    /* mark the interface as up */
    if (netmask) {
        sprintf(buf,"%s sl%d %s pointopoint %s netmask %s mtu %d up",
	    PATH_IFCONFIG,link_iface,local_ip,remote_ip,netmask,mtu);
    } else {
        sprintf(buf,"%s sl%d %s pointopoint %s mtu %d up",
	    PATH_IFCONFIG,link_iface,local_ip,remote_ip,mtu);
    }
    system(buf);

    /* Set the routing for the new slip interface */
    set_ptp("sl",link_iface,remote_ip);

    /* run bootp if it is asked for */
    if (dynamic_addrs && dynamic_mode == DMODE_BOOTP) start_bootp();

    dead = 0;
}

int slip_set_addrs()
{
    char buf[128];
    char type[30];
    char addr[30];

    if (waiting_for_bootp && (!have_local || !have_remote)) {
	int i;
	char *buf, *tail;
	buf = tail = bootp_pipe.buf;
	i = pipe_read(&bootp_pipe);
	if (i <= 0) return 0;
	while (i--) {
	   if (*tail == '\n') {
		*tail = 0;
		/* Got a line, deal with it */
		if (sscanf(buf,"%[^=]=%s\n",type,addr) == 2) {
		    if (strcmp(type,"SERVER")==0) {
			have_remote = 1;
			remote_ip = strdup(addr);
		    } else if (strcmp(type,"IPADDR")==0) {
			have_local = 1;
			local_ip = strdup(addr);
		    }
		}
		buf = tail+1;
	   }
	   tail++;
	}
	pipe_flush(&bootp_pipe,buf-bootp_pipe.buf);
    }

    if (waiting_for_bootp && (!have_local || !have_remote))
        return 0;

    if (waiting_for_bootp) {
        pclose(bootpfp);
	syslog(LOG_INFO,"New addresses: local %s, remote %s.",
	    local_ip,remote_ip);
    	waiting_for_bootp = 0;
    }


    /* redo the interface marking and the routing since BOOTP will change it */
    if (netmask) {
        sprintf(buf,"%s sl%d %s pointopoint %s netmask %s mtu %d up",
	    PATH_IFCONFIG,link_iface,local_ip,remote_ip,netmask,mtu);
    } else {
        sprintf(buf,"%s sl%d %s pointopoint %s mtu %d up",
	    PATH_IFCONFIG,link_iface,local_ip,remote_ip,mtu);
    }
    system(buf);

    /* Set the routing for the new slip interface */
    set_ptp("sl",link_iface,remote_ip);

    if (dynamic_addrs) {
	local_addr = inet_addr(local_ip);
	proxy_config(local_ip,remote_ip);
#ifndef UNSAFE_ROUTING
	add_routes("sl",proxy_iface,local_ip,remote_ip);
#endif
    }

#ifdef UNSAFE_ROUTING
    add_routes("sl",link_iface,local_ip,remote_ip);
#endif

    return 1;
}

int slip_dead()
{
    return (dead);
}

/* Stop the slip interface. I'm not sure this needs to do anything
 * other than down the interface. In fact I'm not even sure I need to do that...
 */
void slip_stop()
{
    /* set the line discipline back to the starting discipline */
    ioctl(modem_fd, TIOCSETD, &start_disc);
    dead = 1;
}

void slip_reroute()
{
    /* Restore the original proxy routing */
    proxy_config(orig_local_ip,orig_remote_ip);
    set_ptp("sl",proxy_iface,orig_remote_ip);
    add_routes("sl",proxy_iface,orig_local_ip,orig_remote_ip);
    local_addr = inet_addr(orig_local_ip);
#ifdef UNSAFE_ROUTING
    /* If we did routing on the slip link, remove it */
    if (link_iface != -1) /* just in case we get called twice */
    	del_routes("sl",link_iface,local_ip,remote_ip);
#endif
    link_iface = -1;
}

/* Dummy proc. This should never get called */
void slip_kill()
{
}
