/*
linux/fs/dmsdos/dmsdos_std.c

DMSDOS filesystem: standard routines.

This file adapts the standard linux filesystem interface to the dmsdos
routines and decides whether to call msdos, dmsdos or umsdos for the desired
operation.

NOTE: umsdos support currently needs a patch in the umsdos' inode.c. See doc.

Email and author see file dmsdos_spc.c.

*/

#ifdef MODULE
#include <linux/module.h>
#include <linux/version.h>
#else
#define MOD_INC_USE_COUNT
#define MOD_DEC_USE_COUNT
#endif

#include <linux/sched.h>
#include <linux/ctype.h>
#include <linux/major.h>
#include <linux/blkdev.h>
#include <linux/fs.h>
#include <linux/stat.h>
#include <linux/locks.h>
#include <asm/segment.h>
#include <linux/mm.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/msdos_fs.h>
#include <linux/dmsdos_fs.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/shm.h>
#include <linux/mman.h>
#include <asm/system.h>

#define PRINTK(X)

#include <linux/autoconf.h>
/* change 'CONFIG_UMSDOS_FS' to 'NOTHING_IMPORTANT' if you have umsdos
   support in your kernel but do not want to use umsdos together with
   dmsdos or if you did not patch linux/fs/umsdos/inode.c */
#ifdef CONFIG_UMSDOS_FS
#define OLD_MSDOS_put_inode UMSDOS_put_inode
#define OLD_MSDOS_put_super UMSDOS_put_super
#define OLD_MSDOS_read_super UMSDOS_read_super
#define OLD_MSDOS_statfs UMSDOS_statfs
#define OLD_MSDOS_read_inode UMSDOS_read_inode
#define OLD_MSDOS_write_inode UMSDOS_write_inode
#define OLD_MSDOS_notify_change UMSDOS_notify_change
#include <linux/umsdos_fs.h>
#else
#define OLD_MSDOS_put_inode msdos_put_inode
#define OLD_MSDOS_put_super msdos_put_super
#define OLD_MSDOS_read_super msdos_read_super
#define OLD_MSDOS_statfs msdos_statfs
#define OLD_MSDOS_read_inode msdos_read_inode
#define OLD_MSDOS_write_inode msdos_write_inode
#define OLD_MSDOS_notify_change msdos_notify_change
#endif
#ifndef CONFIG_MSDOS_FS
#error DMSDOS filesystem needs MSDOS filesystem support!
#endif

static int dmsdos_dir_read(struct inode * inode,struct file * filp, char * buf,int count)
{
	return -EISDIR;
}

static struct file_operations dmsdos_dir_operations = {
	NULL,			/* lseek - default */
	dmsdos_dir_read,		/* read */
	NULL,			/* write - bad */
	dmsdos_readdir,		/* readdir */
	NULL,			/* select - default */
	NULL,			/* ioctl - default */
	NULL,			/* mmap */
	NULL,			/* no special open code */
	NULL,			/* no special release code */
	NULL     		/* fsync */
};

struct inode_operations dmsdos_dir_inode_operations = {
	&dmsdos_dir_operations,	/* default directory file-ops */
	NULL,   		/* create */
	dmsdos_lookup,		/* lookup */
	NULL,			/* link */
	NULL,    		/* unlink */
	NULL,			/* symlink */
	NULL,   		/* mkdir */
	NULL,   		/* rmdir */
	NULL,			/* mknod */
	NULL,   		/* rename */
	NULL,			/* readlink */
	NULL,			/* follow_link */
	NULL,    		/* bmap */
	NULL,			/* truncate */
	NULL			/* permission */
};

int dmsdos_readdir(
	struct inode *inode,
	struct file *filp,
	struct dirent *dirent,
	int count)
{	/*printk("DMSDOS: readdir ino=%ld\n",inode->i_ino);*/
        if(dostest(inode))
        { printk("DMSDOS: readdir: msdos!\n");
          return -EPERM;
        }
        return dmsdos_readdirx(inode,filp,dirent,count);
}

static struct file_operations dmsdos_file_operations = {
	NULL,			/* lseek - default */
	dmsdos_file_read,	/* read */
	NULL,           	/* write */
	NULL,			/* readdir - bad */
	NULL,			/* select - default */
	NULL,			/* ioctl - default */
	dmsdos_mmap,    	/* mmap */
	NULL,			/* no special open is needed */
	NULL,			/* release */
	NULL    		/* fsync */
};

struct inode_operations dmsdos_file_inode_operations = {
	&dmsdos_file_operations,	/* default file operations */
	NULL,			/* create */
	NULL,			/* lookup */
	NULL,			/* link */
	NULL,			/* unlink */
	NULL,			/* symlink */
	NULL,			/* mkdir */
	NULL,			/* rmdir */
	NULL,			/* mknod */
	NULL,			/* rename */
	NULL,			/* readlink */
	NULL,			/* follow_link */
	NULL,   		/* bmap */
	NULL,   		/* truncate */
	NULL,			/* permission */
	NULL			/* smap */
};

static struct file_operations dmsdos_file_operations_1024 = {
	NULL,			/* lseek - default */
	dmsdos_file_read,	/* read */
	NULL,           	/* write */
	NULL,			/* readdir - bad */
	NULL,			/* select - default */
	NULL,			/* ioctl - default */
	dmsdos_mmap,     	/* mmap */
	NULL,			/* no special open is needed */
	NULL,			/* release */
	NULL     		/* fsync */
};

struct inode_operations dmsdos_file_inode_operations_1024 = {
	&dmsdos_file_operations_1024,	/* default file operations */
	NULL,			/* create */
	NULL,			/* lookup */
	NULL,			/* link */
	NULL,			/* unlink */
	NULL,			/* symlink */
	NULL,			/* mkdir */
	NULL,			/* rmdir */
	NULL,			/* mknod */
	NULL,			/* rename */
	NULL,			/* readlink */
	NULL,			/* follow_link */
	NULL,			/* bmap */
	NULL,   		/* truncate */
	NULL,			/* permission */
	NULL			/* smap */
};

int dmsdos_file_read(
	struct inode *inode,
	struct file *filp,
	char *buf,
	int count)
{       /*printk("DMSDOS file_read ino=%ld\n",inode->i_ino);*/
        if(dostest(inode))
        { printk("DMSDOS: file_read: msdos!\n");
          return -EPERM;
        }
        return dmsdos_file_readx(inode,filp,buf,count);
}

void dmsdos_put_inode(struct inode *inode)
{	/*printk("DMSDOS: put inode ino=%ld\n",inode->i_ino);*/
        if(dostest(inode))OLD_MSDOS_put_inode(inode);
        else clear_inode(inode);
}

void dmsdos_put_super(struct super_block *sb)
{ 	/*printk("DMSDOS put super\n");*/
        exit_dbl(sb);
        OLD_MSDOS_put_super(sb);
}

static struct super_operations dmsdos_sops = { 
	dmsdos_read_inode,
	dmsdos_notify_change,
	dmsdos_write_inode,
	dmsdos_put_inode,
	dmsdos_put_super,
	NULL, /* added in 0.96c */
	dmsdos_statfs,
	NULL
};

struct super_block *dmsdos_read_super(struct super_block *sb,void *data,
				     int silent)
{	struct super_block *s;

        /*printk("DMSDOS read_super\n");*/
        s=OLD_MSDOS_read_super(sb,data,silent);
        s->s_op=&dmsdos_sops;
        dbl_init(s,data,silent);
        return s;
}

void dmsdos_statfs(struct super_block *sb,struct statfs *buf)
{	/*printk("DMSDOS statfs\n");*/
        OLD_MSDOS_statfs(sb,buf);
}

void dmsdos_read_inode(struct inode *inode)
{	/*printk("DMSDOS read_inode ino=%ld\n",inode->i_ino);*/
        if(dostest(inode))OLD_MSDOS_read_inode(inode);
	else dmsdos_read_inodex(inode);
}

void dmsdos_write_inode(struct inode *inode)
{	/*printk("DMSDOS write inode ino=%ld\n",inode->i_ino);*/
        if(dostest(inode))OLD_MSDOS_write_inode(inode);
        /*else printk("DMSDOS: writeinode ignored\n");*/
}

int dmsdos_notify_change(struct inode * inode,struct iattr * attr)
{	/*printk("DMSDOS notify_change ino=%ld\n",inode->i_ino);*/
        if(dostest(inode))return OLD_MSDOS_notify_change(inode,attr);
        printk("DMSDOS: notifychange unsupported\n");
        return -EPERM;
}

#ifdef MODULE

char kernel_version[] = UTS_RELEASE;

static struct file_system_type dmsdos_fs_type = {
	dmsdos_read_super, "dmsdos", 1, NULL
};

int init_module(void)
{
	register_filesystem(&dmsdos_fs_type);
	return 0;
}

void cleanup_module(void)
{
	unregister_filesystem(&dmsdos_fs_type);
}

#endif

/*
 * Fill in the supplied page for mmap
 */
static unsigned long dmsdos_file_mmap_nopage(
	struct vm_area_struct * area,
	unsigned long address,
	unsigned long page,
	int error_code)
{
	struct inode * inode = area->vm_inode;
	unsigned int clear;
	int pos;
	long gap;	/* distance from eof to pos */
	
	/*printk("DMSDOS_file_mmap_nopage\n");*/

	address &= PAGE_MASK;
	pos = address - area->vm_start + area->vm_offset;

	clear = 0;
	gap = inode->i_size - pos;
	if (gap <= 0){
		/* mmaping beyond end of file */
		clear = PAGE_SIZE;
	}else{
		int cur_read;
		int need_read;
		struct file filp;
		if (gap < PAGE_SIZE){
			clear = PAGE_SIZE - gap;
		}
		filp.f_reada = 0;
		filp.f_pos = pos;
		need_read = PAGE_SIZE - clear;
		{
			unsigned long cur_fs = get_fs();
			set_fs (KERNEL_DS);
			cur_read = dmsdos_file_read (inode,&filp,(char*)page
				,need_read);
			set_fs (cur_fs);
		}
		if (cur_read != need_read){
			printk ("DMSDOS: Error while reading an mmap file %d <> %d\n"
				,cur_read,need_read);
		}
	}
	if (clear > 0){
		memset ((char*)page+PAGE_SIZE-clear,0,clear);
	}
	return page;
}

struct vm_operations_struct dmsdos_file_mmap = {
	NULL,			/* open */
	NULL,			/* close */
	NULL,			/* unmap */
	NULL,			/* protect */
	NULL,			/* sync */
	NULL,			/* advise */
	dmsdos_file_mmap_nopage,	/* nopage */
	NULL,			/* wppage */
	NULL,			/* swapout */
	NULL,			/* swapin */
};

/*
 * This is used for a general mmap of an dmsdos file
 * Returns 0 if ok, or a negative error code if not.
 */
int dmsdos_mmap(struct inode * inode, struct file * file, struct vm_area_struct * vma)
{
	/*printk("DMSDOS mmap ino=%ld\n",inode->i_ino);*/
	if (vma->vm_flags & VM_SHARED)	/* only PAGE_COW or read-only supported now */
		return -EINVAL;
	if (vma->vm_offset & (inode->i_sb->s_blocksize - 1))
		return -EINVAL;
	if (!inode->i_sb || !S_ISREG(inode->i_mode))
		return -EACCES;
	if (!IS_RDONLY(inode)) {
		inode->i_atime = CURRENT_TIME;
		inode->i_dirt = 1;
	}

	vma->vm_inode = inode;
	inode->i_count++;
	vma->vm_ops = &dmsdos_file_mmap;
	return 0;
}

int dmsdos_lookup(struct inode *dir,const char *name,int len,
    struct inode **result)
{	/*printk("DMSDOS lookup ino=%ld name=%s\n",dir->i_ino,name);*/
        if(dostest(dir))
        { printk("DMSDOS: lookup: msdos!\n");
          return -EPERM;
        }
	return dmsdos_lookupx(dir,name,len,result);
}
