
/* TclQddb_Utils.c - TCL interface routines for Qddb schemas.
 *
 * Copyright (C) 1994 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */


#include "tcl.h"
#include "Qddb.h"
#include "tclQddb.h"

void TclQddb_FreeArgs(argc, argv)
    int				argc;
    char			*argv[];
{
    int				i;

    if (argc > 0) {
	for (i = 0; i < argc; i++)
	    Free(argv[i]);
    } else {
	for (i = 0; argv[i] != NULL; i++)
	    Free(argv[i]);
    }
    Free(argv);
}

char **TclQddb_SplitAttributeName(name)
    char			*name;
{
    char			**retval, *chidx, *newname;
    int				i, numdots;

    numdots = 0;
    for (chidx = name; *chidx != '\0'; chidx++)
	if (*chidx == '.')
	    numdots++;
    retval = (char **)Malloc(sizeof(char *)*(numdots+2));
    retval[numdots+1] = NULL;
    newname = Malloc(strlen(name)+1);
    strcpy(newname, name);
    chidx = strtok(newname, ".");
    if (chidx == NULL) {
	retval[0] = newname;
	retval[1] = '\0';
	return retval;
    }
    for (i = 0; i <= numdots; i++) {
	retval[i] = Malloc(strlen(chidx)+1);
	strcpy(retval[i], chidx);
	chidx = strtok(NULL, ".");
    }
    Free(newname);
    return retval;
}

char *TclQddb_ConcatAttributeName(name_list, num)
    char			**name_list;
    int				num;
{
    int				i;
    size_t			len;
    char			*retval;

    for (len = 0, i = 0; i < num; i++)
	len += strlen(name_list[i]);
    retval = Malloc(len+num);
    retval[0] = '\0';
    for (i = 0; i < num-1; i++) {
	strcat(retval, name_list[i]);
	strcat(retval, ".");
    }
    strcat(retval, name_list[i]);
    return retval;
}

int TclQddb_FindAttribute(tree, name)
    DataTree			**tree;
    char			*name;
{
    int				i;

    for (i = 0; tree[i] != NULL; i++)
	if (strcmp(tree[i]->datatree_schema->schemanode->Name, name) == 0)
	    return i;
    return -1;
}

char **TclQddb_ParseAttr(name)
    char			*name;
{
    int				numdots, i;
    size_t			len;
    char			**retval;

    if (name == NULL)
	return NULL;
    retval = (char **)Calloc(sizeof(char *)*(MAXIMUM_NUMBER_OF_SCHEMA_LEVELS+1));
    len = strlen(name);
    for (i = 0, numdots = 0; i < len; i++) {
	if (name[i] == '.') {
	    name[i] = '\0';
	    retval[numdots] = Malloc((size_t)i+1);
	    strcpy(retval[numdots++], name);
	    name[i] = '.';
	}
    }
    retval[numdots] = Malloc(len+1);
    strcpy(retval[numdots], name);
    retval[numdots+1] = NULL;
    return retval;
}

int TclQddb_CountAttrs(attrs)
    char			*attrs;
{
    int				i, numdots;
    if (attrs[0] == '\0')
	return 0;
    for (i = 0, numdots = 0; attrs[i] != '\0'; i++)
	if (attrs[i] == '.')
	    numdots++;
    return ++numdots;
}

char *TclQddb_StringRef(tuple, rownode)
    TclQddb_Tuple		*tuple;
    TclQddb_Rows		*rownode;
{
    DataTree			**dt;
    char			**attrs, *retval = NULL;
    size_t			inst[MAXIMUM_NUMBER_OF_SCHEMA_LEVELS];
    int				num, nextattr;
    int				i;

    attrs = TclQddb_SplitAttributeName(rownode->attr_name);
    num = StringToInstance(rownode->instance, inst);
    dt = tuple->datatree;
    i = 0;
    while ((nextattr = TclQddb_FindAttribute(dt, attrs[i])) != -1) {
	if (i >= num) {
	    TclQddb_FreeArgs(-1, attrs);
	    return NULL;
	}
	if (dt[nextattr]->datatree_type == DATATREE_CHILDREN)
	    dt = dt[nextattr][inst[i] - 1].datatree_children;
	else {
	    retval = TclQddb_TreeDataToString(dt[nextattr] + inst[i] - 1);
	    break;
	}
	i++;
    }
    TclQddb_FreeArgs(-1, attrs);
    return retval;
}

char *TclQddb_TreeDataToString(tree_entry)
    DataTree			*tree_entry;
{
    char			*retval;
    char			buf[BUFSIZ];

    switch(tree_entry->datatree_type) {
    case DATATREE_STRING:
	retval = Malloc(strlen(tree_entry->datatree_string)+1);
	strcpy(retval, tree_entry->datatree_string);
	break;
    case DATATREE_DATE:
	retval = Malloc(strlen(tree_entry->datatree_date)+1);
	strcpy(retval, tree_entry->datatree_date);
	break;
    case DATATREE_INT:
	sprintf(buf, "%d", tree_entry->datatree_int);
	retval = Malloc(strlen(buf)+1);
	strcpy(retval, buf);
	break;
    case DATATREE_REAL:
	sprintf(buf, "%f", tree_entry->datatree_real);
	retval = Malloc(strlen(buf)+1);
	strcpy(retval, buf);
	break;
    case DATATREE_NOVAL:
    default:
	return NULL;
    }
    return retval;
}


DataTree *TclQddb_StringToTreeData(node, value)
    SchemaTreeNode		*node;
    char			*value;
{
    DataTree			*retval;

    retval = (DataTree *)Malloc(sizeof(DataTree));
    switch(node->schemanode->Type) {
    case String:
	retval->datatree_type = DATATREE_STRING;
	retval->datatree_string = Malloc(strlen(value)+1);
	retval->datatree_schema = node;
	strcpy(retval->datatree_string, value);
	break;
    case Date:
	retval->datatree_type = DATATREE_DATE;
	retval->datatree_date = Malloc(strlen(value)+1);
	retval->datatree_schema = node;
	strcpy(retval->datatree_date, value);
	break;
    case Integer:
	retval->datatree_type = DATATREE_DATE;
	retval->datatree_int = atoi(value);
	retval->datatree_schema = node;
	break;
    case Real:
	retval->datatree_type = DATATREE_REAL;
	retval->datatree_real = strtod(value, NULL);
	retval->datatree_schema = node;
    default:
	PANIC("TclQddb_StringToTreeData");
    }
    return retval;
}


void TclQddb_NullOutNode(dt, do_realloc)
    DataTree		*dt;
    int			do_realloc;
{
    switch (dt->datatree_type) {
    case DATATREE_INT:
	dt->datatree_int = 0.0;
	break;
    case DATATREE_REAL:
	dt->datatree_real = 0.0;
	break;
    case DATATREE_DATE:
	if (dt->datatree_date != NULL)
	    Free(dt->datatree_date);
	if (do_realloc)
	    dt->datatree_date = Calloc(1);
	else
	    dt->datatree_date = NULL;
	break;
    case DATATREE_STRING:
	if (dt->datatree_string != NULL)
	    Free(dt->datatree_string);
	if (do_realloc)
	    dt->datatree_string = Calloc(1);
	else
	    dt->datatree_string = NULL;
	break;
    default:
	return;
    }
}

Boolean TclQddb_IsEmptyDataTree(dt)
    DataTree		**dt;
{
    DataTree		*inst;
    int			i;

    while (*dt != NULL) {
	inst = *dt;
	for (i = 0; inst[i].datatree_type != DATATREE_END; i++) {
	    switch (inst[i].datatree_type) {
	    case DATATREE_INT:
		return False;
		break;
	    case DATATREE_REAL:
		return False;
		break;
	    case DATATREE_DATE:
		if (inst[i].datatree_date[0] != '\0')
		    return False;
		break;
	    case DATATREE_STRING:
		if (inst[i].datatree_string[0] != '\0')
		    return False;
		break;
	    case DATATREE_CHILDREN:
		if (TclQddb_IsEmptyDataTree(inst[i].datatree_children) == False)
		    return False;
	    default:
		break;
	    }
	}
	dt++;
    }
    return True;
}
