
/* TclQddb_Tuple.c - TCL interface routines for Qddb tuples.
 *
 * Copyright (C) 1994, 1995 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */

#include "tcl.h"
#include "Qddb.h"
#include "tclQddb.h"

static Tcl_HashTable 	TclQddb_TupleHashTable;
static int		TclQddb_TupleHashTableInit = 0;
static unsigned int	TclQddb_TupleNextNumber = 0;

static int TclQddb_ReadTuple _ANSI_ARGS_((Tcl_Interp *, char *, KeyList *));
static int TclQddb_WriteTuple _ANSI_ARGS_((Tcl_Interp *, char *));
static int TclQddb_ConvertTuple _ANSI_ARGS_((Tcl_Interp *, char *, char *));
static int TclQddb_PutTuple _ANSI_ARGS_((Tcl_Interp *, char *, char *, char *));
static int TclQddb_CreateNewTuple _ANSI_ARGS_((Tcl_Interp *, char *));
static int TclQddb_RemoveTuple _ANSI_ARGS_((Tcl_Interp *, char *));
static int TclQddb_RefreshTuple _ANSI_ARGS_((Tcl_Interp *, char *));
static int TclQddb_IsEmptyTuple _ANSI_ARGS_((Tcl_Interp *, char *));
static int TclQddb_LockTuple _ANSI_ARGS_((Tcl_Interp *, char *));
static int TclQddb_UnlockTuple _ANSI_ARGS_((Tcl_Interp *, char *));
static TclQddb_Tuple *TclQddb_GrabTuple _ANSI_ARGS_((Tcl_Interp *, char *));

/* TclQddb_TupleProc - qddb_tuple command.
 *
 * Usage:
 *	<tuple desc>    <- qddb_tuple read <schema desc> <keylist quadruple>
 *			   qddb_tuple write <tuple desc>
 *	<tuple string>  <- qddb_tuple get <format> <tuple desc>
 *	<tuple desc>    <- qddb_tuple put <format> <schema desc> <tuple string>
 *	<tuple desc>    <- qddb_tuple new <schema desc>
 *                         qddb_tuple lock <tuple desc>
 *                         qddb_tuple unlock <tuple desc>
 *			   qddb_tuple refresh <tuple desc>
 *                         qddb_tuple remove <tuple desc>
 *			   qddb_tuple delete <tuple desc>|all
 *
 */
int TclQddb_TupleProc(clientData, interp, argc, argv)
    ClientData			clientData;
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    int				nargc;
    char			**nargv;
    KeyList			keylist;

    if (argc < 3 || argc > 5) {
	Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    if (TclQddb_TupleHashTableInit == 0) {
	TclQddb_TupleHashTableInit = 1;
	Tcl_InitHashTable(&TclQddb_TupleHashTable, TCL_STRING_KEYS);
    }
    switch (argv[1][0]) {
    case 'r': /* refresh, remove, read */
	if (strcmp(argv[1], "refresh") == 0) {
	    if (argc != 3) {
		Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
		return TCL_ERROR;	    
	    }
	    if (TclQddb_RefreshTuple(interp, argv[2]) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp(argv[1], "remove") == 0) {
	    if (argc != 3) {
		Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
		return TCL_ERROR;	    
	    }
	    if (TclQddb_RemoveTuple(interp, argv[2]) != TCL_OK)
		return TCL_ERROR;
	} else {
	    if (strcmp(argv[1], "read") != 0) {
		Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[2], NULL);
		return TCL_ERROR;
	    }
	    if (argc != 4) {
		Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
		return TCL_ERROR;	    
	    }
	    if (Tcl_SplitList(interp, argv[3], &nargc, &nargv) != TCL_OK)
		return TCL_ERROR;
	    if (nargc != 4) { /* a quadruple */
		Tcl_AppendResult(interp, argv[0], ": wrong # elements in keylist (should be 4)", NULL);
		return TCL_ERROR;	    
	    }
	    keylist.Start = atoi(nargv[0]);
	    keylist.Length = atoi(nargv[1]);
	    keylist.Number = atoi(nargv[2]);
	    keylist.Type = atoi(nargv[3]);
	    Free(nargv);
	    if (TclQddb_ReadTuple(interp, argv[2], &keylist) != TCL_OK)
		return TCL_ERROR;
	}
	break;
    case 'w': /* write */
	if (strcmp(argv[1], "write") != 0) {
	    Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	    return TCL_ERROR;
	}
	if (argc != 3) {
	    Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
	    return TCL_ERROR;	    
	}
	if (TclQddb_WriteTuple(interp, argv[2]) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 'g': /* get (convert into string) */
	if (strcmp(argv[1], "get") != 0) {
	    Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	    return TCL_ERROR;
	}
	if (argc != 4) {
	    Tcl_AppendResult(interp, argv[0], ": wrong # args for get", NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_ConvertTuple(interp, argv[2], argv[3]) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 'p': /* put (convert from string) */
	if (strcmp(argv[1], "put") != 0) {
	    Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	    return TCL_ERROR;
	}
	if (argc != 5) {
	    Tcl_AppendResult(interp, argv[0], ": wrong # args for put", NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_PutTuple(interp, argv[2], argv[3], argv[4]) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 'f': /* flush (flush out all the tuple data structures) */
	if (strcmp(argv[1], "flush") != 0) {
	    Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	    return TCL_ERROR;
	}
	if (argc != 3) {
	    Tcl_AppendResult(interp, argv[0], ": wrong # args for flush", NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_FlushTuple(interp, argv[2]) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 'd': /* delete */
	if (strcmp(argv[1], "delete") != 0) {
	    Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	    return TCL_ERROR;
	}
	if (argc != 3) {
	    Tcl_AppendResult(interp, argv[0], ": wrong # args for delete", NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_DeleteTuple(interp, argv[2]) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 'n': /* new */
	if (strcmp(argv[1], "new") != 0) {
	    Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	    return TCL_ERROR;
	}
	if (argc != 3) {
	    Tcl_AppendResult(interp, argv[0], ": wrong # args for new", NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_CreateNewTuple(interp, argv[2]) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 'i':
	if (strcmp(argv[1], "isempty") != 0) {
	    Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	    return TCL_ERROR;
	}
	if (argc != 3) {
	    Tcl_AppendResult(interp, argv[0], ": wrong # args for isempty", NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_IsEmptyTuple(interp, argv[2]) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 'l':
	if (strcmp(argv[1], "lock") != 0) {
	    Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	    return TCL_ERROR;
	}
	if (argc != 3) {
	    Tcl_AppendResult(interp, argv[0], ": wrong # args for lock", NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_LockTuple(interp, argv[2]) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 'u':
	if (strcmp(argv[1], "unlock") != 0) {
	    Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	    return TCL_ERROR;
	}
	if (argc != 3) {
	    Tcl_AppendResult(interp, argv[0], ": wrong # args for unlock", NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_UnlockTuple(interp, argv[2]) != TCL_OK)
	    return TCL_ERROR;
	break;
    default:
	Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

static int TclQddb_LockTuple(interp, token)
    Tcl_Interp			*interp;
    char			*token;
{
#if defined(RECORDLOCKING)
    TclQddb_Tuple		*tuple;
    Schema			*schema;
    int				fd, old_type;
    off_t			start = 0;
    size_t			length = 0;

    tuple = TclQddb_GetTuple(interp, token);
    if (tuple == NULL) {
	Tcl_AppendResult(interp, "cannot find tuple \"", token, "\" (Qddb Error)", NULL);
	return TCL_ERROR;	
    }
    if ((schema = TclQddb_GetSchema(tuple->schema_name)) == NULL) {
	Tcl_AppendResult(interp, "bad schema \"", tuple->schema_name, "\"", NULL);
	return TCL_ERROR;
    }
    if (tuple->fd != -1) {
	TclQddb_UnlockTuple(interp, token);
    }
    old_type = tuple->keylist.Type;
    switch (tuple->keylist.Type) {
    case ADDITION:
	tuple->fd = OpenAdditionFile(schema->RelationName, tuple->keylist.Number, 0);
	fd = tuple->fd;
	break;
    case CHANGE:
	tuple->fd = OpenChangeFile(schema->RelationName, tuple->keylist.Number, 0);
	fd = tuple->fd;
	break;
    case ORIGINAL:
	fd = schema->database_fd;
	tuple->fd = fd;
	start = tuple->keylist.Start;
	length = tuple->keylist.Length;
	break;
    default:
	fd = -1;
	tuple->fd = -1;
	;
    }
    if (fd != -1) {
	/* obtain a write lock */
	if (LockSection(fd, F_WRLCK, start, (off_t)length, False) == -1) {
	    Tcl_SetResult(interp, "0", TCL_STATIC);
	    return TCL_OK;
	}
	/* convert it to a read lock */
	if (LockSection(fd, F_RDLCK, start, (off_t)length, False) == -1) {
	    Tcl_SetResult(interp, "0", TCL_STATIC);
	    return TCL_OK;
	}
    }
    /* The lock has been obtained; refresh the tuple to make sure this is the
     * latest version.
     */
    TclQddb_RefreshTuple(interp, token);
    tuple = TclQddb_GetTuple(interp, token);
    if (tuple == NULL) {
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "tuple \"", token, "\" has been deleted", NULL);
	return TCL_ERROR;	
    }
    if (tuple->keylist.Type != old_type) {
	/* The tuple has been refreshed and the keylist type has changed;
	 * the lock previously obtained is invalid, so try again.
	 */
	if (TclQddb_LockTuple(interp, token) != TCL_OK)
	    return TCL_ERROR;
	return TCL_OK;
    }
    Tcl_SetResult(interp, "1", TCL_STATIC);
    return TCL_OK;
#else
    Tcl_SetResult(interp, "1", TCL_STATIC);
    return TCL_OK;
#endif
}

static int TclQddb_UnlockTuple(interp, token)
    Tcl_Interp			*interp;
    char			*token;
{
#if defined(RECORDLOCKING)
    TclQddb_Tuple		*tuple;
    Schema			*schema;
    int				fd;
    off_t			start = 0;
    size_t			length = 0;

    tuple = TclQddb_GetTuple(interp, token);
    if (tuple == NULL) {
	Tcl_AppendResult(interp, "cannot find tuple \"", token, "\" (Qddb Error)", NULL);
	return TCL_ERROR;	
    }
    if ((schema = TclQddb_GetSchema(tuple->schema_name)) == NULL) {
	Tcl_AppendResult(interp, "bad schema \"", tuple->schema_name, "\"", NULL);
	return TCL_ERROR;
    }
    switch (tuple->keylist.Type) {
    case ORIGINAL:
	fd = schema->database_fd;
	start = tuple->keylist.Start;
	length = tuple->keylist.Length;
	break;
    default:
	fd = tuple->fd;
    }
    if (fd != -1) {
	UnlockSection(fd, start, (off_t)length);
	if (tuple->fd != -1) {
	    if (tuple->fd != schema->database_fd)
		Close(tuple->fd);
	    tuple->fd = -1;
	}
    }
    return TCL_OK;
#else
    return TCL_OK;
#endif
}

static int TclQddb_CreateNewTuple(interp, schema_desc)
    Tcl_Interp			*interp;
    char			*schema_desc;
{
    Schema			*schema;
    DataTree			**dt;
    KeyList			keylist;
    int				seqno;
    char			*tuple_token;

    if ((schema = TclQddb_GetSchema(schema_desc)) == NULL) {
	Tcl_AppendResult(interp, "bad schema \"", schema_desc, "\"", NULL);
	return TCL_ERROR;
    }
    if ((dt = Qddb_DataTreeProcess(schema, NULL, schema->Tree,
				   QDDB_DATATREE_PROC_NEWINSTANCE, 0)) == NULL) {
	Tcl_AppendResult(interp, "cannot create new tuple for schema \"", schema_desc, "\" (Qddb error)",
			 "(Qddb_DataTreeProcess failed)", NULL);
	return TCL_ERROR;
    }
    seqno = -1;
    Qddb_DataTreeProcess(schema, dt, &seqno, QDDB_DATATREE_PROC_SETSEQ, 0);
    QDDB_KEYLIST_SET_TYPE(&keylist, NEWENTRY);
    keylist.Start = 0;
    keylist.Length = 0;
    keylist.Number = 0;
    keylist.Instance = NULL;
    keylist.Attribute = 0;
    if ((tuple_token = TclQddb_NewTuple(interp, schema_desc, dt, &keylist)) == NULL) {
	Tcl_AppendResult(interp, "cannot create new tuple for schema \"", schema_desc, "\" (Qddb error)",
			 "(Qddb_DataTreeProcess failed)", NULL);
	return TCL_ERROR;
    }
#if defined(DATATREE_DEBUG)
    fprintf(stderr, "Created new tuple, %s\n", tuple_token);
#endif
    Tcl_SetResult(interp, tuple_token, TCL_DYNAMIC);
    return TCL_OK;
}

char *TclQddb_NewTuple(interp, schema_desc, datatree, keylist)
    Tcl_Interp			*interp;
    char			*schema_desc;
    DataTree			**datatree;
    KeyList			*keylist;
{
    TclQddb_Tuple		*tuple;
    Tcl_HashEntry		*hash_ptr;
    char			token[BUFSIZ], *retval;
    int				newPtr;

    if (TclQddb_TupleHashTableInit == 0) {
	TclQddb_TupleHashTableInit = 1;
	Tcl_InitHashTable(&TclQddb_TupleHashTable, TCL_STRING_KEYS);
    }
    sprintf(token, "qddb_tuple%d", TclQddb_TupleNextNumber++);
    tuple = (TclQddb_Tuple *)Malloc(sizeof(TclQddb_Tuple));
    tuple->schema_name = Malloc(strlen(schema_desc)+1);
    strcpy(tuple->schema_name, schema_desc);
    tuple->fd = -1;
    tuple->datatree = datatree;
    tuple->keylist = *keylist;
    tuple->keylist.Instance = NULL;
    tuple->keylist.Attribute = 0;
    hash_ptr = Tcl_CreateHashEntry(&TclQddb_TupleHashTable, token, &newPtr);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot create hash entry \"", token, "\" (TCL error)", NULL);
	Qddb_Free(QDDB_TYPE_DATATREE, tuple->datatree);
	Free(tuple);
	return NULL;
    }
    Tcl_SetHashValue(hash_ptr, (ClientData)tuple);
    retval = Malloc(strlen(token)+1);
    strcpy(retval, token);
    return retval;
}

static int TclQddb_ReadTuple(interp, schema_desc, keylist)
    Tcl_Interp			*interp;
    char			*schema_desc;
    KeyList			*keylist;
{
    Schema			*schema;
    char			*token;
    int				fd;
    DataTree			**datatree;
    Entry			external = NULL;

    if ((schema = TclQddb_GetSchema(schema_desc)) == NULL) {
	Tcl_AppendResult(interp, "bad schema \"", schema_desc, "\"", NULL);
	return TCL_ERROR;
    }
    fd = schema->database_fd;
    if (Qddb_ReadEntryByKeyList(fd, schema->RelationName, &external, keylist, True) == -1) {
	token = "";
	Tcl_SetResult(interp, token, TCL_STATIC);
	return TCL_OK;
    } else {
	Qddb_ReducedAttrToFullAttr(schema, external);
	datatree = Qddb_Convert(schema, QDDB_ENTRYTYPE_EXTERNAL, external, QDDB_ENTRYTYPE_DATATREE);
	Qddb_Free(QDDB_TYPE_ENTRY, external);
	if (datatree == NULL) {
	    Tcl_AppendResult(interp, "conversion failed", NULL);
	    return TCL_ERROR;
	}
	if ((token = TclQddb_NewTuple(interp, schema_desc, datatree, keylist)) == NULL) {
	    Qddb_Free(QDDB_TYPE_DATATREE, datatree);
	    return TCL_ERROR;
	}
    }
    Tcl_SetResult(interp, token, TCL_DYNAMIC);
    return TCL_OK;
}

static int TclQddb_WriteTuple(interp, token)
    Tcl_Interp			*interp;
    char			*token;
{
    TclQddb_Tuple		*tuple;
    Schema			*schema;
    Entry			external;
    int				number;

    tuple = TclQddb_GetTuple(interp, token);
    if (tuple == NULL) {
	Tcl_AppendResult(interp, "cannot find tuple \"", token, "\" (Qddb Error)", NULL);
	return TCL_ERROR;	
    }
    if ((schema = TclQddb_GetSchema(tuple->schema_name)) == NULL) {
	Tcl_AppendResult(interp, "bad schema \"", tuple->schema_name, "\"", NULL);
	return TCL_ERROR;
    }
    external = Qddb_Convert(schema, QDDB_ENTRYTYPE_DATATREE, tuple->datatree, QDDB_ENTRYTYPE_EXTERNAL);
    if (external == NULL) {
	Tcl_AppendResult(interp, "Cannot convert tuple to External type", NULL);
	return TCL_ERROR;
    }
    switch (QDDB_KEYLIST_TYPE(&tuple->keylist)) {
    case ADDITION:
	Qddb_ChangeEntry(schema, tuple->fd, tuple->fd == -1?False:True, external, ADDITION);
	break;
    case CHANGE:
	Qddb_ChangeEntry(schema, tuple->fd, tuple->fd == -1?False:True, external, CHANGE);
	break;
    case ORIGINAL:
	Qddb_InvalidateEntry(schema, tuple->keylist.Start, tuple->keylist.Length);
	Qddb_ChangeEntry(schema, -1, False, external, CHANGE);
	if (tuple->fd != -1) {
	    TclQddb_LockTuple(interp, token);
	}
	tuple->keylist.Number = GetEntryNumber(external);
	QDDB_KEYLIST_SET_TYPE(&tuple->keylist, CHANGE);
	break;
    case NEWENTRY:
	number = Qddb_AddEntry(schema, &(tuple->fd), True, external);
	if (number == -1) {
	    Qddb_Free(QDDB_TYPE_ENTRY, external);
	    Tcl_AppendResult(interp, "Could not obtain lock for addition.  ", 
			     "You may be running over NFS without a lockd.", NULL);
	    return TCL_ERROR;
	}
	tuple->keylist.Number = (size_t)number;
	QDDB_KEYLIST_SET_TYPE(&tuple->keylist, ADDITION);
	break;
    default:
	Qddb_Free(QDDB_TYPE_ENTRY, external);
	Tcl_AppendResult(interp, "Internal TclQddb error: bad keylist associated with tuple", NULL);
	return TCL_ERROR;
    }
    Qddb_Free(QDDB_TYPE_ENTRY, external);
    return TCL_OK;
}

static int TclQddb_DeleteTupleOne(interp, token)
    Tcl_Interp			*interp;
    char			*token;
{
    TclQddb_Tuple		*tuple;
    Tcl_HashEntry		*hash_ptr;

    /* Make sure that the view is deleted BEFORE the tuples;
     * Tcl_UnlinkVar gets confused otherwise.
     */
    if (TclQddb_DeleteView(interp, token) != TCL_OK) {
	Tcl_AppendResult(interp, "cannot delete views associated with tuple \"", 
			 token, "\" (TCL error)", NULL);
	return TCL_ERROR;
    }
    if (TclQddb_DeleteRows(interp, token) != TCL_OK) {
	Tcl_AppendResult(interp, "cannot delete rows associated with tuple \"", 
			 token, "\" (TCL error)", NULL);
	return TCL_ERROR;
    }
    hash_ptr = Tcl_FindHashEntry(&TclQddb_TupleHashTable, token);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot find keylist \"", token, "\" (TCL error)", NULL);
	return TCL_ERROR;	
    }
    tuple = (TclQddb_Tuple *)Tcl_GetHashValue(hash_ptr);
    Tcl_DeleteHashEntry(hash_ptr);
    if (tuple != NULL) {
	Schema		*schema;

	Qddb_Free(QDDB_TYPE_DATATREE, tuple->datatree);
	schema = TclQddb_GetSchema(tuple->schema_name);
	if (schema == NULL) {
	    if (tuple->fd != -1)
		Close(tuple->fd);
	} else {
	    if (tuple->fd != -1 && tuple->fd != schema->database_fd)
		Close(tuple->fd);
	}
	Free(tuple->schema_name);
	Free(tuple);
    }
    return TCL_OK;
}

int TclQddb_DeleteTuple(interp, Token)
    Tcl_Interp			*interp;
    char			*Token;
{
    Tcl_HashEntry		*hash_ptr;
    Tcl_HashSearch		hash_search;
    char			*hash_key;
    int				deletebyschema = 0;

    if (TclQddb_TupleHashTableInit == 0)
	return TCL_OK;
    if (strncmp("qddb_schema", Token, 11) == 0)
	deletebyschema = 1;
    if (strcmp(Token, "all") == 0 || deletebyschema == 1) {
	hash_ptr = Tcl_FirstHashEntry(&TclQddb_TupleHashTable, &hash_search);
	while (hash_ptr != NULL) {
	    if (deletebyschema == 1) {
		TclQddb_Tuple	*tuple;

		tuple = (TclQddb_Tuple *)Tcl_GetHashValue(hash_ptr);
		if (strcmp(tuple->schema_name, Token) != 0) {
		    hash_ptr = Tcl_NextHashEntry(&hash_search);
		    continue;
		}
	    }
	    hash_key = Tcl_GetHashKey(&TclQddb_TupleHashTable, hash_ptr);
	    if (hash_key == NULL) {
		Tcl_AppendResult(interp, "TclQddb_DeleteTuple: ", 
				 "Tcl_GetHashKey failed (TCL ERROR)", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_DeleteTupleOne(interp, hash_key) != TCL_OK)
		return TCL_ERROR;
	    hash_ptr = Tcl_NextHashEntry(&hash_search);
	}
	if (deletebyschema == 0) {
	    TclQddb_TupleHashTableInit = 0;
	    Tcl_DeleteHashTable(&TclQddb_TupleHashTable);
	}
    } else if (TclQddb_DeleteTupleOne(interp, Token) != TCL_OK)
	return TCL_ERROR;
    return TCL_OK;
}

void TclQddb_DeleteTupleProc(clientData)
    ClientData			clientData;
{
#if defined(DEBUG_MALLOC)
    fprintf(stderr, "TclQddb_DeleteTupleProc\n");
#endif
    (void)TclQddb_DeleteTuple((Tcl_Interp *)clientData, "all");
}


TclQddb_Tuple *TclQddb_GetTuple(interp, token)
    Tcl_Interp			*interp;
    char			*token;
{
    TclQddb_Tuple		*tuple;
    Tcl_HashEntry		*hash_ptr;

    if (TclQddb_TupleHashTableInit == 0) {
	TclQddb_TupleHashTableInit = 1;
	Tcl_InitHashTable(&TclQddb_TupleHashTable, TCL_STRING_KEYS);
	Tcl_AppendResult(interp, "cannot find tuple \"", token, "\" (TCL error)", NULL);
	return NULL;
    }
    hash_ptr = Tcl_FindHashEntry(&TclQddb_TupleHashTable, token);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot find tuple \"", token, "\" (TCL error)", NULL);
	return NULL;
    }
    tuple = (TclQddb_Tuple *)Tcl_GetHashValue(hash_ptr);
    if (tuple->datatree == NULL)
	TclQddb_RefreshTuple(interp, token);
    return tuple;
}

static int TclQddb_RemoveTuple(interp, token)
    Tcl_Interp			*interp;
    char			*token;
{
    Schema			*schema;
    TclQddb_Tuple		*tuple;

    tuple = TclQddb_GetTuple(interp, token);
    if (tuple == NULL) {
	Tcl_AppendResult(interp, "cannot find tuple \"", token, "\"", NULL);
	return TCL_ERROR;
    }
    schema = TclQddb_GetSchema(tuple->schema_name);
    if (schema == NULL) {
	Tcl_AppendResult(interp, "cannot find schema \"", tuple->schema_name, 
			 "\" for tuple ", "\"", token, "\"", NULL);
	return TCL_ERROR;
    }
    DeleteEntryByKeyList(schema, &tuple->keylist);
    TclQddb_DeleteTuple(interp, token);
    return TCL_OK;
}

static TclQddb_Tuple *TclQddb_GrabTuple(interp, token)
    Tcl_Interp			*interp;
    char			*token;
{
    TclQddb_Tuple		*tuple;
    Tcl_HashEntry		*hash_ptr;

    if (TclQddb_TupleHashTableInit == 0) {
	TclQddb_TupleHashTableInit = 1;
	Tcl_InitHashTable(&TclQddb_TupleHashTable, TCL_STRING_KEYS);
	Tcl_AppendResult(interp, "cannot find tuple \"", token, "\" (TCL error)", NULL);
	return NULL;
    }
    hash_ptr = Tcl_FindHashEntry(&TclQddb_TupleHashTable, token);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot find tuple \"", token, "\" (TCL error)", NULL);
	return NULL;
    }
    tuple = (TclQddb_Tuple *)Tcl_GetHashValue(hash_ptr);
    return tuple;
}


static int TclQddb_RefreshTuple(interp, token)
    Tcl_Interp			*interp;
    char			*token;
{
    Schema			*schema;
    DataTree			**datatree;
    TclQddb_Tuple		*tuple;
    Entry			external = NULL;
    unsigned int		entry_number;
    int				bad_tuple;

    tuple = TclQddb_GrabTuple(interp, token);
    if (tuple == NULL) {
	Tcl_AppendResult(interp, "cannot find tuple \"", token, "\"", NULL);
	return TCL_ERROR;
    }
    if (QDDB_KEYLIST_TYPE(&(tuple->keylist)) == NEWENTRY)
	return TCL_OK;
    schema = TclQddb_GetSchema(tuple->schema_name);
    if (schema == NULL) {
	Tcl_AppendResult(interp, "cannot find schema \"", tuple->schema_name, 
			 "\" for tuple ", "\"", token, "\"", NULL);
	return TCL_ERROR;
    }
    if (tuple->fd != -1) {
	bad_tuple = Qddb_ReadEntryByKeyList2(tuple->fd, schema->RelationName, &external, 
					     &tuple->keylist, False);
	Qddb_ReducedAttrToFullAttr(schema, external);
    } else {
	bad_tuple = Qddb_ReadEntryByKeyList(schema->database_fd, schema->RelationName, 
					    &external, &tuple->keylist, True);
	Qddb_ReducedAttrToFullAttr(schema, external);
    }
    if (bad_tuple == -1) {
	/* Entry has been deleted or changed */
	TclQddb_UnlockTuple(interp, token);
	if (tuple->keylist.Type != ADDITION) {
	    if (external == NULL) {
		TclQddb_DeleteTuple(interp, token);
		return TCL_OK;
	    }
	    entry_number = GetEntryNumber(external);
	    tuple->keylist.Start = 0;
	    tuple->keylist.Length = 0;
	    tuple->keylist.Type = CHANGE;
	    tuple->keylist.Number = entry_number;
	}
	if (Qddb_ReadEntryByKeyList(schema->database_fd, schema->RelationName, 
				    &external, &tuple->keylist, True) == -1) {
	    /* Tuple deleted */
	    TclQddb_DeleteTuple(interp, token);
	    return TCL_OK;
	}
	Qddb_ReducedAttrToFullAttr(schema, external);
    }
    if (TclQddb_ViewLockTuple(interp, token) != TCL_OK) {
	return TCL_ERROR;
    }
    datatree = Qddb_Convert(schema, QDDB_ENTRYTYPE_EXTERNAL, external, QDDB_ENTRYTYPE_DATATREE);
    Qddb_Free(QDDB_TYPE_ENTRY, external);
    if (datatree == NULL) {
	Tcl_AppendResult(interp, "Cannot convert External tuple to DataTree\n", NULL);
	return TCL_ERROR;
    }
    if (tuple->datatree != NULL)
	Qddb_Free(QDDB_TYPE_DATATREE, tuple->datatree);
    tuple->datatree = datatree;
    if (TclQddb_ViewUnlockTuple(interp, token) != TCL_OK) {
	return TCL_ERROR;
    }
    return TCL_OK;
}


static int TclQddb_ConvertTuple(interp, type, token)
    Tcl_Interp			*interp;
    char			*type;
    char			*token;
{
    TclQddb_Tuple		*tuple;
    Schema			*schema;

    tuple = TclQddb_GetTuple(interp, token);
    if (tuple == NULL) {
	Tcl_AppendResult(interp, "cannot find tuple \"", token, "\"", NULL);
	return TCL_ERROR;
    }
    if ((schema = TclQddb_GetSchema(tuple->schema_name)) == NULL) {
	Tcl_AppendResult(interp, "bad schema \"", tuple->schema_name, "\"", NULL);
	return TCL_ERROR;
    }    
    if (strcmp(type, "external") == 0) {
	Entry			entry, idx;
	int			i;
	char			*retbuf;

	entry = Qddb_Convert(schema, QDDB_ENTRYTYPE_DATATREE, tuple->datatree, QDDB_ENTRYTYPE_EXTERNAL);
	if (entry == NULL) {
	    Tcl_AppendResult(interp, "Cannot convert tuple to External type", NULL);
	    return TCL_ERROR;
	}
	for (i = 0, idx = entry; *idx != NULL; idx++, i++);
	retbuf = Tcl_Merge(i, entry);
	Qddb_Free(QDDB_TYPE_ENTRY, entry);
	Tcl_SetResult(interp, retbuf, TCL_DYNAMIC);
    } else if (strcmp(type, "tclexternal") == 0) {
	char			*entry;

	entry = Qddb_Convert(schema, QDDB_ENTRYTYPE_DATATREE, tuple->datatree, QDDB_ENTRYTYPE_TCLEXTERNAL);
	if (entry == NULL) {
	    Tcl_AppendResult(interp, "Cannot convert tuple to TCLExternal type", NULL);
	    return TCL_ERROR;
	}
	Tcl_SetResult(interp, entry, TCL_DYNAMIC);
    } else if (strcmp(type, "tclvalue") == 0) {
	char			*tclvalue;

	tclvalue = Qddb_Convert(schema, QDDB_ENTRYTYPE_DATATREE, tuple->datatree, QDDB_ENTRYTYPE_TCLVALUE);
	if (tclvalue == NULL) {
	    Tcl_AppendResult(interp, "Cannot convert tuple to TCLvalue type", NULL);
	    return TCL_ERROR;
	}
	Tcl_AppendResult(interp, "qddb_tuple get tclvalue: not yet implemented", NULL);
	return TCL_ERROR;
    } else if (strcmp(type, "readable") == 0) {
	char			*readable;

	readable = Qddb_Convert(schema, QDDB_ENTRYTYPE_DATATREE, tuple->datatree, QDDB_ENTRYTYPE_READABLE);
	if (readable == NULL) {
	    Tcl_AppendResult(interp, "Cannot convert tuple to Readable type", NULL);
	    return TCL_ERROR;
	}
	Tcl_SetResult(interp, readable, TCL_DYNAMIC);
    } else {
	Tcl_AppendResult(interp, "bad conversion type \"", type, "\"",
			 "\nshould be external, tclexternal, tclvalue, or readable", NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

static int TclQddb_PutTuple(interp, type, schema_desc, data)
    Tcl_Interp			*interp;
    char			*type;
    char			*schema_desc;
    char			*data;
{
    Schema			*schema;
    DataTree			**tree;
    KeyList			keylist;
    char			*token;
    char			**argv;
    int				argc;

    if ((schema = TclQddb_GetSchema(schema_desc)) == NULL) {
	Tcl_AppendResult(interp, "bad schema \"", schema_desc, "\"", NULL);
	return TCL_ERROR;
    }
    if (strcmp(type, "external") == 0) {
	int			i;
	int			redone = False;
	char			**nargv;

	if (Tcl_SplitList(interp, data, &argc, &argv) != TCL_OK)
	    return TCL_ERROR;
	/* For "Entry" types, we first check the first
	 * element to see if it is a '%0 V'.   If it isn't,
	 * then we add a '%0 V 1' line.
	 */
	if (strncmp("%0 V ", argv[0], 5) != 0) {
	    nargv = (char **)Malloc(sizeof(char *)*(argc+2));
	    nargv[0] = Malloc(7);
	    strcpy(nargv[0], "%0 V 1");
	    for (i = 1; i <= argc; i++)
		nargv[i] = argv[i-1];
	    nargv[i] = NULL;
	    redone = True;
	} else
	    nargv = argv;
	tree = Qddb_Convert(schema, QDDB_ENTRYTYPE_EXTERNAL, nargv, QDDB_ENTRYTYPE_DATATREE);
	if (tree == NULL) {
	    Tcl_AppendResult(interp, "Cannot convert tuple from External type", NULL);
	    return TCL_ERROR;
	}
	Free(argv);
	if (redone == True) {
	    Free(nargv[0]);
	    Free(nargv);
	}
	keylist.Type = NEWENTRY;
	keylist.Start = 0;
	keylist.Length = 0;
	keylist.Number = 0;
	keylist.Instance = NULL;
	keylist.Attribute = 0;
	if ((token = TclQddb_NewTuple(interp, schema_desc, tree, &keylist)) == NULL) {
	    Qddb_Free(QDDB_TYPE_DATATREE, tree);
	    return TCL_ERROR;
	}
	Tcl_SetResult(interp, token, TCL_DYNAMIC);
    } else if (strcmp(type, "tclexternal") == 0) {
	char			*entry;
	int			i;
	int			redone = False;
	char			**nargv;

	if (Tcl_SplitList(interp, data, &argc, &argv) != TCL_OK)
	    return TCL_ERROR;
	/* For "TCLExternal" types, we first check the 1st & 2nd
	 * elements to see if it is a '$NUMBER$ X'.   If it isn't,
	 * then we add it before conversion.
	 */
	if (strncmp("$NUMBER$", argv[0], 8) != 0) {
	    nargv = (char **)Malloc(sizeof(char *)*(argc+3));
	    nargv[0] = Malloc(9);
	    strcpy(nargv[0], "$NUMBER$");
	    nargv[1] = Malloc(2);
	    nargv[1][0] = '1'; nargv[1][1] = '\0';
	    for (i = 2; i <= argc+1; i++)
		nargv[i] = argv[i-1];
	    nargv[i] = NULL;
	    argc = i;
	    redone = True;
	} else
	    nargv = argv;
	Qddb_InitBuffer();
	for (i = 0; i < argc; i++) {
	    if (i % 2 == 1)
		Qddb_ConcatBuffer(" ");
	    Qddb_ConcatBuffer(nargv[i]);
	    if (i % 2 == 1)
		Qddb_ConcatBuffer("\n");
	}
	entry = Qddb_GetBuffer();
	tree = Qddb_Convert(schema, QDDB_ENTRYTYPE_TCLEXTERNAL, entry, QDDB_ENTRYTYPE_DATATREE);
	Free(entry);
	if (tree == NULL) {
	    Tcl_AppendResult(interp, "Cannot convert tuple from TCLExternal type", NULL);
	    return TCL_ERROR;
	}
	Free(argv);
	if (redone == True) {
	    Free(nargv[0]);
	    Free(nargv[1]);
	    Free(nargv);
	}
	keylist.Type = NEWENTRY;
	keylist.Start = 0;
	keylist.Length = 0;
	keylist.Number = 0;
	keylist.Instance = NULL;
	keylist.Attribute = 0;
	if ((token = TclQddb_NewTuple(interp, schema_desc, tree, &keylist)) == NULL) {
	    Qddb_Free(QDDB_TYPE_DATATREE, tree);
	    return TCL_ERROR;
	}
	Tcl_SetResult(interp, token, TCL_DYNAMIC);
    } else if (strcmp(type, "tclvalue") == 0) {
#if defined(notyet)
	char			*tclvalue;
#endif
	if (Tcl_SplitList(interp, data, &argc, &argv) != TCL_OK)
	    return TCL_ERROR;
#if defined(notyet)
	/* Not yet implemented */
	tclvalue = Qddb_Convert(schema, QDDB_ENTRYTYPE_DATATREE, tuple->datatree, QDDB_ENTRYTYPE_TCLVALUE);
	if (tclvalue == NULL) {
	    Tcl_AppendResult(interp, "Cannot convert tuple from TCLvalue type", NULL);
	    return TCL_ERROR;
	}
#endif
	Tcl_AppendResult(interp, "qddb_tuple get tclvalue: not yet implemented", NULL);
	return TCL_ERROR;
    } else if (strcmp(type, "readable") == 0) {
	char			*readable;
	Boolean			free_readable = False;

	/* For readable format, we check the beginning of the element to look
	 * for a '$NUMBER$ = "X"; .   If it doesn't exist, we add it.
	 */
	if (strncmp(data, "$NUMBER$", 8) != 0) {
	    readable = data;
	    if (isspace(readable[0])) {
		while (*readable != '\0' && isspace(*readable))
		    readable++;
	    }
	    if (strncmp(readable, "$NUMBER$", 8) != 0) {
		Qddb_InitBuffer();
		Qddb_ConcatBuffer("$NUMBER$ = \"1\";\n");
		Qddb_ConcatBuffer(readable);
		readable = Qddb_GetBuffer();
		free_readable = True;
	    }
	} else
	    readable = data;
	tree = Qddb_Convert(schema, QDDB_ENTRYTYPE_READABLE, readable, QDDB_ENTRYTYPE_DATATREE);
	if (free_readable == True)
	    Free(readable);
	if (tree == NULL) {
	    Tcl_AppendResult(interp, "Bad input: ", qddb_errmsg, NULL);
	    return TCL_ERROR;
	}
	keylist.Type = NEWENTRY;
	keylist.Start = 0;
	keylist.Length = 0;
	keylist.Number = 0;
	keylist.Instance = NULL;
	keylist.Attribute = 0;
	if ((token = TclQddb_NewTuple(interp, schema_desc, tree, &keylist)) == NULL) {
	    Qddb_Free(QDDB_TYPE_DATATREE, tree);
	    return TCL_ERROR;
	}
	Tcl_SetResult(interp, token, TCL_DYNAMIC);
    } else {
	Tcl_AppendResult(interp, "bad conversion type \"", type, "\"",
			 "\nshould be external, tclexternal, tclvalue, or readable", NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

static int TclQddb_IsEmptyTuple(interp, token)
    Tcl_Interp			*interp;
    char			*token;
{
    TclQddb_Tuple		*tuple;

    tuple = TclQddb_GetTuple(interp, token);
    if (tuple == NULL) {
	Tcl_AppendResult(interp, "cannot find tuple \"", token, "\" (Qddb Error)", NULL);
	return TCL_ERROR;	
    }
    if (TclQddb_IsEmptyDataTree(tuple->datatree) == True)
	Tcl_SetResult(interp, "1", TCL_STATIC);
    else
	Tcl_SetResult(interp, "0", TCL_STATIC);
    return TCL_OK;
}

int TclQddb_FlushTuple(interp, Token)
    Tcl_Interp			*interp;
    char			*Token;
{
    Tcl_HashEntry		*hash_ptr;
    Tcl_HashSearch		hash_search;
    TclQddb_Tuple		*tuple;
    int				flushbyschema = 0;

    if (TclQddb_TupleHashTableInit == 0)
	return TCL_OK;
    if (strncmp("qddb_schema", Token, 11) == 0)
	flushbyschema = 1;
    if (strcmp(Token, "all") == 0 || flushbyschema == 1) {
	hash_ptr = Tcl_FirstHashEntry(&TclQddb_TupleHashTable, &hash_search);
	while (hash_ptr != NULL) {
	    tuple = (TclQddb_Tuple *)Tcl_GetHashValue(hash_ptr);
	    if (flushbyschema == 1 && strcmp(tuple->schema_name, Token) != 0) {
		hash_ptr = Tcl_NextHashEntry(&hash_search);
		continue;
	    }
	    /* flush it out */
	    if (tuple->datatree != NULL) {
		Qddb_Free(QDDB_TYPE_DATATREE, tuple->datatree);
		tuple->datatree = NULL;
	    }
	    hash_ptr = Tcl_NextHashEntry(&hash_search);
	}
    } else {
	hash_ptr = Tcl_FindHashEntry(&TclQddb_TupleHashTable, Token);
	if (hash_ptr != NULL) {
	    tuple = (TclQddb_Tuple *)Tcl_GetHashValue(hash_ptr);
	    if (tuple->datatree != NULL) {
		Qddb_Free(QDDB_TYPE_DATATREE, tuple->datatree);
		tuple->datatree = NULL;
	    }
	} else {
	    Tcl_AppendResult(interp, "qddb_tuple flush: cannot find tuple \"", Token, "\"", NULL);
	}
    }
    return TCL_OK;
}
