
/* TclQddb_RowList.c - TCL interface routines for Qddb KeyLists.
 *
 * Copyright (C) 1994 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */

#include "Qddb.h"
#include "tclQddb.h"

/* EXPORTED:
 *	TclQddb_RowsProc
 */

static Tcl_HashTable 	TclQddb_RowsHashTable;
static int		TclQddb_RowsHashTableInit = 0;
static unsigned int	TclQddb_RowsNextNumber = 0;

static int TclQddb_SelectRows _ANSI_ARGS_((Tcl_Interp *, TclQddb_KeyListHeader *, int, char **));

static int TclQddb_BuildNewRows _ANSI_ARGS_((Tcl_Interp *, char *, char *, TclQddb_EntryRows *, char *, \
					     int, char **, size_t *,
					     char ***, char ***, Boolean, Boolean));
static int TclQddb_SelectRowsBuildList _ANSI_ARGS_((Tcl_Interp *, char *, Schema *, KeyList *, \
						    char *, int, char **, \
						    int, char **, int, char **, \
						    int, char **, size_t *, \
						    Boolean, Boolean, Boolean));
static char *TclQddb_PrintRow _ANSI_ARGS_((Tcl_Interp *, Schema *, TclQddb_Tuple *, TclQddb_Rows *, \
					   char *, int, char **, size_t *));
static char *TclQddb_RowAsElements _ANSI_ARGS_((Tcl_Interp *, Schema *, TclQddb_Tuple *, TclQddb_Rows *, \
						char *, int, char **, size_t *));
static int TclQddb_TranslateRows _ANSI_ARGS_((Tcl_Interp *, char *));
static int TclQddb_AllRows _ANSI_ARGS_((Tcl_Interp *, int, char **));
static void TclQddb_FreeRow _ANSI_ARGS_((TclQddb_Rows *));
static TclQddb_Rows **TclQddb_BuildAllRows _ANSI_ARGS_((Tcl_Interp *, Schema *, TclQddb_Tuple *, \
							char **, int, char *, char *));
static int TclQddb_FormatRows _ANSI_ARGS_((Tcl_Interp *, int, char **));
static TclQddb_Rows *TclQddb_RowCopy _ANSI_ARGS_((TclQddb_Rows *, TclQddb_Rows **));
static TclQddb_Rows *TclQddb_RowConcat _ANSI_ARGS_((TclQddb_Rows *, TclQddb_Rows *));
static TclQddb_Rows **TclQddb_RowCartesianProduct _ANSI_ARGS_((TclQddb_Rows **, TclQddb_Rows **));
static TclQddb_Rows **TclQddb_RowUnion _ANSI_ARGS_((TclQddb_Rows **, TclQddb_Rows **));
static TclQddb_Rows **TclQddb_RowInst _ANSI_ARGS_((Schema *, DataTree **, char **, int, char *, char *));
static TclQddb_Rows **TclQddb_RowAttr _ANSI_ARGS_((Schema *, DataTree *, char **, int, char *, char *));
#if defined(notdef)
static int TclQddb_DeleteDuplicateRows _ANSI_ARGS_((TclQddb_Rows **, int, char **));
#endif
static char *TclQddb_AddRow _ANSI_ARGS_((Tcl_Interp *, char *, char *, TclQddb_Rows *));
static int TclQddb_FormatAndBuildRows _ANSI_ARGS_((Tcl_Interp *, TclQddb_Tuple *, char *, TclQddb_Rows **, \
						   char *, int, char **, size_t *));

static int TclQddb_GetVal _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_DoSortRows _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_InterestingAttr _ANSI_ARGS_((char *, char **, int));

/* TclQddb_RowsProc -- manipulate rows.
 *
 * <row_desc list>              <- qddb_rows all ?-instance inst? ?<options>? 
 *                                      ?-sortby <attr list>? ?-ascending on|off? <tuple desc>
 *	{ <row_desc> "row value" }
 *	{ <row_desc> "row value" }
 *	...
 * <row_desc list> <- qddb_rows select ?-query off? ?<options>? <keylist desc>
 *	{ <row_desc> "row value" }
 *	{ <row_desc> "row value" }
 *	...
 * <query(1)-style list> 	<- qddb_rows select -query on ?-sep string? ?-norows on|off? \
 *					?<options>? <keylist desc>
 *	{ { row value } { Start Length Number Type } { attr,inst attr,inst ... } }
 *	{ { row value } { Start Length Number Type } { attr,inst attr,inst ... } }
 *	...
 * NOTE: -norows is ignored unless -query is on.
 *
 * { attr,inst attr,inst }      <- qddb_rows get <row desc>
 * <values>                     <- qddb_rows getval <attr_list> <row_desc>
 * { <row_desc> <row_desc> }    <- qddb_rows sort ?-ascending on|off? <attr_list> <row_desc list>
 *                                 qddb_rows delete <row_desc>|all
 *
 * inst -- <attr>,<inst> pair
 * <options>:
 *	-deldup_rows on|off (default: off)	- delete duplicate rows
 *	-attrs <attribute list>			- attributes used to generate keylist
 *	-print <print list>			- attributes printed as 'row value'
 *	   { attr:width attr:width ... }
 */
int TclQddb_RowsProc(clientData, interp, argc, argv)
    ClientData			clientData;
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    TclQddb_KeyListHeader	*keylist_header = NULL;
    TclQddb_RowHeader		*rows;
    
    if (argc < 3) {
	Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    if (TclQddb_RowsHashTableInit == 0) {
	TclQddb_RowsHashTableInit = 1;
	Tcl_InitHashTable(&TclQddb_RowsHashTable, TCL_STRING_KEYS);
    }
    switch (argv[1][0]) {
    case 'a':
	if (strcmp(argv[1], "all") != 0) {
	    Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	    return TCL_ERROR;
	}
	if (argc < 3) {
	    Tcl_AppendResult(interp, argv[0], ": wrong # args for all", NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_AllRows(interp, argc, argv) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 's':
	if (strcmp(argv[1], "select") != 0) {
	    if (strcmp(argv[1], "sort") != 0) {
		Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
		return TCL_ERROR;
	    }
	    if (argc < 4) {
		Tcl_AppendResult(interp, argv[0], ": wrong # args for sort", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_DoSortRows(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else {
	    if (argc < 3) {
		Tcl_AppendResult(interp, argv[0], ": wrong # args for select", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_GetKeyList(interp, argv[argc-1], &keylist_header) != TCL_OK) {
		Tcl_AppendResult(interp, argv[0], ": bad keylist \"", argv[argc-1], "\"", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_SelectRows(interp, keylist_header, argc, argv) != TCL_OK)
		return TCL_ERROR;
	}
	break;
    case 'g':
	if (strcmp(argv[1], "get") != 0) {
	    if (strcmp(argv[1], "getval") != 0) {
		Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
		return TCL_ERROR;
	    }
	    if (argc != 4) {
		Tcl_AppendResult(interp, argv[0], ": wrong # args for getval", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_GetVal(interp, argc, argv) != TCL_OK) 
		return TCL_ERROR;
	} else {
	    if (argc != 3) {
		Tcl_AppendResult(interp, argv[0], ": wrong # args for get", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_TranslateRows(interp, argv[2]) != TCL_OK)
		return TCL_ERROR;
	}
	break;
    case 'd':
	if (strcmp(argv[1], "delete") != 0) {
	    Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	    return TCL_ERROR;
	}
	if (argc != 3) {
	    Tcl_AppendResult(interp, argv[0], ": wrong # args for delete", NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_DeleteRows(interp, argv[2]) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 't':
	if (strcmp(argv[1], "tuplename") != 0) {
	    Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	    return TCL_ERROR;
	}
	if (argc != 3) {
	    Tcl_AppendResult(interp, argv[0], ": wrong # args for tuplename", NULL);
	    return TCL_ERROR;
	}
	if ((rows = TclQddb_GetRows(interp, argv[2])) == NULL)
	    return TCL_ERROR;
	Tcl_SetResult(interp, rows->tuple_name, TCL_VOLATILE);
	break;
    case 'f':
	if (strcmp(argv[1], "format") != 0) {
	    Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	    return TCL_ERROR;
	}
	if (argc != 4 && argc != 6) {
	    Tcl_AppendResult(interp, argv[0], ": wrong # args for format", NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_FormatRows(interp, argc, argv) != TCL_OK)
	    return TCL_ERROR;
	break;
    default:
	Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

static int TclQddb_AllRows(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			**argv;
{
    TclQddb_Tuple		*tuple;
    Schema			*schema;
    TclQddb_Rows		**row_array;
    int				deldup_rows = 0;
    int				attr_argc = 0, print_argc = 0, sort_argc = 0, ascending_argc = 0, i, j;
    size_t			*print_widths = NULL;
    int				getint;
    char			**attr_argv = NULL, **print_argv = NULL, **sort_argv = NULL;
    char			**ascending_argv = NULL;
    char			*sepstring = ":";
    char			*check_attr = NULL, *check_inst = NULL;

    if ((tuple = TclQddb_GetTuple(interp, argv[argc-1])) == NULL) {
	Tcl_AppendResult(interp, argv[0], ": bad tuple \"", argv[argc-1], "\"", NULL);
	return TCL_ERROR;
    }
    if ((schema = TclQddb_GetSchema(tuple->schema_name)) == NULL) {
	Tcl_AppendResult(interp, argv[0], ": bad schema \"", tuple->schema_name, "\"", NULL);
	return TCL_ERROR;
    }
    for (i = 2; i < argc-2; i++) {
	if (strcmp(argv[i], "-attrs") == 0) {
	    if (attr_argc != 0) {
		Free(attr_argv);
		attr_argv = NULL;
	    }
	    if (Tcl_SplitList(interp, argv[++i], &attr_argc, &attr_argv) != TCL_OK)
		goto error_ret;
	} else if (strcmp(argv[i], "-sortby") == 0) {
	    if (sort_argc != 0) {
		Free(sort_argv);
		sort_argv = NULL;
	    }
	    if (Tcl_SplitList(interp, argv[++i], &sort_argc, &sort_argv) != TCL_OK)
		goto error_ret;
	} else if (strcmp(argv[i], "-ascending") == 0) {
	    if (ascending_argc != 0) {
		Free(ascending_argv);
		ascending_argv = NULL;
	    }
	    if (Tcl_SplitList(interp, argv[++i], &ascending_argc, &ascending_argv) != TCL_OK)
		goto error_ret;
	} else if (strcmp(argv[i], "-print") == 0) {
	    if (print_argc != 0) {
		Free(print_argv);
		print_argv = NULL;
	    }
	    if (Tcl_SplitList(interp, argv[++i], &print_argc, &print_argv) != TCL_OK)
		goto error_ret;
	    print_widths = (size_t *)Malloc(sizeof(size_t)*print_argc);
	    for (j = 0; j < print_argc; j++) {
		char		*c_buf, *n_buf;

		c_buf = strtok(print_argv[j], ":");
		n_buf = strtok(NULL, ":");
		if (n_buf == NULL)
		    print_widths[j] = 0;
		else if (Tcl_GetInt(interp, n_buf, &getint) != TCL_OK)
		    goto error_ret;
		else
		    print_widths[j] = getint;
	    }
	} else if (strcmp(argv[i], "-instance") == 0) {
	    check_attr = strtok(argv[++i], ",");
	    if (check_attr == NULL || check_attr[0] == '\0' || 
		Qddb_ConvertAttributeNameToNumber(schema, check_attr) == -1) {
		Tcl_AppendResult(interp, "bad instance \"", argv[i], "\"", NULL);
		goto error_ret;
	    }
	    check_inst = strtok(NULL, ",");
	    if (check_inst == NULL)
		check_inst = "";
	} else if (strcmp(argv[i], "-sep") == 0) {
	    sepstring = argv[++i];
	} else 	if (strcmp(argv[i], "-deldup_rows") == 0) {
	    if (Tcl_GetBoolean(interp, argv[++i], &deldup_rows) != TCL_OK)
		goto error_ret;
	} else {
	    Tcl_AppendResult(interp, "bad option \"", argv[i], "\": must be -attrs or -print", NULL);
	    goto error_ret;
	}
    }
    if ((row_array = TclQddb_BuildAllRows(interp, schema, tuple, attr_argv, attr_argc,
					  check_attr, check_inst)) == NULL)
	goto error_ret;
    /* Now we have a complete set of existing rows starting at (if specified)
     * check_attr,check_inst for this tuple.   We have to prune the list of 
     * rows based upon the attrs specified in attr_argv by deleting duplicates.
     * Generally, if you want a unique list of rows, then you specify
     * the same attrs in attr_argv as in print_argv.   If you want all rows, then
     * you specify nothing in attr_argv.   You CAN specify different values for
     * attr_argv & print_argv if you want to prune duplicate rows based on
     * attributes different from those being printed.
     */
    if (sort_argc > 0)
	TclQddb_SortRowsSameTuple(interp, tuple, row_array, sort_argc, sort_argv, 
				  ascending_argc, ascending_argv);
    if (TclQddb_FormatAndBuildRows(interp, tuple, argv[argc-1], row_array, sepstring,
				   print_argc, print_argv, print_widths) != TCL_OK)
	goto error_ret;
    Free(row_array);
    if (print_argv != NULL)
	Free(print_argv);
    if (print_widths != NULL)
	Free(print_widths);
    if (attr_argv != NULL)
	Free(attr_argv);
    if (sort_argv != NULL)
	Free(sort_argv);
    if (ascending_argv != NULL)
	Free(ascending_argv);
    return TCL_OK;
error_ret:
    if (print_argv != NULL)
	Free(print_argv);
    if (print_widths != NULL)
	Free(print_widths);
    if (attr_argv != NULL)
	Free(attr_argv);
    if (sort_argv != NULL)
	Free(sort_argv);
    if (ascending_argv != NULL)
	Free(ascending_argv);
    return TCL_ERROR;    
}

/* TclQddb_FormatAndBuildRows --
 *	Takes the rows, prunes them via argv, then keeps the individual rows
 * for later use.   The rows (rows[i]) should not be used again.
 */
static int TclQddb_FormatAndBuildRows(interp, tuple, tuple_name, rows, sep, argc, argv, widths)
    Tcl_Interp			*interp;
    TclQddb_Tuple		*tuple;
    char			*tuple_name;
    TclQddb_Rows		**rows;
    char			*sep;
    int				argc;
    char			**argv;
    size_t			*widths;
{
    TclQddb_Rows		*ptr;
    char			*buf, *tmpbuf, *cargv[2], **retval = NULL;
    int				i, num_retval = 0;

    while (*rows != NULL) {
	Qddb_InitBuffer();
	for (i = 0; i < argc; i++) {
	    ptr = *rows;
	    while (ptr != NULL) {
		if (strcmp(argv[i], ptr->attr_name) == 0) {
		    buf = TclQddb_StringRef(tuple, ptr);
		    if (buf == NULL) {
			Tcl_AppendResult(interp, "failed for format string for row node \"",
					 ptr->attr_name, ",", ptr->instance, "\"", NULL);
			return TCL_ERROR;
		    }
		    if (i != 0)
			Qddb_ConcatBuffer(sep);
		    tmpbuf = Qddb_FieldString(buf, widths[i]);
		    Qddb_ConcatBuffer(tmpbuf);
		    Free(tmpbuf);
		    Free(buf);
		}
		ptr = ptr->next;
	    }
	}
	cargv[0] = TclQddb_AddRow(interp, tuple->schema_name, tuple_name, *rows);
	cargv[1] = Qddb_GetBuffer();
	tmpbuf = Tcl_Merge(2, cargv);
	Free(cargv[0]);
	Free(cargv[1]);
	retval = (char **)Realloc(retval, sizeof(char *)*(num_retval+1));
	retval[num_retval++] = tmpbuf;
	rows++;
    }
    tmpbuf = Tcl_Merge(num_retval, retval);
    for (i = 0; i < num_retval; i++)
	Free(retval[i]);
    if (retval != NULL)
	Free(retval);
    Tcl_SetResult(interp, tmpbuf, TCL_DYNAMIC);
    return TCL_OK;
}

static TclQddb_Rows *TclQddb_RowCopy(row, lastnode)
    TclQddb_Rows		*row, **lastnode;
{
    TclQddb_Rows		*retval = NULL, *lastretval = NULL;

    while (row != NULL) {
	if (retval == NULL) {
	    retval = lastretval = (TclQddb_Rows *)Malloc(sizeof(TclQddb_Rows));
	    lastretval->next = NULL;
	} else {
	    lastretval->next = (TclQddb_Rows *)Malloc(sizeof(TclQddb_Rows));
	    lastretval = lastretval->next;
	    lastretval->next = NULL;
	}
	lastretval->attr_name = Malloc(strlen(row->attr_name)+1);
	strcpy(lastretval->attr_name, row->attr_name);
	lastretval->instance = Malloc(strlen(row->instance)+1);
	strcpy(lastretval->instance, row->instance);
	row = row->next;
    }
    if (lastnode != NULL)
	*lastnode = lastretval;
    return retval;
}

static TclQddb_Rows *TclQddb_RowConcat(row1, row2)
    TclQddb_Rows		*row1, *row2;
{
    TclQddb_Rows		*lastnode;

    row1 = TclQddb_RowCopy(row1, &lastnode);
    row2 = TclQddb_RowCopy(row2, NULL);
    lastnode->next = row2;
    return row1;
}

static TclQddb_Rows **TclQddb_RowCartesianProduct(row1, row2)
    TclQddb_Rows		**row1, **row2;
{
    TclQddb_Rows		**ptr1, **ptr2, **retval, **place;
    int				row1_len, row2_len;

    if (row1 == NULL && row2 == NULL) {
	return NULL;
    }
    if (row1 == NULL) {
	for (ptr2 = row2; *ptr2 != NULL; ptr2++) {
	    TclQddb_FreeRow(*ptr2);
	}
	Free(row2);
	return NULL;
    }
    if (row2 == NULL) {
	for (ptr1 = row1; *ptr1 != NULL; ptr1++) {
	    TclQddb_FreeRow(*ptr1);
	}
	Free(row1);
	return NULL;
    }
    for (ptr1 = row1, row1_len = 0; *ptr1 != NULL; ptr1++, row1_len++);
    for (ptr2 = row2, row2_len = 0; *ptr2 != NULL; ptr2++, row2_len++);
    retval = (TclQddb_Rows **)Malloc(sizeof(TclQddb_Rows *)*(row1_len*row2_len+1));
    place = retval;
    for (ptr1 = row1; *ptr1 != NULL;  ptr1++) {
	for (ptr2 = row2; *ptr2 != NULL; ptr2++) {
	    *place++ = TclQddb_RowConcat(*ptr1, *ptr2);
	}
	TclQddb_FreeRow(*ptr1);
    }
    *place = NULL;
    for (ptr2 = row2; *ptr2 != NULL; ptr2++)
	TclQddb_FreeRow(*ptr2);
    Free(row1);
    Free(row2);
    return retval;
}

/* TclQddb_RowUnion --
 * 	Take the union of two rows.
 */
static TclQddb_Rows **TclQddb_RowUnion(row1, row2)
    TclQddb_Rows		**row1, **row2;
{
    TclQddb_Rows		**ptr, **retval, **place;
    int				row1_len, row2_len;

    if (row1 == NULL)
	return row2;
    if (row2 == NULL)
	return row1;
    for (ptr = row1, row1_len = 0; *ptr != NULL; ptr++, row1_len++);
    for (ptr = row2, row2_len = 0; *ptr != NULL; ptr++, row2_len++);
    retval = (TclQddb_Rows **)Malloc(sizeof(TclQddb_Rows *)*(row1_len+row2_len+1));
    place = retval;
    for (ptr = row1; *ptr != NULL; ptr++)
	*place++ = *ptr;
    for (ptr = row2; *ptr != NULL; ptr++)
	*place++ = *ptr;
    *place = NULL;
    Free(row1);
    Free(row2);
    return retval;
}

static TclQddb_Rows **TclQddb_RowInst(schema, attrs, attr_argv, attr_argc, name, inst_name)
    Schema			*schema;
    DataTree			**attrs;
    char			**attr_argv;
    int				attr_argc;
    char			*name, *inst_name;
{
    TclQddb_Rows		**retval = NULL;
    int				i;

    for (i = 0; attrs[i] != NULL; i++) {
	if (retval == NULL) {
	    retval = TclQddb_RowAttr(schema, attrs[i], attr_argv, attr_argc, name, inst_name);
	    if (retval == NULL)
		break;
	} else {
	    retval = TclQddb_RowCartesianProduct(retval, 
						 TclQddb_RowAttr(schema, attrs[i], attr_argv, attr_argc,
								 name, inst_name));
	    if (retval == NULL)
		break;
	}
    }
    return retval;
}

static int TclQddb_InterestingAttr(name, attr_argv, attr_argc)
    char			*name, **attr_argv;
    int				attr_argc;
{
    int				i;

    if (attr_argc == 0)
	return True; /* All nodes are interesting */
    for (i = 0; i < attr_argc; i++)
	if (strcmp(name, attr_argv[i]) == 0) 
	    return True;
    return False;
}

static int TclQddb_InterestingAttrsBelow(name, attr_argv, attr_argc)
    char			*name, **attr_argv;
    int				attr_argc;
{
    int				i;

    if (attr_argc == 0)
	return True; /* All nodes are interesting */
    for (i = 0; i < attr_argc; i++)
	if (strncmp(name, attr_argv[i], strlen(name)) == 0) 
	    return True;
    return False;
}

static TclQddb_Rows **TclQddb_RowAttr(schema, insts, attr_argv, attr_argc, name, inst_name)
    Schema			*schema;
    DataTree			*insts;
    char 			**attr_argv;
    int				attr_argc;
    char			*name, *inst_name;
    
{
    TclQddb_Rows		**retval = NULL;
    int				i, retval_top;
    size_t			namelen, instlen;
    char			*thisname, *thisinstname, *comp;

    /* Compose this level's attribute name */
    comp = insts->datatree_schema->schemanode->Name;
    if (name[0] == '\0') { /* top level */
	thisname = alloca(strlen(comp)+1);
	strcpy(thisname, comp);
    } else {
	namelen = strlen(name);
	thisname = alloca(namelen+strlen(comp)+2);
	strcpy(thisname, name);
	thisname[namelen] = '.';
	strcpy(thisname+namelen+1, comp);
    }
    instlen = strlen(inst_name);
    thisinstname = alloca(instlen+MAXINTEGERLEN+2);
    strcpy(thisinstname, inst_name);
    if (instlen > 0) {
	thisinstname[instlen++] = '.';
    }
    retval_top = 0;
    for (i = 0; insts[i].datatree_type != DATATREE_END; i++) {
	sprintf(thisinstname+instlen, "%d", i+1);
	if (insts[i].datatree_type != DATATREE_CHILDREN) {
	    /* This node is a leaf node; if we are interested in this
	     * attribute, then walk through all instances.  Otherwise,
	     * just return the first non-false one.
	     */
	    if (insts[i].marked == False) {
		continue;
	    }
	    retval = (TclQddb_Rows **)Realloc(retval, sizeof(TclQddb_Rows *)*(retval_top+2));
	    retval[retval_top] = (TclQddb_Rows *)Malloc(sizeof(TclQddb_Rows));
	    retval[retval_top]->attr_name = Malloc(strlen(thisname)+1);
	    strcpy(retval[retval_top]->attr_name, thisname);
	    retval[retval_top]->instance = Malloc(strlen(thisinstname)+1);
	    strcpy(retval[retval_top]->instance, thisinstname);
	    retval[retval_top]->next = NULL;
	    retval[++retval_top] = NULL;
	    if (TclQddb_InterestingAttr(thisname, attr_argv, attr_argc) == False)
		break;
	} else {
	    if (retval == NULL) {
		retval = TclQddb_RowInst(schema, insts[i].datatree_children, attr_argv, attr_argc,
					 thisname, thisinstname);
	    } else {
		retval = TclQddb_RowUnion(retval, TclQddb_RowInst(schema, insts[i].datatree_children, 
								  attr_argv, attr_argc,
								  thisname, thisinstname));
	    }
	    if (TclQddb_InterestingAttrsBelow(thisname, attr_argv, attr_argc) == False && retval != NULL)
		break;
	}
    }
    return retval;
}

static void *TclQddb_FindSpot(tree, attr, inst, do_attr)
    DataTree			**tree;
    char			*attr, *inst;
    int				*do_attr;
{
    DataTree			**dt;
    char			**attr_arr;
    size_t			inst_arr[MAXIMUM_NUMBER_OF_SCHEMA_LEVELS];
    size_t			num_inst, cur;
    int				i;

    if (attr[0] == '\0') {
	*do_attr = 0;
	return (void *)tree;
    }
    attr_arr = TclQddb_SplitAttributeName(attr);
    num_inst = StringToInstance(inst, inst_arr);
    dt = tree;
    for (i = 0; attr_arr[i] != NULL; i++) {
	cur = TclQddb_FindAttribute(dt, attr_arr[i]);
	if (i >= num_inst) {
	    *do_attr = 1;
	    TclQddb_FreeArgs(-1, attr_arr);
	    return (void *)dt[cur];
	}
#if defined(DIAGNOSTIC)
	if (dt[cur][inst_arr[i]-1].datatree_type != DATATREE_CHILDREN) {
	    fprintf(stderr, "bad attr,inst specified (%s,%s)\n", attr, inst);
	    fflush(stderr);
	    abort();
	}
#endif
	dt = dt[cur][inst_arr[i]-1].datatree_children;
    }
    TclQddb_FreeArgs(-1, attr_arr);
    *do_attr = 0;
    return (void *)dt;
}

/* TclQddb_BuildAllRows -- 
 *	Build an array of (TclQddb_Rows *) consisting of all interesting
 * rows based upon check_attr,check_inst.   If check_attr,check_inst is
 * non-existant, then just return a list of ALL rows in the tuple matching
 * the attributes in attr_argv.
 */
static TclQddb_Rows **TclQddb_BuildAllRows(interp, schema, tuple, attr_argv, attr_argc, 
					   check_attr, check_inst)
    Tcl_Interp			*interp;
    Schema			*schema;
    TclQddb_Tuple		*tuple;
    char			**attr_argv;
    int				attr_argc;
    char			*check_attr, *check_inst;
{
    TclQddb_Rows		**rows = NULL;
    void			*dt;
    int				do_attr;
    char			*lastdot;

    if (check_attr != (char *)NULL) {
	dt = TclQddb_FindSpot(tuple->datatree, check_attr, check_inst, &do_attr);
	if (do_attr == 1) {
	    lastdot = rindex(check_attr, '.');
	    if (lastdot == (char *)NULL)
		check_attr = "";
	    else
		*lastdot = '\0';
	    rows = TclQddb_RowAttr(schema, (DataTree *)dt, attr_argv, attr_argc, check_attr, check_inst);
	    if (lastdot != (char *)NULL)
		*lastdot = '.';
	} else {
	    rows = TclQddb_RowInst(schema, (DataTree **)dt, attr_argv, attr_argc, check_attr, check_inst);
	}
    } else {
	rows = TclQddb_RowInst(schema, tuple->datatree, attr_argv, attr_argc, "", "");
    }
    return rows;
}

/* TclQddb_SelectRows -- Build a list of <row_desc,entry_desc> pairs
 * describing the rows from the given keylist.
 */
static int TclQddb_SelectRows(interp, keylist_header, argc, argv)
    Tcl_Interp			*interp;
    TclQddb_KeyListHeader	*keylist_header;
    int				argc;
    char			*argv[];
{
    KeyList			*keylist = keylist_header->keylist;
    Schema			*schema;
    int				query_mode = 0, deldup_rows = 0;
    int				attr_argc = 0, print_argc = 0, sort_argc = 0, ascending_argc = 0;
    int				i, j;
    size_t			*print_nums = NULL;
    int				getint;
    char			**attr_argv = NULL, **print_argv = NULL;
    char			**sort_argv = NULL, **ascending_argv = NULL;
    char			*sepstring = ":";
    int				DontPrintRows = 0;
    int				suppress_printing = 0, print_tcl_elements = 0;
    int				flush = 0;

    if ((schema = TclQddb_GetSchema(keylist_header->schema_name)) == NULL) {
	Tcl_AppendResult(interp, argv[0], ": bad schema \"", argv[2], "\"", NULL);
	return TCL_ERROR;
    }
    for (i = 2; i <= argc-2; i++) {
	if (strcmp(argv[i], "-attrs") == 0) {
	    if (attr_argc != 0) {
		Free(attr_argv);
		attr_argv = NULL;
	    }
	    if (Tcl_SplitList(interp, argv[++i], &attr_argc, &attr_argv) != TCL_OK)
		goto error_ret;
	} else if (strcmp(argv[i], "-print") == 0) {
	    if (print_argc != 0) {
		Free(print_argv);
		print_argv = NULL;
	    }
	    if (Tcl_SplitList(interp, argv[++i], &print_argc, &print_argv) != TCL_OK)
		goto error_ret;
	    print_nums = (size_t *)Malloc(sizeof(size_t)*print_argc);
	    for (j = 0; j < print_argc; j++) {
		char		*c_buf, *n_buf;

		c_buf = strtok(print_argv[j], ":");
		n_buf = strtok(NULL, ":");
		if (n_buf == NULL)
		    print_nums[j] = 0;
		else if (Tcl_GetInt(interp, n_buf, &getint) != TCL_OK)
		    goto error_ret;
		else
		    print_nums[j] = getint;
	    }
	}  else if (strcmp(argv[i], "-sortby") == 0) {
	    if (sort_argc != 0) {
		Free(sort_argv);
		sort_argv = NULL;
	    }
	    if (Tcl_SplitList(interp, argv[++i], &sort_argc, &sort_argv) != TCL_OK)
		goto error_ret;
	} else if (strcmp(argv[i], "-ascending") == 0) {
	    if (ascending_argc != 0) {
		Free(ascending_argv);
		ascending_argv = NULL;
	    }
	    if (Tcl_SplitList(interp, argv[++i], &ascending_argc, &ascending_argv) != TCL_OK)
		goto error_ret;
	} else if (strcmp(argv[i], "-sep") == 0) {
	    sepstring = argv[++i];
	} else 	if (strcmp(argv[i], "-deldup_rows") == 0) {
	    if (Tcl_GetBoolean(interp, argv[++i], &deldup_rows) != TCL_OK)
		goto error_ret;
	} else if (strcmp(argv[i], "-query") == 0) {
	    if (Tcl_GetBoolean(interp, argv[++i], &query_mode) != TCL_OK)
		goto error_ret;
	} else if (strcmp(argv[i], "-norows") == 0) {
	    if (Tcl_GetBoolean(interp, argv[++i], &DontPrintRows) != TCL_OK)
		goto error_ret;
	} else if (strcmp(argv[i], "-suppress") == 0) {
	    if (Tcl_GetBoolean(interp, argv[++i], &suppress_printing) != TCL_OK)
		goto error_ret;
	} else if (strcmp(argv[i], "-tclelements") == 0) {
	    if (Tcl_GetBoolean(interp, argv[++i], &print_tcl_elements) != TCL_OK)
		goto error_ret;
	} else if (strcmp(argv[i], "-flush") == 0) {
	    if (Tcl_GetBoolean(interp, argv[++i], &flush) != TCL_OK)
		goto error_ret;
	} else {
	    Tcl_AppendResult(interp, "bad option \"", argv[i], "\": must be -query, -attrs or -print", NULL);
	    goto error_ret;
	}
    }
    if (query_mode == 1) {
	if (Qddb_RowListQuery(interp, schema, keylist, sepstring, print_argv, print_nums, 
			      (size_t)print_argc, attr_argv, (size_t)attr_argc, attr_argc == 0, 
			      True, (Boolean)DontPrintRows,(Boolean)print_tcl_elements,
			      (Boolean)suppress_printing) != 0)
	    goto error_ret;
    } else {
	if (TclQddb_SelectRowsBuildList(interp, keylist_header->schema_name, schema, keylist, sepstring, 
					attr_argc, attr_argv, sort_argc, sort_argv,
					ascending_argc, ascending_argv,
					print_argc, print_argv, print_nums, (Boolean)print_tcl_elements,
					(Boolean)suppress_printing, (Boolean)flush) != TCL_OK)
	    goto error_ret;
    }
    if (print_argv != NULL)
	Free(print_argv);
    if (print_nums != NULL)
	Free(print_nums);
    if (attr_argv != NULL)
	Free(attr_argv);
    if (sort_argv != NULL)
	Free(sort_argv);
    if (ascending_argv != NULL)
	Free(ascending_argv);
    return TCL_OK;
error_ret:
    if (print_argv != NULL)
	Free(print_argv);
    if (print_nums != NULL)
	Free(print_nums);
    if (attr_argv != NULL)
	Free(attr_argv);
    if (sort_argv != NULL)
	Free(sort_argv);
    if (ascending_argv != NULL)
	Free(ascending_argv);
    return TCL_ERROR;
}

int TclQddb_GenerateRowLists(interp, schema_desc, schema, list, attr_argc, attr_argv, 
			     print_argc, print_argv,
			     entry_list_argc, entry_list)
    Tcl_Interp			*interp;
    char			*schema_desc;
    Schema			*schema;
    KeyList			*list;
    int				attr_argc;
    char			*attr_argv[];
    int				print_argc;
    char			*print_argv[];
    int				*entry_list_argc;
    TclQddb_EntryRows		**entry_list;
{
    Entry			this_entry;
    InCoreEntry			*core_entry = NULL;
    DataTree			**datatree;
    char			*relation_fn = schema->RelationName;
    size_t			*print_argv_nums = NULL;
    int				db_file;
    int				i;
    int				inapplicable = Inapplicable;
    TclQddb_Tuple		*tuple;
    char			*tuple_desc;
    KeyList			**split_list, **orig_split_list, *tmplist, key, *keylist;
    Qddb_PruneArgs		prune_args;
    
    if (list == NULL)
	return TCL_OK;
    /* Check the attributes at this level for validity */
    for (i = 0; i < attr_argc; i++) {
	if (Qddb_ConvertAttributeNameToNumber(schema, attr_argv[i]) == -1) {
	    Tcl_AppendResult(interp, "bad attribute: \"", attr_argv[i], "\"", NULL);
	    return TCL_ERROR;
	}
    }
    if (print_argc > 0)
	print_argv_nums = (size_t *)Malloc(sizeof(size_t)*print_argc);
    for (i = 0; i < print_argc; i++) {
	if ((print_argv_nums[i] = Qddb_ConvertAttributeNameToNumber(schema, print_argv[i])) == -1) {
	    Free(print_argv_nums);
	    Tcl_AppendResult(interp, "bad attribute: \"", print_argv[i], "\"", NULL);
	    return TCL_ERROR;
	}
    }
    /* Here we copy the keylist into a split list; each element of the
     * split list is deleted from MARKFROMINCORE   We need to make sure that
     * we use a copy because TclQddb probably has a token set up for this KeyList.
     */
    keylist = Qddb_KeyListProcess(schema, list, NULL, QDDB_KEYLIST_PROC_NULLOP, QDDB_KEYLIST_FLAG_COPY);
    if (attr_argc > 0) {
	prune_args.attr_len = attr_argc;
	prune_args.attrs = attr_argv;
	keylist = Qddb_KeyListProcess(schema, keylist, &prune_args, QDDB_KEYLIST_PROC_PRUNE, 
				      QDDB_KEYLIST_FLAG_PRUNEBYROW);
    }
    *entry_list_argc = 0;
    *entry_list = NULL;
    if (keylist == NULL) {
	if (print_argc > 0)
	    Free(print_argv_nums);
	return TCL_OK;
    }
    (void)Qddb_KeyListProcess(schema, keylist, &split_list, QDDB_KEYLIST_PROC_SPLIT, 0);
    orig_split_list = split_list;
    db_file = schema->database_fd;
    while (*split_list != NULL) {
	tmplist = *split_list;
	this_entry = NULL;
	if (Qddb_ReadEntryByKeyList(db_file, relation_fn, &this_entry, tmplist, True) == -1) { 
	    /* invalid entry */
	    Qddb_Free(QDDB_TYPE_KEYLIST, *split_list);
	    split_list++;
	    continue;
	}
	Qddb_ReducedAttrToFullAttr(schema, this_entry);
	core_entry = (InCoreEntry *)Qddb_Convert(schema, QDDB_ENTRYTYPE_EXTERNAL, this_entry, 
						 QDDB_ENTRYTYPE_INTERNAL);
	Qddb_Free(QDDB_TYPE_ENTRY, this_entry);
	key = **split_list;
	key.Instance = NULL;
	key.next = NULL;
	(void)Qddb_KeyListProcess(schema, *split_list, core_entry, QDDB_KEYLIST_PROC_MARK, 
				  QDDB_KEYLIST_FLAG_MARKFROMINCORE);
	/* *split_list was deleted by Qddb_KeyListProcess */
	*split_list = NULL;
	datatree = (DataTree **)Qddb_Convert(schema, QDDB_ENTRYTYPE_INTERNAL, core_entry,
						 QDDB_ENTRYTYPE_DATATREE);
	Qddb_Free(QDDB_TYPE_INCOREENTRY, core_entry);
	*entry_list = (TclQddb_EntryRows *)Realloc(*entry_list, 
						   sizeof(TclQddb_EntryRows)*(++(*entry_list_argc)));
	tuple_desc = (*entry_list)[*entry_list_argc-1].tuple_desc = 
	    TclQddb_NewTuple(interp, schema_desc, datatree, &key);
	tuple = TclQddb_GetTuple(interp, tuple_desc);
	for (i = 0; i < attr_argc; i++) { 
	    /* Unfortunately, this is necessary to take care of NULL values.
	     */
	    (void)Qddb_DataTreeProcess(schema, tuple->datatree, attr_argv[i], QDDB_DATATREE_PROC_SETATTR, 0);
	}
	(*entry_list)[*entry_list_argc-1].rows = 
	    TclQddb_BuildAllRows(interp, schema, tuple, print_argv, print_argc, NULL, NULL);
	if ((*entry_list)[*entry_list_argc-1].rows == NULL) {
	    /* This can occur when doing unattributed searches, since the keylist was not pruned
	     * by row.
	     */
	    TclQddb_DeleteTuple(interp, tuple_desc); 
	    (*entry_list_argc)--;
	} else {
	    (void)Qddb_DataTreeProcess(schema, tuple->datatree, 
				       &inapplicable, QDDB_DATATREE_PROC_SETMARK, 0);
	}
	split_list++;
    }
    if (print_argc > 0)
	Free(print_argv_nums);
    Free(orig_split_list);
    return TCL_OK;
}

static int TclQddb_SelectRowsBuildList(interp, schema_desc, schema, list, sep, attr_argc, attr_argv, 
				       sort_argc, sort_argv, ascending_argc, ascending_argv,
				       print_argc, print_argv, print_widths, print_tcl_elements,
				       suppress_printing, flush)
    Tcl_Interp			*interp;
    char			*schema_desc;
    Schema			*schema;
    KeyList			*list;
    char			*sep;
    int				attr_argc;
    char			**attr_argv;
    int				sort_argc;
    char			**sort_argv;
    int				ascending_argc;
    char			**ascending_argv;
    int				print_argc;
    char			**print_argv;
    size_t			*print_widths;
    Boolean			print_tcl_elements, suppress_printing, flush;
{
    TclQddb_EntryRows		*entry_rows;
    char			*tuple_token;
    char			**row_descs, **row_vals;
    char			**ret_argv, **rowdesc_argv, *retval;
    int				ret_argc, rowdesc_argc;
    int				entry_rows_argc;
    int				i, j;

    if (list == NULL)
	return TCL_OK;
    if (TclQddb_GenerateRowLists(interp, schema_desc, schema, list, attr_argc, attr_argv, 
				 print_argc, print_argv,
				 &entry_rows_argc, &entry_rows) != TCL_OK)
	return TCL_ERROR;
    ret_argc = 0;
    rowdesc_argc = 0;
    ret_argv = NULL;
    rowdesc_argv = NULL;
    retval = NULL;
    for (i = 0; i < entry_rows_argc; i++) {
	/* Need 3 things: (1) new tuple, (2) list of rows, (3) row values (if necessary) */
	tuple_token = entry_rows[i].tuple_desc;
	row_descs = row_vals = NULL;
	if (print_argc > 0) {
	    if (TclQddb_BuildNewRows(interp, schema_desc, tuple_token, 
				     entry_rows+i, sep, print_argc, print_argv, print_widths,
				     &row_descs, &row_vals, 
				     print_tcl_elements, suppress_printing) != TCL_OK) {
		/* FIXME: Free up stuff */
		return TCL_ERROR;
	    }
	} else {
	    if (TclQddb_BuildNewRows(interp, schema_desc, tuple_token, entry_rows+i, sep, 0, NULL, NULL,
				     &row_descs, NULL, 
				     print_tcl_elements, suppress_printing) != TCL_OK) {
		/* FIXME: Free up stuff */
		return TCL_ERROR;
	    }
	}
	/* Now we need to take the tuple, row_descs and row_vals (if any) and do a
	 * cartesian product (tuple X (row_desc,row_val)) to produce the correct
	 * return value.   The return value should look something like:
	 *	{ { <tuple_desc> <row_desc> "row value" }
	 *	  { <tuple_desc> <row_desc> "row value" }
	 *	}
	 */
	for (j = 0; row_descs[j] != NULL; j++) {
	    char		*tmp_argv[3], *buf;

	    tmp_argv[0] = tuple_token;
	    tmp_argv[1] = row_descs[j];
	    if (row_vals != NULL) {
		tmp_argv[2] = row_vals[j];
	    } else {
		tmp_argv[2] = "";
	    }
	    buf = Tcl_Merge(3, tmp_argv);
	    ret_argv = (char **)Realloc(ret_argv, sizeof(char *)*(++ret_argc));
	    ret_argv[ret_argc-1] = buf;
	    rowdesc_argv = (char **)Realloc(rowdesc_argv, sizeof(char *)*(++rowdesc_argc));
	    rowdesc_argv[rowdesc_argc-1] = row_descs[j];
	    if (row_vals != NULL && row_vals[j] != NULL)
		Free(row_vals[j]);
	}
	if (row_descs != NULL)
 	    Free(row_descs);
	if (row_vals != NULL)
	    Free(row_vals);
	if (flush)
	    TclQddb_FlushTuple(interp, tuple_token);
	Free(tuple_token);
    }
    if (entry_rows_argc != 0)
	Free(entry_rows);
    if (ret_argc > 0) {
	if (sort_argc > 0) {
	    if (TclQddb_SortSelectRows(interp, rowdesc_argc, rowdesc_argv, ret_argv, sort_argc, sort_argv, 
				 ascending_argc, ascending_argv) != TCL_OK) {
		for (i = 0; i < ret_argc; i++)
		    Free(ret_argv[i]);
		Free(ret_argv);
		for (i = 0; i < rowdesc_argc; i++)
		    Free(rowdesc_argv[i]);
		Free(rowdesc_argv);
		return TCL_ERROR;
	    }
	}
	retval = Tcl_Merge(ret_argc, ret_argv);
	for (i = 0; i < ret_argc; i++)
	    Free(ret_argv[i]);
	Free(ret_argv);
	for (i = 0; i < rowdesc_argc; i++)
	    Free(rowdesc_argv[i]);
	Free(rowdesc_argv);
	Tcl_SetResult(interp, retval, TCL_DYNAMIC);
    }
    return TCL_OK;
}


static int TclQddb_BuildNewRows(interp, schema_desc, tuple_desc, row, sep, 
				print_argc, print_argv, print_widths,
				row_descs, row_vals,
				print_tcl_elements, suppress_printing)
    Tcl_Interp			*interp;
    char			*schema_desc, *tuple_desc;
    TclQddb_EntryRows		*row;
    char			*sep;
    int				print_argc;
    char			*print_argv[];
    size_t			*print_widths;
    char			***row_descs, ***row_vals;
    Boolean			print_tcl_elements, suppress_printing;
{
    Schema			*schema;
    TclQddb_Tuple		*tuple;
    char			*newtok;
    int				num_rows, i;

    if ((schema = TclQddb_GetSchema(schema_desc)) == NULL) {
	Tcl_AppendResult(interp, "bad schema \"", schema_desc, "\"", NULL);
	return TCL_ERROR;
    }
    if ((tuple = TclQddb_GetTuple(interp, tuple_desc)) == NULL) {
	Tcl_AppendResult(interp, "bad tuple \"", schema_desc, "\"", NULL);
	return TCL_ERROR;
    }
    for (num_rows = 0; row->rows[num_rows] != NULL; num_rows++);
    *row_descs = (char **)Malloc(sizeof(char *)*(num_rows+1));
    (*row_descs)[num_rows] = NULL;
    if (row_vals != NULL && !suppress_printing) {
	*row_vals = (char **)Malloc(sizeof(char *)*(num_rows+1));
	(*row_vals)[num_rows] = NULL;
    }
    if (print_tcl_elements != False) {
	for (i = 0; row->rows[i] != NULL; i++) {
	    if ((newtok=TclQddb_AddRow(interp, schema_desc, tuple_desc, row->rows[i])) == NULL) {
		/* FIXME: Free up all storage allocated so far */
		return TCL_ERROR;
	    }
	    (*row_descs)[i] = newtok;
	    if (row_vals != NULL && !suppress_printing) {
		(*row_vals)[i] = TclQddb_RowAsElements(interp, schema, tuple, row->rows[i], sep,
						  print_argc, print_argv, print_widths);
	    }
	}
    } else {
	for (i = 0; row->rows[i] != NULL; i++) {
	    if ((newtok=TclQddb_AddRow(interp, schema_desc, tuple_desc, row->rows[i])) == NULL) {
		/* FIXME: Free up all storage allocated so far */
		return TCL_ERROR;
	    }
	    (*row_descs)[i] = newtok;
	    if (row_vals != NULL && !suppress_printing) {
		(*row_vals)[i] = TclQddb_PrintRow(interp, schema, tuple, row->rows[i], sep,
						  print_argc, print_argv, print_widths);
	    }
	}
    }
    Free(row->rows);
    return TCL_OK;
}

static char *TclQddb_PrintRow(interp, schema, tuple, row, sep, print_argc, print_argv, print_widths)
    Tcl_Interp			*interp;
    Schema			*schema;
    TclQddb_Tuple		*tuple;
    TclQddb_Rows		*row;
    char			*sep;
    int				print_argc;
    char			*print_argv[];
    size_t			*print_widths;
{
    DataTree			*node;
    TclQddb_Rows		*list;
    char			*retval, *element;
    int				i, j;

    Qddb_InitBuffer();
    for (i = 0; i < print_argc; i++) {
	for (list = row; list != NULL; list = list->next) {
	    if (strcmp(list->attr_name, print_argv[i]) == 0)
		break;
	}
	if (list == NULL) {
	    /* element doesn't exist */
	    if (print_widths[i] <= 0) {
		Qddb_ConcatBuffer(sep);
	    } else {
		retval = Malloc(print_widths[i]+1);
		for (j = 0; j < print_widths[i]; j++)
		    retval[j] = ' ';
		retval[j] = '\0';
		Qddb_ConcatBuffer(retval);
		Qddb_ConcatBuffer(sep);
		Free(retval);
	    }
	} else {
	    if (print_widths[i] <= 0) {
		node = TclQddb_DataTreeLookupNode(interp, tuple, list->attr_name, list->instance);
#if defined(DIAGNOSTIC)
		if (node == NULL) {
		    PANIC("TclQddb_PrintRow: cannot find node");
		}
#endif
		element = (char *)Qddb_DataTreeProcess(NULL, NULL, (void *)node, 
						       QDDB_DATATREE_PROC_GETDATA, 0);
#if defined(DIAGNOSTIC)
		if (element == NULL) {
		    PANIC("TclQddb_PrintRow: cannot find data for a node");
		}
#endif
		Qddb_ConcatBuffer(element);
		Free(element);
		Qddb_ConcatBuffer(sep);
	    } else {
		node = TclQddb_DataTreeLookupNode(interp, tuple, list->attr_name, list->instance);
#if defined(DIAGNOSTIC)
		if (node == NULL) {
		    PANIC("TclQddb_PrintRow: cannot find node");
		}
#endif
		element = (char *)Qddb_DataTreeProcess(NULL, NULL, (void *)node, 
						       QDDB_DATATREE_PROC_GETDATA, 0);
#if defined(DIAGNOSTIC)
		if (element == NULL) {
		    PANIC("TclQddb_PrintRow: cannot find data for a node");
		}
#endif
		retval = Qddb_FieldString(element, print_widths[i]);
		Free(element);
		Qddb_ConcatBuffer(retval);
		Qddb_ConcatBuffer(sep);
		Free(retval);
	    }
	}
    }
    return Qddb_GetBuffer();
}

static char *TclQddb_RowAsElements(interp, schema, tuple, row, sep, print_argc, print_argv, print_widths)
    Tcl_Interp			*interp;
    Schema			*schema;
    TclQddb_Tuple		*tuple;
    TclQddb_Rows		*row;
    char			*sep;
    int				print_argc;
    char			*print_argv[];
    size_t			*print_widths;
{
    char			**merge_buf;
    TclQddb_Rows		*list;
    DataTree			*node;
    char			*element;
    char			*retval;
    size_t			data_len;
    int				i, j;

    merge_buf = (char **)Malloc(sizeof(char *)*print_argc);
    for (i = 0; i < print_argc; i++) {
	for (list = row; list != NULL; list = list->next) {
	    if (strcmp(list->attr_name, print_argv[i]) == 0)
		break;
	}
	if (list == NULL) {
	    /* element doesn't exist */
	    if (print_widths[i] <= 0) {
		merge_buf[i] = Malloc(QDDB_MIN_MALLOC_SIZE);
		*(merge_buf[i]) = '\0';
	    } else {
		retval = merge_buf[i] = Malloc(print_widths[i]+1);
		for (j = 0; j < print_widths[i]; j++)
		    *retval++ = ' ';
	    }
	} else {
	    node = TclQddb_DataTreeLookupNode(interp, tuple, list->attr_name, list->instance);
#if defined(DIAGNOSTIC)
	    if (node == NULL) {
		PANIC("TclQddb_PrintRow: cannot find node");
	    }
#endif
	    element = (char *)Qddb_DataTreeProcess(NULL, NULL, (void *)node, 
						   QDDB_DATATREE_PROC_GETDATA, 0);
#if defined(DIAGNOSTIC)
	    if (element == NULL) {
		PANIC("TclQddb_PrintRow: cannot find data for a node");
	    }
#endif
	    if (print_widths[i] <= 0) {
		data_len = strlen(element);
		merge_buf[i] = Malloc(data_len+strlen(sep)+1);
		strcpy(merge_buf[i], element);
	    } else {
		retval = Qddb_FieldString(element, print_widths[i]);
		merge_buf[i] = retval;
	    }
	    Free(element);
	}
    }
    retval = Tcl_Merge(print_argc, merge_buf);
    TclQddb_FreeArgs(print_argc, merge_buf);
    return retval;
}

TclQddb_RowHeader *TclQddb_GetRows(interp, token)
    Tcl_Interp			*interp;
    char			*token;
{
    TclQddb_RowHeader		*rows;
    Tcl_HashEntry		*hash_ptr;

    if (TclQddb_RowsHashTableInit == 0) {
	TclQddb_RowsHashTableInit = 1;
	Tcl_InitHashTable(&TclQddb_RowsHashTable, TCL_STRING_KEYS);
	return NULL;
    }
    hash_ptr = Tcl_FindHashEntry(&TclQddb_RowsHashTable, token);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot find row \"", token, "\" (TCL error)", NULL);
	return NULL;
    }
    rows = (TclQddb_RowHeader *)Tcl_GetHashValue(hash_ptr);
    return rows;
}

static int TclQddb_TranslateRows(interp, token)
    Tcl_Interp			*interp;
    char			*token;
{
    TclQddb_RowHeader		*rows;
    TclQddb_Rows		*r;
    char			buf[BUFSIZ];

    rows = TclQddb_GetRows(interp, token);
    for (r = rows->row; r != NULL; r = r->next) {
	if (r->next == NULL)
	    sprintf(buf, "%s,%s", r->attr_name, r->instance);
	else
	    sprintf(buf, "%s,%s ", r->attr_name, r->instance);
	Tcl_AppendResult(interp, buf, NULL);
    }
    return TCL_OK;
}


int TclQddb_SetRow(interp, row_header)
    Tcl_Interp			*interp;
    TclQddb_RowHeader		*row_header;
{
    Tcl_HashEntry		*hash_ptr;
    char			token[BUFSIZ];
    int				newPtr;

    if (TclQddb_RowsHashTableInit == 0) {
	TclQddb_RowsHashTableInit = 1;
	Tcl_InitHashTable(&TclQddb_RowsHashTable, TCL_STRING_KEYS);
    }
    sprintf(token, "qddb_row%d", TclQddb_RowsNextNumber++);
    hash_ptr = Tcl_CreateHashEntry(&TclQddb_RowsHashTable, token, &newPtr);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot create hash entry \"", token, "\" (TCL error)", NULL);
	return TCL_ERROR;
    }
    Tcl_SetHashValue(hash_ptr, (ClientData)row_header);
    Tcl_SetResult(interp, token, TCL_VOLATILE);
    return TCL_OK;
}


static char *TclQddb_AddRow(interp, schema_desc, tuple_desc, row)
    Tcl_Interp			*interp;
    char			*schema_desc, *tuple_desc;
    TclQddb_Rows		*row;
{
    TclQddb_RowHeader		*row_header;
    Tcl_HashEntry		*hash_ptr;
    char			token[BUFSIZ], *retval;
    int				newPtr;

    if (TclQddb_RowsHashTableInit == 0) {
	TclQddb_RowsHashTableInit = 1;
	Tcl_InitHashTable(&TclQddb_RowsHashTable, TCL_STRING_KEYS);
    }
    sprintf(token, "qddb_row%d", TclQddb_RowsNextNumber++);
    hash_ptr = Tcl_CreateHashEntry(&TclQddb_RowsHashTable, token, &newPtr);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot create hash entry \"", token, "\" (TCL error)", NULL);
	return NULL;
    }
    row_header = (TclQddb_RowHeader *)Malloc(sizeof(TclQddb_RowHeader));
    row_header->schema_name = Malloc(strlen(schema_desc)+1);
    strcpy(row_header->schema_name, schema_desc);
    row_header->tuple_name = Malloc(strlen(tuple_desc)+1);
    strcpy(row_header->tuple_name, tuple_desc);
    row_header->row = row;
    Tcl_SetHashValue(hash_ptr, (ClientData)row_header);
    retval = Malloc(strlen(token)+1);
    strcpy(retval, token);
    return retval;
}

static void TclQddb_FreeRow(row)
    TclQddb_Rows		*row;
{
    TclQddb_Rows		*old_row;

    while (row != NULL) {
	old_row = row;
	row = row->next;
	Free(old_row->attr_name);
	Free(old_row->instance);
	Free(old_row);
    }
}

static int TclQddb_DeleteRowsOne(interp, token)
    Tcl_Interp			*interp;
    char			*token;
{
    TclQddb_RowHeader		*row_header;
    Tcl_HashEntry		*hash_ptr;

    hash_ptr = Tcl_FindHashEntry(&TclQddb_RowsHashTable, token);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot find row \"", token, "\" (TCL error)", NULL);
	return TCL_ERROR;	
    }
    row_header = (TclQddb_RowHeader *)Tcl_GetHashValue(hash_ptr);
    Tcl_DeleteHashEntry(hash_ptr);
    TclQddb_FreeRow(row_header->row);
    Free(row_header->schema_name);
    Free(row_header->tuple_name);
    Free(row_header);
    return TCL_OK;
}

int TclQddb_DeleteRows(interp, token)
    Tcl_Interp			*interp;
    char			*token;
{
    Tcl_HashEntry		*hash_ptr;
    Tcl_HashSearch		hash_search;
    char			*hash_key;
    int				deletebytuple = 0;

    if (TclQddb_RowsHashTableInit == 0)
	return TCL_OK;
    if (strncmp("qddb_tuple", token, 10) == 0)
	deletebytuple = 1;
    if (strcmp(token, "all") == 0 || deletebytuple == 1) {
	hash_ptr = Tcl_FirstHashEntry(&TclQddb_RowsHashTable, &hash_search);
	while (hash_ptr != NULL) {
	    if (deletebytuple == 1) {
		TclQddb_RowHeader *rows;

		rows = (TclQddb_RowHeader *)Tcl_GetHashValue(hash_ptr);
		if (strcmp(rows->tuple_name, token) != 0) {
		    hash_ptr = Tcl_NextHashEntry(&hash_search);
		    continue;
		}		
	    }
	    hash_key = Tcl_GetHashKey(&TclQddb_RowsHashTable, hash_ptr);
	    if (hash_key == NULL) {
		Tcl_AppendResult(interp, "TclQddb_DeleteRows: ", 
				 "Tcl_GetHashKey failed (TCL ERROR)", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_DeleteRowsOne(interp, hash_key) != TCL_OK)
		return TCL_ERROR;
	    hash_ptr = Tcl_NextHashEntry(&hash_search);
	}
	if (deletebytuple == 0) {
	    TclQddb_RowsHashTableInit = 0;
	    Tcl_DeleteHashTable(&TclQddb_RowsHashTable);
	}
    } else if (TclQddb_DeleteRowsOne(interp, token) != TCL_OK)
	return TCL_ERROR;
    return TCL_OK;
}

void TclQddb_DeleteRowProc(clientData)
    ClientData			clientData;
{
#if defined(DEBUG_MALLOC)
    fprintf(stderr, "TclQddb_DeleteRowProc\n");
#endif
    (void)TclQddb_DeleteRows((Tcl_Interp *)clientData, "all");
}


static int TclQddb_GetVal(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			**argv;
{
    TclQddb_RowHeader		*row_header;
    TclQddb_Tuple		*tuple;
    TclQddb_Rows		*row;
    DataTree			*node;
    int				attr_argc;
    char			**attr_argv = NULL, *element;
    int				i;

    row_header = TclQddb_GetRows(interp, argv[3]);
    if (row_header == NULL) {
	Tcl_AppendResult(interp, "Cannot find row for row descriptor \"", argv[3], "\"", NULL);
	return TCL_ERROR;
    }
    if ((tuple = TclQddb_GetTuple(interp, row_header->tuple_name)) == NULL) {
	Tcl_AppendResult(interp, "Cannot find tuple \"", row_header->tuple_name,
			 "\" for row descriptor \"", argv[3], "\"", NULL);
	return TCL_ERROR;
    }
    if (Tcl_SplitList(interp, argv[2], &attr_argc, &attr_argv) != TCL_OK)
	return TCL_ERROR;
    for (i = 0; i < attr_argc; i++) {
	row = row_header->row;
	while (row != NULL && strcmp(attr_argv[i], row->attr_name) != 0)
	    row = row->next;
	if (row == NULL) {
	    Tcl_AppendElement(interp, "");
	    continue;
	}
	if ((node = TclQddb_DataTreeLookupNode(interp, tuple, row->attr_name, row->instance)) == NULL) {
	    if (attr_argv != NULL)
		Free(attr_argv);
	    Tcl_ResetResult(interp);
	    Tcl_AppendResult(interp, "Cannot find part of datatree node \"", row->attr_name, ",",
			     row->instance, "\"", NULL);

	    return TCL_ERROR;
	}
	if ((element = (char *)Qddb_DataTreeProcess(NULL, NULL, (void *)node, 
						    QDDB_DATATREE_PROC_GETDATA, 0)) == NULL) {
	    if (attr_argv != NULL)
		Free(attr_argv);
	    Tcl_ResetResult(interp);
	    Tcl_AppendResult(interp, "Datatree node corrupted \"", row->attr_name, ",",
			     row->instance, "\"", NULL);

	    return TCL_ERROR;	    
	}
	Tcl_AppendElement(interp, element);
	Free(element);
    }
    return TCL_OK;
}

/* TclQddb_DoSortRows -- Front-end wrapper for TclQddb_SortRows.
 */
static int TclQddb_DoSortRows(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			**argv;
{
    int				attr_argc, row_argc, ascending_argc = 0;
    char			**attr_argv = NULL, **row_argv = NULL, **ascending_argv = NULL, *retval;
    char			*attrs_str, *rows_str;

    /* qddb_rows sort ?-ascending <attr list>? <attr list> <row_desc list> */
    if (argc == 6) {
	if (Tcl_SplitList(interp, argv[3], &ascending_argc, &ascending_argv) != TCL_OK)
	    goto error_ret;
	attrs_str = argv[4];
	rows_str = argv[5];
    } else if (argc != 4) {
	Tcl_AppendResult(interp, "wrong # args for sort", NULL);
	return TCL_ERROR;
    } else {
	attrs_str = argv[2];
	rows_str = argv[3];
    }
    if (Tcl_SplitList(interp, attrs_str, &attr_argc, &attr_argv) != TCL_OK)
	goto error_ret;
    if (Tcl_SplitList(interp, rows_str, &row_argc, &row_argv) != TCL_OK)
	goto error_ret;
    if (TclQddb_SortRows(interp, row_argc, row_argv, attr_argc, attr_argv, 
			 ascending_argc, ascending_argv) != TCL_OK)
	goto error_ret;
    /* package result */
    retval = Tcl_Merge(row_argc, row_argv);
    Tcl_SetResult(interp, retval, TCL_DYNAMIC);
    if (attr_argv != NULL)
	Free(attr_argv);
    if (row_argv != NULL)
	Free(row_argv);
    if (ascending_argv != NULL)
	Free(ascending_argv);
    return TCL_OK;
error_ret:
    if (attr_argv != NULL)
	Free(attr_argv);
    if (row_argv != NULL)
	Free(row_argv);
    if (ascending_argv != NULL)
	Free(ascending_argv);
    return TCL_ERROR;
}

DataTree *TclQddb_DataTreeLookupNode(interp, tuple, attr, inst)
    Tcl_Interp			*interp;
    TclQddb_Tuple		*tuple;
    char			*attr, *inst;
{
    DataTree			**dt, *dinst;
    char			**attr_array;
    size_t			Instance[MAXIMUM_NUMBER_OF_SCHEMA_LEVELS];
    int				spot, level;
    int				i;

    dinst = NULL;
    attr_array = TclQddb_SplitAttributeName(attr);
    StringToInstance(inst, Instance);
    dt = tuple->datatree; 
    level = 0;
    while (dt != NULL) {
	spot = TclQddb_FindAttribute(dt, attr_array[level]);
	if (spot == -1)
	    goto error_ret;
        for (i = 0; dt[spot][i].datatree_type != DATATREE_END; i++);
	if (Instance[level] > i)
	    goto error_ret;
	dinst = dt[spot] + (Instance[level] - 1);
	if (dinst->datatree_type == DATATREE_CHILDREN) {
	    dt = dinst->datatree_children;
	} else if (level == dinst->datatree_schema->schemanode->Level-1) {
	    TclQddb_FreeArgs(-1, attr_array);
	    return dinst;
	} else {
	    break;
	}
	level++;
    }
error_ret:
    TclQddb_FreeArgs(-1, attr_array);
    Tcl_AppendResult(interp, "bad instance number \"", inst, "\"", NULL);
    return NULL;
}

int TclQddb_FormatRows(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			**argv;
{
    return TCL_OK;
}
