
/* qddb/Lib/LibQddb/KeyList.c 
 *
 * Copyright (C) 1993, 1994 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */

#include "Qddb.h"

/* EXPORTED:
 *	KeyList *Qddb_KeyListOp(Schema *, KeyList *, KeyList *, int, int)
 *	KeyList *Qddb_KeyListProcess(Schema *, KeyList *, void *, int, int)
 *	KeyList *Qddb_KeyListConcat(KeyList *, KeyList *, KeyList **);
 */

static KeyList *KeyListIntersection _ANSI_ARGS_((Schema *, KeyList *, KeyList *, int));
static KeyList *KeyListUnion _ANSI_ARGS_((Schema *, KeyList *, KeyList *, int));
static KeyList *KeyListExclusion _ANSI_ARGS_((Schema *, KeyList *, KeyList *, int));

static KeyList *KeyListMarkEntry _ANSI_ARGS_((Schema *, KeyList *, InCoreEntry *));
static KeyList *KeyListPruneByRow _ANSI_ARGS_((Schema *, KeyList *, char **, int));
static KeyList **KeyListSplitList _ANSI_ARGS_((Schema *, KeyList *));
static KeyList *KeyListGetList _ANSI_ARGS_((KeyList **, int));

static KeyList *PruneListByAttribute _ANSI_ARGS_((Schema *, KeyList *, char *));

static KeyList *KeyListHashTable _ANSI_ARGS_((KeyList **));

/* Qddb_KeyListOp -- Standard binary operations on pairs of KeyLists.
 *
 * Operations
 * ----------
 * 	QDDB_KEYLIST_OP_UNION: 			List1 + List2
 * 	QDDB_KEYLIST_OP_INTERSECTION:		List1 * List2
 * 	QDDB_KEYLIST_OP_EXCLUSION:		List1 * ^List2
 *
 * Flags
 * -----
 *	QDDB_KEYLIST_FLAG_COPY:			Copy KeyList nodes instead of using existing nodes.
 *	QDDB_KEYLIST_FLAG_DELDUP_SAMEENTRY:	Delete all duplicate nodes in same entry.
 *	QDDB_KEYLIST_FLAG_DELDUP_SAMEATTR:	Delete all duplicate nodes in same entry with
 *						the same (attribute,instance) pair.  These are
 *						EXACT duplicates.
 *  	QDDB_KEYLIST_FLAG_EXACT:		Forces the operation to match (attr,inst) pairs
 *						as well as the quadruples.
 */
KeyList *Qddb_KeyListOp(schema, List1, List2, op, flags)
    Schema		*schema;
    KeyList		*List1, *List2;
    int			op, flags;
{
    KeyList		*retval;

    qddb_errno = 0;
    switch (op) {
    case QDDB_KEYLIST_OP_UNION:
	retval = KeyListUnion(schema, List1, List2, flags);
	break;
    case QDDB_KEYLIST_OP_INTERSECTION:
	retval = KeyListIntersection(schema, List1, List2, flags);
	break;
    case QDDB_KEYLIST_OP_EXCLUSION:
	retval = KeyListExclusion(schema, List1, List2, flags);
	break;
    default:
	qddb_errno = QDDB_ERRNO_INVALID_ARG;
	return NULL;
    }
    return retval;
}

/* Qddb_KeyListProcess -- Perform various operations on a single KeyList.
 *
 * Operations
 * ----------
 *	QDDB_KEYLIST_PROC_MARK:			Mark the KeyList as specified.
 *	QDDB_KEYLIST_PROC_PRUNE:		Prune the KeyList by the given row markers.
 *	QDDB_KEYLIST_PROC_SORT:			Sort the KeyList.
 *	QDDB_KEYLIST_PROC_SPLIT:		Split the list into an array of lists, each
 *						element containing a pointer to the first
 *						element of a list.   Terminate the array with
 *						a NULL pointer.
 *	QDDB_KEYLIST_PROC_NULLOP:		Perform no operation, just return result based
 *						on the value of 'flags.'
 *
 * Generic Flags
 * -------------
 *	QDDB_KEYLIST_FLAG_COPY:
 *	QDDB_KEYLIST_FLAG_DELDUP_SAMEENTRY:
 *	QDDB_KEYLIST_FLAG_DELDUP_SAMEATTR:
 *
 * QDDB_KEYLIST_PROC_MARK specific flags
 * -------------------------------------
 *	QDDB_KEYLIST_FLAG_MARKASANY:		Just mark all KeyList nodes as matching an
 *						'Any' search.
 *	QDDB_KEYLIST_FLAG_MARKFROMINCORE:	Mark up an InCoreEntry (args == InCoreEntry *)
 *	QDDB_KEYLIST_FLAG_MARKFROMDATATREE:	Mark up a DataTree (args == DataTree *)
 *	QDDB_KEYLIST_FLAG_PRUNEBYROW:
 *	QDDB_KEYLIST_FLAG_PRUNEBYATTR:
 */
void *Qddb_KeyListProcess(schema, List, args, op, flags)
    Schema		*schema;
    KeyList		*List;
    void		*args;
    int			op, flags;
{
    KeyList		*retval = NULL;
    Qddb_PruneArgs	*prune_args;

    qddb_errno = 0;
    List = KeyListGetList(&List, flags);
    switch (op) {
    case QDDB_KEYLIST_PROC_MARK:
	if ((flags & QDDB_KEYLIST_FLAG_MARKFROMDATATREE) != 0)
	    return NULL;
	else if ((flags & QDDB_KEYLIST_FLAG_MARKFROMINCORE) != 0)
	    retval = KeyListMarkEntry(schema, List, (InCoreEntry *)args);
	break;
    case QDDB_KEYLIST_PROC_PRUNE:
	if ((flags & QDDB_KEYLIST_FLAG_PRUNEBYATTR) != 0)
	    retval = PruneListByAttribute(schema, List, (char *)args);
	else if ((flags & QDDB_KEYLIST_FLAG_PRUNEBYROW) != 0) {
	    prune_args = (Qddb_PruneArgs *)args;
	    retval = KeyListPruneByRow(schema, List, prune_args->attrs, prune_args->attr_len);
	} else {
	    qddb_errno = QDDB_ERRNO_INVALID_ARG;
	    retval = NULL;
	}
	break;
    case QDDB_KEYLIST_PROC_SPLIT:
	/* args is (KeyList ***) */
	*(KeyList ***)args = KeyListSplitList(schema, List);
	if ((flags & QDDB_KEYLIST_FLAG_COPY) != 0)
	    retval = List;
	else
	    retval = NULL;
	break;
    case QDDB_KEYLIST_PROC_SORT:
    case QDDB_KEYLIST_PROC_NULLOP:
	retval = List;
	break;
    default:
	qddb_errno = QDDB_ERRNO_INVALID_ARG;
	return NULL;
    }
    return (void *)retval;
}

/* KeyListAddToList -- Add a KeyList node to the given KeyList.   This routine assumes that
 * Src should be freed if it is not to be placed in the KeyList (due to DELDUP flags).
 */
static void KeyListAddToList(Dest, Src, flags)
    KeyList		**Dest, *Src;
    int			flags;
{
    KeyList		**BackPtr;
    int			start = 0;

    if (*Dest == NULL) {
	*Dest = Src;
        Src->next = NULL;
	return;
    }
    /* *Dest != NULL */
    BackPtr = Dest;
    while (*BackPtr != NULL) {
	if (QDDB_KEYLIST_SAMEENTRY(Src, *BackPtr)) {
	    start = 1;
	    /* If we're not checking for dups, then just place the new node here */
	    if ((flags & (QDDB_KEYLIST_FLAG_DELDUP_SAMEENTRY|QDDB_KEYLIST_FLAG_DELDUP_SAMEATTR)) == 0) {
		break;
	    }
	} else if (start == 1)
	    break;
	if (start == 1 && QDDB_KEYLIST_DELDUP_MATCH(Src, *BackPtr, flags)) {
	    /* This node is a duplicate that we want to dispose of */
	    Qddb_Free(QDDB_TYPE_KEYLIST, Src);
	    return;
	}
	BackPtr = &((*BackPtr)->next);
    }
    /* Found the right place to insert Src */
    Src->next = *BackPtr;
    *BackPtr = Src;
    return;
}


static KeyList *KeyListCopyNode(node)
    KeyList		*node;
{
    KeyList		*retval;

    retval = (KeyList *)Malloc(sizeof(KeyList));
    *retval = *node;
    retval->Instance = Malloc(strlen(node->Instance)+1);
    strcpy(retval->Instance, node->Instance);
    retval->next = NULL;
    return retval;
}

static KeyList *KeyListGetNode(node, flags)
    KeyList		***node;
    int			flags;
{
    KeyList		*retval;

#if defined(DIAGNOSTIC)
    if (node == NULL)
	PANIC("KeyListGetNode: KeyList node == NULL\nInternal Error\n");
    if (*node == NULL)
	PANIC("KeyListGetNode: KeyList *node == NULL\nInternal Error\n");
#endif
    if (**node == NULL)
	return NULL;
    if ((flags & QDDB_KEYLIST_FLAG_COPY) == 0) {
	retval = **node;
	**node = (**node)->next;
	retval->next = NULL;
    } else {
	retval = KeyListCopyNode(**node);
	*node = &(**node)->next;
    }
    return retval;
}

static KeyList *KeyListGetList(node, flags)
    KeyList		**node;
    int			flags;
{
    KeyList		*retval, *tmp, **ptr;
    KeyList		*hashtable[QDDB_KEYLIST_HASH_NUM];

    bzero(hashtable, sizeof(hashtable));
#if defined(DIAGNOSTIC)
    if (node == NULL)
	PANIC("KeyListGetList: KeyList **node == NULL");
#endif
    ptr = node;
    retval = NULL;
    while (*ptr != NULL) {
	tmp = KeyListGetNode(&ptr, flags);
	KeyListAddToList(&hashtable[QDDB_KEYLIST_HASH_VAL(tmp->Start)], tmp, flags);
    }
    retval = KeyListHashTable(hashtable);
    return retval;
}

static KeyList **KeyListBuildHashTable(List, flags)
    KeyList		**List;
    int			flags;
{
    KeyList		**retval, **ptr, *tmp;

    retval = (KeyList **)Calloc(sizeof(KeyList *)*QDDB_KEYLIST_HASH_NUM);
    ptr = List;
    while (*ptr != NULL) {
	tmp = KeyListGetNode(&ptr, flags);
	KeyListAddToList(&retval[QDDB_KEYLIST_HASH_VAL(tmp->Start)], tmp, flags);
    }
    return retval;
}

static KeyList **KeyListHashTableUnion(hash1, hash2, flags)
    KeyList		**hash1, **hash2;
    int			flags;
{
    KeyList		**retval, **ptr;
    int			i;

    /* HashTable operations do not recognize the COPY flag */
    flags &= ~QDDB_KEYLIST_FLAG_COPY;
    ptr = retval = (KeyList **)Calloc(sizeof(KeyList *)*QDDB_KEYLIST_HASH_NUM);
    for (i = 0; i < QDDB_KEYLIST_HASH_NUM; i++, hash1++, hash2++, ptr++) {
	KeyList		**hash2_next;

	if (*hash1 == NULL) {
	    if (*hash2 == NULL) {
		*ptr = NULL;
	    } else {
		*ptr = KeyListGetList(hash2, flags);
	    }
	    continue;
	} else {
	    if (*hash2 == NULL) {
		*ptr = KeyListGetList(hash1, flags);
		continue;
	    }
	}
	/* hash1 != NULL && hash2 != NULL, merge both into ptr */
	hash2_next = hash2;
	*ptr = KeyListGetList(hash1, flags);
	while (*hash2_next != NULL) {
	    KeyList	*node;

	    node = KeyListGetNode(&hash2_next, flags);
	    KeyListAddToList(ptr, node, flags);
	}
    }
    return retval;
}

static KeyList **KeyListHashTableIntersection(hash1, hash2, flags)
    KeyList		**hash1, **hash2;
    int			flags;
{
    KeyList		**retval, **ptr;
    KeyList		**list1, **list2;
    int			i;

    /* HashTable operations do not recognize the COPY flag */
    flags &= ~QDDB_KEYLIST_FLAG_COPY;
    ptr = retval = (KeyList **)Calloc(QDDB_KEYLIST_HASH_NUM*sizeof(KeyList *));
    for (i = 0; i < QDDB_KEYLIST_HASH_NUM; i++, hash1++, hash2++, ptr++) {
	if (*hash1 == NULL || *hash2 == NULL) {
	    *ptr = NULL;
	    Qddb_Free(QDDB_TYPE_KEYLIST, *hash1);
	    Qddb_Free(QDDB_TYPE_KEYLIST, *hash2);
	    *hash1 = NULL;
	    *hash2 = NULL;
	} else {
	    list1 = hash1;
	    list2 = hash2;
	    /* Do a true intersection, either exact or not */
	    if ((flags & QDDB_KEYLIST_FLAG_EXACT) != 0) {
		/* Exact intersection */
		while (*list1 != NULL) {
		    int		flag;
		    KeyList	*node, *last, *tmp;

		    flag = 0;
		    node = *list2;
		    last = NULL;
		    while (node != NULL) {
			if (QDDB_KEYLIST_SAMEATTR(*list1, node)) {
			    flag = 1;
			    if (last == NULL) {
				tmp = node;
				node = node->next;
				*list2 = node;
			    } else {
				last->next = node->next;
				tmp = node;
				node = node->next;
			    }
			    KeyListAddToList(ptr, tmp, flags);
			} else {
			    last = node;
			    node = node->next;
			}
		    }
		    if (flag == 0) {
			node = *ptr;
			while (node != NULL) {
			    if (QDDB_KEYLIST_SAMEATTR(*list1, node)) {
				flag = 1;
				break;
			    }
			    node = node->next;
			}
		    }
		    node = KeyListGetNode(&list1, flags);
		    if (flag == 1)
			KeyListAddToList(ptr, node, flags);
		    else
			Qddb_Free(QDDB_TYPE_KEYLIST, node);
		}
	    } else {
		/* Non-Exact intersection */
		while (*list1 != NULL) {
		    int		flag;
		    KeyList	*node, *last, *tmp;

		    flag = 0;
		    node = *list2;
		    last = NULL;
		    while (node != NULL) {
			if (QDDB_KEYLIST_SAMEENTRY(*list1, node)) {
			    flag = 1;
			    if (last == NULL) {
				tmp = node;
				node = node->next;
				*list2 = node;
			    } else {
				last->next = node->next;
				tmp = node;
				node = node->next;
			    }
			    KeyListAddToList(ptr, tmp, flags);
			} else {
			    last = node;
			    node = node->next;
			}
		    }
		    if (flag == 0) {
			node = *ptr;
			while (node != NULL) {
			    if (QDDB_KEYLIST_SAMEENTRY(*list1, node)) {
				flag = 1;
				break;
			    }
			    node = node->next;
			}
		    }
		    node = KeyListGetNode(&list1, flags);
		    if (flag == 1)
			KeyListAddToList(ptr, node, flags);
		    else
			Qddb_Free(QDDB_TYPE_KEYLIST, node);
		}
	    }
	}
    }
    return retval;
}


static KeyList **KeyListHashTableExclusion(hash1, hash2, flags)
    KeyList		**hash1, **hash2;
    int			flags;
{
    KeyList		**retval, **ptr;
    KeyList		**list1, **list2;
    int			i;

    /* HashTable operations do not recognize the COPY flag */
    flags &= ~QDDB_KEYLIST_FLAG_COPY;
    ptr = retval = (KeyList **)Calloc(QDDB_KEYLIST_HASH_NUM*sizeof(KeyList *));
    for (i = 0; i < QDDB_KEYLIST_HASH_NUM; i++, hash1++, hash2++, ptr++) {
	if (*hash1 == NULL || *hash2 == NULL) {
	    *ptr = *hash1;
	    Qddb_Free(QDDB_TYPE_KEYLIST, *hash2);
	    *hash1 = NULL;
	    *hash2 = NULL;
	} else {
	    list1 = hash1;
	    list2 = hash2;
	    /* Do a true exclusion, either exact or not */
	    if ((flags & QDDB_KEYLIST_FLAG_EXACT) != 0) {
		/* Exact intersection */
		while (*list1 != NULL) {
		    int		flag;
		    KeyList	*node;

		    flag = 0;
		    node = *list2;
		    while (node != NULL) {
			if (QDDB_KEYLIST_SAMEATTR(*list1, node)) {
			    flag = 1;
			    break;
			} else
			    node = node->next;
		    }
		    node = KeyListGetNode(&list1, flags);
		    if (flag == 0)
			KeyListAddToList(ptr, node, flags);
		    else
			Qddb_Free(QDDB_TYPE_KEYLIST, node);
		}
	    } else {
		/* Non-Exact exclusion */
		while (*list1 != NULL) {
		    int		flag;
		    KeyList	*node;

		    flag = 0;
		    node = *list2;
		    while (node != NULL) {
			if (QDDB_KEYLIST_SAMEENTRY(*list1, node)) {
			    flag = 1;
			    break;
			} else
			    node = node->next;
		    }
		    node = KeyListGetNode(&list1, flags);
		    if (flag == 0)
			KeyListAddToList(ptr, node, flags);
		    else
			Qddb_Free(QDDB_TYPE_KEYLIST, node);
		}
	    }
	}
    }
    return retval;
}


/* Qddb_KeyListConcat -- Concat list2 onto list1; *last points to the last 
 *	element of list1 or is NULL.  If NULL, list1 is traversed to
 *	find the last element.
 */
KeyList *Qddb_KeyListConcat(list1, list2, last)
    KeyList		*list1, *list2, **last;
{
    KeyList		*ptr;

    if (list1 == NULL) {
	if (list2 != NULL) {
	    ptr = list2;
	    while (ptr->next != NULL)
		ptr = ptr->next;
	    *last = ptr;
	}
	return list2;
    }
    if (list2 == NULL) {
	if (list1 != NULL) {
	    ptr = list1;
	    while (ptr->next != NULL)
		ptr = ptr->next;
	    *last = ptr;
	}
	return list1;
    }
    if (*last == NULL) {
	ptr = list1;
	while (ptr->next != NULL)
	    ptr = ptr->next;
    } else
	ptr = *last;
    ptr->next = list2;
    while (ptr->next != NULL)
	ptr = ptr->next;
    *last = ptr;
    return list1;

}

static KeyList *KeyListHashTable(hash)
    KeyList		**hash;
{
    KeyList		*retval, *last = NULL;
    int			i;

    retval = NULL;
    for (i = 0; i < QDDB_KEYLIST_HASH_NUM; i++) {
	retval = Qddb_KeyListConcat(retval, hash[i], &last);
	hash[i] = NULL;
    }
    return retval;
}

static void KeyListFreeHashTable(hash)
    KeyList		**hash;
{
    int			i;

    for (i = 0; i < QDDB_KEYLIST_HASH_NUM; i++)
	Qddb_Free(QDDB_TYPE_KEYLIST, hash[i]);
    Free(hash);
}

static KeyList *KeyListUnion(schema, List1, List2, flags)
    Schema		*schema;
    KeyList		*List1, *List2;
    int			flags;
{
    KeyList		*retval, **hash_table1, **hash_table2, **hash_table3;

    if (List1 == NULL)
	return (KeyList *)Qddb_KeyListProcess(schema, List2, NULL, QDDB_KEYLIST_PROC_NULLOP, flags);
    if (List2 == NULL)
	return (KeyList *)Qddb_KeyListProcess(schema, List1, NULL, QDDB_KEYLIST_PROC_NULLOP, flags);
    hash_table1 = KeyListBuildHashTable(&List1, flags);
    hash_table2 = KeyListBuildHashTable(&List2, flags);
    hash_table3 = KeyListHashTableUnion(hash_table1, hash_table2, flags);
    KeyListFreeHashTable(hash_table1);
    KeyListFreeHashTable(hash_table2);
    /* hash_table3 contains the new hash table */
    retval = KeyListHashTable(hash_table3);
    KeyListFreeHashTable(hash_table3);
    return retval;
}

static KeyList *KeyListIntersection(schema, List1, List2, flags)
    Schema		*schema;
    KeyList		*List1, *List2;
    int			flags;
{
    KeyList		*retval, **hash_table1, **hash_table2, **hash_table3;

    if (List1 == NULL || List2 == NULL) {
	if ((flags & QDDB_KEYLIST_FLAG_COPY) == 0) {
	    if (List1 != NULL)
		Qddb_Free(QDDB_TYPE_KEYLIST, List1);
	    if (List2 != NULL)
		Qddb_Free(QDDB_TYPE_KEYLIST, List2);
	}
	return NULL;
    }
    hash_table1 = KeyListBuildHashTable(&List1, flags);
    hash_table2 = KeyListBuildHashTable(&List2, flags);
    hash_table3 = KeyListHashTableIntersection(hash_table1, hash_table2, flags);
    KeyListFreeHashTable(hash_table1);
    KeyListFreeHashTable(hash_table2);
    retval = KeyListHashTable(hash_table3);
    KeyListFreeHashTable(hash_table3);
    return retval;
}

static KeyList *KeyListExclusion(schema, List1, NotList, flags)
    Schema		*schema;
    KeyList        	*List1, *NotList;
    int			flags;
{
    KeyList		*retval, **hash_table1, **hash_table2, **hash_table3;

    if (List1 == NULL) {
	if ((flags & QDDB_KEYLIST_FLAG_COPY) == 0)
	    Qddb_Free(QDDB_TYPE_KEYLIST, NotList);
	return NULL;
    }
    if (NotList == NULL)
	return (KeyList *)Qddb_KeyListProcess(schema, List1, NULL, QDDB_KEYLIST_PROC_NULLOP, flags);
    hash_table1 = KeyListBuildHashTable(&List1, flags);
    hash_table2 = KeyListBuildHashTable(&NotList, flags);
    hash_table3 = KeyListHashTableExclusion(hash_table1, hash_table2, flags);
    KeyListFreeHashTable(hash_table1);
    KeyListFreeHashTable(hash_table2);
    retval = KeyListHashTable(hash_table3);
    KeyListFreeHashTable(hash_table3);
    if ((flags & QDDB_KEYLIST_FLAG_COPY) == 0) {
	/* Delete anything left over */
	Qddb_Free(QDDB_TYPE_KEYLIST, List1);
	Qddb_Free(QDDB_TYPE_KEYLIST, NotList);
    }
    return retval;
}

static int CompareKeyNodesByRow(schema, Node1, Node2)
    Schema		*schema;
    KeyList		*Node1, *Node2;
{
    int			i, j, numdots, attr1, attr2;
    SchemaNode		*node1, *node2;
    char		buf1[BUFSIZ], buf2[BUFSIZ], *tmpbuf;

    if (Node1 == NULL)
	return 1;
    if (Node2 == NULL)
	return -1;
    /* Nodes with same attribute number have the leaf as the lowest
     * common ancestor.  Just compare the instance and return.
     */
    attr1 = Node1->Attribute;
    attr2 = Node2->Attribute;
    if (attr1 == attr2)
	return strcmp(Node1->Instance, Node2->Instance);
    /* Find the number of places in the lowest common
     * ancestor.
     */
    node1 = schema->Entries+attr1;
    node2 = schema->Entries+attr2;
    if (node1->AncestorNumber[1] != node2->AncestorNumber[1]) {
	return 0; /* highest level mismatches always match in a join */
    } 
    for (i = 2; i < MIN(node1->Level, node2->Level); i++) {
	if (node1->AncestorNumber[i] != node2->AncestorNumber[i]) {
	    break;
	}
    }
    numdots = i-1;
    /* Instance numbers must match for the number of places in
     * the lowest common ancestor.
     */
    strcpy(buf1, Node1->Instance);
    strcpy(buf2, Node2->Instance);
    tmpbuf = buf1;
    j = 0;
    while (*tmpbuf != '\0') {
	if (*tmpbuf == '.') {
	    j++;
	    if (j == numdots) {
		*tmpbuf = '\0';
	    }
	}
	tmpbuf++;
    }
    tmpbuf = buf2;
    j = 0;
    while (*tmpbuf != '\0') {
	if (*tmpbuf == '.') {
	    j++;
	    if (j == numdots) {
		*tmpbuf = '\0';
	    }
	}
	tmpbuf++;
    }
    return strcmp(buf1, buf2);
}

static int MarkPaths(schema, Node, KeyMatrix, MarkedMatrix, AttrArray, KeyTop, AttrLen)
    Schema		*schema;
    KeyList		*Node, ***KeyMatrix;
    int			**MarkedMatrix, *AttrArray, *KeyTop, AttrLen;
{
    int			i, FoundOne;
    KeyList		*tmp;

    if (AttrLen == 0)
	return True;
    FoundOne = False;
    for (i = 0; i < KeyTop[0]; i++) {
	tmp = Node;
	while (tmp != NULL && CompareKeyNodesByRow(schema, tmp, KeyMatrix[0][i]) == 0)
	    tmp = tmp->next;
	if (tmp == NULL) {
	    KeyMatrix[0][i]->next = Node;
	    if (MarkPaths(schema, KeyMatrix[0][i], KeyMatrix+1, MarkedMatrix+1, 
			  AttrArray+1, KeyTop+1, AttrLen-1) == True) {
		MarkedMatrix[0][i] = 1;
		FoundOne = True;
	    }
	    KeyMatrix[0][i]->next = NULL;
	}
    }
    return FoundOne;
}

static KeyList *ProcessListByAttr(schema, List, Length, AttrList, AttrLen)
    Schema		*schema;
    KeyList		*List;
    int			Length;
    char		*AttrList[];
    int			AttrLen;
{
    int			*AttrArray = (int *)Malloc(sizeof(int)*AttrLen);
    int			*KeyTop    = (int *)Calloc(sizeof(int)*AttrLen);
    KeyList		***KeyMatrix = (KeyList ***)Malloc(sizeof(KeyList **)*AttrLen), *TmpList;
    int			**MarkedMatrix = (int **)Malloc(sizeof(int *)*AttrLen);
    int			i, j;

    for (i = 0; i < AttrLen; i++)
	KeyMatrix[i] = (KeyList **)Malloc(sizeof(KeyList *)*Length);
    for (i = 0; i < AttrLen; i++)
	MarkedMatrix[i] = (int *)Calloc(sizeof(int)*Length);
    for (i = 0; i < AttrLen; i++)
	AttrArray[i] = Qddb_ConvertAttributeNameToNumber(schema, AttrList[i]);
    TmpList = List;
    /* Build a matrix of KeyList nodes sorted by attribute number.
     */
    while (TmpList != NULL) {
	for (i = 0; i < AttrLen; i++) {
	    if (AttrArray[i] == TmpList->Attribute) {
		KeyMatrix[i][KeyTop[i]] = TmpList;
		KeyTop[i]++;
                break;
	    }
	}
	TmpList = TmpList->next;
    }
    /* Now we need to mark paths in the matrix.
     */
    for (i = 0; i < KeyTop[0]; i++) {
	KeyMatrix[0][i]->next = NULL;
	if (MarkPaths(schema, KeyMatrix[0][i], KeyMatrix+1, MarkedMatrix+1, 
		      AttrArray+1, KeyTop+1, AttrLen-1) == True)
	    MarkedMatrix[0][i] = 1;
    }
    TmpList = NULL;
    List = NULL;
    /* Finally, prune out unmarked nodes.
     */
    for (i = 0; i < AttrLen; i++) {
	for (j = 0; j < KeyTop[i]; j++) {
	    if (MarkedMatrix[i][j] == 1) {
		KeyMatrix[i][j]->next = NULL;
		if (TmpList == NULL) {
		    TmpList = KeyMatrix[i][j];
		    List = KeyMatrix[i][j];
		} else {
		    TmpList->next = KeyMatrix[i][j];
		    TmpList = TmpList->next;
		}
	    } else {
		Free(KeyMatrix[i][j]);
	    }
	}
	Free(KeyMatrix[i]);
	Free(MarkedMatrix[i]);
    }
    Free(AttrArray);
    Free(KeyTop);
    Free(MarkedMatrix);
    Free(KeyMatrix);
    return List;
}

static KeyList **KeyListSplitList(schema, List)
    Schema		*schema;
    KeyList		*List;
{
    KeyList		**SplitList, *tmp, *tmp2;
    int			len, reallen;

    for (len = 0, tmp = List; tmp != NULL; tmp = tmp->next, len++);
    SplitList = (KeyList **)Malloc(sizeof(KeyList *)*(len+1));
    if (SplitList == NULL)
	PANIC("KeyListSplitList: out of memory");
    reallen = 0;
    for (tmp = List, SplitList[reallen] = List; tmp->next != NULL;) {
	if (!QDDB_KEYLIST_SAMEENTRY(tmp, tmp->next)) {
	    SplitList[++reallen] = tmp->next;
	    tmp2 = tmp;
	    tmp = tmp->next;
	    tmp2->next = NULL;
	    if (tmp->next == NULL)
		break;
	} else
	    tmp = tmp->next;
    }
    if (SplitList[reallen] == NULL)
	reallen--;
    SplitList[reallen+1] = NULL;
    return SplitList;
}

static KeyList *KeyListPruneByRow(schema, List, AttrList, AttrLen)
    Schema		*schema;
    KeyList	      	*List;
    char		*AttrList[];
    int			AttrLen;
{
    KeyList		**SplitList, *tmp;
    int			len, reallen, i;

    if (List == NULL)
	return NULL;
    if (AttrLen == 0)
	return List;
    for (len = 0, tmp = List; tmp != NULL; tmp = tmp->next, len++);
    SplitList = KeyListSplitList(schema, List);
    for (reallen = 0; SplitList[reallen] != NULL; reallen++);
    for (i = 0; i < reallen; i++) {
	SplitList[i] = ProcessListByAttr(schema, SplitList[i], len, AttrList, AttrLen);
    }
    List = NULL;
    /* merge list back together again */
    for (i = 0; i < reallen; i++) {
	tmp = SplitList[i];
	if (tmp == NULL)
	    continue;
	if (List == NULL)
	    List = SplitList[i];
	if (i == reallen)
	    break;
	while (tmp->next != NULL)
	    tmp = tmp->next;
	while (i+1 < reallen && SplitList[i+1] == NULL)
	    i++;
	tmp->next = SplitList[i+1];
    }
    Free(SplitList);
    return List;
}




static void KeyListMarkAttrs(schema, List, ThisEntry)
    Schema		*schema;
    KeyList		*List;
    InCoreEntry		*ThisEntry;
{
    int			Found;
    char		*instance;

    Found = False;
    while (ThisEntry->SequenceNumber != 0) {
	if (List->Attribute == ThisEntry->AttributeNumber) {
	    if (Found == False) {
		instance = InstanceToString(ThisEntry->Instance,
					    (size_t)schema->Entries[ThisEntry->AttributeNumber].Level);
		if (strcmp(List->Instance, instance) == 0) {
		    Found = True;
		    ThisEntry->Marked = True;
		} else {
	            if (ThisEntry->Marked != True) {
			if (QDDB_KEYLIST_ISANY(List) && ThisEntry->Marked != False)
			    ThisEntry->Marked = Inapplicable;
			else
			    ThisEntry->Marked = False;
		    }
		}
		Free(instance);
	    } else {
	        if (ThisEntry->Marked != True) {
		    if (QDDB_KEYLIST_ISANY(List) && ThisEntry->Marked != False)
			ThisEntry->Marked = Inapplicable;
		    else 
			ThisEntry->Marked = False;
		}	
	    }
	}
	ThisEntry++;
    }
}

/* KeyListMarkEntry -- Mark all searched on entries as either True or False.
 * False indicates that a match on this attr failed.  All other nodes
 * are marked True.  Returns the portion of the remainder of the list.
 */
static KeyList *KeyListMarkEntry(schema, List, ThisEntry)
    Schema		*schema;
    KeyList		*List;
    InCoreEntry		*ThisEntry;
{
    KeyList		*retval, *BackPtr;
    InCoreEntry		*TmpEntry;

    if (List == NULL)
	return NULL;
    retval = List;
    BackPtr = List;
    while (retval != NULL && QDDB_KEYLIST_SAMEENTRY(List, retval)) {
	BackPtr = retval;
	retval = retval->next;
    }
    if (retval == BackPtr)
	retval = retval->next;
    BackPtr->next = NULL;
    /* Three stages:
     * 	1. Mark all leaves as uninitialized (-1)
     *  2. For each node in KeyList
     *		Mark matching node as True, all other unTrue nodes False.
     *  3. Mark all uninitialized nodes Inapplicable.
     */
    /* 1. */
    TmpEntry = ThisEntry;
    while (TmpEntry->SequenceNumber != 0) {
	TmpEntry->Marked = -1;
	TmpEntry++;
    }
    /* 2. */
    while (List != NULL) {
	KeyListMarkAttrs(schema, List, ThisEntry);
	BackPtr = List;
	List = List->next;
	BackPtr->next = NULL;
	Qddb_Free(QDDB_TYPE_KEYLIST, BackPtr);
    }
    /* 3. */
    TmpEntry = ThisEntry;
    while (TmpEntry->SequenceNumber != 0) {
	if (TmpEntry->Marked == -1) {
	    TmpEntry->Marked = Inapplicable;
	}
	TmpEntry++;
    }
    return retval;
}

#if defined(notyet)
static void KeyListMarkAsAny(list)
    KeyList		*list;
{
    while (list != NULL) {
	QDDB_KEYLIST_SET_ANY(list);
	list = list->next;
    }
}
#endif

static KeyList *PruneListByAttribute(schema, list, attr)
    Schema			*schema;
    KeyList			*list;
    char			*attr;
{
    int				attr_num;
    KeyList			*last_place, *place, *retval;

    attr_num = Qddb_ConvertAttributeNameToNumber(schema, attr);
    if (attr_num == -1) {
	Qddb_Free(QDDB_TYPE_KEYLIST, list);
	return NULL;
    }
    last_place = NULL;
    place = retval = list;
    while (place != NULL) {
	if (place->Attribute != attr_num) {
	    /* prune this KeyList node */
	    if (last_place == NULL) {
		retval = place->next;
		Free(place->Instance);
		Free(place);
		place = retval;
	    } else {
		last_place->next =  place->next;
		Free(place->Instance);
		Free(place);
		place = last_place->next;
	    }
	} else {
	    last_place = place;
	    place = place->next;
	}
    }
    return retval;
}
