/* Copyright (c) 1991
 *      Juergen Weigert (jnweiger@immd4.informatik.uni-erlangen.de)
 *      Michael Schroeder (mlschroe@immd4.informatik.uni-erlangen.de)
 * Copyright (c) 1987 Oliver Laumann
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file COPYING); if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Noteworthy contributors to screen's design and implementation:
 *	Wayne Davison (davison@borland.com)
 *	Patrick Wolfe (pat@kai.com, kailand!pat)
 *	Bart Schaefer (schaefer@cse.ogi.edu)
 *	Nathan Glasser (nathan@brokaw.lcs.mit.edu)
 *	Larry W. Virden (lvirden@cas.org)
 *	Howard Chu (hyc@hanauma.jpl.nasa.gov)
 *	Tim MacKenzie (tym@dibbler.cs.monash.edu.au)
 *	Markku Jarvinen (mta@{cc,cs,ee}.tut.fi)
 *	Marc Boucher (marc@CAM.ORG)
 *
 ****************************************************************
 */

#include "rcs.h"
RCS_ID("$Id: screen.c,v 1.54 92/12/01 16:17:57 jnweiger Exp Locker: jnweiger $ FAU")


#include <sys/param.h>
/* #include <signal.h> */
#include <ctype.h>
#ifdef __sgi
# include <stdio.h> /* needed before pwd.h to avoid ansi compiler whining */
#endif /* __sgi */
#include <pwd.h>
#include <fcntl.h>
#ifdef sgi
# include <sys/sysmacros.h>
#endif /* sgi */
#if !defined(sun) && !defined(B43) && !defined(ISC) && !defined(pyr)
# include <time.h>
#endif
#include <sys/time.h>
#if defined(M_XENIX) || defined(M_UNIX)
#include <sys/select.h> /* for timeval */
#endif
#include <sys/types.h>
#ifdef ISC
# include <sys/bsdtypes.h>
#endif
#include <sys/stat.h>
#ifndef sgi
# include <sys/file.h>
#endif /* sgi */
#ifndef sun
# include <sys/ioctl.h>
#endif /* sun */

#include <signal.h>

#if defined(SVR4) && !defined(NSIG)
# define NSIG 32
#endif /* SVR4 && !NSIG */

#include "config.h"

#ifdef SHADOWPW
# include <shadow.h>
#endif /* SHADOWPW */

#ifdef SVR4
# include <sys/stropts.h>
#endif

#ifdef SYSV
# include <sys/utsname.h>
#endif

#if defined(sequent) || defined(SVR4)
# include <sys/resource.h>
#endif /* sequent || SVR4 */

#ifdef ISC
# include <sys/tty.h>
# include <sys/sioctl.h>
# include <sys/pty.h>
#endif /* ISC */

#include "screen.h"

#include "patchlevel.h"

#if defined(xelos) || defined(m68k) || defined(M_XENIX)
 struct passwd *getpwuid __P((uid_t));
 struct passwd *getpwnam __P((char *));
#endif

#ifdef USEVARARGS
# if defined(__STDC__)
#  include <stdarg.h>
# else
#  include <varargs.h>
# endif
#endif

#ifdef DEBUG
FILE *dfp;
#endif


extern char *blank, *null, Term[], screenterm[], **environ, Termcap[];
int force_vt = 1, assume_LP = 0;
extern struct display *display; 
extern struct layer BlankLayer;
int VBellWait, MsgWait, MsgMinWait;

extern char *expand_vars __P((char *));

/* tty.c */
extern int intrc, origintrc;


extern int use_hardstatus;
#ifndef linux	/* all done in <errno.h> */
extern int errno;
extern sys_nerr;
extern char *sys_errlist[];
#endif /* linux */
#ifdef COPY_PASTE
extern char mark_key_tab[];
#endif
extern char version[];
extern char DefaultShell[];


char *ShellProg;
char *ShellArgs[2];

extern struct NewWindow nwin_undef, nwin_default, nwin_options;

static char *MakeWinMsg __P((char *, int));
static void  MakeNewEnv __P((void));
static void  SigChldHandler __P((void));
static sig_t SigChld __P(SIGPROTOARG);
static sig_t SigInt __P(SIGPROTOARG);
static sig_t CoreDump __P((int));
static void  DoWait __P((void));


#ifdef PASSWORD
extern char Password[];
#endif


/* the attacher */
struct passwd *ppp;
char *attach_tty;
char *attach_term;
char *LoginName;
struct mode attach_Mode;


#ifdef SOCKDIR
char *SockDir = SOCKDIR;
#else
char *SockDir = ".iscreen";
#endif
extern char SockPath[], *SockNamePtr, *SockName;
int ServerSocket = -1;
char **NewEnv;

char *RcFileName = NULL;
extern char Esc;
char *home;

char *BellString;
char *VisualBellString;
char *ActivityString;
#ifdef COPY_PASTE
char *BufferFile;
#endif
#ifdef POW_DETACH
char *PowDetachString;
#endif
int auto_detach = 1;
int iflag, rflag, dflag, lsflag, quietflag, wipeflag, xflag;
int adaptflag;

#ifdef MULTIUSER
char *multi;
char *multi_home;
int multi_uid;
int own_uid;
int multiattach;
int tty_mode;
int tty_oldmode = -1;
#endif

char HostName[MAXSTR];
int MasterPid;
int real_uid, real_gid, eff_uid, eff_gid;
int default_startup;
int slowpaste;

#ifdef NETHACK
int nethackflag = 0;
#endif


struct display *displays;
struct win *fore = NULL;
struct win *windows = NULL;


/*
 * Do this last
 */
#include "extern.h"

/*
 * XXX: Missing system header files.
 */
#ifdef USEVARARGS
# ifndef VPRNT_DECLARED
int vsprintf __P((char *, char *, va_list));
# endif /* VPRNT_DECLARED */
#endif
#ifndef SELECT_DECLARED
int select __P((int, fd_set *, fd_set *, fd_set *, struct timeval *));
#endif

#ifdef NETHACK
char strnomem[] = "Who was that Maude person anyway?";
#else
char strnomem[] = "Out of memory.";
#endif


/*
 *  ====> tty.c
 */

/*ARGSUSED*/
void
brktty(fd)
int fd;
{
#ifdef POSIX
  setsid();		/* will break terminal affiliation */
# ifdef BSD
  ioctl(fd, TIOCSCTTY, 0);
# endif /* BSD */
#else /* POSIX */
# ifdef SYSV
  setpgrp();		/* will break terminal affiliation */
# else /* SYSV */
#  ifdef BSDJOBS
  int devtty;

  if ((devtty = open("/dev/tty", O_RDWR | O_NDELAY)) >= 0)
    {
      if (ioctl(devtty, TIOCNOTTY, (char *) 0))
        debug2("brktty: ioctl(devtty=%d, TIOCNOTTY, 0) = %d\n", devtty, errno);
      close(devtty);
    }
#  endif /* BSDJOBS */
# endif /* SYSV */
#endif /* POSIX */
}

void
freetty()
{
  close(userfd);
  debug1("did freetty %d\n", userfd);
  userfd = -1;
  obufp = 0;
  obuffree = 0;
  if (obuf)
    free(obuf);
  obuf = 0;
  obuflen = 0;
}

int
fgtty(fd)
int fd;
{
#ifdef BSDJOBS
  int mypid;

  mypid = getpid();

# ifdef BSDI
  setsid();	/* should be obsolete */
  ioctl(fd, TIOCSCTTY, 0);
# endif /* BSDI */

# ifdef POSIX
  if (tcsetpgrp(fd, mypid))
    {
      debug1("fgtty: tcsetpgrp: %d\n", errno);
      return -1;
    }
# else /* POSIX */
  if (ioctl(fd, TIOCSPGRP, &mypid) != 0)
    debug1("fgtty: TIOSETPGRP: %d\n", errno);
  if (setpgrp(fd, mypid))
    debug1("fgtty: setpgrp: %d\n", errno);
# endif /* POSIX */
#endif /* BSDJOBS */
  return 0;
}

  
static int InterruptPlease = 0;
static int GotSigChld;


void
main(ac, av)
int ac;
char **av;
{
  register int n, len;
  register struct win *p;
  char *ap;
  char *av0;
  char socknamebuf[2 * MAXSTR];
  fd_set r, w;
  int mflag = 0;
  struct timeval tv;
  int nsel;
  char buf[IOSIZE], *myname = (ac == 0) ? "screen" : av[0];
  struct stat st;
  int buflen, tmp;
#ifdef _MODE_T			/* (jw) */
  mode_t oumask;
#else
  int oumask;
#endif
#ifdef SYSV
  struct utsname utsnam;
#endif
  struct NewWindow nwin;
  int detached = 0;		/* start up detached */
  struct display *ndisplay;
#ifdef MULTIUSER
  char *sockp;
#endif

  /*
   *  First, close all unused descriptors
   *  (otherwise, we might have problems with the select() call)
   */
  closeallfiles();
#ifdef DEBUG
  (void) mkdir("/tmp/debug", 0777);
  if ((dfp = fopen("/tmp/debug/screen.front", "w")) == NULL)
    dfp = stderr;
  else
    (void) chmod("/tmp/debug/screen.front", 0666);
#endif
  sprintf(version, "%d.%.2d.%.2d%s (%s) %s", REV, VERS,
	  PATCHLEVEL, STATE, ORIGIN, DATE);
  debug2("-- screen debug started %s (%s)\n", *av, version);
#ifdef POSIX
  debug("POSIX\n");
#endif
#ifdef TERMIO
  debug("TERMIO\n");
#endif
#ifdef SYSV
  debug("SYSV\n");
#endif
#ifdef NAMEDPIPE
  debug("NAMEDPIPE\n");
#endif
#if defined(SIGWINCH) && defined(TIOCGWINSZ)
  debug("Window changing enabled\n");
#endif
#ifdef NOREUID
  debug("NOREUID\n");
#endif
#ifdef hpux
  debug("hpux\n");
#endif
#ifdef USEBCOPY
  debug("USEBCOPY\n");
#endif
#ifdef UTMPOK
  debug("UTMPOK\n");
#endif
#ifdef LOADAV
  debug("LOADAV\n");
#endif
#ifdef NETHACK
  debug("NETHACK\n");
#endif
#ifdef TERMINFO
  debug("TERMINFO\n");
#endif
#ifdef SHADOWPW
  debug("SHADOWPW\n");
#endif
#ifdef NAME_MAX
  debug1("NAME_MAX = %d\n", NAME_MAX);
#endif

  BellString = SaveStr("Bell in window %");
  VisualBellString = SaveStr("   Wuff,  Wuff!!  ");
  ActivityString = SaveStr("Activity in window %");
#ifdef COPY_PASTE
  BufferFile = SaveStr(DEFAULT_BUFFERFILE);
#endif
  ShellProg = NULL;
#ifdef POW_DETACH
  PowDetachString = 0;
#endif
  default_startup = (ac > 1) ? 0 : 1;
  adaptflag = 0;
  slowpaste = 0;
  VBellWait = VBELLWAIT;
  MsgWait = MSGWAIT;
  MsgMinWait = MSGMINWAIT;
#ifdef COPY_PASTE
  CompileKeys((char *)NULL, mark_key_tab);
#endif
  nwin = nwin_undef;
  nwin_options = nwin_undef;

  av0 = *av;
  while (ac > 0)
    {
      ap = *++av;
      if (--ac > 0 && *ap == '-')
	{
	  switch (ap[1])
	    {
	    case 'a':
	      nwin_options.aflag = 1;
	      break;
	    case 'A':
	      adaptflag = 1;
	      break;
	    case 'c':
	      if (ap[2])
		RcFileName = ap + 2;
	      else
		{
		  if (--ac == 0)
		    exit_with_usage(myname);
		  RcFileName = *++av;
		}
	      break;
	    case 'e':
	      if (ap[2])
		ap += 2;
	      else
		{
		  if (--ac == 0)
		    exit_with_usage(myname);
		  ap = *++av;
		}
	      if (ParseEscape(ap))
		Panic(0, "Two characters are required with -e option.");
	      break;
	    case 'f':
	      switch (ap[2])
		{
		case 'n':
		case '0':
		  nwin_options.flowflag = FLOW_NOW * 0;
		  break;
		case 'y':
		case '1':
		case '\0':
		  nwin_options.flowflag = FLOW_NOW * 1;
		  break;
		case 'a':
		  nwin_options.flowflag = FLOW_AUTOFLAG;
		  break;
		default:
		  exit_with_usage(myname);
		}
	      break;
            case 'h':
	      if (ap[2])
		nwin_options.histheight = atoi(ap + 2);
	      else
		{
		  if (--ac == 0)
		    exit_with_usage(myname);
		  nwin_options.histheight = atoi(*++av);
		}
	      if (nwin_options.histheight < 0)
		exit_with_usage(myname);
	      break;
	    case 'i':
	      iflag = 1;
	      break;
	    case 't': /* title is a synonym for AkA */
	    case 'k':
	      if (ap[2])
		nwin_options.aka = ap + 2;
	      else
		{
		  if (--ac == 0)
		    exit_with_usage(myname);
		  nwin_options.aka = *++av;
		}
	      break;
	    case 'l':
	      switch (ap[2])
		{
		case 'n':
		case '0':
		  nwin_options.lflag = 0;
		  break;
		case 'y':
		case '1':
		case '\0':
		  nwin_options.lflag = 1;
		  break;
		case 's':
		case 'i':
		  lsflag = 1;
		  if (ac > 1)
		    {
		      SockName = *++av;
		      ac--;
		    }
		  break;
		default:
		  exit_with_usage(myname);
		}
	      break;
	    case 'w':
	      lsflag = 1;
	      wipeflag = 1;
	      break;
	    case 'L':
	      assume_LP = 1;
	      break;
	    case 'm':
	      mflag = 1;
	      break;
	    case 'O':
	      force_vt = 0;
	      break;
	    case 'T':
              if (ap[2])
		{
		  if (strlen(ap+2) < 20)
                    strcpy(screenterm, ap + 2);
		}
              else
                {
                  if (--ac == 0)
                    exit_with_usage(myname);
		  if (strlen(*++av) < 20)
                    strcpy(screenterm, *av);
                }
	      nwin_options.term = screenterm;
              break;
	    case 'q':
	      quietflag = 1;
	      break;
	    case 'r':
	    case 'R':
#ifdef MULTI
	    case 'x':
#endif
	      if (ap[2])
		{
		  SockName = ap + 2;
		  if (ac != 1)
		    exit_with_usage(myname);
		}
	      else if (ac > 1 && *av[1] != '-')
		{
		  SockName = *++av;
		  ac--;
		}
#ifdef MULTI
	      if (ap[1] == 'x')
		xflag = 1;
	      else
#endif
	        rflag = (ap[1] == 'r') ? 1 : 2;
	      break;
#ifdef REMOTE_DETACH
	    case 'd':
	      dflag = 1;
	      /* FALLTHRU */
	    case 'D':
	      if (!dflag)
		dflag = 2;
	      if (ap[2])
		SockName = ap + 2;
	      if (ac == 2)
		{
		  if (*av[1] != '-')
		    {
		      SockName = *++av;
		      ac--;
		    }
		}
	      break;
#endif
	    case 's':
	      if (ap[2])
		{
		  if (ShellProg)
		    Free(ShellProg);
		  ShellProg = SaveStr(ap + 2);
		}
	      else
		{
		  if (--ac == 0)
		    exit_with_usage(myname);
		  if (ShellProg)
		    Free(ShellProg);
		  ShellProg = SaveStr(*++av);
		}
	      debug1("ShellProg: '%s'\n", ShellProg);
	      break;
	    case 'S':
	      if (ap[2])
		SockName = ap + 2;
	      else
		{
		  if (--ac == 0)
		    exit_with_usage(myname);
		  SockName = *++av;
		  if (!*SockName)
		    exit_with_usage(myname);
		}
	      break;
	    case 'v':
	      Panic(0, "Screen version %s", version);
	      /* NOTREACHED */
	    default:
	      exit_with_usage(myname);
	    }
	}
      else
	break;
    }
  if (dflag && mflag && SockName && !(rflag || xflag))
    detached = 1;
  nwin = nwin_options;
  if (ac)
    nwin.args = av;
  real_uid = getuid();
  real_gid = getgid();
  eff_uid = geteuid();
  eff_gid = getegid();
  if (eff_uid != real_uid)
    {		
      /* if running with s-bit, we must install a special signal
       * handler routine that resets the s-bit, so that we get a
       * core file anyway.
       */
#ifdef SIGBUS /* OOPS, linux has no bus errors ??? */
      signal(SIGBUS, CoreDump);
#endif /* SIGBUS */
      signal(SIGSEGV, CoreDump);
    }
  if (!ShellProg)
    {
      register char *sh;

      sh = getenv("SHELL");
      ShellProg = SaveStr(sh ? sh : DefaultShell);
    }
  ShellArgs[0] = ShellProg;
#ifdef NETHACK
  nethackflag = (getenv("NETHACKOPTIONS") != NULL);
#endif
#ifdef MULTIUSER
  own_uid = multi_uid = real_uid;
  if (SockName && (sockp = index(SockName, '/')))
    {
      if (eff_uid)
        Panic(0, "Must run suid root for multi support.");
      *sockp = 0;
      multi = SockName;
      SockName = sockp + 1;
      if (*multi)
	{
	  if ((ppp = getpwnam(multi)) == (struct passwd *) 0)
	    Panic(0, "Cannot identify account '%s'.", multi);
	  multi_uid = ppp->pw_uid;
	  multi_home = SaveStr(ppp->pw_dir);
#ifdef MULTI
	  xflag = 1;
	  rflag = 0;
#else
	  rflag = 1;
#endif
	  detached = 0;
	  multiattach = 1;
	}
    }
  if (SockName && *SockName == 0)
    SockName = 0;
#endif
  if ((LoginName = getlogin()) && LoginName[0] != '\0')
    {
      if ((ppp = getpwnam(LoginName)) != (struct passwd *) 0)
	if (ppp->pw_uid != real_uid)
	  ppp = (struct passwd *) 0;
    }
  if (ppp == 0)
    {
      if ((ppp = getpwuid(real_uid)) == 0)
        {
#ifdef NETHACK
          if (nethackflag)
	    Panic(0, "An alarm sounds through the dungeon...\nWarning, the kops are coming.");
	  else
#endif
	  Panic(0, "getpwuid() can't identify your account!");
	  exit(1);
        }
      LoginName = ppp->pw_name;
    }
  home = getenv("HOME");	/* may or may not return a result. jw. */
#if !defined(SOCKDIR) && defined(MULTIUSER)
  if (multi && !multiattach)
    {
      if (home && strcmp(home, ppp->pw_dir))
        Panic(0, "$HOME must match passwd entry for multi screens");
    }
#endif
  if (home == 0 || *home == '\0')
    home = ppp->pw_dir;
  if (strlen(LoginName) > 20)
    Panic(0, "LoginName too long - sorry.");
  if (strlen(home) > MAXPATH - 25)
    Panic(0, "$HOME too long - sorry.");
#ifdef PASSWORD
  strcpy(Password, ppp->pw_passwd);
#endif

  if (!detached && !lsflag)
    {
      /* ttyname implies isatty */
      if (!(attach_tty = ttyname(0)))
	{
#ifdef NETHACK
	  if (nethackflag)
	    Panic(0, "You must play from a terminal.");
	  else
#endif
	  Panic(0, "Must be connected to a terminal.");
	  exit(1);
	}
      if (strlen(attach_tty) >= MAXPATH)
	Panic(0, "TtyName too long - sorry.");
      if (stat(attach_tty, &st))
	Panic(errno, "Cannot access '%s'", attach_tty);
#ifdef MULTIUSER
      tty_mode = st.st_mode & 0777;
#endif
      if ((n = secopen(attach_tty, O_RDWR, 0)) < 0)
	Panic(0, "Cannot open '%s' - please check.", attach_tty);
      close(n);
      debug1("attach_tty is %s\n", attach_tty);
      if ((attach_term = getenv("TERM")) == 0)
	Panic(0, "Please set a terminal type.");
      if (strlen(attach_term) > sizeof(termname) - 1)
	Panic(0, "$TERM too long - sorry.");
      GetTTY(0, &attach_Mode);
#ifdef DEBUG
      DebugTTY(&attach_Mode);
#endif /* DEBUG */
    }
  
#ifdef _MODE_T
  oumask = umask(0);		/* well, unsigned never fails? jw. */
#else
  if ((oumask = umask(0)) == -1)
    Panic(errno, "Cannot change umask to zero");
#endif
  if ((SockDir = getenv("ISCREENDIR")) == NULL)
    SockDir = getenv("SCREENDIR");
  if (SockDir)
    {
      if (strlen(SockDir) >= MAXPATH - 1)
	Panic(0, "Ridiculously long $(I)SCREENDIR - try again.");
#ifdef MULTIUSER
      if (multi)
	Panic(0, "No $(I)SCREENDIR with multi screens, please.");
#endif
    }
#ifdef MULTIUSER
  if (multiattach)
    {
# ifndef SOCKDIR
      sprintf(SockPath, "%s/.iscreen", multi_home);
      SockDir = SockPath;
# else
      SockDir = SOCKDIR;
      sprintf(SockPath, "%s/S-%s", SockDir, multi);
# endif
    }
  else
#endif
    {
#ifndef SOCKDIR
      if (SockDir == 0)
	{
	  sprintf(SockPath, "%s/.iscreen", home);
	  SockDir = SockPath;
	}
#endif
      if (SockDir)
	{
	  if (access(SockDir, F_OK))
	    {
	      if (UserContext() > 0)
		{
		  if (mkdir(SockDir, 0700))
		    UserReturn(0);
		  UserReturn(1);
		}
	      if (UserStatus() <= 0)
		Panic(0, "Cannot make directory '%s'.", SockDir);
	    }
	  if (SockDir != SockPath)
	    strcpy(SockPath, SockDir);
	}
#ifdef SOCKDIR
      else
	{
	  SockDir = SOCKDIR;
	  if (stat(SockDir, &st))
	    {
	      if (mkdir(SockDir, eff_uid ? 0777 : 0755) == -1)
		Panic(errno, "Cannot make directory '%s'", SockDir);
	    }
	  else
	    {
	      n = eff_uid ? 0777 : 0755;
	      if ((st.st_mode & 0777) != n)
		Panic(0, "Directory '%s' must have mode %03o.", SockDir, n);
	    }
	  sprintf(SockPath, "%s/S-%s", SockDir, LoginName);
	  if (access(SockPath, F_OK))
	    {
	      if (mkdir(SockPath, 0700) == -1)
		Panic(errno, "Cannot make directory '%s'", SockPath);
	      (void) chown(SockPath, real_uid, real_gid);
	    }
	}
#endif
    }

  if (stat(SockPath, &st) == -1)
    {
      Panic(errno, "Cannot access %s", SockPath);
    }
  else
    {
#ifdef _POSIX_SOURCE
      if (S_ISDIR(st.st_mode) == 0)
#else
      if ((st.st_mode & S_IFMT) != S_IFDIR)
#endif
	Panic(0, "%s is not a directory.", SockPath);
#ifdef MULTIUSER
      if (multi)
	{
	  if (st.st_uid != multi_uid)
	    Panic(0, "%s is not the owner of %s.", multi, SockPath);
	}
      else
#endif
	{
	  if (st.st_uid != real_uid)
	    Panic(0, "You are not the owner of %s.", SockPath);
	}
      if ((st.st_mode & 0777) != 0700)
	Panic(0, "Directory %s must have mode 700.", SockPath);
    }
  strcat(SockPath, "/");
  SockNamePtr = SockPath + strlen(SockPath);
  (void) umask(oumask);
  debug2("SockPath: %s  SockName: %s\n", SockPath, SockName ? SockName : "NULL");

#if defined(SYSV) && !defined(ISC)
  if (uname(&utsnam) == -1)
    Panic(0, "uname() failed, errno = %d.", errno);
  else
    {
      strncpy(HostName, utsnam.nodename, MAXSTR);
      HostName[(sizeof(utsnam.nodename) <= MAXSTR) ? 
               sizeof(utsnam.nodename) : MAXSTR] = '\0';
    }
#else
  (void) gethostname(HostName, MAXSTR);
#endif
  HostName[MAXSTR - 1] = '\0';
  if ((ap = index(HostName, '.')) != NULL)
    *ap = '\0';

  if (lsflag)
    {
      int i;

#ifdef MULTIUSER
      if (multi)
	real_uid = multi_uid;
      setuid(real_uid);
      setgid(real_gid);
      eff_uid = real_uid;
      eff_gid = real_gid;
#endif
      i = FindSocket(0, (int *)NULL);
      /* MakeClientSocket appended the last (Sock)Name there: */
      *SockNamePtr = '\0';
      if (i == 0)
	{
#ifdef NETHACK
          if (nethackflag)
	    Panic(0, "This room is empty (%s).\n", SockPath);
          else
#endif /* NETHACK */
          Panic(0, "No Sockets found in %s.\n", SockPath);
        }
      Panic(0, "%d Socket%s in %s.\n", i, i > 1 ? "s" : "", SockPath);
      /* NOTREACHED */
    }
  if (rflag || xflag)
    {
      debug("screen -r: - is there anybody out there?\n");
#ifdef SHADOWPW
      setspent();  /* open shadow file while we are still root */
#endif /* SHADOWPW */
      if (Attach(MSG_ATTACH))
	{
	  Attacher();
	  /* NOTREACHED */
	}
      debug("screen -r: backend not responding -- still crying\n");
    }
  else if (dflag && !mflag)
    {
      (void) Attach(MSG_DETACH);
      Msg(0, "[%s %sdetached.]\n", SockName, (dflag > 1 ? "power " : ""));
      eexit(0);
      /* NOTREACHED */
    }
  if (!SockName && !mflag)
    {
      register char *sty;
      int s;

      if ((sty = getenv("STY")) != 0 && *sty != '\0')
	{
	  setuid(real_uid);
	  setgid(real_gid);
	  eff_uid = real_uid;
	  eff_gid = real_gid;
	  if ((s = MakeClientSocket(1, sty)) > 0)
	    {
	      nwin_options.args = av;
	      SendCreateMsg(s, &nwin);
	      close(s);
	    }
	  exit(0);
	  /* NOTREACHED */
	}
    }
  nwin_compose(&nwin_default, &nwin_options, &nwin_default);
  if (SockName && !*SockName)
    SockName = NULL;
  switch (MasterPid = fork())
    {
    case -1:
      Panic(errno, "fork");
      /* NOTREACHED */
    case 0:
      break;
    default:
      if (detached)
        exit(0);
      if (SockName)
	{
	  /* user started us with -S option */
	  sprintf(socknamebuf, "%d.%s", MasterPid, SockName);
	}
      else
	{
	  sprintf(socknamebuf, "%d.%s.%s", MasterPid, stripdev(attach_tty),
		  HostName);
	}
      for (ap = socknamebuf; *ap; ap++)
	if (*ap == '/')
	  *ap = '-';
      SockName = socknamebuf;
#ifdef SHADOWPW
      setspent();  /* open shadow file while we are still root */
#endif /* SHADOWPW */
      setuid(real_uid);
      setgid(real_gid);
      eff_uid = real_uid;
      eff_gid = real_gid;
      Attacher();
      /* NOTREACHED */
    }

  ap = av0 + strlen(av0) - 1;
  while (ap >= av0)
    {
      if (!strncmp("screen", ap, 6))
	{
	  strncpy(ap, "SCREEN", 6); /* name this process "SCREEN-BACKEND" */
	  break;
	}
      ap--;
    }
  if (ap < av0)
    *av0 = 'S';

#ifdef DEBUG
  if (dfp != stderr)
    fclose(dfp);
  if ((dfp = fopen("/tmp/debug/screen.back", "w")) == NULL)
    dfp = stderr;
  else
    (void) chmod("/tmp/debug/screen.back", 0666);
#endif
  if (!detached)
    n = dup(0);
  else
    n = -1;
  freopen("/dev/null", "r", stdin);
  freopen("/dev/null", "w", stdout);
#ifdef DEBUG
  if (dfp != stderr)
#endif
  freopen("/dev/null", "w", stderr);
  debug("-- screen.back debug started\n");

  if (!detached)
    {
      if (MakeDisplay(LoginName, attach_tty, attach_term, n, getppid(), &attach_Mode) == 0)
	Panic(0, "Could not alloc display");
    }

  if (SockName)
    {
      /* user started us with -S option */
      sprintf(socknamebuf, "%d.%s", getpid(), SockName);
    }
  else
    {
      sprintf(socknamebuf, "%d.%s.%s", getpid(), stripdev(attach_tty),
	      HostName);
    }
  for (ap = socknamebuf; *ap; ap++)
    if (*ap == '/')
      *ap = '-';
  SockName = socknamebuf;
  ServerSocket = MakeServerSocket();
#ifdef ETCSCREENRC
  if ((ap = getenv("SYSSCREENRC")) == NULL)
    StartRc(ETCSCREENRC);
  else
    StartRc(ap);
#endif
  StartRc(RcFileName);
# ifdef UTMPOK
#  ifndef UTNOKEEP
  InitUtmp(); 
#  endif /* UTNOKEEP */
# endif /* UTMPOK */
  if (display)
    {
      if (InitTermcap())
	{
	  if (userpid)
	    Kill(userpid, SIG_BYE);
	  eexit(1);
	}
      InitTerm(0);
#ifdef UTMPOK
      RemoveLoginSlot();
#endif
    }
  else
    {
      MakeTermcap(1);
    }
#ifdef LOADAV
  InitLoadav();
#endif /* LOADAV */
  MakeNewEnv();
  signal(SIGHUP, SigHup);
  signal(SIGINT, Finit);
  signal(SIGQUIT, Finit);
  signal(SIGTERM, Finit);
#ifdef BSDJOBS
  signal(SIGTTIN, SIG_IGN);
  signal(SIGTTOU, SIG_IGN);
#endif
  InitKeytab();
  if (display)
    {
      brktty(userfd);
      SetMode(&OldMode, &NewMode);
      /* Note: SetMode must be called _before_ FinishRc. */
      SetTTY(userfd, &NewMode);
      if (fcntl(userfd, F_SETFL, FNDELAY))
	Msg(errno, "Warning: NDELAY fcntl failed");
    }
#ifdef ETCSCREENRC
  if ((ap = getenv("SYSSCREENRC")) == NULL)
    FinishRc(ETCSCREENRC);
  else
    FinishRc(ap);
#endif
  FinishRc(RcFileName);

  debug2("UID %d  EUID %d\n", getuid(), geteuid());
  if (windows == NULL)
    {
      debug("We open one default window, as screenrc did not specify one.\n");
      if (MakeWindow(&nwin) == -1)
	{
	  AddStr("Sorry, could not find a PTY.");
	  sleep(2);
	  Finit(0);
	  /* NOTREACHED */
	}
    }
  if (default_startup)
    display_copyright();
#ifdef SYSV
  signal(SIGCLD, SigChld);
#else
  signal(SIGCHLD, SigChld);
#endif
  signal(SIGINT, SigInt);
  tv.tv_usec = 0;
  if (rflag == 2)
    {
#ifdef NETHACK
      if (nethackflag)
        Msg(0, "I can't seem to find a... Hey, wait a minute!  Here comes a screen now.");
      else
#endif
      Msg(0, "New screen...");
      rflag = 0;
    }
  debug1("display? = %#x\n", (unsigned int)display);
  for (;;)
    {
      /*
       * check to see if message line should be removed
       */
      tv.tv_sec = 0;
      for (display = displays; display; display = display->d_next)
	{
	  int time_left;

	  if (status == 0)
	    continue;
	  debug("checking status...\n");
	  time_left = status_time + (status_bell ? VBellWait : MsgWait) - time((time_t *)0);
	  if (time_left > 0)
	    {
	      if (tv.tv_sec == 0 || time_left < tv.tv_sec)
	        tv.tv_sec = time_left;
	      debug(" not yet.\n");
	    }
	  else
	    {
	      debug(" removing now.\n");
	      RemoveStatus();
	    }
	 }
      /*
       * check for I/O on all available I/O descriptors
       */
      FD_ZERO(&r);
      FD_ZERO(&w);
      for (display = displays; display; display = display->d_next)
	{
	  len = obufp - obuf;
	  if (len != 0)
	    FD_SET(userfd, &w);

	  if (dfore == 0)
	    {
	      FD_SET(userfd, &r);
	      continue;
	    }
#ifdef COPY_PASTE
	  if (pastelen)
	    FD_SET(dfore->w_ptyfd, &w);
	  else
#endif
	  if (dfore->inlen < sizeof(dfore->inbuf))
	    FD_SET(userfd, &r);
	}
      for (p = windows; p; p = p->next)
	{
	  if (p->inlen > 0)
	    FD_SET(p->w_ptyfd, &w);
          display = p->display;
	  if (p->active && status && !status_bell && !(use_hardstatus && HS))
	    continue;
	  if (p->outlen > 0)
	    continue;
	  if (p->wlay->l_block)
	    continue;
	/* Don't accept input if there is too much output pending on display*/
	  if (p->active && (obufp - obuf) > obufmax)
	    continue;  
	  FD_SET(p->w_ptyfd, &r);
	}
      FD_SET(ServerSocket, &r);
      if (GotSigChld && !tv.tv_sec)
	{
	  SigChldHandler();
	  continue;
	}
      if ((nsel = select(FD_SETSIZE, &r, &w, (fd_set *)0, tv.tv_sec ? &tv : (struct timeval *) 0)) < 0)
	{
	  debug1("Bad select - errno %d\n", errno);
	  if (errno != EINTR)
	    Panic(errno, "select");
	  errno = 0;
	  nsel = 0;
	}
      if (GotSigChld && !tv.tv_sec)
	{
	  SigChldHandler();
	  continue;
	}
      if (InterruptPlease)
	{
	  debug("Backend received interrupt\n");
	  if (fore)
	    {
	      char ibuf;
	      ibuf = intrc;
	      write(fore->w_ptyfd, &ibuf, 1);
	      debug1("Backend wrote interrupt to %d\n", fore->number);
	    }
	  InterruptPlease = 0;
	}

      /*
       *   Process a client connect attempt and message
       */
      if (nsel && FD_ISSET(ServerSocket, &r))
	{
          nsel--;
	  ReceiveMsg();
	  continue;
	}

#ifdef COPY_PASTE
      /*
       *  Write the copybuffer contents first, if any.
       */
      if (nsel)
	{
	  for (display = displays; display; display = display->d_next)
	    {
	      if ((fore = dfore) == 0 || pastelen == 0)
		continue;
	      tmp = fore->w_ptyfd;
	      if (! FD_ISSET(tmp, &w))
		continue;
	      debug1("writing pastebuffer (%d)\n", pastelen);
	      len = write(tmp, pastebuffer, pastelen > IOSIZE ? IOSIZE : pastelen);
	      if (len < 0)	/* Problems... window is dead */
		pastelen = 0;
	      if (len > 0)
		{
		  pastebuffer += len;
		  pastelen -= len;
		}
	      debug1("%d bytes pasted\n", len);
	      if (slowpaste > 0)
		{
		  struct timeval t;

		  debug1("slowpaste %d\n", slowpaste);
		  t.tv_usec = (long) (slowpaste * 1000);
		  t.tv_sec = 0;
		  select(0, (fd_set *)0, (fd_set *)0, (fd_set *)0, &t);
		}
	      if (--nsel == 0)
		break;
	    }
	}
#endif

      /*
       * Write the (already processed) user input to the window
       * descriptors first. We do not want to choke, if he types fast.
       */
      if (nsel)
	{
	  for (p = windows; p; p = p->next)
	    {
	      if (p->inlen <= 0)
		continue;
#ifdef COPY_PASTE
	      if (p->display && p->display->d_pastelen)
		continue;
#endif
	      tmp = p->w_ptyfd;
              if (FD_ISSET(tmp, &w))
                {
		  if ((len = write(tmp, p->inbuf, p->inlen)) > 0)
		    {
		      if ((p->inlen -= len))
		        bcopy(p->inbuf + len, p->inbuf, p->inlen);
		    }
		  if (--nsel == 0)
		    break;
		}
	    }
	}

      /*
       * Read, process, and store the user input
       */
      for (display = displays; nsel && display; display = ndisplay)
	{
	  int maxlen;

	  ndisplay = display->d_next;
	  if (FD_ISSET(userfd, &w)) 
	    {
	      int size = OUTPUT_BLOCK_SIZE;

	      len = obufp - obuf;
	      nsel --;
	      if (len < size)
		size = len;
	      debug1("ASYNC: writing %d", size);
	      size = write(userfd, obuf, size);
	      debug1(" - wrote %d\n", size);
	      if (size >= 0) 
		{
		  len -= size;
		  if (len)
		    bcopy(obuf + size, obuf, len);
		  obufp -= size;
		  obuffree += size;
		} 
	      else
		{
# ifdef EWOULDBLOCK
		  if (errno != EWOULDBLOCK)
# endif
		    Msg(errno, "Error writing output to display");
		}
	    }
	  if (! FD_ISSET(userfd, &r))
	    continue;
	  nsel--;
	  if (status && !(use_hardstatus && HS))
	    RemoveStatus();
	  if (dfore == 0)
	    maxlen = IOSIZE;
	  else
	    maxlen = sizeof(dfore->inbuf) - dfore->inlen;
	  if (maxlen > IOSIZE)
	    maxlen = IOSIZE;
	  if (maxlen == 0)
	    continue;
	  if (ESCseen)
	    {
	      if (maxlen == 1)
		continue;
	      buf[0] = Esc;
	      buflen = read(userfd, buf + 1, maxlen - 1) + 1;
	      ESCseen = 0;
	    }
	  else
	    buflen = read(userfd, buf, maxlen);
	  if (buflen < 0)
	    {
	      if (errno == EINTR)
		continue;
	      debug1("Read error: %d - SigHup()ing!\n", errno);
	      SigHup(SIGARG);
	      sleep(1);
	      continue;
	    }
	  if (buflen == 0)
	    {
	      debug("Found EOF - SigHup()ing!\n");
	      SigHup(SIGARG);
	      sleep(1);
	      continue;
	    }
	  ProcessInput(buf, buflen);
	}

      /*
       * Read and process the output from the window descriptors
       */
      for (p = windows; p; p = p->next)
	{
	  if (p->wlay->l_block)
	    continue;
	  display = p->display;
	  if (p->outlen)
	    WriteString(p, p->outbuf, p->outlen);
	  else if (nsel && FD_ISSET(p->w_ptyfd, &r))
	    {
	      nsel--;
	      if ((len = read(p->w_ptyfd, buf, IOSIZE)) <= 0)
		{
		  if (errno == EINTR)
		    continue;
 		  debug2("Window %d: read error (errno %d) - killing window\n", p->number, len ? errno : 0);
		  KillWindow(p);
		  nsel = 0;	/* KillWindow may change window order */
		  break;	/* so we just break */
  		}
#ifdef TIOCPKT
	      if ((p->t.flags & TTY_FLAG_PLAIN) == 0)
		{
		  if (buf[0])
		    {
		      debug1("PAKET %x\n", buf[0]);
		      if (buf[0] & TIOCPKT_NOSTOP)
			NewAutoFlow(p, 0);
		      if (buf[0] & TIOCPKT_DOSTOP)
			NewAutoFlow(p, 1);
		    }
		  if (len > 1)
		    WriteString(p, buf + 1, len - 1);
		}
	      else
#endif /* TIOCPKT */
		{
	          if (len > 0)
		    WriteString(p, buf, len);
		}
	    }
	  if (p->bell == BELL_ON)
	    {
	      p->bell = BELL_MSG;
	      for (display = displays; display; display = display->d_next)
	        Msg(0, MakeWinMsg(BellString, p->number));
	      if (p->monitor == MON_FOUND)
		p->monitor = MON_DONE;
	    }
	  else if (p->bell == BELL_VISUAL)
	    {
	      if (display && !status_bell)
		{
		  p->bell = BELL_DONE;
		  Msg(0, VisualBellString);
		  if (status)
		    status_bell = 1;
		}
	    }
	  if (p->monitor == MON_FOUND)
	    {
	      p->monitor = MON_MSG;
	      for (display = displays; display; display = display->d_next)
	        Msg(0, MakeWinMsg(ActivityString, p->number));
	    }
	}
#ifdef DEBUG
      if (nsel)
	debug1("*** Left over nsel: %d\n", nsel);
#endif
    }
  /* NOTREACHED */
}

static void
SigChldHandler()
{
  struct stat st;
  while (GotSigChld)
    {
      GotSigChld = 0;
      DoWait();
#ifdef SYSV
      signal(SIGCLD, SigChld);
#endif
    }
  if (stat(SockPath, &st) == -1)
    {
      debug1("SigChldHandler: Yuck! cannot stat '%s'\n", SockPath);
      if (!RecoverSocket())
	{
	  debug("SCREEN cannot recover from corrupt Socket, bye\n");
	  Finit(1);
	}
      else
	debug1("'%s' reconstructed\n", SockPath);
    }
  else
    debug2("SigChldHandler: stat '%s' o.k. (%03o)\n", SockPath, st.st_mode);
}

static sig_t
SigChld(SIGDEFARG)
{
  debug("SigChld()\n");
  GotSigChld = 1;
#ifndef SIGVOID
  return((sig_t) 0);
#endif
}

sig_t
SigHup(SIGDEFARG)
{
  if (display == 0)
    return;
  debug("SigHup()\n");
  if (auto_detach || displays->d_next)
    Detach(D_DETACH);
  else
    Finit(0);
#ifndef SIGVOID
  return((sig_t) 0);
#endif
}

/* 
 * the backend's Interrupt handler
 * we cannot insert the intrc directly, as we never know
 * if fore is valid.
 */
static sig_t
SigInt(SIGDEFARG)
{
#if HAZARDOUS
  char buf[1];

  debug("SigInt()\n");
  *buf = (char) intrc;
  if (fore)
    fore->inlen = 0;
  if (fore)
    write(fore->w_ptyfd, buf, 1);
#else
  signal(SIGINT, SigInt);
  debug("SigInt() careful\n");
  InterruptPlease = 1;
#endif
#ifndef SIGVOID
  return((sig_t) 0);
#endif
}

static sig_t
CoreDump(sig)
int sig;
{
  struct display *disp;
  char buf[80];

#ifdef SYSV
  signal(sig, SIG_IGN);
#endif /* SYSV */
  setgid(getgid());
  setuid(getuid());
  unlink("core");
  sprintf(buf, "\r\n[screen caught signal %d.%s]\r\n", sig,
#if defined(SHADOWPW) && !defined(DEBUG)
              ""
#else /* SHADOWPW  && !DEBUG */
              " (core dumped)"
#endif /* SHADOWPW  && !DEBUG */
              );
  for (disp = displays; disp; disp = disp->d_next)
    {
      write(disp->d_userfd, buf, strlen(buf));
      Kill(disp->d_userpid, SIG_BYE);
    }
#if defined(SHADOWPW) && !defined(DEBUG)
  eexit(sig);
#else /* SHADOWPW && !DEBUG */
  abort();
#endif /* SHADOWPW  && !DEBUG */
#ifndef SIGVOID
  return((sig_t) 0);
#endif
}

static void
DoWait()
{
  register int pid;
  struct win *p, *next;
#ifdef BSDWAIT
  union wait wstat;
#else
  int wstat;
#endif

#ifdef BSDJOBS
# ifndef BSDWAIT
  while ((pid = waitpid(-1, &wstat, WNOHANG | WUNTRACED)) > 0)
# else
  while ((pid = wait3(&wstat, WNOHANG | WUNTRACED, (struct rusage *) 0)) > 0)
# endif
#else	/* BSDJOBS */
  while ((pid = wait(&wstat)) < 0)
    if (errno != EINTR)
      break;
  if (pid > 0)
#endif	/* BSDJOBS */
    {
      for (p = windows; p; p = next)
	{
	  next = p->next;
	  if (pid == p->w_pid)
	    {
#ifdef BSDJOBS
	      if (WIFSTOPPED(wstat))
		{
		  debug3("Window %d pid %d: WIFSTOPPED (sig %d)\n", p->number, p->w_pid, WSTOPSIG(wstat));
#ifdef SIGTTIN
		  if (WSTOPSIG(wstat) == SIGTTIN)
		    {
		      Msg(0, "Suspended (tty input)");
		      continue;
		    }
#endif
#ifdef SIGTTOU
		  if (WSTOPSIG(wstat) == SIGTTOU)
		    {
		      Msg(0, "Suspended (tty output)");
		      continue;
		    }
#endif
		  /* Try to restart process */
# ifdef NETHACK	
                  if (nethackflag)
		    Msg(0, "You regain consciousness.");
		  else
# endif /* NETHACK */
		  Msg(0, "Child has been stopped, restarting.");
		  if (killpg(p->w_pid, SIGCONT))
		    kill(p->w_pid, SIGCONT);
		}
	      else
#endif
		KillWindow(p);
	      break;
	    }
	}
      if (p == 0)
	debug1("pid %d not found - hope that's ok\n", pid);
    }
}


sig_t
Finit(i)
int i;
{
  struct win *p, *next;

#ifdef SYSV
  signal(SIGCLD, SIG_IGN);
#else
  signal(SIGCHLD, SIG_IGN);
#endif
  signal(SIGHUP, SIG_IGN);
  debug1("Finit(%d);\n", i);
  for (p = windows; p; p = next)
    {
      next = p->next;
      FreeWindow(p);
    }
  if (ServerSocket != -1)
    {
      debug1("we unlink(%s)\n", SockPath);
#ifndef NOREUID
      setreuid(eff_uid, real_uid);
      setregid(eff_gid, real_gid);
#endif
      (void) unlink(SockPath);
#ifndef NOREUID
      setreuid(real_uid, eff_uid);
      setregid(real_gid, eff_gid);
#endif
    }
  for (display = displays; display; display = display->d_next)
    {
      if (status)
	RemoveStatus();
      FinitTerm();
#ifdef UTMPOK
      RestoreLoginSlot();
#endif
      AddStr("[screen is terminating]\r\n");
      Flush();
      SetTTY(userfd, &OldMode);
      fcntl(userfd, F_SETFL, 0);
      freetty();
      Kill(userpid, SIG_BYE);
    }
  /*
   * we _cannot_ call eexit(i) here, 
   * instead of playing with the Socket above. Sigh.
   */
  exit(i);
#ifndef SIGVOID
  return((sig_t) 0);
#endif
}

void
eexit(e)
int e;
{
  if (ServerSocket != -1)
    {
      debug1("we unlink(%s)\n", SockPath);
      setuid(real_uid);
      setgid(real_gid);
      (void) unlink(SockPath);
    }
  exit(e);
}


/*
 * Detach now has the following modes:
 *	D_DETACH	SIG_BYE		detach backend and exit attacher
 *	D_STOP		SIG_STOP	stop attacher (and detach backend)
 *	D_REMOTE	SIG_BYE		remote detach -- reattach to new attacher
 *	D_POWER 	SIG_POWER_BYE 	power detach -- attacher kills his parent
 *	D_REMOTE_POWER	SIG_POWER_BYE	remote power detach -- both
 *	D_LOCK		SIG_LOCK	lock the attacher
 * (jw)
 * we always remove our utmp slots. (even when "lock" or "stop")
 * Note: Take extra care here, we may be called by interrupt!
 */
void
Detach(mode)
int mode;
{
  int sign = 0, pid;
#ifdef UTMPOK
  struct win *p;
#endif

  if (display == 0)
    return;
  debug1("Detach(%d)\n", mode);
  if (status)
    RemoveStatus();
  signal(SIGHUP, SIG_IGN);
  FinitTerm();
  switch (mode)
    {
    case D_DETACH:
      AddStr("[detached]\r\n");
      sign = SIG_BYE;
      break;
#ifdef BSDJOBS
    case D_STOP:
      sign = SIG_STOP;
      break;
#endif
#ifdef REMOTE_DETACH
    case D_REMOTE:
      AddStr("[remote detached]\r\n");
      sign = SIG_BYE;
      break;
#endif
#ifdef POW_DETACH
    case D_POWER:
      AddStr("[power detached]\r\n");
      if (PowDetachString) 
	{
	  AddStr(expand_vars(PowDetachString));
	  AddStr("\r\n");
	}
      sign = SIG_POWER_BYE;
      break;
#ifdef REMOTE_DETACH
    case D_REMOTE_POWER:
      AddStr("[remote power detached]\r\n");
      if (PowDetachString) 
	{
	  AddStr(expand_vars(PowDetachString));
	  AddStr("\r\n");
	}
      sign = SIG_POWER_BYE;
      break;
#endif
#endif
    case D_LOCK:
      ClearDisplay();
      sign = SIG_LOCK;
      /* tell attacher to lock terminal with a lockprg. */
      break;
    }
#ifdef UTMPOK
  if (displays->d_next == 0)
    for (p = windows; p; p = p->next)
      if (p->slot != (slot_t) -1)
	{
	  RemoveUtmp(p);
	  /*
	   * Set the slot to 0 to get the window
	   * logged in again.
	   */
	  p->slot = (slot_t) 0;
	}
  RestoreLoginSlot();
#endif
  if (dfore)
    {
      dfore->active = 0;
      dfore->display = 0;
      lay = &BlankLayer;
      layfn = lay->l_layfn;
    }
  while (lay != &BlankLayer)
    ExitOverlayPage();
  Flush();
  SetTTY(userfd, &OldMode);
  fcntl(userfd, F_SETFL, 0);
  freetty();
  pid = userpid;
  debug2("display: %#x displays: %#x\n", (unsigned int)display, (unsigned int)displays);
  FreeDisplay();
  if (displays == 0)
    /* Flag detached-ness */
    (void) chsock();
  /*
   * tell father to father what to do. We do that after we
   * freed the tty, thus getty feels more comfortable on hpux
   * if it was a power detach.
   */
  Kill(pid, sign);
  debug2("Detach: Signal %d to Attacher(%d)!\n", sign, pid);
  debug("Detach returns, we are successfully detached.\n");
}

void
Kill(pid, sig)
int pid, sig;
{
  if (pid < 2)
    return;
  (void) kill(pid, sig);
}

static int
IsSymbol(e, s)
register char *e, *s;
{
  register int l;

  l = strlen(s);
  return strncmp(e, s, l) == 0 && e[l] == '=';
}

static void
MakeNewEnv()
{
  register char **op, **np;
  static char buf[MAXSTR];

  for (op = environ; *op; ++op)
    ;
  NewEnv = np = (char **) malloc((unsigned) (op - environ + 6 + 1) * sizeof(char **));
  if (!NewEnv)
    Panic(0, strnomem);
  if (strlen(SockName) > MAXSTR - 5)
    SockName = "?";
  sprintf(buf, "STY=%s", SockName);
  *np++ = buf;	                /* NewEnv[0] */
  *np++ = Term;	                /* NewEnv[1] */
#ifdef TIOCSWINSZ
  np += 2;	/* room for TERMCAP and WINDOW */
#else
  np += 4;	/* room for TERMCAP WINDOW LINES COLUMNS */
#endif

  for (op = environ; *op; ++op)
    {
      if (!IsSymbol(*op, "TERM") && !IsSymbol(*op, "TERMCAP")
	  && !IsSymbol(*op, "STY") && !IsSymbol(*op, "WINDOW")
	  && !IsSymbol(*op, "SCREENCAP")
#ifndef TIOCGWINSZ
	  && !IsSymbol(*op, "LINES") && !IsSymbol(*op, "COLUMNS")
#endif
	  )
	*np++ = *op;
    }
  *np = 0;
}

void
#ifdef USEVARARGS
/*VARARGS2*/
# if defined(__STDC__)
Msg(int err, char *fmt, ...)
# else /* __STDC__ */
Msg(err, fmt, va_alist)
int err;
char *fmt;
va_dcl
# endif /* __STDC__ */
{
  static va_list ap;
#else /* USEVARARRGS */
/*VARARGS2*/
Msg(err, fmt, p1, p2, p3, p4, p5, p6)
int err;
char *fmt;
unsigned long p1, p2, p3, p4, p5, p6;
{
#endif /* USEVARARRGS */
  char buf[MAXPATH*2];
  char *p = buf;

#ifdef USEVARARGS
# if defined(__STDC__)
  va_start(ap, fmt);
# else /* __STDC__ */
  va_start(ap);
# endif /* __STDC__ */
  (void) vsprintf(p, fmt, ap);
  va_end(ap);
#else /* USEVARARRGS */
  sprintf(p, fmt, p1, p2, p3, p4, p5, p6);
#endif /* USEVARARRGS */
  if (err)
    {
      p += strlen(p);
      if (err > 0 && err < sys_nerr)
	sprintf(p, ": %s", sys_errlist[err]);
      else
	sprintf(p, ": Error %d", err);
    }
  debug2("Msg('%s') (%#x);\n", buf, (unsigned int)display);
  if (display)
    MakeStatus(buf);
  else if (displays)
    {
      for (display = displays; display; display = display->d_next)
	MakeStatus(buf);
    }
  else
    printf("%s\r\n", buf);
}

void
#ifdef USEVARARGS
/*VARARGS2*/
# if defined(__STDC__)
Panic(int err, char *fmt, ...)
# else /* __STDC__ */
Panic(err, fmt, va_alist)
int err;
char *fmt;
va_dcl
# endif /* __STDC__ */
{
  static va_list ap;
#else /* USEVARARRGS */
/*VARARGS2*/
Panic(err, fmt, p1, p2, p3, p4, p5, p6)
int err;
char *fmt;
unsigned long p1, p2, p3, p4, p5, p6;
{
#endif /* USEVARARRGS */
  char buf[MAXPATH*2];
  char *p = buf;

#ifdef USEVARARGS
# if defined(__STDC__)
  va_start(ap, fmt);
# else /* __STDC__ */
  va_start(ap);
# endif /* __STDC__ */
  (void) vsprintf(p, fmt, ap);
  va_end(ap);
#else /* USEVARARRGS */
  sprintf(p, fmt, p1, p2, p3, p4, p5, p6);
#endif /* USEVARARRGS */
  if (err)
    {
      p += strlen(p);
      if (err > 0 && err < sys_nerr)
	sprintf(p, ": %s", sys_errlist[err]);
      else
	sprintf(p, ": Error %d", err);
    }
  debug1("Panic('%s');\n", buf);
  if (displays == 0)
    printf("%s\r\n", buf);
  else
    for (display = displays; display; display = display->d_next)
      {
        FinitTerm();
        Flush();
#ifdef UTMPOK
        RestoreLoginSlot();
#endif
        SetTTY(userfd, &OldMode);
        fcntl(userfd, F_SETFL, 0);
        write(userfd, buf, strlen(buf));
        write(userfd, "\n", 1);
        freetty();
	if (userpid)
	  Kill(userpid, SIG_BYE);
      }
#ifdef MULTIUSER
  if (tty_oldmode >= 0)
    {
# ifdef NOREUID
      setuid(eff_uid);
# else
      setreuid(real_uid, eff_uid);
# endif
      debug1("Panic: changing back modes from %s\n", attach_tty);
      chmod(attach_tty, tty_oldmode);
    }
#endif
  eexit(1);
}


/*
 * '^' is allowed as an escape mechanism for control characters. jw.
 */
static char *
MakeWinMsg(s, n)
register char *s;
int n;
{
  static char buf[MAXSTR];
  register char *p = buf;
  register int ctrl;

  ctrl = 0;
  for (; *s && p < buf + MAXSTR - 1; s++, p++)
    if (ctrl)
      {
        ctrl = 0;
        if (*s == '^' || *s < 64)
          *p = *s;
        else 
          *p = *s - 64;
      }
    else
      {
        switch (*s)
          {
          case '%':
    	    *p = n + '0';
	    break;
          case '~':
	    *p = BELL;
	    break;
	  case '^':
	    ctrl = 1;
	    *p-- = '^';
	    break;
          default:
	    *p = *s;
	    break;
          }
      }
  *p = '\0';
  return buf;
}

void
DisplaySleep(d, n)
struct display *d;
int n;
{
  if (!d)
    {
      debug("DisplaySleep has no display sigh\n");
      sleep(n);
    }
  else
    {
      char buf[2];
      fd_set r;
      struct timeval t;

      t.tv_usec = 0;
      t.tv_sec = n;
      FD_ZERO(&r);
      FD_SET(d->d_userfd, &r);
      if (select(FD_SETSIZE, &r, (fd_set *)0, (fd_set *)0, &t) > 0)
	{
	  debug("display activity stopped sleep\n");
	  read(d->d_userfd, buf, 1);
	}
      debug1("DisplaySleep(%d) ending\n", n);
    }
}
