/* Copyright (c) 1991
 *      Juergen Weigert (jnweiger@immd4.informatik.uni-erlangen.de)
 *      Michael Schroeder (mlschroe@immd4.informatik.uni-erlangen.de)
 * Copyright (c) 1987 Oliver Laumann
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file COPYING); if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Noteworthy contributors to screen's design and implementation:
 *	Wayne Davison (davison@borland.com)
 *	Patrick Wolfe (pat@kai.com, kailand!pat)
 *	Bart Schaefer (schaefer@cse.ogi.edu)
 *	Nathan Glasser (nathan@brokaw.lcs.mit.edu)
 *	Larry W. Virden (lvirden@cas.org)
 *	Howard Chu (hyc@hanauma.jpl.nasa.gov)
 *	Tim MacKenzie (tym@dibbler.cs.monash.edu.au)
 *	Markku Jarvinen (mta@{cc,cs,ee}.tut.fi)
 *	Marc Boucher (marc@CAM.ORG)
 *
 ****************************************************************
 */

#include "rcs.h"
RCS_ID("$Id: display.c,v 1.12 92/11/25 20:47:46 mlschroe Exp Locker: mlschroe $ FAU")


#include <sys/types.h>
#include <fcntl.h>

#include "config.h"
#include "screen.h"
#include "extern.h"

static void CountChars __P((int));
static void PutChar __P((int));
static int  BlankResize __P((int, int));


extern char *tgoto __P((char *, int, int));

extern struct display *display, *displays;
extern struct win *windows;
extern int  use_hardstatus;
extern int  MsgMinWait;
extern int  Z0width, Z1width;
extern char *blank, *null;

/*
 * tputs needs this to calculate the padding
 */
short ospeed;

#ifndef MULTI
struct display TheDisplay;
#endif

/*
 *  Default layer management
 */

void
DefProcess(bufp, lenp)
char **bufp;
int *lenp;
{
  *bufp += *lenp;
  *lenp = 0;
}

void
DefRedisplayLine(y, xs, xe, isblank)
int y, xs, xe, isblank;
{
  if (isblank == 0 && y >= 0)
    DefClearLine(y, xs, xe);
}

void
DefClearLine(y, xs, xe)
int y, xs, xe;
{
  DisplayLine(null, null, null, blank, null, null, y, xs, xe);
}

/*ARGSUSED*/
int
DefRewrite(y, xs, xe, doit)
int y, xs, xe, doit;
{
  return EXPENSIVE;
}

void
DefSetCursor()
{
  GotoPos(0, 0);
}

/*ARGSUSED*/
int
DefResize(wi, he)
int wi, he;
{
  return -1;
}

void
DefRestore()
{
  InsertMode(0);
  ChangeScrollRegion(0, height - 1);
  KeypadMode(0);
  CursorkeysMode(0);
  SetAttrFont(0, ASCII);
  SetFlow(FLOW_NOW);
}

/*
 *  Blank layer management
 */

struct LayFuncs BlankLf =
{
  DefProcess,
  0,
  DefRedisplayLine,
  DefClearLine,
  DefRewrite,
  DefSetCursor,
  BlankResize,
  DefRestore
};

struct layer BlankLayer =
{
  0,
  0,
  &BlankLf,
  0
};

/*ARGSUSED*/
static int
BlankResize(wi, he)
int wi, he;
{
  return 0;
}


/*
 *  Generate new display
 */

struct display *
MakeDisplay(uname, utty, term, fd, pid, Mode)
char *uname, *utty, *term;
int fd, pid;
struct mode *Mode;
{
#ifdef MULTI
  if ((display = (struct display *)malloc(sizeof(*display))) == 0)
    return 0;
  bzero((char *) display, sizeof(*display));
#else
  if (displays)
    return 0;
  display = &TheDisplay;
#endif
  display->d_next = displays;
  displays = display;
  flow = 1;
  userfd = fd;
  OldMode = *Mode;
  Resize_obuf();  /* Allocate memory for buffer */
  obufmax = OBUF_MAX;
  obufp = obuf;
  userpid = pid;
#ifdef POSIX
  dospeed = (short) cfgetospeed(&OldMode.tio);
#else
# ifndef TERMIO
  dospeed = (short) OldMode.m_ttyb.sg_ospeed;
# endif
#endif
  debug1("New displays ospeed = %d\n", dospeed);
  strcpy(usertty, utty);
  strcpy(termname, term);
  strncpy(user, uname, sizeof(user) - 1);
  lay = &BlankLayer;
  layfn = BlankLayer.l_layfn;
  return display;
}

void
FreeDisplay()
{
  struct display *d, **dp;

#ifdef MULTI
  for (dp = &displays; (d = *dp) ; dp = &d->d_next)
    if (d == display)
      break;
  ASSERT(d);
  if (status_lastmsg)
    free(status_lastmsg);
# ifdef COPY_PASTE
  if (copybuffer)
    free(copybuffer);
# endif
  if (obuf)
    free(obuf);
  *dp = display->d_next;
  free(display);
#else
  ASSERT(display == displays);
  ASSERT(display == &TheDisplay);
  displays = 0;
#endif
  display = 0;
}

/*
 * if the adaptflag is on, we keep the size of this display, else
 * we may try to restore our old window sizes.
 */
void
InitTerm(adapt)
int adapt;
{
  ASSERT(display);
  top = bot = -1;
  PutStr(TI);
  PutStr(IS);
  /* Check for toggle */
  if (IM && strcmp(IM, EI))
    PutStr(EI);
  insert = 0;
  /* Check for toggle */
  if (KS && strcmp(KS, KE))
    PutStr(KE);
  keypad = 0;
  if (CCS && strcmp(CCS, CCE))
    PutStr(CCE);
  cursorkeys = 0;
  PutStr(CE0);
  font = ASCII;
  if (adapt == 0)
    ResizeDisplay(defwidth, defheight);
  ChangeScrollRegion(0, height - 1);
  curx = cury = 0;
  Flush();
  ClearDisplay();
  debug1("we %swant to adapt all our windows to the display\n", 
	 (adapt) ? "" : "don't ");
  /* In case the size was changed by a init sequence */
  CheckScreenSize((adapt) ? 2 : 0);
}

void
FinitTerm()
{
  ASSERT(display);
  ResizeDisplay(defwidth, defheight);
  DefRestore();
  SetAttrFont(0, ASCII);
  curx = cury = -1;
  GotoPos(0, height - 1);
  AddChar('\n');
  PutStr(TE);
  Flush();
}


void
INSERTCHAR(c)
int c;
{
  ASSERT(display);
  if (!insert && curx < width - 1)
    {
      if (IC || CIC)
	{
	  if (IC)
	    PutStr(IC);
	  else
	    CPutStr(CIC, 1);
	  RAW_PUTCHAR(c);
	  return;
	}
      InsertMode(1);
      if (!insert)
	{
          RefreshLine(cury, curx, width-1, 0);
	  return;
	}
    }
  RAW_PUTCHAR(c);
}

void
PUTCHAR(c)
int c;
{
  ASSERT(display);
  if (insert && curx < width - 1)
    InsertMode(0);
  RAW_PUTCHAR(c);
}

void
PUTCHARLP(c)
int c;
{
  if (curx < width - 1)
    {
      if (insert)
	InsertMode(0);
      RAW_PUTCHAR(c);
      return;
    }
  if (CLP || cury != bot)
    {
      RAW_PUTCHAR(c);
      return;
    }
  lp_missing = 1;
  lp_image = c;
  lp_attr = attr;
  lp_font = font;
}

/*
 * RAW_PUTCHAR() is for all text that will be displayed.
 * NOTE: charset Nr. 0 has a conversion table, but c1, c2, ... don't.
 */

void
RAW_PUTCHAR(c)
int c;
{
  ASSERT(display);
  if (font == '0')
    {
      AddChar(c0_tab[c]);
    }
  else
    AddChar(c);
  if (++curx >= width)
    {
      if ((AM && !CLP) || curx > width)
	{
	  curx -= width;
	  if (cury < height-1 && cury != bot)
	    cury++;
	}
    }
}

static void
PutChar(c)
int c;
{
  /* this PutChar for ESC-sequences only (AddChar is a macro) */
  AddChar(c);
}

void
PutStr(s)
char *s;
{
  if (display && s)
    {
      ospeed = dospeed;
      tputs(s, 1, PutChar);
    }
}

void
CPutStr(s, c)
char *s;
int c;
{
  if (display && s)
    {
      ospeed = dospeed;
      tputs(tgoto(s, 0, c), 1, PutChar);
    }
}


/* Insert mode is a toggle on some terminals, so we need this hack:
 */
void
InsertMode(on)
int on;
{
  if (display && on != insert && IM)
    {
      insert = on;
      if (insert)
	PutStr(IM);
      else
	PutStr(EI);
    }
}

/* ...and maybe keypad application mode is a toggle, too:
 */
void
KeypadMode(on)
int on;
{
  if (display && keypad != on && KS)
    {
      keypad = on;
      if (keypad)
	PutStr(KS);
      else
	PutStr(KE);
    }
}

void
CursorkeysMode(on)
int on;
{
  if (display && cursorkeys != on && CCS)
    {
      cursorkeys = on;
      if (cursorkeys)
	PutStr(CCS);
      else
	PutStr(CCE);
    }
}

static int StrCost;

/* ARGSUSED */
static void
CountChars(c)
int c;
{
  StrCost++;
}

int
CalcCost(s)
register char *s;
{
  ASSERT(display);
  if (s)
    {
      StrCost = 0;
      ospeed = dospeed;
      tputs(s, 1, CountChars);
      return StrCost;
    }
  else
    return EXPENSIVE;
}

void
GotoPos(x2, y2)
int x2, y2;
{
  register int dy, dx, x1, y1;
  register int costx, costy;
  register int m;
  register char *s;
  int CMcost;
  enum move_t xm = M_NONE, ym = M_NONE;

  if (!display)
    return;

  x1 = curx;
  y1 = cury;

  if (x1 == width)
    if (CLP && AM)
      x1 = -1;		/* don't know how the terminal treats this */
    else
      x1--;
  if (x2 == width)
    x2--;
  dx = x2 - x1;
  dy = y2 - y1;
  if (dy == 0 && dx == 0)
    {
      return;
    }
  if (!MS && attr)	/* Safe to move in SO mode ? */
    SetAttr(0);
  if (y1 < 0			/* don't know the y position */
      || (y2 > bot && y1 <= bot)	/* have to cross border */
      || (y2 < top && y1 >= top))	/* of scrollregion ?    */
    {
    DoCM:
      if (HO && !x2 && !y2)
        PutStr(HO);
      else
        PutStr(tgoto(CM, x2, y2));
      curx = x2;
      cury = y2;
      return;
    }
  /* Calculate CMcost */
  if (HO && !x2 && !y2)
    s = HO;
  else
    s = tgoto(CM, x2, y2);
  CMcost = CalcCost(s);

  /* Calculate the cost to move the cursor to the right x position */
  costx = EXPENSIVE;
  if (x1 >= 0)	/* relativ x positioning only if we know where we are */
    {
      if (dx > 0)
	{
	  if (CRI && (dx > 1 || !ND))
	    {
	      costx = CalcCost(tgoto(CRI, 0, dx));
	      xm = M_CRI;
	    }
	  if ((m = NDcost * dx) < costx)
	    {
	      costx = m;
	      xm = M_RI;
	    }
	  /* Speedup: dx <= Rewrite() */
	  if (dx < costx && (m = Rewrite(y1, x1, x2, 0)) < costx)
	    {
	      costx = m;
	      xm = M_RW;
	    }
	}
      else if (dx < 0)
	{
	  if (CLE && (dx < -1 || !BC))
	    {
	      costx = CalcCost(tgoto(CLE, 0, -dx));
	      xm = M_CLE;
	    }
	  if ((m = -dx * LEcost) < costx)
	    {
	      costx = m;
	      xm = M_LE;
	    }
	}
      else
	costx = 0;
    }
  /* Speedup: Rewrite() >= x2 */
  if (x2 + CRcost < costx && (m = (x2 ? Rewrite(y1, 0, x2, 0) : 0) + CRcost) < costx)
    {
      costx = m;
      xm = M_CR;
    }

  /* Check if it is already cheaper to do CM */
  if (costx >= CMcost)
    goto DoCM;

  /* Calculate the cost to move the cursor to the right y position */
  costy = EXPENSIVE;
  if (dy > 0)
    {
      if (CDO && dy > 1)	/* DO & NL are always != 0 */
	{
	  costy = CalcCost(tgoto(CDO, 0, dy));
	  ym = M_CDO;
	}
      if ((m = dy * ((x2 == 0) ? NLcost : DOcost)) < costy)
	{
	  costy = m;
	  ym = M_DO;
	}
    }
  else if (dy < 0)
    {
      if (CUP && (dy < -1 || !UP))
	{
	  costy = CalcCost(tgoto(CUP, 0, -dy));
	  ym = M_CUP;
	}
      if ((m = -dy * UPcost) < costy)
	{
	  costy = m;
	  ym = M_UP;
	}
    }
  else
    costy = 0;

  /* Finally check if it is cheaper to do CM */
  if (costx + costy >= CMcost)
    goto DoCM;

  switch (xm)
    {
    case M_LE:
      while (dx++ < 0)
	PutStr(BC);
      break;
    case M_CLE:
      CPutStr(CLE, -dx);
      break;
    case M_RI:
      while (dx-- > 0)
	PutStr(ND);
      break;
    case M_CRI:
      CPutStr(CRI, dx);
      break;
    case M_CR:
      PutStr(CR);
      curx = 0;
      x1 = 0;
      /* FALLTHROUGH */
    case M_RW:
      if (x1 < x2)
	(void) Rewrite(y1, x1, x2, 1);
      break;
    default:
      break;
    }

  switch (ym)
    {
    case M_UP:
      while (dy++ < 0)
	PutStr(UP);
      break;
    case M_CUP:
      CPutStr(CUP, -dy);
      break;
    case M_DO:
      s =  (x2 == 0) ? NL : DO;
      while (dy-- > 0)
	PutStr(s);
      break;
    case M_CDO:
      CPutStr(CDO, dy);
      break;
    default:
      break;
    }
  curx = x2;
  cury = y2;
}

void
ClearDisplay()
{
  ASSERT(display);
  Clear(0, 0, width - 1, height - 1);
}

void
Clear(xs, ys, xe, ye)
int xs, ys, xe, ye;
{
  int y, xxe;

  ASSERT(display);
  if (xs == width)
    xs--;
  if (xe == width)
    xe--;
  if (lp_missing && ys <= bot)
    {
      if (ye > bot || (ye == bot && xe == width - 1))
	lp_missing = 0;
    }
  if (xe == width - 1 && ye == height - 1)
    {
#ifdef AUTO_NUKE
      if (xs == 0 && ys == 0 && auto_nuke)
	NukePending();
#endif
      if (xs == 0 && ys == 0 && CL)
	{
	  PutStr(CL);
	  cury = curx = 0;
	  return;
	}
      if (CD)
	{
	  GotoPos(xs, ys);
	  PutStr(CD);
	  return;
	}
    }
  xxe = width - 1;
  for (y = ys; y <= ye; y++, xs = 0)
    {
      if (y == ye)
	xxe = xe;
      if (xs == 0 && CB && (xxe != width - 1 || (curx == xxe && cury == y)))
	{
	  GotoPos(xxe, y);
	  PutStr(CB);
	  continue;
	}
      if (xxe == width - 1 && CE)
	{
	  GotoPos(xs, y);
	  PutStr(CE);
	  continue;
	}
      ClearLine(y, xs, xxe);
    }
}


/*
 * if cur_only > 0, we only redisplay current line, as a full refresh is
 * too expensive over a low baud line.
 */
void
Redisplay(cur_only)
int cur_only;
{
  register int i, stop;

  ASSERT(display);
  DefRestore();
  ClearDisplay();
  stop = height;
  i = 0;
  if (cur_only > 0 && dfore)
    {
      i = stop = dfore->y;
      stop++;
    }
  else RedisplayLine(-1, 0, width - 1, 1);
  for (; i < stop; i++)
    RedisplayLine(i, 0, width - 1, 1);
  SetCursor();
  Restore();
}


void
ScrollRegion(ys, ye, n)
int ys, ye, n;
{
  int i;
  int up;
  int oldtop, oldbot;
  int alok, dlok, aldlfaster;
  int missy = 0;

  ASSERT(display);
  if (n == 0)
    return;
  if (ys == 0 && ye == height - 1 && 
      (n >= height || -n >= height))
    {
      ClearDisplay();
      return;
    }

  if (lp_missing)
    {
      if (bot > ye || bot < ys)
	missy = bot;
      else
	{
	  missy = bot - n;
          if (missy>ye || missy<ys)
	    lp_missing = 0;
	}
    }

  up = 1;
  if (n < 0)
    {
      up = 0;
      n = -n;
    }
  if (n >= ye-ys+1)
    n = ye-ys+1;

  oldtop = top;
  oldbot = bot;
  if (bot != ye)
    ChangeScrollRegion(ys, ye);
  alok = (AL || CAL || (ye == bot &&  up));
  dlok = (DL || CDL || (ye == bot && !up));
  if (top != ys && !(alok && dlok))
    ChangeScrollRegion(ys, ye);

  if (lp_missing && 
      (oldbot != bot ||
       (oldbot == bot && up && top == ys && bot == ye)))
    {
      FixLP(width - 1, oldbot);
      if (oldbot == bot)		/* have scrolled */
	{
	  if (--n == 0)
	    {
	      ChangeScrollRegion(oldtop, oldbot);
	      return;
	    }
	}
    }

  aldlfaster = (n > 1 && ye == bot && ((up && CDL) || (!up && CAL)));

  if ((up || SR) && top == ys && bot == ye && !aldlfaster)
    {
      if (up)
	{
	  GotoPos(0, ye);
	  while (n-- > 0)
	    PutStr(NL);		/* was SF, I think NL is faster */
	}
      else
	{
	  GotoPos(0, ys);
	  while (n-- > 0)
	    PutStr(SR);
	}
    }
  else if (alok && dlok)
    {
      if (up || ye != bot)
	{
          GotoPos(0, up ? ys : ye+1-n);
          if (CDL && !(n == 1 && DL))
	    CPutStr(CDL, n);
	  else
	    for(i = n; i--; )
	      PutStr(DL);
	}
      if (!up || ye != bot)
	{
          GotoPos(0, up ? ye+1-n : ys);
          if (CAL && !(n == 1 && AL))
	    CPutStr(CAL, n);
	  else
	    for(i = n; i--; )
	      PutStr(AL);
	}
    }
  else
    {
      Redisplay(0);
      return;
    }
  if (lp_missing && missy != bot)
    FixLP(width - 1, missy);
  ChangeScrollRegion(oldtop, oldbot);
  if (lp_missing && missy != bot)
    FixLP(width - 1, missy);
}

void
SetAttr(new)
register int new;
{
  register int i, old;

  if (!display || (old = attr) == new)
    return;
  attr = new;
  for (i = 1; i <= A_MAX; i <<= 1)
    {
      if ((old & i) && !(new & i))
	{
	  PutStr(UE);
	  PutStr(SE);
	  PutStr(ME);
	  if (new & A_DI)
	    PutStr(attrtab[ATTR_DI]);
	  if (new & A_US)
	    PutStr(attrtab[ATTR_US]);
	  if (new & A_BD)
	    PutStr(attrtab[ATTR_BD]);
	  if (new & A_RV)
	    PutStr(attrtab[ATTR_RV]);
	  if (new & A_SO)
	    PutStr(attrtab[ATTR_SO]);
	  if (new & A_BL)
	    PutStr(attrtab[ATTR_BL]);
	  return;
	}
    }
  if ((new & A_DI) && !(old & A_DI))
    PutStr(attrtab[ATTR_DI]);
  if ((new & A_US) && !(old & A_US))
    PutStr(attrtab[ATTR_US]);
  if ((new & A_BD) && !(old & A_BD))
    PutStr(attrtab[ATTR_BD]);
  if ((new & A_RV) && !(old & A_RV))
    PutStr(attrtab[ATTR_RV]);
  if ((new & A_SO) && !(old & A_SO))
    PutStr(attrtab[ATTR_SO]);
  if ((new & A_BL) && !(old & A_BL))
    PutStr(attrtab[ATTR_BL]);
}

void
SetFont(new)
int new;
{
  if (!display || font == new)
    return;
  font = new;
  if (new == ASCII)
    PutStr(CE0);
  else
    CPutStr(CS0, new);
}

void
SetAttrFont(newattr, newcharset)
int newattr, newcharset;
{
  SetAttr(newattr);
  SetFont(newcharset);
}

void
MakeStatus(msg)
char *msg;
{
  register char *s, *t;
  register int max, ti;

  if (!display)
    return;
  
  if (!tcinited)
    {
      debug("tc not inited, just writing msg\n");
      AddStr(msg);
      AddStr("\r\n");
      Flush();
      return;
    }
  if (!use_hardstatus || !HS)
    {
      max = width;
      if (CLP == 0)
	max--;
    }
  else
    max = WS;
  if (status)
    {
      if (!status_bell)
	{
	  ti = time((time_t *) 0) - status_time;
	  if (ti < MsgMinWait)
	    sleep(MsgMinWait - ti);
	}
      RemoveStatus();
    }
  for (s = t = msg; *s && t - msg < max; ++s)
    if (*s == BELL)
      PutStr(BL);
    else if (*s >= ' ' && *s <= '~')
      *t++ = *s;
  *t = '\0';
  if (t > msg)
    {
      if (t - msg >= status_buflen)
        {
          char *buf;
          if (status_lastmsg)
	    buf = realloc(status_lastmsg, t - msg + 1);
	  else
	    buf = malloc(t - msg + 1);
	  if (buf)
	    {
              status_lastmsg = buf;
              status_buflen = t - msg + 1;
            }
        }
      if (t - msg < status_buflen)
        strcpy(status_lastmsg, msg);
      status = 1;
      status_len = t - msg;
      status_lastx = curx;
      status_lasty = cury;
      if (!use_hardstatus || !HS)
	{
	  debug1("using STATLINE %d\n", STATLINE);
	  GotoPos(0, STATLINE);
          SetAttrFont(A_SO, ASCII);
	  InsertMode(0);
	  AddStr(msg);
          curx = -1;
	}
      else
	{
	  debug("using HS\n");
          SetAttrFont(0, ASCII);
	  InsertMode(0);
	  CPutStr(TS, 0);
	  AddStr(msg);
	  PutStr(FS);
	}
      Flush();
      (void) time(&status_time);
    }
}

void
RemoveStatus()
{
  struct win *p;

  if (!display)
    return;
  if (!status)
    return;
  
  /*
   * now we need to find the window that caused an activity or bell
   * message, to reenable this function there.
   */
  for (p = windows; p; p = p->next)
    { 
      if (p->display != display)
	continue;
      if (p->monitor == MON_MSG)
	{
	  debug1("RemoveStatus clearing monitor win %d\n", p->number);
	  p->monitor = MON_DONE;
	}
      if (p->bell == BELL_MSG)
	{
	  debug1("RemoveStatus clearing bell win %d\n", p->number);
	  p->bell = BELL_DONE;
	}
    }
  status = 0;
  status_bell = 0;
  if (!use_hardstatus || !HS)
    {
      GotoPos(0, STATLINE);
      RefreshLine(STATLINE, 0, status_len - 1, 0);
      GotoPos(status_lastx, status_lasty);
    }
  else
    {
      SetAttrFont(0, ASCII);
      PutStr(DS);
    }
  SetCursor();
}

void
RefreshLine(y, from, to, isblank)
int y, from, to, isblank;
{
  ASSERT(display);
  debug2("RefreshLine %d %d", y, from);
  debug2(" %d %d\n", to, isblank);
  if (isblank == 0 && CE && to == width - 1)
    {
      GotoPos(from, y);
      PutStr(CE);
      isblank = 1;
    }
  RedisplayLine(y, from, to, isblank);
}

void
FixLP(x2, y2)
register int x2, y2;
{
  int oldattr = attr, oldfont = font;

  ASSERT(display);
  GotoPos(x2, y2);
  SetAttrFont(lp_attr, lp_font);
  PUTCHAR(lp_image);
  lp_missing = 0;
  SetAttrFont(oldattr, oldfont);
}

void
DisplayLine(os, oa, of, s, as, fs, y, from, to)
int from, to, y;
register char *os, *oa, *of, *s, *as, *fs;
{
  register int x;
  int last2flag = 0, delete_lp = 0;

  ASSERT(display);
  if (!CLP && y == bot && to == width - 1)
    if (lp_missing
	|| s[to] != os[to] || as[to] != oa[to] || of[to] != fs[to])
      {
	if ((IC || IM) && from < to)
	  {
	    to -= 2;
	    last2flag = 1;
	    lp_missing = 0;
	  }
	else
	  {
	    to--;
	    delete_lp = (CE || DC || CDC);
	    lp_missing = (s[to] != ' ' || as[to] || fs[to]);
	    lp_image = s[to];
	    lp_attr = as[to];
	    lp_font = fs[to];
	  }
      }
    else
      to--;
  for (x = from; x <= to; ++x)
    {
      if (s[x] == os[x] && as[x] == oa[x] && of[x] == fs[x])
	continue;
      GotoPos(x, y);
      SetAttr(as[x]);
      SetFont(fs[x]);
      PUTCHAR(s[x]);
    }
  if (last2flag)
    {
      GotoPos(x, y);
      SetAttr(as[x + 1]);
      SetFont(fs[x + 1]);
      PUTCHAR(s[x + 1]);
      GotoPos(x, y);
      SetAttr(as[x]);
      SetFont(fs[x]);
      INSERTCHAR(s[x]);
    }
  else if (delete_lp)
    {
      if (DC)
	PutStr(DC);
      else if (CDC)
	CPutStr(CDC, 1);
      else if (CE)
	PutStr(CE);
    }
}

void
SetLastPos(x,y)
int x,y;
{
  ASSERT(display);
  curx = x;
  cury = y;
}

int
ResizeDisplay(wi, he)
int wi, he;
{
  ASSERT(display);
  debug2("ResizeDisplay: to (%d,%d).\n", wi, he);
  if (width == wi && height == he)
    {
      debug("ResizeDisplay: No change\n");
      return 0;
    }
  if (CWS)
    {
      debug("ResizeDisplay: using WS\n");
      PutStr(tgoto(CWS, wi, he));
      ChangeScreenSize(wi, he, 0);
      return 0;
    }
  else if (CZ0 && (wi == Z0width || wi == Z1width))
    {
      debug("ResizeDisplay: using Z0/Z1\n");
      PutStr(wi == Z0width ? CZ0 : CZ1);
      ChangeScreenSize(wi, height, 0);
      return (he == height) ? 0 : -1;
    }
  return -1;
}

void
ChangeScrollRegion(newtop, newbot)
int newtop, newbot;
{
  if (display == 0)
    return;
  if (CS == 0)
    {
      top = 0;
      bot = height - 1;
      return;
    }
  if (top == newtop && bot == newbot)
    return;
  debug2("ChangeScrollRegion: (%d - %d)\n", newtop, newbot);
  PutStr(tgoto(CS, newbot, newtop));
  top = newtop;
  bot = newbot;
  cury = curx = -1;		/* Just in case... */
}


/*
 *  Layer creation / removal
 */

int
InitOverlayPage(datasize, lf, block)
int datasize;
struct LayFuncs *lf;
int block;
{
  char *data;
  struct layer *newlay;

  RemoveStatus();
  debug3("Entering new layer  display %#x  dfore %#x  oldlay %#x\n", 
  	 (unsigned int)display, (unsigned int)dfore, (unsigned int)lay);
  if ((newlay = (struct layer *)malloc(sizeof(struct layer))) == 0)
    {
      Msg(0, "No memory for layer struct");
      return(-1);
    }
  data = 0;
  if (datasize)
    {
      if ((data = malloc(datasize)) == 0)
	{
	  free(newlay);
	  Msg(0, "No memory for layer data");
	  return(-1);
	}
      bzero(data, datasize);
    }
  newlay->l_layfn = lf;
  newlay->l_block = block | lay->l_block;
  newlay->l_data = data;
  newlay->l_next = lay;
  if (dfore)
    {
      dfore->wlay = newlay;
      dfore->active = 0;
    }
  lay = newlay;
  layfn = newlay->l_layfn;
  Restore();
  return(0);
}

void
ExitOverlayPage()
{
  struct layer *oldlay;

  debug3("Exiting layer  display %#x  fore %#x  lay %#x\n", 
         (unsigned int)display, (unsigned int)dfore, (unsigned int)lay);
  oldlay = lay;
  if (oldlay->l_data)
    free(oldlay->l_data);
  lay = oldlay->l_next;
  layfn = lay->l_layfn;
  free(oldlay);
  if (dfore)
    dfore->wlay = lay;
  Restore();
  SetCursor();
}


/*
 *  Output buffering routines
 */

void
AddStr(str)
char *str;
{
  register char c;

  ASSERT(display);
  while (c = *str++)
    AddChar(c);
}

void
AddStrn(str, n)
char *str;
int n;
{
  register char c;

  ASSERT(display);
  while ((c = *str++) && n-- > 0)
    AddChar(c);
  while (n-- > 0)
    AddChar(' ');
}

void
Flush()
{
  register int l;
  register char *p;

  ASSERT(display);
  l = obufp - obuf;
  debug1("Flush(): %d\n", l);
  ASSERT(l + obuffree == obuflen);
  if (l == 0)
    return;
  p = obuf;
  if (fcntl(userfd, F_SETFL, 0))
    debug1("Warning: DELAY fcntl failed: %d\n", errno);
  while (l)
    {
      register int wr;
      wr = write(userfd, p, l);
      if (wr <= 0) 
	{
	  if (errno == EINTR) 
	    continue;
	  debug1("Writing to display: %d\n", errno);
	  wr = l;
	}
      obuffree += wr;
      p += wr;
      l -= wr;
    }
  obuffree += l;
  obufp = obuf;
  if (fcntl(userfd, F_SETFL, FNDELAY))
    debug1("Warning: NDELAY fcntl failed: %d\n", errno);
}


/*
 *  Asynchronous output routines by
 *  Tim MacKenzie (tym@dibbler.cs.monash.edu.au)
 */

void
Resize_obuf()
{
  register int ind;

  ASSERT(display);
  ind  = obufp - obuf;
  if (obuflen && obuf)
    {
      obuflen += GRAIN;
      obuffree += GRAIN;
      obuf = realloc(obuf, obuflen);
    }
  else
    {
      obuflen = GRAIN;
      obuffree = GRAIN;
      obuf = malloc(obuflen);
    }
  if (!obuf)
    Panic(0, "Out of memory");
  obufp = obuf + ind;
  debug1("ResizeObuf: resized to %d\n", obuflen);
}

#ifdef AUTO_NUKE
void
NukePending()
{/* Nuke pending output in current display, clear screen */
  register int len;
  int oldfont = font, oldattr = attr, oldtop = top, oldbot = bot;
  int oldkeypad = keypad, oldcursorkeys = cursorkeys;

  len = obufp - obuf;
  debug1("NukePending: nuking %d chars\n", len);
  
  /* Throw away any output that we can... */
# ifdef POSIX
  tcflush(userfd, TCOFLUSH);
# else
#  ifdef TCFLUSH
  (void) ioctl(userfd, TCFLUSH, (char *) 1);
#  endif
# endif

  obufp = obuf;
  obuffree += len;
  top = bot = -1;
  PutStr(TI);
  PutStr(IS);
  /* Check for toggle */
  if (IM && strcmp(IM, EI))
    PutStr(EI);
  insert = 0;
  /* Check for toggle */
  if (KS && strcmp(KS, KE))
    PutStr(KE);
  keypad = 0;
  if (CCS && strcmp(CCS, CCE))
    PutStr(CCE);
  cursorkeys = 0;
  PutStr(CE0);
  font = ASCII;
  ChangeScrollRegion(oldtop, oldbot);
  SetAttrFont(oldattr, oldfont);
  KeypadMode(oldkeypad);
  CursorkeysMode(oldcursorkeys);
  if (CWS)
    {
      debug("ResizeDisplay: using WS\n");
      PutStr(tgoto(CWS, width, height));
    }
  else if (CZ0 && (width == Z0width || width == Z1width))
    {
      debug("ResizeDisplay: using Z0/Z1\n");
      PutStr(width == Z0width ? CZ0 : CZ1);
    }
}
#endif /* AUTO_NUKE */
