/* Copyright (c) 1991
 *      Juergen Weigert (jnweiger@immd4.informatik.uni-erlangen.de)
 *      Michael Schroeder (mlschroe@immd4.informatik.uni-erlangen.de)
 * Copyright (c) 1987 Oliver Laumann
 * All rights reserved.  Not derived from licensed software.
 *
 * Permission is granted to freely use, copy, modify, and redistribute
 * this software, provided that no attempt is made to gain profit from it,
 * the authors are not construed to be liable for any results of using the
 * software, alterations are clearly marked as such, and this notice is
 * not modified.
 *
 * Noteworthy contributors to screen's design and implementation:
 *	Wayne Davison (davison@borland.com)
 *	Patrick Wolfe (pat@kai.com, kailand!pat)
 *	Bart Schaefer (schaefer@cse.ogi.edu)
 *	Nathan Glasser (nathan@brokaw.lcs.mit.edu)
 *	Larry W. Virden (lvirden@cas.org)
 *	Howard Chu (hyc@hanauma.jpl.nasa.gov)
 *	Tim MacKenzie (tym@dibbler.cs.monash.edu.au)
 *	Markku Jarvinen (mta@{cc,cs,ee}.tut.fi)
 *	Marc Boucher (marc@CAM.ORG)
 *
 ****************************************************************
 */

#include "rcs.h"
RCS_ID("$Id: ansi.c,v 1.35 92/11/25 20:47:39 mlschroe Exp $ FAU")

#include <stdio.h>
#include <sys/types.h>
#ifdef BSDI
#include <sys/signal.h>
#endif /* BSDI */
#include <fcntl.h>
#ifndef sun	/* we want to know about TIOCPKT. */
# include <sys/ioctl.h>
#endif
#include "config.h"
#include "screen.h"
#include "extern.h"

extern char *getenv(), *tgetstr(), *tgoto();
#ifndef __STDC__
extern char *malloc();
#endif

extern struct win *windows;	/* linked list of all windows */
extern struct win *fore;
extern int  force_vt;
extern int  all_norefresh;	/* => display */
extern char Esc, MetaEsc;	/* => display */

int maxwidth;			/* maximum of all widths so far */

int Z0width, Z1width;		/* widths for Z0/Z1 switching */

struct display *display;

static struct win *curr;	/* window we are working on */
static int rows, cols;		/* window size of the curr window */

int default_wrap = 1;		/* default: wrap on */
int default_monitor = 0; 

int visual_bell = 0;
int use_hardstatus = 1;

char *blank;			/* line filled with spaces */
char *null;			/* line filled with '\0' */
char *OldImage, *OldAttr, *OldFont;	/* temporary buffers */

time_t time();

static void WinProcess __P((char **, int *));
static void WinRedisplayLine __P((int, int, int, int));
static void WinClearLine __P((int, int, int));
static int  WinRewrite __P((int, int, int, int));
static void WinSetCursor __P(());
static int  WinResize __P((int, int));
static void WinRestore __P((void));
static int  Special __P((int));
static void DoESC __P((int, int ));
static void DoCSI __P((int, int ));
static void SetChar __P(());
static void StartString __P((enum string_t));
static void SaveChar __P((int ));
static void PrintChar __P((int ));
static void PrintFlush __P((void));
static void DesignateCharset __P((int, int ));
static void MapCharset __P((int));
static void SaveCursor __P((void));
static void RestoreCursor __P((void));
static void BackSpace __P((void));
static void Return __P((void));
static void LineFeed __P((int));
static void ReverseLineFeed __P((void));
static void InsertAChar __P((int));
static void InsertChar __P((int));
static void DeleteChar __P((int));
static void DeleteLine __P((int));
static void InsertLine __P((int));
static void ScrollUpMap __P((int));
static void ScrollDownMap __P((int));
static void Scroll __P((char *, int, int, char *));
static void ForwardTab __P((void));
static void BackwardTab __P((void));
static void ClearScreen __P((void));
static void ClearFromBOS __P((void));
static void ClearToEOS __P((void));
static void ClearFullLine __P((void));
static void ClearToEOL __P((void));
static void ClearFromBOL __P((void));
static void ClearInLine __P((int, int, int));
static void CursorRight __P(());
static void CursorUp __P(());
static void CursorDown __P(());
static void CursorLeft __P(());
static void ASetMode __P((int));
static void SelectRendition __P((void));
static void RestorePosAttrFont __P((void));
static void FillWithEs __P((void));
static void ChangeLine __P((char *, char *, char *, int, int, int ));
static void FindAKA __P((void));
static void Report __P((char *, int, int));


/*
 *  The window layer functions
 */

struct LayFuncs WinLf =
{
  WinProcess,
  0,
  WinRedisplayLine,
  WinClearLine,
  WinRewrite,
  WinSetCursor,
  WinResize,
  WinRestore
};

static void
WinProcess(bufpp, lenp)
char **bufpp;
int *lenp;
{
  int f, l = *lenp;
  
  fore = dfore;
  f = sizeof(fore->inbuf) - fore->inlen;
  if (l > f)
    l = f;
  if (l > 0)
    {
      bcopy(*bufpp, fore->inbuf + fore->inlen, l);
      fore->inlen += l;
    }
  *bufpp += *lenp;
  *lenp = 0;
}

static void
WinRedisplayLine(y, from, to, isblank)
int y, from, to, isblank;
{
  if (y < 0)
    return;
  fore = dfore;
  DisplayLine(isblank ? blank: null, null, null, fore->w_image[y],
	      fore->w_attr[y], fore->w_font[y], y, from, to);
}

static int
WinRewrite(y, x1, x2, doit)
int y, x1, x2, doit;
{
  register int cost, dx;
  register char *p, *f, *i;

  fore = dfore;
  dx = x2 - x1;
  if (doit)
    {
      i = fore->w_image[y] + x1;
      while (dx-- > 0)
	PUTCHAR(*i++);
      return(0);
    }
  p = fore->w_attr[y] + x1;
  f = fore->w_font[y] + x1;

  cost = dx = x2 - x1;
  if (insert)
    cost += EIcost + IMcost;
  while(dx-- > 0)
    {
      if (*p++ != attr || *f++ != font)
	return EXPENSIVE;
    }
  return cost;
}

static void
WinClearLine(y, xs, xe)
int y, xs, xe;
{
  fore = dfore;
  DisplayLine(fore->w_image[y], fore->w_attr[y], fore->w_font[y],
	      blank, null, null, y, xs, xe);
}

static void
WinSetCursor()
{
  fore = dfore;
  GotoPos(fore->x, fore->y);
}

static int
WinResize(wi, he)
int wi, he;
{
  fore = dfore;
  if (fore)
    ChangeWindowSize(fore, wi, he);
  return 0;
}

static void
WinRestore()
{
  fore = dfore;
  if (ResizeDisplay(fore->w_width, fore->w_height))
    {
      debug2("Cannot resize from (%d,%d)", width, height);
      debug2(" to (%d,%d) -> resize window\n", fore->w_width, fore->w_height);
      DoResize(width, height);
    }
  ChangeScrollRegion(fore->w_top, fore->w_bot);
  KeypadMode(fore->w_keypad);
  CursorkeysMode(fore->w_cursorkeys);
  SetFlow(fore->w_flow & FLOW_NOW);
  InsertMode(fore->w_insert);
  fore->active = 1;
}

/* 
 *  Activate - make fore window active
 *  norefresh = -1 forces a refresh, disregard all_norefresh then.
 */
void
Activate(norefresh)
int norefresh;
{
  debug1("Activate(%d)\n", norefresh);
  if (display == 0)
    return;
  RemoveStatus();
  if (fore = dfore)
    {
      ASSERT(fore->display == display);
      fore->active = layfn == &WinLf;
      if (fore->monitor != MON_OFF)
	fore->monitor = MON_ON;
      fore->bell = BELL_OFF;
    }
  Redisplay(norefresh + all_norefresh);
}

void
ResetWindow(p)
register struct win *p;
{
  register int i;

  p->wrap = default_wrap;
  p->origin = 0;
  p->w_insert = 0;
  p->vbwait = 0;
  p->w_keypad = 0;
  p->w_cursorkeys = 0;
  p->w_top = 0;
  p->w_bot = p->w_height - 1;
  p->saved = 0;
  p->Attr = 0;
  p->x = p->y = 0;
  p->state = LIT;
  p->StringType = NONE;
  p->ss = 0;
  p->Charset = G0;
  bzero(p->tabs, p->w_width);
  for (i = 8; i < p->w_width; i += 8)
    p->tabs[i] = 1;
  for (i = G0; i <= G3; i++)
    p->charsets[i] = ASCII;
}


/*
 *  Here comes the vt100 emulator
 */
void
WriteString(wp, buf, len)
struct win *wp;
register char *buf;
register int len;
{
  register int c;

  if (!len)
    return;
  if (wp->logfp != NULL)
    if ((int)fwrite(buf, len, 1, wp->logfp) < 1)
      {
	extern int errno;

	Msg(errno, "Error writing logfile");
	fclose(wp->logfp);
	wp->logfp = NULL;
      }
  /*
   * SetCurr() here may prevent output, as it may set display = 0
   */
  SetCurr(wp);
  if (display)
    {
      if (status && !(use_hardstatus && HS))
	RemoveStatus();
    }
  else
    {
      if (curr->monitor == MON_ON || curr->monitor == MON_DONE)
	{
          debug2("ACTIVITY %d %d\n", curr->monitor, curr->bell);
          curr->monitor = MON_FOUND;
	}
    }
  do
    {
      c = (unsigned char)*buf++;
      if (c == '\177')
	continue;

      /* The next part is only for speedup */
      if (curr->state == LIT &&
          c >= ' ' && ((c & 0x80) == 0 || !CB8) &&
          !curr->w_insert && !curr->ss)
	{
	  register int currx;
	  register char *imp, *atp, *fop, at, fo;

	  currx = curr->x;
	  imp = curr->w_image[curr->y] + currx;
	  atp = curr->w_attr[curr->y] + currx;
	  fop = curr->w_font[curr->y] + currx;
	  at = curr->Attr;
	  fo = curr->charsets[curr->Charset];
	  if (display)
	    {
	      if (curx != currx || cury != curr->y)
		GotoPos(currx, curr->y);
	      if (at != attr)
		SetAttr(at);
	      if (fo != font)
		SetFont(fo);
	      if (insert)
		InsertMode(0);
	    }
	  while (currx < cols - 1)
	    {
	      if (display)
		AddChar(font != '0' ? c : c0_tab[c]);
	      *imp++ = c;
	      *atp++ = at;
	      *fop++ = fo;
	      currx++;
skip:	      if (--len == 0)
		break;
              c = (unsigned char)*buf++;
	      if (c == '\177')
		goto skip;
	      if (c < ' ' || ((c & 0x80) && CB8))
		break;
	    }
	  curr->x = currx;
	  if (display)
	    curx = currx;
	  if (len == 0)
	    break;
	}
      /* end of speedup code */

      if ((c & 0x80) && CB8)
	{
	  FILE *logfp = wp->logfp;
	  char *cb8 = CB8;
	
	  wp->logfp = NULL;	/* a little hack */
	  CB8 = NULL;		/* dito */
	  WriteString(wp, cb8, strlen(cb8));
	  wp->logfp = logfp;
	  CB8 = cb8;
	  c &= 0x7f;
	}
    tryagain:
      switch (curr->state)
	{
	case PRIN:
	  switch (c)
	    {
	    case '\033':
	      curr->state = PRINESC;
	      break;
	    default:
	      PrintChar(c);
	    }
	  break;
	case PRINESC:
	  switch (c)
	    {
	    case '[':
	      curr->state = PRINCSI;
	      break;
	    default:
	      PrintChar('\033');
	      PrintChar(c);
	      curr->state = PRIN;
	    }
	  break;
	case PRINCSI:
	  switch (c)
	    {
	    case '4':
	      curr->state = PRIN4;
	      break;
	    default:
	      PrintChar('\033');
	      PrintChar('[');
	      PrintChar(c);
	      curr->state = PRIN;
	    }
	  break;
	case PRIN4:
	  switch (c)
	    {
	    case 'i':
	      curr->state = LIT;
	      PrintFlush();
	      break;
	    default:
	      PrintChar('\033');
	      PrintChar('[');
	      PrintChar('4');
	      PrintChar(c);
	      curr->state = PRIN;
	    }
	  break;
	case STRESC:
	  switch (c)
	    {
	    case '\\':
	      curr->state = LIT;
	      *(curr->stringp) = '\0';
	      switch (curr->StringType)
		{
		case PM:
		  if (!display)
		    break;
		  MakeStatus(curr->string);
		  if (status && !(use_hardstatus && HS) && len > 1)
		    {
		      curr->outlen = len - 1;
		      bcopy(buf, curr->outbuf, curr->outlen);
		      return;
		    }
		  break;
		case DCS:
		  if (display)
		    AddStr(curr->string);
		  break;
		case AKA:
		  if (curr->akapos == 0 && !*curr->string)
		    break;
		  strncpy(curr->cmd + curr->akapos, curr->string, 20);
		  if (!*curr->string)
		    curr->autoaka = curr->y + 1;
		  break;
		default:
		  break;
		}
	      break;
	    default:
	      curr->state = ASTR;
	      SaveChar('\033');
	      SaveChar(c);
	    }
	  break;
	case ASTR:
	  switch (c)
	    {
	    case '\0':
	      break;
	    case '\033':
	      curr->state = STRESC;
	      break;
	    default:
	      SaveChar(c);
	    }
	  break;
	case ESC:
	  switch (c)
	    {
	    case '[':
	      curr->NumArgs = 0;
	      curr->intermediate = 0;
	      bzero((char *) curr->args, MAXARGS * sizeof(int));
	      curr->state = CSI;
	      break;
	    case ']':
	      StartString(OSC);
	      break;
	    case '_':
	      StartString(APC);
	      break;
	    case 'P':
	      StartString(DCS);
	      break;
	    case '^':
	      StartString(PM);
	      break;
	    case '"':
	    case 'k':
	      StartString(AKA);
	      break;
	    default:
	      if (Special(c))
		break;
	      debug1("not special. c = %x\n", c);
	      if (c >= ' ' && c <= '/')
		curr->intermediate = curr->intermediate ? -1 : c;
	      else if (c >= '0' && c <= '~')
		{
		  DoESC(c, curr->intermediate);
		  curr->state = LIT;
		}
	      else
		{
		  curr->state = LIT;
		  goto tryagain;
		}
	    }
	  break;
	case CSI:
	  switch (c)
	    {
	    case '0':
	    case '1':
	    case '2':
	    case '3':
	    case '4':
	    case '5':
	    case '6':
	    case '7':
	    case '8':
	    case '9':
	      if (curr->NumArgs < MAXARGS)
		{
		  curr->args[curr->NumArgs] =
		    10 * curr->args[curr->NumArgs] + c - '0';
		}
	      break;
	    case ';':
	    case ':':
	      curr->NumArgs++;
	      break;
	    default:
	      if (Special(c))
		break;
	      if (c >= '@' && c <= '~')
		{
		  curr->NumArgs++;
		  DoCSI(c, curr->intermediate);
		  if (curr->state != PRIN)
		    curr->state = LIT;
		}
	      else if ((c >= ' ' && c <= '/') || (c >= '<' && c <= '?'))
		curr->intermediate = curr->intermediate ? -1 : c;
	      else
		{
		  curr->state = LIT;
		  goto tryagain;
		}
	    }
	  break;
	case LIT:
	default:
	  if (c < ' ')
	    {
	      if (c == '\033')
		{
		  curr->intermediate = 0;
		  curr->state = ESC;
		  if (display && lp_missing && (CIC || IC || IM))
		    {
		      ChangeLine(blank, null, null, bot,
				 cols - 2, cols - 1);
		      GotoPos(curr->x, curr->y);
		    }
		  if (curr->autoaka < 0)
		    curr->autoaka = 0;
		}
	      else
		Special(c);
	      break;
	    }
	  if (display)
	    {
	      if (attr != curr->Attr)
		SetAttr(curr->Attr);
	      if (font != curr->charsets[curr->ss ? curr->ss : curr->Charset])
		SetFont(curr->charsets[curr->ss ? curr->ss : curr->Charset]);
	    }
	  if (curr->x < cols - 1)
	    {
	      if (curr->w_insert)
		InsertAChar(c);
	      else
		{
		  if (display)
		    PUTCHAR(c);
		  SetChar(c);
		}
	      curr->x++;
	    }
	  else if (curr->x == cols - 1)
	    {
	      if (display && curr->wrap && (CLP || !force_vt || COP))
		{
		  RAW_PUTCHAR(c);	/* don't care about insert */
		  SetChar(c);
		  if (AM && !CLP)
		    {
		      curr->x = 0;	/* terminal auto-wrapped */
		      LineFeed(0);
		    }
		  else
		    curr->x++;
		}
	      else
		{
		  if (display)
		    {
		      if (CLP || curr->y != bot)
			{
			  RAW_PUTCHAR(c);
			  GotoPos(curr->x, curr->y);
			}
		      else
			CheckLP(c);
		    }
		  SetChar(c);
		  if (curr->wrap)
		    curr->x++;
		}
	    }
	  else /* curr->x > cols - 1 */
	    {
	      LineFeed(2); /* cr+lf, handle LP */
	      if (curr->w_insert)
		InsertAChar(c);
	      else
		{
		  if (display)
		    PUTCHAR(c);
		  SetChar(c);
		}
	      curr->x = 1;
	    }
	  if (curr->ss)
	    {
	      SetFont(curr->charsets[curr->Charset]);
	      curr->ss = 0;
	    }
	  break;
	}
    } while (--len);
  curr->outlen = 0;
  if (curr->state == PRIN)
    PrintFlush();
}

static int
Special(c)
register int c;
{
  switch (c)
    {
    case '\b':
      BackSpace();
      return 1;
    case '\r':
      Return();
      return 1;
    case '\n':
      if (curr->autoaka)
	FindAKA();
      LineFeed(1);
      return 1;
    case '\007':
      if (display == 0)
	curr->bell = BELL_ON;
      else
	{
	  if (!visual_bell)
	    PutStr(BL);
	  else
	    {
	      if (!VB)
		curr->bell = BELL_VISUAL;
	      else
		PutStr(VB);
	    }
	}
      return 1;
    case '\t':
      ForwardTab();
      return 1;
    case '\017':		/* SI */
      MapCharset(G0);
      return 1;
    case '\016':		/* SO */
      MapCharset(G1);
      return 1;
    }
  return 0;
}

static void
DoESC(c, intermediate)
int c, intermediate;
{
  debug2("DoESC: %x - inter = %x\n", c, intermediate);
  switch (intermediate)
    {
    case 0:
      switch (c)
	{
	case 'E':
	  LineFeed(2);
	  break;
	case 'D':
	  LineFeed(1);
	  break;
	case 'M':
	  ReverseLineFeed();
	  break;
	case 'H':
	  curr->tabs[curr->x] = 1;
	  break;
	case 'Z':		/* jph: Identify as VT100 */
	  Report("\033[?%d;%dc", 1, 2);
	  break;
	case '7':
	  SaveCursor();
	  break;
	case '8':
	  RestoreCursor();
	  break;
	case 'c':
	  ClearScreen();
	  ResetWindow(curr);
	  SetAttrFont(0, ASCII);
	  InsertMode(0);
	  KeypadMode(0);
	  CursorkeysMode(0);
	  ChangeScrollRegion(0, rows - 1);
	  break;
	case '=':
	  KeypadMode(curr->w_keypad = 1);
#ifndef TIOCPKT
	  NewAutoFlow(curr, 0);
#endif /* !TIOCPKT */
	  break;
	case '>':
	  KeypadMode(curr->w_keypad = 0);
#ifndef TIOCPKT
	  NewAutoFlow(curr, 1);
#endif /* !TIOCPKT */
	  break;
	case 'n':		/* LS2 */
	  MapCharset(G2);
	  break;
	case 'o':		/* LS3 */
	  MapCharset(G3);
	  break;
	case 'N':		/* SS2 */
	  if (curr->charsets[curr->Charset] != curr->charsets[G2])
	    curr->ss = G2;
	  else
	    curr->ss = 0;
	  break;
	case 'O':		/* SS3 */
	  if (curr->charsets[curr->Charset] != curr->charsets[G3])
	    curr->ss = G3;
	  else
	    curr->ss = 0;
	  break;
	}
      break;
    case '#':
      switch (c)
	{
	case '8':
	  FillWithEs();
	  break;
	}
      break;
    case '(':
      DesignateCharset(c, G0);
      break;
    case ')':
      DesignateCharset(c, G1);
      break;
    case '*':
      DesignateCharset(c, G2);
      break;
    case '+':
      DesignateCharset(c, G3);
      break;
    }
}

static void
DoCSI(c, intermediate)
int c, intermediate;
{
  register int i, a1 = curr->args[0], a2 = curr->args[1];

  if (curr->NumArgs > MAXARGS)
    curr->NumArgs = MAXARGS;
  switch (intermediate)
    {
    case 0:
      switch (c)
	{
	case 'H':
	case 'f':
	  if (a1 < 1)
	    a1 = 1;
	  if (curr->origin)
	    a1 += curr->w_top;
	  if (a1 > rows)
	    a1 = rows;
	  if (a2 < 1)
	    a2 = 1;
	  if (a2 > cols)
	    a2 = cols;
	  GotoPos(--a2, --a1);
	  curr->x = a2;
	  curr->y = a1;
	  if (curr->autoaka)
	    curr->autoaka = a1 + 1;
	  break;
	case 'J':
	  if (a1 < 0 || a1 > 2)
	    a1 = 0;
	  switch (a1)
	    {
	    case 0:
	      ClearToEOS();
	      break;
	    case 1:
	      ClearFromBOS();
	      break;
	    case 2:
	      ClearScreen();
	      GotoPos(curr->x, curr->y);
	      break;
	    }
	  break;
	case 'K':
	  if (a1 < 0 || a1 > 2)
	    a1 %= 3;
	  switch (a1)
	    {
	    case 0:
	      ClearToEOL();
	      break;
	    case 1:
	      ClearFromBOL();
	      break;
	    case 2:
	      ClearFullLine();
	      break;
	    }
	  break;
	case 'A':
	  CursorUp(a1 ? a1 : 1);
	  break;
	case 'B':
	  CursorDown(a1 ? a1 : 1);
	  break;
	case 'C':
	  CursorRight(a1 ? a1 : 1);
	  break;
	case 'D':
	  CursorLeft(a1 ? a1 : 1);
	  break;
	case 'm':
	  SelectRendition();
	  break;
	case 'g':
	  if (a1 == 0)
	    curr->tabs[curr->x] = 0;
	  else if (a1 == 3)
	    bzero(curr->tabs, cols);
	  break;
	case 'r':
	  if (!a1)
	    a1 = 1;
	  if (!a2)
	    a2 = rows;
	  if (a1 < 1 || a2 > rows || a1 >= a2)
	    break;
	  curr->w_top = a1 - 1;
	  curr->w_bot = a2 - 1;
	  ChangeScrollRegion(curr->w_top, curr->w_bot);
	  if (curr->origin)
	    {
	      GotoPos(0, curr->w_top);
	      curr->y = curr->w_top;
	      curr->x = 0;
	    }
	  else
	    {
	      GotoPos(0, 0);
	      curr->y = curr->x = 0;
	    }
	  break;
	case 's':
	  SaveCursor();
	  break;
	case 't':
	  if (a1 != 8)
	    break;
	  a1 = curr->args[2];
	  if (a1 < 1)
	    a1 = curr->w_width;
	  if (a2 < 1)
	    a2 = curr->w_height;
	  if (CWS == NULL)
	    {
	      a2 = curr->w_height;
	      if (CZ0 == NULL || (a1 != Z0width && a1 != Z1width))
	        a1 = curr->w_width;
 	    }
	  if (a1 == curr->w_width && a2 == curr->w_height)
	    break;
          ChangeWindowSize(curr, a1, a2);
	  SetCurr(curr);
	  if (display)
	    Activate(0);
	  break;
	case 'u':
	  RestoreCursor();
	  break;
	case 'I':
	  if (!a1)
	    a1 = 1;
	  while (a1--)
	    ForwardTab();
	  break;
	case 'Z':
	  if (!a1)
	    a1 = 1;
	  while (a1--)
	    BackwardTab();
	  break;
	case 'L':
	  InsertLine(a1 ? a1 : 1);
	  break;
	case 'M':
	  DeleteLine(a1 ? a1 : 1);
	  break;
	case 'P':
	  DeleteChar(a1 ? a1 : 1);
	  break;
	case '@':
	  InsertChar(a1 ? a1 : 1);
	  break;
	case 'h':
	  ASetMode(1);
	  break;
	case 'l':
	  ASetMode(0);
	  break;
	case 'i':
	  if (PO && a1 == 5)
	    {
	      curr->stringp = curr->string;
	      curr->state = PRIN;
	    }
	  break;
	case 'n':
	  if (a1 == 6)		/* Report cursor position */
	    Report("\033[%d;%dR", curr->y + 1, curr->x + 1);
	  break;
	case 'c':		/* Identify as VT100 */
	  Report("\033[?%d;%dc", 1, 2);
	  break;
	}
      break;
    case '?':
      debug2("\\E[?%d%c\n",a1,c);
      if (c != 'h' && c != 'l')
	break;
      i = (c == 'h');
      switch (a1)
	{
	case 1:
	  CursorkeysMode(curr->w_cursorkeys = i);
#ifndef TIOCPKT
	  NewAutoFlow(curr, !i);
#endif /* !TIOCPKT */
	  break;
	case 3:
	  i = (i ? Z0width : Z1width);
	  if ((CZ0 || CWS) && curr->w_width != i)
	    {
              ChangeWindowSize(curr, i, curr->w_height);
	      SetCurr(curr);
	      if (display)
		Activate(0);
	    }
	  break;
	case 5:
	  if (i)
	    curr->vbwait = 1;
	  else
	    {
	      if (curr->vbwait)
		PutStr(VB);
	      curr->vbwait = 0;
	    }
	  break;
	case 6:
	  if ((curr->origin = i) != 0)
	    {
	      GotoPos(0, curr->w_top);
	      curr->y = curr->w_top;
	      curr->x = 0;
	    }
	  else
	    {
	      GotoPos(0, 0);
	      curr->y = curr->x = 0;
	    }
	  break;
	case 7:
	  curr->wrap = i;
	  break;
	case 35:
	  debug1("Cursor %svisible\n", i ? "in" : "");
	  curr->cursor_invisible = i;
	  break;
	}
      break;
    }
}


static void
SetChar(c)
register int c;
{
  register struct win *p = curr;

  p->w_image[p->y][p->x] = c;
  p->w_attr[p->y][p->x] = p->Attr;
  p->w_font[p->y][p->x] = p->charsets[p->ss ? p->ss : p->Charset];
}

static void
StartString(type)
enum string_t type;
{
  curr->StringType = type;
  curr->stringp = curr->string;
  curr->state = ASTR;
}

static void
SaveChar(c)
int c;
{
  if (curr->stringp >= curr->string + MAXSTR - 1)
    curr->state = LIT;
  else
    *(curr->stringp)++ = c;
}

static void
PrintChar(c)
int c;
{
  if (curr->stringp >= curr->string + MAXSTR - 1)
    PrintFlush();
  *(curr->stringp)++ = c;
}

static void
PrintFlush()
{
  if (curr->stringp > curr->string)
    {
      PutStr(PO);
      AddStrn(curr->string, curr->stringp - curr->string);
      PutStr(PF);
      Flush();
      curr->stringp = curr->string;
    }
}


void
NewAutoFlow(win, on)
struct win *win;
int on;
{
  debug1("NewAutoFlow: %d\n", on);
  SetCurr(win);
  if (win->w_flow & FLOW_AUTOFLAG)
    win->w_flow = FLOW_AUTOFLAG | (FLOW_AUTO|FLOW_NOW) * on;
  else
    win->w_flow = (win->w_flow & ~FLOW_AUTO) | FLOW_AUTO * on;
  if (display)
    SetFlow(win->w_flow & FLOW_NOW);
}

static void
DesignateCharset(c, n)
int c, n;
{
  curr->ss = 0;
  if (c == 'B')
    c = ASCII;
  if (curr->charsets[n] != c)
    {
      curr->charsets[n] = c;
      if (curr->Charset == n)
	SetFont(c);
    }
}

static void
MapCharset(n)
int n;
{
  curr->ss = 0;
  if (curr->Charset != n)
    {
      curr->Charset = n;
      SetFont(curr->charsets[n]);
    }
}

static void
SaveCursor()
{
  curr->saved = 1;
  curr->Saved_x = curr->x;
  curr->Saved_y = curr->y;
  curr->SavedAttr = curr->Attr;
  curr->SavedCharset = curr->Charset;
  bcopy((char *) curr->charsets, (char *) curr->SavedCharsets,
	4 * sizeof(int));
}

static void
RestoreCursor()
{
  if (curr->saved)
    {
      GotoPos(curr->Saved_x, curr->Saved_y);
      curr->x = curr->Saved_x;
      curr->y = curr->Saved_y;
      curr->Attr = curr->SavedAttr;
      SetAttr(curr->Attr);
      bcopy((char *) curr->SavedCharsets, (char *) curr->charsets,
	    4 * sizeof(int));
      curr->Charset = curr->SavedCharset;
      SetFont(curr->charsets[curr->Charset]);
    }
}

static void
BackSpace()
{
  if (curr->x > 0)
    {
      curr->x--;
    }
  else if (curr->wrap && curr->y > 0)
    {
      curr->x = cols - 1;
      curr->y--;
    }
  if (display)
    GotoPos(curr->x, curr->y);
}

static void
Return()
{
  if (curr->x > 0)
    {
      curr->x = 0;
      if (display)
        GotoPos(curr->x, curr->y);
    }
}

static void
LineFeed(out_mode)
int out_mode;
{
  /* out_mode: 0=no-output lf, 1=lf, 2=cr+lf */
  if (out_mode == 2)
    curr->x = 0;
  if (curr->y != curr->w_bot)		/* Don't scroll */
    {
      if (curr->y < rows-1)
	curr->y++;
      if (out_mode && display)
	GotoPos(curr->x, curr->y);
      return;
    }
  ScrollUpMap(1);
  if (curr->autoaka > 1)
    curr->autoaka--;
  if (out_mode && display)
    {
      ScrollRegion(curr->w_top, curr->w_bot, 1);
      GotoPos(curr->x, curr->y);
    }
}

static void
ReverseLineFeed()
{
  if (curr->y == curr->w_top)
    {
      ScrollDownMap(1);
      if (!display)
	return;
      ScrollRegion(curr->w_top, curr->w_bot, -1);
      GotoPos(curr->x, curr->y);
    }
  else if (curr->y > 0)
    CursorUp(1);
}

static void
InsertAChar(c)
int c;
{
  register int y = curr->y, x = curr->x;

  if (x == cols)
    x--;
  bcopy(curr->w_image[y], OldImage, cols);
  bcopy(curr->w_attr[y], OldAttr, cols);
  bcopy(curr->w_font[y], OldFont, cols);
  bcopy(curr->w_image[y] + x, curr->w_image[y] + x + 1, cols - x - 1);
  bcopy(curr->w_attr[y] + x, curr->w_attr[y] + x + 1, cols - x - 1);
  bcopy(curr->w_font[y] + x, curr->w_font[y] + x + 1, cols - x - 1);
  SetChar(c);
  if (!display)
    return;
  if (CIC || IC || IM)
    {
      InsertMode(curr->w_insert);
      INSERTCHAR(c);
      if (y == bot)
	lp_missing = 0;
    }
  else
    {
      ChangeLine(OldImage, OldAttr, OldFont, y, x, cols - 1);
      GotoPos(++x, y);
    }
}

static void
InsertChar(n)
int n;
{
  register int i, y = curr->y, x = curr->x;

  if (n <= 0)
    return;
  if (x == cols)
    --x;
  bcopy(curr->w_image[y], OldImage, cols);
  bcopy(curr->w_attr[y], OldAttr, cols);
  bcopy(curr->w_font[y], OldFont, cols);
  if (n > cols - x)
    n = cols - x;
  bcopy(curr->w_image[y] + x, curr->w_image[y] + x + n, cols - x - n);
  bcopy(curr->w_attr[y] + x, curr->w_attr[y] + x + n, cols - x - n);
  bcopy(curr->w_font[y] + x, curr->w_font[y] + x + n, cols - x - n);
  ClearInLine(y, x, x + n - 1);
  if (!display)
    return;
  if (IC || CIC || IM)
    {
      if (y == bot)
	lp_missing = 0;
      if (!insert)
	{
	  if (n == 1 && IC)
	    {
	      PutStr(IC);
	      return;
            }
	  if (CIC)
	    {
	      CPutStr(CIC, n);
	      return;
            }
	}
      InsertMode(1);
      for (i = n; i--; )
	INSERTCHAR(' ');
      GotoPos(x, y);
    }
  else
    {
      ChangeLine(OldImage, OldAttr, OldFont, y, x, cols - 1);
      GotoPos(x, y);
    }
}

static void
DeleteChar(n)
int n;
{
  register int i, y = curr->y, x = curr->x;

  if (x == cols)
    --x;
  bcopy(curr->w_image[y], OldImage, cols);
  bcopy(curr->w_attr[y], OldAttr, cols);
  bcopy(curr->w_font[y], OldFont, cols);
  if (n > cols - x)
    n = cols - x;
  bcopy(curr->w_image[y] + x + n, curr->w_image[y] + x, cols - x - n);
  bcopy(curr->w_attr[y] + x + n, curr->w_attr[y] + x, cols - x - n);
  bcopy(curr->w_font[y] + x + n, curr->w_font[y] + x, cols - x - n);
  ClearInLine(y, cols - n, cols - 1);
  if (!display)
    return;
  if (CDC && !(n == 1 && DC))
    {
      CPutStr(CDC, n);
      if (lp_missing && y == bot)
	{
	  FixLP(cols - 1 - n, y);
          GotoPos(x, y);
	}
    }
  else if (DC)
    {
      for (i = n; i; i--)
	PutStr(DC);
      if (lp_missing && y == bot)
	{
	  FixLP(cols - 1 - n, y);
          GotoPos(x, y);
	}
    }
  else
    {
      ChangeLine(OldImage, OldAttr, OldFont, y, x, cols - 1);
      GotoPos(x, y);
    }
}

static void
DeleteLine(n)
int n;
{
  register int old = curr->w_top;
  
  if (curr->y < curr->w_top || curr->y > curr->w_bot)
    return;
  if (n > curr->w_bot - curr->y + 1)
    n = curr->w_bot - curr->y + 1;
  curr->w_top = curr->y;
  ScrollUpMap(n);
  curr->w_top = old;
  if (!display)
    return;
  ScrollRegion(curr->y, curr->w_bot, n);
  GotoPos(curr->x, curr->y);
}

static void
InsertLine(n)
int n;
{
  register int old = curr->w_top;

  if (curr->y < curr->w_top || curr->y > curr->w_bot)
    return;
  if (n > curr->w_bot - curr->y + 1)
    n = curr->w_bot - curr->y + 1;
  curr->w_top = curr->y;
  ScrollDownMap(n);
  curr->w_top = old;
  if (!display)
    return;
  ScrollRegion(curr->y, curr->w_bot, -n);
  GotoPos(curr->x, curr->y);
}


static void
ScrollUpMap(n)
int n;
{
  char tmp[256 * sizeof(char *)];
  register int ii, i, cnt1, cnt2;
  register char **ppi, **ppa, **ppf;

  i = curr->w_top + n;
  cnt1 = n * sizeof(char *);
  cnt2 = (curr->w_bot - i + 1) * sizeof(char *);
  ppi = curr->w_image + i;
  ppa = curr->w_attr + i;
  ppf = curr->w_font + i;
#ifdef COPY_PASTE
  for(ii = curr->w_top; ii < i; ii++)
     AddLineToHist(curr, &curr->w_image[ii], &curr->w_attr[ii], &curr->w_font[ii]);
#endif
  for (i = n; i; --i)
    {
      bclear(*--ppi, cols);
      bzero(*--ppa, cols);
      bzero(*--ppf, cols);
    }
  Scroll((char *) ppi, cnt1, cnt2, tmp);
  Scroll((char *) ppa, cnt1, cnt2, tmp);
  Scroll((char *) ppf, cnt1, cnt2, tmp);
}

static void
ScrollDownMap(n)
int n;
{
  char tmp[256 * sizeof(char *)];
  register int i, cnt1, cnt2;
  register char **ppi, **ppa, **ppf;

  i = curr->w_top;
  cnt1 = (curr->w_bot - i - n + 1) * sizeof(char *);
  cnt2 = n * sizeof(char *);
  Scroll((char *) (ppi = curr->w_image + i), cnt1, cnt2, tmp);
  Scroll((char *) (ppa = curr->w_attr + i), cnt1, cnt2, tmp);
  Scroll((char *) (ppf = curr->w_font + i), cnt1, cnt2, tmp);
  for (i = n; i; --i)
    {
      bclear(*ppi++, cols);
      bzero(*ppa++, cols);
      bzero(*ppf++, cols);
    }
}

static void
Scroll(cp, cnt1, cnt2, tmp)
char *cp, *tmp;
int cnt1, cnt2;
{
  if (!cnt1 || !cnt2)
    return;
  if (cnt1 <= cnt2)
    {
      bcopy(cp, tmp, cnt1);
      bcopy(cp + cnt1, cp, cnt2);
      bcopy(tmp, cp + cnt2, cnt1);
    }
  else
    {
      bcopy(cp + cnt1, tmp, cnt2);
      bcopy(cp, cp + cnt2, cnt1);
      bcopy(tmp, cp, cnt2);
    }
}

static void
ForwardTab()
{
  register int x = curr->x;

  if (x == cols)
    {
      LineFeed(2);
      x = 0;
    }
  if (curr->tabs[x] && x < cols - 1)
    x++;
  while (x < cols - 1 && !curr->tabs[x])
    x++;
  GotoPos(x, curr->y);
  curr->x = x;
}

static void
BackwardTab()
{
  register int x = curr->x;

  if (curr->tabs[x] && x > 0)
    x--;
  while (x > 0 && !curr->tabs[x])
    x--;
  GotoPos(x, curr->y);
  curr->x = x;
}

static void
ClearScreen()
{
  register int i;
  register char **ppi = curr->w_image, **ppa = curr->w_attr, **ppf = curr->w_font;

  for (i = 0; i < rows; ++i)
    {
#ifdef COPY_PASTE
      AddLineToHist(curr, ppi, ppa, ppf);
#endif
      bclear(*ppi++, cols);
      bzero(*ppa++, cols);
      bzero(*ppf++, cols);
    }
  if (display)
    ClearDisplay();
}

static void
ClearFromBOS()
{
  register int n, y = curr->y, x = curr->x;

  if (display)
    Clear(0, 0, x, y);
  for (n = 0; n < y; ++n)
    ClearInLine(n, 0, cols - 1);
  ClearInLine(y, 0, x);
  RestorePosAttrFont();
}

static void
ClearToEOS()
{
  register int n, y = curr->y, x = curr->x;

  if (display)
    Clear(x, y, width - 1, height - 1);
  ClearInLine(y, x, cols - 1);
  for (n = y + 1; n < rows; n++)
    ClearInLine(n, 0, cols - 1);
  RestorePosAttrFont();
}

static void
ClearFullLine()
{
  register int y = curr->y;

  if (display)
    Clear(0, y, width - 1, y);
  ClearInLine(y, 0, cols - 1);
  RestorePosAttrFont();
}

static void
ClearToEOL()
{
  register int y = curr->y, x = curr->x;

  if (display)
    Clear(x, y, width - 1, y);
  ClearInLine(y, x, cols - 1);
  RestorePosAttrFont();
}

static void
ClearFromBOL()
{
  register int y = curr->y, x = curr->x;

  if (display)
    Clear(0, y, x, y);
  ClearInLine(y, 0, x);
  RestorePosAttrFont();
}

static void
ClearInLine(y, x1, x2)
int y, x1, x2;
{
  register int n;

  if (x1 == cols)
    x1--;
  if (x2 == cols)
    x2--;
  if ((n = x2 - x1 + 1) != 0)
    {
      bclear(curr->w_image[y] + x1, n);
      bzero(curr->w_attr[y] + x1, n);
      bzero(curr->w_font[y] + x1, n);
    }
}

static void
CursorRight(n)
register int n;
{
  register int x = curr->x;

  if (x == cols)
    {
      LineFeed(2);
      x = 0;
    }
  if ((curr->x += n) >= cols)
    curr->x = cols - 1;
  GotoPos(curr->x, curr->y);
}

static void
CursorUp(n)
register int n;
{
  if (curr->y < curr->w_top)		/* if above scrolling rgn, */
    {
      if ((curr->y -= n) < 0)		/* ignore its limits      */
         curr->y = 0;
    }
  else
    if ((curr->y -= n) < curr->w_top)
      curr->y = curr->w_top;
  GotoPos(curr->x, curr->y);
}

static void
CursorDown(n)
register int n;
{
  if (curr->y > curr->w_bot)		/* if below scrolling rgn, */
    {
      if ((curr->y += n) > rows - 1)	/* ignore its limits      */
        curr->y = rows - 1;
    }
  else
    if ((curr->y += n) > curr->w_bot)
      curr->y = curr->w_bot;
  GotoPos(curr->x, curr->y);
}

static void
CursorLeft(n)
register int n;
{
  if ((curr->x -= n) < 0)
    curr->x = 0;
  GotoPos(curr->x, curr->y);
}

static void
ASetMode(on)
int on;
{
  register int i;

  for (i = 0; i < curr->NumArgs; ++i)
    {
      switch (curr->args[i])
	{
	case 4:
	  curr->w_insert = on;
	  InsertMode(on);
	  break;
	}
    }
}

static void
SelectRendition()
{
  register int i = 0, a = curr->Attr;

  do
    {
      switch (curr->args[i])
	{
	case 0:
	  a = 0;
	  break;
	case 1:
	  a |= A_BD;
	  break;
	case 2:
	  a |= A_DI;
	  break;
	case 3:
	  a |= A_SO;
	  break;
	case 4:
	  a |= A_US;
	  break;
	case 5:
	  a |= A_BL;
	  break;
	case 7:
	  a |= A_RV;
	  break;
	case 22:
	  a &= ~(A_BD | A_SO | A_DI);
	  break;
	case 23:
	  a &= ~A_SO;
	  break;
	case 24:
	  a &= ~A_US;
	  break;
	case 25:
	  a &= ~A_BL;
	  break;
	case 27:
	  a &= ~A_RV;
	  break;
	}
    } while (++i < curr->NumArgs);
  SetAttr(curr->Attr = a);
}

static void
FillWithEs()
{
  register int i;
  register char *p, *ep;

  curr->y = curr->x = 0;
  for (i = 0; i < rows; ++i)
    {
      bzero(curr->w_attr[i], cols);
      bzero(curr->w_font[i], cols);
      p = curr->w_image[i];
      ep = p + cols;
      while (p < ep)
	*p++ = 'E';
    }
  if (display)
    Redisplay(0);
}


static void
ChangeLine(os, oa, of, y, from, to)
int from, to, y;
char *os, *oa, *of;
{
  ASSERT(display);
  DisplayLine(os, oa, of, curr->w_image[y], curr->w_attr[y],
	      curr->w_font[y], y, from, to);
  RestorePosAttrFont();
}

void
CheckLP(n_ch)
char n_ch;
{
  register int y, x;
  register char n_at, n_fo, o_ch, o_at, o_fo;

  ASSERT(display);
  x = cols - 1;
  y = bot;
  o_ch = curr->w_image[y][x];
  o_at = curr->w_attr[y][x];
  o_fo = curr->w_font[y][x];

  n_at = curr->Attr;
  n_fo = curr->charsets[curr->Charset];

  lp_image = n_ch;
  lp_attr = n_at;
  lp_font = n_fo;
  lp_missing = 0;
  if (n_ch == o_ch && n_at == o_at && n_fo == o_fo)
    return;
  if (n_ch != ' ' || n_at || n_fo)
    lp_missing = 1;
  if (o_ch != ' ' || o_at || o_fo)
    {
      if (DC)
	PutStr(DC);
      else if (CDC)
	CPutStr(CDC, 1);
      else if (CE)
	PutStr(CE);
      else
	lp_missing = 1;
    }
}

static void
FindAKA()
{
  register char *cp, *line;
  register struct win *wp = curr;
  register int len = strlen(wp->cmd);
  int y;

  y = (wp->autoaka > 0 && wp->autoaka <= wp->w_height) ? wp->autoaka - 1 : wp->y;
  cols = wp->w_width;
 try_line:
  cp = line = wp->w_image[y];
  if (wp->autoaka > 0 &&  *wp->cmd != '\0')
    {
      for (;;)
	{
	  if (cp - line >= cols - len)
	    {
	      if (++y == wp->autoaka && y < rows)
		goto try_line;
	      return;
	    }
	  if (strncmp(cp, wp->cmd, len) == 0)
	    break;
	  cp++;
	}
      cp += len;
    }
  for (len = cols - (cp - line); len && *cp == ' '; len--, cp++)
    ;
  if (len)
    {
      if (wp->autoaka > 0 && (*cp == '!' || *cp == '%' || *cp == '^'))
	wp->autoaka = -1;
      else
	wp->autoaka = 0;
      line = wp->cmd + wp->akapos;
      while (len && *cp != ' ')
	{
	  if ((*line++ = *cp++) == '/')
	    line = wp->cmd + wp->akapos;
	  len--;
	}
      *line = '\0';
    }
  else
    wp->autoaka = 0;
}

void
MakeBlankLine(p, n)
register char *p;
register int n;
{
  while (n--)
    *p++ = ' ';
}

void
SetCurr(wp)
struct win *wp;
{
  curr = wp;
  if (curr == 0)
    return;
  cols = curr->w_width;
  rows = curr->w_height;
  display = curr->active ? curr->display : 0;
}

static void
RestorePosAttrFont()
{
  GotoPos(curr->x, curr->y);
  SetAttr(curr->Attr);
  SetFont(curr->charsets[curr->Charset]);
}

/* Send a terminal report as if it were typed. */ 
static void
Report(fmt, n1, n2)
char *fmt;
int n1, n2;
{
  register int len;
  char rbuf[40];

  sprintf(rbuf, fmt, n1, n2);
  len = strlen(rbuf);

  if ((unsigned)(curr->inlen + len) <= sizeof(curr->inbuf))
    {
      bcopy(rbuf, curr->inbuf + curr->inlen, len);
      curr->inlen += len;
    }
}

#ifdef COPY_PASTE
void
AddLineToHist(wp, pi, pa, pf)
struct win *wp;
char **pi, **pa, **pf;
{
  register char *q;

  if (wp->histheight == 0)
    return;
  q = *pi; *pi = wp->ihist[wp->histidx]; wp->ihist[wp->histidx] = q;
  q = *pa; *pa = wp->ahist[wp->histidx]; wp->ahist[wp->histidx] = q;
  q = *pf; *pf = wp->fhist[wp->histidx]; wp->fhist[wp->histidx] = q;
  if (++wp->histidx >= wp->histheight)
    wp->histidx = 0;
}
#endif
