
/*
ptl2com (c) 2010-always Jan Panteltje

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


/* Panteltje ptl2com, communication program with MCS52 BASIC 8052AH */

#include "ptl2com.h"

struct termios terminal;
struct termios port;
int port_fd;
int terminal_fd;
fd_set readfs;
struct timeval timeout;
time_t idletimer;
int wait_ack_flag;
int txflag;
int rxflag;
char txbuf[60];
char rxbuf[60];
int rx_lf_flag;
struct passwd *userinfo;
char *home_dir;
int debug_flag;
long baudrate;
char *device;
speed_t cbaud;

tcflag_t _res_iflg, _res_lflg;

#define savetty() ((void) tcgetattr(terminal_fd, &terminal), \
	_res_iflg = terminal.c_iflag, _res_lflg = terminal.c_lflag)

#define mytty() (terminal.c_lflag &= 0, \
	tcsetattr(terminal_fd, TCSADRAIN, &terminal))

#define resettty() (terminal.c_iflag = _res_iflg, terminal.c_lflag = _res_lflg,\
	(void) tcsetattr(terminal_fd, TCSADRAIN, &terminal))

#define cfsetispeed(X, Y) (X)->c_cflag &= ~CBAUD; (X)->c_cflag |= Y
#define cfsetospeed(X, Y) (X)->c_cflag &= ~CBAUD; (X)->c_cflag |= Y

static struct
	{
	char *proto;
	int  clear;
	int  set;
	} prot_tbl[] =
		{
		{"8N1",	~(CSIZE | PARENB | CSTOPB), CS8},
		{"7E2",	~(CSIZE | PARODD), CS7 | PARENB | CSTOPB},
		{"7O2",	~CSIZE,	CS7 | PARENB | PARODD | CSTOPB},
		{0, 0, 0}
		};

static struct
	{
	char *bps;
	unsigned rate;
	speed_t cbaud;
	} bps_tbl[] =
		{
		{"300",  300,  B300},
		{"600",  600,  B600},
		{"1200", 1200, B1200},
		{"2400", 2400, B2400},
		{"4800", 4800, B4800},
		{"9600", 9600, B9600},
		#ifdef B19200
			{"19200",19200,B19200},
		#endif
		#ifdef B38400
			{"38400",38400,B38400},
			{"57600",57600,B50},
		#endif
		{"0", 0, B0}
	};


char *strsave(char *s) /*save char array s somewhere*/
{
char *p;
//*malloc();

p = malloc(strlen(s) +  1);
if(p)  strcpy(p, s);
return p;
}


int readline(FILE *file, char *contents)
{
int a, c, i;

for(i = 0; i < READSIZE - 1; i++)
	{
	while(1)
		{
		c = getc(file);
		a = ferror(file);
		if(a)
			{
			perror("readline():");
			continue;
			}
		break;
		}
	if(feof(file) )
		{
		fclose(file);
		contents[i] = 0;/* EOF marker */
		return EOF;
		}
	if(c == '\n')
		{
		contents[i] = 0;/* string termination */
		return 1;/* end of line */
		} 
	contents[i] = c;
	}
/*
mmm since we are here, the line must be quite long, possibly something
is wrong.
Since I do not always check for a return 0 in the use uf this function,
just to be safe, gona force a string termination.
This prevents stack overflow, and variables getting overwritten.
*/
contents[i] = 0;/* force string termination */
fclose(file);
return 0;
}/* end function readline */


void set_rtscts(int state)
{
if(state) port.c_cflag |= CRTSCTS;
else port.c_cflag &= ~CRTSCTS;
   
if (port_fd != -1) tcsetattr(port_fd, TCSANOW, &port);
}


void set_flow(int flow)
{
if (flow) port.c_iflag |= IXON | IXOFF | IXANY;
else  port.c_iflag &= ~(IXON | IXOFF | IXANY);
if(port_fd != -1) tcsetattr(port_fd, TCSANOW, &port);
}


/* get/set character size and parity on the port */
char *set_proto(char *p)
{
int i;

if (!p) return 0;

for (i = 0; prot_tbl[i].proto; i++)
	{
	if(!strcmp(p,prot_tbl[i].proto))
		{
		port.c_cflag &= prot_tbl[i].clear;
		port.c_cflag |= prot_tbl[i].set;
		if(port_fd != -1) tcsetattr(port_fd, TCSANOW, &port);
		return strsave(p);
		}
	}
return 0;
}


/*	Get/set the bps of the modem port; set the terminal rate to correspond. */
unsigned mrate(s)
char *s;
{
int  i;

if (s)
	{
	for (i = 0; bps_tbl[i].cbaud; i++)
		{
		if (!strcmp(s, bps_tbl[i].bps))
			{
			printf("Baudrate found s=%s\n", s);
			cbaud = bps_tbl[i].cbaud;
			cfsetispeed(&port, cbaud);
			cfsetospeed(&port, cbaud);
			tcsetattr(port_fd, TCSANOW, &port);
			return (bps_tbl[i].rate);
			}
		}

	for (i=0; bps_tbl[i].cbaud; i++)
		{
		if(cfgetospeed(&port) == bps_tbl[i].cbaud)
			{
			return bps_tbl[i].rate;
			}
		}
	return 0;
	}
return 1;
}


int save_setting(char *type, char *string)
{
char pathfilename[512];
FILE *setting_file;

/* test if a helpersfile helpers is present */
/* combine with home directory */
sprintf(pathfilename, "%s/.ptl2com/setup/%s", home_dir, type);
setting_file = fopen(pathfilename, "w");
if(! setting_file)
	{
	fprintf(stdout, "Cannot open file %s for write", pathfilename);
	if(debug_flag)
		{
		fprintf(stdout, "\nCannot open setting file %s for write\n",\
		pathfilename);
		}
	return 0;
	}

fprintf(setting_file, "%s", string);
fclose(setting_file);
return 1;	
}/* end function save_setting */


char *load_setting(char *type)
{
int a;
char temp[1024];
char pathfilename[512];
FILE *setting_file;
extern char *strsave();

/* combine with home directory */
sprintf(pathfilename, "%s/.ptl2com/setup/%s", home_dir, type);
setting_file = fopen(pathfilename, "r");
if(! setting_file)
	{
	fprintf(stdout, "Cannot open file %s for read\n", pathfilename);
	if(debug_flag)
		{
		fprintf(stdout, "\nCannot open setting file %s for read\n",\
		pathfilename);
		}
	return 0;
	}

a = readline(setting_file, temp);/* closes file if EOF */	
if(a != EOF) fclose(setting_file);
return strsave(temp);
}/* end function load_settings */


void set_baudrate()
{
char temp[512];
unsigned new_baudrate;

resettty();
fprintf(stdout, "Set baudrate to?\n");
fscanf(stdin, "%100s", temp);
baudrate = atol(temp);
sprintf(temp, "%ld", baudrate);
save_setting("baudrate", temp); 
new_baudrate = mrate(temp);

fprintf(stdout, "baudrate set to %u.\n", new_baudrate); 
mytty();
}


void set_device()
{
char temp[512];
resettty();
fprintf(stdout, "Set device to?\n");
fscanf(stdin, "%100s", temp);
free(device);
device = strsave(temp);
save_setting("device", device); 

close(port_fd);
port_fd = open(device, O_RDWR | O_NDELAY);
if(port_fd < 0)
	{
	fprintf(stdout, "Cannot open device %s\n", device);
	}

tcgetattr(port_fd, &port);
set_rtscts(1);
set_proto("8N1");
set_flow(0);

txflag = 0;
wait_ack_flag = 0;

fprintf(stdout, "device set to %s.\n", device);
mytty();
}


int main(int argc, char *argv[])
{
int a, c;
char cc;
FILE *send_fd = 0; // -Wall
FILE *receive_fd = 0; // -Wall
char tx_filename[512];
char rx_filename[512];
char *ptr;

terminal_fd = fileno(stdin);

/* select con io 1 byte at the time */
setbuf(stdout,NULL);
setbuf(stdin,NULL);
 
fprintf(stdout,\
"\n                 Panteltje (c) ptl2com-%s\n\
A program for communication with MCS BASIC 52 based controllers.\n", PROG_VERSION);

/* get user info */
userinfo = getpwuid(getuid() );

/* get home directory */
home_dir = strsave(userinfo -> pw_dir);

/* defaults */
debug_flag = 0;
baudrate = DEFAULT_BAUDRATE;

device = strsave(DEFAULT_SERIAL_DEVICE_NAME);
if(! device)
	{
	fprintf(stdout, "ptl2com: Cannot allocate space for device, aborting.\n");

	exit(1);
	}

/* end defaults */

/* try to create directory for setup */
system("mkdir -p ~/.ptl2com/setup");

ptr = load_setting("baudrate");
if( ptr)
	{
	baudrate = atol(ptr);

	free(ptr);
	}

ptr = load_setting("device");
if(ptr)
	{
	free(device);
	device = strsave(ptr);
	if(! device)
		{
		fprintf(stdout, "ptl2com: Cannot allocate space for device, aborting.\n");

		exit(1);
		}

	free(ptr);
	}

fprintf(stdout, "^B set baudrate, current %ld, select from either 75, 150, 300, 600, 1200, 2400, 4800, 9600, 19200, 38400, 57600, or 115200.\n", baudrate);
fprintf(stdout, "^D download file.\n");
fprintf(stdout, "^P set serial device neame, current %s.\n", device);
fprintf(stdout, "^U upload file.\n");
fprintf(stdout, "^V enter debug mode.\n");
fprintf(stdout, "^X exit program.\n");


port_fd = open(device, O_RDWR | O_NDELAY);
if(port_fd < 0)
	{
	fprintf(stdout, "Cannot open device %s\n", device);
	}

tcgetattr(port_fd, &port);
set_rtscts(0);
set_proto("8N1");
set_flow(0);

savetty();
mytty();/* do not echo to con */

txflag = 0;
rxflag = 0;
wait_ack_flag = 0;

/* reset idle timer */
idletimer = time(0);
while(1)/* main communications loop */
	{
	/* this macro initializes the file descriptor readfs to to be the empty set */
	FD_ZERO(&readfs);

	/* this macro adds port_fd to the file descriptor readfs */
	if(port_fd >= 0)FD_SET(port_fd, &readfs);

	/* this macro adds fileno(stdin) to the file descriptor readfs */
	FD_SET(fileno(stdin), &readfs);

	if(txflag) if(send_fd > 0) FD_SET(fileno(send_fd), &readfs);
	 
	timeout.tv_sec = 10;
	timeout.tv_usec = 0;
	
	/* int FD_SETSIZE  macro is maximum number of filedescriptors that fd_set can hold */
 	/* function select waits for specified filedescr. to have a signal */
	/* last argument struct timeval *timeout */
	a = select(FD_SETSIZE, &readfs, NULL, NULL, NULL); /* &timeout); */
	if(a == -1)/* 0 is timeout, -1 error (in errno) */
		{
		perror("select(): ");
		break;
		}

	/* test if port has data */
	if( FD_ISSET(port_fd, &readfs) )
		{	
		/* read from port, write to console */
		a = read(port_fd, &c, 1);
		if(debug_flag)
			{
			fprintf(stdout, "READ FROM PORT a=%d c=%c (%d)\n", a, c, c);
			}
		else
			{
			fprintf(stdout, "%c", c);
			}

		if(rx_lf_flag)
			{
			if(c == '>')
				{
				wait_ack_flag = 0;
				}
			rx_lf_flag = 0;
			}
		if(c == 10) rx_lf_flag = 1;

		/* write to file if download enabled */
		if(rxflag)
			{
			while(1)
				{
				fputc(c, receive_fd);
				if(! ferror(receive_fd) ) break;/* maybe busy */
				}
			}
		}
	
	/* test if user has data */
	if( FD_ISSET(fileno(stdin), &readfs) )
		{
/*	 	fprintf(stdout, "USER HAS DATA\n");*/
		/* reset idletimer */
		idletimer = time(0);

		/* read a user key */	
		c = getc(stdin);

		if(c == EXIT_PROGRAM) /* ^X */
			{
			fprintf(stdout, "EXIT PROGRAM: are you sure? (y/n)\n");
			resettty();
			fscanf(stdin, "%c", &cc);
			a = toupper(cc);
			if(a == 'Y') break;
			mytty();
			}
		if(c == SET_BAUDRATE) /* ^B */
			{
			set_baudrate();
			}
		if(c == SET_DEVICE)/* ^P */
			{
			set_device();
			}
		if(c == UPLOAD_PROGRAM) /* ^U */
			{
			if(txflag)
				{
				txflag = 0;
				wait_ack_flag = 0;

				FD_CLR(fileno(send_fd), &readfs);

				fclose(send_fd);

				fprintf(stdout,\
				"\n\nUpload of file %s aborted by user.\n\n",\
				tx_filename);

				continue;
				}
			if(c == DEBUG_MODE) /* ^V */
				{
				debug_flag = 1;
				}

			resettty();
			fprintf(stdout, "tx filename?\n");
			fscanf(stdin, "%s", tx_filename);
			mytty();
			if(! strchr(tx_filename, '.') ) strcat(tx_filename, ".bas");
			send_fd = fopen(tx_filename, "r");
			if(! send_fd)
				{
				fprintf(stdout, "Cannot open file %s for read.\n", tx_filename);
				}
			else
				{
				fprintf(stdout, "Sending file %s ^U aborts.\n", tx_filename);
				FD_SET(fileno(send_fd), &readfs);
				txflag = 1;
				wait_ack_flag = 0;
				rx_lf_flag = 0;
				}
			}
		if(c == DOWNLOAD_PROGRAM) /* ^D */
			{
			if(rxflag)
				{
				rxflag = 0;
				FD_CLR(fileno(receive_fd), &readfs);

				fclose(receive_fd);

				fprintf(stdout,\
				"\n\nDownload file %s closed by user.\n\n",\
				rx_filename);
				continue;
				}

			rxflag = 0;
			resettty();
			
			fprintf(stdout, "rx filename?\n");
			fscanf(stdin, "%s", rx_filename);
			mytty();
			if(! strchr(rx_filename, '.') ) strcat(rx_filename, ".bas");
			receive_fd = fopen(rx_filename, "w");
			if(! receive_fd)
				{
				fprintf(stdout, "Cannot open file %s for write\n", rx_filename);
				}
			else
				{
				fprintf(stdout, "Downloading to file %s, type 'list' to save listing, ^D closes file.\n", rx_filename);
				FD_SET(fileno(receive_fd), &readfs);
				rxflag = 1;
				}
			}

		txbuf[0] = c;
		txbuf[1] = 0;

		write(port_fd, txbuf, 1);
		if(c == 10) 
			{
			txbuf[0]  = 13;
			write(port_fd, txbuf, 1);
			}
		c = 'a';
		}

	/* test if transmit file has data */
	if(txflag)
		{
		if(wait_ack_flag) continue;
		if(FD_ISSET(fileno(send_fd), &readfs) )
			{
			while(1)
				{
				c = getc(send_fd);
				if(! ferror(send_fd) ) break;
				}

			if(feof(send_fd) )
				{
				txflag = 0;
				wait_ack_flag = 0;
				FD_CLR(fileno(send_fd), &readfs);

				fclose(send_fd);
				} 
			else
				{
				txbuf[0] = c;
				txbuf[1] = 0;

				write(port_fd, txbuf, 1);
				if(c == 10)
					{
					txbuf[0]  = 13;
					write(port_fd, txbuf, 1);
					wait_ack_flag = 1;
					}
				}/* end else not EOF */
			}/* end file has data */
 		}/* end if tx_flag */
	}/* end main communications loop */

close(port_fd);
resettty();
fprintf(stdout, "\n");
exit(1);
}/* end main */


