/*
                                 REPSTR.C
                               
                   Program to replace string(s) in file(s)
            
                      Copyright (C) Laszlo Menczel 2006

                    This is free software with no warranty.
             Distributed under the General Public Licence version 2.
*/
    
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <malloc.h>
#include <ctype.h>
#include <errno.h>

#include "mutil.h"

/*****************************************************************************/

/*
   The constants RULE_STR_LEN and FILE_LINE_LEN limit the amount of
   text which can be processed as a single block. Their values come
   from the following considerations:

   1. In programming practice names are rarely if ever longer than
      64 characters.
 
   2. Single lines in typical source files almost never exceed 256
      characters.

   2. The maximum number of replacements in a single line is the
      maximum length of line / 2 (since the shortest string to replace
      is minimum 2 characters long) if recursive replacements are
      not allowed.

   3. The maximum length of the resulting line is the number of
      of possible replacements times the length of the longest
      replacement string allowed.

   With the values defined below, the maximum length of a transformed
   line is 8K which is long but not excessive (altough quite a few text
   editors would choke on a line this long).
*/

#define RULE_STR_LEN    64
#define FILE_LINE_LEN   256
#define TEXT_BUF_LEN    ((RULE_STR_LEN * FILE_LINE_LEN) / 2)

static char txtbuf[3][TEXT_BUF_LEN];
static int text_in = 0;
static int text_out = 1;

#define TEXT_IN  txtbuf[text_in]
#define TEXT_OUT txtbuf[text_out]
#define COMM_INF txtbuf[2]

static int match[TEXT_BUF_LEN / 2 + 1];

/*****************************************************************************/

typedef struct
{
  int oldlen;
  int newlen;
  char old[RULE_STR_LEN];
  char new[RULE_STR_LEN];
} rule_t;


/*****************************************************************************/

#define MAX_FNAME_LEN        (MUT_MAX_PATH_LEN + MUT_MAX_NAME_LEN)
#define MAX_GLOB_NUM         8
#define MAX_EXT_LEN          8
#define MAX_DELIM_ARG_LEN    64
#define MAX_COMM_DELIM_LEN   4

enum { OUTSIDE_COMM, INSIDE_COMM };

enum { COMM_NONE, COMM_EOL, COMM_BEG, COMM_END };

#define NO_MATCH  -1

/*****************************************************************************/

/* argument list for command line processing */

static int allow_empty = 0;
static int do_log = 0;
static int allow_recurse = 0;
static int batch_mode = 0;
static int skip_c_comment = 0;
static int enter_subdirs = 0;
static int ignore_case = 0;
static int match_word_space = 0;
static int process_hidden = 0;
static int supress_sorting = 0;

static char comm_delim_def[MUT_ARG_MAXLEN];
static char explicit_rule[MUT_ARG_MAXLEN];
static char ext_list[MUT_ARG_MAXLEN];
static char glob_pattern[MUT_ARG_MAXLEN];
static char input_file[MUT_ARG_MAXLEN];
static char backup_ext[MUT_ARG_MAXLEN];
static char rule_delim[MUT_ARG_MAXLEN];
static char rule_file[MUT_ARG_MAXLEN];
static char word_match_delim[MUT_ARG_MAXLEN];

static arglist_t arglist[] =
{
  { MUT_ARG_SWITCH, MUT_ARG_OPTIONAL, "a", (void *) &ignore_case }, 
  { MUT_ARG_SWITCH, MUT_ARG_OPTIONAL, "B", (void *) &batch_mode }, 
  { MUT_ARG_SWITCH, MUT_ARG_OPTIONAL, "C", (void *) &skip_c_comment }, 
  { MUT_ARG_SWITCH, MUT_ARG_OPTIONAL, "H", (void *) &process_hidden }, 
  { MUT_ARG_SWITCH, MUT_ARG_OPTIONAL, "l", (void *) &do_log }, 
  { MUT_ARG_SWITCH, MUT_ARG_OPTIONAL, "n", (void *) &supress_sorting }, 
  { MUT_ARG_SWITCH, MUT_ARG_OPTIONAL, "r", (void *) &allow_recurse }, 
  { MUT_ARG_SWITCH, MUT_ARG_OPTIONAL, "s", (void *) &enter_subdirs }, 
  { MUT_ARG_SWITCH, MUT_ARG_OPTIONAL, "W", (void *) &match_word_space }, 
  { MUT_ARG_SWITCH, MUT_ARG_OPTIONAL, "y", (void *) &allow_empty }, 

  { MUT_ARG_PAIRED, MUT_ARG_OPTIONAL, "c", (void *) comm_delim_def }, 
  { MUT_ARG_PAIRED, MUT_ARG_OPTIONAL, "d", (void *) rule_delim }, 
  { MUT_ARG_PAIRED, MUT_ARG_OPTIONAL, "e", (void *) ext_list }, 
  { MUT_ARG_PAIRED, MUT_ARG_OPTIONAL, "f", (void *) input_file }, 
  { MUT_ARG_PAIRED, MUT_ARG_OPTIONAL, "g", (void *) glob_pattern }, 
  { MUT_ARG_PAIRED, MUT_ARG_OPTIONAL, "b", (void *) backup_ext }, 
  { MUT_ARG_PAIRED, MUT_ARG_OPTIONAL, "t", (void *) rule_file }, 
  { MUT_ARG_PAIRED, MUT_ARG_OPTIONAL, "w", (void *) word_match_delim }, 
  { MUT_ARG_PAIRED, MUT_ARG_OPTIONAL, "x", (void *) explicit_rule }, 

  { MUT_ARG_END }
};

/*****************************************************************************/

/* string constants */

enum
{
  NO_PERR,
  PERR_CREATE,
  PERR_OPEN,

  PERR_ACCESS_R,
  PERR_BUSY_R,
  PERR_RONLY_R,

  PERR_ACCESS_D,
  PERR_BUSY_D,
  PERR_RONLY_D,

  PERR_UNDEF_R,
  PERR_UNDEF_D
};

static char *perr_str[] =
{
  "",
  "creation of temporary file failed when processing",
  "cannot open source file",

  "'access denied' error when renaming",
  "'in use' error when renaming",
  "'read-only' error when renaming",

  "'access denied' error when deleting",
  "'in use' error when deleting",
  "'read-only' error when deleting",

  "unknown error when renaming",
  "unknown error when deleting",
};

static char *err_1 = "Command line parsing failed";
static char *err_2 = "No source file(s) specified";
static char *err_3 = "More than one argument for source file specification";
static char *err_4 = "Cannot access the file";
static char *err_5 = "String replacement table is empty";
static char *err_6 = "Could not load string replacement table (memory allocation failed)";
static char *err_8 = "Memory allocation failed";
static char *err_9 = "Recursive rule(s) detected";
static char *err_10 = "No search pattern(s) for finding the source files";
// static char *err_11 = "";
static char *err_12 = "Too many replacements (matches) found";
static char *err_13 = "Rule file and explicit rule are both specified";
static char *err_14 = "Either a rule file or an explicit rule must be specified";
static char *err_15 = "Error when parsing the explicit rule";
static char *err_16 = "Switch '-s' should not be used if a single file is specified";
static char *err_17 = "No filename part in path";
static char *err_18 = "Extension string to append is too long";
static char *err_19 = "Transformed line is too long";
static char *err_20 = "Too many word delimiter characters specified";
static char *err_21 = "Line is too long";
static char *err_22 = "Failed when parsing rule string #";
static char *err_23 = "Oops: more rules than lines? Please, report the bug. :-("; 
static char *err_24 = "File or directory name too long";
static char *err_25 = "Failed when parsing comment delimiter definition";
static char *err_26 = "Comment delimiter conflict";
static char *err_27 = "Both '-C' and '-c <delim>' are specified";
static char *err_28 = "Access to directory denied";
static char *err_29 = "A system call (getcwd, chdir, etc.) failed";

static char *help_txt[] =
{
  "",
  "REPSTR (C) 2006 by Laszlo Menczel",
  "This is free software with no warranty.",
  "",
  "  repstr [options] [opt-args] source-arg rule-arg",
  "",
  "[options]",
  "-a   ignore case when replacing strings",
  "-B   batch mode (no screen output)",
  "-C   ignore text within C-style comments",
  "-h   you should already know this :-)",
  "-H   also process hidden files",
  "-l   write messages also to a logfile",
  "-n   do not sort replacement rules (string pairs)",
  "-r   allow recursive replacements",
  "-s   process subdirectories recursively",
  "-W   match only whole words delimited by whitespace(s)",
  "-y   allow empty strings in rules (string removal)",
  "",
  "[opt-args]",
  "-b <app>   make backup file(s) by appending 'app' to names",
  "-c <com>   use comment delimiter definitions in 'com'",
  "-d <sep>   strings in rules are separated by the 'sep' character",
  "-u <file>  check the rule set in 'file' and report the result",
  "-w <del>   match only whole words delimited by chars in 'del'",
  "",
  "'source-arg' (use only one) specify file(s) to process",
  "-e <ext>   list of extensions to use for file search",
  "-f <file>  path to single file to process",
  "-g <patt>  wildcard pattern(s) for filename globbing",
  "",
  "'rule-arg' (use only one) specify replacement rules",
  "-t <table> name of rule table containing string pairs",
  "-x <rule>  single explicit rule in the form <old,new>",
  "",
  NULL
};

static char *logname = "repstr.log";
static char *whitespace = " \t\n";

/*****************************************************************************/

/* global variables */

static rule_t single_rule;
static rule_t *single_rule_p = &single_rule;
static rule_t *rule = NULL;
static rule_t **rule_arr = NULL;

static finfo_t single_finfo;

static flist_t single_flist;
static flist_t *flist[MAX_GLOB_NUM];

static char comm_delim[3][MAX_COMM_DELIM_LEN + 1];
static char glob_string[MAX_GLOB_NUM][MUT_MAX_NAME_LEN + 1];

static int glob_num = 0;
static int rule_count = 0;
static int comm_delim_len[3] = { 0, 0, 0 };
static int ignore_comments = 0;
static int comment_stat = OUTSIDE_COMM;

/*****************************************************************************/

static void print_help(void)
{
  int i;

  i = 0;
  while (help_txt[i] != NULL)
  {
    printf("%s\n", help_txt[i]);
    i++;
  }
}

/***************************************************************************/

static void cleanup(void)
{
  int i;

  if (rule != NULL)
    free(rule);

  if (explicit_rule[0] == 0 && rule_arr != NULL)
    free(rule_arr);

  if (glob_pattern[0] != 0)
    for (i = 0; i < MAX_GLOB_NUM; i++)
      if (flist[i] != NULL)
        mut_glob_discard(&flist[i]);
}

/*****************************************************************************/

static int load_rules(char *name, char *delim, int count)
{
  FILE *f;
  char token[RULE_STR_LEN + 1], line[MUT_MAX_LINE_LEN + 1], *pos, *end;
  int i, ret;

  f = fopen(name, "rt");
  if (f == NULL)
  {
    mut_logprintf("Error: %s '%s'.", err_4, name);
    if (! batch_mode)
      printf("Error: %s '%s'.\n", err_4, name);
    return 0;
  }

  i = 0;

  while (1)
  {
    if (! mut_fget_line(line, MUT_MAX_LINE_LEN, f))
    {
      ret = mut_last_error();

      if (ret == MUTERR_PARTIAL_READ)
      {
        mut_logprintf("Error: %s (line %d in '%s').", err_21, i + 1, name);
        if (! batch_mode)
          printf("Error: %s (line %d in '%s').", err_21, i + 1, name);
        fclose(f);
        return 0;
      }
      else if (ret == MUTERR_EMPTY)
        continue;
      else if (ret == MUTERR_END_OF_FILE)
        break;
    }

    if (i == count)
    {
      mut_logprintf("%s", err_23);
        if (! batch_mode)
          printf("%s\n", err_23); 
      fclose(f);
      return 0;
    }

    end = mut_str_end(line);
    pos = mut_get_token_ex(token, line, end, delim, RULE_STR_LEN); 

    if (pos == NULL)
    {
      mut_logprintf("Error: %s %d (line %d).", err_22, 1, i + 1);
      if (! batch_mode)
        printf("Error: %s %d (line %d)\n.", err_22, 1, i + 1);
      fclose(f);
      return 0;
    }

    strcpy(rule[i].old, token);
    rule[i].oldlen = strlen(token);

    pos = mut_get_token_ex(token, pos, end, delim, RULE_STR_LEN); 
    if (pos == NULL)
    {
      if (! allow_empty)
      {
        mut_logprintf("Error: %s %d (line %d).", err_22, 2, i + 1);
        if (! batch_mode)
          printf("Error: %s %d (line %d)\n.", err_22, 2, i + 1);
        fclose(f);
        return 0;
      }
      else
      {
        rule[i].new[0] = 0;
        rule[i].newlen = 0;
      }
    }
    else
    {
      strcpy(rule[i].new, token);
      rule[i].newlen = strlen(token);
    }

    i++;
  }

  mut_logprintf("Parsed %d rules in '%s'.", i, name);
  if (! batch_mode)
    printf("Parsed %d rules in '%s'.\n", i, name);

  fclose(f);
  return 1;
}

/***************************************************************************/

static void sort_rules(rule_t **arr, int count)
{
  rule_t *tmp;
  int i, swapped;

  do
  {
    swapped = 0;
    for (i = 1; i < count; i++)
      if (arr[i]->oldlen > arr[i - 1]->oldlen)
      {
        tmp = arr[i];
        arr[i] = arr[i - 1];
        arr[i - 1] = tmp;
        swapped = 1;
      }
  }
  while (swapped);
}

/***************************************************************************/

static int is_recursive(rule_t **rep, int count)
{
  int target, probe, ret;

  ret = 0;

  for (probe = 0; probe < count; probe++)
    for (target = 0; target < count; target++)
    {
      if (probe == target || rep[probe]->oldlen > rep[target]->newlen)
        continue;

      if (strstr(rep[target]->new, rep[probe]->old))
      {
        mut_logprintf
        (
          "recursive: %s -> %s [%s -> %s]",
          rep[probe]->old, rep[probe]->new,  rep[target]->old, rep[target]->new
        );
        ret = 1;
      }
    }

  return ret;
}

/***************************************************************************/

static int parse_single_rule(void)
{
  char token[MUT_MAX_NAME_LEN + 1], *pos, *end;

  pos = explicit_rule;
  end = mut_str_end(explicit_rule);

  pos = mut_get_token_ex(token, pos, end, ",", MUT_MAX_NAME_LEN);
  if (pos == NULL)
    return 0;

  strcpy(single_rule.old, token);
  single_rule.oldlen = strlen(token);

  pos = mut_get_token_ex(token, pos, end, ",", MUT_MAX_NAME_LEN);
  if (pos == NULL)
  {
    if (! allow_empty)
      return 0;
    else
    {
      single_rule.new[0] = 0;
      single_rule.newlen = 0;
    }
  }
  else
  {  
    strcpy(single_rule.new, token);
    single_rule.newlen = strlen(token);
  }
  
  return 1;
}

/***************************************************************************/

static int load_and_check_rules(void)
{
  int i, ret;

  rule_count = mut_file_line_count(rule_file, "\n#");
  if (rule_count < 1)
  {
    mut_logprintf("Error: %s.", err_5);
    if (! batch_mode)
    printf("Error: %s.\n", err_5);
    return 0;
  }

  rule = (rule_t *) malloc(rule_count * sizeof(rule_t));
  if (rule == NULL)
  {
    mut_logprintf("Error: %s.", err_6);
    if (! batch_mode)
      printf("Error: %s.\n", err_6);
    return 0;
  }

  memset((void *) rule, 0, rule_count * sizeof(rule_t));

  if (rule_delim[0] == 0)
    strcpy(rule_delim, whitespace);

  ret = load_rules(rule_file, rule_delim, rule_count);
  if (ret == 0)
    return 0;

  rule_arr = (rule_t **) malloc(rule_count * sizeof(rule_t *));
  if (rule_arr == NULL)
  {
    mut_logprintf("Error: %s.", err_8);
    if (! batch_mode)
      printf("Error: %s.\n", err_8);
    return 0;
  }

  for (i = 0; i < rule_count; i++)
    rule_arr[i] = &(rule[i]);

  if (! supress_sorting)
    sort_rules(rule_arr, rule_count);

  if (is_recursive(rule_arr, rule_count))
  {
    mut_logprintf("%s.", err_9);
    if (! batch_mode)
      printf("%s.\n", err_9);

    if (! allow_recurse)
      return 0;
  }

  return 1;
}

/***************************************************************************/

static int parse_glob_data(char *buf)
{
  int i;
  char token[MUT_MAX_NAME_LEN + 1], *pos, *end;

  pos = buf;
  end = mut_str_end(buf);

  i = 0;
  while (1)
  {
    if (i == MAX_GLOB_NUM)
    {
      mut_logputs("Warning: Too many search patterns or extensions specified.");
      if (! batch_mode)
        printf("Warning: Too many search patterns or extensions specified.");
      break;
    }
     
    pos = mut_get_token_ex(token, pos, end, ",", MUT_MAX_NAME_LEN);
    if (pos == NULL)
      break;

    strcpy(glob_string[i], token);
    i++;
  }

  return i;
}

/***************************************************************************/

static int parse_comm_delim(void)
{
  int i;
  char token[3][MAX_COMM_DELIM_LEN + 1], *pos, *end;

  for (i = 0; i < 3; i++)
    token[i][0] = 0;

  pos = comm_delim_def;
  end = mut_str_end(comm_delim_def);

  pos = mut_get_token_ex(token[0], pos, end, ",", MAX_COMM_DELIM_LEN); 
  if (pos == NULL)
  {
    mut_logprintf("Error: %s.", err_25);
    if (! batch_mode)
      printf("Error: %s.\n", err_25);
    return 0;
  }

  pos = mut_get_token_ex(token[1], pos, end, ",", MAX_COMM_DELIM_LEN); 
  if (pos == NULL)  
  {
    strcpy(comm_delim[0], token[0]);
    comm_delim_len[0] = strlen(token[0]);

    mut_logprintf("Parsed single comment delimiter '%s'.", comm_delim[0]);
    if (! batch_mode)
      printf("Parsed single comment delimiter %s.\n", comm_delim[0]);
    return 1;
  }

  pos = mut_get_token_ex(token[2], pos, end, ",", MAX_COMM_DELIM_LEN);
  if (pos == NULL)  
  {
    strcpy(comm_delim[1], token[0]);
    strcpy(comm_delim[2], token[1]);
    comm_delim_len[1] = strlen(token[0]);
    comm_delim_len[2] = strlen(token[1]);

    mut_logprintf("Parsed comment delimiter pair '%s' '%s'.", comm_delim[1], comm_delim[2]);
    if (! batch_mode)
      printf("Parsed comment delimiter pair '%s' '%s'.\n", comm_delim[1], comm_delim[2]);
    return 1;
  }

  if (strcmp(token[0], token[1]) == 0 || strcmp(token[0], token[2]) == 0)
  {
    mut_logprintf("Error: %s (identical strings).", err_26);
    if (! batch_mode)
      printf("Error: %s (identical strings).\n", err_26);
    return 0;
  }

  for (i = 0; i < 3; i++)
  {
    strcpy(comm_delim[i], token[i]);
    comm_delim_len[i] = strlen(token[i]);
  }

  mut_logprintf
  (
    "Parsed three comment delimiters '%s' '%s' '%s'.",
    comm_delim[0], comm_delim[1], comm_delim[2]
  );

  if (! batch_mode)
    printf
    (
      "Parsed three comment delimiters '%s' '%s' '%s'.\n",
      comm_delim[0], comm_delim[1], comm_delim[2]
    );

  return 1;
}

/***************************************************************************/

static int is_comment_delimiter(char *s)
{
  int i, k;

  for (i = 0; i < 3; i++)
  {
    if (comm_delim_len[i] == 0)
      continue;

    for (k = 0; k < comm_delim_len[i]; k++)
      if (*(s + k) != comm_delim[i][k])
        break;

    if (k == comm_delim_len[i])
      return i + 1;
  }

  return 0;
}

/***************************************************************************/

static int mark_comment_regions(int *comm_stat)
{
  int i, k, end, ret;

  for (i = 0; i < TEXT_BUF_LEN; i++)
    COMM_INF[i] = 0;

  i = 0;
  end = strlen(TEXT_IN);

  while (i < end)
  {
    ret = is_comment_delimiter(&TEXT_IN[i]);

    if (ret == COMM_NONE)
    {
      if (*comm_stat == INSIDE_COMM)
        COMM_INF[i] = 1;
      else
        COMM_INF[i] = 0;
      i++;
    }
    else
    {
      if (ret == COMM_EOL)
      {
        if (*comm_stat == OUTSIDE_COMM)
        {
          for ( ; i < end; i++)
            COMM_INF[i] = 1;
          break;
        }
      }
      else if (ret == COMM_BEG)
        *comm_stat = INSIDE_COMM;
      else if (ret == COMM_END)
        *comm_stat = OUTSIDE_COMM;

      for (k = 0; k < comm_delim_len[ret - 1]; k++, i++)
        COMM_INF[i] = 1;
    }
  }

  return 1;
}

/***************************************************************************/

static int is_word_delimiter(char c)
{
  int i;

  i = 0;
  while (word_match_delim[i] != 0)
  {
    if (c == word_match_delim[i])
      return 1;
    i++;
  }

  return 0;
}

/***************************************************************************/

static int match_whole_word(char *s, int pos, int *match_pos)
{
  int i;

  if (! is_word_delimiter(TEXT_IN[pos]))
    return 0;

  while (is_word_delimiter(TEXT_IN[pos]))
  {
    pos++;
    if (ignore_comments && COMM_INF[pos] != 0)
      return 0;
  }

  *match_pos = pos;

  i = 0;
  while (s[i] != 0)
  {
    if (ignore_comments && COMM_INF[pos + i] != 0)
      break;
    else if (is_word_delimiter(TEXT_IN[pos + i]))
      break;
    else if (ignore_case && tolower(TEXT_IN[pos + i]) != tolower(s[i]))
      break;
    else if (! ignore_case && TEXT_IN[pos + i] != s[i])
      break;

    i++;
  }

  if (s[i] != 0 || ! is_word_delimiter(TEXT_IN[pos + i]))
    return 0;
  else
    return i;
}

/***************************************************************************/

/*
   Tries to match the string 's' in the text input buffer starting at position
   '*pos'. Return values:

   SUCCESS: returns the number of characters matched
   FAILURE: returns zero
*/

static int match_string(char *s, int pos, int *match_pos)
{
  int i;

  *match_pos = pos;

  i = 0;
  while (s[i] != 0)
  {
    if (ignore_comments && COMM_INF[pos + i] != 0)
      break;
    else if (ignore_case && tolower(TEXT_IN[pos + i]) != tolower(s[i]))
      break;
    else if (! ignore_case && TEXT_IN[pos + i] != s[i])
      break;

    i++;
  }

  if (s[i] != 0)
    return 0;
  else
    return i;
}

/***************************************************************************/

static int find_matches(char *s, int max_match, int match_word)
{
  int next, len, pos, match_pos, end, ret;

  for (pos = 0; pos < max_match; pos++)
    match[pos] = NO_MATCH;

  next = 0;
  len = strlen(s);
  end = strlen(TEXT_IN);
  pos = 0;

  while (pos < end)
  {
    if (len > end - pos + 1)
      break;

    if (ignore_comments && COMM_INF[pos] != 0)
    {
      pos++;
      continue;
    }

    if (match_word)
      ret = match_whole_word(s, pos, &match_pos);
    else
      ret = match_string(s, pos, &match_pos);

    if (ret)
    {
      if (next == max_match)
      {
        next = -1;
        break;
      }
      else
      {
        match[next] = match_pos;
        next++;
        pos += ret;
      }
    }
    else
      pos++;
  }

  return next;
}

/***************************************************************************/

static int replace_and_copy(char *new, int oldlen, int match_count)
{
  int k, in, out, newlen, next;
  
  newlen = strlen(new);
  next = in = out = 0;

  for (next = 0; next < match_count; next++)
  {
    while (in < match[next])
    {
      TEXT_OUT[out] = TEXT_IN[in];
      in++;
      out++;
      if (out == TEXT_BUF_LEN)
      {
        TEXT_OUT[out - 1] = 0;
        return 0;
      }
    }

    for (k = 0; k < newlen; k++)
    {
      TEXT_OUT[out] = new[k];
      out++;
      if (out == TEXT_BUF_LEN)
      {
        TEXT_OUT[out - 1] = 0;
        return 0;
      }
    }

    in += oldlen;
  }

  while (TEXT_IN[in] != 0)
  {
    TEXT_OUT[out] = TEXT_IN[in];
    in++;
    out++;
    if (out == TEXT_BUF_LEN)
    {
      TEXT_OUT[out - 1] = 0;
      return 0;
    }
  }

  TEXT_OUT[out] = 0;
  return out;
}
    
/***************************************************************************/

static void swap_txtbuf(void)
{
  int tmp;

  tmp = text_in;
  text_in = text_out;
  text_out = tmp;
}

/***************************************************************************/

static FILE *create_temp_file(char *tmp_name)
{
  sprintf(tmp_name, "repstr-tmp-%d", rand() & 0xffff);
  return fopen(tmp_name, "wt");
}

/***************************************************************************/

static int rename_file(char *old, char *new)
{
  if (rename(old, new) != 0)
  {
    if (errno == EACCES)
      return PERR_ACCESS_R;
    else if (errno == EBUSY)
      return PERR_BUSY_R;
    else if (errno == EROFS)
      return PERR_RONLY_R;
    else
      return PERR_UNDEF_R;
  }

  return 0;
}

/***************************************************************************/

static int delete_file(char *name)
{
  if (unlink(name) != 0)
  {
    if (errno == EACCES)
      return PERR_ACCESS_D;
    else if (errno == EBUSY)
      return PERR_BUSY_D;
    else if (errno == EROFS)
      return PERR_RONLY_D;
    else
      return PERR_UNDEF_D;
  }

  return 0;
}

/***************************************************************************/

static int process_file(char *path)
{
  FILE *f, *tmp;
  char tmp_name[64], bak_name[MAX_FNAME_LEN + MAX_EXT_LEN + 1];
  int i, ret, lnum, mnum, size, mode, buf_swapped;

printf("Processing '%s'\n", path);

  tmp = create_temp_file(tmp_name);
  if (tmp == NULL)
  {
    mut_logprintf("%s '%s'.", perr_str[PERR_CREATE], tmp_name);
    if (! batch_mode)
      printf("%s '%s'.\n", perr_str[PERR_CREATE], tmp_name);
    return 0;
  }

  f = fopen(path, "rt");
  if (f == NULL)
  {
    fclose(tmp);
    unlink(tmp_name);
    mut_logprintf("%s '%s'.", perr_str[PERR_OPEN], path);
    if (! batch_mode)
      printf("%s '%s'.\n", perr_str[PERR_OPEN], path);
    return 0;
  }

  if (match_word_space || word_match_delim[0] != 0)
    mode = 1;
  else
    mode = 0;

  text_in = 0;
  text_out = 1;
  lnum = mnum = 0;
  comment_stat = OUTSIDE_COMM;

  while (1)
  {
    if (! mut_fget_line(TEXT_IN, FILE_LINE_LEN, f))
    {
      ret = mut_last_error();

      if (ret == MUTERR_PARTIAL_READ)
      {
        mut_logprintf("Error: %s (line %d in '%s').", err_21, lnum + 1, path);
        if (! batch_mode)
          printf("Error: %s (line %d in '%s').", err_21, lnum + 1, path);
        fclose(f);
        fclose(tmp);
        unlink(tmp_name);
        return 0;
      }
      else if (ret == MUTERR_EMPTY)
      {
        fprintf(tmp, "\n");
        continue;
      }
      else if (ret == MUTERR_END_OF_FILE)
        break;
    }

    buf_swapped = 1;    // so that comments in the first round are also marked if needed

    for (i = 0; i < rule_count; i++)
    {
      if (ignore_comments && buf_swapped)
        mark_comment_regions(&comment_stat);

      ret = find_matches(rule_arr[i]->old, strlen(TEXT_IN) / 2, mode);
      if (ret == -1)
      {
        mut_logprintf("Error: %s (line %d in '%s').", err_12, lnum + 1, path);
        if (! batch_mode)
          printf("Error: %s (line %d in '%s').", err_12, lnum + 1, path);
        fclose(f);
        fclose(tmp);
        unlink(tmp_name);
        return 0;
      }
      else if (ret > 0)
      {
        mnum += ret;
        size = replace_and_copy(rule_arr[i]->new, rule_arr[i]->oldlen, ret);
        if (size == 0)
        {
          mut_logprintf("Error: %s (line %d in '%s').", err_19, lnum + 1, path);
          if (! batch_mode)
            printf("Error: %s (line %d in '%s').\n", err_19, lnum + 1, path);
          fclose(f);
          fclose(tmp);
          unlink(tmp_name);
          return 0;
        }
        swap_txtbuf();
        buf_swapped = 1;
      }
      else
       buf_swapped = 0;
    }

    fprintf(tmp, "%s\n", TEXT_IN);
    lnum++;
  }

  fclose(f);
  fclose(tmp);

  strcpy(bak_name, path);

  if (backup_ext[0] != 0)
    strcat(bak_name, backup_ext);
  else
    strcat(bak_name, "~~");
    
  ret = rename_file(path, bak_name);
  if (ret != 0)
  {
    mut_logprintf("%s '%s' to '%s'.", perr_str[ret], path, bak_name);
    if (! batch_mode)
      printf("%s '%s' to '%s'.", perr_str[ret], path, bak_name);
    return 0;
  }

  ret = rename_file(tmp_name, path);
  if (ret != 0)
  {
    rename(bak_name, path);
    mut_logprintf("%s '%s' to '%s'.", perr_str[ret], tmp_name, path);
    if (! batch_mode)
      printf("%s '%s' to '%s'.", perr_str[ret], tmp_name, path);
    return 0;
  }

  if (backup_ext[0] == 0)
  {
    ret = delete_file(bak_name);
    if (ret != 0)
    {
      mut_logprintf("%s '%s'.", perr_str[ret], bak_name);
      if (! batch_mode)
        printf("%s '%s'.", perr_str[ret], bak_name);
      return 0;
    }
  }
  
  mut_logprintf("Replaced %d strings in '%s' (%d lines).", mnum, path, lnum);
  if (! batch_mode)
    printf("Replaced %d strings in '%s' (%d lines).\n", mnum, path, lnum);

  return 1;
}

/***************************************************************************/

static int get_yes_no(char *msg)
{
  int c;

  printf("%s (y/n)...", msg);

  while (1)
  {
    c = getchar();
    if (c == 'y' || c == 'Y')
      return 1;
    else if (c == 'n' || c == 'N')
      return 0;
  }
}

/***************************************************************************/

int main(int argc, char *argv[])
{
  flist_t *curr;
  int i, k, ret;
  char path[MUT_MAX_PATH_LEN + MUT_MAX_NAME_LEN + 1], name[MUT_MAX_NAME_LEN + 1];
  char date[32], *err_str;

  /*=========================================*/
  /* check for single argument '-h' for help */

  if (argc == 2 && (strcmp(argv[1], "-h") == 0 || strcmp(argv[1], "--help") == 0))
  {
    print_help();
    return 0;
  }

  /*==================================*/
  /* set up logging & exit processing */

  mut_set_logname(logname);
  mut_setlog(1);

  atexit(cleanup);

  mut_get_date(date);
  mut_logprintf("REPSTR version 1.0 %s", date);

  /*==============================*/
  /* check rule file if requested */

  if (argc == 3 && (strcmp(argv[1], "-u") == 0 || strcmp(argv[1], "--check-rules") == 0))
  {
    unlink(logname);
    strcpy(rule_file, argv[2]);
    ret = load_and_check_rules();
    if (ret == 1)
    {
      mut_logprintf("The rule set in '%s' is not recursive.", rule_file);
      if (! batch_mode)
        printf("The rule set in '%s' is not recursive.\n", rule_file);
      return 0;
    }
    else
      return 1;
  }

  /*=================*/
  /* parse arguments */

  if (! mut_parse_cmd_line(argc, argv, arglist, NULL))
  {
    unlink(logname);
    mut_logprintf("%s:\n%s.", err_1, mut_errmsg(mut_last_error()));
    if (! batch_mode)
      printf("%s:\n%s.\n", err_1, mut_errmsg(mut_last_error()));
    return 1;
  }

  /*======================================================*/
  /* check '-l' switch and disable logging if not present */

  if (! do_log)
    mut_setlog(0);
  else
    unlink(logname);
  
  /*=====================*/
  /* check the arguments */

  ret = 0;
  if (glob_pattern[0] != 0)
    ret++;
  if (input_file[0] != 0)
    ret++;
  if (ext_list[0] != 0)
    ret++;

  if (ret == 0)
  {
    mut_logprintf("Error: %s.", err_2);
    if (! batch_mode)
      printf("Error: %s.\n", err_2);
    return 1;
  }
  else if (ret > 1)
  {
    mut_logprintf("Error: %s.", err_3);
    if (! batch_mode)
      printf("Error: %s.\n", err_3);
    return 1;
  }

  if (input_file[0] != 0)
  {
    if (enter_subdirs)
    {
      mut_logprintf("Error: %s.", err_16);
      if (! batch_mode)
        printf("Error: %s.\n", err_16);
      return 1;
    }
  }

  if (rule_file[0] != 0 && explicit_rule[0] != 0)
  {
    mut_logprintf("Error: %s.", err_13);
    if (! batch_mode)
      printf("Error: %s.\n", err_13);
    return 1;
  }

  if (rule_file[0] == 0 && explicit_rule[0] == 0)
  {
    mut_logprintf("Error: %s.", err_14);
    if (! batch_mode)
      printf("Error: %s.\n", err_14);
    return 1;
  }

  if (backup_ext[0] != 0 && strlen(backup_ext) > MAX_EXT_LEN)
  {
    mut_logprintf("Error: %s.", err_18);
    if (! batch_mode)
      printf("Error: %s.\n", err_18);
    return 1;
  }

  if (word_match_delim[0] != 0)
  {
    if (strlen(word_match_delim) > MAX_DELIM_ARG_LEN)
    {
      mut_logprintf("Error: %s.", err_20);
      if (! batch_mode)
        printf("Error: %s.\n", err_20);
      return 1;
    }

    if (match_word_space)
      strcat(word_match_delim, whitespace);
  }
  else
  {
    if (match_word_space)
      strcpy(word_match_delim, whitespace);
  }

  if (word_match_delim[0] != 0)
  {
    for (i = 0; i < MUT_MAX_NAME_LEN; i++)
      name[i] = 0;

    i = 0;
    while (word_match_delim[i] != 0)
    {
      if (word_match_delim[i] == ' ')
        strcat(name, " SPACE");
      else if (word_match_delim[i] == '\t')
        strcat(name, " TAB");
      else if (word_match_delim[i] == '\n')
        strcat(name, " NEWLINE");
      else
        name[strlen(name)] = word_match_delim[i];

      i++;
    }
    
    mut_logprintf("Word delimiter(s): %s", name);
    if (! batch_mode)
      printf("Word delimiter(s): %s\n", name);
  }

  /*=========================*/
  /* get replacement rule(s) */

  if (explicit_rule[0] != 0)
  {
    if (! parse_single_rule())
    {
      mut_logprintf("%s '%s'.", err_15, explicit_rule);
      if (! batch_mode)
        printf("%s '%s'.\n", err_15, explicit_rule);
      return 1;
    }
      
    rule_arr = &single_rule_p;
    rule_count = 1;
  }
  else
  {
    if (! load_and_check_rules())
    {
      mut_logprintf("Program aborted.");
      if (! batch_mode)
        printf("Aborted.\n");
      return 1;
    }
  }

  /*===============================*/
  /* get comment delimiters if any */

  for (i = 0; i < 3; i++)
    comm_delim[i][0] = 0;

  if (comm_delim_def[0] != 0)
  {
    if (skip_c_comment)
    {
      mut_logprintf("%s.", err_27);
      mut_logprintf("Program aborted.");
      if (! batch_mode)
        printf("%s.\nAborted.\n", err_27);
      return 1;
    }

    if (! parse_comm_delim())
    {
      mut_logprintf("Program aborted.");
      if (! batch_mode)
        printf("Aborted.\n");
      return 1;
    }
    else
      ignore_comments = 1;
  }
  else
  {
    if (skip_c_comment)
    {
      strcpy(comm_delim[0], "//");
      strcpy(comm_delim[1], "/*");
      strcpy(comm_delim[2], "*/");
      comm_delim_len[0] = comm_delim_len[1] = comm_delim_len[2] = 2;
      ignore_comments = 1;
    }
  }

  /*======================================*/
  /* prepare the list of files to process */

  if (input_file[0] != 0)                   /* single file argument */
  {
    if (! mut_file_exist(input_file))
    {
      mut_logprintf("Error: %s '%s'.", err_4, input_file);
      if (! batch_mode)
        printf("Error: %s '%s'.\n", err_4, input_file);
      return 1;
    }
    
    mut_fname_split(input_file, path, name);

    if (name[0] == 0)
    {
      mut_logprintf("Error: %s '%s'.", err_17, input_file);
      if (! batch_mode)
        printf("Error: %s '%s'.\n", err_17, input_file);
      return 1;
    }

    strcpy(single_finfo.name, name);
    single_finfo.len = strlen(name);
    single_finfo.flag = 0;

    if (path[0] == 0)
    {
      single_flist.dir[0] = 0;
      single_flist.dirlen = 0;
    }
    else
    {
      strcpy(single_flist.dir, path);
      single_flist.dirlen = strlen(path);
    }

    single_flist.flag = 0;
    single_flist.count = 1;
    single_flist.filenum = 1;
    single_flist.files = &single_finfo;
    single_flist.next = NULL;

    flist[0] = &single_flist;
  }

  else
  {
    for (i = 0; i < MAX_GLOB_NUM; i++)
      flist[i] = NULL;

    if (ext_list[0] != 0)                   /* extension list argument */
    {
      glob_num = parse_glob_data(ext_list);
      if (glob_num == 0)
      {
        mut_logprintf("Error: %s.", err_10);
        if (! batch_mode)
          printf("Error: %s.\n", err_10);
        return 1;
      }

      for (i = 0; i < glob_num; i++)
      {
        sprintf(name, "*.%s", glob_string[i]);
        strcpy(glob_string[i], name);
      }
    }
    else                                   /* file search pattern argument */
    {
      glob_num = parse_glob_data(glob_pattern);
      if (glob_num == 0)
      {
        mut_logprintf("Error: %s.", err_10);
        if (! batch_mode)
          printf("Error: %s.\n", err_10);
        return 1;
      }
    }

    i = 0;
    while (i < glob_num)
    {
      if (enter_subdirs)
        flist[i] = mut_glob_dir(NULL, glob_string[i], MUT_FLIST_RECURS);
      else
        flist[i] = mut_glob_dir(NULL, glob_string[i], MUT_FLIST_SIMPLE);

      if (flist[i] == NULL)
      {
        ret = mut_last_error();

        if (ret == MUTERR_ALLOC)
          err_str = err_8;
        else if (ret == MUTERR_OVERFLOW)
          err_str = err_24;
        else if (ret == MUTERR_ACCESS_DIR)
          err_str = err_28;
        else if (ret == MUTERR_SYSCALL)
          err_str = err_29;

        mut_logprintf("%s.", err_str);
        if (! batch_mode)
          printf("%s.\n", err_str);
        return 1;
      }
      i++;
    }
  }

  /*================================*/
  /* process the files in the lists */      

  for (i = 0; i < MAX_GLOB_NUM && flist[i] != NULL; i++)
  {
    curr = flist[i];
    while (curr != NULL)
    {
      if (curr->filenum == 0)
      {
        curr = curr->next;
        continue;
      }

      for (k = 0; k < curr->count; k++)
      {
        if (curr->files[k].flag & MUT_FINFO_IS_DIR)
          continue;

        if (! process_hidden && (curr->files[k].flag & MUT_FINFO_HIDDEN))
          continue;

        strcpy(path, curr->dir);
        strcat(path, curr->files[k].name);

        if (! process_file(path))
        {
          ret = -1;   /* assume unconditional abort */

          if (! batch_mode && input_file[0] == 0)
            ret = get_yes_no("Proceed to next file?");

          if (ret == -1)
          {
            mut_logprintf("Aborted.");
            if (! batch_mode)
              printf("Aborted.\n");
            return 1;
          }
          else if (ret == 0)
          {
            mut_logprintf("Aborted by user.");
            printf("Aborted.\n");
            return 1;
          }
          else
          {
            mut_logprintf("Continued by user.");
            printf("continuing...\n");
          }
        }
      }                     /* end FOR (k) */

      curr = curr->next;
    }                       /* end WHILE   */
  }                         /* end FOR (i) */

  if (! batch_mode)
    printf("OK\n");

  return 0;
}
