/* vim:tw=78:ts=8:sw=4:set ft=c:  */
/*
    Copyright (C) 2006-2010 Ben Kibbey <bjk@luxsci.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1301  USA
*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <err.h>
#include <errno.h>
#include <string.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <ctype.h>
#include <glib.h>
#include <gcrypt.h>
#include <libxml/xmlwriter.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "pwmd_error.h"
#include "misc.h"
#include "xml.h"

/*
 * 'element' must be allocated.
 */
gboolean is_literal_element(gchar **element)
{
    gchar *p;

    if (!element || !*element)
	return FALSE;

    if (*(*element) == '!') {
	gchar *c;

	for (p = *element, c = p+1; *c; c++)
	    *p++ = *c;

	*p = 0;
	return TRUE;
    }

    return FALSE;
}

gboolean valid_xml_element(xmlChar *element)
{
    if (!element)
	return FALSE;

    if (strchr((gchar *)element, '\t') || strchr((gchar *)element, ' ') ||
	    *element == '!')
	return FALSE;

    return TRUE;
}

gpg_error_t new_root_element(xmlDocPtr doc, gchar *name)
{
    xmlNodePtr root = xmlDocGetRootElement(doc);
    xmlAttrPtr a;
    xmlNodePtr n;
    gchar *p = name;

    if (!p || !root)
	return EPWMD_LIBXML_ERROR;

    if (*p == '!')
	p++;

    if (!valid_xml_element((xmlChar *)p))
	return EPWMD_INVALID_ELEMENT;

    n = xmlNewNode(NULL, (xmlChar *)"element");
    n = xmlAddChild(root, n);
    a = xmlNewProp(n, (xmlChar *)"_name", (xmlChar *)p);
    return 0;
}

xmlDocPtr create_dtd()
{
    xmlDocPtr doc;
    xmlTextWriterPtr wr = xmlNewTextWriterDoc(&doc, 0);
    
    if (!wr)
	return NULL;

    if (xmlTextWriterStartDocument(wr, NULL, NULL, NULL))
	goto fail;

    if (xmlTextWriterStartDTD(wr, (xmlChar *)"pwmd", NULL, NULL) == -1)
	goto fail;

    if (xmlTextWriterWriteDTDElement(wr, (xmlChar *)"pwmd",
		(xmlChar *)"(element)") == -1)
	goto fail;

    xmlTextWriterEndDTDElement(wr);

    if (xmlTextWriterWriteDTDAttlist(wr, (xmlChar *)"element",
	    (xmlChar *)"_name CDATA #REQUIRED") == -1)
	goto fail;

    xmlTextWriterEndDTDAttlist(wr);
    xmlTextWriterEndDTD(wr);

    if (xmlTextWriterStartElement(wr, (xmlChar *)"pwmd"))
	goto fail;

    xmlTextWriterEndElement(wr);
    xmlTextWriterEndDocument(wr);
    xmlFreeTextWriter(wr);
    return doc;

fail:
    xmlTextWriterEndDocument(wr);
    xmlFreeTextWriter(wr);
    xmlFreeDoc(doc);
    return NULL;
}

xmlChar *new_document()
{
    xmlChar *xml;
    gint len;
    xmlDocPtr doc = create_dtd();

    if (!doc)
	return NULL;

    xmlDocDumpMemory(doc, &xml, &len);
    xmlFreeDoc(doc);
    return xml;
}

/*
 * Lists root element names; the value of the attribute "_name" of an element
 * "element". If there's a target attribute both literal and non-literal
 * element names will be added. This is the primary reason why XML entities
 * cannot be used. There wouldn't be a way to get the literal an non-literal
 * element paths.
 */
gpg_error_t list_root_elements(xmlDocPtr doc, GString **result)
{
    xmlNodePtr n = NULL;
    GSList *list = NULL;
    gint total, i;
    GString *string;
    gpg_error_t rc = 0;

    n = xmlDocGetRootElement(doc);

    if (!n || !n->children)
	return EPWMD_EMPTY_ELEMENT;

    for (n = n->children; n; n = n->next) {
	xmlAttrPtr a;
	xmlChar *val, *target;
	GSList *tlist;
	gchar *tmp;

	if (n->type != XML_ELEMENT_NODE)
	    continue;

	a = xmlHasProp(n, (xmlChar *)"_name");

	if (!a || !a->children->content)
	    continue;

	val = xmlNodeGetContent(a->children);

	if (!val) {
	    rc = gpg_error_from_errno(ENOMEM);
	    goto fail;
	}

	tmp = g_strdup_printf("!%s", (gchar *)val);

	if (!tmp) {
	    xmlFree(val);
	    rc = gpg_error_from_errno(ENOMEM);
	    goto fail;
	}

	tlist = g_slist_append(list, tmp);

	if (!tlist) {
	    xmlFree(val);
	    rc = gpg_error_from_errno(ENOMEM);
	    goto fail;
	}

	list = tlist;
	target = node_has_attribute(n, (xmlChar *)"target");

	if (target) {
	    gchar *t = g_strdup((gchar *)val);

	    if (!t) {
		xmlFree(val);
		xmlFree(target);
		rc = gpg_error_from_errno(ENOMEM);
		goto fail;
	    }

	    tlist = g_slist_append(list, t);

	    if (!tlist) {
		g_free(t);
		xmlFree(target);
		rc = gpg_error_from_errno(ENOMEM);
		goto fail;
	    }
	    
	    list = tlist;
	}

	xmlFree(val);
	xmlFree(target);
    }

    total = g_slist_length(list);

    if (!total)
	return EPWMD_EMPTY_ELEMENT;

    string = g_string_new(NULL);

    if (!string) {
	rc = gpg_error_from_errno(ENOMEM);
	goto fail;
    }

    for (i = 0; i < total; i++) {
	gchar *val = g_slist_nth_data(list, i);

	g_string_append_printf(string, "%s\n", val);
    }

    string = g_string_truncate(string, string->len - 1);
    *result = string;

fail:
    total = g_slist_length(list);

    for (i = 0; i < total; i++)
	g_free(g_slist_nth_data(list, i));

    g_slist_free(list);
    return rc;
}

/*
 * Prevents a sibling element past the current element path with the same
 * element name.
 */
static xmlNodePtr find_stop_node(xmlNodePtr node)
{
    xmlNodePtr n;

    for (n = node->parent->children; n; n = n->next) {
	if (n == node)
	    return n->next;
    }

    return NULL;
}

/*
 * Alot like create_elements_cb() but doesn't use the last element of 'req' as
 * content but as an element.
 */
xmlNodePtr create_target_elements_cb(xmlNodePtr node, gchar **path,
	gpg_error_t *rc, void *data)
{
    gint i;
    char **req = path;

    for (i = 0; req[i]; i++) {
	xmlNodePtr n;

	if ((n = find_element(node, req[i], find_stop_node(node))) == NULL ||
		(n && n->parent == node->parent)) {
	    is_literal_element(&req[i]);

	    if (!valid_xml_element((xmlChar *)req[i])) {
		*rc = EPWMD_INVALID_ELEMENT;
		return NULL;
	    }

	    //n = xmlNewNode(NULL, (xmlChar *)req[i]);
	    n = xmlNewNode(NULL, (xmlChar *)"element");

	    if (!n) {
		*rc = gpg_error_from_errno(ENOMEM);
		return NULL;
	    }

	    add_attribute(n, "_name", req[i]);
	    node = xmlAddChild(node, n);

	    if (!node) {
		*rc = gpg_error_from_errno(ENOMEM);
		return NULL;
	    }
	}
	else
	    node = n;
    }

    return node;
}

xmlNodePtr find_text_node(xmlNodePtr node)
{
    xmlNodePtr n = node;

    if (n && n->type == XML_TEXT_NODE)
	return n;

    for (n = node; n; n = n->next) {
	if (n->type == XML_TEXT_NODE)
	    return n;
    }

    return NULL;
}

xmlNodePtr create_elements_cb(xmlNodePtr node, gchar **elements,
	gpg_error_t *rc, void *data)
{
    gint i;
    gchar **req = elements;

    if (node->type == XML_TEXT_NODE)
	node = node->parent;

    xmlChar *a = node_has_attribute(node, (xmlChar *)"_name");

    if (a)
	xmlFree(a);

    for (i = 0; req[i]; i++) {
	xmlNodePtr n;

	if (req[i+1]) {
	    /*
	     * Strip the first '!' if needed. If there's another, it's an
	     * rc. The syntax has already been checked before calling this
	     * function.
	     */
	    is_literal_element(&req[i]);
	}

	/*
	 * The value of the element tree.
	 */
	if (!req[i+1]) {
	    n = find_text_node(node->children);

	    if (!n)
		/* Use AddContent here to prevent overwriting any children. */
		xmlNodeAddContent(node, (xmlChar *)req[i]);
	    else if (n && !*req[i])
		xmlNodeSetContent(n, NULL);
	    else
		xmlNodeSetContent(n, (xmlChar *)req[i]);

	    break;
	}

	n = find_element(node, req[i], find_stop_node(node));

	/*
	 * If the found element has the same parent as the current element,
	 * they are siblings and the new element needs to be created as a
	 * child of the current element (node).
	 */
	if (n && n->parent == node->parent)
	    n = NULL;

	if (!n) {
	    if (!valid_xml_element((xmlChar *)req[i])) {
		*rc = EPWMD_INVALID_ELEMENT;
		return NULL;
	    }

	    //n = xmlNewNode(NULL, (xmlChar *)req[i]);
	    n = xmlNewNode(NULL, (xmlChar *)"element");

	    if (!n) {
		*rc = gpg_error_from_errno(ENOMEM);
		return NULL;
	    }

	    *rc = add_attribute(n, "_name", req[i]);

	    if (*rc)
		return NULL;

	    node = xmlAddChild(node, n);

	    if (!node) {
		*rc = gpg_error_from_errno(ENOMEM);
		return NULL;
	    }
	}
	else
	    node = n;
    }

    return node;
}

/* The root element is really req[0]. It is need as a pointer in case there is
 * a target attribute so it can be updated. */
xmlNodePtr find_root_element(xmlDocPtr doc, gchar ***req, gpg_error_t *rc,
	gboolean *target, gint recursion_depth, gboolean stop)
{
    xmlNodePtr n = xmlDocGetRootElement(doc);
    gint depth = 0;
    gchar *root = g_strdup(*req[0]);
    gboolean literal = is_literal_element(&root);

    if (!root) {
	*rc = gpg_error_from_errno(ENOMEM);
	return NULL;
    }

    *rc = 0;
    recursion_depth++;

    if (max_recursion_depth >= 1 && recursion_depth > max_recursion_depth) {
	xmlChar *t = xmlGetNodePath(n);

	log_write("%s: %s", pwmd_strerror(EPWMD_LOOP), t);
	xmlFree(t);
	g_free(root);
	*rc = EPWMD_LOOP;
	return NULL;
    }

    while (n) {
	if (n->type == XML_ELEMENT_NODE) {
	    if (depth == 0 && xmlStrEqual(n->name, (xmlChar *)"pwmd")) {
		n = n->children;
		depth++;
		continue;
	    }

	    if (depth == 1 && xmlStrEqual(n->name, (xmlChar *)"element")) {
		xmlChar *content = node_has_attribute(n, (xmlChar *)"_name");

		if (!content)
		    continue;

		if (xmlStrEqual(content, (xmlChar *)root)) {
		    gchar **nreq, **tmp = NULL;

		    if (literal == TRUE) {
			xmlFree(content);
			g_free(root);
			return n;
		    }

		    xmlFree(content);
		    content = node_has_attribute(n, (xmlChar *)"target");

		    if (target)
			*target = TRUE;

		    if (!content || stop) {
			if (content)
			    xmlFree(content);

			g_free(root);
			return n;
		    }

		    if (strchr((gchar *)content, '\t')) {
			nreq = split_input_line((gchar *)content, "\t", 0);
			xmlFree(content);

#if 0
			/*
			 * FIXME ENOMEM
			 */
			if (!nreq) {
			    *rc = gpg_error_from_errno(ENOMEM);
			    return NULL;
			}
#endif

			tmp = *req;
			tmp = strvcatv(nreq, tmp+1);
			g_strfreev(nreq);

			if (!tmp) {
			    g_free(root);
			    *rc = gpg_error_from_errno(ENOMEM);
			    return NULL;
			}

			g_strfreev(*req);
			*req = tmp;
		    }
		    else {
			if (strv_printf(&tmp, "%s", content) == FALSE) {
			    xmlFree(content);
			    g_free(root);
			    *rc = gpg_error_from_errno(ENOMEM);
			    return NULL;
			}

			xmlFree(content);
			nreq = *req;
			nreq = strvcatv(tmp, nreq+1);
			g_strfreev(tmp);

			if (!nreq) {
			    *rc = gpg_error_from_errno(ENOMEM);
			    g_free(root);
			    return NULL;
			}

			g_strfreev(*req);
			*req = nreq;
		    }

		    g_free(root);
		    n = find_root_element(doc, req, rc, target, recursion_depth, FALSE);
		    return n;
		}

		xmlFree(content);
	    }
	}

	n = n->next;
    }

    g_free(root);
    *rc = EPWMD_ELEMENT_NOT_FOUND;
    return NULL;
}

xmlNodePtr find_element(xmlNodePtr node, gchar *element, xmlNodePtr stop)
{
    xmlNodePtr n;

    if (!node || !element)
	return NULL;

    for (n = node; n; n = n->next) {
	if (n->type != XML_ELEMENT_NODE)
	    continue;

	if (n == stop)
	    break;

	xmlChar *a = node_has_attribute(n, (xmlChar *)"_name");

	if (a && xmlStrEqual(a, (xmlChar *)element)) {
	    xmlFree(a);
	    return n;
	}

	xmlFree(a);
    }

    return NULL;
}

xmlChar *node_has_attribute(xmlNodePtr n, xmlChar *attr)
{
    xmlAttrPtr a = xmlHasProp(n, attr);

    if (!a)
	return NULL;

    if (!a->children || !a->children->content)
	return NULL;

    return xmlGetProp(n, attr);
}

static gboolean element_to_literal(gchar **element)
{
    gchar *p = g_strdup_printf("!%s", *element);

    if (!p)
	return FALSE;

    g_free(*element);
    *element = p;
    return TRUE;
}

/* Resolves elements in 'req' one at a time. It's recursive in case of
 * "target" attributes. */
xmlNodePtr find_elements(xmlDocPtr doc, xmlNodePtr node,
	gchar **req, gpg_error_t *rc, gboolean *target,
	xmlNodePtr (*found_fn)(xmlNodePtr, gchar **, gpg_error_t *, gchar **, void *),
	xmlNodePtr (*not_found_fn)(xmlNodePtr, gchar **, gpg_error_t *, void *),
	gboolean is_list_command, gint recursion_depth, void *data, gboolean stop)
{
    xmlNodePtr n, last, last_node;
    gchar **p;
    gint found = 0;

    *rc = 0;
    recursion_depth++;

    if (max_recursion_depth >= 1 && recursion_depth > max_recursion_depth) {
	xmlChar *t = xmlGetNodePath(node);

	log_write("%s: %s", pwmd_strerror(EPWMD_LOOP), t);
	xmlFree(t);
	recursion_depth--;
	*rc = EPWMD_LOOP;
	return NULL;
    }

    for (last_node = last = n = node, p = req; *p; p++) {
	xmlNodePtr tmp;
	gchar *t = g_strdup(*p);
	gboolean literal;

	if (!t) {
	    *rc = gpg_error_from_errno(ENOMEM);
	    return NULL;
	}

	literal = is_literal_element(&t);
	n = find_element(last, t, NULL);
	g_free(t);

	if (!n) {
	    if (not_found_fn)
		return not_found_fn(found ? last_node : last_node->parent, p, rc, data);

	    *rc = EPWMD_ELEMENT_NOT_FOUND;
	    return NULL;
	}

	last = n->children;
	last_node = n;
	found = 1;

	if (literal == FALSE) {
	    xmlChar *content = node_has_attribute(n, (xmlChar *)"target");
	    gchar **nreq = NULL, **nnreq;

	    if (!content) {
		if (is_list_command == TRUE) {
		    if (element_to_literal(&(*p)) == FALSE) {
			*rc = gpg_error_from_errno(ENOMEM);
			return NULL;
		    }
		}

		continue;
	    }

	    if (target)
		*target = TRUE;

	    if (!*(p+1) && stop) {
		xmlFree(content);
		return n;
	    }

	    if (strchr((gchar *)content, '\t') != NULL) {
		if ((nreq = split_input_line((gchar *)content, "\t", 0)) == NULL) {
		    xmlFree(content);
		    *rc = EPWMD_INVALID_ELEMENT;
		    return NULL;
		}
	    }
	    else {
		if ((nreq = split_input_line((gchar *)content, " ", 0)) == NULL) {
		    xmlFree(content);
		    *rc = EPWMD_INVALID_ELEMENT;
		    return NULL;
		}
	    }

	    xmlFree(content);
	    tmp = find_root_element(doc, &nreq, rc, target, 0, FALSE);

	    if (!tmp) {
		g_strfreev(nreq);
		return NULL;
	    }

	    if (found_fn) {
		found_fn(tmp, nreq, rc, p+1, data);

		if (*rc) {
		    g_strfreev(nreq);
		    return NULL;
		}
	    }

	    nnreq = strvcatv(nreq+1, p+1);
	    g_strfreev(nreq);

	    // FIXME ENOMEM
	    if (!nnreq || !*nnreq) {
		if (nnreq)
		    g_strfreev(nnreq);

		return tmp;
	    }

	    n = find_elements(doc, tmp->children, nnreq, rc, NULL, found_fn,
		    not_found_fn, is_list_command, recursion_depth, data, stop);

	    if (*(p+1)) {
		gchar **zz = p+1, **qq = nnreq;

		if (g_strv_length(nnreq) > g_strv_length(p+1))
		    qq = nnreq+1;

		for (; *qq && *zz; zz++) {
		    g_free(*zz);
		    *zz = g_strdup(*qq++);

		    if (!*zz) {
			*rc = gpg_error_from_errno(ENOMEM);
			n = NULL;
			break;
		    }
		}
	    }

	    g_strfreev(nnreq);
	    return n;
	}
    }

    return n;
}

static gboolean update_element_list(struct element_list_s *elements)
{
    gchar *line;
    GSList *l;

    if (!elements || !elements->elements)
	return TRUE;

    line = g_strjoinv("\t", elements->elements);

    if (!line)
	return FALSE;

    g_strfreev(elements->elements);
    elements->elements = NULL;
    l = g_slist_append(elements->list, line);

    if (!l)
	return FALSE;

    elements->list = l;
    return TRUE;
}

static gpg_error_t path_list_recurse(xmlDocPtr doc, xmlNodePtr node,
	struct element_list_s *elements)
{
    gpg_error_t rc = 0;
    xmlNodePtr n;

    for (n = node; n; n = n->next) {
	xmlChar *target = NULL;
	xmlChar *a = node_has_attribute(n, (xmlChar *)"_name");

	if (!a)
	    continue;

	if (n->type != XML_ELEMENT_NODE)
	    goto children;

	if (strv_printf(&elements->elements, "%s\t!%s", elements->prefix, a) == FALSE) {
	    xmlFree(a);
	    return gpg_err_code_from_errno(ENOMEM);
	}

	if (update_element_list(elements) == FALSE) {
	    xmlFree(a);
	    return gpg_err_code_from_errno(ENOMEM);
	}

	target = node_has_attribute(n, (xmlChar *)"target");

	if (target) {
	    gchar *tmp;
	    gchar *save = elements->prefix;
	    gboolean r = elements->resolving;

	    elements->depth++;

	    if (max_recursion_depth >= 1 && elements->depth > max_recursion_depth) {
		xmlChar *t = xmlGetNodePath(n);
		log_write("%s: %s", pwmd_strerror(EPWMD_LOOP), t);
		xmlFree(t);
		xmlFree(target);
		xmlFree(a);
		return EPWMD_LOOP;
	    }

	    if (strv_printf(&elements->elements, "%s\t%s", elements->prefix, a) == FALSE) {
		xmlFree(a);
		xmlFree(target);
		return gpg_err_code_from_errno(ENOMEM);
	    }

	    tmp = g_strjoinv("\t", elements->elements);

	    if (!tmp) {
		xmlFree(a);
		xmlFree(target);
		return gpg_err_code_from_errno(ENOMEM);
	    }

	    if (update_element_list(elements) == FALSE) {
		xmlFree(a);
		xmlFree(target);
		return gpg_err_code_from_errno(ENOMEM);
	    }

	    elements->prefix = tmp;
	    elements->resolving = TRUE;
	    rc = create_path_list(doc, elements, (gchar *)target);
	    xmlFree(target);
	    elements->resolving = r;
	    elements->depth--;
	    g_free(tmp);
	    elements->prefix = save;

	    if (rc) {
		xmlFree(a);
		return rc;
	    }
	}

children:
	if (n->children) {
	    gchar *tmp = g_strdup_printf("%s\t!%s", elements->prefix, a);
	    gchar *save = elements->prefix;

	    if (!tmp) {
		xmlFree(a);
		return gpg_err_code_from_errno(ENOMEM);
	    }

	    elements->prefix = tmp;
	    rc = path_list_recurse(doc, n->children, elements);
	    g_free(elements->prefix);
	    elements->prefix = save;

	    if (rc) {
		xmlFree(a);
		return rc;
	    }
	}

	xmlFree(a);
    }

    return rc;
}

gpg_error_t add_attribute(xmlNodePtr node, const gchar *name,
	const gchar *value)
{
    xmlAttrPtr a;

    if ((a = xmlHasProp(node, (xmlChar *)name)) == NULL) {
	a = xmlNewProp(node, (xmlChar *)name, (xmlChar *)value);

	if (!a)
	    return EPWMD_LIBXML_ERROR;
    }
    else
	xmlNodeSetContent(a->children, (xmlChar *)value);

    return 0;
}

gpg_error_t update_timestamp(xmlDocPtr doc)
{
    xmlNodePtr n = xmlDocGetRootElement(doc);
    gchar *t = g_strdup_printf("%li", time(NULL));
    gpg_error_t rc;

    rc = add_attribute(n, "age", t);
    g_free(t);
    return rc;
}

/*
 * From the element path 'path', find sub-nodes and append them to the list.
 */
gpg_error_t create_path_list(xmlDocPtr doc, struct element_list_s *elements,
	gchar *path)
{
    gpg_error_t rc;
    gchar **req, **req_orig;
    xmlNodePtr n;
    gboolean a_target = FALSE;

    req = split_input_line(path, "\t", 0);

    if (!req) {
	req = split_input_line(path, " ", 0);

	if (!req)
	    return EPWMD_COMMAND_SYNTAX;
    }

    req_orig = g_strdupv(req);

    if (!req_orig) {
	rc = gpg_err_code_from_errno(ENOMEM);
	goto fail;
    }

    n = find_root_element(doc, &req, &rc, &a_target, 0, FALSE);

    if (!n && rc == EPWMD_ELEMENT_NOT_FOUND && elements->resolving == TRUE) {
	rc = 0;
	goto fail;
    }
    else if (!n)
	goto fail;

    if (a_target == TRUE) {
	g_free(*req);
	*req = g_strdup(*req_orig);
    }

    if (*(req+1)) {
	gboolean e_target = FALSE;

	n = find_elements(doc, n->children, req+1, &rc, &e_target, NULL, NULL, TRUE, 0, NULL, FALSE);

	if (!n && rc == EPWMD_ELEMENT_NOT_FOUND && elements->resolving == TRUE) {
	    rc = 0;
	    goto fail;
	}
	else if (!n)
	    goto fail;
    }

    if (!elements->prefix) {
	/*
	 * FIXME
	 *
	 * If any req_orig element contains no target the element should be
	 * prefixed with the literal character. Not really crucial if the
	 * client isn't human because child elements are prefixed for the
	 * current path. But may be confusing if editing by hand.
	 */
	elements->prefix = g_strjoinv("\t", req_orig);

	if (!elements->prefix) {
	    rc = gpg_err_code_from_errno(ENOMEM);
	    goto fail;
	}

	if (strv_printf(&elements->elements, "%s", elements->prefix) == FALSE) {
	    rc = gpg_err_code_from_errno(ENOMEM);
	    goto fail;
	}

	if (update_element_list(elements) == FALSE) {
	    rc = gpg_err_code_from_errno(ENOMEM);
	    goto fail;
	}
    }

    rc = path_list_recurse(doc, n->children, elements);

fail:
    if (req_orig)
	g_strfreev(req_orig);

    g_strfreev(req);
    return rc;
}

gpg_error_t recurse_xpath_nodeset(xmlDocPtr doc, xmlNodeSetPtr nodes,
	xmlChar *value, xmlBufferPtr *result, gboolean cmd, const xmlChar *attr)
{
    gint i = value ? nodes->nodeNr - 1 : 0;
    xmlBufferPtr buf;

    buf = xmlBufferCreate();

    if (!buf)
	return gpg_err_code_from_errno(ENOMEM);

    for (; value ? i >= 0 : i < nodes->nodeNr; value ? i-- : i++) {
	xmlNodePtr n = nodes->nodeTab[i];

	if (!n)
	    continue;

	if (!value && !attr) {
	    if (xmlNodeDump(buf, doc, n, 0, 0) == -1) {
		*result = buf;
		return EPWMD_LIBXML_ERROR;
	    }

	    continue;  
	}

	if (!attr)
	    xmlNodeSetContent(n, value);
	else {
	    gpg_error_t rc;
	    
	    if (!cmd)
		rc = add_attribute(n, (gchar *)attr, (gchar *)value);
	    else
		rc = delete_attribute(n, attr);

	    if (rc)
		return rc;
	}
    }

    *result = buf;
    return 0;
}

static gpg_error_t convert_root_element(xmlNodePtr n)
{
    xmlChar *a = xmlGetProp(n, (xmlChar *)"_name");
    gpg_error_t rc;

    if (a) {
	xmlFree(a);
	xmlChar *t = xmlGetNodePath(n);

	log_write(N_("An existing \"_name\" attribute already exists. Please rename this attribute before converting. Path is: %s"), t);
	xmlFree(t);
	return GPG_ERR_AMBIGUOUS_NAME;
    }

    a = xmlGetProp(n, (xmlChar *)"name");

    if (a) {
	rc = add_attribute(n, "_name", (gchar *)a);
	xmlFree(a);

	if (rc)
	    return rc;

	rc = delete_attribute(n, (xmlChar *)"name");

	if (rc)
	    return rc;

	xmlNodeSetName(n, (xmlChar *)"element");
    }

    return 0;
}

/* Updates the DTD and renames the root "accounts" and "account" elements. */
gpg_error_t convert_xml(gchar **xml, goffset *len)
{
    gpg_error_t rc = EPWMD_LIBXML_ERROR;
    xmlDocPtr doc, new = NULL;
    xmlNodePtr n;

    doc = xmlReadMemory(*xml, *len, NULL, "UTF-8", XML_PARSE_NOBLANKS);

    if (!doc)
	return EPWMD_LIBXML_ERROR;

    gcry_free(*xml);
    *xml = NULL;
    n = xmlDocGetRootElement(doc);
    xmlNodeSetName(n, (xmlChar *)"pwmd");

    for (n = n->children; n; n = n->next) {
	if (xmlStrEqual(n->name, (xmlChar *)"account")) {
	    rc = convert_root_element(n);

	    if (rc) {
		xmlFreeDoc(doc);
		return rc;
	    }
	}
    }

    new = create_dtd();

    if (!new)
	goto fail;

    n = xmlDocGetRootElement(doc);
    xmlDocSetRootElement(new, n);
    xmlDocDumpMemory(new, (xmlChar **)xml, (gint *)len);
    xmlDocSetRootElement(new, xmlCopyNode(n, 0));
    rc = 0;

fail:
    if (new)
	xmlFreeDoc(new);

    xmlFreeDoc(doc);
    return rc;
}

gpg_error_t delete_attribute(xmlNodePtr n, const xmlChar *name)
{
    xmlAttrPtr a;

    if ((a = xmlHasProp(n, name)) == NULL)
	return EPWMD_ATTR_NOT_FOUND;

    if (xmlRemoveProp(a) == -1)
	return EPWMD_LIBXML_ERROR;

    return 0;
}

static gpg_error_t convert_elements_recurse(xmlDocPtr doc, xmlNodePtr n,
	guint depth)
{
    gpg_error_t rc;

    depth++;

    for (n = n->children; n; n = n->next) {
	if (n->type == XML_ELEMENT_NODE) {
	    xmlChar *a = NULL;

	    if (depth > 1) {
		if (xmlStrEqual(n->name, (xmlChar *)"element")) {
		    xmlChar *t = xmlGetNodePath(n);

		    log_write(N_("An existing \"element\" already exists. Please rename this element before converting. Path is: %s"), t);
		    xmlFree(t);
		    return GPG_ERR_AMBIGUOUS_NAME;
		}

		a = xmlGetProp(n, (xmlChar *)"_name");

		if (a) {
		    xmlFree(a);
		    xmlChar *t = xmlGetNodePath(n);

		    log_write(N_("An existing \"_name\" attribute already exists. Please rename this attribute before converting. Path is: %s"), t);
		    xmlFree(t);
		    return GPG_ERR_AMBIGUOUS_NAME;
		}

		xmlChar *tmp = xmlStrdup(n->name);

		if (!tmp)
		    return gpg_error_from_errno(ENOMEM);

		xmlNodeSetName(n, (xmlChar *)"element");
		rc = add_attribute(n, "_name", (gchar *)tmp);
		xmlFree(tmp);

		if (rc)
		    return rc;
	    }
	    else {
		rc = convert_root_element(n);

		if (rc)
		    return rc;
	    }
	}

	if (n->children) {
	    rc = convert_elements_recurse(doc, n, depth);

	    if (rc)
		return rc;
	}
    }

    return 0;
}

/* Renames ALL elements to the new "element" name. Existing element names are
 * stored as an attribute "_name". This was introduced in pwmd 2.12 so
 * elements can contain common characters that the XML parser barfs on (an
 * email address for example. */
gpg_error_t convert_elements(xmlDocPtr doc)
{
    xmlNodePtr n = xmlDocGetRootElement(doc);
    gpg_error_t rc;

    log_write(N_("Converting pre 2.12 data file..."));
    rc = convert_elements_recurse(doc, n, 0);

    if (!rc)
	log_write(N_("Finished converting. Please SAVE to update."));

    return rc;
}

gpg_error_t validate_import(xmlNodePtr node)
{
    gpg_error_t rc;

    if (!node)
	return 0;

    for (xmlNodePtr n = node; n; n = n->next) {
	if (n->type == XML_ELEMENT_NODE) {
	    if (xmlStrEqual(n->name, (xmlChar *)"element")) {
		xmlChar *a = xmlGetProp(n, (xmlChar *)"_name");

		if (!a) {
		    xmlChar *t = xmlGetNodePath(n);

		    log_write(N_("Missing attribute '_name' at %s."), t);
		    xmlFree(t);
		    return EPWMD_INVALID_ELEMENT;
		}

		if (!valid_xml_element(a)) {
		    xmlChar *t = xmlGetNodePath(n);

		    log_write(N_("'%s' is not a valid element name at %s."), a, t);
		    xmlFree(a);
		    xmlFree(t);
		    return EPWMD_INVALID_ELEMENT;
		}

		xmlFree(a);
	    }
	    else {
		xmlChar *t = xmlGetNodePath(n);

		log_write(N_("Warning: unknown element '%s' at %s. Ignoring."), n->name, t);
		xmlFree(t);
		continue;
	    }
	}

	if (n->children) {
	    rc = validate_import(n->children);

	    if (rc)
		return rc;
	}
    }

    return rc;
}
