/* vim:tw=78:ts=8:sw=4:set ft=c:  */
/*
    Copyright (C) 2006-2010 Ben Kibbey <bjk@luxsci.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1301  USA
*/
#ifndef COMMON_H
#define COMMON_H

#include <sys/types.h>
#include <sys/stat.h>
#include <pth.h>
#include <gcrypt.h>

#define _ASSUAN_ONLY_GPG_ERRORS	1
#include <assuan.h>

#ifdef ENABLE_NLS
#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif
#endif

#include "gettext.h"
#define N_(msgid)	gettext(msgid)
#include "status.h"

enum {
    STATE_CONNECTED,
    STATE_OPEN
};

typedef enum {
    INQUIRE_INIT,
    INQUIRE_BUSY,
    INQUIRE_DONE
} inquire_status_t;

typedef enum {
    PINENTRY_OPEN,
    PINENTRY_SAVE
} pinentry_cmd_t;

#ifdef WITH_PINENTRY
typedef struct {
    size_t len;
    void *buf;
} membuf_t;

typedef enum {
    PINENTRY_NONE,
    PINENTRY_INIT,
    PINENTRY_PID,
    PINENTRY_RUNNING,
    PINENTRY_TIMEOUT
} pinentry_status_t;

typedef struct {
    gint fd;
    gpg_error_t error;
    pinentry_status_t status;
    union {
	gchar key[ASSUAN_LINELENGTH];
	pid_t pid;
    } what;
} pinentry_key_s;

typedef gpg_error_t (*pinentry_finalize_cb)(assuan_context_t, guchar *, gboolean);

struct pinentry_s {
    pth_t tid;
    pth_mutex_t status_mutex;
    pth_mutex_t cond_mutex;
    pth_cond_t cond;
    pinentry_cmd_t which;
    gchar *filename;
    assuan_context_t ctx;
    pinentry_finalize_cb cb;
    pid_t pid;
    pid_t pin_pid;
    gint fd;
    pinentry_status_t status;
    gchar *name;
    gchar *title;
    gchar *desc;
    gchar *prompt;
    gchar *ttyname;
    gchar *ttytype;
    gchar *display;
    gchar *path;
    gchar *lcctype;
    gchar *lcmessages;
    gint timeout;
    gboolean has_lock;
    gint enable;
    membuf_t data;
    assuan_error_t (*inquire_cb)(void *data, const char *line);
    void *inquire_data;
};
#endif

typedef struct {
    guint8 magic[5];
    guint16 version;
    guint64 iter;
    guint64 flags;
    guint8 iv[16];
} file_header_t;

typedef struct {
    gint iter;
    guint8 iv[16];
} file_header_v1_t;

typedef struct {
    gboolean v1;
    gint fd;
    struct stat st;
    gpointer doc;
    goffset len;

    union {
	file_header_v1_t fh1;
	file_header_t fh2;
    } ver;
} file_header_internal_t;

struct client_thread_s {
    pth_t tid;
    pth_msgport_t mp;
    pth_t msg_tid;
    pth_mutex_t mp_mutex;
    gint fd;
    struct client_s *cl;
};

struct assuan_cmd_s {
    assuan_context_t ctx;
    const gchar *line;
    gint line_len;
    const gchar *line2;
};

/* For use with .flags in the data file header. */
#define PWMD_CIPHER_OFFSET	(1)
#define PWMD_CIPHER(n)		(PWMD_CIPHER_OFFSET << n)
#define PWMD_CIPHER_AES128	PWMD_CIPHER(0)
#define PWMD_CIPHER_AES192	PWMD_CIPHER(1)
#define PWMD_CIPHER_AES256	PWMD_CIPHER(2)
#define PWMD_CIPHER_SERPENT128	PWMD_CIPHER(3)
#define PWMD_CIPHER_SERPENT192	PWMD_CIPHER(4)
#define PWMD_CIPHER_SERPENT256	PWMD_CIPHER(5)
#define PWMD_CIPHER_CAMELLIA128	PWMD_CIPHER(6)
#define PWMD_CIPHER_CAMELLIA192	PWMD_CIPHER(7)
#define PWMD_CIPHER_CAMELLIA256	PWMD_CIPHER(8)
#define PWMD_CIPHER_3DES	PWMD_CIPHER(9)
#define PWMD_CIPHER_CAST5	PWMD_CIPHER(10)
#define PWMD_CIPHER_BLOWFISH	PWMD_CIPHER(11)
#define PWMD_CIPHER_TWOFISH	PWMD_CIPHER(12)
#define PWMD_CIPHER_TWOFISH128	PWMD_CIPHER(13)

#define PWMD_FLAG_OFFSET	(1<<15)
#define PWMD_FLAG(n)		(PWMD_FLAG_OFFSET << n)

struct client_crypto_s {
    gpointer iv;
    gpointer key;
    gpointer tkey;
    gpointer tkey2;
    gpointer inbuf;
    goffset insize;
    gpointer outbuf;
    goffset outsize;
    file_header_internal_t *fh;
    gcry_cipher_hd_t gh;
    gsize blocksize;
    gsize keysize;
};

/* These are flags that are set by a client via the OPTION command. */
#define OPT_ITERATIONS	0x1
#define OPT_PINENTRY	0x2
#define OPT_PINENTRY_TO	0x4

struct client_s {
    assuan_context_t ctx;
#ifdef WITH_PINENTRY
    struct pinentry_s *pinentry;
#endif
    gpointer doc; /* xmlDocPtr */
    gpointer xml_error;
    gpointer xml;
    gint len;
    gint state;
    gchar *filename;
    guchar md5file[16];
    gboolean new;
    gboolean freed;
    time_t mtime;
    gboolean has_lock;
    gboolean is_lock_cmd;
    inquire_status_t inquire_status;
    struct client_thread_s *thd;
    struct client_crypto_s *crypto;
    guchar opts;
    gpg_error_t last_rc;
    gboolean lockonopen;
    gboolean rc_on_locked;
};

GKeyFile *keyfileh;
gboolean log_syslog;
gint zlib_bufsize;
pth_mutex_t rcfile_mutex;
pth_mutex_t cn_mutex;
GSList *cn_thread_list;

#define log_write0 log_write

#define log_write1(...) { \
    if (get_key_file_integer("global", "log_level") >= 1) \
	log_write(__VA_ARGS__); \
    }

#define log_write2(...) { \
    if (get_key_file_integer("global", "log_level") >= 2) \
	log_write(__VA_ARGS__); \
    }

void log_write(const gchar *fmt, ...);
gpg_error_t send_error(assuan_context_t ctx, gpg_error_t pwmd_errno);
gpg_error_t send_syserror(assuan_context_t ctx, int e);
gint open_file(const gchar *filename, struct stat *st);
gpg_error_t do_xml_encrypt(struct client_s *client, struct client_crypto_s *,
	const gchar *filename);
gint get_key_file_integer(const gchar *section, const gchar *what);
gdouble get_key_file_double(const gchar *section, const gchar *what);
gboolean get_key_file_boolean(const gchar *section, const gchar *what);
gchar *get_key_file_string(const gchar *section, const gchar *what);
gchar *expand_homedir(gchar *str);
void free_client(struct client_s *client);
void cleanup_crypto(struct client_crypto_s **);
struct client_crypto_s *init_client_crypto();
gpg_error_t init_client_crypto2(const char *filename,
	struct client_crypto_s *crypto);
gpg_error_t do_assuan_command(assuan_context_t ctx,
	void *(*cb)(void *data), void *data);
void close_file_header(file_header_internal_t *fh);
void cleanup_ev_cb(void *arg);
void cleanup_mutex_cb(void *arg);
void cleanup_fd_cb(void *arg);
void cleanup_unlink_cb(void *arg);
void cleanup_attr_cb(void *arg);
void cleanup_cancel_cb(void *arg);
guint pwmd_cipher_str_to_cipher(const gchar *str);

#endif
