/*********************************************************************************/
/*                                                                               */
/*  makeform (Part of the FenForm Document Printer project)                      */
/*  Vn 1.0                                                                       */
/*                                                                               */
/*  Project Leader - Mike Eggleston                                              */
/*                                                                               */
/*  Copyright 2005 - 2008 Fenland Software Ltd                                   */
/*  Date: 28/08/2008                                                             */
/*                                                                               */
/*  Form Creation Program - makeform.c                                           */
/*                                                                               */
/*********************************************************************************/
/*                                                                               */
/*  Licence: GPL Vn2                                                             */
/*                                                                               */
/*  This program is free software; you can redistribute it and/or modify         */
/*  it under the terms of the GNU General Public License as published by         */
/*  the Free Software Foundation; either version 2 of the License, or            */
/*  (at your option) any later version.                                          */
/*                                                                               */
/*  This program is distributed in the hope that it will be useful,              */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of               */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                */
/*  GNU General Public License for more details.                                 */
/*                                                                               */
/*  You should have received a copy of the GNU General Public License            */
/*  along with this program; if not, write to the Free Software                  */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA    */
/*                                                                               */
/*                                                                               */
/*                                                                               */
/*********************************************************************************/


#include "fenform.h"
#include "ierrors.h"
#include "iapi.h"
#include <getopt.h>
#include <ctype.h>

#define BUFFER_LEN	1024
#define THIS_PROGNAME	"makeform"
#define THIS_VERSION	"(Vn 1.0)"


/************************/
/* Function Prototypes  */ 
/************************/

void MAKEFORM_Usage( void );
int  MAKEFORM_BuildHeader( FF_Handle*, FILE* );
int  MAKEFORM_GetMediaParameters( FF_Handle*, char* );
int  MAKEFORM_GetOrientation( FF_Handle*, char* );
int  MAKEFORM_Build_EPS_File( FF_Handle*, char*, char* );
int  MAKEFORM_Copy_EPS_to_Form( FF_Handle*, FILE*, FILE* );
char *MAKEFORM_BuildFileName( FF_Handle*, char* );




/************************/
/* Functions            */ 
/************************/


void MAKEFORM_Usage( )
{
    fprintf( stderr, "\n%s %s\n%s\n%s\n\n", THIS_PROGNAME, THIS_VERSION, COPYRIGHT, LICENCE );
    fprintf( stderr, "Commercial support for this software is available from\n" );
    fprintf( stderr, "%s    %s\n\n", COMPANY, WEB_SITE );
    fprintf( stderr, "Usage:  " );
    fprintf( stderr, "makeform [-c CONFIG_FILE] [-f FORM_NAME] INPUT_FILE_NAME\n\n" );
    fprintf( stderr, "For 'Special Forms' the FORM_NAME *must* be given.\n" );
    fprintf( stderr, "        makeform [-c CONFIG_FILE] -f FORM_NAME *lMEDIA_TYPE\n" );
    fprintf( stderr, "        makeform [-c CONFIG_FILE] -f FORM_NAME *pMEDIA_TYPE\n\n\n" );
}


int MAKEFORM_BuildHeader( FF_Handle *Hndl, FILE *form_file_ptr )
{

    /***********************************************************/
    /* Header Layout                                           */
    /*                                                         */
    /* 9 Bytes Const %FENFORM%                                 */
    /* 2 Bytes - Header Size (Bytes)                           */
    /* 1 Byte - Major version                                  */
    /* 2 Bytes - Minor version                                 */
    /* NULL terminated string - User Name                      */
    /* NULL terminated string - Form Name                      */
    /* NULL terminated string - Creation Date                  */
    /* NULL terminated string - Creation Program               */
    /* NULL terminated string - Copyright Notice               */
    /* 2 Bytes - Form Length                                   */
    /* 2 Bytes - Form Width                                    */
    /* 2 Bytes - Left Margin                                   */
    /* 2 Bytes - Top Margin                                    */
    /* 1 Byte - LPI                                            */
    /* 1 Byte - CPI                                            */
    /*                                                         */
    /***********************************************************/

    FF_HeaderData Hdr;

    int temp_int,
        i;

    char *x;

    time_t tim;
    struct tm *tm;

    Hdr.file_id = "%FENFORM%";
    Hdr.major_version = 2;
    Hdr.minor_version = 0;
    Hdr.header_size = 14;

    /* Get user's passwd entry. */
    Hndl->pswd = getpwuid( getuid ( ) );
    if( Hndl->pswd == NULL )
    {
     fprintf( stderr, "Couldn't find passwd entry for UID - %d", getuid( ) );
     return( -1 );
    } else {

     Hdr.user_name = Hndl->pswd->pw_gecos;
     if( *Hdr.user_name )
     {

      if( x = strchr( Hdr.user_name, ',' ) )
       *x = '\0';

     } else {

      Hdr.user_name = Hndl->pswd->pw_name;
     }
     Hdr.header_size += ( strlen( Hdr.user_name ) + 1 );

     Hdr.form_name = Hndl->frm.form_name;
     Hdr.header_size += ( strlen( Hdr.form_name ) + 1 );

     /* Create date string. */
     time( &tim );
     tm = (struct tm *)localtime( &tim );

     Hdr.creation_date = strdup( asctime( tm ) );
     i = strlen ( Hdr.creation_date );
     Hdr.creation_date[ i - 1 ] = '\0';
     Hdr.header_size += ( strlen( Hdr.creation_date ) + 1 );

     Hdr.creation_pgm = "makeform (Vn 2.0)";
     Hdr.header_size += ( strlen( Hdr.creation_pgm ) + 1 );

     Hdr.copyright_notice = "Copyright 2008 Fenland Software Ltd.";
     Hdr.header_size += ( strlen( Hdr.copyright_notice ) + 1 );

     Hdr.form_length = Hndl->cfg.paper_length;
     Hdr.header_size += 2;

     Hdr.form_width = Hndl->cfg.paper_width;
     Hdr.header_size += 2;

     Hdr.left_margin = 0;
     Hdr.header_size += 2;

     Hdr.top_margin = 0;
     Hdr.header_size += 2;

     Hdr.lpi = DFT_LPI;
     Hdr.header_size += 1;

     Hdr.cpi = DFT_CPI;
     Hdr.header_size += 1;

     /* Now write the elements of the header */

     fwrite( Hdr.file_id, 9, 1, form_file_ptr );
     fwrite( &Hdr.header_size, 2, 1, form_file_ptr );
     fwrite( &Hdr.major_version, 1, 1, form_file_ptr );
     fwrite( &Hdr.minor_version, 2, 1, form_file_ptr );
     fwrite( Hdr.user_name, strlen( Hdr.user_name ) + 1, 1, form_file_ptr );
     fwrite( Hdr.form_name, strlen( Hdr.form_name ) + 1, 1, form_file_ptr );
     fwrite( Hdr.creation_date, strlen( Hdr.creation_date ) + 1, 1, form_file_ptr );
     fwrite( Hdr.creation_pgm, strlen( Hdr.creation_pgm ) + 1, 1, form_file_ptr );
     fwrite( Hdr.copyright_notice, strlen( Hdr.copyright_notice ) + 1, 1, form_file_ptr );

     if( Hndl->cfg.paper_orientation == 'L' ) {

      fprintf( stderr, "makeform: Orientation: Landscape\n" );
      fwrite( &Hdr.form_width, 2, 1, form_file_ptr );
      fwrite( &Hdr.form_length, 2, 1, form_file_ptr );

     } else {

      fprintf( stderr, "makeform: Orientation: Portrait\n" );
      fwrite( &Hdr.form_length, 2, 1, form_file_ptr );
      fwrite( &Hdr.form_width, 2, 1, form_file_ptr );

     }

     fwrite( &Hdr.left_margin, 2, 1, form_file_ptr );
     fwrite( &Hdr.top_margin, 2, 1, form_file_ptr );
     fwrite( &Hdr.lpi, 1, 1, form_file_ptr );
     fwrite( &Hdr.cpi, 1, 1, form_file_ptr );
    }

    return( 0 );
}


int MAKEFORM_GetMediaParameters( FF_Handle *Hndl, char *media_type )
{
    /* "Special" Forms */

    FILE *media_file_ptr;
    char stationery_record[ LINE_LEN + 1 ],
         *param_array[ 6 ],
         *s1,
         *s2;

    int  elements,
         ret = -1;

    switch( *( media_type + 1 ) ) {

     /* Landscape */
     case 'L':
     case 'l':
               Hndl->cfg.paper_orientation = 'L';
               break;

     /* Portrait */
     case 'P':
     case 'p':
               Hndl->cfg.paper_orientation = 'P';
               break;

     default:  fprintf( stderr, "makeform: Invalid media type - %s.\n", media_type );
               return( -1 );
               break;
    }



    /* Convert paper size to lower case */
    Hndl->cfg.paper_size = malloc( strlen( media_type ) );
    if( ! Hndl->cfg.paper_size ) {
     fprintf( stderr, "\nmakeform: FATAL ERROR - memory allocation failed for paper_size.\n" );
     return( -1 );
    }

    s1 = media_type + 2;
    s2 = Hndl->cfg.paper_size;
    while( *s1 )
     *( s2++ ) = tolower( *( s1++ ) );

    *s2 = '\0';

    media_file_ptr = fopen( Hndl->cfg.paper_size_file, "r" );

    if( ! media_file_ptr ) {
     fprintf( stderr, "makeform: Unable to open %s.\n", Hndl->cfg.paper_size_file );
     return( -1 );
    }

    while( fgets( stationery_record, LINE_LEN, media_file_ptr ) ) {

     if( strncasecmp( stationery_record, Hndl->cfg.paper_size, strlen( Hndl->cfg.paper_size ) ) ) {
      continue;

     } else {

      if( elements = FENFORM_CFG_ParseCSV( stationery_record, param_array, 5, ',' ) >= 5 ) {

       fprintf( stderr, "makeform: Media type (%s) found.\n", media_type + 2 );

       Hndl->cfg.paper_length = atof( param_array[ 3 ] );
       Hndl->cfg.paper_width = atof( param_array[ 4 ] );

       /* Media Type found */
       ret = 0;
       break;

      } else {
       fprintf( stderr, "makeform: Invalid 'stationery.sizes' record found.  Continuing...\n" );
      }
     }
    }

    fclose( media_file_ptr );

    if( ret < 0 )
     fprintf( stderr, "makeform: Media type (%s) not found in 'stationery.sizes' file.\n", media_type + 2 );

    return( ret );

}


int MAKEFORM_GetOrientation( FF_Handle *Hndl, char *input_file )
{
    FILE *input_file_ptr;

    char buffer[ BUFFER_LEN + 1 ];
    char *mediabox_ptr,
         *ptr;

    int length,
        width,
        start_w = 0,
        end_w = 0,
        start_l = 0,
        end_l = 0;

    if( ( input_file_ptr = fopen( input_file, "r" ) ) ) {

     fseek( input_file_ptr, 0L, SEEK_SET );
     while( fgets( buffer, BUFFER_LEN, input_file_ptr ) ) {

      if( mediabox_ptr = strstr( buffer, "MediaBox" ) ) {
       ptr = strchr( mediabox_ptr, '[' );
       start_w = atoi( ptr++ );
       ptr = strchr( ptr, ' ' );
       start_l = atoi( ptr++ );
       ptr = strchr( ptr, ' ' );
       end_w = atoi( ptr++ );
       ptr = strchr( ptr, ' ' );
       end_l = atoi( ptr++ );

       width = end_w - start_w;
       length = end_l - start_l;

       break;
      }
     }

     if( width == length ) {

      fprintf( stderr, "makeform: Unable to determine the form's orientation.\n" );

      do {

       fprintf( stderr, "makeform: Enter (L)andscape or (P)ortrait  \n" );
       Hndl->cfg.paper_orientation = fgetc( stdin );

      } while( Hndl->cfg.paper_orientation != 'L' && Hndl->cfg.paper_orientation != 'P' );
       
     } else {

      if( width > length )
       Hndl->cfg.paper_orientation = 'L';
      else
       Hndl->cfg.paper_orientation = 'P';

     }

     fclose( input_file_ptr );
     return( 0 );

    } else {

     fprintf( stderr, "\nmakeform: FATAL ERROR in MAKEFORM_GetOrientation()\n" );
     fprintf( stderr, "makeform: Unable to open %s\n\n", input_file );
     return( -1 );

    }
}


int MAKEFORM_Build_EPS_File( FF_Handle *Hndl, char *input_file, char *temp_file )
{

    gs_main_instance *minst;

    char *output_file,
         *paper_size,
         *s1,
         *s2,
         *gsargv[ 11 ];

    int code = 0,
        code1,
        gsargc;

    fprintf( stderr, "makeform: Writing temporary EPS file.\n" );

    output_file = malloc( 13 + strlen( temp_file ) );
    if( ! output_file ) {
     fprintf( stderr, "\nmakeform: FATAL ERROR - memory allocation failed for output_file.\n" );
     return( -1 );
    }
    sprintf( output_file, "-sOutputFile=%s", temp_file );

    paper_size = malloc( 12 + strlen( Hndl->cfg.paper_size ) );
    if( ! paper_size ) {
     fprintf( stderr, "\nmakeform: FATAL ERROR - memory allocation failed for paper_size.\n" );
     return( -1 );
    }
    strcpy( paper_size, "-sPAPERSIZE=" );

    /* Convert paper size to lower case */
    s1 = Hndl->cfg.paper_size;
    s2 = paper_size + 12;

    while( *s1 )
     *( s2++ ) = tolower( *( s1++ ) );

    *s2 = '\0';

    gsargv[ 0 ] = "pdf2eps";	/* actual value doesn't matter */
    gsargv[ 1 ] = "-dNOPAUSE";
    gsargv[ 2 ] = "-dBATCH";
    gsargv[ 3 ] = "-dSAFER";
    gsargv[ 4 ] = "-sDEVICE=epswrite";
    gsargv[ 5 ] = paper_size;
    gsargv[ 6 ] = output_file;
    gsargv[ 7 ] = "-q";
    gsargv[ 8 ] = "-f";
    gsargv[ 9 ] = input_file;
    gsargc=10;

    code = gsapi_new_instance( &minst, NULL );
    if ( code < 0 ) {
     fprintf( stderr, "makeform: GSAPI(1) error %d\n", code );
     return( -1 );
    }

    code = gsapi_init_with_args( minst, gsargc, gsargv );
    code1 = gsapi_exit( minst );
    gsapi_delete_instance( minst );
    free( output_file );

    if ( ( code == 0 ) || ( code == e_Quit ) ) {
     code = code1;
     return( 0 );
    } else {
     fprintf( stderr, "makeform: GSAPI(2) error %d\n", code );
     return( -1 );
    }

}


int MAKEFORM_Copy_EPS_to_Form( FF_Handle *Hndl, FILE *temp_file_ptr, FILE *form_file_ptr )
{
    char buffer[ BUFFER_LEN + 1 ];

    fseek( temp_file_ptr, 0L, SEEK_SET );

    if( MAKEFORM_BuildHeader( Hndl, form_file_ptr ) != 0 )
     return( -1 );

    while( fgets( buffer, BUFFER_LEN, temp_file_ptr ) 
     && strncasecmp( buffer, "%%Page: ", 8 ) ) {
     fprintf( form_file_ptr, "%s", buffer );
    }

    while( fgets( buffer, BUFFER_LEN, temp_file_ptr ) ) {
     if( strncmp( buffer, "%%", 2 ) )
      fprintf( form_file_ptr, "%s", buffer );
    }

    return( 0 );
}


char *MAKEFORM_BuildFileName( FF_Handle *Hndl, char *input_file )
{
    static char *form_file_name;
    char *s;

    if( *input_file == '*' )
     s = input_file + 2;
    else
     s = input_file;

    form_file_name = malloc( strlen( Hndl->cfg.forms_dir ) + strlen( s ) + 5 );
    strcpy( form_file_name, Hndl->cfg.forms_dir );

    Hndl->frm.form_name = malloc( strlen( s ) + 6 );
    memset( Hndl->frm.form_name, 0, strlen( s ) + 6 );

    if( strcasecmp( s + ( strlen( s ) - 4 ), ".pdf" ) == 0 ) {
     if( strchr( s, '/' ) ) {
      strncat( form_file_name, strrchr( s, '/' ), strlen( strrchr( s, '/' ) ) - 4 );
      strncpy( Hndl->frm.form_name, strrchr( s, '/' ) + 1, strlen( strrchr( s, '/' ) ) - 5 );
     } else {
      strcat( form_file_name, "/" );
      strncat( form_file_name, s, strlen( s ) - 4 );
      strncpy( Hndl->frm.form_name, s, strlen( s ) - 4 );
     }
    } else {
     if( strchr( s, '/' ) ) {
      strcat( form_file_name, strrchr( s, '/' ) );
      strcpy( Hndl->frm.form_name, strrchr( s, '/' ) + 1 );
     } else {
      strcat( form_file_name, "/" );
      strcat( form_file_name, s );
      strcpy( Hndl->frm.form_name, s );
     }
    }

    return( form_file_name );

}


int main( int argc, char *argv[ ] )
{
    FILE *form_file_ptr;
    FILE *temp_file_ptr;

    FF_Handle Hndl;

    int opt,
        cfg_flag = 0,
        form_flag = 0,
        error_flag = 0,
        temp_file_dscptr;

    char *input_file,
         *output_file;
    char temp_file[ ] = { "/tmp/fenform.XXXXXX" };

    struct option long_options[ ] = {
                                     { "help", no_argument, NULL, 'h' },
                                     { "version", no_argument, NULL, 'v' },
                                     { "config-file", required_argument, NULL, 'c' },
                                     { "form", required_argument, NULL, 'f' },
                                    };


    if( argc < 2 || argc > 4 ) {
     MAKEFORM_Usage( );
     fprintf( stderr, "makeform: Program ending with errors.\n\n" );
     exit( EXIT_FAILURE );
    }
  
    fprintf( stderr, "makeform: Load form to FenForm directory.\n" );

    /* Clear the structure */
    memset( &Hndl, 0, sizeof( FF_Handle ) );

    Hndl.config_file = _CFG_FILE;

    opterr = 0;
    while( ( opt = getopt_long( argc, argv, "c:f:hv", long_options, NULL ) ) != -1 )
    {
     switch( opt )
     {
      case 'c': if( ! cfg_flag )
                {
                 Hndl.config_file = malloc( (size_t) ( strlen( optarg ) + 2 ) * sizeof( char ) );
                 strcpy( Hndl.config_file, optarg );
                 cfg_flag = 1;
                } else {
                 fprintf( stderr, "\nmakeform: Parameter Error.\n" );
                 MAKEFORM_Usage( );
                 fprintf( stderr, "makeform: Program ending with errors.\n\n" );
                 exit( EXIT_FAILURE );
                }
                break;

      case 'f': if( ! form_flag )
                {
                 Hndl.curr_form = malloc( (size_t) ( strlen( optarg ) + 2 ) * sizeof( char ) );
                 strcpy( Hndl.curr_form, optarg );
                 form_flag = 1;
                } else {
                 fprintf( stderr, "\nmakeform: Parameter Error.\n" );
                 MAKEFORM_Usage( );
                 fprintf( stderr, "makeform: Program ending with errors.\n\n" );
                 exit( EXIT_FAILURE );
                }
                break;

      case '?': fprintf( stderr, "\nmakeform: Unrecognized option - '%c'\n", optopt );
      case 'h':
      case 'v': 
      default:  MAKEFORM_Usage( );
                exit( EXIT_SUCCESS );
                break;

     }
    }


    error_flag = FENFORM_CFG_GetConfig( &Hndl );
    if( error_flag < 0 ) {
     fprintf( stderr, "makeform: Configuration Error.\n" );
     exit( EXIT_FAILURE );

    } else {

     if( ( error_flag > 0 ) && cfg_flag ) {
      fprintf( stderr, "makeform: Specified configuration file not found.\n" );
      fprintf( stderr, "makeform: Program ending with errors.\n\n" );
      exit( EXIT_FAILURE );
     }
    }

    if( optind == argc ) {

     fprintf( stderr, "\nmakeform: Parameter Error.\n" );
     MAKEFORM_Usage( );
     fprintf( stderr, "makeform: Program ending with errors.\n\n" );
     exit( EXIT_FAILURE );

    }

    if( *argv[ optind ] == '*' ) {

     /* "Special" Form */
     if( ! form_flag ) {

      fprintf( stderr, "makeform: Formname required.\n" );
      MAKEFORM_Usage( );
      exit( EXIT_FAILURE );

     } else {

     if( MAKEFORM_GetMediaParameters( &Hndl, argv[ optind ] ) != 0 ) {
      fprintf( stderr, "makeform: Program ending with errors.\n\n" );
      exit( EXIT_FAILURE );

      }

     }

    } else {

     /* Normal Form */
     temp_file_dscptr = mkstemp( temp_file );

     if( temp_file_dscptr == -1 ) {
      fprintf( stderr, "\nmakeform: FATAL ERROR - makeform was unable to open a temporary work-file.\n" );
      fprintf( stderr, "makeform: Program ending with errors.\n\n" );
      exit( EXIT_FAILURE );
     }

     if( MAKEFORM_GetOrientation( &Hndl, argv[ optind ] ) != 0 ) {
      fprintf( stderr, "makeform: Program ending with errors.\n\n" );
      exit( EXIT_FAILURE );
     }

     if( MAKEFORM_Build_EPS_File( &Hndl, argv[ optind ], temp_file ) ) {
      fprintf( stderr, "makeform: Program ending with errors.\n\n" );
      exit( EXIT_FAILURE );
     }

    }

    if( form_flag )
     output_file = MAKEFORM_BuildFileName( &Hndl, Hndl.curr_form );
    else
     output_file = MAKEFORM_BuildFileName( &Hndl, argv[ optind ] );

    form_file_ptr = fopen( output_file, "w" );

    if( form_file_ptr ) {

     if( *argv[ optind ] == '*' ) {

      if( MAKEFORM_BuildHeader( &Hndl, form_file_ptr ) == 0 )
       fprintf( stderr, "makeform: Form written to %s\n", output_file );

     } else {

      temp_file_ptr = fdopen( temp_file_dscptr, "r+" );

      MAKEFORM_Copy_EPS_to_Form( &Hndl, temp_file_ptr, form_file_ptr );
      fprintf( stderr, "makeform: Form written to %s\n", output_file );

      fclose( temp_file_ptr );

      fprintf( stderr, "makeform: Deleting temporary EPS file.\n" );
      unlink( temp_file );

     }

     fclose( form_file_ptr );

     fprintf( stderr, "makeform: Program ending.\n\n" );
     exit( EXIT_SUCCESS );

    } else {

     fprintf( stderr, "\nmakeform: FATAL ERROR - makeform was unable to open the output file.\n\n" );
     fprintf( stderr, "makeform: Program ending with errors.\n\n" );
     exit( EXIT_FAILURE );

    }

}

/*********************************************************************************/
/*                                                                               */
/*                         End of Source File - makeform.c                       */
/*                                                                               */
/*********************************************************************************/

