
/* $Id: netscan.c,v 1.86 2009/07/20 17:37:39 jayrfink Exp $ */

/* gnetscan source file: netscan.c see COPYING for license */

#include <sys/socket.h>
#include <sys/time.h>
#include <sys/types.h>

#include <arpa/inet.h>

#include <netinet/in.h>

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <netdb.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#define DEFAULT_START_PORT 1 /* default starting port to scan */
#define DEFAULT_END_PORT 1024 /* default ending port to scan */
#define DEFAULT_INET_TIMEOUT 2 /* default AF_INET connect timeout */
#define PACKAGE "ns" /* binary exec name */

short int cflag = 0; /* full connect flag */
short int iflag = 0; /* isalive only  flag */
short int vflag = 0; /* verbosity */
short int xflag = 0; /* xtra ports flag */

short int inet_timeo = DEFAULT_INET_TIMEOUT; /* Set connect timeout */
short int inet_utimeo = 0; /* default usec for timer struct */

static void ns_usage (void)
{
	printf(PACKAGE " [[option][arguments]][ipadxn-N|host]\n"
		PACKAGE " [-C][-P][-p n-N][-t n.n][-u][-v][-x]\n"
		"OPTIONS:\n"
		" -C       Connect for each port(not default, slower)\n"
		" -P       Is alive only check (ping select)\n"
		" -p n[-N] Scan port number n or a range of n-N\n"
		"          Defaults are: %i-%i\n"
		" -t n[.n] Set the default scan timeout to SECONDS.USECONDS\n"
		"          Defaults are: %i.0\n"
		" -x       Xtra thorough scan (slower non-strobe scan)\n"
		" -u       Print usage message\n"
		" -v       Be verbose\n"
		"EXAMPLES:\n"
		"  " PACKAGE " -v -p 22-80 192.168.1.10\n"
		"  " PACKAGE " -t 5 -p 22-80 192.168.1.2-254\n"
		"  " PACKAGE " -t 2.050 -v -x  somehost.domain.net\n",
			DEFAULT_START_PORT, DEFAULT_END_PORT, 
			DEFAULT_INET_TIMEOUT
	);
}

static void sho_time(char * msg)
{
	char buffer[256];
	time_t curtime;
	struct tm *loctime;

	curtime = time(NULL);
	loctime = localtime (&curtime);
	printf("%s", msg);
	fputs (asctime(loctime), stdout);
}

/*
 * scan functions-
 *    ns_sockerr: parse connect() errors
 *    ns_shoport: helper for ns_scan; based on v and t print info
 *    ns_isup: helper for ns_scan; check host first before looping ports
 *    ns_scan: scan an individual host using port range
 * These are mostly helpers but could later be broken into modules if need be
 */
static void ns_sockerr(int res)
{
	fprintf(stderr,"Connect error: ");

	switch (res) { 
	case EADDRINUSE:
		fprintf(stderr,"EADDRINUSE\n");
		break;
	case EADDRNOTAVAIL:
		fprintf(stderr,"EADDRNOTAVAIL\n");
		break;
	case EALREADY:
		fprintf(stderr,"EALREADY\n");
		break;
	case ECONNREFUSED:
		fprintf(stderr,"ECONNREFUSED\n");
		break;
	case EHOSTUNREACH:
		fprintf(stderr,"EHOSTUNREACH\n");
		break;
	case ENETDOWN:
		fprintf(stderr,"ENETDOWN\n");
		break;
	case ENETUNREACH:
		fprintf(stderr,"ENETUNREACH\n");
		break;
	case ETIMEDOUT:
		fprintf(stderr,"ETIMEDOUT\n");
		break;
	default:
		fprintf(stderr,"Host timed out (exists?)\n");
		break;
	}
}

static void ns_shoport(u_short port_start, u_short current_port,
                       struct servent *service_info)
{
	switch (vflag) {
	case 0:
		if (!xflag)
			printf("%-5d\n", ntohs(service_info->s_port));
		else
			printf("%-5d\n", port_start + current_port);
		break;
	case 1:
		if (xflag) {
			service_info = getservbyport(htons
			    (port_start + current_port), "tcp");
			if (!service_info) {
				printf("%-5d unknown\n", port_start + current_port);
			} else {
				printf("%-5d %-30s\n",
					port_start + current_port,
					service_info->s_name);
			}
		} else {
			printf("%-5d %-30s\n",
			    ntohs(service_info->s_port),
			    service_info->s_name);
		}
		break;
	}
}

static int ns_isup(struct sockaddr_in scanaddr) 
{
	short int sock;          /* our main socket */
	long arg;                /* for non-block */
	fd_set wset;             /* file handle for bloc mode */
	struct timeval timeout;  /* timeout struct for connect() */

	sock = -1;

	sock = socket(AF_INET, SOCK_STREAM, 0);

	if( (arg = fcntl(sock, F_GETFL, NULL)) < 0) { 
		fprintf(stderr, 
		"Error fcntl(..., F_GETFL) (%s)\n",
			strerror(errno)); 
		return 1;
	}
                                        
	arg |= O_NONBLOCK; 
	if(fcntl(sock, F_SETFL, arg) < 0) {
		fprintf(stderr, 
		"Error fcntl(..., F_SETFL)  (%s)\n",                                            strerror(errno)); 
		return 1;
	}
                                
	/* 
	 * set result stat then try a select if it can take
	 * awhile. This is dirty but works 
	 */
	int res = connect(sock,(struct sockaddr *)&scanaddr,sizeof(scanaddr));
	if (res < 0) {
		if (errno == EINPROGRESS) {
			timeout.tv_sec = inet_timeo;
			timeout.tv_usec = inet_utimeo;
			FD_ZERO(&wset);
			FD_SET(sock, &wset);
			int rc = select(sock + 1,NULL,
			&wset,NULL,&timeout); 

			/* This works great on dead hosts */
			if (rc == 0 && errno != EINTR) {
				ns_sockerr(res);
				close (sock);
				return 1; 
			}
		}
	}
	close(sock);
	return 0;
} 

static void ns_scan(u_short port_start, u_short port_end, 
                    struct sockaddr_in scanaddr)
{
	u_short current_port;         /* the current port being scanned */
	short int sock;               /* our main socket */
	short int hostalive;          /* node is alive - used for skipping */
	register u_int finished;      /* 1 when deep scanning is finished */
	register u_int goodproto;     /* 1 when a protocol mapped to service */
	struct servent *service_info; /* service information structure */

	service_info = getservent();
	current_port = 0; 
	finished = 0;
	goodproto = 0;
	hostalive = 0;
	sock = -1;

	if (!xflag) setservent(1); /* not thorough get /etc/services handle */
	if (vflag) sho_time("Scan start: ");
	while (((port_start + current_port) <= port_end) || !finished) {
		scanaddr.sin_family = AF_INET;
		sock = socket(AF_INET, SOCK_STREAM, 0);
		if (sock == -1) {
			fprintf(stderr, "Error  assigning master socket\n");
			exit (EXIT_FAILURE);
		}

		/* not thorough then strobe using known services */
		if (!xflag) {
			while (!goodproto) {
				service_info = getservent();
				if (!service_info)
					break;
				if (!strcmp(service_info->s_proto, "tcp")
				    && (ntohs(service_info->s_port) <= port_end)
				    && (ntohs(service_info->s_port) >=
					port_start)) {
					goodproto = 1;
					break; 
				}
			}

			if (!goodproto) break;

			if (!service_info) {
				finished = 1;
				break;
			}

			if (goodproto) scanaddr.sin_port = service_info->s_port;
			goodproto = 0;
		} else
			scanaddr.sin_port = htons(port_start + current_port);

		/* do a basic connect/select test before the real check */
		if (!finished) {
			/* If we have not checked already see if it is alive */
			if (hostalive == 0)
				if (ns_isup(scanaddr) > 0) return;

			if (!cflag)
				hostalive = 1;

			if (iflag > 0) {
				printf("Host is alive\n");
				close (sock);
				break;
			}

			if (connect
			    (sock, (struct sockaddr *)&scanaddr,
			     sizeof(scanaddr)) == 0) {
				ns_shoport(port_start, current_port, 
		 		    service_info);
			} else if (errno == 113) { /* Crap */
				fprintf(stderr, "No route to host\n");
				finished = 1;
				break;
			} 
		}

		close(sock);
		current_port++;
		if (port_start + current_port >= port_end) 
			finished = 1;
	}

	if (vflag) sho_time("Scan end: ");
	if (!xflag) endservent();
}

/* MAIN */
int main(int argc, char **argv)
{
	register int i, y;                        
	u_short port_start; /* Starting port */
	u_short port_end;   /* Last port to scan */
	register int subnet_end; /* Last address in the _subnet_ to scan */
	register int subnet_start; /* First address in the _subnet_ to scan */
	char *tmpvar_subnet, *token_subnet;   /* Parses the - operator for net */
	char *tmpvar_port, *token_port; /* Parses the - operator for port */
	char *tmpvar_timeval, *token_timeval; /* parses . for timeval */
	char *addr_start_vp; /* starting adx string pointer */
	char *addr_end_vp; /* ending string adx pointer */
	char *argv_cp; /* copy of argp pointer */
	struct sockaddr *cur_address;     /* Current address in subnet */
	struct sockaddr *end_address; /* Ending subnet address */
	char addr[1024];   /* Initial ascii addr to connect to */
	struct sockaddr_in address, address_end, scanaddr; /* Address structs*/
	struct hostent *host_info;            /* Host information structure */
	char start_addr_cp[1023];
	char *tmp_subnet_start;

	if (!argv[1]) {
		fprintf(stderr, "Syntax error\n");
		ns_usage();
		return EXIT_FAILURE;
	 }

	/* Init defaults */
	port_start = DEFAULT_START_PORT; 
	port_end = DEFAULT_END_PORT;   
	subnet_end = 0;

	/* XXX Thoughts about these:
		- Be better in data structures?
		- Be easier with posix getopt?
		- Tokenizing should be broken off and just 
	      return the values
	*/
	switch (argc) {
	case 2: /* Trap help print request */
		if ((!strcmp(argv[1], "-?")) || (!strcmp(argv[1], "-u"))) {
			ns_usage();
			return EXIT_SUCCESS;
		} else {
			break;
		}

	default:
		for (i = 1; i < argc - 1; i++) {
			/* verbose flag */
			if (!strcmp(argv[i], "-v")) {
				++vflag;
			/* isalive only check */
			} else if (!strcmp(argv[i], "-P")) {
				++vflag;
				++iflag;
			/* force connect on every port flag */
			} else if (!strcmp(argv[i], "-C")) {
				++cflag;
			/* port specification */
			} else if (!strcmp(argv[i], "-p")) {
				tmpvar_port = (argv[i+1]);
				token_port = strtok (tmpvar_port, "-");
				if (!token_port) {
					fprintf(stderr,
						"Error! No port specified\n");
					ns_usage();
					return EXIT_FAILURE;
				} else {
					port_start = atoi(token_port);
					token_port = strtok (NULL, "-");
					if (token_port)
						port_end = atoi(token_port);
					else
						port_end = port_start;
				}

				i++;
			/* timer specification */
			} else if (!strcmp(argv[i], "-t")) {
				tmpvar_timeval = (argv[i+1]);
				token_timeval = strtok (tmpvar_timeval, ".");
				if (!token_timeval) {
					fprintf(stderr,
					  "Error! No time specified\n");
					ns_usage();
					return EXIT_FAILURE;
				} else {
					inet_timeo = atoi(token_timeval);
					token_timeval = strtok (NULL, ".'");
					if (token_timeval)
						inet_utimeo=atoi(token_timeval);
				}

				i++;
			/* Don't strobe (e.g do not rely on /etc/services) */
			} else if (!strcmp(argv[i], "-x")) {
				++xflag;
			}
		}
	}

	/*
	 * If the base port is bigger than end exit or if base is less
	 * than 0 exit.
	 */
	if ((port_start > port_end) || ((short)port_start < 0)) {
		fprintf(stderr, "Bad port range : start=%d end=%d\n",
			port_start, port_end);
		ns_usage();
		return EXIT_FAILURE;
	}

	/* XXX There is something off about this code - it works
	       but is confusing as hell. Grabbing the end and start
	       addresses in base10 int form might be something
	       better served with a helper function for clarity 
	*/
	argv_cp = argv[argc - 1];
	addr_start_vp = strtok(argv_cp, "-");
	addr_end_vp = strtok(NULL, "-");

	/* If there is not a -then make subnet end same as the start */
	if ( addr_end_vp)  {
		subnet_end = atoi(addr_end_vp);
		strncpy(start_addr_cp, addr_start_vp, 1023);
		tmp_subnet_start = strtok(start_addr_cp, ".");
		for (y = 3; y != 0; y--) 
			tmp_subnet_start = strtok(NULL, ".");

		subnet_start = atoi(tmp_subnet_start);
	} else 
		subnet_end = subnet_start;

	strncpy(addr, addr_start_vp, 1023);

	/* Initialize the address structure - zero out and assign */
	bzero((char *)&address, sizeof(address));
	address.sin_family = AF_INET;
	addr[1023] = 0; /* set the last element in the array to 0 */

	/* Try to resolve host in either direction */
	if ((host_info = gethostbyname(addr))) {
		bcopy(host_info->h_addr, (char *)&address.sin_addr,
		      host_info->h_length);
	} else if ((address.sin_addr.s_addr = inet_addr(addr_start_vp)) ==
		   INADDR_NONE) {
		fprintf(stderr, "Could not resolve host\n" );
		return EXIT_FAILURE;
	} else if (vflag)
		printf(" address valid\n");

	if (vflag)
		printf("Timeouts: sec %i usec %i\n",
			 inet_timeo, inet_utimeo);

	while (subnet_start <= subnet_end) {
			cur_address = (struct sockaddr *)&address; 
			end_address = (struct sockaddr *)&address_end; 
			end_address->sa_data[5] = subnet_end;
			printf("Host: ");
			printf("%d.%d.%d.%d\n", 
			    (u_char) cur_address->sa_data[2],
			    (u_char) cur_address->sa_data[3],
			    (u_char) cur_address->sa_data[4],
			    (u_char) cur_address->sa_data[5]);

			bcopy(cur_address, &scanaddr, sizeof(scanaddr));
			if (vflag) 
				if ((port_start != port_end) && (iflag == 0))
					printf("Port range: %d-%d\n", 
					    port_start, port_end);

			ns_scan(port_start, port_end, scanaddr);
			cur_address->sa_data[5]++;
			subnet_start++;
	}

	return EXIT_SUCCESS;
}
