/*
   ringbuf.c -- Functions for ring buffer management

   Copyright (C) 2007 Laszlo Menczel
   
   This is free software with NO WARRANTY. Distributed under
   the GNU Library General Public Licence (LGPL) version 2.1.
*/

#include <stdio.h>
#include <malloc.h>
#include <string.h>

#include "sndcap.h"
#include "sndcaplib.h"

enum { INSERT, REMOVE };

//=======================================================================
//  Local functions
//=======================================================================

static void ring_incr(ringbuf_t *buf, int ptr)
{
  if (ptr == RBUF_TAIL)
  {
    buf->tail += buf->incr;
    if (buf->tail == buf->size)
      buf->tail = 0;
  }
  else if (ptr == RBUF_HEAD)
  {
    buf->head += buf->incr;
    if (buf->head == buf->size)
      buf->head = 0;
  }
  else
    return;
}

//=======================================================================

static void copy_byte(void *buf, void *data, int dir, int count)
{
  int i;
  byte *src, *dst;
    
  if (dir == INSERT)
  {
    src = (byte *) data;
    dst = (byte *) buf;
  }
  else if (dir == REMOVE)
  {
    dst = (byte *) data;
    src = (byte *) buf;
  }
  else
    return;
    
  for (i = 0; i < count; i++)
    dst[i] = src[i];
}

//=======================================================================

static void copy_word(void *buf, void *data, int dir, int count)
{
  int i;
  word *src, *dst;
    
  if (dir == INSERT)
  {
    src = (word *) data;
    dst = (word *) buf;
  }
  else if (dir == REMOVE)
  {
    dst = (word *) data;
    src = (word *) buf;
  }
  else
    return;
    
  for (i = 0; i < count / sizeof(word); i++)
    dst[i] = src[i];
}

//=======================================================================

static void copy_dword(void *buf, void *data, int dir, int count)
{
  int i;
  dword *src, *dst;
    
  if (dir == INSERT)
  {
    src = (dword *) data;
    dst = (dword *) buf;
  }
  else if (dir == REMOVE)
  {
    dst = (dword *) data;
    src = (dword *) buf;
  }
  else
    return;
    
  for (i = 0; i < count / sizeof(dword); i++)
    dst[i] = src[i];
}

//=======================================================================
// Public functions
//=======================================================================

ringbuf_t *rbuf_create(int count, int size)
{
  int buf_size;
  ringbuf_t *buf;
  
  if (count < 2 || size < 2)
    ERR(RBERR_BAD_ARG)

  buf = (ringbuf_t *) malloc(sizeof(ringbuf_t));

  if (buf == NULL)
    ERR(RBERR_ALLOC)
    
  buf_size = count * size;
  buf->data = (byte *) malloc(buf_size);
  
  if (buf == NULL)
  {
    free(buf);
    ERR(RBERR_ALLOC)
  }

  memset((void *) buf->data, 0, size);
  buf->size = buf_size;
  buf->incr = size;
  buf->dynamic = 1;
  buf->count = buf->tail = buf->head = buf->overflow = 0;

  return buf;
}
  
//=======================================================================

int rbuf_destroy(ringbuf_t *buf)
{
  if (buf == NULL)
    ERR(RBERR_NULL_PTR);
    
  if (! buf->dynamic)
    ERR(RBERR_BUF_STATIC)
    
  if (buf->data != NULL)
    free(buf->data);

  free(buf);
  RETURN(1)
}

//=======================================================================

int rbuf_reset(ringbuf_t *b)
{
  int i;

  if (b == NULL)
    ERR(RBERR_NULL_PTR)
    
  if (b->data == NULL || b ->size == 0)
    ERR(RBERR_BAD_BUF)
    
  for (i = 0; i < b->size; i++)
    b->data[i] = 0;
       
  b->count = b->tail = b->head = b->overflow = 0;
  RETURN(1)
}

//=======================================================================

/*
   Increments the tail or head pointer of a type 'ringbuf_t' ring buffer.
   The pointer wraps around to zero if it reaches the end of the buffer.
   The field 'count' is incremented when the head pointer is incremented,
   and is decremented if the tail pointer is incremented. So its value is
   the number of data items in the buffer.
   
   The buffer may overflow (i.e. a new item is added to a buffer already full).
   In this case the oldest item is overwritten and the 'overflow' field of the
   buffer structure is set to 1. Otherwise the value of the 'overflow' field is
   zero. The consumer of buffer data should check this flag, and perform the
   necessary adjustment if it is non-zero. When the item at 'tail' has been
   processed and removed from the buffer, the flag is set back to zero. Please
   note that a non-zero flag value means at least one (but possibly several)
   items overwritten and lost.

   Returns failure (zero) if 'buf' is invalid.
*/

int rbuf_incr(ringbuf_t *buf, int which)
{
  if (buf == NULL)
    ERR(RBERR_NULL_PTR)

  if (buf->data == NULL)
    ERR(RBERR_BAD_BUF)
 
  //======== removing an item

  if (which == RBUF_TAIL)
  {
    if (buf->count == 0)				// buffer empty
      ERR(RBERR_EMPTY)

    ring_incr(buf, RBUF_TAIL);
    buf->overflow = 0;
    buf->count--;
    RETURN(1)
  }

  //======== adding an item

  else if (which == RBUF_HEAD)
  {
    // test for overflow

    if (buf->tail == buf->head && buf->count > 0)	// buffer full
    {
      buf->overflow = 1;
      ring_incr(buf, RBUF_TAIL);
    }
    else
      buf->count++;

    ring_incr(buf, RBUF_HEAD);
    RETURN(1)
  }

  ERR(RBERR_BAD_ARG)
}

//=======================================================================

/*
   Returns non-zero (empty) if 'buf' is invalid, so that a non-existent buffer
   is always treated as empty and (hopefully) no attempt is made to read data
   from it.
*/

int rbuf_empty(ringbuf_t *buf)
{
  if (buf == NULL)
  {
    __scap_err = RBERR_NULL_PTR;  
    return 1;
  }

  if (buf->data == NULL)
  {
    __scap_err = RBERR_BAD_BUF;  
    return 1;
  }

  __scap_err = RBERR_NONE;

  if (buf->count == 0)
    return 1;
    
  return 0;
}

//=======================================================================

/*
   Returns non-zero (full) if 'buf' is invalid, so that a non-existent buffer
   is always treated as full and (hopefully) no attempt is made to add data.
*/

int rbuf_full(ringbuf_t *buf)
{
  if (buf == NULL)
  {
    __scap_err = RBERR_NULL_PTR;  
    return 1;
  }

  if (buf->data == NULL)
  {
    __scap_err = RBERR_BAD_BUF;  
    return 1;
  }

  __scap_err = RBERR_NONE;

  if (buf->tail == buf->head && buf->count > 0)
    return 1;
    
  return 0;
}

//=======================================================================

int rbuf_insert(ringbuf_t *buf, void *data, int data_type)
{
  if (buf == NULL)
    ERR(RBERR_NULL_PTR)
    
  if (buf->data == NULL)
    ERR(RBERR_BAD_BUF)
    
  if (buf->tail == buf->head && buf->count > 0)
  {
    ring_incr(buf, RBUF_TAIL);
    buf->overflow = 1;
  }
  
  switch (data_type)
  {
    case RBUF_BYTE:
     copy_byte((void *) (buf->data + buf->head), data, INSERT, buf->incr);
     break;
          
    case RBUF_WORD:
     copy_word((void *) (buf->data + buf->head), data, INSERT, buf->incr);
     break;
     
    case RBUF_DWORD:
     copy_dword((void *) (buf->data + buf->head), data, INSERT, buf->incr);
     break;
     
    default:
      return 0;
  }

  ring_incr(buf, RBUF_HEAD);
  buf->count++;
  RETURN(1)
}

//=======================================================================

int rbuf_remove(ringbuf_t *buf, void *data, int data_type)
{
  if (buf == NULL)
    ERR(RBERR_NULL_PTR)
    
  if (buf->data == NULL)
    ERR(RBERR_BAD_BUF)
    
  if (buf->count == 0)
    ERR(RBERR_EMPTY)

  if (data != NULL)		// copy data before removing
  {
    switch (data_type)
    {
      case RBUF_BYTE:
      copy_byte((void *) (buf->data + buf->head), data, REMOVE, buf->incr);
      break;
          
      case RBUF_WORD:
       copy_word((void *) (buf->data + buf->head), data, REMOVE, buf->incr);
       break;
     
      case RBUF_DWORD:
       copy_dword((void *) (buf->data + buf->head), data, REMOVE, buf->incr);
       break;
     
      default:
        return 0;
    }
  }
  
  ring_incr(buf, RBUF_TAIL);
  buf->overflow = 0;
  buf->count--;
  RETURN(1)
}
