/*
   capture.c  --  Main module
   
   SCAP Voice Capture Library
   
   Copyright (C) 2007 Laszlo Menczel
   
   This is free software with NO WARRANTY. Distributed under
   the GNU Library General Public Licence (LGPL) version 2.1.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "sndcap.h"
#include "sndcaplib.h"

//============================================================================

#define DEF_NUM_CHAN	1
#define DEF_CAP_FREQ	8000
#define DEF_DAT_SIZE	16
#define MIN_BUF_INCR	800

ringbuf_t *__scap_buf = NULL;

int __scap_err       = 0;
int __scap_init      = 0;
int __scap_recording = 0;
int __scap_num_buf   = 0;
int __scap_num_chan  = DEF_NUM_CHAN;
int __scap_dat_size  = DEF_DAT_SIZE;
int __scap_cap_freq  = DEF_CAP_FREQ;

//============================================================================

// Sleep() for Linux

#if defined _LINUX

#include <time.h>

static struct timespec sleep_time;
static struct timespec rem_time;

void Sleep(int msec)
{
  sleep_time.tv_sec = 0;
  sleep_time.tv_nsec = 1000000 * msec;
  nanosleep(&sleep_time, &rem_time);
}

#endif

//============================================================================

int scap_init(int chan, int freq, int bits, ringbuf_t *buf)
{
  if (chan != 0)
  {
    switch (chan)
    {
      case SCAP_MONO:
      case SCAP_STEREO:
        __scap_num_chan = chan;
        break;
      
      default:
        ERR(CAPERR_BAD_ARG)
    }
  }
  else
    __scap_num_chan = DEF_NUM_CHAN;
      
  if (freq != 0)
  {
    switch (freq)
    {
      case SCAP_8K:
      case SCAP_11K:
      case SCAP_22K:
      case SCAP_44K:
        __scap_cap_freq = freq;
        break;

      default:
        ERR(CAPERR_BAD_ARG)
        break;
    }
  }
  else
    __scap_cap_freq = DEF_CAP_FREQ;
          
  if (bits != 0)
  {
    switch (bits)
    {
      case SCAP_BYTE:
      case SCAP_WORD:
        __scap_dat_size = bits;
        break;
        
      default:
        ERR(CAPERR_BAD_ARG)
        break;
    }
  }        
  else
    __scap_dat_size = DEF_DAT_SIZE;
    
  if (buf == NULL)
    ERR(CAPERR_NULL_PTR)

  if (
       buf->data == NULL         ||
       buf->incr < MIN_BUF_INCR  ||
       buf->size / buf->incr < 2 ||
       buf->size % buf->incr != 0
     )
     ERR(CAPERR_BAD_BUF)

  __scap_buf = buf;

  if (! __scap_open_device())
    ERR(CAPERR_DEV_OPEN)

  __scap_init = 1;

/* !!!  atexit(scap_exit); */

  RETURN(1)
}

//============================================================================

void scap_exit(void)
{
  if (__scap_recording)
    __scap_stop();
    
  __scap_close_device();
  __scap_buf = NULL;
  __scap_init = 0;
  __scap_err = CAPERR_NONE;
}  

//============================================================================

int scap_start(void)
{
  if (! __scap_init)
    ERR(CAPERR_INIT)

  if (__scap_recording)
    ERR(CAPERR_IN_RECORD)

  if (! __scap_start())
    return 0;

  __scap_recording = 1;
  RETURN(1)
}

//============================================================================

int scap_stop(void)
{
  if (! __scap_init)
    ERR(CAPERR_INIT)

  if (! __scap_recording)
    ERR(CAPERR_NOT_RECORD)

  __scap_stop();
  __scap_recording = 0;
  RETURN(1)
}
