/*
                               PARSE.C

         Functions for parsing different tokens from a string buffer.
   
                  Copyright (C) Laszlo Menczel, 2005
                         menczel@invitel.hu

        This is free software without warranty. See 'licence.txt'.
*/                  

#include <stdio.h>
#include <string.h>

#include "mutil.h"
#include "mutlib.h"

//==========================================================================

/*   
   Extracts a token from 'buf'. Returns a pointer to the next character
   beyond the token, or NULL in case of error. The zero character is
   interpreted as the end of buffer.
   
   Assumes that 'dest' can hold 'maxlen' + 1 characters (there is room for
   the terminating zero even if 'maxlen' characters have been copied).
*/

char *mut_get_token(char *dest, char *buf, char *delim, int maxlen)
{
  int n;

  if (delim == NULL)
    delim = __mut_white_sp;

  dest[0] = 0;

  while (1)
  {
    if (__mut_is_delimiter(*buf, delim))
      buf++;
    else if (*buf == 0)
      MUTERR_P(MUTERR_END_OF_STR)
    else
      break;
  }

  n = 0;
  while (n < maxlen)
  {
    if (__mut_is_delimiter(*buf, delim) || *buf == 0)
       break;
    else
    {
      dest[n] = *buf;
      buf++;
      n++;
    }
  }

  dest[n] = 0;

  if (n == 0)
    MUTERR_P(MUTERR_NOT_FOUND)
  else if (n == maxlen)
    MUTERR_P(MUTERR_TOO_MANY)

  RETURN(buf)
}

//==========================================================================

/*
   This function can be used for extracting tokens from a buffer which starts
   at 'buf' and ends at 'end' ('end' is a pointer to the last character in the
   buffer). Zero characters are also interpreted as delimiters. Returns a
   pointer to the next character beyond the token, or NULL if error.
   
   Assumes that 'dest' can hold 'maxlen' + 1 characters (there is room for
   the terminating zero even if 'maxlen' characters have been copied).
*/

char *mut_get_token_ex(char *dest, char *buf, char *end, char *delim, int maxlen)
{
  int n;

  if (delim == NULL)
    delim = __mut_white_sp;

  dest[0] = 0;

  while (1)
  {
    if (buf > end)
      MUTERR_P(MUTERR_END_OF_FILE)
    else if (*buf == 0 || __mut_is_delimiter(*buf, delim))
      buf++;
    else
      break;
  }

  n = 0;
  while (n < maxlen)
  {
    if (buf > end || __mut_is_delimiter(*buf, delim) || *buf == 0)
       break;
    else
    {
      dest[n] = *buf;
      buf++;
      n++;
    }
  }

  dest[n] = 0;

  if (n == 0)
    MUTERR_P(MUTERR_NOT_FOUND)
  else if (buf > end || n == maxlen)
    MUTERR_P(MUTERR_TOO_MANY)

  RETURN(buf)
}

//===========================================================================

char *mut_parse_int(char *buf, char *delim, int base, int *dest)
{
  int maxlen;
  char token[MUT_MAX_NUM_LEN + 2], *next;

  if (buf == NULL || dest == NULL)
    MUTERR_P(MUTERR_BAD_ARG)
  
  if (base == MUT_BASE_BIN)
    maxlen = MUT_MAX_BIN_LEN;
  else if (base == MUT_BASE_DEC)
    maxlen = MUT_MAX_DEC_LEN;
  else if (base == MUT_BASE_HEX)
    maxlen = MUT_MAX_HEX_LEN;
  else
    MUTERR_P(MUTERR_BAD_ARG)

  next = mut_get_token(token, buf, delim, maxlen);
  if (next == NULL)
    return NULL;

  if (! mut_stoi(token, base, dest))
    return NULL;

  RETURN(next)
}

//===========================================================================

char *mut_parse_float(char *buf, char *delim, double *dest)
{
  char token[MUT_MAX_FLT_LEN + 2], *next;

  if (buf == NULL || dest == NULL)
    MUTERR_P(MUTERR_BAD_ARG)
  
  next = mut_get_token(token, buf, delim, MUT_MAX_FLT_LEN);
  if (next == NULL)
    return NULL;

  if (! mut_stof(token, dest))
    return NULL;

  RETURN(next)
}

//===========================================================================

char *mut_parse_int_ex(char *buf, char *end, char *delim, int base, int *dest)
{
  int maxlen;
  char token[MUT_MAX_NUM_LEN + 2], *next;

  if (buf == NULL || end == NULL || dest == NULL || end < buf)
    MUTERR_P(MUTERR_BAD_ARG)
  
  if (base == MUT_BASE_BIN)
    maxlen = MUT_MAX_BIN_LEN;
  else if (base == MUT_BASE_DEC)
    maxlen = MUT_MAX_DEC_LEN;
  else if (base == MUT_BASE_HEX)
    maxlen = MUT_MAX_HEX_LEN;
  else
    MUTERR_P(MUTERR_BAD_ARG)

  if (maxlen > end - buf + 1)
    maxlen = end - buf + 1;

  next = mut_get_token_ex(token, buf, end, delim, maxlen);
  if (next == NULL)
    return NULL;

  if (! mut_stoi(token, base, dest))
    return NULL;

  RETURN(next)
}

//===========================================================================

char *mut_parse_float_ex(char *buf, char *end, char *delim, double *dest)
{
  int maxlen;
  char token[MUT_MAX_FLT_LEN + 2], *cp;

  maxlen = end - buf + 1;
  if (maxlen > MUT_MAX_FLT_LEN)
    maxlen = MUT_MAX_FLT_LEN;
    
  if (buf == NULL || end == NULL || dest == NULL || end < buf)
    MUTERR_P(MUTERR_BAD_ARG)
  
  token[0] = 0;

  cp = mut_get_token_ex(token, buf, end, delim, maxlen);
  if (cp == NULL)
    return NULL;

  if (! mut_stof(token, dest))
    return NULL;

  RETURN(cp)
}
