/*
                               NUMCONV.C

                Functions for stirng/number conversions
                    
                   Copyright (C) Laszlo Menczel, 2005
                         menczel@invitel.hu

        This is free software without warranty. See 'licence.txt'.
*/

#include <stdio.h>
#include <string.h>

#include "mutil.h"
#include "mutlib.h"

//==========================================================================

static int conv_bin_digit(char c);
static int conv_dec_digit(char c);
static int conv_hex_digit(char c);

/* ------- not used at present ------------------
static char *bin_char = "01";
static char *dec_char = "+-0123456789";
static char *hex_char = "0123456789aAbBcCdDeEfF";
static char *flt_char = "+-.0123456789eE";
------------------------------------------------*/

//==========================================================================

#define MAX_DIGIT_NUM 16
#define MAX_EXP_VAL   (308 - MAX_DIGIT_NUM)

//==========================================================================

int mut_itobs(void *data, int len, char *buf)
{
  int n;
  long mask, flag;
  char tmp[34];

  if (len != 8 && len != 16 && len != 32)
    MUTERR(MUTERR_BAD_ARG)

  if (data == NULL || buf == NULL)
    MUTERR(MUTERR_BAD_ARG)

  mask = 1;
  for (n = len - 1; n >= 0; n--)
  {
    if (len == 8)
      flag = (long) (*((char *) data) & (char) mask);
    else if (len == 16)
      flag = (long) (*((short *) data) & (short) mask);
    else if (len == 32)
      flag = *((long *) data) & mask;

    if (flag)
      tmp[n] = '1';
    else
      tmp[n] = '0';

    mask = mask << 1;
  }

  tmp[len] = 0;
  strcpy(buf, tmp);

  RETURN(1)
}

//==========================================================================

int mut_stoi(char *s, int base, int *dest)
{
  int n, start, len, maxlen, tmp;
  int (* conv_func)(char);
  long fact, mult, sum, sign;

  if (s == NULL || dest == NULL)
    MUTERR(MUTERR_BAD_ARG)

  if (base == MUT_BASE_BIN)
  {
    mult = 2L;
    maxlen = 32;
    conv_func = conv_bin_digit;
  }
  else if (base == MUT_BASE_DEC)
  {
    sign = 1L;
    mult = 10L;
    maxlen = 10;
    conv_func = conv_dec_digit;
  }
  else if (base == MUT_BASE_HEX)
  {
    mult = 16L;
    maxlen = 8;
    conv_func = conv_hex_digit;
  }
  else
    MUTERR(MUTERR_INT_BASE)

  start = len = 0;
  while (len < maxlen && s[len] != 0)
  {
    if (len == 0 && s[len] == '-')
    {
      if (base == MUT_BASE_DEC)
      {
        sign = -1L;
        maxlen++;
        start++;
        len++;
        continue;
      }
      else
        MUTERR(MUTERR_NOT_DIG)
    }

    len++;
  }

  if (len == maxlen)
    MUTERR(MUTERR_EXTRA_CHAR)

  fact = 1L;
  sum = 0L;
  for (n = len - 1; n >= start; n--)          /* process in reverse order */
  {
    tmp = conv_func(s[n]);
    if (tmp < 0)
      return tmp;
    sum += fact * (long) tmp;
    fact *= mult;
  }

  if (base == MUT_BASE_DEC)
    sum *= sign;

  *dest = (int) sum;

  RETURN(1)
}

//==========================================================================

int mut_stof(char *s, double *dest)
{
  double fact, valsum, valsign, expmult;
  int n,
      strend,
      dotpos,
      valstart,
      valend,
      valcount,
      expstart,
      expend,
      expsum,
      expsign;

  valsign = 1.0;
  valstart = 0;
  valend = 0;
  valcount = 0;

  expsign = 1;
  expstart = 0;
  expend = 0;

  strend = 0;
  dotpos = -1;
  n = 0;

  /*******************/
  /* check arguments */
  /*******************/

  if (s == NULL || dest == NULL || s[0] == 0)
    MUTERR(MUTERR_BAD_ARG)

  /****************************************/
  /* check for leading plus or minus sign */
  /****************************************/

  if (s[0] == '-')
  {
    valsign = -1.0;
    n++;
    valstart++;
  }
  
  if (s[0] == '+')
  {
    n++;
    valstart++;
  }
  
  /*******************************/
  /* scan digits & decimal point */
  /*******************************/

  while (1)
  {
    if (valcount == MAX_DIGIT_NUM)
      MUTERR(MUTERR_TOO_MANY_DIG)                         /* too many digits */

    if (s[n] == 0)
    {
      strend = n;
      if (valcount > 0)
      {
        valend = n - 1;
        break;
      }
      else
        MUTERR(MUTERR_NO_DIG)                                /* no digits */
    }

    if (s[n] == '.')
    {
      if (dotpos == -1)
      {
        dotpos = n;
        n++;
        continue;
      }
      else
        MUTERR(MUTERR_TOO_MANY_DOTS)        /* more than one decimal point */
    }

    if (s[n] == 'e' || s[n] == 'E')
    {
      if (valcount > 0)
      {
        valend = n - 1;
        n++;
        break;
      }
      else
        MUTERR(MUTERR_NO_EXP)            /* no digits before exponent */
    }

    if (s[n] < '0' || s[n] > '9')
      MUTERR(MUTERR_NOT_DIG)                           /* not a digit */

    valcount++;
    n++;
  }

  /*****************************/
  /* scan exponent part if any */
  /*****************************/

  if (strend)
    goto string_scanned;

  if (s[n] == '+')
    n++;
  else if (s[n] == '-')
  {
    expsign = -1;
    n++;
  }

  if (s[n] == 0)
    MUTERR(MUTERR_END_OF_STR)              /* premature end or not a digit */
  else if (s[n] < '0' || s[n] > '9')
    MUTERR(MUTERR_BAD_EXP)

  expstart = expend = n;

  n++;

  if (s[n] == 0)
    goto string_scanned;
  else if (s[n] < '0' || s[n] > '9')
    MUTERR(MUTERR_BAD_EXP)                                  /* not a digit */

  expend++;

  n++;

  if (s[n] == 0)
    goto string_scanned;
  else if (s[n] < '0' || s[n] > '9')
    MUTERR(MUTERR_BAD_EXP)                                  /* not a digit */

  expend++;

  n++;
  if (s[n] != 0)
    MUTERR(MUTERR_EXTRA_CHAR)            /* excess characters in the string */

string_scanned:

  /***************************************************************
    Now we have all info and can convert the string. We start by
    converting the integer part backward.
  ***************************************************************/

  if (dotpos != -1)
  {
    if (valstart > dotpos)
      goto skip_integer_part;
    else
      n = dotpos - 1;
  }
  else
    n = valend;

  fact = 1.0;
  valsum = 0.0;
  while (n >= valstart)
  {
    valsum += fact * (double) (s[n] - '0');
    fact = fact * 10.0;
    n--;
  }

skip_integer_part:

  /***********************************/
  /* convert fractional part forward */
  /***********************************/

  if (dotpos == -1 || valend < dotpos)
    goto skip_fractional_part;
  else
    n = dotpos + 1;

  fact = 0.1;
  while (n <= valend)
  {
    valsum += fact * (double) (s[n] - '0');
    fact = fact / 10.0;
    n++;
  }

skip_fractional_part:

  valsum *= valsign;

  /*****************************/
  /* convert exponent backward */
  /*****************************/

  expsum = 0;
  if (expstart != 0)
  {
    fact = 1;
    n = expend;
    while (n >= expstart)
    {
      expsum += fact * (s[n] - '0');
      fact = fact * 10;
      n--;
    }

    if (expsum > MAX_EXP_VAL)
      MUTERR(MUTERR_EXP_RANGE)
  }

  /*************************************************************/
  /* multiply with appropriate value derived from the exponent */
  /*************************************************************/

  if (expsum > 0)
  {
    if (expsign > 0)
      fact = 10.0;
    else
      fact = 0.1;

    expmult = fact;
    for (n = 1; n < expsum; n++)
      expmult *= fact;

    valsum *= expmult;
  }

  *dest = valsum;

  RETURN(1)
}

//==========================================================================
// Local functions
//==========================================================================

/*
   Checks if <c> is a binary digit (0 or 1). Returns the integer value
   of the digit, or MUTERR_NOT_DIG if error.
*/

static int conv_bin_digit(char c)
{
  if (c == '0')
    return 0;
  else if (c == '1')
    return 1;
  else
    return MUTERR_NOT_DIG;
}

//==========================================================================

/*
   Checks if <c> is a decimal digit (0 to 9). Returns the integer value
   of the digit, or MUTERR_NOT_DIG if error.

   Note: Assumes ASCII encoding of the character converted.
*/

static int conv_dec_digit(char c)
{
  if (c >= '0' && c <= '9')
    return (int) c - '0';
  else 
    return MUTERR_NOT_DIG;
}

//==========================================================================

/*
   Checks if <c> is a hexadecimal digit (0-9, a-f, A-F). Returns the
   integer value of the digit, or MUTERR_NOT_DIG if error.

   Note: Assumes ASCII encoding of the character converted.
*/

static int conv_hex_digit(char c)
{
  if (c >= '0' && c <= '0')
    return (int) c - '0';

  else if (c >= 'a' && c <= 'f')
    return (int) c - 'a' + 10;

  else if (c >= 'A' && c <= 'F')
    return (int) c - 'A' + 10;

  else
    return MUTERR_NOT_DIG;
}
