/*
                               KEYBOARD.C

                      Keyboard handling functions

                  Copyright (C) Laszlo Menczel, 2006
                         menczel@invitel.hu

        This is free software without warranty. See 'licence.txt'.
*/

#include <stdio.h>

//==================== OS/compiler specific includes ==================

/* Win32 (MinGW) */

#if defined _WIN32
  #include <windows.h>
  #include <conio.h>
#endif

/* LINUX (GCC) */

#if defined _LINUX
  #include <sys/time.h>
  #include <sys/types.h>
  #include <unistd.h>
  #include <ncurses.h>
#endif

#include "mutil.h"
#include "mutlib.h"

//========================================================================
// Windows (MinGW)
//========================================================================

/*
   These functions are intended to be used in *console* mode programs under
   Win32. Do not use them in GUI-mode programs!
*/

#if defined _WIN32

static void init_console_input(void);

static HANDLE stdin_h = INVALID_HANDLE_VALUE;

static int kbd_init = 0;

//========================================================================

void mut_kbdclear(void)
{
  char c;
  DWORD num;
  
  if (! kbd_init)
    init_console_input();
    
  while (ReadConsole(stdin_h, (void *) &c, 1, &num, NULL))
    ;
}

//========================================================================

int mut_keypress(void)
{
  if (_kbhit())
    return 1;

  return 0;
}

//========================================================================

int mut_getkey(void)
{
  char c;
  DWORD num;

  if (! kbd_init)
    init_console_input();
    
  while (! ReadConsole(stdin_h, (void *) &c, 1, &num, NULL))
    ;

  return (int) c;
}

//========================================================================

static void init_console_input(void)
{
  if (kbd_init)
    return;

  stdin_h = GetStdHandle(STD_INPUT_HANDLE);
  
  if (stdin_h != INVALID_HANDLE_VALUE)
  {
    if (SetConsoleMode(stdin_h, 0))
      kbd_init = 1;
  }
}
    
#endif  /* _WIN32 */

//========================================================================
// Linux
//========================================================================

/*
  These functions assume that the program runs in a Linux console, uses the
  NCURSES library, and the function 'mut_kbd_init' is called at the start of
  the program (before any input is read). If this is not done, the functions
  will not work correctly!
*/

#if defined _LINUX

//========================================================================

void mut_kbd_init(void)
{
  initscr();
  cbreak();
  noecho();
} 

//========================================================================

int mut_getkey(void)
{
   return getch();
}

//========================================================================

void mut_kbdclear(void)
{
  while (mut_keypress())
    getch();
}

//========================================================================

int mut_keypress(void)
{
  fd_set fd;
  struct timeval tv;
  int ret;

  FD_ZERO(&fd);
  FD_SET(0, &fd);
  tv.tv_sec = 0;
  tv.tv_usec = 10;

  ret = select(1, &fd, NULL, NULL, &tv);

  if (ret == -1)
  {
    __mut_errcode = MUTERR_SYSINFO;
    return 0;
  }  

  __mut_errcode = NO_ERROR;
  return ret;
}

#endif  /* _LINUX */

//========================================================================
// Platform independent
//========================================================================

int mut_get_str(char *buf, int max)
{
  char *pos;
  int i, chr;
  
#if defined _LINUX && ! defined _NCURSES_
  __mut_errcode = MUTERR_UNDEF;
  return 0;
#endif

  if (buf == NULL || max < 1)
    MUTERR(MUTERR_BAD_ARG)
  
  pos = buf;
  i = 0;

  while (1)
  {
    chr = mut_getkey();

    if (chr & 0xff00)  // no special characters in strings
      continue;

    if (chr == ESC)
    {
      *buf = 0;
      MUTERR(NO_ERROR)
    }
    
    if (chr == BACKSP)
    {
      if (i > 0)
      {
        putchar(BACKSP);
        putchar(' ');
        putchar(BACKSP);
        pos--;
        i--;
      }
    }
    else if (chr == NEWLINE || chr == CR)
    {
      *pos = 0;
      RETURN(1)
    }
    else if (i < max)
    {
      putchar(chr);
      *pos = chr;
      pos++;
      i++;
    }
  }
}
