/*
                               FILE.C

                        File system utilities
                    
                   Copyright (C) Laszlo Menczel, 2005
                         menczel@invitel.hu

        This is free software without warranty. See 'licence.txt'.
*/

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <malloc.h>
#include <errno.h>
#include <time.h>

#if defined _WIN32
  #include <io.h>
#endif

#include "mutil.h"
#include "mutlib.h"

static int is_path_delimiter(char c);
static int do_ignore(int c, char *pattern);

#if defined _WIN32
  static char *win_mut_fname_char = " !@#$%^&*()[]{}=+<>";
#endif

/**************************************************************************/

int mut_file_exist(char *name)
{
  FILE *f;
  
  if (name == NULL || name[0] == 0)
    MUTERR(MUTERR_BAD_ARG)
  
  f = fopen(name, "rt");
  if (f != NULL)
  {
    fclose(f);
    RETURN(1)
  }
  else
    MUTERR(MUTERR_FILE)
}
   
/***************************************************************************/

int mut_file_rename(char *old, char *new)
{
  if (rename(old, new) != 0)
  {
    if (errno == EACCES)
      MUTERR(MUTERR_ACCESS_FILE)
    else if (errno == EBUSY)
      MUTERR(MUTERR_BUSY)
    else if (errno == EROFS)
      MUTERR(MUTERR_READ_ONLY)
    else
      MUTERR(MUTERR_UNDEF)
  }

  RETURN(1)
}

/***************************************************************************/

int mut_file_delete(char *name)
{
  if (unlink(name) != 0)
  {
    if (errno == EACCES)
      MUTERR(MUTERR_ACCESS_FILE)
    else if (errno == EBUSY)
      MUTERR(MUTERR_BUSY)
    else if (errno == EROFS)
      MUTERR(MUTERR_READ_ONLY)
    else
      MUTERR(MUTERR_UNDEF)
  }

  RETURN(1)
}

/***************************************************************************/

int mut_fget_line(char *buf, int len, FILE *f)
{
  char *start;
  int c, count;
  
  if (
       buf == NULL       ||
       f == NULL         ||
       len < 1           ||
       len > MUT_MAX_LINE_LEN
     )
     MUTERR(MUTERR_BAD_ARG)

  start = buf;
  count = 0;

  while (1)
  {
    if (count == len)
    {
      *buf = 0;
      __mut_errcode = MUTERR_PARTIAL_READ;
      return -1;
    }

    c = fgetc(f);

    if (c == EOF)
    {
      if (feof(f))
      {
        *buf = 0;
        __mut_errcode = MUTERR_END_OF_FILE;
      }
      else
      {
        *start = 0;
        __mut_errcode = MUTERR_FILE_READ;
      }

      return 0;
    }

    if (c == '\n')
    {
      *buf = 0;
      if (count == 0)
        MUTERR(MUTERR_EMPTY)
      else
        RETURN(count)
    }
    
    *buf = (char) c;
    buf++;
    count++;
  }
}  

/***************************************************************************/

int mut_fget_line_ex(char *buf, char *ignore, int len, int *count, FILE *f)
{
  int c;
  
  if (
       buf == NULL       ||
       count == NULL     ||
       f == NULL         ||
       len < 1           ||
       len > MUT_MAX_LINE_LEN
     )
     MUTERR(MUTERR_BAD_ARG)

  *buf = 0;
  *count = 0;

  while (1)
  {
    if (*count == len)
    {
      *buf = 0;
      MUTERR(MUTERR_PARTIAL_READ)
    }

    c = fgetc(f);
    if (c == EOF)
    {
      *buf = 0;
      MUTERR(MUTERR_END_OF_FILE)
    }

    if (*count == 0 && do_ignore(c, ignore))
      MUTERR(MUTERR_EMPTY)

    if (c == '\n')
    {
      *buf = 0;
      RETURN(1)
    }
    
    *buf = (char) c;
    buf++;
    (*count)++;
  }
}  

/***************************************************************************/

/*
   Replaces '/' and '\' delimiters so that 'path' conforms to the pathname
   conventions of the currently used operating system. Returns zero if no
   replacement, 1 if delimiters had to be replaced, a negative number in
   case of error. If 'path' is longer than MUT_MAX_PATH_LEN, it is truncated
   and MUTERR_TOO_MANY is returned.
*/

int mut_fname_fix_delim(char *path)
{
  int res, i;
  char bad, good;
  
  if (path == NULL)
    MUTERR(MUTERR_BAD_ARG)

#if defined _WIN32 || defined _DOS
  bad = '/';
  good = '\\';
#endif

#if defined _LINUX
  bad = '\\';
  good = '/';
#endif
  
  res = i = 0;

  while (i < MUT_MAX_NAME_LEN + MUT_MAX_PATH_LEN && path[i] != 0 )
  {
    if (path[i] == bad)
    {
      path[i] = good;
      res = 1;
    }

    i++;
  }
  
  if (i == MUT_MAX_PATH_LEN + MUT_MAX_NAME_LEN)
  {
    path[i - 1] = 0;
    MUTERR(MUTERR_TOO_MANY)
  }

  RETURN(res)
}

/**************************************************************************/

int mut_fname_char(char c)
{
  if (
       (c >= 'a' && c <= 'z') ||
       (c >= 'A' && c <= 'Z') ||
       (c >= '0' && c <= '9') ||
       c == '.'               ||
       c == '_'               ||
       c == '-'
     )
     return 1;
    
#if defined _WIN32
  int i = 0;
  while (win_mut_fname_char[i] != 0)
  {
    if (c == win_mut_fname_char[i])
      return 1;
    i++;
  }
#endif

#if defined _LINUX
  if (c != '/')
    return 1;
#endif

  return 0;
}

/***************************************************************************/

int mut_fname_split(char *s, char *path, char *name)
{
  char *name_part;
  int i, delim;

  i = 0;
  delim = -1;

  while (i < MUT_MAX_PATH_LEN + MUT_MAX_NAME_LEN && s[i] != 0)
  {
    if (is_path_delimiter(s[i]))
      delim = i;
    i++;
  }

  if (
       i == MUT_MAX_PATH_LEN + MUT_MAX_NAME_LEN ||
       delim > MUT_MAX_PATH_LEN - 1         ||
       i - delim > MUT_MAX_NAME_LEN
     )
  {
    if (path != NULL)
      *path = 0;

    if (name != NULL)
      *name = 0;

    MUTERR(MUTERR_TOO_MANY)
  }

  if (name != NULL)
  {
    if (delim < i - 1)
    {
      name_part = &(s[delim + 1]);
      strcpy(name, name_part);
    }
    else
      *name = 0;
  }

  if (path != NULL)
  {
    if (delim > -1)
      strncpy(path, s, delim + 1);
    else
      *path = 0;
  }

  RETURN(1)
}  

/***************************************************************************/

int mut_fname_get_path(char *s, char *path)
{
  return mut_fname_split(s, path, NULL);
}

/***************************************************************************/

int mut_fname_get_name(char *s, char *name)
{
  return mut_fname_split(s, NULL, name);
}

/***************************************************************************/

int mut_fname_check_len(char *s)
{
  int i, delim;

  i = 0;
  delim = 0;

  while (i < MUT_MAX_PATH_LEN + MUT_MAX_NAME_LEN && s[i] != 0)
  {
    if (is_path_delimiter(s[i]))
      delim = i;
    i++;
  }

  if (
       i == MUT_MAX_PATH_LEN + MUT_MAX_NAME_LEN ||
       delim > MUT_MAX_PATH_LEN - 1         ||
       i - delim > MUT_MAX_NAME_LEN - 1
     )
     return 0;

  return 1;
}

/***************************************************************************/

int mut_fname_check_str(char *s)
{
  int i;

  i = 0;
  while (i < MUT_MAX_PATH_LEN + MUT_MAX_NAME_LEN - 1)
  {
    if (mut_fname_char(s[i]) || is_path_delimiter(s[i]))
      i++;
    else
      return 0;
  }
    
  return 1;
}

/***************************************************************************/

int mut_fname_change_ext(char *name, char *ext)
{
  char *pos;

  if (name == NULL || name[0] == 0 || ext == NULL || ext[0] == 0)
    MUTERR(MUTERR_BAD_ARG)

  pos = mut_str_end(name);
    
  while (1)
  {
    if (pos == name)      // returned to start of string, no extension
    {
      strcat(name, ".");  // just append
      strcat(name, ext);
      RETURN(1)
    }
    else if (*pos == '.')
    {
      pos++;
      strcpy(pos, ext);
      RETURN(1)
    }
    else
      pos--;
  }
}

/*****************************************************************************/

int mut_file_line_count(char *name, char *ignore)
{
  FILE *f;
  int count;
  char line[MUT_MAX_LINE_LEN + 1];

  if (name == NULL || name[0] == 0)
    MUTERR(MUTERR_BAD_ARG)

  f = fopen(name, "rt");

  if (f == NULL)
    MUTERR(MUTERR_FILE)
 
  count = 0;
  while (fgets(line, MUT_MAX_LINE_LEN, f) != NULL)
  {
    if (ignore != NULL && do_ignore(line[0], ignore))
      continue;
    else
      count++;
  }

  fclose(f);

  RETURN(count)
}

/***************************************************************************/
/*************************** local functions *******************************/
/***************************************************************************/

static int is_path_delimiter(char c)
{
#if defined _WIN32 || defined _DOS
  if (c == ':' || c == '\\')
    return 1;
#endif

#if defined _LINUX
  if (c == '/')
    return 1;
#endif

  return 0;
}

/***************************************************************************/

static int do_ignore(int c, char *pattern)
{
  int i;

  i = 0;
  while (pattern[i] != 0)
  {
    if (c == pattern[i])
      return 1;
    i++;
  }

  return 0;
}
