/*
 *   Copyright 1999 Bent Bagger (OZ6BL)
 *              All Rights Reserved
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 1, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
        download.c

        Determines whether or not a file should be downloaded from the
        satellite
*/

#define ID_STRING "$Id: download.c,v 1.3 2007/12/30 15:44:13 oz6bl Exp $"
#define VERSION_STRING "(version 1.1 by oz6bl)"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "download.h"
#include "header.h"

#define FILES 2

LOADLIST *autoload_list = NULL;
FILE     *autof;
 
/* 'ThisFileShouldBeDownloaded' is called from pb.c whenever a new complete
   directory entry is received from the satellite. 
  
   Entry: HEADER *hdr points to a structure containg the decoded file header 
          items. THis structure is defined in 'header.h'.

   Exit: return value = 0 ('false') - this file should not be downloaded
                        -1 ('true') - this file should be downloaded

   Note that the download management is done purely by pb.c

*/


int ThisFileShouldBeDownloaded(HEADER *hdr) {
    LOADLIST *ll ;
    char fileName[80];
    struct stat status;
    ll = autoload_list;
        for (;;ll = ll->next) {
            if (ll == NULL) break;
            if (ll->dest[0] != '\0' && ! strcasestr (hdr->destination, ll->dest)) continue;
            if (ll->src [0] != '\0' && ! strcasestr (hdr->source,      ll->src))  continue;
            if (! strcasestr (hdr->title,       ll->title)) continue;
            if (! strcasestr (hdr->keyWords,    ll->key))   continue;
            if (hdr->fileSize >             ll->size)   continue;
	    /* Check to see if this file has been downloaded already
	     * If it has then don't download it again - unless the size of the file
	     * has changed, in which case we delete the downloaded files and request it again.
	     * This may be the situation with the satellite-generated log files (AL, BL, EL, etc)
	     * which grow during the day */
	    sprintf(fileName, "%x.dl", hdr->fileId); // create the path of the (possibly) downloaded file
	    if (stat(fileName, &status) != 0) // not downloaded (does not exist) - request download
            	return (-1) ;
	    else { // downloaded - compare files sizes
		if (hdr->fileSize != status.st_size) { // file sizes differ, delete existing
			if (unlink(fileName) == 0) { // deleted - request download
				return (-1);
			} else { // something went astray, leave it alone
				return (0);
			}
		}
		return (0); // same file sizes
	    }
        }
    return (0) ;
}

/* 'InitializeDownload' is called at program start to perform the necessary 
   setup activities.

   It may be called during program execution to re-initialize 

*/

void InitializeDownload(void) {

if ((autof = fopen ("autoload.dat", "r")) != NULL) {
	LOADLIST *ll = NULL;
	for (;;) {
		if (feof (autof)) break;
		autoload_list = malloc (sizeof (LOADLIST));
		autoload_list->next = ll;
		ll = autoload_list;
		fscanf (autof, "%s %s %s %s %d\n", ll->dest, ll->src,
				ll->title, ll->key, &ll->size);
		if (ll->dest  [0] == '*') ll->dest  [0] = '\0';
		if (ll->src   [0] == '*') ll->src   [0] = '\0';
		if (ll->title [0] == '*') ll->title [0] = '\0';
		if (ll->key   [0] == '*') ll->key   [0] = '\0';
		if (ll->size      == 0)   ll->size      = 0xFFFFFFFF;
	}
	fclose (autof);
	writelog (FILES, "Download control re-initialized");
	

}
} 
