/* $Id: env.c,v 1.4 2000/05/08 11:50:30 malekith Exp $ */

#include "h.h"

struct evar {
	struct evar *next;
	char *name;
	char *val;	/* == 0 if unset */
	int size;	/* space allocated for name + val */
	int flags;
};

static struct evar *ehead;

static struct evar *find_env(const char *n)
{
	struct evar *p;
	
	if (!n || !*n)
		return 0;
		
	for (p = ehead; p; p = p->next)
		if (strcmp(p->name, n) == 0)
			return p;
	return 0;
}

char *getenv(const char *n)
{
	struct evar *p;
	
	p = find_env(n);
	return p ? p->val : 0;
}

extern long sec_start;
extern char *ifs;

static int magic_env(const char *n, int nl, const char *v)
{
	switch (nl) {
	case 3:
		if (strncmp(n, "IFS", 3) == 0)
			ifs = pstrdup(v);
		goto def;
	case 6:
		if (strncmp(n, "RANDOM", 6) == 0)
			srand(v ? atoi(v) : 0);
		goto def;
	case 7:
		if (strncmp(n, "SECONDS", 7) == 0)
			sec_start = xtime() - (v ? atoi(v) : 0);
		goto def;
	default:
	def:
		return 0;
	}
	return 1;
}

static void addenv(const char *n, int nl, const char *v, int flags)
{
	struct evar *p, *vnew = 0;
	int vl, dl = -1;
	
	if (!n || !*n)
		return; /* cannot do that */
		
	if (magic_env(n, nl, v))
		return;
		
	vl = nl + 1 + (v ? strlen(v) : 0) + 1;

	for (p = ehead; p; p = p->next) {
		int ndl;
		
		if (strncmp(p->name, n, nl) == 0 && p->name[nl] == 0) 
			/* found, unset */
			*p->name = 0;
		
		if (*p->name == 0 && ((ndl = p->size - vl) >= 0) && 
			 (dl == -1 || ndl < dl)) {
			vnew = p;
			dl = ndl;
		}
	}
	
	/* only unset ? */
	if (v == 0)
		return;
	
	if (vnew == 0) { /* need to allocate more space */
		int size = 16;
		
		vnew = palloc(sizeof(struct evar));
		/* alloc more space in case of enlerging data (i.e.
		 * ``while true ; do v="${v}x" ; done'' -- be able to eat
		 * our memory faster ;) */
		while (size < vl)
			size <<= 1;
		vnew->name = palloc(size);
		vnew->size = size;
		/* link our new node */
		vnew->next = ehead;
		ehead = vnew;
	}
	
	vnew->flags = flags;
	
	/* val has to be exactly one char past end of name, to allow 
	 * changing this into name=val with one assigment */
	strncpy(vnew->name, n, nl); 
	vnew->name[nl] = 0;
	
	vnew->val = vnew->name + nl + 1;
	strcpy(vnew->val, v);
}

void putenv_f(const char *n, const char *v, int f)
{
	addenv(n, strlen(n), v, f);
}

/* use putenv("VAR", 0) to unset */
void putenv(const char *n, const char *v)
{
	addenv(n, strlen(n), v, ENV_EXPORT);
}

/* add string NAME=value */
void putenv_eq(const char *s, int flags)
{
	const char *v;
	int len;
	
	if (!s)
		return;
	v = strchr(s, '=');
	
	if (v)
		len = v++ - s;
	else {
		len = strlen(s);
		v = "";
	}
		
	addenv(s, len, v, flags);
}

char **buildenv(void)
{
	char **ret, **e;
	struct evar *p;
	
	ret = e = alloc(sizeof(char*));
	
	/* first add ehead */
	for (p = ehead; p; p = p->next)	{
		if (*p->name && (p->flags & (ENV_EXPORT|ENV_EXPORT_ONCE))) {
			*strchr(p->name, 0) = '=';
			*e++ = p->name;
			alloc(sizeof(char*));
		}
	}
	*e = 0;
	return ret;
}

extern char *xversion, *xdate, *ifs;

void setupenv(void)
{
	char **p;
	p = environ;
	
	while (*p)
		putenv_eq(*p++, ENV_EXPORT);
		
	putenv_f("IFS", ifs, 0);
	putenv_f("TRASH_VERSION", sprintf("trash v%s %s", xversion, xdate), 0);
}

void clear_export_once(void)
{
	struct evar *p;
	
	for (p = ehead; p; p = p->next)
		p->flags &= ~ENV_EXPORT_ONCE;
}

void delete_export_once(void)
{
	struct evar *p;
	
	for (p = ehead; p; p = p->next)
		if (p->flags&ENV_EXPORT_ONCE)
			*p->name = 0;
}

void set_env_flags(const char *n, int f)
{
	struct evar *p;
	
	p = find_env(n);
	
	if (p)
		p->flags = f;
	else
		addenv(n, strlen(n), "", f);
}

int get_env_flags(const char *n)
{
	struct evar *p;
	
	p = find_env(n);
	
	return p ? p->flags : -1;
}

void printenv(int flags)
{
	struct evar *p;
	for (p = ehead; p; p = p->next)
		if (*p->name && (p->flags & flags))
			printf("%s=%s\n", p->name, p->val);
}

void unsetall()
{
	struct evar *p;
	for (p = ehead; p; p = p->next)
		*p->name = 0;
}
