/*
 * tcpd_tools.c
 *
 * 		This program is free software; you can redistribute it and/or
 * 		modify it under the terms of the BSD style license (see
 * 		COPYING file included with this software).
 * 		
 * Authors:	Kazunori Fujiwara <fujiwara@rcac.tdi.co.jp>
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "rcsid.h"
RCSID("$Id: tcpd_tools.c,v 1.7 2000/10/01 21:19:57 baggins Exp $")

#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include "tcpd.h"
#include "tcpd_local.h"

int tcpd_separator(char **p)
{
	SPACESKIP(*p);
	if (**p != SEPARATOR)
		return 0;
	(*p)++;
	SPACESKIP(*p);
	return 1;
}

void tcpd_to_nextseparator(char **p)
{
	char *q = *p;

	while (*q != SEPARATOR && *q != '\0')
		q++;
	if (*q == SEPARATOR)
		q++;
	*p = q;
}

int tcpd_is_separator(int c)
{
	if (c == '\0')
		return 1;
	if (isalnum(c) || c == '_' || c == '.' || c == '-' || c == ':')
		return 0;
	return 1;
}

int tcpd_is_wordsep(int c)
{
	return c == '\0' || c == ' ' || c == ',' || c == '\t';
}

int tcpd_word(char *str, const char *keyword, char **left)
{
	int len;

	if (left != NULL)
		*left = str;
	len = strlen(keyword);
	if (len == 0 || *str == '\0')
		return 0;
	if (strncasecmp(str, keyword, len))
		return 0;
	str += len;
	if (!tcpd_is_separator(*str))
		return 0;
	if (left != NULL)
		*left = str;
	return 1;
}

/*
	word check (word+space)
	space == ' ', ',', '\t', '\0'
*/

int tcpd_word_ws(char *str, const char *keyword, char **left)
{
	int len;

	if (left != NULL)
		*left = str;
	len = strlen(keyword);
	if (len == 0 || *str == '\0')
		return 0;
	if (strncasecmp(str, keyword, len))
		return 0;
	str += len;
	if (!tcpd_is_wordsep(*str))
		return 0;
	if (left != NULL)
		*left = str;
	return 1;
}

int tcpd_pstrcmp(char *str, const char *keyword, char **left)
{
	int len;

	if (left != NULL)
		*left = str;
	len = strlen(keyword);
	if (len == 0 || *str == '\0')
		return 0;
	if (strncasecmp(str, keyword, len))
		return 0;
	str += len;
	if (left != NULL)
		*left = str;
	return 1;
}


/*
	get token from string (str)
	no space skip
	return zero when no token.
	return token length
	token = [0-9a-z-._:]+
*/

int tcpd_get_token(char *str, char **left, char *buf, int buflen)
{
	char *p, *q;
	int len;

	buflen--;
	for (p = str, q = buf, len = 0;
		!tcpd_is_separator(*p) && len <= buflen;
		*q++ = *p++, len++);
	*q = '\0';
	*left = p;
	return len;
}

static const unsigned char prefixlen2mask_table[7] = { 0x80, 0xc0, 0xe0, 0xf0, 0xf8, 0xfc, 0xfe };

void prefixlen2mask(int prefixlen, unsigned char *dest, int size)
{
	int i;

	if (prefixlen < 0)
		bzero(dest, size);
	else
	for (i = 0; i < size; i++, dest++, prefixlen -= 8)
		if (prefixlen >= 8)
			*dest = 0xff;
		else if (prefixlen > 0)
			*dest = prefixlen2mask_table[prefixlen-1];
		else
			*dest = 0;
}

/*
	get token from string (str)
	space is skipped
	return zero when no token.
	return token length
	token = [0-9a-z-._:]+
*/

int tcpd_get_token2(char *str, char **left, char *buf, int buflen)
{
	char *p, *q;
	int len;

	SPACESKIP(str);
	buflen--;
	if (*str == '\'') {
		str++;
		for (p = str, q = buf, len = 0;
			*p != '\0' && *p != '\'' && len < buflen;
			*q++ = *p++, len++);
		if (*p == '\'')
			p++;
	} else {
		for (p = str, q = buf, len = 0;
			!tcpd_is_wordsep(*p) && len <= buflen;
			*q++ = *p++, len++);
	}
	*q = '\0';
	*left = p;
	return len;
}


/*
   int tcpd_parse_options(char *str, char **left, struct _code *flags)

      parse flags and options
 */

int tcpd_parse_options(char *str, char **left, struct _code *flags)
{
  int i, u, sign;
  char buf1[TCPD_STRINGLENL];
  char buf2[TCPD_STRINGLENL];

  SPACESKIP(str);
  if (*str != SEPARATOR && *str != '\0')
    for ( ; ; ) {
      for (i = 0; flags[i].tag != NULL; i++) {
        if (tcpd_word_ws(str, flags[i].tag, &str)) {
	  SPACESKIP(str);
          switch(flags[i].type) {
            case T_UINTEGER:
	      if (!isdigit(*str)) {
		*left = str;
	        return TCPD_ERROR;
	      }
	      *(int *)flags[i].ptr = strtoul(str, &str, 10);
	      break;
            case T_INTEGER:
	      if (*str == '-') {
	        sign = 1;
	        str++;
	      } else {
	        sign = 0;
	      }
	      if (!isdigit(*str)) {
		*left = str;
	        return TCPD_ERROR;
	      }
	      u = strtoul(str, &str, 10);
	      *(int *)flags[i].ptr = sign ? -u : u;
	      break;
            case T_SET1:
	      *(int *)flags[i].ptr = 1;
	      break;
            case T_SET0:
	      *(int *)flags[i].ptr = 0;
	      break;
	    case T_STR:
	      tcpd_get_token2(str, &str, flags[i].ptr, flags[i].len);
	      break;
	    case T_FUNC_STR1:
	      tcpd_get_token2(str, &str, buf1, sizeof(buf1));
	      if (((int (*)(char *))flags[i].ptr)(buf1) == 0) {
		*left = str;
	        return TCPD_ERROR;
	      }
	      break;
	    case T_FUNC_STR2:
	      tcpd_get_token2(str, &str, buf1, sizeof(buf1));
	      SPACESKIP(str);
	      tcpd_get_token2(str, &str, buf2, sizeof(buf2));
	      if (((int (*)(char *, char *))flags[i].ptr)(buf1, buf2) == 0) {
		*left = str;
	        return TCPD_ERROR;
	      }
	      break;
            case T_FUNC_UINT1:
	      if (!isdigit(*str)) {
		*left = str;
	        return TCPD_ERROR;
	      }
	      u = strtoul(str, &str, 10);
	      if (((int (*)(int))flags[i].ptr)(u) == 0) {
		*left = str;
	        return TCPD_ERROR;
	      }
	      break;
            case T_FUNC_SET0:
	      if (((int (*)(int))flags[i].ptr)(0) == 0) {
		*left = str;
	        return TCPD_ERROR;
	      }
	      break;
            case T_FUNC_SET1:
	      if (((int (*)(int))flags[i].ptr)(1) == 0) {
		*left = str;
	        return TCPD_ERROR;
	      }
	      break;
	    default:
	      *left = str;
	      return TCPD_ERROR;
	  }
	  SPACESKIP(str);
	  break;
        }
      }
      if (flags[i].tag == NULL) {
	*left = str;
        return TCPD_ERROR;
      }
      if (*str == ',') {
        str++;
        SPACESKIP(str);
      } else
      if (*str == SEPARATOR || *str == '\0')
        break;
      else {
	*left = str;
        return TCPD_ERROR;
      }
    }
  *left = str;

  return TCPD_NOERROR;
}

char *tcpd_lineno(struct request_info *req)
{
	static char buff[TCPD_STRINGLEN];

	if (req->lineno > 0)
		snprintf(buff, sizeof(buff), "at line %d", req->lineno);
	else if (req->lineno == -1)
		snprintf(buff, sizeof(buff), "by default configuration");
	else
		snprintf(buff, sizeof(buff), "at unknown line (%d?)", req->lineno);
	return (buff);
}

