/*
 * libwrap.c
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the BSD style license (see
 *		COPYING file included with this software).
 *
 * Authors:     Kazunori Fujiwara <fujiwara@rcac.tdi.co.jp>
 *                  
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "rcsid.h"
RCSID(PKG_VER "$Id: libwrap.c,v 1.17 2000/10/01 21:19:57 baggins Exp $")

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>
#include <errno.h>
#include <ctype.h>
#include <string.h>
#include <limits.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <syslog.h>

#include "tcpd.h"
#include "tcpd_local.h"

struct __service_options {
	int refuse_wait;
	int rfc1413_timeout;
	int allow_severity;
	int deny_severity;
	int ident;
	int defaultrule;
	int ignore_src_dns_fake;
	int ignore_dst_dns_fake;
};

static char access_table[TCPD_STRINGLENL] = "/etc/hosts.access";
static int userresolvreq;
static int defaultallow;
static int ignore_src_dns_fake;
static int ignore_dst_dns_fake;

static void strtosa(char *str, struct sockaddr_storage *sa)
{
	struct addrinfo hints, *res, *res0;
	
	memset(sa, 0, sizeof(*sa));
	memset(&hints, 0, sizeof(hints));
	hints.ai_family = AF_UNSPEC;
	hints.ai_socktype = SOCK_DGRAM;
	hints.ai_flags = AI_NUMERICHOST;
	if (!getaddrinfo(str, NULL, &hints, &res0)) {
		for (res = res0; res; res = res->ai_next) {
			if ((res->ai_family != AF_INET)
					&& (res->ai_family != AF_INET6))
				continue;
			memcpy(sa, res->ai_addr, res->ai_addrlen);
			sa->__ss_family = res->ai_family;
			break;
		}
		freeaddrinfo(res0);
	}
}

#ifdef IN6_IS_ADDR_V4MAPPED
static void mappedtov4(struct sockaddr_storage *ss)
{
	struct sockaddr_in sin;
	struct sockaddr_in6 *sin6 = (struct sockaddr_in6 *)ss;

	if (ss->__ss_family == AF_INET6 &&
			IN6_IS_ADDR_V4MAPPED(&sin6->sin6_addr) ) {
		memcpy(&sin.sin_addr, sin6->sin6_addr.s6_addr+12,
				sizeof(sin.sin_addr));
		sin.sin_port = ((struct sockaddr_in6 *)ss)->sin6_port;
		sin.sin_family = AF_INET;
#ifdef SIN6_LEN
		sin.sin_len = 16;
#endif
		memcpy(ss, &sin, sizeof(sin));
	}
}
#else
#define        mappedtov4(A)
#endif
		
static struct request_info *request_fill(struct request_info *req, va_list ap)
{
	struct sockaddr_storage *ss;
	int cmd;

	do {
		cmd = va_arg(ap, int);
		switch(cmd) {
		case RQ_FILE:
			req->sock = va_arg(ap, int);
			break;
		case RQ_DAEMON:
			strncpy(req->service, va_arg(ap, char *), sizeof(req->service));
			break;
		case RQ_USER:
			strncpy(req->user, va_arg(ap, void *), sizeof(req->user));
			break;
		case RQ_SERVER_SIN:
			ss = va_arg(ap, struct sockaddr_storage *);
			if (ss)
				memcpy(&req->dst, ss, SA_LEN((struct sockaddr *)ss));
			break;
		case RQ_CLIENT_SIN:
			ss = va_arg(ap, struct sockaddr_storage *);
			if (ss)
				memcpy(&req->src, ss, SA_LEN((struct sockaddr *)ss));
			break;
		case RQ_SERVER_NAME:
			strncpy(req->dst_name, va_arg(ap, void *), sizeof(req->dst_name));
			break;
		case RQ_CLIENT_NAME:
			strncpy(req->src_name, va_arg(ap, void *), sizeof(req->src_name));
			break;
		case RQ_SERVER_ADDR:
			strtosa(va_arg(ap, void *), &req->dst);
			break;
		case RQ_CLIENT_ADDR:
			strtosa(va_arg(ap, void *), &req->src);
			break;
		case RQ_SETTABLE:
			strncpy(access_table, va_arg(ap, void *), sizeof(access_table));
			break;
		case RQ_END:
			break;
		default:
			cmd = RQ_END;
			break;
		}
	} while(cmd != RQ_END);
	return (req);
}

struct request_info *request_init(struct request_info *req, ...)
{
	struct request_info *r;
	va_list ap;

	va_start(ap, req);

	userresolvreq = 1;
	defaultallow = TCPD_DENY;
	ignore_src_dns_fake = 0;
	ignore_dst_dns_fake = 0;

	memset(req, 0, sizeof(struct request_info));
	req->sock = -1;
	req->userresolved = 0;
	req->src.__ss_family = -1;
	req->dst.__ss_family = -1;
	allow_severity = deny_severity = LOG_DAEMON | LOG_NOTICE;
	req->udp = 0;
	req->refuse_wait = 5;
	req->src_wrongdns = 0;
	req->dst_wrongdns = 0;
	req->rfc1413_timeout = TCPD_DEFAULT_RFC1413_TIMEOUT;
	r = request_fill(req, ap);
	va_end(ap);
	return (r);
}

struct request_info *request_set(struct request_info *req, ...)
{
	struct request_info *r;
	va_list ap;

	va_start(ap, req);
	r = request_fill(req, ap);
	va_end(ap);
	return(r);
}

static int gethostaddr(struct sockaddr_storage *sa, char *str, int slen)
{
	struct addrinfo hints, *res, *res0 = NULL;
	char hbuf[NI_MAXHOST], hipbuf[NI_MAXSERV];

	str[0] = '\0';
	if (getnameinfo((struct sockaddr *)sa, SA_LEN((struct sockaddr *)sa),
			hbuf, sizeof(hbuf), NULL, 0, NI_NAMEREQD))
		return 1;
	if (getnameinfo((struct sockaddr *)sa, SA_LEN((struct sockaddr *)sa),
			hipbuf, sizeof(hipbuf), NULL, 0, NI_NUMERICHOST))
		return 1;
	if (strlen(hbuf) >= slen)
		return -1;
	hbuf[sizeof(hbuf) - 1] = '\0';
	memset(&hints, 0, sizeof(hints));
	hints.ai_family = AF_UNSPEC;
	hints.ai_socktype = SOCK_DGRAM;
	if (getaddrinfo(hbuf, NULL, &hints, &res0))
		return -1;
	for (res = res0; res; res = res->ai_next) {
		char hipbuf2[NI_MAXHOST];
		if (!getnameinfo(res->ai_addr, res->ai_addrlen, hipbuf2,
				sizeof(hipbuf2), NULL, 0, NI_NUMERICHOST)
				&& (!strcmp(hipbuf, hipbuf2))) {
			strcpy(str, hbuf);
			freeaddrinfo(res0);
			return 0;
		}
	}
	freeaddrinfo(res0);
	
	return -1;
}

void fromhost(struct request_info *req)
{
	int salen;

	salen = sizeof(req->src);
	if (getpeername(req->sock, (struct sockaddr *)&req->src, &salen) < 0) {
		char buf[BUFSIZ];
		req->udp = 1;
		salen = sizeof(req->src);
		if (recvfrom(req->sock, buf, sizeof(buf), MSG_PEEK, (struct sockaddr *)&req->src, &salen) < 0) {
			return;
		}
	}
	salen = sizeof(req->dst);
	if (getsockname(req->sock, (struct sockaddr *)&req->dst, &salen) < 0)
		return;
	mappedtov4(&req->dst);
	mappedtov4(&req->src);
	if (gethostaddr(&req->src, req->src_name, sizeof(req->src_name)) == -1)
		req->src_wrongdns = 1;
	if (gethostaddr(&req->dst, req->dst_name, sizeof(req->dst_name)) == -1)
		req->dst_wrongdns = 1;
}

/*
    char *tcpd_resolv_user(struct request_info *&req, int force)

	get requrest's username
		parameter: force = 1 ... force resolv user
		           force = 0 ... depends on req->userreqolvreq
		returns:   NULL ... No username
			   username (char *)
*/

static char *tcpd_resolv_user(struct request_info *req, int force)
{
	if ((userresolvreq || force) && (!req->userresolved)) {
		rfc1413(&req->dst, &req->src, req->user, sizeof(req->user), req->rfc1413_timeout);
		req->userresolved = 1;
	}
	return *req->user == '\0' ? NULL : req->user;
}

/*
	Does struct sockaddr_storage *sa match addr/mask ?
 */

#ifdef DEBUG
static void dump_binary(char *msg, void *data, size_t len)
{
	unsigned char *p = data;

	printf("%s [ ", msg);
	while (len-- > 0)
		printf("%02x ", *p++);
	printf("]\n");
}
#endif
	
static int match_ipaddress(struct sockaddr_storage *sa, char *addr, int prefixlen)
{
	unsigned char inaddr_sa[16], inaddr[16], mask[16];
	int size, i;

	if (!inet_pton(sa->__ss_family, addr, &inaddr))
		return TCPD_NOMATCH;
	switch(sa->__ss_family) {
	case AF_INET:
		size = 4;
		memcpy(inaddr_sa, &((struct sockaddr_in *)sa)->sin_addr, size);
		break;
	case AF_INET6:
		size = 16;
		memcpy(inaddr_sa, &((struct sockaddr_in6 *)sa)->sin6_addr, size);
		break;
	default:
		return TCPD_NOMATCH;
	}
	if (prefixlen >= 0) {
		prefixlen2mask(prefixlen, mask, size);
		for (i = 0; i < size; i++) {
			inaddr_sa[i] &= mask[i];
			inaddr[i] &= mask[i];
		}
	}
	return memcmp(inaddr_sa, inaddr, size) ? TCPD_NOMATCH : TCPD_MATCH;
}

/*
	compare req's userhost and user/host/mask/ipaddress
*/

static int match_host_part(struct request_info *req, char *user, char *host, int mask)
{
	char *p;

	if (user[0] != '\0' &&
	     strcmp(tcpd_resolv_user(req, 1), user))
		return TCPD_NOMATCH;
	if (mask == -1) {
		if (!strcasecmp(host, "ALL"))
			return TCPD_MATCH;
		if (!strcasecmp(host, "KNOWN"))
			return *req->src_name != '\0';
		if (!strcasecmp(host, req->src_name))
			return TCPD_MATCH;
		if (*host == '.') {
			p = req->src_name;
			while((p = strchr(++p, '.')) != NULL)
				if (!strcasecmp(p, host))
					return TCPD_MATCH;
			return TCPD_NOMATCH;
		}
	}
	return match_ipaddress(&req->src, host, mask);
}

static int service_match(char *str, char *service, struct request_info *req, char **left)
{
	char *p;
	char host[TCPD_HOSTNAMELEN];

	if (!tcpd_word(str, service, &p))
		return TCPD_NOMATCH;
	if (*p == '@') {
		p++;
		if (tcpd_pstrcmp(p, "inet6#", &p)) {
			if (req->dst.__ss_family != AF_INET6)
				return TCPD_NOMATCH;
		} else
		if (tcpd_pstrcmp(p, "inet#", &p)) {
			if (req->dst.__ss_family != AF_INET)
				return TCPD_NOMATCH;
		}
		if (tcpd_word(p, req->dst_name, &p)) {
			*left = p;
			return TCPD_MATCH;
		}
		if (tcpd_get_token(p, &p, host, sizeof(host)) == 0)
			return TCPD_NOMATCH;
		switch(match_ipaddress(&req->dst, host, -1)) {
		case TCPD_MATCH:
			*left = p;
			return TCPD_MATCH;
		default:
			return TCPD_NOMATCH;
		}
	}
	if (!tcpd_is_separator(*p))
		return TCPD_NOMATCH;
	*left = p;
	return TCPD_MATCH;
}

/*
	str ... HOST | options | misc
*/

static int host_match(struct request_info *req, char *strp, char **left)
{
	int mask;
	int ret;
	int af;
	char host[TCPD_HOSTNAMELEN];
	char user[TCPD_HOSTNAMELEN];

	do {
		if (!tcpd_get_token(strp, &strp, host, sizeof(host)))
			return TCPD_ERROR;
		mask = -1;
		af = AF_UNSPEC;
		if (*strp == '@') {
			strncpy(user, host, sizeof(user));
			strp++;
			if (!tcpd_get_token(strp, &strp, host, sizeof(host)))
				return TCPD_ERROR;
		} else {
			*user = '\0';
		}
		if (*strp == '#') {
			if (!strcasecmp(host, "inet")) {
				af = AF_INET;
			} else
			if (!strcasecmp(host, "inet6")) {
				af = AF_INET6;
			} else
				return TCPD_ERROR;
			strp++;
			if (!tcpd_get_token(strp, &strp, host, sizeof(host)))
				return TCPD_ERROR;
		}
		if (*strp == '/') {
			strp++;
			if (!isdigit(*strp))
				return TCPD_ERROR;
			mask = strtoul(strp, &strp, 10);
		}
		SPACESKIP(strp);
		switch(*strp) {
		case ',':
			strp++;
		case '\0':
		case SEPARATOR:
			break;
		default:
			return TCPD_ERROR;
		}

		/* compare req and pattern */
		/* user@host/mask */
		/* if match ... retrn mode */
		if (af == AF_UNSPEC ||
		     (af == AF_INET && req->src.__ss_family == AF_INET)
		      || (af == AF_INET6 && req->src.__ss_family == AF_INET6)
			) {
			switch(ret = match_host_part(req,user,host,mask)) {
			case TCPD_MATCH:
				*left = strp;
				return TCPD_MATCH;
				break;
			case TCPD_NOMATCH:
				break;
			case TCPD_ERROR:
			default:
				return TCPD_ERROR;
			}
		}
		SPACESKIP(strp);
		if (*strp == ',')
			strp++;
		SPACESKIP(strp);
	} while (*strp != SEPARATOR && *strp != '\0');
	return TCPD_CONTINUE;
}

/*
   int tcpd_parse_option(struct request_info *req, char *str, int mode)

      parse flags and options to 'req'
	parameter:  mode = 0 ... default set mode
	            mode = 1 ... append mode
 */

static struct request_info *sreq;
static struct __service_options ALL_service = {-1, -1, -1, -1, -1, -1, -1};
static struct __service_options current_service = {-1, -1, -1, -1, -1, -1, -1};
static struct __service_options *sservicep = &ALL_service;

static int set_syslogallow(char *s)
{
	int i = tcpd_parse_syslog(s, NULL);

	if (i >= 0) {
		sservicep->allow_severity = i;
		return 1;
	}
	return 0;
}

static int set_syslogdeny(char *s)
{
	int i = tcpd_parse_syslog(s, NULL);

	if (i >= 0) {
		sservicep->deny_severity = i;
		return 1;
	}
	return 0;
}

static int set_syslog(char *s)
{
	int i = tcpd_parse_syslog(s, NULL);

	if (i >= 0) {
		sservicep->allow_severity = sservicep->deny_severity = i;
		return 1;
	}
	return 0;
}

static int set_refuse_wait(int val) { sservicep->refuse_wait = val; return 1; }
static int set_rfc1413_timeout(int val) { sservicep->rfc1413_timeout = val; return 1; }
static int set_ident(int val) { sservicep->ident = val; return 1; }
static int set_defaultrule(int val) { sservicep->defaultrule = val; return 1; }
static int set_dns(int val) { sservicep->ignore_src_dns_fake = sservicep->ignore_dst_dns_fake = val; return 1; }
static int set_srcdns(int val) { sservicep->ignore_src_dns_fake = val; return 1; }
static int set_dstdns(int val) { sservicep->ignore_dst_dns_fake = val; return 1; }

struct _code flagsheader[] = {
	{ "syslogallow", T_FUNC_STR1, &set_syslogallow},
	{ "syslogdeny", T_FUNC_STR1, &set_syslogdeny},
	{ "syslog", T_FUNC_STR1, &set_syslog},
	{ "defaultallow", T_FUNC_SET1, &set_defaultrule },
	{ "defaultdeny", T_FUNC_SET0, &set_defaultrule },
	{ "refuse_wait", T_FUNC_UINT1, &set_refuse_wait },
	{ "rfc1413_timeout", T_FUNC_UINT1, &set_rfc1413_timeout },
	{ "ignorednsfake", T_FUNC_SET1, &set_dns },
	{ "ignoresrcdnsfake", T_FUNC_SET1, &set_srcdns },
	{ "ignoredstdnsfake", T_FUNC_SET1, &set_dstdns },
	{ "denydnsfake", T_FUNC_SET0, &set_dns },
	{ "denysrcdnsfake", T_FUNC_SET0, &set_srcdns },
	{ "denydstdnsfake", T_FUNC_SET0, &set_dstdns },
	{ "ident", T_FUNC_SET1, &set_ident },
	{ "noident", T_FUNC_SET0, &set_ident },
	{ NULL }
};


static int tcpd_parse_option(struct request_info *req, char *str, int mode)
{
	sreq = req;

	if (tcpd_parse_options(str, &str, flagsheader) == TCPD_ERROR)
		return TCPD_ERROR;
	if (*str == SEPARATOR) {
		str++;
		SPACESKIP(str);
		if (strlen(str) > 0) {
			if (mode == 0)
				strncpy(req->commands_default, str, sizeof(req->commands_default));
			else
				strncpy(req->commands, str, sizeof(req->commands));
		} else {
			if (mode == 0)
				*req->commands_default = '\0';
			else
				*req->commands = '\0';
		}
	}
	return TCPD_MATCH;
}

static int hosts_access0(FILE *fp, struct request_info *req)
{
	char buff[TCPD_BUFLEN];
	char *p, *q, *r, *service;
#define SERVICE_ALL 2
#define SERVICE_IN  1
#define SERVICE_OTHER 0
	int in_service = SERVICE_ALL;
	char service_name[TCPD_STRINGLEN];
	int result = defaultallow;

	*service_name = '\0';
	while(fgets(buff, sizeof(buff), fp) != NULL) {
		req->lineno++;
		p = buff + strlen(buff) -1;
		if (*p == '\n') *p = '\0';
		p = buff;
		SPACESKIP(p);
		if (*p == '#') /* comment */
			continue;
		if (*p == '\0')
			continue;
		if (tcpd_word(p, "ALLOW", &q)) {
			if (in_service != SERVICE_OTHER) {
				tcpd_separator(&q);
				switch(host_match(req, q, &r)) {
				case TCPD_MATCH:
					tcpd_to_nextseparator(&r);
					if (tcpd_parse_option(req, r, 1) == TCPD_ERROR)
						return TCPD_ERROR; /* parse error */
					tcpd_to_nextseparator(&r);
					if (tcpd_exec_option(req) == TCPD_ERROR)
						return TCPD_ERROR; /* parse error */
					result = TCPD_ALLOW;
				case TCPD_CONTINUE:
					continue;
				case TCPD_ERROR:
				default:
					return TCPD_ERROR;
				}
			}
			continue;
		}
		if (tcpd_word(p, "DENY", &q)) {
			if (in_service != SERVICE_OTHER) {
				tcpd_separator(&q);
				switch(host_match(req, q, &r)) {
				case TCPD_MATCH:
					tcpd_to_nextseparator(&r);
					if (tcpd_parse_option(req, r, 1) == TCPD_ERROR)
						return TCPD_ERROR; /* parse error */
					tcpd_to_nextseparator(&r);
					if (tcpd_exec_option(req) == TCPD_ERROR)
						return TCPD_ERROR; /* parse error */
					result = TCPD_DENY;
				case TCPD_CONTINUE:
					continue;
				case TCPD_ERROR:
				default:
					return TCPD_ERROR;
				}
			}
			continue;
		}
		if (tcpd_word(p, "DEFAULT", &q) ) {
			(void)tcpd_separator(&q);
			if (tcpd_parse_option(req, q, 0) == TCPD_ERROR)
				return TCPD_ERROR; /* parse error */
			continue;
		}
		if (!tcpd_word(p, service = "ALL", &q) ) {
			service = req->service;
			sservicep = &current_service;
		} else {
			sservicep = &ALL_service;
		}
		switch(service_match(p, service, req, &q)) {
		case TCPD_MATCH:
			if (*q != '\0') {
				if (*q == ' ' || *q == SEPARATOR || *q == '\t')
					*q++ = '\0';
				else
					return TCPD_ERROR;
			}
			/* Next service? */
			if (in_service != SERVICE_IN ||
					!strcasecmp(p, service_name)) {
				in_service = SERVICE_IN;
				strncpy(service_name, p, sizeof(service_name));
			}
			SPACESKIP(q);
			if (tcpd_parse_option(req, q, 0) == TCPD_ERROR)
				return TCPD_ERROR; /* parse error */
			continue;
		case TCPD_NOMATCH:
			/* Next service */
			if (in_service == SERVICE_IN) {
				in_service = SERVICE_OTHER;
				continue;
			}
		default:
			break;
		}
		in_service = SERVICE_OTHER;
	}
	req->lineno = -1;
	return result;
}

int hosts_access(struct request_info *req)
{
	FILE *fp;
	int ret;

	req->error = 0;
	req->lineno = 0;
	if ((fp = fopen(access_table, "r")) == NULL) {
		req->error = errno;
		syslog(deny_severity, "can't open config file %s: %m", access_table);
		return TCPD_DENY;
	}
	ret = hosts_access0(fp, req);
	fclose(fp);

#define SET_VAR(g,a,c) ((((a)<0)&&((c)<0)) ? g : (((c)<0) ? a : c))

	req->refuse_wait = SET_VAR(req->refuse_wait, ALL_service.refuse_wait, current_service.refuse_wait);
	req->rfc1413_timeout = SET_VAR(req->rfc1413_timeout, ALL_service.rfc1413_timeout, current_service.rfc1413_timeout);
	allow_severity = SET_VAR(allow_severity, ALL_service.allow_severity, current_service.allow_severity);
	deny_severity = SET_VAR(deny_severity, ALL_service.deny_severity, current_service.deny_severity);
	userresolvreq = SET_VAR(userresolvreq, ALL_service.ident, current_service.ident);
	defaultallow = SET_VAR(defaultallow, ALL_service.defaultrule, current_service.defaultrule);
	ignore_src_dns_fake =SET_VAR(ignore_src_dns_fake, ALL_service.ignore_src_dns_fake, current_service.ignore_src_dns_fake);
	ignore_dst_dns_fake = SET_VAR(ignore_dst_dns_fake, ALL_service.ignore_dst_dns_fake, current_service.ignore_dst_dns_fake);

	if (ret == TCPD_ALLOW && !ignore_src_dns_fake && req->src_wrongdns)
		ret = TCPD_DENY;
	else
	if (ret == TCPD_ALLOW && !ignore_dst_dns_fake && req->dst_wrongdns)
		ret = TCPD_DENY;
	else
	if (ret == TCPD_ERROR) {
		req->error = -1;
		ret = TCPD_DENY;
	}
	(void)tcpd_resolv_user(req, 0);

	return ret;
}

int hosts_ctl(char *daemon, char *client_name, char *client_addr,
	      char *client_user)
{
    struct request_info request;

    request_init(&request, RQ_DAEMON, daemon, RQ_CLIENT_NAME, client_name,
		    RQ_CLIENT_ADDR, client_addr, RQ_USER, client_user, RQ_END);
    switch(hosts_access(&request)) {
    case TCPD_ALLOW:
	    return (TCPD_ALLOW);
	    break;
    default:
	    return (TCPD_DENY);
    }
}
