/*
 * libwrap.c
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the BSD style license (see
 *		COPYING file included with this software).
 *
 * Authors:     Kazunori Fujiwara <fujiwara@rcac.tdi.co.jp>
 *                  
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "rcsid.h"
RCSID(PKG_VER "$Id: libwrap.c,v 1.3 2000/02/24 15:07:39 misiek Exp $")

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>
#include <errno.h>
#include <ctype.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <syslog.h>

#include "tcpd.h"
#include "tcpd_local.h"

static char access_table[TCPD_STRINGLENL] = "/etc/hosts.access";
static int userresolvreq;
static int defaultallow;
static int ignore_dns_fake;

static void strtosa(char *str, struct sockaddr_storage *sa)
{
	struct addrinfo hints, *res, *res0;
	
	memset(sa, 0, sizeof(*sa));
	memset(&hints, 0, sizeof(hints));
	hints.ai_flags = AF_UNSPEC;
	hints.ai_flags = SOCK_STREAM;
	if (!getaddrinfo(str, NULL, &hints, &res0)) {
		for (res = res0; res; res = res->ai_next) {
			if ((res->ai_family != AF_INET)
					&& (res->ai_family != AF_INET6))
				continue;
			memcpy(sa, res->ai_addr, res->ai_addrlen);
			sa->__ss_family = res->ai_family;
			break;
		}
		freeaddrinfo(res0);
	}
}

void request_init(struct request_info *req, ...)
{
	va_list ap;
	int cmd;

	va_start(ap, req);

	userresolvreq = 1;
	defaultallow = TCPD_DENY;
	ignore_dns_fake = 0;

	memset(req, 0, sizeof(struct request_info));
	req->sock = -1;
	req->userresolved = 0;
	req->src.__ss_family = -1;
	req->dst.__ss_family = -1;
	allow_severity = deny_severity = LOG_DAEMON | LOG_NOTICE;
	req->udp = 0;
	req->refuse_wait = 5;
	req->wrongdns = 0;
	req->rfc1413_timeout = TCPD_DEFAULT_RFC1413_TIMEOUT;

	do {
		cmd = va_arg(ap, int);
		switch(cmd) {
		case RQ_FILE:
			req->sock = va_arg(ap, int);
			break;
		case RQ_DAEMON:
			strncpy(req->service, va_arg(ap, char *), sizeof(req->service));
			break;
		case RQ_USER:
			strncpy(req->user, va_arg(ap, void *), sizeof(req->user));
			break;
		case RQ_SERVER_SIN:
		case RQ_CLIENT_SIN:
			(void)va_arg(ap, void *);
			break;
		case RQ_SERVER_NAME:
			strncpy(req->dst_name, va_arg(ap, void *), sizeof(req->dst_name));
			break;
		case RQ_CLIENT_NAME:
			strncpy(req->src_name, va_arg(ap, void *), sizeof(req->src_name));
			break;
		case RQ_SERVER_ADDR:
			strtosa(va_arg(ap, void *), &req->dst);
			break;
		case RQ_CLIENT_ADDR:
			strtosa(va_arg(ap, void *), &req->src);
			break;
		case RQ_SETTABLE:
			strncpy(access_table, va_arg(ap, void *), sizeof(access_table));
			break;
		case RQ_END:
			break;
		default:
			cmd = RQ_END;
			break;
		}
	} while(cmd != RQ_END);

	va_end(ap);
}

static int gethostaddr(struct sockaddr_storage *sa, char *str, int slen)
{
	struct addrinfo hints, *res, *res0 = NULL;
	char hbuf[NI_MAXHOST], hipbuf[NI_MAXSERV];

	str[0] = '\0';
	if (getnameinfo((struct sockaddr *)sa, SA_LEN((struct sockaddr *)sa),
			hbuf, sizeof(hbuf), NULL, 0, NI_NAMEREQD))
		return 1;
	if (getnameinfo((struct sockaddr *)sa, SA_LEN((struct sockaddr *)sa),
			hipbuf, sizeof(hipbuf), NULL, 0, NI_NUMERICHOST))
		return 1;
	/* Convert IPv4-mapped IPv6 address to native IPv4 */
	if (sa->__ss_family == AF_INET6 && IN6_IS_ADDR_V4MAPPED((&((struct sockaddr_in6 *)sa)->sin6_addr)))
		strncpy(hipbuf, hipbuf + 7, sizeof(hipbuf));
	if (strlen(hbuf) >= slen)
		return -1;
	hbuf[sizeof(hbuf) - 1] = '\0';
	memset(&hints, 0, sizeof(hints));
	hints.ai_family = AF_UNSPEC;
	hints.ai_socktype = SOCK_STREAM;
	if (getaddrinfo(hbuf, NULL, &hints, &res0))
		return -1;
	for (res = res0; res; res = res->ai_next) {
		char hipbuf2[NI_MAXHOST];
		if (!getnameinfo(res->ai_addr, res->ai_addrlen, hipbuf2,
				sizeof(hipbuf2), NULL, 0, NI_NUMERICHOST)
				&& (!strcmp(hipbuf, hipbuf2))) {
			strcpy(str, hbuf);
			freeaddrinfo(res0);
			return 0;
		}
	}
	freeaddrinfo(res0);
	
	return -1;
}

void fromhost(struct request_info *req)
{
	int salen;

	salen = sizeof(req->src);
	if (getpeername(req->sock, (struct sockaddr *)&req->src, &salen) < 0) {
		char buf[BUFSIZ];
		req->udp = 1;
		salen = sizeof(req->src);
		if (recvfrom(req->sock, buf, sizeof(buf), MSG_PEEK, (struct sockaddr *)&req->src, &salen) < 0) {
			return;
		}
	}
	salen = sizeof(req->dst);
	if (getsockname(req->sock, (struct sockaddr *)&req->dst, &salen) < 0)
		return;
	if (gethostaddr(&req->src, req->src_name, sizeof(req->src_name)) == -1)
		req->wrongdns = 1;
	if (gethostaddr(&req->dst, req->dst_name, sizeof(req->dst_name)) == -1)
		req->wrongdns = 1;
}

/*
    char *tcpd_resolv_user(struct request_info *&req, int force)

	get requrest's username
		parameter: force = 1 ... force resolv user
		           force = 0 ... depends on req->userreqolvreq
		returns:   NULL ... No username
			   username (char *)
*/

static char *tcpd_resolv_user(struct request_info *req, int force)
{
	if ((userresolvreq || force) && (!req->userresolved)) {
		rfc1413(&req->dst, &req->src, req->user, sizeof(req->user), req->rfc1413_timeout);
		req->userresolved = 1;
	}
	return *req->user == '\0' ? NULL : req->user;
}

/*
	Does struct sockaddr_storage *sa match addr/mask ?
 */

static int match_ipaddress(struct sockaddr_storage *sa, char *addr, int prefixlen)
{
	unsigned char inaddr_sa[16], inaddr[16], mask[16];
	int size, i;

	if (!inet_pton(sa->__ss_family, addr, &inaddr))
		return TCPD_NOMATCH;
	switch(sa->__ss_family) {
	case AF_INET:
		size = 4;
		memcpy(inaddr_sa, &((struct sockaddr_in *)&sa)->sin_addr, size);
		break;
	case AF_INET6:
		size = 16;
		memcpy(inaddr_sa, &((struct sockaddr_in6 *)&sa)->sin6_addr, size);
		break;
	default:
		return TCPD_NOMATCH;
	}
	if (prefixlen >= 0) {
		prefixlen2mask(prefixlen, mask, size);
		for (i = 0; i < size; i++) {
			inaddr_sa[i] &= mask[i];
			inaddr[i] &= mask[i];
		}
	}
	return memcmp(inaddr_sa, inaddr, size) ? TCPD_NOMATCH : TCPD_MATCH;
}

/*
	compare req's userhost and user/host/mask/ipaddress
*/

static int match_host_part(struct request_info *req, char *user, char *host, int mask)
{
	char *p;

	if (user[0] != '\0' &&
	     strcmp(tcpd_resolv_user(req, 1), user))
		return TCPD_NOMATCH;
	if (mask == -1) {
		if (!strcasecmp(host, "ALL"))
			return TCPD_MATCH;
		if (!strcasecmp(host, "KNOWN"))
			return *req->src_name != '\0';
		if (!strcasecmp(host, req->src_name))
			return TCPD_MATCH;
		if (*host == '.') {
			p = req->src_name;
			while((p = strchr(p, '.')) != NULL)
				if (strcasecmp(p, host))
					return TCPD_MATCH;
			return TCPD_NOMATCH;
		}
	}
	return match_ipaddress(&req->src, host, mask);
}

static int service_match(char *str, char *service, struct request_info *req, char **left)
{
	char *p;
	char host[TCPD_HOSTNAMELEN];

	if (!tcpd_word(str, service, &p))
		return TCPD_NOMATCH;
	if (*p == '@') {
		p++;
		if (tcpd_pstrcmp(p, "inet6#", &p)) {
			if (req->dst.__ss_family != AF_INET6)
				return TCPD_NOMATCH;
		} else
		if (tcpd_pstrcmp(p, "inet#", &p)) {
			if (req->dst.__ss_family != AF_INET)
				return TCPD_NOMATCH;
		}
		if (tcpd_word(p, req->dst_name, &p)) {
			*left = p;
			return TCPD_MATCH;
		}
		if (tcpd_get_token(p, &p, host, sizeof(host)) == 0)
			return TCPD_NOMATCH;
		switch(match_ipaddress(&req->dst, host, -1)) {
		case TCPD_MATCH:
			*left = p;
			return TCPD_MATCH;
		default:
			return TCPD_NOMATCH;
		}
	}
	if (!tcpd_is_separator(*p))
		return TCPD_NOMATCH;
	*left = p;
	return TCPD_MATCH;
}

/*
	str ... HOST | options | misc
*/

static int host_match(struct request_info *req, char *strp, char **left)
{
	int mask;
	int ret;
	int af;
	char host[TCPD_HOSTNAMELEN];
	char user[TCPD_HOSTNAMELEN];

	do {
		if (!tcpd_get_token(strp, &strp, host, sizeof(host)))
			return TCPD_ERROR;
		mask = -1;
		af = AF_UNSPEC;
		if (*strp == '@') {
			strncpy(user, host, sizeof(user));
			strp++;
			if (!tcpd_get_token(strp, &strp, host, sizeof(host)))
				return TCPD_ERROR;
		} else {
			*user = '\0';
		}
		if (*strp == '#') {
			if (!strcasecmp(host, "inet")) {
				af = AF_INET;
			} else
			if (!strcasecmp(host, "inet6")) {
				af = AF_INET6;
			} else
				return TCPD_ERROR;
			strp++;
			if (!tcpd_get_token(strp, &strp, host, sizeof(host)))
				return TCPD_ERROR;
		}
		if (*strp == '/') {
			strp++;
			if (!isdigit(*strp))
				return TCPD_ERROR;
			mask = strtoul(strp, &strp, 10);
		}
		SPACESKIP(strp);
		switch(*strp) {
		case ',':
			strp++;
		case '\0':
		case SEPARATOR:
			break;
		default:
			return TCPD_ERROR;
		}

		/* compare req and pattern */
		/* user@host/mask */
		/* if match ... retrn mode */
		if (af == AF_UNSPEC ||
		     (af == AF_INET && req->src.__ss_family == AF_INET)
		      || (af == AF_INET6 && req->src.__ss_family == AF_INET6)
			) {
			switch(ret = match_host_part(req,user,host,mask)) {
			case TCPD_MATCH:
				*left = strp;
				return TCPD_MATCH;
				break;
			case TCPD_NOMATCH:
				break;
			case TCPD_ERROR:
			default:
				return TCPD_ERROR;
			}
		}
		SPACESKIP(strp);
		if (*strp == ',')
			strp++;
		SPACESKIP(strp);
	} while (*strp != SEPARATOR && *strp != '\0');
	return TCPD_CONTINUE;
}

/*
   int tcpd_parse_option(struct request_info *req, char *str, int mode)

      parse flags and options to 'req'
	parameter:  mode = 0 ... default set mode
	            mode = 1 ... append mode
 */

static struct request_info *sreq;

static int set_syslogallow(char *s)
{
  int i = tcpd_parse_syslog(s, NULL);

  if (i >= 0) {
    allow_severity = i;
    return 1;
  }
  return 0;
}

static int set_syslogdeny(char *s)
{
  int i = tcpd_parse_syslog(s, NULL);

  if (i >= 0) {
    deny_severity = i;
    return 1;
  }
  return 0;
}

static int set_syslog(char *s)
{
  int i = tcpd_parse_syslog(s, NULL);

  if (i >= 0) {
    allow_severity = deny_severity = i;
    return 1;
  }
  return 0;
}

static int set_refuse_wait(int val){sreq->refuse_wait = val; return 1;}
static int set_rfc1413_timeout(int val){sreq->rfc1413_timeout = val;return 1;}

struct _code flagsheader[] = {
	{ "syslogallow", T_FUNC_STR1, &set_syslogallow},
	{ "syslogdeny", T_FUNC_STR1, &set_syslogdeny},
	{ "syslog", T_FUNC_STR1, &set_syslog},
	{ "defaultallow", T_SET1, &defaultallow },
	{ "defaultdeny", T_SET0, &defaultallow },
	{ "refuse_wait", T_FUNC_UINT1, &set_refuse_wait },
	{ "rfc1413_timeout", T_FUNC_UINT1, &set_rfc1413_timeout },
	{ "ignorednsfake", T_SET1, &ignore_dns_fake },
	{ "denydnsfake", T_SET0, &ignore_dns_fake },
	{ "ident", T_SET1, &userresolvreq },
	{ "noident", T_SET0, &userresolvreq },
	{ NULL }
};


static int tcpd_parse_option(struct request_info *req, char *str, int mode)
{
	sreq = req;

	if (tcpd_parse_options(str, &str, flagsheader) == TCPD_ERROR)
		return TCPD_ERROR;
	if (*str == SEPARATOR) {
		str++;
		SPACESKIP(str);
		if (strlen(str) > 0) {
			if (mode == 0)
				strncpy(req->commands_default, str, sizeof(req->commands_default));
			else
				strncpy(req->commands, str, sizeof(req->commands));
		} else {
			if (mode == 0)
				*req->commands_default = '\0';
			else
				*req->commands = '\0';
		}
	}
	return TCPD_MATCH;
}


static int hosts_access0(FILE *fp, struct request_info *req)
{
	char buff[TCPD_BUFLEN];
	char *p, *q, *r, *service;
#define SERVICE_ALL 2
#define SERVICE_IN  1
#define SERVICE_OTHER 0
	int in_service = SERVICE_ALL;
	char service_name[TCPD_STRINGLEN];

	*service_name = '\0';
	while(fgets(buff, sizeof(buff), fp) != NULL) {
		req->lineno++;
		p = buff + strlen(buff) -1;
		if (*p == '\n') *p = '\0';
		p = buff;
		SPACESKIP(p);
		if (*p == '#') /* comment */
			continue;
		if (*p == '\0')
			continue;
		if (tcpd_word(p, "ALLOW", &q)) {
			if (in_service != SERVICE_OTHER) {
				tcpd_separator(&q);
				switch(host_match(req, q, &r)) {
				case TCPD_MATCH:
					tcpd_to_nextseparator(&r);
					if (tcpd_parse_option(req, r, 1) == TCPD_ERROR)
						return TCPD_ERROR; /* parse error */
					return TCPD_ALLOW;
				case TCPD_CONTINUE:
					continue;
				case TCPD_ERROR:
				default:
					return TCPD_ERROR;
				}
			}
			continue;
		}
		if (tcpd_word(p, "DENY", &q)) {
			if (in_service != SERVICE_OTHER) {
				tcpd_separator(&q);
				switch(host_match(req, q, &r)) {
				case TCPD_MATCH:
					tcpd_to_nextseparator(&r);
					if (tcpd_parse_option(req, r, 1) == TCPD_ERROR)
						return TCPD_ERROR; /* parse error */
					return TCPD_DENY;
				case TCPD_CONTINUE:
					continue;
				case TCPD_ERROR:
				default:
					return TCPD_ERROR;
				}
			}
			continue;
		}
		if (tcpd_word(p, "DEFAULT", &q) ) {
			(void)tcpd_separator(&q);
			if (tcpd_parse_option(req, q, 0) == TCPD_ERROR)
				return TCPD_ERROR; /* parse error */
			continue;
		}
		if (!tcpd_word(p, service = "ALL", &q) )
			service = req->service;
		switch(service_match(p, service, req, &q)) {
		case TCPD_MATCH:
			if (*q != '\0') {
				if (*q == ' ' || *q == SEPARATOR || *q == '\t')
					*q++ = '\0';
				else
					return TCPD_ERROR;
			}
			if (in_service == SERVICE_IN
			    && strcasecmp(p, service_name))
				return defaultallow;
			in_service = SERVICE_IN;
			strncpy(service_name, p, sizeof(service_name));
			SPACESKIP(q);
			if (tcpd_parse_option(req, q, 0) == TCPD_ERROR)
				return TCPD_ERROR; /* parse error */
			continue;
		case TCPD_NOMATCH:
			if (in_service == SERVICE_IN)
				return defaultallow;
		default:
			break;
		}
		in_service = SERVICE_OTHER;
	}
	req->lineno = -1;
	return defaultallow;
}

int hosts_access(struct request_info *req)
{
	FILE *fp;
	int ret;

	req->error = 0;
	req->lineno = 0;
	if ((fp = fopen(access_table, "r")) == NULL) {
		req->error = errno;
		syslog(deny_severity, "can't open config file %s: %m", access_table);
		return 0;
	}
	ret = hosts_access0(fp, req);
	fclose(fp);

	if (ret == TCPD_ALLOW && !ignore_dns_fake && req->wrongdns)
		ret = TCPD_DENY;
	else
	if (ret == TCPD_ERROR) {
		req->error = -1;
		ret = TCPD_DENY;
	}
	(void)tcpd_resolv_user(req, 0);

	return ret;
}

int hosts_ctl(char *daemon, char *client_name, char *client_addr,
	      char *client_user)
{
    struct request_info request;

    request_init(&request, RQ_DAEMON, daemon, RQ_CLIENT_NAME, client_name,
		    RQ_CLIENT_ADDR, client_addr, RQ_USER, client_user, RQ_END);
    switch(hosts_access(&request)) {
    case TCPD_ALLOW:
	    return (TCPD_ALLOW);
	    break;
    default:
	    return (TCPD_DENY);
    }
}
