/*
  fix-info-dir - create a top-level `dir' file in the Info system
  Copyright (C) 2000 Pawel A. Gajda (mis@pld.org.pl)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
 
  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  59 Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/*
  $Id: fix-info-dir.c,v 1.6 2000/07/20 16:12:16 mis Exp $
*/

#include <alloca.h>
#include <ctype.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <limits.h>
#include <time.h>
#include <unistd.h>
#include <dirent.h>
#include <fnmatch.h>
#include <getopt.h>
#include <sys/stat.h>
#include <sys/types.h>

#include <zlib.h>
#include "trurl/narray.h"

#ifndef VERSION
# error "-DVERSION=<version> missing"
#endif

static char default_infodirfile[] = "dir";
static char default_infodir[] = "/usr/share/info"; /* FHS 2.{0,1} */
static char default_infosection[] = "Miscellaneous";

static const char progname[] = "fix-info-dir";
static const char info_section_tag[] = "INFO-DIR-SECTION ";
static const char info_entry_btag[]  = "START-INFO-DIR-ENTRY";
static const char info_entry_etag[]  = "END-INFO-DIR-ENTRY";
static int info_section_tag_len;
static int info_entry_btag_len;
static int info_entry_etag_len;

static char *argv0;
static int verbose = 0, force = 0, add_incomplete = 0;
static const char version[] = VERSION;

struct dir_section {
    char      *name;            /* *must* be first member */
    tn_array  *entries;         /* array of strings */
};

struct vfile {
    char  *path;
    void  *fp;
    void  *(*open)  (const char*, const char *);
    char  *(*gets)  (void*, char*, int);
    int   (*close)  (void*);
};


void dir_section_free(struct dir_section *sec) 
{
    free(sec->name);
    n_array_free(sec->entries);
    free(sec);
}


struct dir_section *dir_section_new(const char *name) 
{
    struct dir_section *sec;
    
    sec = malloc(sizeof(*sec));
    sec->name = strdup(name);
    sec->entries = n_array_new(64, free, (t_fn_cmp)strcmp);
    return sec;
}


int dir_section_cmp(struct dir_section *a, struct dir_section *b) 
{
    return strcasecmp(a->name, b->name);
}


static char *file_gets(void *f, char *buf, int size) 
{
    return fgets(buf, size, f);
}


static void errmsg(char *fmt, ...)
{
    va_list args;
    
    fprintf(stderr, "%s: ", progname);
    va_start(args, fmt);
    vfprintf(stderr, fmt, args);
    va_end(args);
}


static void msg(char *fmt, ...)
{
    va_list args;
    
    va_start(args, fmt);
    vfprintf(stderr, fmt, args);
    va_end(args);
}


/*
  eat up non alphanumeric characters from the beginning and
  the end of section name -- this is frequent typo in info files
 */
static char *prepare_section(char *str)
{
    char *p;

    while (*str && !isalnum(*str))
        str++;

    if (*str == '\0')
        return NULL;
    
    p = str + strlen(str) - 1;
    while (!isalnum(*p)) {
        *p = '\0';
        if (p == str)
            break;
        p--;
    }
    
    if (*str == '\0')
        return NULL;
    
    return str;
}


#define MAX_SECTIONS   64  /* max number of sections in one file */
int add_file(tn_array *dirnode, struct vfile *f)
{
    char buf[256];
    int in_entries = 0, n;
    struct dir_section tmp, *sec[MAX_SECTIONS];
    int i, sec_idx = 0, has_info_tags = 0, is_info = 1;
    
    while (f->gets(f->fp, buf, sizeof(buf))) {
        n++;
        
        if (*buf == '') {
            if (*(buf + 1) == '\n' || *(buf + 1) == '\r')
                is_info = 1;
            break;
        }
        
        if (!in_entries &&
            strncmp(buf, info_section_tag, info_section_tag_len) == 0) {
            if (verbose > 2)
                msg("sec: %s", buf);
            
            tmp.name = prepare_section(buf + info_section_tag_len);
            if (tmp.name == NULL) {
                if (verbose)
                    msg("%s has invalid section, using '%s'.\n", 
                        f->path, default_infosection);
                
                tmp.name = default_infosection;
            }
            
            if ((sec[sec_idx] = n_array_bsearch(dirnode, &tmp)) == NULL) {
                if (sec_idx >= MAX_SECTIONS)
                    continue;

                sec[sec_idx] = dir_section_new(tmp.name);
                n_array_push(dirnode, sec[sec_idx]);
                n_array_sort(dirnode);
            }
            sec_idx++;
            continue;
        }
        
        if (in_entries) {
            if (verbose > 2)
                msg("ent: %s", buf);
            
            if (sec_idx <= 0) {
                errmsg("This should not happen (buf = %s).\n", buf);
                exit(EXIT_FAILURE);
            }
            
            if (*buf == '*') 
                for (i=0; i<sec_idx; i++) 
                    n_array_push(sec[i]->entries, strdup(buf));
                
            else if (strcmp(buf, info_entry_etag) == 0) 
                break;
        }
        

        if (strncmp(buf, info_entry_btag, info_entry_btag_len) == 0) {
            in_entries = 1;
            if (sec_idx == 0) { /* no sections secified */
                sec_idx++;
                tmp.name = default_infosection; 
                if ((sec[0] = n_array_bsearch(dirnode, &tmp)) == NULL) {
                    sec[0] = dir_section_new(tmp.name);
                    n_array_push(dirnode, sec[0]);
                    n_array_sort(dirnode);
                }
            }
            
        } else if (strncmp(buf, info_entry_etag, info_entry_etag_len) == 0) {
            sec_idx = 0;        /* reset sections */
            in_entries = 0;
            has_info_tags = 1;
        }
    }

    if (is_info && !has_info_tags) {
        if (verbose && !add_incomplete)  
            msg("No entries info in %s, skipping it\n", f->path);
        
        if (add_incomplete) {
            char entry[256], *p, *name;

            if ((p = strrchr(f->path, '/'))) 
                name = p + 1;
            else
                name = f->path;
            
            if ((p = strrchr(name, '.')) && strcmp(p, ".gz") == 0)
                *p = '\0';
            
            if ((p = strrchr(name, '.')) && strcmp(p, ".info") == 0) {
                *p = '\0';

                if (verbose) 
                    msg("Generating dummy entry for %s\n", f->path);
                
                snprintf(entry, sizeof(entry),
                         "* %s:  (%s).    %s (generated by %s)\n",
                         name, name, name, progname);

                
                tmp.name = default_infosection;
                if ((sec[0] = n_array_bsearch(dirnode, &tmp)) == NULL) {
                    sec[0] = dir_section_new(tmp.name);
                    n_array_push(dirnode, sec[0]);
                    n_array_sort(dirnode);
                }
            
                n_array_push(sec[0]->entries, strdup(entry));
            }
        }
    }
    
    return 1;
}


void print_dirheader(FILE *f) 
{
    time_t t;
    char datestr[128];
    t = time(0);
    strftime(datestr, sizeof(datestr),
             "%a, %d %b %Y %H:%M:%S GMT", gmtime(&t));
    
    fprintf(f, "-*- Text -*-\n"
"This file was automatically generated by %s %s on \n%s. "
"This is the file, which contains the topmost node of the Info hierarchy.\n"
"\n"
"File: dir\tNode: Top\tThis is the top of the INFO tree\n"
"\n"
"This (the Directory node) gives a menu of major topics.\n"
"Typing \"q\" exits, \"?\" lists all Info commands, \"d\" returns here,\n"
"\"h\" gives a primer for first-timers.\n"
"\n"            
"* Menu:\n\n", argv0 ? argv0 : progname, version, datestr);
}


int backup_file(const char *pathname) 
{
    char backup[PATH_MAX];

    snprintf(backup, sizeof(backup), "%s.old", pathname);
    if (rename(pathname, backup) != 0 && errno != ENOENT) {
        errmsg("rename %s %s: %s\n", pathname, backup, strerror(errno));
        return 0;
    }
    
    return 1;
}


time_t mtime(const char *pathname) 
{
    struct stat st;
    
    if (stat(pathname, &st) != 0)
        return 0;

    return st.st_mtime;
}


int mk_dirnode(tn_array *dirnode, const char *path) 
{
    int   i, j;
    FILE  *f;

    if (n_array_size(dirnode) == 0)
        return 1;
    
    /* For these paranoid guys having umask 077 ;) */
    /* ...we set umask */
    umask(022);
    
    backup_file(path);
    if ((f = fopen(path, "w")) == NULL) {
	errmsg("open %s: %s\n", path, strerror(errno));
	return 0;
    }

    print_dirheader(f);

    for (i=0; i<n_array_size(dirnode); i++) {
        struct dir_section *sec;
        
        sec = n_array_nth(dirnode, i);
        n_array_sort(sec->entries);
        n_array_uniq(sec->entries);

        if (verbose > 2)
            msg("write sec: %s\n", sec->name);

        if (n_array_size(sec->entries) == 0) {
            if (verbose)
                msg("Omiting empty section '%s'\n", sec->name);
            continue;
        }

        fprintf(f, "\n%s:\n\n", sec->name);
        
        for (j=0; j<n_array_size(sec->entries); j++)
            fprintf(f, "%s", (char*)n_array_nth(sec->entries, j));
    }
    
    fclose(f);

    return 1;
}


void usage(void) 
{
    printf(""
"Usage: %s [OPTION]... [INFO_DIR/[DIR_FILE]]\n\n"
"Rebuild info dir file if info dir was changed. This is replacement\n"
"of similar utility from texinfo package.\n"
"Options:\n"
"  -c, --create      ignored, for compatibility with texinfo's fix-info-dir\n"
"  -d, --delete      ignored, for compatibility with texinfo's fix-info-dir\n"
"  --force           rebuild DIR_FILE even if its modify time is greater \n"
"                    or equal INFO_DIR's time\n"
"  --add-incomplete  add entries for files without entries information\n"
"                    (works with *.info[.gz] files only), default action\n"
"                    is omit them\n"
"  --verbose[=level] be more verbose\n"
"  --debug           for compatibility with texinfo's fix-info-dir, \n"
"                    the same as --verbose=2\n"
"  --help            print this help message and exit\n"
"  --version         print current version and exit\n\n"
"Defaults INFO_DIR and DIR_FILE are '%s' and '%s'.\n"
"Utility makes backup of the info node with a '.old' suffix added.\n"
"\nReport bugs to <mis@pld.org.pl>\n",
          progname, default_infodir, default_infodirfile);
}


int parse_options(int argc, char *argv[], char **infodir)
{
    struct option long_options[] = {
        {"verbose", optional_argument, 0, 0},
        {"debug",   0, 0, 0},
        {"help",    0, 0, 0},
        {"version", 0, 0, 0},
        {"delete",  0, 0, 0},
        {"create",  0, 0, 0},
        {"force",   0, &force, 1},
        {"add-incomplete", 0, &add_incomplete, 1 },
        {0, 0, 0, 0}
    };
    
    while (1) {
         int c, option_index = 0;
         c = getopt_long(argc, argv, "cdh", long_options, &option_index);
         
         if (c == -1)
             break;
     
         switch (c) {
              case 0:
                  switch (option_index) {
                      case 0:       /* verbose */
                          if (optarg)
                              verbose = atoi(optarg);
                          else
                              verbose = 1;
                          break;
                          
                      case 1:       /* debug  */
                          verbose = 2;
                          break;
                          
                      case 2:
                          usage();
                          exit(EXIT_SUCCESS);
                          break;

                      case 3:
                          printf("%s %s\n", progname, version);
                  }
                  break;
                  
              case 'c':
              case 'd':
                  if (verbose > 1)
                      msg("Ignore option %c\n", c);
                  break;
                  
             case 'h':
                  usage();
                  exit(EXIT_SUCCESS);
                  break;
     
             case '?': /* getopt_long's error */
             default:
                 exit(EXIT_FAILURE);
         }
    }
    
    if (optind == argc) {
        *infodir = default_infodir; /* assume FHS 2.x info dir location */
        
    } else if (optind < argc) {
        *infodir = argv[optind++];
        
        if (optind < argc) {
            errmsg("Too many arguments.\n");
            exit(EXIT_FAILURE);
        }
        
    } else {
        errmsg("optind > argc. This should not happen.\n");
        exit(EXIT_FAILURE);
    }
    
    return 0;
}


int is_main_info_file(const char *fname) 
{
    char *p;

    if ((p = strrchr(fname, '-')) == NULL)
        return 1;

    if (isdigit(*(p + 1)))
        return 0;

    return 1;
}


int is_file(const char *path) 
{
    struct stat st;
    
    if (stat(path, &st) != 0) {
        if (verbose)
            errmsg("stat %s: %s\n", path, strerror(errno));
        return 0;
    }

    return S_ISREG(st.st_mode);
}


int main(int argc, char *argv[])
{
    char           path[PATH_MAX], *dirpath, *dirfile;
    int            dirfile_len;
    struct dirent  *ent;
    struct vfile   vf;
    tn_array       *dirnode;              /* array of dir_sections */
    DIR            *dir;
    int            nfiles = 0;
    time_t         mtime_infodir, mtime_dirfile;


    if (parse_options(argc, argv, &dirpath) != 0)
        exit(EXIT_FAILURE);
    
    argv0 = argv[0];
    info_section_tag_len = strlen(info_section_tag);
    info_entry_btag_len = strlen(info_entry_btag);
    info_entry_etag_len = strlen(info_entry_etag);

    dirfile = default_infodirfile;

    if ((dir = opendir(dirpath)) == NULL) {
        if (errno == ENOTDIR) {
            char *p;

            if ((p = strrchr(dirpath, '/'))) {
                dirfile = p + 1;
                *p = '\0';
                if ((dir = opendir(dirpath)) == NULL) {
                    errmsg("opendir %s: %s\n", dirpath, strerror(errno));
                    exit(EXIT_FAILURE);
                }
            }
            
        } else {
            errmsg("open %s: %s\n", dirpath, strerror(errno));
            exit(EXIT_FAILURE);
        }
    }

    snprintf(path, sizeof(path), "%s/%s", dirpath, dirfile);
    mtime_dirfile = mtime(path);
    mtime_infodir = mtime(dirpath);

    if (mtime_dirfile >= mtime_infodir && force == 0) {
        if (verbose)
            msg("%s is up to date. Nothing to do.\n", path);
        exit(EXIT_SUCCESS);
    }

    dirfile_len = strlen(dirfile);
    
    vf.open = gzopen;
    vf.close = gzclose;
    vf.gets = gzgets;

    dirnode = n_array_new(64, (t_fn_free)dir_section_free,
                          (t_fn_cmp)dir_section_cmp);

    
    while( (ent = readdir(dir)) ) {
        if (!is_main_info_file(ent->d_name)   ||
            strcmp(ent->d_name, ".") == 0     ||
            strcmp(ent->d_name, "..") == 0    ||
            strncmp(ent->d_name, dirfile, dirfile_len) == 0) {

            if (verbose > 1)
                msg("skipping %s\n", ent->d_name);
            continue;
        }
        
        if (fnmatch("*.gz", ent->d_name, 0) == 0) {
            if (vf.open != gzopen) {
                vf.open  = gzopen;
                vf.close = gzclose;
                vf.gets  = gzgets;
            }

        } else {
            if ((void*)vf.open != (void*)fopen) {
                vf.open = (void *(*)(const char *, const char *))fopen;
                vf.close = (int (*)(void*))fclose;
                vf.gets = file_gets;
            }
        }

        snprintf(path, sizeof(path), "%s/%s", dirpath, ent->d_name);
        
        if (!is_file(path)) {
            if (verbose > 1)
                msg("Skipping %s\n", ent->d_name);
            continue;
        }
            
        if ((vf.fp = vf.open(path, "r")) == NULL) {
            errmsg("open %s: %s\n", path, strerror(errno));
            continue;
        }
        
        if (verbose > 1)
            msg("file: %s\n", path);
        
        vf.path = path;
        add_file(dirnode, &vf);
        vf.close(vf.fp);
        nfiles++;
    }
    closedir(dir);
    
    snprintf(path, sizeof(path), "%s/%s", dirpath, dirfile);
    mk_dirnode(dirnode, path);
    
    if (verbose) {
        int i, nentries = 0;

        for (i=0; i<n_array_size(dirnode); i++) {
            struct dir_section *ds;
            ds = n_array_nth(dirnode, i);
            nentries += n_array_size(ds->entries);
        }
        
        msg("%d files processed, %d sections, %d entries\n",
            nfiles, n_array_size(dirnode), nentries);
    }
    
    n_array_free(dirnode);
    exit(EXIT_SUCCESS);
}
