/*
 * xferstats, a wu-ftpd and ncftpd logfile parser and report generator
 * Copyright 1997  Phil Schwan <pschwan@apk.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 Version change information previously located here has been moved to the NEWS
 file.

 */

#include <stdlib.h>
#include <stdio.h>
#include <strings.h>
#include <math.h>
#include <ctype.h>
#include <time.h>
#include <fcntl.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <netinet/in.h>
#include <unistd.h>
#include "config.h"
#include "xferstats.h"

#ifndef NOGRAPH
	#include "gd/gd.h"
	#include "gd/gdfontt.h"
	#include "gd/gdfonts.h"
	#include "gd/gdfontmb.h"
	#include "gd/gdfontl.h"
	#include "gd/gdfontg.h"
#endif

const char VERSION[] = "xferstats 1.16";

const char *MONTHS[] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul",
	"Aug", "Sep", "Oct", "Nov", "Dec", NULL};

FILE *gif;

int
parseline(struct ftp_entry * ftp_line, struct config_ops * config)
{
	char foo[1024] = {0}, junkChar, tmpStr[1024], foo2[1024] = {0};

	memset(foo, 0, sizeof(foo));
	memset(foo2, 0, sizeof(foo2));
	memset(ftp_line->date, 0, sizeof(ftp_line->date));
	ftp_line->seconds = -1;
	ftp_line->data = 0;
	if (config->map[0] == '\0')
		return -1;
	strncpy(foo, config->map, 1023);
	strncpy(foo2, strchr(foo, '\n'), 1023);
	foo[strlen(foo) - strlen(foo2)] = '\0';
#ifdef DEBUG
	printf("SNIP: %s :PINS\n", foo);
	printf("location: %d\n", strlen(foo));
#endif
	if (config->log_style == 0) {
#ifdef DEBUG
		printf("RAW: %s\n", foo);
#endif
		config->map = config->map + strlen(foo) + 1;
		strncpy(ftp_line->date, foo, 24);
		sscanf(foo + 25, "%d %s %lu %s %c %c %c %c", &ftp_line->seconds, ftp_line->host, &ftp_line->data, ftp_line->file_name, &junkChar, &junkChar, &ftp_line->in_out, &ftp_line->user_type);
#ifdef DEBUG
		printf("%s %d %s %lu %s %c %c\n", ftp_line->date, ftp_line->seconds, ftp_line->host, ftp_line->data, ftp_line->file_name, ftp_line->in_out, ftp_line->user_type);
#endif
	} else {

	/* this is a fairly ugly quick hack, but it seems to work.  I
	   don't plan on replacing this parsing code util a) ncftpd
	   logfiles change or b) someone sends me better code :) */

		char userName[10], *tmpToken;
		int temp;

		config->map = config->map + strlen(foo) + 1;
		if (foo[29] != ',' || (foo[28] != 'R' && foo[28] != 'S')) {
			return 0;
		}
#ifdef DEBUG
		else
			printf("RAW: %s\n", foo);
#endif
		if ((foo[28] == 'R' || foo[28] == 'S') && foo[29] == ',') {
			strncpy(ftp_line->date, "    ", 4);
			memset(userName, 0, sizeof(userName));
			strncpy(userName, foo+3, 2);
			temp = atoi(userName);
			strncat(ftp_line->date, MONTHS[temp - 1], 3);
			strcat(ftp_line->date, " ");
			strncat(ftp_line->date, foo+6, 2);
			strcat(ftp_line->date, " ");
			strncat(ftp_line->date, foo+9, 8);
			strcat(ftp_line->date, " 19");
			strncat(ftp_line->date, foo, 2);
			tmpToken = strtok(foo+30, ",");
			strcpy(ftp_line->file_name, tmpToken);
			tmpToken = strtok(NULL, ",");
			strcpy(tmpStr, tmpToken);
			sscanf(tmpStr, "%lu", &ftp_line->data);
			tmpToken = strtok(NULL, ",");
			strcpy(tmpStr, tmpToken);
			sscanf(tmpStr, "%d", &ftp_line->seconds);
			tmpToken = strtok(NULL, ",");
			tmpToken = strtok(NULL, ",");
			tmpToken = strtok(NULL, ",");
			tmpToken = strtok(NULL, ",");
			strcpy(ftp_line->host, tmpToken);
			if (foo[28] == 'R') ftp_line->in_out = 'o';
			else ftp_line->in_out = 'i';
			if (!strcmp(userName, "anonymous")) ftp_line->user_type = 'a';
			else ftp_line->user_type = 'r';
#ifdef DEBUG
			printf("%s %d %s %lu %s %c %c\n", ftp_line->date, ftp_line->seconds, ftp_line->host, ftp_line->data, ftp_line->file_name, ftp_line->in_out, ftp_line->user_type);
#endif
		}	
	}
	if (!strcmp(ftp_line->date, "") || ftp_line->seconds < 0)  
		return 0; 	/* the line is invalid */

	return 1;
} /* parseline */


void
doHours(struct ftp_entry ftp_line)
{
	int hours;
	char foo[3] = {0};

	strncpy(foo, ftp_line.date + 11, 2);
	hours = atoi(foo);
	hourStruct[hours].data += ftp_line.data;
	hourStruct[hours].fileCount++;
	hourStruct[hours].seconds += ftp_line.seconds;
#ifndef NOGRAPH
	if ((float) ftp_line.data / ftp_line.seconds / 1024 > hourStruct[hours].hixfer)
		hourStruct[hours].hixfer = (float) ftp_line.data / ftp_line.seconds / 1024;
#endif
} /* doHours */


int
finalDir(DIRPTR *sPtr, char dcat[4096], struct ftp_entry ftp_line,
	struct config_ops config)
{
	DIRPTR newPtr;

	if (*sPtr == NULL) {
		newPtr = malloc(sizeof(DirNode));
		
		if (newPtr != NULL) {  /* is space available? */
			memset(newPtr->name, 0, sizeof(newPtr->name));
			if (config.html_output) strncpy(newPtr->name, dcat, 1023);
			else strncpy(newPtr->name, dcat, 46);
			newPtr->data = ftp_line.data;
			newPtr->fileCount = 1;
			newPtr->nextPtr = NULL;
			newPtr->sorted = newPtr->datak = 0;
			*sPtr = newPtr;
#ifdef DEBUG
			printf("1st directory structure for \"%s\" created.\n", newPtr->name);
#endif
			return 1;
		} else {
			printf("fatal: out of memory.\n");
			exit(1);
		}
	}

	{
		DIRPTR currentPtr = *sPtr;
		char tempDir[1024] = {0};

		if (config.html_output) strncpy(tempDir, dcat, 1023);
		else strncpy(tempDir, dcat, 46);
		while (currentPtr != NULL) {
			if (!strcmp(currentPtr->name, tempDir)) {
				currentPtr->fileCount++;
				currentPtr->data += ftp_line.data;
				if (currentPtr->data > 3e9) {
					currentPtr->datak += currentPtr->data / 1024;
					currentPtr->data = 0;
				}
#ifdef DEBUG
				printf("Added stats to the dir entry \"%s\".\n", currentPtr->name);
#endif
				return 1;
			}
			currentPtr = currentPtr->nextPtr;
		}
		newPtr = malloc(sizeof(DirNode));
	
		if (newPtr != NULL) {			/* is space available? */
	    		memset(newPtr->name, 0, sizeof(newPtr->name));
			if (config.html_output) strncpy(newPtr->name, dcat, 1023);
			else strncpy(newPtr->name, dcat, 46);
			newPtr->data = ftp_line.data;
			newPtr->fileCount = 1;
			newPtr->sorted = newPtr->datak = 0;
			currentPtr = *sPtr;
			while (currentPtr->nextPtr != NULL)
				currentPtr = currentPtr->nextPtr;
			currentPtr->nextPtr = newPtr;
			newPtr->nextPtr = NULL;
#ifdef DEBUG
			printf("New directory structure for \"%s\" created.\n", newPtr->name);
#endif
		} else {
			printf("fatal: out of memory.\n");
			exit(1);
		}
	}
	return 1;
} /* finalDir */


int
finalFile(FILEPTR *fPtr, char fcat[1024], struct ftp_entry ftp_line)
{
	FILEPTR newFPtr;

	if (*fPtr == NULL) {
		newFPtr = malloc(sizeof(FileNode));
		
		if (newFPtr != NULL) {  /* is space available? */
			memset(newFPtr->name, 0, sizeof(newFPtr->name));
			strncpy(newFPtr->name, fcat, 1023);
			newFPtr->data = ftp_line.data;
			newFPtr->fileCount = 1;
			newFPtr->nextPtr = NULL;
			newFPtr->sorted = 0;
			*fPtr = newFPtr;
#ifdef DEBUG
			printf("1st file structure for \"%s\" created.\n", newFPtr->name);
#endif
			return 1;
		} else {
			printf("fatal: out of memory.\n");
			exit(1);
		}
	}

	{
		FILEPTR currentPtr = *fPtr;

		while (currentPtr != NULL) {
			if (!strcmp(currentPtr->name, fcat)) {
				currentPtr->fileCount++;
				currentPtr->data += ftp_line.data;
#ifdef DEBUG
				printf("Added stats to the file entry \"%s\".\n", currentPtr->name);
#endif
				return 1;
			}
			currentPtr = currentPtr->nextPtr;
		}
		newFPtr = malloc(sizeof(FileNode));
	
		if (newFPtr != NULL) {			/* is space available? */
	    		memset(newFPtr->name, 0, sizeof(newFPtr->name));
			strncpy(newFPtr->name, fcat, 1023);
			newFPtr->data = ftp_line.data;
			newFPtr->fileCount = 1;
			newFPtr->sorted = 0;
			currentPtr = *fPtr;
			while (currentPtr->nextPtr != NULL)
				currentPtr = currentPtr->nextPtr;
			currentPtr->nextPtr = newFPtr;
			newFPtr->nextPtr = NULL;
#ifdef DEBUG
			printf("New file structure for \"%s\" created.\n", newFPtr->name);
#endif
			return 1;
		} else {
			printf("fatal: out of memory.\n");
			exit(1);
		}
	}
	return 1;
} /* finalFile */


int
doDirs(DIRPTR *sPtr, FILEPTR *fPtr, struct ftp_entry ftp_line,
	struct config_ops config)
{
	char **tokens = NULL, *tmpToken, dcat[4096], fcat[1024];
	int i = 0, j = 1;

	tmpToken = strtok(ftp_line.file_name, "/");
	for(; tmpToken; i++) {
		tokens = realloc(tokens, sizeof(char *) * (i + 1));
		tokens[i] = strdup(tmpToken);
		tmpToken = strtok(NULL, "/");
	}

	if(!tokens) {	
#ifdef DEBUG
		printf("Bad file name while splitting in doDirs.\n");
#endif
		return 0;
	}

	bzero(dcat, 4096);
	bzero(fcat, 1024);
	for (; j <= i; j++) {
		if (j < config.depth && j < i) {
			strcat(dcat, "/");
			strcat(dcat, tokens[j - 1]);
		}
		strcat(fcat, "/");
		strcat(fcat, tokens[j - 1]);
	}

#ifdef DEBUG
	{
		printf("Final dir: \"%s\"\n", dcat);
		printf("Final file: \"%s\"\n", fcat);
	}
#endif

	free(tokens);

	if (config.most_downloads && ftp_line.in_out == 'o')
		if (!finalFile(fPtr, fcat, ftp_line)) return 0;
	if (!finalDir(sPtr, dcat, ftp_line, config)) return 0;
	return 1;
} /* doDirs */


int
doDaily(DAYPTR *sPtr, struct ftp_entry ftp_line)
{
	DAYPTR newPtr;
	
	if (*sPtr == NULL) {
		newPtr = malloc(sizeof(DayNode));
		
		if (newPtr != NULL) {  /* is space available? */
			memset(newPtr->name, 0, sizeof(newPtr->name));
			strncpy(newPtr->name, ftp_line.date, 10);
			strncat(newPtr->name, ftp_line.date + 19, 5);
			newPtr->data = ftp_line.data;
			newPtr->fileCount = 1;
			newPtr->nextPtr = NULL;
			newPtr->datak = 0;
			newPtr->seconds = ftp_line.seconds;
			*sPtr = newPtr;
			return 1;
		} else {
			printf("fatal: out of memory.\n");
			exit(1);
		}
	} else {
		DAYPTR currentPtr = *sPtr;
		char tempDay[16] = {0};

		strncpy(tempDay, ftp_line.date, 10);
		strncat(tempDay, ftp_line.date + 19, 5);

		while (currentPtr != NULL) {
#ifdef DEBUG
			printf("currPtr->name: %s\n", currentPtr->name);
#endif
			if (!strcmp(currentPtr->name, tempDay)) {
				currentPtr->fileCount++;
				currentPtr->data += ftp_line.data;
				if (currentPtr->data > 3e9) {
					currentPtr->datak += currentPtr->data / 1024;
					currentPtr->data = 0;
				}
				currentPtr->seconds += ftp_line.seconds;
#ifdef DEBUG
				printf("Added stats to the day \"%s\".\n", currentPtr->name);
#endif
				return 1;
			}
			currentPtr = currentPtr->nextPtr;
		}

	        newPtr = malloc(sizeof(DayNode));
        
       		if (newPtr != NULL) {  /* is space available? */
       			memset(newPtr->name, 0, sizeof(newPtr->name));
			strncpy(newPtr->name, ftp_line.date, 10);
			strncat(newPtr->name, ftp_line.date + 19, 5);
			newPtr->data = ftp_line.data;
			newPtr->fileCount = 1;
			newPtr->datak = 0;
			newPtr->seconds = ftp_line.seconds;
			currentPtr = *sPtr;
			while (currentPtr->nextPtr != NULL)
				currentPtr = currentPtr->nextPtr;
			currentPtr->nextPtr = newPtr;
			newPtr->nextPtr = NULL;
#ifdef DEBUG
			printf("New day structure for \"%s\" created.\n", newPtr->name);
#endif
			return 1;
		} else {
			printf("fatal: out of memory.\n");
			exit(1);
		}
	}	
	return 0;
} /* doDaily */
	

int

doDomain(DOMPTR *sPtr, char domain[11], struct ftp_entry ftp_line)
{
	DOMPTR newPtr;
	
	if (*sPtr == NULL) {
		newPtr = malloc(sizeof(DomainNode));
		
		if (newPtr != NULL) {  /* is space available? */
			memset(newPtr->name, 0, sizeof(newPtr->name));
			strncpy(newPtr->name, domain, 10);
			newPtr->data = ftp_line.data;
			newPtr->fileCount = 1;
			newPtr->nextPtr = NULL;
			newPtr->datak = 0;
			newPtr->seconds = ftp_line.seconds;
			*sPtr = newPtr;
#ifdef DEBUG
			printf("1st domain (\"%s\") created.\n", newPtr->name);
#endif
			return 1;
		} else {
			printf("fatal: out of memory.\n");
			exit(1);
		}
	} else {
		DOMPTR currentPtr = *sPtr;

#ifdef DEBUG
		printf("DOMAIN: %s\n", domain);
#endif
		while (currentPtr != NULL) {
#ifdef DEBUG
 			printf("cPtr  : %s\n", currentPtr->name);
#endif
			if (!strcmp(currentPtr->name, domain)) {
				currentPtr->fileCount++;
				currentPtr->data += ftp_line.data;
				if (currentPtr->data > 3e9) {
					currentPtr->datak += currentPtr->data / 1024;
					currentPtr->data = 0;
				}
				currentPtr->seconds += ftp_line.seconds;
#ifdef DEBUG
				printf("Added to existing domain \"%s\".\n", currentPtr->name);
#endif
				return 1;
			}
			currentPtr = currentPtr->nextPtr;
		}

	        newPtr = malloc(sizeof(DomainNode));
        
       		if (newPtr != NULL) {  /* is space available? */
			memset(newPtr->name, 0, sizeof(newPtr->name));
			strncpy(newPtr->name, domain, 10);
			newPtr->data = ftp_line.data;
			newPtr->fileCount = 1;
			newPtr->datak = 0;
			newPtr->seconds = ftp_line.seconds;
			currentPtr = *sPtr;
			while (currentPtr->nextPtr != NULL)
				currentPtr = currentPtr->nextPtr;
			currentPtr->nextPtr = newPtr;
			newPtr->nextPtr = NULL;
#ifdef DEBUG
			printf("New domain structure for \"%s\" created.\n", newPtr->name);
#endif
			return 1;
		} else {
			printf("fatal: out of memory.\n");
			exit(1);
		}
	}	
	return 0;
} /* doDomain */


int
compareDates(char date1[25], char date2[25])
{
	float dw1, dw2;
	int i, m = 0;
	char foo[5] = {0};

	if (strlen(date1) == 24) {
		strncpy(foo, date1 + 20, 4);
		dw1 = atoi(foo) * 4800;
		strncpy(foo, date2 + 20, 4);
		dw2 = atoi(foo) * 4800;
		strncpy(foo, date1 + 4, 3);
		foo[3] = '\0';
		for (i=0; i <= 11; i++) {
			if (!strcmp(MONTHS[i], foo)) {
				m = i;
				break;
			}
		}
#ifdef DEBUG
		if (!m) printf("ERROR: invalid month\n");
#endif	
		if (m) dw1 += m * 400;

		m = 0;
		strncpy(foo, date2 + 4, 3);
		for (i=0; i <= 11; i++) {
			if (!strcmp(MONTHS[i], foo)) {
				m = i;
				break;
			}
		}
#ifdef DEBUG
		if (!m) printf("ERROR: invalid month\n");
#endif	
		if (m) dw2 += m * 400;

		strncpy(foo, date1 + 8, 2);
		foo[3] = '\0';
		dw1 += atoi(foo);
		strncpy(foo, date2 + 8, 2);
		dw2 += atoi(foo);
	} else {
		strncpy(foo, date1 + 16, 4);
		dw1 = atoi(foo) * 4800;
		strncpy(foo, date2 + 16, 4);
		dw2 = atoi(foo) * 4800;
		strncpy(foo, date1, 3);
		foo[3] = '\0';
		for (i=0; i <= 11; i++) {
			if (!strcmp(MONTHS[i], foo)) {
				m = i;
				break;
			}
		}
#ifdef DEBUG
		if (!m) printf("ERROR: invalid month\n");
#endif	
		if (m) dw1 += m * 400;

		m = 0;
		strncpy(foo, date2, 3);
		for (i=1; i <= 12; i++) {
			if (!strcmp(MONTHS[i], foo)) {
				m = i;
				break;
			}
		}
#ifdef DEBUG
		if (!m) printf("ERROR: invalid month\n");
#endif	
		if (m) dw2 += m * 400;

		strncpy(foo, date1 + 4, 2);
		foo[3] = '\0';
		dw1 += atoi(foo);
		strncpy(foo, date2 + 4, 2);
		dw2 += atoi(foo);
	}
	if (dw1 == dw2) return 0;
	if (dw1 > dw2)
		return 1;
	else
		return -1;
}	


void
displayTotals(DOMPTR *sPtr, DAYPTR *dPtr, DIRPTR *drPtr, FILEPTR *fPtr,
	struct config_ops config, char lowDate[16], char highDate[16],
	unsigned int fileCount, unsigned long total, unsigned long totalk)
{
	DOMPTR currentPtr, hiDPtr = NULL;
	DAYPTR currentDayPtr;
	DIRPTR currentDirPtr, hiPtr = NULL;
	FILEPTR currentFilePtr, hiFPtr = NULL;
	int i = 0, whee = 0, total_mod = 1, indiv_mod, hi_mod;
	char date1[16] = {0}, date2[16] = {0};
	unsigned long hibyte;

	if (totalk) {
		total = (totalk += (total / 1024));
		total_mod = 1024;
	}
	strncpy(date1, lowDate, 10);
	strncat(date1, lowDate + 19, 5);
	strncpy(date2, highDate, 10);
	strncat(date2, highDate + 19, 5);

	printf("TOTALS FOR SUMMARY PERIOD %s TO %s\n\n", date1, date2);
	printf("Files Transmitted During Summary Period     %11u\n", fileCount);
	if (totalk)
		printf("Kilobytes Transmitted During Summary Period %11lu\n\n", total);
	else
		printf("Bytes Transmitted During Summary Period     %11lu\n\n", total);
	currentDayPtr = *dPtr;
	for (; currentDayPtr != NULL; i++)
		currentDayPtr = currentDayPtr->nextPtr;
	if (totalk)
		printf("Average Kilobytes Transmitted Per File      %11lu\n", total / fileCount);
	else
		printf("Average Bytes Transmitted Per File          %11lu\n", total / fileCount);
	printf("Average Files Transmitted Daily             %11u\n", fileCount / i);
	if (totalk)
		printf("Average Kilobytes Transmitted Daily         %11lu\n\n", total / i);
	else
		printf("Average Bytes Transmitted Daily             %11lu\n\n", total / i);
	if (!config.number_daily_stats || config.number_daily_stats >= i) printf("Daily Transmission Statistics\n\n");
	else printf("Daily Transmission Statistics (last %d days)\n\n", config.number_daily_stats);
	printf("                 Number Of    Number of    Average    Percent Of  Percent Of\n");
	printf("     Date        Files Sent  Bytes  Sent  Xmit  Rate  Files Sent  Bytes Sent\n");
	printf("---------------  ----------  -----------  ----------  ----------  ----------\n");
	currentDayPtr = *dPtr;
	while(currentDayPtr != NULL) {
		if (currentDayPtr->datak) {
			indiv_mod = 1024;
			currentDayPtr->datak += (currentDayPtr->data / 1024);
			currentDayPtr->data = currentDayPtr->datak;
		} else
			indiv_mod = 1;
		whee++;
		if (!config.number_daily_stats || whee > i - config.number_daily_stats) {
			if (indiv_mod == 1024)
				printf("%s  %10u  %10luk  %5.1f KB/s  %10.2f  %10.2f\n", currentDayPtr->name, currentDayPtr->fileCount, currentDayPtr->data,
					(float) currentDayPtr->data / currentDayPtr->seconds, (float) currentDayPtr->fileCount / fileCount * 100, (float) currentDayPtr->data / total / total_mod * 102400);
			else
				printf("%s  %10u  %11lu  %5.1f KB/s  %10.2f  %10.2f\n", currentDayPtr->name, currentDayPtr->fileCount, currentDayPtr->data,
					(float) currentDayPtr->data / currentDayPtr->seconds / 1024, (float) currentDayPtr->fileCount / fileCount * 100, (float) currentDayPtr->data / total / total_mod * 100);
		}
		currentDayPtr = currentDayPtr->nextPtr;
	}
	if (config.section_traffic) {
		if (config.number_section_stats) printf("%c\nTop %d Total Transfers from each Archive Section (By bytes)\n\n", 12, config.number_section_stats);
		else printf("%c\nTotal Transfers from each Archive Section (By bytes)\n\n", 12);
		printf("                                                                 ---- %%  Of ----\n");
		printf("                 Archive Section                Files Bytes Sent  Files   Bytes\n");
   		printf(" ---------------------------------------------- ----- ---------- ------- -------\n");
		currentDirPtr = *drPtr;
		for (i = whee = 0; currentDirPtr != NULL; i++)
			currentDirPtr = currentDirPtr->nextPtr;
		while (whee < i && (!config.number_section_stats || whee < config.number_section_stats)) {
			hibyte = 0;
			currentDirPtr = *drPtr;
			while(currentDirPtr != NULL) {
				if (!currentDirPtr->sorted && currentDirPtr->data >= hibyte) {
					hiPtr = currentDirPtr;
					hibyte = currentDirPtr->data;
				}
				currentDirPtr = currentDirPtr->nextPtr;
			}
			if (hiPtr->datak) {
				indiv_mod = 1024;
				hiPtr->datak += (hiPtr->data / 1024);
				hiPtr->data = hiPtr->datak;
			} else
				indiv_mod = 1;
			whee++;
			hiPtr->sorted = 1;
			if (indiv_mod == 1024)
				printf(" %-46s %5u %9luk %7.2f %7.2f\n", hiPtr->name, hiPtr->fileCount, hibyte, (float) hiPtr->fileCount / fileCount * 100, (float) hibyte / total / total_mod * 102400);
			else
				printf(" %-46s %5u %10lu %7.2f %7.2f\n", hiPtr->name, hiPtr->fileCount, hibyte, (float) hiPtr->fileCount / fileCount * 100, (float) hibyte / total / total_mod * 100);
		}
	}
	if (config.top_domain_traffic) {
		if (!config.number_top_domain_stats) printf("%c\nTotal Transfer Amount By Domain (By bytes)\n\n", 12);
		else printf("%c\nTop %d Total Transfer Amounts By Domain (By bytes)\n\n", 12, config.number_top_domain_stats);
		printf("             Number Of    Number of     Average    Percent Of  Percent Of\n");
		printf("Domain Name  Files Sent   Bytes Sent   Xmit  Rate  Files Sent  Bytes Sent\n");
		printf("-----------  ----------  ------------  ----------  ----------  ----------\n");
		currentPtr = *sPtr;
		for (i = whee = 0; currentPtr != NULL; i++)
			currentPtr = currentPtr->nextPtr;
		while (whee < i && (!config.number_top_domain_stats || whee < config.number_top_domain_stats)) {
			hibyte = 0;
			currentPtr = *sPtr;
			while (currentPtr != NULL) {
				if (!currentPtr->sorted && currentPtr->data > hibyte) {
					hiDPtr = currentPtr;
					hibyte = currentPtr->data;
				}
				currentPtr = currentPtr->nextPtr;
			}
			if (!hibyte) break;
			whee++;
			hiDPtr->sorted = 1;
			printf("%-11s  %10u  %12lu  %5.1f KB/s  %10.2f  %10.2f\n", hiDPtr->name, hiDPtr->fileCount, hibyte,
				(float) hibyte / hiDPtr->seconds / 1024, (float) hiDPtr->fileCount / fileCount * 100, (float) hibyte / total / total_mod * 100);
		}
	}
	printf("\nThese figures only reflect FTP transfers.  There are many sites\n");
	printf("which mount the archives via NFS, and those transfers are not logged\n"); 
	printf("and reported by this program.\n\n");
	if (config.hourly_traffic) {
		printf("%c\nHourly Transmission Statistics\n\n", 12);
		printf("                Number Of    Number of     Average    Percent Of  Percent Of\n");
		printf("     Time       Files Sent  Bytes  Sent  Xmit   Rate  Files Sent  Bytes Sent\n");
		printf("--------------  ----------  -----------  -----------  ----------  ----------\n");
		for (i = 0; i <= 23; i++)
			if (hourStruct[i].fileCount)
				printf("%-14d  %10u  %11lu  %6.2f KB/s  %10.2f  %10.2f\n",
					i, hourStruct[i].fileCount, hourStruct[i].data, (float) hourStruct[i].data / hourStruct[i].seconds / 1024, (float) hourStruct[i].fileCount / fileCount * 100, (float) hourStruct[i].data / total / total_mod * 100);
	}
	if (config.most_downloads) {
		printf("%c\nTop %d Most Downloaded Files (By number)\n\n", 12, config.number_most_downloads);
		printf("                                                                ---- %%  Of ----\n");
		printf("                           Filename                       Files  Files   Bytes\n");
		printf(" -------------------------------------------------------- ----- ------- -------\n");
		currentFilePtr = *fPtr;
		for (i = whee = 0; currentFilePtr != NULL; i++)
			currentFilePtr = currentFilePtr->nextPtr;
		while (whee < config.number_most_downloads) {
			hibyte = 0;
			currentFilePtr = *fPtr; 
			while(currentFilePtr != NULL) {
				if (!currentFilePtr->sorted && currentFilePtr->fileCount >= hibyte) {
					hiFPtr = currentFilePtr;
					hibyte = currentFilePtr->fileCount;
				}
				currentFilePtr = currentFilePtr->nextPtr;
			}
			if (!hibyte) break;
			whee++;
			hiFPtr->sorted = 1;
			if (strlen(hiFPtr->name) > 56) {
				char tempName[55];
				strcpy(tempName, hiFPtr->name + strlen(hiFPtr->name) - 54); 
				printf("...%-54s %5lu %7.2f %7.2f\n", tempName, hibyte, (float) hibyte / fileCount * 100, (float) hiFPtr->data / total / total_mod * 100);
			} else
				printf(" %-56s %5lu %7.2f %7.2f\n", hiFPtr->name, hibyte, (float) hibyte / fileCount * 100, (float) hiFPtr->data / total / total_mod * 100);
		}
	}
} /* displayTotals */


void
displayHTMLTotals(DOMPTR *sPtr, DAYPTR *dPtr, DIRPTR *drPtr, FILEPTR *fPtr,
	struct config_ops config, char lowDate[16], char highDate[16],
	unsigned int fileCount, unsigned long total, unsigned long totalk)
{
	DOMPTR currentPtr, hiDPtr = NULL;
	DAYPTR currentDayPtr;
	DIRPTR currentDirPtr, hiPtr = NULL;
	FILEPTR currentFilePtr, hiFPtr = NULL;
	int i = 0, whee = 0;
	char date1[16] = {0}, date2[16] = {0};
	float hixfer = 0;
	unsigned long hibyte;
#ifndef NOGRAPH
	gdImagePtr image1, image2, image3;
	int black, black2, black3, white, white2, white3, blue, blue2, blue3,
		green, green2, green3;
	char foo[5], tempPath[1024], *html_path = HTML_GRAPH_PATH;
	char *real_path = REAL_GRAPH_PATH;
	float hixfer2 = 0;
#endif

	if (real_path[strlen(html_path)+1] != '/')
		strcat(html_path, "/");
	if (real_path[strlen(real_path)+1] != '/')
		strcat(real_path, "/");
	strncpy(date1, lowDate, 10);
	strncat(date1, lowDate + 19, 5);
	strncpy(date2, highDate, 10);
	strncat(date2, highDate + 19, 5);

	printf("<HTML><HEAD>\n");
	printf("<TITLE>xferstats: %s to %s</TITLE>\n", date1, date2);
	printf("</HEAD><BODY TEXT=\"#FFFFFF\" BGCOLOR=\"#000000\">\n");

	printf("<H1>TOTALS FOR SUMMARY PERIOD %s TO %s</H1><br>\n", date1, date2);
	printf("<TABLE BORDER = 2>\n");
	printf("<TR><TH ALIGN=LEFT>Files Transmitted During Summary Period</TH><TH ALIGN=RIGHT>%11u</TH></TR>\n", fileCount);
	printf("<TR><TH ALIGN=LEFT>Bytes Transmitted During Summary Period</TH><TH ALIGN=RIGHT>%11lu</TH></TR>\n", total);
	currentDayPtr = *dPtr;
	for (; currentDayPtr != NULL; i++)
		currentDayPtr = currentDayPtr->nextPtr;
	printf("<TR><TH ALIGN=LEFT>Average Bytes Transmitted Per File</TH><TH ALIGN=RIGHT>%11lu</TH></TR>\n", total / fileCount);
	printf("<TR><TH ALIGN=LEFT>Average Files Transmitted Daily</TH><TH ALIGN=RIGHT>%11u</TH><TR>\n", fileCount / i);
	printf("<TR><TH ALIGN=LEFT>Average Bytes Transmitted Daily</TH><TH ALIGN=RIGHT>%11lu</TH></TR></TABLE><BR><BR>\n", total / i);
	if (!config.number_daily_stats || config.number_daily_stats >= i) printf("<H1>Daily Transmission Statistics</H1><BR>\n");
	else printf("<H1>Daily Transmission Statistics (last %d days)</H1><BR>\n", config.number_daily_stats);
	printf("<TABLE BORDER=2>\n");
	printf("<TR><TH></TH><TH>Number Of</TH><TH>Number of</TH><TH>Average</TH><TH>Percent Of</TH><TH>Percent Of</TH></TR>\n");
	printf("<TR><TH>Date</TH><TH>Files Sent</TH><TH>Bytes Sent</TH><TH>Xmit Rate</TH><TH>Files Sent</TH><TH>Bytes Sent</TH></TR>\n");
	currentDayPtr = *dPtr;
	while(currentDayPtr != NULL) {
		whee++;
		if (!config.number_daily_stats || whee > i - config.number_daily_stats) {
			printf("<TR><TH>%s</TH><TH ALIGN=RIGHT>%u</TH><TH ALIGN=RIGHT>%lu</TH><TH ALIGN=RIGHT>%5.1f KB/s</TH><TH ALIGN=RIGHT>%10.2f</TH><TH ALIGN=RIGHT>%10.2f</TH></TR>\n", currentDayPtr->name, currentDayPtr->fileCount, currentDayPtr->data,
				(float) currentDayPtr->data / currentDayPtr->seconds / 1024, (float) currentDayPtr->fileCount / fileCount * 100, (float) currentDayPtr->data / total * 100);
		}
		currentDayPtr = currentDayPtr->nextPtr;
	}
	printf("</TABLE><BR><BR>\n");
	if (config.section_traffic) {
		if (config.number_section_stats) printf("<H1>Top %d Total Transfers from each Archive Section (By bytes)</H1><BR><BR>\n", config.number_section_stats);
		else printf("<H1>Total Transfers from each Archive Section (By bytes)</H1><BR><BR>\n");
		printf("<TABLE BORDER=2>\n");
		printf("<TR><TH>Archive Section</TH><TH>Files</TH><TH>Bytes Sent</TH><TH>%% of Files</TH><TH>%% of Bytes</TH></TR>\n");
		currentDirPtr = *drPtr;
		for (i = 0; currentDirPtr != NULL; i++)
			currentDirPtr = currentDirPtr->nextPtr;
		whee = 0;
		while (whee < i && (!config.number_section_stats || whee < config.number_section_stats)) {
			hibyte = 0;
			currentDirPtr = *drPtr;
			while(currentDirPtr != NULL) {
				if (!currentDirPtr->sorted && currentDirPtr->data >= hibyte) {
					hiPtr = currentDirPtr;
					hibyte = currentDirPtr->data;
				}
				currentDirPtr = currentDirPtr->nextPtr;
			}
			whee++;
			hiPtr->sorted = 1;
			printf("<TR><TH ALIGN=LEFT>%s</TH><TH ALIGN=RIGHT>%u</TH><TH ALIGN=RIGHT>%lu</TH><TH ALIGN=RIGHT>%7.2f</TH><TH ALIGN=RIGHT>%7.2f</TH></TR>\n",
			hiPtr->name, hiPtr->fileCount, hibyte, (float) hiPtr->fileCount / fileCount * 100, (float) hibyte / total * 100);
		}
		printf("</TABLE><BR<BR>\n");
	}
	if (config.top_domain_traffic) {
		if (!config.number_top_domain_stats) printf("<H1>Total Transfer Amount By Domain (By bytes)</H1><BR>\n");
		else printf("<H1>Top %d Total Transfer Amounts By Domain (By bytes)</H1><BR>\n", config.number_top_domain_stats);
		printf("<TABLE BORDER=2>\n");
		printf("<TR><TH></TH><TH>Number Of</TH><TH>Number of</TH><TH>Average</TH><TH>Percent Of</TH><TH>Percent Of</TH></TR>\n");
		printf("<TR><TH>Domain Name</TH><TH>Files Sent</TH><TH>Bytes Sent</TH><TH>Xmit Rate</TH><TH>Files Sent</TH><TH>Bytes Sent</TH></TR>\n");
		currentPtr = *sPtr;
		for (i = whee = 0; currentPtr != NULL; i++)
			currentPtr = currentPtr->nextPtr;
		while (whee < i && (!config.number_top_domain_stats || whee < config.number_top_domain_stats)) {
			hibyte = 0;
			currentPtr = *sPtr;
			while (currentPtr != NULL) {
				if (!currentPtr->sorted && currentPtr->data > hibyte) {
					hiDPtr = currentPtr;
					hibyte = currentPtr->data;
				}
				currentPtr = currentPtr->nextPtr;
			}
			if (!hibyte) break;
			whee++;
			hiDPtr->sorted = 1;
			printf("<TR><TH>%s</TH><TH ALIGN=RIGHT>%u</TH><TH ALIGN=RIGHT>%lu</TH><TH ALIGN=RIGHT>%5.1f KB/s</TH><TH ALIGN=RIGHT>%10.2f</TH><TH ALIGN=RIGHT>%10.2f</TH></TR>\n", hiDPtr->name, hiDPtr->fileCount, hibyte,
				(float) hibyte / hiDPtr->seconds / 1024, (float) hiDPtr->fileCount / fileCount * 100, (float) hibyte / total * 100);
		}


		while(currentPtr != NULL) {
			currentPtr = currentPtr->nextPtr;
		}
		printf("</TABLE><BR><BR>\n");
	}
	printf("These figures only reflect FTP transfers.  There are many sites\n");
	printf("which mount the archives via NFS, and those transfers are not logged\n"); 
	printf("and reported by this program.<BR><BR>\n");
	if (config.hourly_traffic) {
#ifndef NOGRAPH
		image1 = gdImageCreate(280, 280);
		image2 = gdImageCreate(280, 280);
		image3 = gdImageCreate(280, 280);
		black = gdImageColorAllocate(image1, 0, 0, 0);
		white = gdImageColorAllocate(image1, 255, 255, 255);
		blue = gdImageColorAllocate(image1, 135, 206, 235);
		green = gdImageColorAllocate(image1, 46, 139, 87);
		black2 = gdImageColorAllocate(image2, 0, 0, 0);
		white2 = gdImageColorAllocate(image2, 255, 255, 255);
		blue2 = gdImageColorAllocate(image2, 135, 206, 235);
		green2 = gdImageColorAllocate(image2, 46, 139, 87);
		black3 = gdImageColorAllocate(image3, 0, 0, 0);
		white3 = gdImageColorAllocate(image3, 255, 255, 255);
		blue3 = gdImageColorAllocate(image3, 135, 206, 235);
		green3 = gdImageColorAllocate(image3, 46, 139, 87);
		gdImageLine(image1, 30, 15, 30, 249, white);
		gdImageLine(image1, 30, 249, 270, 249, white);
		gdImageLine(image2, 30, 15, 30, 249, white2);
		gdImageLine(image2, 30, 249, 270, 249, white2);
		gdImageLine(image3, 30, 15, 30, 249, white3);
		gdImageLine(image3, 30, 249, 270, 249, white3);
		gdImageString(image1, gdFontTiny, 32, 250, "0   2   4   6   8", white);
		gdImageString(image1, gdFontTiny, 131, 250, "10  12  14  16  18  20  22", white);
		gdImageString(image2, gdFontTiny, 32, 250, "0   2   4   6   8", white2);
		gdImageString(image2, gdFontTiny, 131, 250, "10  12  14  16  18  20  22", white2);
		gdImageString(image3, gdFontTiny, 32, 250, "0   2   4   6   8", white3);
		gdImageString(image3, gdFontTiny, 131, 250, "10  12  14  16  18  20  22", white3);
		gdImageString(image1, gdFontTiny, 155 - 2 * gdFontTiny->w, 260, "Hour", white);
		gdImageString(image2, gdFontTiny, 155 - 2 * gdFontTiny->w, 260, "Hour", white2);
		gdImageString(image3, gdFontTiny, 155 - 2 * gdFontTiny->w, 260, "Hour", white3);
		gdImageStringUp(image1, gdFontTiny, 15, 140 + 12 * gdFontTiny->w, "Percent of Total Traffic", white);
		gdImageStringUp(image2, gdFontTiny, 15, 140 + 13 * gdFontTiny->w, "Average Throughput (in K/s)", white2);
		gdImageStringUp(image3, gdFontTiny, 15, 140 + 13 * gdFontTiny->w, "Maximum Throughput (in K/s)", white3);
#endif
		printf("<H1>Hourly Transmission Statistics</H1><BR>\n");
		printf("<IMG SRC=\"%s%s\"><IMG SRC=\"%s%s\"><BR>", html_path, HOURLY_DATA_GRAPH, html_path, HOURLY_THROUGHPUT_GRAPH);
		printf("<IMG SRC=\"%s%s\">", html_path, HOURLY_MAX_THROUGHPUT_GRAPH);
		printf("<TABLE BORDER=2>\n");
		printf("<TR><TH></TH><TH>Number Of</TH><TH>Number of</TH><TH>Average</TH><TH>Percent Of</TH><TH>Percent Of</TH></TR>\n");
		printf("<TR><TH>Time</TH><TH>Files Sent</TH><TH>Bytes Sent</TH><TH>Xmit Rate</TH><TH>Files Sent</TH><TH>Bytes Sent</TH></TR>\n");
		hibyte = 0;
		for (i = 0; i <= 23; i++) {
			if (hourStruct[i].data > hibyte)
				hibyte = hourStruct[i].data;
			if ((float) hourStruct[i].data / hourStruct[i].seconds / 1024 > hixfer)
				hixfer = (float) hourStruct[i].data / hourStruct[i].seconds / 1024;
#ifndef NOGRAPH
			if (hourStruct[i].hixfer > hixfer2)
				hixfer2 = hourStruct[i].hixfer;
#endif
		}
		for (i = 0; i <= 23; i++)
			if (hourStruct[i].fileCount) {
				printf("<TR><TH>%d</TH><TH ALIGN=RIGHT>%u</TH><TH ALIGN=RIGHT>%lu</TH><TH ALIGN=RIGHT>%6.2f KB/s</TH><TH ALIGN=RIGHT>%10.2f</TH><TH ALIGN=RIGHT>%10.2f</TH></TR>\n",
					i, hourStruct[i].fileCount, hourStruct[i].data, (float) hourStruct[i].data / hourStruct[i].seconds / 1024, (float) hourStruct[i].fileCount / fileCount * 100, (float) hourStruct[i].data / total * 100);
#ifndef NOGRAPH
				if ((i % 2) == 0) {
					gdImageFilledRectangle(image1, 10*i+31, (int) (248 - (float) hourStruct[i].data * 233 / hibyte), 10*(i+1)+30, 248, blue);
					gdImageFilledRectangle(image2, 10*i+31, (int) (248 - (float) hourStruct[i].data * 233 / hourStruct[i].seconds / 1024 / hixfer), 10*(i+1)+30, 248, blue2);
					gdImageFilledRectangle(image3, 10*i+31, (int) (248 - (float) hourStruct[i].hixfer * 233 / hixfer2), 10*(i+1)+30, 248, blue2);
				} else {
					gdImageFilledRectangle(image1, 10*i+31, (int) (248 - (float) hourStruct[i].data * 233 / hibyte), 10*(i+1)+30, 248, green);
					gdImageFilledRectangle(image2, 10*i+31, (int) (248 - (float) hourStruct[i].data * 233 / hourStruct[i].seconds / 1024 / hixfer), 10*(i+1)+30, 248, green2);
					gdImageFilledRectangle(image3, 10*i+31, (int) (248 - (float) hourStruct[i].hixfer * 233 / hixfer2), 10*(i+1)+30, 248, green2);
				}
#endif
			}
		printf("</TABLE><BR><BR>\n");
#ifndef NOGRAPH
		if (total) {
			sprintf(foo, "%3.1f", ((float) hibyte * 100 / total));
			gdImageString(image1, gdFontTiny, 29 - gdFontTiny->w * strlen(foo), 15, foo, white);
			sprintf(foo, "%3.1f", (float) hixfer);
 			gdImageString(image2, gdFontTiny, 29 - gdFontTiny->w * strlen(foo), 15, foo, white2);
			sprintf(foo, "%3.1f", (float) hixfer2);
 			gdImageString(image3, gdFontTiny, 29 - gdFontTiny->w * strlen(foo), 15, foo, white3);
		}
		memset(tempPath, 0, 1024);
		strncpy(tempPath, REAL_GRAPH_PATH, 1024);
		strcat(tempPath, HOURLY_DATA_GRAPH);
		gif = fopen(tempPath, "wb");
		gdImageGif(image1, gif);
		fclose(gif);
		gdImageDestroy(image1);
		memset(tempPath, 0, 1024);
		strncpy(tempPath, REAL_GRAPH_PATH, 1024);
		strcat(tempPath, HOURLY_THROUGHPUT_GRAPH);
		gif = fopen(tempPath, "wb");
		gdImageGif(image2, gif);
		fclose(gif);
		gdImageDestroy(image2);
		memset(tempPath, 0, 1024);
		strncpy(tempPath, REAL_GRAPH_PATH, 1024);
		strcat(tempPath, HOURLY_MAX_THROUGHPUT_GRAPH);
		gif = fopen(tempPath, "wb");
		gdImageGif(image3, gif);
		fclose(gif);
		gdImageDestroy(image3);
#endif
	}
	if (config.most_downloads) {
		whee = 0;
		printf("<H1>Top %d Most Downloaded Files (by number)</H1><BR>\n", config.number_most_downloads);
		printf("<TABLE BORDER=2>\n");
		printf("<TR><TH></TH><TH>Number Of</TH><TH>Percent Of</TH><TH>Percent Of</TH></TR>\n");
		printf("<TR><TH>Filename</TH><TH>Files Sent</TH><TH>Files Sent</TH><TH>Bytes Sent</TH></TR>\n");
		currentFilePtr = *fPtr;
		for (i = 0; currentFilePtr != NULL; i++)
			currentFilePtr = currentFilePtr->nextPtr;
		while (whee < config.number_most_downloads) {
			hibyte = 0;
			currentFilePtr = *fPtr; 
			while(currentFilePtr != NULL) {
				if (!currentFilePtr->sorted && currentFilePtr->fileCount >= hibyte) {
					hiFPtr = currentFilePtr;
					hibyte = currentFilePtr->fileCount;
				}
				currentFilePtr = currentFilePtr->nextPtr;
			}
			if (hibyte == 0)
				break;
			whee++;
			hiFPtr->sorted = 1;
			if (strlen(hiFPtr->name) > 5000) {
				char tempName[55];
				strcpy(tempName, hiFPtr->name + strlen(hiFPtr->name) - 54); 
				printf("...%-54s %5lu %7.2f %7.2f\n", tempName, hibyte, (float) hibyte / fileCount * 100, (float) hiFPtr->data / total * 100);
			} else
				printf("<TR><TH ALIGN=LEFT>%s</TH><TH ALIGN=RIGHT>%lu</TH><TH ALIGN=RIGHT>%7.2f</TH><TH ALIGN=RIGHT>%7.2f</TH></TR>\n", hiFPtr->name, hibyte, (float) hibyte / fileCount * 100, (float) hiFPtr->data / total * 100);
		}
		printf("</TABLE>\n");
	}
} /* displayHTMLTotals */


void
usage(void)
{
	printf("%s\n", VERSION);
	printf("usage: xferstats <options>\n\n"
		" -			grab the logfile from stdin\n"
		" -f <filename>		use <filename> for the log file\n"
		" -T			toggle logfile type (wuftpd or ncftpd)\n"
		" -H			toggle HTML output\n"
		" -r			include real users\n"
		" -a			include anonymous users\n"
		" -h			include report on hourly traffic\n"
		" -d [number]		include report on domain traffic\n"
		" -t [number]		report on total traffic by section\n" 
		" -m [number]		include a \"most downloaded files\" report\n"
		" -L <number>		limit the report-by-day report to [number] listings\n"
		" -A			include all users, all reports\n"
		" -D <domain>		report only on traffic from <domain>\n"
		" -l <number>		depth of path detail for sections (default 2)\n"
		" -s <section>		section to report on, for example: -s /pub will\n"
		"			report only on paths under /pub\n"
		" -v, --version		display version information\n"
		" --help			this listing\n\n"
		"Report bugs to pschwan@apk.net\n");

} /* usage */


void
parse_cmdline(int argc, char *argv[], char *fileStr, struct config_ops * config)
{
	int i = 1, j, realArg = 0;
	char c;

	for (; i < argc; i++) {
		if (!strncmp(argv[i], "--", 2)) {
                                       /* argv[i] contains a long option */
			if (!strcmp(argv[i]+2, "help")) {
				usage();
				exit(0);
			} else if (!strcmp(argv[i]+2, "version")) {
				printf("%s\n", VERSION);
				printf("Copyright (C) 1997 Phil Schwan\n");
#ifndef NOGRAPH
				printf("Using the graphics library gd 1.2\n");
#endif
				exit(0);
			}
		} else if (*argv[i] == '-') {
                                       /* argv[i] contains a short option */
			j = 1;
			if (*(argv[i] + 1) == '\0') config->use_stdin = 1;
			while ((c = *(argv[i] + j)) != '\0') {
				switch (c) {
					case 'f':
						if (i < argc - 1 && *argv[i + 1] != '-')
							strcpy(fileStr,argv[i + 1]);
						else {
							fprintf(stderr, "fatal: the -f parameter requires a filename following it.\n");
							exit(1);
						}
						break;
					case 'T':
						if (config->log_style == 0) config->log_style = 1;
						else config->log_style = 0;
						break;
					case 'H':
						if (config->html_output) config->html_output = 0;
						else config->html_output = 1;
						break;
					case 'r':
						realArg = 1;
						config->real_traffic = 2;
						break;
					case 'a':
						realArg = 1;
						config->anon_traffic = 2;
						break;
					case 'h':
						realArg = 1;
						config->hourly_traffic = 2;
						break;
					case 'd':
						realArg = 1;
						config->top_domain_traffic = 2;
						if (i < argc - 1 && *argv[i + 1] != '-') 
							if ((config->number_top_domain_stats = atoi(argv[i + 1])) < 0) {
								fprintf(stderr, "fatal: the -d parameter accepts only an optional number following it.\n");
								exit(1);
							}
						break;
					case 't':
						realArg = 1;
						config->section_traffic = 2;
						if (i < argc - 1 && *argv[i + 1] != '-')
							if ((config->number_section_stats = atoi(argv[i + 1])) < 0) {
								fprintf(stderr, "fatal: the -t parameter accepts only an optional number following it.\n");
								exit(1);
							}
						break;
					case 'm':
						realArg = 1;
						config->most_downloads = 2;
						if (i < argc - 1 && *argv[i+1] != '-')
							if ((config->number_most_downloads = atoi(argv[i + 1])) < 1) {
								fprintf(stderr, "fatal: the -m parameter accepts only an optional number following it.\n");
								exit(1);
							}
						break;
					case 'L':
						if (i < argc - 1 && *argv[i + 1] != '-') {
							if ((config->number_daily_stats = atoi(argv[i + 1])) < 0) {
								fprintf(stderr, "fatal: the -L parameter requires a number greater than or equal to zero following it.\n");
								exit(1);
							}
						} else {
							fprintf(stderr, "fatal: the -L parameter requires a number following it.\n");
							exit(1);
						}
						break;
					case 'D':
						if (i < argc - 1 && *argv[i+1] != '-')
							strncpy(config->only_domain, argv[i+1], 10);
						else {
							fprintf(stderr, "fatal: the -D parameter requires a top-level domain following it.\n");
							exit(1);
						}
						printf("Transfer totals include the '%s' domain only.\n", config->only_domain);
						printf("All other domains are filtered out for this report.\n\n");
						break;
					case 'l':
						if (i < argc  - 1 && *argv[i+1] != '-')
							if ((config->depth = atoi(argv[i + 1]) < 1))
								fprintf(stderr, "fatal: the -l parameter accepts only a number greater than zero following it.\n");
						else {
							fprintf(stderr, "fatal: the -l parameter requires a number following it.\n");
							exit(1);
						}
						break;
					case 's':
						if (i < argc - 1 && *argv[i+1] != '-')
							if (*argv[i+1] != '/') {
								strcpy(config->only_section, "/");
								strcat(config->only_section, argv[i + 1]);
							} else
								strncpy(config->only_section, argv[i + 1], 254);
						else {
							fprintf(stderr, "fatal: the -s parameter requires a path following it.\n");
							exit(1);
						}
						printf("Transfer totals include the '%s' section only.\n", config->only_section);
						printf("All other sections are filtered out for this report.\n\n");
						break;
					case 'v':
						fprintf(stderr, "%s\n", VERSION);
						exit(1);
					case 'A':
						realArg = 1;
#ifdef DEFAULT_ANON
						config->anon_traffic = 2;
#endif
#ifdef DEFAULT_HOURLY
						config->hourly_traffic = 2;
#endif
#ifdef DEFAULT_TOP_DOMAIN
						config->top_domain_traffic = 2;
#endif
#ifdef DEFAULT_SECTION
						config->section_traffic = 2;
#endif
#ifdef DEFAULT_REAL
						config->real_traffic = 2;
#endif
#ifdef DEFAULT_MOSTDL
						config->most_downloads = 2;
#endif
						break;
					default:
						usage();
						exit(1);
				}
				j++;
			}
		}
	}
	if (realArg) {
		if (config->real_traffic < 2) config->real_traffic = 0;
		if (config->anon_traffic < 2) config->anon_traffic = 0;
		if (config->hourly_traffic < 2) config->hourly_traffic = 0;
		if (config->top_domain_traffic < 2) config->top_domain_traffic = 0;
		if (config->section_traffic < 2) config->section_traffic = 0;
		if (config->most_downloads < 2) config->most_downloads = 0;
	}
} /* parse_cmdline */


void
main(int argc, char **argv)
{
	struct ftp_entry ftp_line;
	struct config_ops config;
	DOMPTR startPtr = NULL;
	DAYPTR firstDayPtr = NULL;
	DIRPTR firstDirPtr = NULL;
	FILEPTR firstFilePtr = NULL;
	char *fileStr = FILENAME, domain[11], lowDate[25] = {0};
	char highDate[25] = {0};
	FILE *filePtr;
	unsigned int fileCount = 0;
	unsigned long total = 0, totalk = 0;

	int file_fd, parse_value = 1;
	struct stat stat_buf;

	setbuf(stdout, NULL);

        config.number_most_downloads = NUMBER_MOSTDL_STATS;
        config.html_output = DEFAULT_HTML;
	config.number_daily_stats = NUMBER_DAILY_STATS;
	config.number_section_stats = NUMBER_SECTION_STATS;
	config.number_top_domain_stats = NUMBER_DOMAIN_STATS;
	config.log_style = DEFAULT_LOG_TYPE;
#ifdef DEFAULT_REAL
	config.real_traffic = 1;
#else
	config.real_traffic = 0;
#endif
#ifdef DEFAULT_ANON
	config.anon_traffic = 1;
#else
	config.anon_traffic = 0;
#endif
#ifdef DEFAULT_TOP_DOMAIN
	config.top_domain_traffic = 1;
#else
	config.top_domain_traffic = 0;
#endif
#ifdef DEFAULT_HOURLY
	config.hourly_traffic = 1;
#else
	config.hourly_traffic = 0;
#endif
#ifdef DEFAULT_SECTION
	config.section_traffic = 1;
#else
	config.section_traffic = 0;
#endif
#ifdef CONFIG_MOSTDL
	config.most_downloads = 1;
#else
	config.most_downloads = 0;
#endif
	config.use_stdin = 0;
	config.depth = 4;
	memset(config.only_section, 0, sizeof(config.only_section));
	memset(config.only_domain, 0, sizeof(config.only_domain));

	parse_cmdline(argc, argv, fileStr, &config);

	if (config.use_stdin) filePtr = stdin;
	else filePtr = fopen(fileStr, "r");
        if (filePtr == NULL) {
		perror(fileStr);
		exit(EXIT_FAILURE);
        }

	stat(fileStr, &stat_buf);
	if ((file_fd = open(fileStr, O_RDONLY)) < 0) {
		perror("file_type: open");
		exit(1);
	}
	if ((config.map = (char *) mmap(NULL, stat_buf.st_size + 1, PROT_READ,
				      MAP_SHARED, file_fd, 0)) < 0) {
		perror("mmap");
		exit(1);
	}

	do {
		if ((parse_value = parseline(&ftp_line, &config)) > 0) {
			char c[11] = {0}, *d;

#ifdef DEBUG
			printf("%s ftped %s\n", ftp_line.host, ftp_line.file_name);
#endif
			if (strchr(ftp_line.host, '.')) {
				strncpy(c, strrchr(ftp_line.host, '.') + 1, 4);
#ifdef DEBUG
				printf("Last letter of domain: %d:%c\n", c[strlen(c)-1], c[strlen(c)-1]);
#endif
				if (isdigit(c[strlen(c)-1])) strcpy(c, "unresolved");
			} else
				strcpy(c, "unresolved");
			for(d = c; (*d = tolower(*d)); d++);
			strncpy(domain, c, 10);
			domain[10] = '\0';
			if (((config.only_domain[0] == '\0' || !strcmp(config.only_domain, domain))) && (config.only_section[0] == '\0' || !strncmp(ftp_line.file_name, config.only_section, strlen(config.only_section)))) {
				total += ftp_line.data;
				if (total > 3e9) {
					totalk += (total / 1024);
					total = 0;
				}
				fileCount++;
				if ((config.real_traffic && ftp_line.user_type == 'r') || (config.anon_traffic && (ftp_line.user_type == 'a' || ftp_line.user_type == 'g'))) {
					if (!doDomain(&startPtr, domain, ftp_line)) {
						printf("fatal: could not doDomain for domain \"%s\".\n", domain);
						exit(2);
					}
					if (!doDaily(&firstDayPtr, ftp_line)) {
						printf("fatal: could not doDaily.\n");
						exit(2);
					}
					if (!doDirs(&firstDirPtr, &firstFilePtr, ftp_line, config)) {
						printf("fatal: could not doDirs.\n");
						exit(2);
					}
					if (config.hourly_traffic) doHours(ftp_line);
					if (!strcmp(lowDate, "")) {
						strncpy(lowDate, ftp_line.date, 24);
						lowDate[24] = '\0';
					} else if (compareDates(ftp_line.date, lowDate) < 0) {
						strncpy(lowDate, ftp_line.date, 24);
						lowDate[24] = '\0';
					}
					if (!strcmp(highDate, "")) {
						strncpy(highDate, ftp_line.date, 24);
						highDate[24] = '\0';
					} else if (compareDates(ftp_line.date, highDate) > 0) {
						strncpy(highDate, ftp_line.date, 24);
						highDate[24] = '\0';
					}
				}
			}
		}
	} while (parse_value >= 0);
	if (startPtr != NULL)
		if (config.html_output)
			displayHTMLTotals(&startPtr, &firstDayPtr, &firstDirPtr, &firstFilePtr, config, lowDate, highDate, fileCount, total, totalk);
		else
			displayTotals(&startPtr, &firstDayPtr, &firstDirPtr, &firstFilePtr, config, lowDate, highDate, fileCount, total, totalk);
	else
		printf("There was no data to process.\n");
}
