/***************************************
  $Header: /home/amb/wwwoffle/src/RCS/wwwoffles.c 2.222 2002/11/28 18:53:19 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 2.7g.
  A server to fetch the required pages.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1996,97,98,99,2000,01,02 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include "autoconfig.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <unistd.h>
#include <errno.h>

#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#include <sys/stat.h>
#include <signal.h>
#include <fcntl.h>

#include "wwwoffle.h"
#include "document.h"
#include "misc.h"
#include "proto.h"
#include "config.h"
#include "sockets.h"
#include "errors.h"


static void uninstall_sighandlers(void);


/*+ The mode of operation of the server. +*/
typedef enum _Mode
{
 None,                          /*+ Undecided. +*/

 Real,                          /*+ From server host to cache and client. +*/
 RealNoCache,                   /*+ From server host to client. +*/
 RealRefresh,                   /*+ Refresh the page, forced from index. +*/
 RealNoPassword,                /*+ From server host to cache, not using supplied password. +*/
 SpoolOrReal,                   /*+ Spool if already cached else Real. +*/

 Fetch,                         /*+ From server host to cache. +*/
 FetchNoPassword,               /*+ From server host to cache, not using supplied password. +*/

 Spool,                         /*+ From cache to client. +*/
 SpoolGet,                      /*+ Not in cache so record request in outgoing. +*/
 SpoolRefresh,                  /*+ Refresh the page, forced from refresh page. +*/
 SpoolPragma,                   /*+ Refresh the page, forced from browser by 'Pragma: no-cache'. +*/

 SpoolInternal                  /*+ The page is one that has been internally generated by WWWOFFLE. +*/
}
Mode;


/*++++++++++++++++++++++++++++++++++++++
  The main server program.

  int wwwoffles Returns the exit status.

  int online Whether the demon is online or not.

  int browser Set to true if there is a browser.

  int client The file descriptor of the client.
  ++++++++++++++++++++++++++++++++++++++*/

int wwwoffles(int online,int browser,int client)
{
 int outgoing=-1,spool=-1,tmpclient=-1,is_server=0;
 int fetch_again=0;
 char *proxy_auth,*proxy_user;
 char *aliasProto=NULL,*aliasHost=NULL,*aliasPath=NULL;
 Header *request_head=NULL,*reply_head=NULL;
 Body *request_body=NULL,*reply_body=NULL;
 int reply_status=-1,head_only=0;
 char *url;
 URL *Url=NULL,*Urlpw=NULL;
 Mode mode=None;
 int outgoing_exists=0,lasttime_exists=0;
 time_t spool_exists=0,spool_exists_pw=0;
 int conditional_request_ims=0,conditional_request_inm=0;
 char *user_agent;
 int offline_request=1;
 int is_client_wwwoffle=0;
 int is_client_searcher=0;
#if USE_ZLIB
 int client_compression=0;
 int request_compression=0;
 int server_compression=0;
#endif


 /*----------------------------------------
   Initialise things, work out the mode from the options.
   ----------------------------------------*/

 uninstall_sighandlers();

 InitErrorHandler("wwwoffles",-1,-1); /* change name nothing else */

 if(online==1 && browser)
    mode=Real;
 else if(online!=0 && !browser)
    mode=Fetch;
 else if(online==-1 && browser)
    mode=SpoolOrReal;
 else if(!online && browser)
    mode=Spool;
 else
    PrintMessage(Fatal,"Started in a mode that is not allowed (online=%d, browser=%d).",online,browser);


 /*----------------------------------------
   mode = Spool, Real, SpoolOrReal or Fetch

   Set up the input file to read the request from, either client or stored in outgoing.
   ----------------------------------------*/

 /* Check the client file descriptor (browser connection). */

 if(client==-1 && mode!=Fetch)
    PrintMessage(Fatal,"Cannot use client file descriptor %d.",client);

 /* Open the temporary client file for writing to while processing. */

 if(mode!=Fetch)
   {
    tmpclient=CreateTempSpoolFile();
    init_buffer(tmpclient);

    if(tmpclient==-1)
      {
       PrintMessage(Warning,"Cannot open temporary spool file; [%!s]."); /* Used in audit-usage.pl */
       HTMLMessage(client,500,"WWWOFFLE Server Error",NULL,"ServerError",
                   "error","Cannot open temporary file.",
                   NULL);
       exit(1);
      }
   }

 /* Open up the outgoing file for reading the stored request from. */

 else
   {
    outgoing=OpenOutgoingSpoolFile(1);

    if(outgoing==-1)
      {
       PrintMessage(Inform,"No more outgoing requests.");
       exit(3);
      }

    init_buffer(outgoing);
   }


 /*----------------------------------------
   mode = Spool, Real, SpoolOrReal or Fetch

   Parse the request and make some checks on the type of request and request options.
   ----------------------------------------*/

 /* Get the URL from the request and read the header and body (if present). */

 if(mode==Real || mode==Spool || mode==SpoolOrReal)
    url=ParseRequest(client,&request_head,&request_body);
 else /* mode==Fetch */
    url=ParseRequest(outgoing,&request_head,&request_body);

 if(DebuggingLevel==ExtraDebug)
   {
    if(request_head)
       PrintMessage(ExtraDebug,"Incoming Request Head (from browser)\n%s",HeaderString(request_head));
    else
       PrintMessage(ExtraDebug,"Incoming Request Head (from browser) is empty");

    if(request_body)
      {
       if(strcmp(request_head->method,"POST")) /* only POST is guaranteed to be ASCII. */
          PrintMessage(ExtraDebug,"Incoming Request Body (from browser) is %d bytes of binary data\n",request_body->length);
       else
          PrintMessage(ExtraDebug,"Incoming Request Body (from browser)\n%s",request_body->content);
      }
   }

 /* Check that a URL was read, means a valid header was received. */

 if(!url)
   {
    PrintMessage(Warning,"Could not parse HTTP request (%s).",request_head?"Parse error":"Empty request"); /* Used in audit-usage.pl */
    if(mode!=Fetch)
       HTMLMessage(client,500,"WWWOFFLE Server Error",NULL,"ServerError",
                   "error",request_head?"Cannot parse the HTTP request":"The HTTP request was empty",
                   NULL);
    exit(1);
   }

 Url=SplitURL(url);

 /* Check if the browser can use compression and work out which type. */

#if USE_ZLIB
 if(ConfigBoolean(ReplyCompressedData))
   {
    char *content_encoding=GetHeader(request_head,"Accept-Encoding");

    if(content_encoding)
      {
       client_compression=WhichCompression(content_encoding);

       PrintMessage(Debug,"Client can 'Accept-Encoding: %s', using %s.",content_encoding,
                    client_compression==1?"deflate":client_compression==2?"gzip":"identity");
      }
   }
#endif

 /* Store the browsers language preferences for message page generation. */

 SetLanguage(GetHeader(request_head,"Accept-Language"));

 /* Check the request's proxy authentication. */

 proxy_auth=GetHeader(request_head,"Proxy-Authorization");
 proxy_user=IsAllowedConnectUser(proxy_auth);

 if(!proxy_user)
   {
    PrintMessage(Inform,"HTTP Proxy connection rejected from unauthenticated user."); /* Used in audit-usage.pl */
    HTMLMessageHead(tmpclient,407,"WWWOFFLE Proxy Authentication Required",
                    "Proxy-Authenticate","Basic realm=\"wwwoffle-proxy\"",
                    NULL);
    HTMLMessageBody(tmpclient,"ProxyAuthFail",
                    NULL);
    mode=SpoolInternal; goto spoolinternal;
   }
 else if(proxy_auth)
    PrintMessage(Inform,"HTTP Proxy connection from user '%s'.",proxy_user); /* Used in audit-usage.pl */

 /* Check the HTTP method that is requested. */

 if(!strcasecmp(request_head->method,"CONNECT"))
   {
    PrintMessage(Inform,"SSL='%s'.",Url->host); /* Used in audit-usage.pl */

    if(mode==Real || mode==SpoolOrReal || (mode==Spool && IsLocalNetHost(Url->host)))
      {
       if(IsSSLAllowedPort(Url->host) && !ConfigBooleanMatchURL(DontGet,Url))
         {
          char *err=SSL_Open(Url);

          if(err && ConfigBoolean(ConnectRetry))
            {
             PrintMessage(Inform,"Waiting to try connection again.");
             sleep(10);
             err=SSL_Open(Url);
            }

          if(err)
            {
             HTMLMessage(client,503,"WWWOFFLE Remote Host Error",NULL,"RemoteHostError",
                         "url",Url->host,
                         "reason",err,
                         "cache",NULL,
                         "backup",NULL,
                         NULL);
             mode=SpoolInternal; goto spoolinternal;
            }

          err=SSL_Request(client,Url,request_head);

          if(err)
            {
             HTMLMessage(client,503,"WWWOFFLE Remote Host Error",NULL,"RemoteHostError",
                         "url",Url->host,
                         "reason",err,
                         "cache",NULL,
                         "backup",NULL,
                         NULL);
             mode=SpoolInternal; goto spoolinternal;
            }

          SSL_Transfer(client);

          SSL_Close();

          CloseTempSpoolFile(tmpclient);
          exit(0);
         }
       else
         {
          PrintMessage(Warning,"A SSL proxy connection for %s was received but is not allowed.",Url->host);
          HTMLMessage(tmpclient,500,"WWWOFFLE Server Error",NULL,"ServerError",
                      "error","SSL proxy connection to specified port is not allowed.",
                      NULL);
          mode=SpoolInternal; goto spoolinternal;
         }
      }
    else /* mode==Fetch || mode==Spool */
      {
       PrintMessage(Warning,"A SSL proxy connection for %s was received but wwwoffles is in wrong mode.",Url->host);
       if(mode==Fetch)
         {
          if(client!=-1)
             write_formatted(client,"Cannot fetch %s [HTTP method 'CONNECT' not supported in this mode]\n",Url->name);
          exit(1);
         }
       else
         {
          HTMLMessage(tmpclient,500,"WWWOFFLE Server Error",NULL,"ServerError",
                      "error","SSL proxy connection while offline is not allowed.",
                      NULL);
          mode=SpoolInternal; goto spoolinternal;
         }
      }
   }
 else if(strcmp(request_head->method,"GET") &&
         strcmp(request_head->method,"HEAD") &&
         strcmp(request_head->method,"POST") &&
         strcmp(request_head->method,"PUT"))
   {
    PrintMessage(Warning,"The requested method '%s' is not supported.",request_head->method);
    if(mode==Fetch)
      {
       if(client!=-1)
          write_formatted(client,"Cannot fetch %s [The %s method is not supported]\n",Url->name,request_head->method);
       exit(1);
      }
    else
      {
       HTMLMessage(tmpclient,501,"WWWOFFLE Method Unsupported",NULL,"MethodUnsupported",
                   "method",request_head->method,
                   "protocol","all",
                   NULL);
       mode=SpoolInternal; goto spoolinternal;
      }
   }
 else if(!strcmp(request_head->method,"HEAD"))
   {
    strcpy(request_head->method,"GET");
    request_head->size-=1;
    head_only=1;
   }


 /*----------------------------------------
   mode = Spool, Real, SpoolOrReal or Fetch

   Modify the URL requested based on Alias and DontGet sections of the configuration file.
   ----------------------------------------*/

 PrintMessage(Inform,"URL='%s'%s.",Url->name,Url->user?" (With username/password)":""); /* Used in audit-usage.pl */
 PrintMessage(Debug,"proto='%s'; host='%s'; path='%s'; args='%s'; user:pass='%s:%s'.",
              Url->proto,Url->host,Url->path,Url->args,Url->user,Url->pass);

 /* Check for an alias. */

 if(IsAliased(Url->proto,Url->host,Url->path,&aliasProto,&aliasHost,&aliasPath))
   {
    URL *newUrl;
    char *newurl=(char*)malloc(strlen(aliasProto)+strlen(aliasHost)+strlen(aliasPath)-
                               strlen(Url->path)+strlen(Url->pathp)+8);

    sprintf(newurl,"%s://%s%s%s",aliasProto,aliasHost,aliasPath,Url->pathp+strlen(Url->path));

    newUrl=SplitURL(newurl);

    if(Url->user)
       AddURLPassword(newUrl,Url->user,Url->pass);

    PrintMessage(Inform,"Aliased URL='%s'%s.",newUrl->name,newUrl->user?" (With username/password)":"");
    PrintMessage(Debug,"Aliased proto='%s'; host='%s'; path='%s'; args='%s'; user:pass='%s:%s'.",
                 newUrl->proto,newUrl->host,newUrl->path,newUrl->args,newUrl->user,newUrl->pass);

    HTMLMessage(tmpclient,302,"WWWOFFLE Alias Redirect",newUrl->file,"Redirect",
                "location",newUrl->file,
                NULL);
    mode=SpoolInternal; goto spoolinternal;
   }

 /* Check for a DontGet URL and its replacement. */
 /* Must be here so that local URLs can be in the DontGet section. */

 if(ConfigBooleanMatchURL(DontGet,Url))
   {
    char *replace=ConfigStringURL(DontGetReplacementURL,Url);

    PrintMessage(Inform,"The URL '%s' matches one in the list not to get.",Url->name);

    if(!replace)
      {
       if(mode==Fetch)
          exit(0);
       else
         {
          HTMLMessage(tmpclient,404,"WWWOFFLE Host Not Got",NULL,"HostNotGot",
                      "url",Url->name,
                      NULL);
          mode=SpoolInternal; goto spoolinternal;
         }
      }
    else
      {
       char *newurl=(char*)malloc(strlen(replace)+1);
       URL *newUrl=SplitURL(replace);

       strcpy(newurl,replace);

       free(url);
       url=newurl;
       FreeURL(Url);
       Url=newUrl;

       PrintMessage(Inform,"Replaced URL='%s'%s.",newUrl->name,newUrl->user?" (With username/password)":"");
       PrintMessage(Debug,"Replaced proto='%s'; host='%s'; path='%s'; args='%s'; user:pass='%s:%s'.",
                    newUrl->proto,newUrl->host,newUrl->path,newUrl->args,newUrl->user,newUrl->pass);
      }
   }

 /* Is the specified protocol valid? */

 if(!Url->Protocol)
   {
    PrintMessage(Inform,"The protocol '%s' is not available.",Url->proto);
    if(mode==Fetch)
      {
       if(client!=-1)
          write_formatted(client,"Cannot fetch %s [Protocol not available]\n",Url->name);
       exit(1);
      }
    else
      {
       HTMLMessage(tmpclient,404,"WWWOFFLE Illegal Protocol",NULL,"IllegalProtocol",
                   "url",Url->name,
                   "protocol",Url->proto,
                   NULL);
       mode=SpoolInternal; goto spoolinternal;
      }
   }

 /* Can a POST or PUT request be made with this protocol? */

 if((!strcmp(request_head->method,"POST") && !Url->Protocol->postable) ||
    (!strcmp(request_head->method,"PUT") && !Url->Protocol->putable))
   {
    PrintMessage(Warning,"The requested method '%s' is not supported for the %s protocol.",request_head->method,Url->Protocol->name);
    if(mode==Fetch)
      {
       if(client!=-1)
          write_formatted(client,"Cannot fetch %s [The %s method is not supported for %s]\n",Url->name,request_head->method,Url->Protocol->name);
       exit(1);
      }
    else
      {
       HTMLMessage(tmpclient,501,"WWWOFFLE Method Unsupported",NULL,"MethodUnsupported",
                   "method",request_head->method,
                   "protocol",Url->Protocol->name,
                   NULL);
       mode=SpoolInternal; goto spoolinternal;
      }
   }


 /*----------------------------------------
   mode = Spool, Real, SpoolOrReal or Fetch

   Handle the local URLs, all built-in web pages.
   ----------------------------------------*/

 if(Url->local)
   {
    /* Refresh requests, recursive fetches, refresh options etc. */

    if(!strncmp("/refresh",Url->path,8))
      {
       int recurse=0;
       char *newurl=RefreshPage(tmpclient,Url,request_body,&recurse);
       URL *newUrl;

       /* An internal page ( /refresh-options/ ) or some sort of error. */

       if(!newurl)
         {
          if(mode==Fetch)
            {
             if(client!=-1)
                write_formatted(client,"Cannot fetch %s [Error with refresh URL]\n",Url->name);
             exit(1);
            }
          else
            {
             mode=SpoolInternal; goto spoolinternal;
            }
         }

       /* A new URL to get to replace the original one. */

       newUrl=SplitURL(newurl);

       PrintMessage(Inform,"Refresh newUrl='%s'%s.",newUrl->name,newUrl->user?" (With username/password)":"");
       PrintMessage(Debug,"Refresh proto='%s'; host='%s'; path='%s'; args='%s'; user:pass='%s:%s'.",
                    newUrl->proto,newUrl->host,newUrl->path,newUrl->args,newUrl->user,newUrl->pass);

       /* The new URL was originally a POST or PUT request. */

       if(newUrl->args && *newUrl->args=='!')
         {
          PrintMessage(Inform,"It is not possible to refresh a URL that used the POST/PUT method.");
          if(mode==Fetch)
            {
             if(client!=-1)
                write_formatted(client,"Cannot fetch %s [Reply from a POST/PUT method]\n",Url->name);
             exit(1);
            }
          else
            {
             HTMLMessage(tmpclient,404,"WWWOFFLE Cant Refresh POST/PUT",NULL,"CantRefreshPosted",
                         "url",newUrl->name,
                         NULL);
             mode=SpoolInternal; goto spoolinternal;
            }
         }

       /* A recursive URL ( /refresh-recurse/ ) in Fetch mode is fetched. */

       if(recurse==1 && mode==Fetch)
         {
          fetch_again=1;
         }

       /* A recursive URL ( /refresh-recurse/ ) in other modes is unchanged. */

       else if(recurse==1)
         {
          fetch_again=1;

          mode=SpoolGet;

          free(newurl);
          FreeURL(newUrl);
          newurl=url;
          newUrl=Url;
         }

       /* A recursive URL ( /refresh-request/ ) in other modes is requested. */

       else if(recurse==-1)
         {
          fetch_again=1;

          mode=SpoolGet;

          FreeHeader(request_head);
          request_head=RequestURL(newUrl,NULL);
          if(request_body)
             FreeBody(request_body);
          request_body=NULL;
         }

       /* A non-recursive URL ( /refresh/ ) in spool mode needs refreshing (redirect to real page). */

       else if(mode==Spool)
          mode=SpoolRefresh;

       /* A non-recursive URL ( /refresh/ ) in online mode needs refreshing (redirect to real page). */

       else if(mode==Real || mode==SpoolOrReal)
         {
          DeleteWebpageSpoolFile(newUrl,0);
          mode=RealRefresh;
         }

       /* A non-recursive URL ( /refresh/ ) or wrong sort of recursive URL ( /refresh-request/ )
          in Fetch mode should not happen. */

       else /* if(mode==Fetch) */
         {
          if(client!=-1)
             write_formatted(client,"Cannot fetch %s [Error with refresh URL]\n",Url->name);
          exit(1);
         }

       /* Swap to the new URL if it is different. */

       if(url!=newurl)
         {
          if(Url->user)
             AddURLPassword(newUrl,Url->user,Url->pass);

          free(url);
          url=newurl;
          FreeURL(Url);
          Url=newUrl;

          PrintMessage(Inform,"Refresh URL='%s'%s.",Url->name,Url->user?" (With username/password)":"");
          PrintMessage(Debug,"Refresh proto='%s'; host='%s'; path='%s'; args='%s'; user:pass='%s:%s'.",
                       Url->proto,Url->host,Url->path,Url->args,Url->user,Url->pass);
         }
      }

    /* Cannot fetch a local page except for refresh requests handled above. */

    else if(mode==Fetch)
      {
       PrintMessage(Inform,"The request to fetch a page from the local host is ignored.");
       if(client!=-1)
          write_formatted(client,"Cannot fetch %s [On local host]\n",Url->name);
       exit(1);
      }

    /* The index pages. */

    else if(!strncmp("/index/",Url->path,7))
      {
       IndexPage(tmpclient,Url);
       mode=SpoolInternal; goto spoolinternal;
      }

    /* The info pages. */

    else if(!strncmp("/info/",Url->path,6))
      {
       InfoPage(tmpclient,Url,request_head,request_body);
       mode=SpoolInternal; goto spoolinternal;
      }

    /* The control pages, deleting and URL replacements for wwwoffle program. */

    else if(!strncmp("/control/",Url->path,9))
      {
       ControlPage(tmpclient,Url,request_body);
       mode=SpoolInternal; goto spoolinternal;
      }

    /* The configuration editing pages. */

    else if(!strncmp("/configuration/",Url->path,15))
      {
       ConfigurationPage(tmpclient,Url,request_body);
       mode=SpoolInternal; goto spoolinternal;
      }

    /* The monitor request and options pages, (note that there is no trailing '/'). */

    else if(!strncmp("/monitor",Url->path,8))
      {
       MonitorPage(tmpclient,Url,request_body);
       mode=SpoolInternal; goto spoolinternal;
      }

    /* The search pages. */

    else if(!strncmp("/search/",Url->path,8))
      {
       SearchPage(tmpclient,Url,request_head,request_body);
       mode=SpoolInternal; goto spoolinternal;
      }

    /* The local pages in the root or local directories. */

    else if(!strchr(Url->path+1,'/') || !strncmp("/local/",Url->path,7))
      {
       LocalPage(tmpclient,Url,request_head,request_body);
       mode=SpoolInternal; goto spoolinternal;
      }

    /* Check for pages like '/http/www.foo/bar.html' and transform to 'http://www.foo/bar.html'. */

    else
      {
       int i;

       for(i=0;i<NProtocols;i++)
          if(!strncmp(Protocols[i].name,Url->pathp+1,strlen(Protocols[i].name)) &&
             Url->pathp[strlen(Protocols[i].name)+1]=='/')
            {
             free(url);
             url=(char*)malloc(strlen(Url->pathp)+4);
             Url->pathp[strlen(Protocols[i].name)+1]=0;
             sprintf(url,"%s://%s",Url->pathp+1,&Url->pathp[strlen(Protocols[i].name)+2]);
             FreeURL(Url);
             Url=SplitURL(url);
             break;
            }

       /* Not found */

       if(i==NProtocols)
         {
          PrintMessage(Inform,"The requested URL '%s' does not exist on the local server.",Url->pathp);
          HTMLMessage(tmpclient,404,"WWWOFFLE Page Not Found",NULL,"PageNotFound",
                      "url",Url->name,
                      NULL);
          mode=SpoolInternal; goto spoolinternal;
         }
      }

    /* Check for a DontGet URL and its replacement. (duplicated code from above.) */

    if(ConfigBooleanMatchURL(DontGet,Url))
      {
       char *replace=ConfigStringURL(DontGetReplacementURL,Url);

       PrintMessage(Inform,"The URL '%s' matches one in the list not to get.",Url->name);

       if(!replace)
         {
          if(mode==Fetch)
             exit(0);
          else
            {
             HTMLMessage(tmpclient,404,"WWWOFFLE Host Not Got",NULL,"HostNotGot",
                         "url",Url->name,
                         NULL);
             mode=SpoolInternal; goto spoolinternal;
            }
         }
       else
         {
          char *newurl=(char*)malloc(strlen(replace)+1);
          URL *newUrl=SplitURL(replace);

          strcpy(newurl,replace);

          free(url);
          url=newurl;
          FreeURL(Url);
          Url=newUrl;

          PrintMessage(Inform,"Replaced URL='%s'%s.",newUrl->name,newUrl->user?" (With username/password)":"");
          PrintMessage(Debug,"Replaced proto='%s'; host='%s'; path='%s'; args='%s'; user:pass='%s:%s'.",
                       newUrl->proto,newUrl->host,newUrl->path,newUrl->args,newUrl->user,newUrl->pass);
         }
      }

    /* Is the specified protocol valid? (duplicated code from above.) */

    if(!Url->Protocol)
      {
       PrintMessage(Inform,"The protocol '%s' is not available.",Url->proto);
       if(mode==Fetch)
         {
          if(client!=-1)
             write_formatted(client,"Cannot fetch %s [Protocol not available]\n",Url->name);
          exit(1);
         }
       else
         {
          HTMLMessage(tmpclient,404,"WWWOFFLE Illegal Protocol",NULL,"IllegalProtocol",
                      "url",Url->name,
                      "protocol",Url->proto,
                      NULL);
          mode=SpoolInternal; goto spoolinternal;
         }
      }

    /* Can a POST or PUT request be made with this protocol? (duplicated code from above.) */

    if((!strcmp(request_head->method,"POST") && !Url->Protocol->postable) ||
       (!strcmp(request_head->method,"PUT") && !Url->Protocol->putable))
      {
       PrintMessage(Warning,"The requested method '%s' is not supported for the %s protocol.",request_head->method,Url->Protocol->name);
       if(mode==Fetch)
         {
          if(client!=-1)
             write_formatted(client,"Cannot fetch %s [The %s method is not supported for %s]\n",Url->name,request_head->method,Url->Protocol->name);
          exit(1);
         }
       else
         {
          HTMLMessage(tmpclient,501,"WWWOFFLE Method Unsupported",NULL,"MethodUnsupported",
                      "method",request_head->method,
                      "protocol",Url->Protocol->name,
                      NULL);
          mode=SpoolInternal; goto spoolinternal;
         }
      }
   }

 /* The special case info pages based on the Referer header. */

 else
   {
    char *referer=GetHeader(request_head,"Referer");

    if(referer && strstr(referer,"/info/request"))
      {
       URL *refUrl=SplitURL(referer);

       if(refUrl->local && !strncmp(refUrl->path,"/info/request",13))
         {
          InfoPage(tmpclient,Url,request_head,request_body);
          FreeURL(refUrl);
          mode=SpoolInternal; goto spoolinternal;
         }

       FreeURL(refUrl);
      }
   }

 /* If not a local request then setup the default Fetch options. */

 if(mode==Fetch)
    DefaultRecurseOptions(Url);


 /*----------------------------------------
   mode = Spool, SpoolGet, SpoolRefresh, Real, RealRefresh, SpoolOrReal or Fetch

   Check for a username / password.
   ----------------------------------------*/

 if(Url->user && ConfigBooleanURL(TryWithoutPassword,Url))
   {
    URL *new=SplitURL(Url->name);

    Urlpw=Url;
    Url=new;

    if(!Urlpw->pass)
      {
       if(mode==Fetch)
         {
          FreeURL(Urlpw);
          Urlpw=NULL;
         }
       else
         {
          HTMLMessage(tmpclient,403,"WWWOFFLE Username Needs Password",NULL,"UserNeedsPass",
                      "url",Urlpw->name,
                      "user",Urlpw->user,
                      NULL);
          mode=SpoolInternal; goto spoolinternal;
         }
      }
   }


 /*----------------------------------------
   mode = Spool, SpoolGet, SpoolRefresh, Real, RealRefresh, SpoolOrReal or Fetch

   Check for an existing cached version, extra checks if there is a password.
   ----------------------------------------*/

 outgoing_exists=ExistsOutgoingSpoolFile(Url);
 spool_exists=ExistsWebpageSpoolFile(Url);

 if(Urlpw)
   {
    spool_exists_pw=ExistsWebpageSpoolFile(Urlpw);

    /* In one of the spool modes we can only return one page, with or without. */

    if(mode==Spool || mode==SpoolGet || mode==SpoolOrReal)
      {
       if(spool_exists)
         {
          /* If the 401 page already exists without the password, then only try the password. */

          if(SpooledPageStatus(Url)==401)
            {
             FreeURL(Url);
             Url=Urlpw;
             Urlpw=NULL;
             spool_exists=spool_exists_pw;
             spool_exists_pw=0;
            }

          /* If a different page already exists without the password, then only try without the password. */

          else
            {
             FreeURL(Urlpw);
             Urlpw=NULL;
             spool_exists_pw=0;
            }
         }
       else /* if(!spool_exists) */
         {
          /* If the password protected page exists then request the non-password version. */

          if(spool_exists_pw)
            {
             int new_outgoing=OpenOutgoingSpoolFile(0);
             init_buffer(new_outgoing);

             if(new_outgoing==-1)
                PrintMessage(Warning,"Cannot open the new outgoing request to write.");
             else
               {
                char *head=HeaderString(request_head);
                if(write_string(new_outgoing,head)==-1)
                   PrintMessage(Warning,"Cannot write to outgoing file; disk full?");
                if(request_body)
                   if(write_data(new_outgoing,request_body->content,request_body->length)==-1)
                      PrintMessage(Warning,"Cannot write to outgoing file; disk full?");
                CloseOutgoingSpoolFile(new_outgoing,Url);
                free(head);
               }
            }

          /* Display the password protected version. */

          FreeURL(Url);
          Url=Urlpw;
          Urlpw=NULL;
          spool_exists=spool_exists_pw;
          spool_exists_pw=0;
         }
      }
    else if(mode==Fetch || mode==Real)
      {
       if(spool_exists)
         {
          /* If the 401 page already exists without the password, then only try the password. */

          if(SpooledPageStatus(Url)==401)
            {
             FreeURL(Url);
             Url=Urlpw;
             Urlpw=NULL;
             spool_exists=spool_exists_pw;
             spool_exists_pw=0;
            }

          /* If a different page already exists without the password, then only try without the password. */

          else
            {
             FreeURL(Urlpw);
             Urlpw=NULL;
             spool_exists_pw=0;
            }
         }
      }

    /* Do nothing for the refresh modes. */

    else if(mode==RealRefresh || mode==SpoolRefresh)
      ;
   }


 /*----------------------------------------
   In Real or Fetch mode with a password Url then get page with
    no password then come here and try again with a password.
   ----------------------------------------*/

passwordagain:


 /*----------------------------------------
   mode = Spool, SpoolGet, SpoolRefresh, Real, RealRefresh, SpoolOrReal or Fetch

   Check if it needs to be cached, depends on mode and password status.
   ----------------------------------------*/

 if(IsLocalNetHost(Url->host))
   {
    /* Don't cache if in a normal mode */

    if(mode==Real || mode==Spool || mode==SpoolOrReal)
       mode=RealNoCache;

    /* Don't do anything if a refresh URL. */

    else if(mode==SpoolGet)
       ;

    /* Give an error if in fetch mode. */

    else if(mode==Fetch)
      {
       PrintMessage(Inform,"The request to fetch a page from the local network host '%s' is ignored.",Url->host);
       if(client!=-1)
          write_formatted(client,"Cannot fetch %s [On local network]\n",Url->name);
       exit(1);
      }

    /* Do nothing for the refresh modes, they will come through again. */

    else if(mode==RealRefresh || mode==SpoolRefresh)
       ;
   }
 else if(ConfigBooleanMatchURL(DontCache,Url))
   {
    /* Don't cache if possibly online */

    if(mode==Real || mode==SpoolOrReal)
       mode=RealNoCache;

    /* Give an error if in fetch mode. */

    else if(mode==Fetch)
      {
       PrintMessage(Inform,"The request to fetch a page from the local network host '%s' is ignored.",Url->host);
       if(client!=-1)
          write_formatted(client,"Cannot fetch %s [On local network]\n",Url->name);
       exit(1);
      }

    /* Give an error if in a spooling mode */

    else if(mode==Spool || mode==SpoolGet)
      {
       PrintMessage(Inform,"It is not possible to request a URL that is not cached when offline.");
       HTMLMessage(tmpclient,404,"WWWOFFLE Cant Spool Not Cached",NULL,"HostNotCached",
                   "url",Url->name,
                   NULL);
       mode=SpoolInternal; goto spoolinternal;
      }

    /* Do nothing for the refresh modes, they will come through again. */

    else if(mode==RealRefresh || mode==SpoolRefresh)
       ;
   }

 /* If not caching then only use the password version. */

 if(mode==RealNoCache && Urlpw)
   {
    FreeURL(Url);
    Url=Urlpw;
    Urlpw=NULL;
    spool_exists=spool_exists_pw;
    spool_exists_pw=0;
   }


 /*----------------------------------------
   mode = Spool, SpoolGet, SpoolRefresh, Real, RealRefresh, SpoolOrReal or Fetch

   Special handling for POST/PUT requests and cached replies.
   ----------------------------------------*/

 /* Check if it was a POST/PUT method. */

 if(!strcmp(request_head->method,"POST") ||
    !strcmp(request_head->method,"PUT"))
   {
    /* In spool mode they can only be requests to get a new page, not requests from the cache. */

    if(mode==Spool)
       mode=SpoolGet;
   }

 /* Check if the URL indicates the request for */

 else if(Url->args && *Url->args=='!')
   {
    /* In fetch mode they cannot be fetched. */

    if(mode==Fetch)
      {
       PrintMessage(Inform,"It is not possible to fetch a URL that used the POST/PUT method.");
       if(client!=-1)
          write_formatted(client,"Cannot fetch %s [Reply from a POST/PUT method]\n",Url->name);
       exit(1);
      }

    /* If they don't already exist then they can't be requested. */

    else if(!spool_exists && !outgoing_exists)
      {
       PrintMessage(Inform,"It is not possible to request a URL that used the POST/PUT method.");
       HTMLMessage(tmpclient,404,"WWWOFFLE Cant Refresh POST/PUT",NULL,"CantRefreshPosted",
                   "url",Url->name,
                   NULL);
       mode=SpoolInternal; goto spoolinternal;
      }

    /* In all other cases they must be requests for a cached reply. */

    else
       mode=Spool;
   }


 /*----------------------------------------
   mode = Spool, SpoolGet, SpoolRefresh, Real, RealRefresh, RealNoCache, SpoolOrReal or Fetch

   Check if it is htdig or mnogosearch (udmsearch) and don't let it request any URLs.
   ----------------------------------------*/

 user_agent=GetHeader(request_head,"User-Agent");
 if(user_agent)
    is_client_searcher=!strncasecmp(user_agent,"htdig",5) ||
                       !strncasecmp(user_agent,"mnoGoSearch",11) ||
                       !strncasecmp(user_agent,"UdmSearch",9);

 if(is_client_searcher)
   {
    /* If it exists then we must supply the cached version. */

    if(spool_exists)
       mode=Spool;

    /* If not then deny the request. */

    else
      {
       PrintMessage(Inform,"URL unavailable to be searched.");
       HTMLMessageHead(tmpclient,404,"WWWOFFLE Not Searched",
                       NULL);
       mode=SpoolInternal; goto spoolinternal;
      }
   }


 /*----------------------------------------
   mode = Spool, SpoolGet, SpoolRefresh, Real, RealRefresh, RealNoCache, SpoolOrReal or Fetch

   Check if a refresh is needed based on pragma, request changes, autodial.
   ----------------------------------------*/

 conditional_request_ims=!!GetHeader(request_head,"If-Modified-Since");
 conditional_request_inm=!!GetHeader(request_head,"If-None-Match");

 /* If in Real mode and there is a lockfile then just spool the cache version. */

 if(mode==Real && ExistsLockWebpageSpoolFile(Url))
   {
    mode=Spool;
   }

 /* If in Fetch mode and there is a lockfile then we don't need to fetch again. */

 else if(mode==Fetch && ExistsLockWebpageSpoolFile(Url))
   {
    PrintMessage(Debug,"Already fetching URL.");
    if(client!=-1)
       write_formatted(client,"Cannot fetch %s [Already fetching]\n",Url->name);
    exit(fetch_again?4:0);
   }

 /* If a forced refresh then change to an appropriate mode. */

 else if(!is_client_searcher &&
         (RefreshForced() ||
          (ConfigBooleanURL(PragmaNoCache,Url) && GetHeader2(request_head,"Pragma","no-cache")) ||
          (ConfigBooleanURL(CacheControlNoCache,Url) && GetHeader2(request_head,"Cache-Control","no-cache")) ||
          GetHeader2(request_head,"Cache-Control","max-age=0")))
   {
    if(conditional_request_ims)
       RemoveFromHeader(request_head,"If-Modified-Since");
    if(conditional_request_inm)
       RemoveFromHeader(request_head,"If-None-Match");

    /* In spool mode either force a new request if cached or make a new request. */

    if(mode==Spool || mode==SpoolGet)
      {
       if(spool_exists && !ConfigBooleanURL(ConfirmRequests,Url))
          mode=SpoolPragma;
       else
          mode=SpoolGet;
      }

    /* In autodial mode we need to make a real request. */

    else if(mode==SpoolOrReal)
       mode=Real;

    /* In real or fetch or nocache or refresh modes make no other changes. */

    else if(mode==Real || mode==SpoolRefresh || mode==RealRefresh || mode==RealNoCache || mode==Fetch)
      ;
   }

 /* In fetch mode when not a forced refresh. */

 else if(mode==Fetch)
   {
    if(conditional_request_ims)
       RemoveFromHeader(request_head,"If-Modified-Since");
    if(conditional_request_inm)
       RemoveFromHeader(request_head,"If-None-Match");

    spool=-1;

    /* If there is already a cached version then open it. */

    if(spool_exists)
      {
       spool=OpenWebpageSpoolFile(1,Url);
       init_buffer(spool);
      }

    /* if not cached or not openable then don't do anything else. */

    if(spool==-1)
       ;

    /* If changes are needed then get them and add the conditional headers. */

    else if(RequireChanges(spool,request_head,Url))
       TouchWebpageSpoolFile(Url,0);

    /* If no change is needed but is a recursive request, parse the document and fetch the images / links. */

    else if(fetch_again)
      {
       lseek(spool,0,SEEK_SET);
       init_buffer(spool);

       if(ParseDocument(spool,Url))
          RecurseFetch(Url,0);

       close(spool);

       exit(4);
      }

    /* Otherwise just exit. */

    else
      {
       close(spool);
       exit(0);
      }
   }

 /* In Real mode when not a forced refresh. */

 else if(mode==Real)
   {
    if(conditional_request_ims)
       RemoveFromHeader(request_head,"If-Modified-Since");
    if(conditional_request_inm)
       RemoveFromHeader(request_head,"If-None-Match");

    spool=-1;

    /* If there is already a cached version then open it. */

    if(spool_exists)
      {
       spool=OpenWebpageSpoolFile(1,Url);
       init_buffer(spool);
      }

    /* if not cached or not openable then don't do anything else. */

    if(spool==-1)
       ;

    /* If changes are needed then get them and add the conditional headers. */

    else if(RequireChanges(spool,request_head,Url))
       TouchWebpageSpoolFile(Url,0);

    /* Otherwise just use the spooled version. */

    else
       mode=Spool;

    if(spool!=-1)
       close(spool);
   }

 /* If in autodial mode when not a forced refresh. */

 else if(mode==SpoolOrReal)
   {
    if(conditional_request_ims)
       RemoveFromHeader(request_head,"If-Modified-Since");
    if(conditional_request_inm)
       RemoveFromHeader(request_head,"If-None-Match");

    /* If it is cached then use Spool mode. */

    if(spool_exists)
       mode=Spool;

    /* If not then use Real mode. */

    else
       mode=Real;
   }

 /* If in Spool mode when not a forced refresh. */

 else if(mode==Spool)
   {
    /* If the cached version does not exist then get it. */

    if(!spool_exists)
       mode=SpoolGet;

    /* If the cached version does exist and a conditional request then see if it is modified. */

    else if(conditional_request_ims || conditional_request_inm)
      {
       spool=OpenWebpageSpoolFile(1,Url);
       init_buffer(spool);

       if(spool!=-1)
         {
          /* Return a not-modified header if it isn't modified. */

          if(!IsModified(spool,request_head))
            {
             HTMLMessageHead(tmpclient,304,"WWWOFFLE Not Modified",
                             NULL);
             mode=SpoolInternal; goto spoolinternal;
            }

          close(spool);
         }
      }

    /* Remove the headers only after calling IsModified(). */

    if(conditional_request_ims)
       RemoveFromHeader(request_head,"If-Modified-Since");
    if(conditional_request_inm)
       RemoveFromHeader(request_head,"If-None-Match");
   }

 /* When in a refresh or get mode then remove the conditional headers. */

 else if(mode==RealRefresh || mode==SpoolRefresh || mode==SpoolGet)
   {
    if(conditional_request_ims)
       RemoveFromHeader(request_head,"If-Modified-Since");
    if(conditional_request_inm)
       RemoveFromHeader(request_head,"If-None-Match");
   }

 /* In no cached mode do nothing. */

 else /* if(mode==RealNoCache) */
   ;


 /*----------------------------------------
   mode = Spool, SpoolGet, SpoolPragma, SpoolRefresh, Real, RealRefresh, RealNoCache or Fetch

   Set up the spool file, outgoing file and server connection as appropriate.
   ----------------------------------------*/

 /* Set up the file descriptor for the spool file (to write). */

 if(mode==Real || mode==Fetch)
   {
    if(spool_exists)
       CreateBackupWebpageSpoolFile(Url);

    spool=OpenWebpageSpoolFile(0,Url);
    init_buffer(spool);

    CreateLockWebpageSpoolFile(Url);

    if(spool==-1)
      {
       DeleteLockWebpageSpoolFile(Url);
       PrintMessage(Warning,"Cannot open the spooled web page to write.");
       if(mode==Real)
         {
          HTMLMessage(tmpclient,500,"WWWOFFLE Server Error",NULL,"ServerError",
                      "error","Cannot open the spooled web page to write.",
                      NULL);
          mode=SpoolInternal; goto spoolinternal;
         }
       else /* mode==Fetch */
         {
          if(client!=-1)
             write_formatted(client,"Internal Error %s [Cannot open spool file]\n",Url->name);
          exit(1);
         }
      }

    lasttime_exists=CreateLastTimeSpoolFile(Url);
   }

 /* Set up the file descriptor for the spool file (to read). */

 else if(mode==Spool || mode==SpoolPragma)
   {
    spool=OpenWebpageSpoolFile(1,Url);
    init_buffer(spool);

    if(spool==-1)
      {
       PrintMessage(Warning,"Cannot open the spooled web page to read.");
       HTMLMessage(tmpclient,500,"WWWOFFLE Server Error",NULL,"ServerError",
                   "error","Cannot open the spooled web page to read.",
                   NULL);
       mode=SpoolInternal; goto spoolinternal;
      }
   }

 /* Set up the outgoing file (to write). */

 offline_request=!ConfigBooleanURL(DontRequestOffline,Url);

 is_client_wwwoffle=GetHeader2(request_head,"Pragma","wwwoffle")!=NULL;

 if((offline_request || online) &&
    (mode==SpoolRefresh ||
     ((mode==SpoolGet || mode==SpoolPragma) &&
      (!ConfigBooleanURL(ConfirmRequests,Url) || is_client_wwwoffle || Url->local || (Url->args && *Url->args=='!')))))
   {
    outgoing=OpenOutgoingSpoolFile(0);
    init_buffer(outgoing);

    if(outgoing==-1)
      {
       PrintMessage(Warning,"Cannot open the outgoing request to write.");
       HTMLMessage(tmpclient,500,"WWWOFFLE Server Error",NULL,"ServerError",
                   "error","Cannot open the outgoing request to write.",
                   NULL);
       mode=SpoolInternal; goto spoolinternal;
      }
   }

 /* Open the connection to the server host. */

 if(mode==Real || mode==RealNoCache || mode==Fetch)
   {
    char *err=(Url->Protocol->open)(Url);

    /* Retry if the option is set. */

    if(err && ConfigBoolean(ConnectRetry))
      {
       PrintMessage(Inform,"Waiting to try connection again.");
       sleep(10);
       err=(Url->Protocol->open)(Url);
      }

    /* In case of an error ... */

    if(err)
      {
       /* Store the error in the cache. */

       if(mode==Real || mode==Fetch)
         {
          lseek(spool,0,SEEK_SET);
          init_buffer(spool);
          ftruncate(spool,0);

          HTMLMessage(spool,503,"WWWOFFLE Remote Host Error",NULL,"RemoteHostError",
                      "url",Url->name,
                      "reason",err,
                      "cache","yes",
                      "backup",spool_exists?"yes":NULL,
                      NULL);
          DeleteLockWebpageSpoolFile(Url);
          close(spool);
         }

       /* In Fetch mode print message and exit. */

       if(mode==Fetch)
         {
          if(client!=-1)
             write_formatted(client,"Fetch Failure %s [Server Connection Failed]\n",Url->name);
          exit(1);
         }

       /* Write the error to the browser. */

       else /* if(mode==Real || mode==RealNoCache) */
         {
          HTMLMessage(tmpclient,503,"WWWOFFLE Remote Host Error",NULL,"RemoteHostError",
                      "url",Url->name,
                      "reason",err,
                      "cache","",
                      "backup",spool_exists?"yes":NULL,
                      NULL);
          mode=SpoolInternal; goto spoolinternal;
         }
      }
   }


 /*----------------------------------------
   mode = Spool, SpoolGet, SpoolPragma, SpoolRefresh, Real, RealRefresh, RealNoCache or Fetch

   Make the request to the server.
   ----------------------------------------*/

 /* Modify the request header. */

 if(mode==SpoolRefresh || mode==Real || mode==RealRefresh || mode==RealNoCache || mode==Fetch)
   {
    /* Censor / Cannonicalise URL / POST & PUT URL hacking / HTTP-1.1 etc. */

    ModifyRequest(Url,request_head);

    /* Add the compression header */

#if USE_ZLIB
    if(ConfigBooleanURL(RequestCompressedData,Url) && !NotCompressed(NULL,Url->path))
      {
       request_compression=1;
       AddToHeader(request_head,"Accept-Encoding","x-gzip, gzip, identity; q=0.1");
/*
       AddToHeader(request_head,"Accept-Encoding","deflate; q=1.0, x-gzip; q=0.9, gzip; q=0.9, identity; q=0.1");
*/
      }
#endif
   }

 /* Display a message if fetching. */

 if(mode==Fetch && client!=-1)
    write_formatted(client,"Fetching %s ...\n",Url->name);

 /* Write request to remote server. */

 if(mode==Real || mode==RealNoCache || mode==Fetch)
   {
    char *err=(Url->Protocol->request)(Url,request_head,request_body);

    /* In case of an error ... */

    if(err)
      {
       /* Store the error in the cache. */

       if(mode==Real || mode==Fetch)
         {
          lseek(spool,0,SEEK_SET);
          init_buffer(spool);
          ftruncate(spool,0);

          HTMLMessage(spool,503,"WWWOFFLE Remote Host Error",NULL,"RemoteHostError",
                      "url",Url->name,
                      "reason",err,
                      "cache","yes",
                      "backup",spool_exists?"yes":NULL,
                      NULL);
          DeleteLockWebpageSpoolFile(Url);
          close(spool);
         }

       /* In Fetch mode print message and exit. */

       if(mode==Fetch)
         {
          if(client!=-1)
             write_formatted(client,"Fetch Failure %s [Server Connection Error]\n",Url->name);
          exit(1);
         }

       /* Write the error to the browser. */

       else /* if(mode==Real || mode==RealNoCache) */
         {
          HTMLMessage(tmpclient,503,"WWWOFFLE Remote Host Error",NULL,"RemoteHostError",
                      "url",Url->name,
                      "reason",err,
                      "cache",NULL,
                      "backup",spool_exists?"yes":NULL,
                      NULL);
          mode=SpoolInternal; goto spoolinternal;
         }
      }
   }

 /* Write request to outgoing file. */

 else if((offline_request || online) &&
         (mode==SpoolRefresh ||
          ((mode==SpoolGet || mode==SpoolPragma) &&
           (!ConfigBooleanURL(ConfirmRequests,Url) || is_client_wwwoffle || Url->local || (Url->args && *Url->args=='!')))))
   {
    char *head=HeaderString(request_head);
    int err=write_string(outgoing,head);

    /* Write the error to the browser. */

    if(err==-1)
      {
       PrintMessage(Warning,"Cannot write the outgoing request.");
       HTMLMessage(tmpclient,500,"WWWOFFLE Server Error",NULL,"ServerError",
                   "error","Cannot write the outgoing request.",
                   NULL);
       mode=SpoolInternal; goto spoolinternal;
      }

    if(request_body)
       if(write_data(outgoing,request_body->content,request_body->length)==-1)
          PrintMessage(Warning,"Cannot write to outgoing file; disk full?");

    free(head);
   }


 /*----------------------------------------
   mode = Spool, SpoolGet, SpoolPragma, SpoolRefresh, Real, RealRefresh, RealNoCache or Fetch

   Check the reply from the server or from the cache.
   ----------------------------------------*/

 /* Read the reply from the server. */

 if(mode==Real || mode==RealNoCache || mode==Fetch)
   {
    int server;
#if USE_ZLIB
    char *content_encoding=NULL;
#endif

    /* Get the header */

    server=Url->Protocol->readhead(&reply_head);

    is_server=1;

    /* In case of error ... */

    if(!reply_head)
      {
       PrintMessage(Warning,"Error reading reply header from remote host [%!s].");

       /* Write the error to the cache. */

       if(mode==Real || mode==Fetch)
         {
          lseek(spool,0,SEEK_SET);
          init_buffer(spool);
          ftruncate(spool,0);

          HTMLMessage(spool,503,"WWWOFFLE Remote Host Error",NULL,"RemoteHostError",
                      "url",Url->name,
                      "reason","TimeoutReply",
                      "cache","yes",
                      "backup",spool_exists?"yes":NULL,
                      NULL);
          DeleteLockWebpageSpoolFile(Url);
         }

       /* In Fetch mode print message and exit. */

       if(mode==Fetch)
         {
          if(mode==Fetch && client!=-1)
             write_formatted(client,"Fetch Failure %s [Server Reply Error]\n",Url->name);
          exit(1);
         }

       /* Write the error to the browser. */

       else /* if(mode==Real || mode==RealNoCache) */
         {
          HTMLMessage(client,503,"WWWOFFLE Remote Host Error",NULL,"RemoteHostError",
                      "url",Url->name,
                      "reason","TimeoutReply",
                      "cache",NULL,
                      "backup",spool_exists?"yes":NULL,
                      NULL);
          mode=SpoolInternal; goto spoolinternal;
         }
      }

    if(DebuggingLevel==ExtraDebug)
       PrintMessage(ExtraDebug,"Incoming Reply Head (from server/proxy)\n%s",HeaderString(reply_head));

    /* Check for compression header */

#if USE_ZLIB
    if(request_compression && (content_encoding=GetHeader(reply_head,"Content-Encoding")))
       server_compression=WhichCompression(content_encoding);
    else
       server_compression=0;

    if(server_compression)
      {
       PrintMessage(Inform,"Server has used 'Content-Encoding: %s'.",content_encoding); /* Used in audit-usage.pl */
       RemoveFromHeader(reply_head,"Content-Encoding");
       init_zlib_buffer(server,server_compression);
      }
#endif

    /* Handle status codes */

    reply_status=reply_head->status;

    /* If fetching and the page has moved, request the new one. */

    if(mode==Fetch && (reply_status==301 || reply_status==302))
      {
       char *new_url=MovedLocation(Url,reply_head);

       if(client!=-1)
          write_formatted(client,"Fetching More %s [Page Moved]\n",Url->name);

       if(!new_url)
          PrintMessage(Warning,"Cannot parse the reply for the new location.");
       else
          fetch_again+=RecurseFetchRelocation(Url,new_url);
      }

    /* If the page has not changed ... */

    else if(reply_status==304)
      {
       PrintMessage(Inform,"Cache Access Status='Unmodified on Server'."); /* Used in audit-usage.pl */

       /* Restore the backup version of the page if writing to the cache. */

       if(mode==Fetch || mode==Real)
         {
          close(spool);
          DeleteLockWebpageSpoolFile(Url);
          RestoreBackupWebpageSpoolFile(Url);
          if(!lasttime_exists)
             DeleteLastTimeSpoolFile(Url);
         }

       /* If fetching then parse the document for links. */

       if(mode==Fetch)
         {
          if(client!=-1)
             write_formatted(client,"Not fetching %s [Page Unchanged]\n",Url->name);

          if(fetch_again)
            {
             spool=OpenWebpageSpoolFile(1,Url);
             init_buffer(spool);

             if(spool!=-1 && ParseDocument(spool,Url))
                RecurseFetch(Url,1);

             close(spool);
            }

          exit(fetch_again?4:0);
         }

       /* If in Real mode then tell the browser or spool the page to the browser. */

       else if(mode==Real)
         {
          if(conditional_request_ims || conditional_request_inm)
            {
             HTMLMessageHead(tmpclient,304,"WWWOFFLE Not Modified",
                             NULL);
             mode=SpoolInternal; goto spoolinternal;
            }
          else
            {
             spool=OpenWebpageSpoolFile(1,Url);
             init_buffer(spool);

             mode=Spool;
            }
         }
      }

    /* If fetching a page that requires a password then change mode. */

    else if(mode==Fetch && reply_status==401 && Urlpw)
      {
       mode=FetchNoPassword;
       if(client!=-1)
          write_formatted(client,"Fetching More %s [URL with password]\n",Url->name);
      }

    /* If in Real mode and a page that requires a password then change mode. */

    else if(mode==Real && reply_status==401 && Urlpw)
      {
       mode=RealNoPassword;
      }
   }

 /* If there is no server then blank the reply header. */

 else
    reply_head=NULL;

 /* Get the header from the cache. */

 if(mode==Spool || mode==SpoolPragma)
   {
    reply_status=ParseReply(spool,&reply_head);

    if(!reply_head)
      {
       PrintMessage(Warning,"Spooled Reply Head (from cache) is empty; deleting it.");
       DeleteWebpageSpoolFile(Url,0);
      }
    else
      {
       char *head=HeaderString(reply_head);
       if(DebuggingLevel==ExtraDebug)
          PrintMessage(ExtraDebug,"Spooled Reply Head (from cache)\n%s",head);
       free(head);
      }
   }

 /* Initialise the body. */

 reply_body=CreateBody(READ_BUFFER_SIZE);

 /* Close the outgoing file if any. */

 if(outgoing>=0)
   {
    if(mode==Fetch)
       close(outgoing);
    if(mode==SpoolGet || mode==SpoolRefresh || mode==SpoolPragma)
       CloseOutgoingSpoolFile(outgoing,Url);
   }


 /*----------------------------------------
   mode = Spool, SpoolGet, SpoolPragma, SpoolRefresh, Real, RealRefresh, RealNoCache, RealNoPassword, Fetch or FetchNoPassword

   The main handling of the body data in the reply.
   ----------------------------------------*/

 /* Check the modified request to get the log messages correct. */

 conditional_request_ims=!!GetHeader(request_head,"If-Modified-Since");
 conditional_request_inm=!!GetHeader(request_head,"If-None-Match");

 /* When reading from the server and writing to the browser and possibly the cache. */

 if(mode==Real || mode==RealNoCache || mode==RealNoPassword)
   {
    char *head;
    int n=0,err=0,bytes=0,headlen=0;
    int modify=0;

    /* Print a message for auditing. */

    if(mode==RealNoCache)
       PrintMessage(Inform,"Cache Access Status='Not Cached'."); /* Used in audit-usage.pl */
    else
      {
       if(!spool_exists)
          PrintMessage(Inform,"Cache Access Status='New Page'."); /* Used in audit-usage.pl */
       else if(conditional_request_ims || conditional_request_inm)
          PrintMessage(Inform,"Cache Access Status='Modified on Server'."); /* Used in audit-usage.pl */
       else
          PrintMessage(Inform,"Cache Access Status='Forced Reload'."); /* Used in audit-usage.pl */
      }

    /* Check if the HTML modifications are to be performed. */

    if(mode==Real &&
       !is_client_wwwoffle &&
       !is_client_searcher &&
       ConfigBooleanURL(EnableHTMLModifications,Url) &&
       ConfigBooleanURL(EnableModificationsOnline,Url) &&
       !GetHeader2(request_head,"Cache-Control","no-transform"))
      {
       char *content_encoding;

       if(ConfigBooleanURL(EnableHTMLModifications,Url) &&
          GetHeader2(reply_head,"Content-Type","text/html"))
          modify=1;
       else if(ConfigBooleanURL(DisableAnimatedGIF,Url) &&
               GetHeader2(reply_head,"Content-Type","image/gif"))
          modify=2;

       /* If the reply uses compression and we are modifying the content then don't (shouldn't happen). */

       if(modify && (content_encoding=GetHeader(reply_head,"Content-Encoding")))
          if(WhichCompression(content_encoding))
             modify=0;
      }

    /* Generate the header and write it to the cache unmodified. */

    if(mode!=RealNoCache)
      {
       head=HeaderString(reply_head);

       write_string(spool,head);

       headlen=strlen(head);

       free(head);
      }

    /* If the client can use compression and did not just ask for the head and we are not modifying the content ... */

#if USE_ZLIB
    if(client_compression)
       if(mode!=RealNoPassword && !head_only && !modify)
         {
          /* If it is not to be compressed then don't */

          if(GetHeader(reply_head,"Content-Encoding") ||
             NotCompressed(GetHeader(reply_head,"Content-Type"),NULL))
             client_compression=0;

          /* Add the compression header for the client. */

          else
            {
             if(client_compression==1)
                AddToHeader(reply_head,"Content-Encoding","deflate");
             else /* if(client_compression==2) */
                AddToHeader(reply_head,"Content-Encoding","gzip");
            }
         }

    /* If the server compressed the data or WWWOFFLE will compress it to the client
       then remove the Content-Length header since it will be wrong, otherwise OK. */

    if(server_compression || client_compression)
       RemoveFromHeader(reply_head,"Content-Length");
#endif

    /* Fix up the reply head from the server before sending to the client. */

    ModifyReply(Url,reply_head);

    head=HeaderString(reply_head);

    /* Write the header to the client if not being modified and this is the actual reply. */

    if(mode!=RealNoPassword && !modify)
      {
       if(DebuggingLevel==ExtraDebug)
          PrintMessage(ExtraDebug,"Outgoing Reply Head (to browser)\n%s",head);

       err=write_string(client,head);
      }

    /* If modifying then write to the temporary file. */

    else if(modify)
       err=write_string(tmpclient,head);

    /* Initialise the client compression. */

#if USE_ZLIB
    if(client_compression)
       if(mode!=RealNoPassword && !head_only && !modify)
          init_zlib_buffer(client,-client_compression);
#endif

    /* While there is data to read ... */

    while(err!=-1 && (n=(Url->Protocol->readbody)(reply_body->content,READ_BUFFER_SIZE))>0)
      {
       bytes+=n;

       /* Write the data to the cache. */

       if(mode!=RealNoCache)
          if(write_data(spool,reply_body->content,n)==-1)
             PrintMessage(Warning,"Cannot write to cache file; disk full?");

       /* Write the data to the client if it wants the body now. */

       if(mode!=RealNoPassword && !head_only && !modify)
          err=write_data(client,reply_body->content,n);

       /* In case of error writing to client decide if to continue reading from server. */

       if(err==-1)
         {
          char *length=GetHeader(reply_head,"Content-Length");

          if(length)
            {
             int size=atoi(length);
             if(size<(ConfigIntegerURL(IntrDownloadSize,Url)<<10) ||
                (100*(double)bytes/(double)size)>ConfigIntegerURL(IntrDownloadPercent,Url))
                err=0;
             head_only=1; /* cheat to stop writing to client */
            }
         }
      }

    /* If writing to the cache ... */

    if(mode!=RealNoCache)
      {
       /* If there is an error with the client and we don't keep partial pages write the error to the cache. */

       if(err==-1 && !ConfigBooleanURL(IntrDownloadKeep,Url))
         {
          lseek(spool,0,SEEK_SET);
          init_buffer(spool);
          ftruncate(spool,0);

          PrintMessage(Warning,"Error writing to client [%!s]; client disconnected?");
          HTMLMessage(spool,503,"WWWOFFLE Remote Host Error",NULL,"RemoteHostError",
                      "url",Url->name,
                      "reason","ClientClose",
                      "cache","yes",
                      "backup",spool_exists?"yes":NULL,
                      NULL);
         }

       /* If there is an error with the server and we don't keep partial pages write the error to the cache. */

       else if(n<0 && !ConfigBooleanURL(TimeoutDownloadKeep,Url))
         {
          lseek(spool,0,SEEK_SET);
          init_buffer(spool);
          ftruncate(spool,0);

          PrintMessage(Warning,"Error reading reply body from remote host [%!s].");
          HTMLMessage(spool,503,"WWWOFFLE Remote Host Error",NULL,"RemoteHostError",
                      "url",Url->name,
                      "reason",errno==ERRNO_USE_Z_ERRNO?"CompressCorrupt":"TimeoutTransfer",
                      "cache","yes",
                      "backup",spool_exists?"yes":NULL,
                      NULL);
         }

       /* Otherwise ... */

       else
         {
          /* In case of error change the cache status code. */

          if(n<0 || err==-1)
            {
             reply_status=ParseReply(spool,&reply_head);

             reply_head->status=503;

             free(head);
             head=HeaderString(reply_head);

             lseek(spool,0,SEEK_SET);
             init_buffer(spool);

             write_string(spool,head);
            }

          /* Delete the backup if there is one. */

          if(spool_exists)
             DeleteBackupWebpageSpoolFile(Url);

          /* If we are modifying the content then do it. */

          if(modify)
            {
             lseek(spool,headlen,SEEK_SET);
             init_buffer(spool);

             if(modify==1)
               OutputHTMLWithModifications(tmpclient,spool,Url);
             else if(modify==2)
               OutputGIFWithModifications(tmpclient,spool,Url);

             mode=SpoolInternal;
            }
         }

       DeleteLockWebpageSpoolFile(Url);
      }

    /* Finish with the client compression. */

#if USE_ZLIB
    if(client_compression)
       finish_zlib_buffer(client);
#endif

    free(head);
   }

 /* When reading from the server and writing to the cache. */

 else if(mode==Fetch || mode==FetchNoPassword)
   {
    char *head;
    int n;

    /* Write a message for auditing. */

    if(!spool_exists)
       PrintMessage(Inform,"Cache Access Status='New Page'."); /* Used in audit-usage.pl */
    else if(conditional_request_ims || conditional_request_inm)
       PrintMessage(Inform,"Cache Access Status='Modified on Server'."); /* Used in audit-usage.pl */
    else
       PrintMessage(Inform,"Cache Access Status='Forced Reload'."); /* Used in audit-usage.pl */

    /* Write the header to the cache. */

    head=HeaderString(reply_head);

    write_string(spool,head);

    free(head);

    /* While the server has data write it to the cache. */

    while((n=(Url->Protocol->readbody)(reply_body->content,READ_BUFFER_SIZE))>0)
       if(write_data(spool,reply_body->content,n)==-1)
          PrintMessage(Warning,"Cannot write to cache file; disk full?");

    /* If there is an error with the server and we don't keep partial pages write the error to the cache. */

    if(n<0 && !ConfigBooleanURL(TimeoutDownloadKeep,Url))
      {
       lseek(spool,0,SEEK_SET);
       init_buffer(spool);
       ftruncate(spool,0);

       PrintMessage(Warning,"Error reading reply body from remote host [%!s].");
       HTMLMessage(spool,503,"WWWOFFLE Remote Host Error",NULL,"RemoteHostError",
                   "url",Url->name,
                   "reason",errno==ERRNO_USE_Z_ERRNO?"CompressCorrupt":"TimeoutTransfer",
                   "cache","yes",
                   "backup",spool_exists?"yes":NULL,
                   NULL);

       if(client!=-1)
          write_formatted(client,"Fetch Failure %s [%s]\n",Url->name,errno==ERRNO_USE_Z_ERRNO?"CompressCorrupt":"Timeout");
      }

    /* If we finished OK then delete the backup. */

    else
      {
       if(spool_exists)
          DeleteBackupWebpageSpoolFile(Url);
       if(client!=-1)
          write_formatted(client,"Fetch Success %s\n",Url->name);
      }

    DeleteLockWebpageSpoolFile(Url);

    /* If the page was OK then fetch the links and images. */

    if(n>=0 && reply_status>=200 && reply_status<400)
      {
       ftruncate(spool,lseek(spool,0,SEEK_CUR));
       lseek(spool,0,SEEK_SET);
       init_buffer(spool);

       if(ParseDocument(spool,Url))
         {
          int links=RecurseFetch(Url,1);

          if(client!=-1 && links)
             write_formatted(client,"Fetching More %s [%d Extra URLs]\n",Url->name,links);

          fetch_again+=links;
         }
      }
   }

 /* If reading the page from the cache to the browser. */

 else if(mode==Spool || mode==SpoolPragma)
   {
    struct stat buf;
    char *remote_error_note="WWWOFFLE Remote Host Error";
    int modify=0;

    /* If the lock file exists ... */

    if(ExistsLockWebpageSpoolFile(Url))
      {
       int t=0;

       /* If we are online then wait for it to become unlocked. */

       if(online)
         {
          t=ConfigInteger(SocketTimeout)/6;

          PrintMessage(Inform,"Waiting for the page to be unlocked.");

          while(--t>0 && ExistsLockWebpageSpoolFile(Url))
             sleep(1);

          if(t<=0)
             PrintMessage(Inform,"Timed out waiting for the page to be unlocked.");
         }

       /* Show an error message if still locked. */

       if(t<=0)
         {
          HTMLMessage(tmpclient,503,"WWWOFFLE File Locked",NULL,"FileLocked",
                      "url",Url->name,
                      NULL);
          mode=SpoolInternal; goto spoolinternal;
         }

       /* Read the header again in case it changed */

       if(reply_head)
          FreeHeader(reply_head);

       lseek(spool,0,SEEK_SET);
       init_buffer(spool);

       reply_status=ParseReply(spool,&reply_head);
      }

    PrintMessage(Inform,"Cache Access Status='Cached Page Used'."); /* Used in audit-usage.pl */

    /* If the page is not empty ... */

    if(reply_head)
      {
       char *head;

       /* If the page is a previous error message then delete the cached one and restore the backup. */

       if(!strcmp(reply_head->note,remote_error_note) || (!fstat(spool,&buf) && buf.st_size==0))
         {
          DeleteWebpageSpoolFile(Url,0);
          RestoreBackupWebpageSpoolFile(Url);
         }

       /* If the file is compressed then prepare to uncompress it. */

#if USE_ZLIB
       if(GetHeader2(reply_head,"Pragma","wwwoffle-compressed"))
         {
          char *content_encoding;

          if((content_encoding=GetHeader(reply_head,"Content-Encoding")))
            {
             PrintMessage(Debug,"Spooled page has 'Content-Encoding: %s'.",content_encoding);
             RemoveFromHeader(reply_head,"Content-Encoding");
             RemoveFromHeader2(reply_head,"Pragma","wwwoffle-compressed");
             init_zlib_buffer(spool,2);
            }
         }
#endif

       /* Write the header to the temporary file. */

       head=HeaderString(reply_head);
       write_string(tmpclient,head);
       free(head);

       /* Decide if we need to modify the content. */

       if(!is_client_wwwoffle &&
          !is_client_searcher &&
          ConfigBooleanURL(EnableHTMLModifications,Url) &&
          !GetHeader2(request_head,"Cache-Control","no-transform"))
         {
          char *content_encoding;

          if(ConfigBooleanURL(EnableHTMLModifications,Url) &&
             GetHeader2(reply_head,"Content-Type","text/html"))
             modify=1;
          else if(ConfigBooleanURL(DisableAnimatedGIF,Url) &&
                  GetHeader2(reply_head,"Content-Type","image/gif"))
             modify=2;

          /* If the spooled page uses compression and we are modifying the content then don't (e.g. very old WWWOFFLE cache). */

          if(modify && (content_encoding=GetHeader(reply_head,"Content-Encoding")))
             if(WhichCompression(content_encoding))
                modify=0;
         }
      }
    else
       PrintMessage(Debug,"Spooled page has no header.");

    /* Modify the content or don't and put it in the temporary file. */

    if(modify==1)
       OutputHTMLWithModifications(tmpclient,spool,Url);
    else if(modify==2)
       OutputGIFWithModifications(tmpclient,spool,Url);
    else
      {
       int n;

       while((n=read_data(spool,reply_body->content,READ_BUFFER_SIZE))>0)
          if(write_data(tmpclient,reply_body->content,n)==-1)
             PrintMessage(Warning,"Cannot write to temporary file; disk full?");
      }

    /* Finish with uncompressing from the cache. */

#if USE_ZLIB
    finish_zlib_buffer(spool);
#endif

    mode=SpoolInternal;
   }

 /* If the page is to be requested. */

 else if(mode==SpoolGet)
   {
    /* If offline requests are allowed or we are online give the appropriate message to the browser. */

    if(offline_request || online)
      {
       if(ConfigBooleanURL(ConfirmRequests,Url) && !Url->local && (!Url->args || *Url->args!='!') && !outgoing_exists)
          HTMLMessage(tmpclient,404,"WWWOFFLE Confirm Request",NULL,"ConfirmRequest",
                      "url",Url->name,
                      NULL);
       else if(fetch_again)
          HTMLMessage(tmpclient,404,"WWWOFFLE Refresh Will Get",NULL,"RefreshWillGet",
                      "url",Url->name,
                      "hash",HashOutgoingSpoolFile(Url),
                      NULL);
       else if(outgoing_exists)
          HTMLMessage(tmpclient,404,"WWWOFFLE Will Get",NULL,"WillGet",
                      "url",Url->name,
                      "hash",NULL,
                      NULL);
       else
         {
          char *hash=HashOutgoingSpoolFile(Url);
          HTMLMessage(tmpclient,404,"WWWOFFLE Will Get",NULL,"WillGet",
                      "url",Url->name,
                      "hash",hash,
                      NULL);
          free(hash);
         }
      }

    /* Else refuse the request. */

    else
       HTMLMessage(tmpclient,404,"WWWOFFLE Refused Request",NULL,"RefusedRequest",
                   "url",Url->name,
                   NULL);

    mode=SpoolInternal;
   }

 /* If the request is for a refresh page when online or offline the redirect the browser. */

 else if(mode==RealRefresh || mode==SpoolRefresh)
   {
    HTMLMessage(tmpclient,302,"WWWOFFLE Refresh Redirect",Url->link,"Redirect",
                "location",Url->link,
                NULL);

    mode=SpoolInternal;
   }


 /*----------------------------------------
   mode = SpoolInternal, Real, RealNoCache, RealNoPassword, Fetch or FetchNoPassword

   Output the data to the browser if we haven't already sent it.
   ----------------------------------------*/

spoolinternal:

 /* If we are handling an internally generated page or modified page or cached page ... */

 if(mode==SpoolInternal)
   {
    int n;
    unsigned long size;

    if(reply_head)
       FreeHeader(reply_head);

    if(!reply_body)
       reply_body=CreateBody(READ_BUFFER_SIZE);

    /* Work out the size of the file. */

    size=lseek(tmpclient,0,SEEK_CUR);
    lseek(tmpclient,0,SEEK_SET);
    init_buffer(tmpclient);

    /* Read the head back in again. */

    reply_status=ParseReply(tmpclient,&reply_head);

    if(!reply_head)
       PrintMessage(Warning,"Outgoing Reply Head (to browser) is empty.");
    else
      {
       char *head,length[10];

       /* Put in the correct content length. */

       head=HeaderString(reply_head);

       size-=strlen(head);

       RemoveFromHeader(reply_head,"Content-Length");

       sprintf(length,"%lu",size);
       AddToHeader(reply_head,"Content-Length",length);

       /* Set up compression for the browser if available. */

#if USE_ZLIB
       if(client_compression)
         {
          /* If it is not to be compressed then don't */

          if(GetHeader(reply_head,"Content-Encoding") ||
             NotCompressed(GetHeader(reply_head,"Content-Type"),NULL))
             client_compression=0;

          /* Add the compression header for the client. */

          else
            {
             if(client_compression==1)
                AddToHeader(reply_head,"Content-Encoding","deflate");
             else /* if(client_compression==2) */
                AddToHeader(reply_head,"Content-Encoding","gzip");
            }

          RemoveFromHeader(reply_head,"Content-Length");
         }
#endif

       /* Modify the reply as appropriate and write it to the browser. */

       ModifyReply(Url,reply_head);

       free(head);
       head=HeaderString(reply_head);

       if(DebuggingLevel==ExtraDebug)
          PrintMessage(ExtraDebug,"Outgoing Reply Head (to browser)\n%s",head);
       write_string(client,head);

       free(head);
      }

    /* If we are doing the body as well then read that from the file and send to the browser. */

    if(!head_only)
      {
#if USE_ZLIB
       if(client_compression)
          init_zlib_buffer(client,-client_compression);
#endif

       while((n=read_data(tmpclient,reply_body->content,READ_BUFFER_SIZE))>0)
          write_data(client,reply_body->content,n);

#if USE_ZLIB
       if(client_compression)
          finish_zlib_buffer(client);
#endif
      }
   }


 /*----------------------------------------
   mode = SpoolInternal, Real, RealNoCache, RealNoPassword, Fetch or FetchNoPassword

   Tidy up and exit.
   ----------------------------------------*/

 /* Delete the outgoing spool file if we just got it from the server. */

 if(is_server && outgoing_exists)
    DeleteOutgoingSpoolFile(Url);

 /* If we had a connection to a server then close it. */

 if(is_server)
    (Url->Protocol->close)();

 /* If we have finished with the temporary file then close it. */

 if(tmpclient>=0 && mode!=RealNoPassword)
    CloseTempSpoolFile(tmpclient);

 /* If there is a spool file open then close it. */

 if(spool>=0)
    close(spool);

 /* If we were searched then reset the cache file time. */

 if(is_client_searcher && request_head && spool_exists)
    TouchWebpageSpoolFile(Url,spool_exists);

 /* If there is a reply head and body free them. */

 if(reply_head)
    FreeHeader(reply_head);
 if(reply_body)
    FreeBody(reply_body);

 /* If we have fetched a version of a URL without a password then fetch the real one. */

 if(mode==RealNoPassword || mode==FetchNoPassword)
   {
    FreeURL(Url);
    Url=Urlpw;
    Urlpw=NULL;

    spool_exists=spool_exists_pw;
    spool_exists_pw=0;

    reply_head=NULL;
    reply_body=NULL;

    if(mode==RealNoPassword)
       mode=Real;
    else
       mode=Fetch;

    goto passwordagain;
   }

 /* Free the original URL */

 FreeURL(Url);
 free(url);

 /* If there is a request head and body then free them. */

 if(request_head)
    FreeHeader(request_head);
 if(request_body)
    FreeBody(request_body);

 /* If there is a client then empty the contents and close it down. */

 if(client>=0)
   {
    empty_buffer(client);
#ifdef __CYGWIN__
    CloseCygwinSocket(client);
#else
    CloseSocket(client);
#endif
   }

 /* If we need to fetch more then tell the parent process. */

 if(fetch_again)
    return(4);
 else
    return(0);
}


/*++++++++++++++++++++++++++++++++++++++
  Uninstall the signal handlers.
  ++++++++++++++++++++++++++++++++++++++*/

static void uninstall_sighandlers(void)
{
 struct sigaction action;

 /* SIGCHLD */
 action.sa_handler = SIG_DFL;
 sigemptyset (&action.sa_mask);
 action.sa_flags = 0;
 if(sigaction(SIGCHLD, &action, NULL) != 0)
    PrintMessage(Warning, "Cannot uninstall SIGCHLD handler.");

 /* SIGINT, SIGQUIT, SIGTERM */
 action.sa_handler = SIG_DFL;
 sigemptyset(&action.sa_mask);
 action.sa_flags = 0;
 if(sigaction(SIGINT, &action, NULL) != 0)
    PrintMessage(Warning, "Cannot uninstall SIGINT handler.");
 if(sigaction(SIGQUIT, &action, NULL) != 0)
    PrintMessage(Warning, "Cannot uninstall SIGQUIT handler.");
 if(sigaction(SIGTERM, &action, NULL) != 0)
    PrintMessage(Warning, "Cannot uninstall SIGTERM handler.");

 /* SIGHUP */
 action.sa_handler = SIG_DFL;
 sigemptyset(&action.sa_mask);
 action.sa_flags = 0;
 if(sigaction(SIGHUP, &action, NULL) != 0)
    PrintMessage(Warning, "Cannot uninstall SIGHUP handler.");

 /* SIGPIPE */
 action.sa_handler = SIG_IGN;
 sigemptyset (&action.sa_mask);
 action.sa_flags = 0;
 if(sigaction(SIGPIPE, &action, NULL) != 0)
    PrintMessage(Warning, "Cannot ignore SIGPIPE.");
}
