/***************************************
  $Header: /home/amb/wwwoffle/RCS/configfile.c 1.15 2001/06/12 18:57:08 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 2.6d.
  Configuration file management functions.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1997,98,99,2000,01 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <limits.h>
#include <unistd.h>
#include <sys/types.h>
#include <pwd.h>
#include <grp.h>

#include "configfile.h"
#include "wwwoffle.h"
#include "errors.h"
#include "misc.h"


#define CONFIG_DEBUG_DUMP 0


/* Local functions */

static int ReadSection(FILE *file,ConfigSection *section);
static char* ReadLine(FILE *file);
static int ParseEntry(ConfigEntry *entry,char *url_str,char *key_str,char *val_str);
static int ParseItem(char *text,ConfigType type,KeyOrValue *pointer);
static int isanumber(const char *string);

static void CreateBackupConfigFile(ConfigFile* current);
static void RestoreBackupConfigFile(ConfigFile* current);
static void PurgeBackupConfigFile(void);

static void FreeConfigItem(ConfigItem item);

#if CONFIG_DEBUG_DUMP
static void DumpConfigFile(ConfigFile* config);
static void DumpConfigItem(ConfigItem item);
#endif


/*+ The error message. +*/
static char *errmsg;

/*+ The line number in the configuration file. +*/
static int line_no;

/*+ The filename of the configuration file. +*/
static char *file_name;

/*+ The backup version of the config file. +*/
static ConfigFile backup;


/*++++++++++++++++++++++++++++++++++++++
  Set the configuration file default values.

  ConfigFile *config The configuration file information to use.
  ++++++++++++++++++++++++++++++++++++++*/

void DefaultConfigFile(ConfigFile *config)
{
 int s,e;

 for(s=0;s<config->nsections;s++)
    for(e=0;e<config->sections[s]->nentries;e++)
       if(config->sections[s]->entries[e].defval)
         {
          ConfigItem *item=config->sections[s]->entries[e].item;

          *item=(ConfigItem)malloc(sizeof(struct _ConfigItem));
          (*item)->entry=&config->sections[s]->entries[e];
          (*item)->nitems=0;
          (*item)->url=NULL;
          (*item)->key=NULL;
          (*item)->val=NULL;
          (*item)->defval=(KeyOrValue*)malloc(sizeof(KeyOrValue));

#if CONFIG_VERIFY_ABORT
          if(config->sections[s]->entries[e].key_type!=Fixed)
             PrintMessage(Fatal,"Configuration file error at %s:%d",__FILE__,__LINE__);

          if(ParseItem(config->sections[s]->entries[e].defval,config->sections[s]->entries[e].val_type,(*item)->defval))
             PrintMessage(Fatal,"Configuration file error at %s:%d; %s",__FILE__,__LINE__,errmsg);
#else
          ParseItem(config->sections[s]->entries[e].defval,config->sections[s]->entries[e].val_type,(*item)->defval);
#endif
         }

#if CONFIG_DEBUG_DUMP
 DumpConfigFile(config);
#endif
}


/*++++++++++++++++++++++++++++++++++++++
  Read the data from the file.

  char *ReadConfigFile Returns the error message or NULL if OK.

  ConfigFile *config The configuration information to read from the file.
  ++++++++++++++++++++++++++++++++++++++*/

char *ReadConfigFile(ConfigFile *config)
{
 static int first_time=1;
 FILE *file;
 char *line;

 file=fopen(config->name,"r");
 if(!file)
   {errmsg=(char*)malloc(48+strlen(config->name));sprintf(errmsg,"Cannot open the configuration file '%s'.",config->name);return(errmsg);}

 CreateBackupConfigFile(config);

 line_no=0;
 file_name=config->name;

 while((line=ReadLine(file)))
   {
    int s;

    for(s=0;s<config->nsections;s++)
       if(!strcmp(config->sections[s]->name,line))
          break;

    if(s==config->nsections)
      {errmsg=(char*)malloc(48+strlen(line));sprintf(errmsg,"Unrecognised section label '%s'.",line);break;}

    if(!(line=ReadLine(file)))
      {errmsg=(char*)malloc(32);strcpy(errmsg,"Unexpected end of file.");break;}

    if(*line!='{' && *line!='[')
      {errmsg=(char*)malloc(48);strcpy(errmsg,"Start of section must be '{' or '['.");break;}

    if(*(line+1))
      {*(line+1)=0;errmsg=(char*)malloc(48);sprintf(errmsg,"Start of section '%c' has trailing junk.",*line);break;}

    if(!first_time && s==0)
      {
       while((line=ReadLine(file)))
          if(*line=='}' || *line==']')
             break;
      }
    else if(*line=='{')
      {
       ReadSection(file,config->sections[s]);
      }
    else if(*line=='[')
      {
       int save_line_no;
       char *name,*r;
       FILE *inc_file;

       if(!(line=ReadLine(file)))
         {errmsg=(char*)malloc(32);strcpy(errmsg,"Unexpected end of file.");break;}

       name=(char*)malloc(strlen(config->name)+strlen(line)+1);

       strcpy(name,config->name);

       r=name+strlen(name)-1;
       while(r>name && *r!='/')
          r--;

       strcpy(r+1,line);

       if(!(inc_file=fopen(name,"r")))
         {errmsg=(char*)malloc(32+strlen(name));sprintf(errmsg,"Cannot open included file '%s.",name);break;}

       save_line_no=line_no;
       file_name=name;
       line_no=0;

       ReadSection(inc_file,config->sections[s]);

       fclose(inc_file);

       if(errmsg)
          break;

       line_no=save_line_no;
       file_name=config->name;

       if(!(line=ReadLine(file)))
         {errmsg=(char*)malloc(32);strcpy(errmsg,"Unexpected end of file.");break;}

       if(*line!=']')
         {errmsg=(char*)malloc(48);strcpy(errmsg,"End of section must be '}' or ']'.");break;}
      }

    if(errmsg)
       break;
   }

 if(errmsg)
    RestoreBackupConfigFile(config);
 else
    PurgeBackupConfigFile();

 fclose(file);

 if(errmsg)
   {
    char *newerrmsg=(char*)malloc(strlen(errmsg)+64+strlen(file_name));
    sprintf(newerrmsg,"Configuration file syntax error at line %d in '%s'; %s\n",line_no,file_name,errmsg);
    free(errmsg);
    errmsg=newerrmsg;
   }
 else
    if(first_time)
       first_time=0;

#if CONFIG_DEBUG_DUMP
 DumpConfigFile(config);
#endif

 return(errmsg);
}


/*++++++++++++++++++++++++++++++++++++++
  Read all the data from a section.

  int ReadSection Returns 1 if there was an error.

  FILE *file The file to read from.

  ConfigSection *section The section to read.
  ++++++++++++++++++++++++++++++++++++++*/

static int ReadSection(FILE *file,ConfigSection *section)
{
 char *line;

 while((line=ReadLine(file)))
   {
    if(*line=='}' || *line==']')
      {
       if(*(line+1))
         {*(line+1)=0;errmsg=(char*)malloc(48);sprintf(errmsg,"End of section '%c' has trailing junk.",*line);return(1);}
       return(0);
      }
    else
      {
       int i;
       char *u=NULL,*l=NULL,*r=NULL;

       if(*line=='<')
         {
          char *uu;

          uu=u=line+1;
          while(*uu && *uu!='>')
             uu++;
          if(!*uu)
            {errmsg=(char*)malloc(32);strcpy(errmsg,"No '>' to match the '<'.");return(1);}

          *uu=0;
          l=uu+1;
          while(*l && isspace(*l))
             l++;
          if(!*l)
            {errmsg=(char*)malloc(48);strcpy(errmsg,"No configuration item following the '<...>'.");return(1);}
         }
       else
          l=line;

       for(i=0;i<section->nentries;i++)
          if(!*section->entries[i].name || !strncmp(section->entries[i].name,l,strlen(section->entries[i].name)))
            {
             char *ll;

             if(*section->entries[i].name)
               {
                ll=l+strlen(section->entries[i].name);

                if(*ll && *ll!='=' && !isspace(*ll))
                   continue;
               }
             else
               {
                ll=l;
                while(*ll && *ll!='=' && !isspace(*ll))
                   ll++;
               }

             if(section->entries[i].url_type==0 && u)
               {errmsg=(char*)malloc(48);strcpy(errmsg,"No URL context '<...>' allowed for this item.");return(1);}

             if(section->entries[i].val_type==None)
               {
                *ll=0;
                r=NULL;
               }
             else
               {
                r=strchr(l,'=');
                if(!r)
                  {errmsg=(char*)malloc(32);strcpy(errmsg,"No equal sign for item.");return(1);}

                *ll=0;
                if(!*l)
                  {errmsg=(char*)malloc(48);strcpy(errmsg,"Nothing to the left of the equal sign.");return(1);}

                r++;
                while(isspace(*r))
                   r++;
               }

             if(ParseEntry(&section->entries[i],u,l,r))
                return(1);

             break;
            }

       if(i==section->nentries)
         {errmsg=(char*)malloc(32+strlen(l));sprintf(errmsg,"Unrecognised entry '%s'.",l);return(1);}
      }
   }

 return(0);
}


/*++++++++++++++++++++++++++++++++++++++
  Read a line from the configuration file, skipping over comments and blank lines.

  char *ReadLine Returns a pointer to the first thing in the line.

  FILE *file The file to read from.
  ++++++++++++++++++++++++++++++++++++++*/

static char *ReadLine(FILE *file)
{
 static char *line=NULL;
 char *l=NULL,*r;

 while((line=fgets_realloc(line,file)))
   {
    l=line;
    r=line+strlen(line)-1;

    line_no++;

    while(isspace(*l))
       l++;

    if(!*l || *l=='#')
       continue;

    while(r>l && isspace(*r))
       *r--=0;

    break;
   }

 if(!line)
    l=NULL;

 return(l);
}


/*++++++++++++++++++++++++++++++++++++++
  Parse an entry from the file.

  int ParseEntry Return 1 in case of error.

  ConfigEntry *entries The list of matching entry in the section.

  char *url_str The string for the URL specification.

  char *key_str The string for the key.

  char *val_str The string to the value.
  ++++++++++++++++++++++++++++++++++++++*/

static int ParseEntry(ConfigEntry *entry,char *url_str,char *key_str,char *val_str)
{
 ConfigItem *item=entry->item;
 UrlSpec *url;
 KeyOrValue key,val;

 if(entry->key_type==Fixed && entry->url_type==0 && (*item) && (*item)->nitems)
   {
    errmsg=(char*)malloc(32+strlen(key_str));sprintf(errmsg,"Duplicated entry: '%s'.",key_str);
    return(1);
   }

 if(!entry->url_type || !url_str)
    url=NULL;
 else
    if(ParseItem(url_str,UrlSpecification,(KeyOrValue*)&url))
       return(1);

 if(entry->key_type==Fixed)
    key.string=entry->name;
 else
    if(ParseItem(key_str,entry->key_type,&key))
       return(1);

 if(!val_str)
    val.string=NULL;
 else if(entry->val_type==None)
    val.string=NULL;
 else
    if(ParseItem(val_str,entry->val_type,&val))
       return(1);

 if(!(*item))
   {
    *item=(ConfigItem)malloc(sizeof(struct _ConfigItem));
    (*item)->entry=entry;
    (*item)->nitems=0;
    (*item)->defval=NULL;
   }
 if(!(*item)->nitems)
   {
    (*item)->nitems=1;
    (*item)->url=(UrlSpec**)malloc(sizeof(UrlSpec*));
    (*item)->key=(KeyOrValue*)malloc(sizeof(KeyOrValue));
    (*item)->val=(KeyOrValue*)malloc(sizeof(KeyOrValue));
   }
 else
   {
    (*item)->nitems++;
    (*item)->url=(UrlSpec**)realloc((void*)(*item)->url,(*item)->nitems*sizeof(UrlSpec*));
    (*item)->key=(KeyOrValue*)realloc((void*)(*item)->key,(*item)->nitems*sizeof(KeyOrValue));
    (*item)->val=(KeyOrValue*)realloc((void*)(*item)->val,(*item)->nitems*sizeof(KeyOrValue));
   }

 (*item)->url[(*item)->nitems-1]=url;
 (*item)->key[(*item)->nitems-1]=key;
 (*item)->val[(*item)->nitems-1]=val;

 return(0);
}


/*++++++++++++++++++++++++++++++++++++++
  Parse the text and put a value into the location.

  int ParseItem Returns non-zero in case of error.

  char *text The text string to parse.

  ConfigType type The type we are looking for.

  KeyOrValue *pointer The location to store the key or value.
  ++++++++++++++++++++++++++++++++++++++*/

static int ParseItem(char *text,ConfigType type,KeyOrValue *pointer)
{
 switch(type)
   {
   case Fixed:
   case None:
    break;

   case CfgMaxServers:
    if(!*text)
      {errmsg=(char*)malloc(56);strcpy(errmsg,"Expecting a maximum server count, got nothing.");}
    else if(!isanumber(text))
      {errmsg=(char*)malloc(48+strlen(text));sprintf(errmsg,"Expecting a maximum server count, got '%s'.",text);}
    else
      {
       pointer->integer=atoi(text);
       if(pointer->integer<=0 || pointer->integer>MAX_SERVERS)
         {errmsg=(char*)malloc(48);sprintf(errmsg,"Invalid maximum server count: %d.",pointer->integer);}
      }
    break;

   case CfgMaxFetchServers:
    if(!*text)
      {errmsg=(char*)malloc(56);strcpy(errmsg,"Expecting a maximum fetch server count, got nothing.");}
    else if(!isanumber(text))
      {errmsg=(char*)malloc(56+strlen(text));sprintf(errmsg,"Expecting a maximum fetch server count, got '%s'.",text);}
    else
      {
       pointer->integer=atoi(text);
       if(pointer->integer<=0 || pointer->integer>MAX_FETCH_SERVERS)
         {errmsg=(char*)malloc(48);sprintf(errmsg,"Invalid maximum fetch server count: %d.",pointer->integer);}
      }
    break;

   case CfgLogLevel:
    if(!*text)
      {errmsg=(char*)malloc(48);strcpy(errmsg,"Expecting a log level, got nothing.");}
    else if(strcasecmp(text,"debug")==0)
       pointer->integer=Debug;
    else if(strcasecmp(text,"info")==0)
       pointer->integer=Inform;
    else if(strcasecmp(text,"important")==0)
       pointer->integer=Important;
    else if(strcasecmp(text,"warning")==0)
       pointer->integer=Warning;
    else if(strcasecmp(text,"fatal")==0)
       pointer->integer=Fatal;
    else
      {errmsg=(char*)malloc(48+strlen(text));sprintf(errmsg,"Expecting a log level, got '%s'.",text);}
    break;

   case Boolean:
    if(!*text)
      {errmsg=(char*)malloc(48);strcpy(errmsg,"Expecting a Boolean, got nothing.");}
    else if(!strcasecmp(text,"yes") || !strcasecmp(text,"true"))
       pointer->integer=1;
    else if(!strcasecmp(text,"no") || !strcasecmp(text,"false"))
       pointer->integer=0;
    else
      {errmsg=(char*)malloc(48+strlen(text));sprintf(errmsg,"Expecting a Boolean, got '%s'.",text);}
    break;

   case PortNumber:
    if(!*text)
      {errmsg=(char*)malloc(48);strcpy(errmsg,"Expecting a port number, got nothing.");}
    else if(!isanumber(text))
      {errmsg=(char*)malloc(40+strlen(text));sprintf(errmsg,"Expecting a port number, got '%s'.",text);}
    else
      {
       pointer->integer=atoi(text);
       if(pointer->integer<=0 || pointer->integer>65535)
         {errmsg=(char*)malloc(32);sprintf(errmsg,"Invalid port number %d.",pointer->integer);}
      }
    break;

   case AgeDays:
    if(!*text)
      {errmsg=(char*)malloc(48);strcpy(errmsg,"Expecting an age in days, got nothing.");}
    else if(isanumber(text))
       pointer->integer=atoi(text);
    else
      {
       int val,len;
       char suffix;

       if(sscanf(text,"%d%1c%n",&val,&suffix,&len)==2 && len==strlen(text) &&
          (suffix=='d' || suffix=='w' || suffix=='m' || suffix=='y'))
         {
          if(suffix=='y')
             pointer->integer=val*365;
          else if(suffix=='m')
             pointer->integer=val*30;
          else if(suffix=='w')
             pointer->integer=val*7;
          else /* if(suffix=='d') */
             pointer->integer=val;
         }
       else
         {errmsg=(char*)malloc(40+strlen(text));sprintf(errmsg,"Expecting an age in days, got '%s'.",text);}
      }
    break;

   case TimeSecs:
    if(!*text)
      {errmsg=(char*)malloc(48);strcpy(errmsg,"Expecting a time in seconds, got nothing.");}
    else if(isanumber(text))
       pointer->integer=atoi(text);
    else
      {
       int val,len;
       char suffix;

       if(sscanf(text,"%d%1c%n",&val,&suffix,&len)==2 && len==strlen(text) &&
          (suffix=='s' || suffix=='m' || suffix=='h' || suffix=='d' || suffix=='w'))
         {
          if(suffix=='w')
             pointer->integer=val*3600*24*7;
          else if(suffix=='d')
             pointer->integer=val*3600*24;
          else if(suffix=='h')
             pointer->integer=val*3600;
          else if(suffix=='m')
             pointer->integer=val*60;
          else /* if(suffix=='s') */
             pointer->integer=val;
         }
       else
         {errmsg=(char*)malloc(40+strlen(text));sprintf(errmsg,"Expecting a time in seconds, got '%s'.",text);}
      }
    break;

   case CacheSize:
    if(!*text)
      {errmsg=(char*)malloc(48);strcpy(errmsg,"Expecting a cache size in MB, got nothing.");}
    else if(!isanumber(text))
      {errmsg=(char*)malloc(40+strlen(text));sprintf(errmsg,"Expecting a cache size in MB, got '%s'.",text);}
    else
      {
       pointer->integer=atoi(text);
       if(pointer->integer<0)
         {errmsg=(char*)malloc(48);sprintf(errmsg,"Invalid cache size %d.",pointer->integer);}
      }
    break;

   case FileSize:
    if(!*text)
      {errmsg=(char*)malloc(48);strcpy(errmsg,"Expecting a file size in kB, got nothing.");}
    else if(!isanumber(text))
      {errmsg=(char*)malloc(40+strlen(text));sprintf(errmsg,"Expecting a file size in kB, got '%s'.",text);}
    else
      {
       pointer->integer=atoi(text);
       if(pointer->integer<0)
         {errmsg=(char*)malloc(48);sprintf(errmsg,"Invalid file size %d.",pointer->integer);}
      }
    break;

   case Percentage:
    if(!*text)
      {errmsg=(char*)malloc(48);strcpy(errmsg,"Expecting a percentage, got nothing.");}
    else if(!isanumber(text))
      {errmsg=(char*)malloc(40+strlen(text));sprintf(errmsg,"Expecting a percentage, got '%s'.",text);}
    else
      {
       pointer->integer=atoi(text);
       if(pointer->integer<0)
         {errmsg=(char*)malloc(48);sprintf(errmsg,"Invalid percentage %d.",pointer->integer);}
      }
    break;

   case UserId:
    if(!*text)
      {errmsg=(char*)malloc(48);strcpy(errmsg,"Expecting a username or uid, got nothing.");}
    else
      {
       int uid;
       struct passwd *userInfo=getpwnam(text);
       if(userInfo)
          uid=userInfo->pw_uid;
       else
         {
          if(sscanf(text,"%d",&uid)!=1)
            {errmsg=(char*)malloc(32+strlen(text));sprintf(errmsg,"Invalid user %s.",text);}
          else if(uid!=-1 && !getpwuid(uid))
            {errmsg=(char*)malloc(32);sprintf(errmsg,"Unknown user id %d.",uid);}
         }
       pointer->integer=uid;
      }
    break;

   case GroupId:
    if(!*text)
      {errmsg=(char*)malloc(48);strcpy(errmsg,"Expecting a group name or gid, got nothing.");}
    else
      {
       int gid;
       struct group *groupInfo=getgrnam(text);
       if(groupInfo)
          gid=groupInfo->gr_gid;
       else
         {
          if(sscanf(text,"%d",&gid)!=1)
            {errmsg=(char*)malloc(32+strlen(text));sprintf(errmsg,"Invalid group %s.",text);}
          else if(gid!=-1 && !getgrgid(gid))
            {errmsg=(char*)malloc(32);sprintf(errmsg,"Unknown group id %d.",gid);}
         }
       pointer->integer=gid;
      }
    break;

   case String:
    if(!*text || !strcasecmp(text,"none"))
       pointer->string=NULL;
    else
      {
       pointer->string=(char*)malloc(strlen(text)+1);
       strcpy(pointer->string,text);
      }
    break;

   case PathName:
    if(!*text)
      {errmsg=(char*)malloc(48);strcpy(errmsg,"Expecting a pathname, got nothing.");}
    else if(*text!='/')
      {errmsg=(char*)malloc(48+strlen(text));sprintf(errmsg,"Expecting an absolute pathname, got '%s'.",text);}
    else
      {
       pointer->string=(char*)malloc(strlen(text)+1);
       strcpy(pointer->string,text);
      }
    break;

   case FileExt:
    if(!*text)
      {errmsg=(char*)malloc(40);strcpy(errmsg,"Expecting a file extension, got nothing.");}
    else if(*text!='.')
      {errmsg=(char*)malloc(40+strlen(text));sprintf(errmsg,"Expecting a file extension, got '%s'.",text);}
    else
      {
       pointer->string=(char*)malloc(strlen(text)+1);
       strcpy(pointer->string,text);
      }
    break;

   case FileMode:
    if(!*text)
      {errmsg=(char*)malloc(40);strcpy(errmsg,"Expecting a file permissions mode, got nothing.");}
    else if(!isanumber(text) || *text!='0')
      {errmsg=(char*)malloc(48+strlen(text));sprintf(errmsg,"Expecting a file permissions mode, got '%s'.",text);}
    else
       pointer->integer=strtol(text,NULL,8);
    break;

   case MIMEType:
     if(!*text)
       {errmsg=(char*)malloc(40);strcpy(errmsg,"Expecting a MIME Type, got nothing.");}
     else
       {
        char *slash=strchr(text,'/');
        if(!slash)
          {errmsg=(char*)malloc(48+strlen(text));sprintf(errmsg,"Expecting a MIME Type/Subtype, got '%s'.",text);}
        pointer->string=(char*)malloc(strlen(text)+1);
        strcpy(pointer->string,text);
       }
    break;

   case HostOrNone:
    if(!*text || !strcasecmp(text,"none"))
      {
       pointer->string=NULL;
       break;
      }

    /* fall through */

   case Host:
     if(!*text)
       {errmsg=(char*)malloc(40);strcpy(errmsg,"Expecting a hostname, got nothing.");}
     else
       {
        int i,colons=0;
        /* This is tricky, we must check for a host:port combination.  But an IPv6 address with no '[...]'
           URL-type formatting cannot have a port anyway (e.g. is 'abcd:*' 'host:port' or IPv6 wildcard?).
           If there are 2 or more ':' then it is IPv6, if it starts with '[' it is IPv6, if it has zero or
           one ':' then it must be a hostname/IPv4 and port
        */

        for(i=0;text[i];i++)
           if(text[i]==':')
              colons++;

        if(colons==1 && *text!='[')
          {errmsg=(char*)malloc(56+strlen(text));sprintf(errmsg,"Expecting a hostname without a port number, got '%s'.",text);}
        else
          {
           char *p;
           pointer->string=(char*)malloc(strlen(text)+1);
           if(*text=='[')
             {
              strcpy(pointer->string,text+1);
              pointer->string[strlen(pointer->string)-1]=0;
             }
           else
              strcpy(pointer->string,text);
           for(p=pointer->string;*p;p++)
              *p=tolower(*p);
          }
       }
     break;

   case HostAndPortOrNone:
    if(!*text || !strcasecmp(text,"none"))
      {
       pointer->string=NULL;
       break;
      }

    /* fall through */

   case HostAndPort:
    if(!*text)
      {errmsg=(char*)malloc(48);strcpy(errmsg,"Expecting a hostname (and port), got nothing.");}
    else
      {
       char *hoststr,*portstr;
       SplitHostPort(text,&hoststr,&portstr);
       RejoinHostPort(text,hoststr,portstr);
       if(*text==':')
         {errmsg=(char*)malloc(48+strlen(text));sprintf(errmsg,"Expecting a hostname before the ':', got '%s'.",text);}
       else
         {
          if(portstr && (!isanumber(portstr) || atoi(portstr)<=0 || atoi(portstr)>65535))
            {errmsg=(char*)malloc(32+strlen(portstr));sprintf(errmsg,"Invalid port number %s.",portstr);}
          else
            {
             char *p;
             pointer->string=(char*)malloc(strlen(text)+1);
             strcpy(pointer->string,text);
             for(p=pointer->string;*p;p++)
                *p=tolower(*p);
            }
         }
      }
    break;

   case UserPass:
    if(!*text)
      {errmsg=(char*)malloc(48);strcpy(errmsg,"Expecting a username and password, got nothing.");}
    else if(!strchr(text,':'))
      {errmsg=(char*)malloc(48+strlen(text));sprintf(errmsg,"Expecting a username and password, got '%s'.",text);}
    else
       pointer->string=Base64Encode(text,strlen(text));
    break;

   case Url:
    if(!*text || !strcasecmp(text,"none"))
       pointer->string=NULL;
    else
      {
       URL *tempUrl=SplitURL(text);
       if(!tempUrl->Protocol)
         {errmsg=(char*)malloc(32+strlen(text));sprintf(errmsg,"Expecting a URL, got '%s'.",text);}
       else
         {
          pointer->string=(char*)malloc(strlen(text)+1);
          strcpy(pointer->string,text);
          FreeURL(tempUrl);
         }
      }
    break;

   case UrlSpecification:
    if(!*text)
      {errmsg=(char*)malloc(64);strcpy(errmsg,"Expecting a URL-SPECIFICATION, got nothing.");}
    else
      {
       char *p,*orgtext=text;

       pointer->urlspec=(UrlSpec*)malloc(sizeof(UrlSpec));
       pointer->urlspec->negated=0;
       pointer->urlspec->proto=NULL;
       pointer->urlspec->host=NULL;
       pointer->urlspec->port=-1;
       pointer->urlspec->path=NULL;
       pointer->urlspec->args=NULL;

       /* ! */

       if(*text=='!')
         {
          pointer->urlspec->negated=1;
          text++;
         }

       /* protocol */

       if(!strncmp(text,"*://",4))
         {
          p=text+4;
          pointer->urlspec->proto=NULL;
         }
       else if(!strncmp(text,"://",3))
         {
          p=text+3;
          pointer->urlspec->proto=NULL;
         }
       else if((p=strstr(text,"://")))
         {
          pointer->urlspec->proto=(char*)malloc(p-text+1);
          strncpy(pointer->urlspec->proto,text,p-text);
          *(pointer->urlspec->proto+(p-text))=0;
          p+=3;
         }
       else
         {errmsg=(char*)malloc(64+strlen(orgtext));sprintf(errmsg,"Expecting a URL-SPECIFICATION, got this '%s'.",orgtext); break;}

       if(pointer->urlspec->proto)
         {
          char *q;
          for(q=pointer->urlspec->proto;*q;q++)
             *q=tolower(*q);
         }

       text=p;

       /* host */

       if(*text=='*' && (*(text+1)=='/' || !*(text+1)))
          p=text+1;
       else if(*text==':')
          p=text;
       else if(*text=='[' && (p=strchr(text,']')))
         {
          p++;
          pointer->urlspec->host=(char*)malloc(p-text+1);
          strncpy(pointer->urlspec->host,text,p-text);
          *(pointer->urlspec->host+(p-text))=0;
         }
       else if((p=strchr(text,':')) && p<strchr(text,'/'))
         {
          pointer->urlspec->host=(char*)malloc(p-text+1);
          strncpy(pointer->urlspec->host,text,p-text);
          *(pointer->urlspec->host+(p-text))=0;
         }
       else if((p=strchr(text,'/')))
         {
          pointer->urlspec->host=(char*)malloc(p-text+1);
          strncpy(pointer->urlspec->host,text,p-text);
          *(pointer->urlspec->host+(p-text))=0;
         }
       else if(*text)
         {
          pointer->urlspec->host=(char*)malloc(strlen(text)+1);
          strcpy(pointer->urlspec->host,text);
          p=text+strlen(text);
         }
       else
          p=text;

       if(pointer->urlspec->host)
         {
          char *q;
          for(q=pointer->urlspec->host;*q;q++)
             *q=tolower(*q);
         }

       text=p;

       /* port */

       if(*text==':' && isdigit(*(text+1)))
         {
          pointer->urlspec->port=atoi(text+1);
          p=text+1;
          while(isdigit(*p))
             p++;
         }
       else if(*text==':' && (*(text+1)=='/' || *(text+1)==0))
         {
          pointer->urlspec->port=0;
          p=text+1;
         }
       else if(*text==':' && *(text+1)=='*')
         {
          p=text+2;
         }
       else if(*text==':')
         {errmsg=(char*)malloc(64+strlen(orgtext));sprintf(errmsg,"Expecting a URL-SPECIFICATION, got this '%s'.",orgtext); break;}

       text=p;

       /* path */

       if(!*text)
          ;
       else if(*text=='/' && !*(text+1))
          p=text+1;
       else if(*text=='?')
          ;
       else if(*text=='/' && (p=strchr(text,'?')))
         {
          pointer->urlspec->path=(char*)malloc(p-text+1);
          strncpy(pointer->urlspec->path,text,p-text);
          pointer->urlspec->path[p-text]=0;
         }
       else if(*text=='/')
         {
          pointer->urlspec->path=(char*)malloc(strlen(text)+1);
          strcpy(pointer->urlspec->path,text);
          p=text+strlen(text);
         }
       else
         {errmsg=(char*)malloc(64+strlen(orgtext));sprintf(errmsg,"Expecting a URL-SPECIFICATION, got this '%s'.",orgtext); break;}

       text=p;

       /* args */

       if(!*text)
          ;
       else if(*text=='?' && !*(text+1))
          pointer->urlspec->args="";
       else if(*text=='?')
         {
          pointer->urlspec->args=(char*)malloc(strlen(text+1)+1);
          strcpy(pointer->urlspec->args,text+1);
         }
       else
         {errmsg=(char*)malloc(64+strlen(orgtext));sprintf(errmsg,"Expecting a URL-SPECIFICATION, got this '%s'.",orgtext); break;}

      }
    break;
   }

 if(errmsg)
    return(1);
 else
    return(0);
}


/*++++++++++++++++++++++++++++++++++++++
  Decide if a string is an integer.

  int isanumber Returns 1 if it is, 0 if not.

  const char *string The string that may be an integer.
  ++++++++++++++++++++++++++++++++++++++*/

static int isanumber(const char *string)
{
 int i=0;

 if(string[i]=='-' || string[i]=='+')
    i++;

 for(;string[i];i++)
    if(!isdigit(string[i]))
       return(0);

 return(1);
}


/*++++++++++++++++++++++++++++++++++++++
  Save the old values in case the re-read of the file fails.

  ConfigFile* current The existing configuration file.
  ++++++++++++++++++++++++++++++++++++++*/

static void CreateBackupConfigFile(ConfigFile* current)
{
 int s,e;

 /* Create a backup of all of the sections. */

 backup=*current;
 backup.sections=(ConfigSection**)malloc(current->nsections*sizeof(ConfigSection*));

 for(s=0;s<current->nsections;s++)
   {
    backup.sections[s]=(ConfigSection*)malloc(sizeof(ConfigSection));
    *backup.sections[s]=*current->sections[s];
    backup.sections[s]->entries=(ConfigEntry*)malloc(current->sections[s]->nentries*sizeof(ConfigEntry));

    for(e=0;e<current->sections[s]->nentries;e++)
      {
       backup.sections[s]->entries[e]=current->sections[s]->entries[e];
       backup.sections[s]->entries[e].item=(ConfigItem*)malloc(sizeof(ConfigItem));

       *backup.sections[s]->entries[e].item=*current->sections[s]->entries[e].item;
       *current->sections[s]->entries[e].item=NULL;
      }
   }

 /* Restore the default values */

 DefaultConfigFile(current);

 /* Restore the StartUp section. */

 s=0;
 for(e=0;e<current->sections[s]->nentries;e++)
   {
    FreeConfigItem(*current->sections[s]->entries[e].item);

    *current->sections[s]->entries[e].item=*backup.sections[s]->entries[e].item;
   }

 free(backup.sections[s]->entries);
 free(backup.sections[s]);
}


/*++++++++++++++++++++++++++++++++++++++
  Restore the old values if the re-read of the file failed.

  ConfigFile* current The existing (broken) configuration file.
  ++++++++++++++++++++++++++++++++++++++*/

static void RestoreBackupConfigFile(ConfigFile* current)
{
 int s,e;

 /* Restore all of the sections except StartUp. */

 for(s=1;s<current->nsections;s++)
   {
    for(e=0;e<current->sections[s]->nentries;e++)
      {
       FreeConfigItem(*current->sections[s]->entries[e].item);

       *current->sections[s]->entries[e].item=*backup.sections[s]->entries[e].item;
      }

    free(backup.sections[s]->entries);
    free(backup.sections[s]);
   }

 free(backup.sections);
}


/*++++++++++++++++++++++++++++++++++++++
  Remove the old values if the re-read of the file succeeded.
  ++++++++++++++++++++++++++++++++++++++*/

static void PurgeBackupConfigFile(void)
{
 int s,e;

 /* Purge all of the sections except StartUp. */

 for(s=1;s<backup.nsections;s++)
   {
    for(e=0;e<backup.sections[s]->nentries;e++)
      {
       FreeConfigItem(*backup.sections[s]->entries[e].item);
      }

    free(backup.sections[s]->entries);
    free(backup.sections[s]);
   }

 free(backup.sections);
}


/*++++++++++++++++++++++++++++++++++++++
  Free a ConfigItem list.

  ConfigItem item The item to free.
  ++++++++++++++++++++++++++++++++++++++*/

static void FreeConfigItem(ConfigItem item)
{
 int i;

 if(!item)
    return;

 for(i=0;i<item->nitems;i++)
   {
    if(item->url[i])
      {
       if(item->url[i]->proto)
          free(item->url[i]->proto);
       if(item->url[i]->host)
          free(item->url[i]->host);
       if(item->url[i]->path)
          free(item->url[i]->path);
       if(item->url[i]->args && *item->url[i]->args)
          free(item->url[i]->args);
       free(item->url[i]);
      }

    switch(item->entry->key_type)
      {
       /* None or Fixed */

      case Fixed:
      case None:
       break;

       /* Integer */

      case CfgMaxServers:
      case CfgMaxFetchServers:
      case CfgLogLevel:
      case Boolean:
      case PortNumber:
      case AgeDays:
      case TimeSecs:
      case CacheSize:
      case FileSize:
      case Percentage:
      case UserId:
      case GroupId:
      case FileMode:
       break;

       /* String */

      case String:
      case PathName:
      case FileExt:
      case MIMEType:
      case HostOrNone:
      case Host:
      case HostAndPortOrNone:
      case HostAndPort:
      case UserPass:
      case Url:
       if(item->key[i].string)
          free(item->key[i].string);
       break;

      case UrlSpecification:
       if(item->key[i].urlspec->proto)
          free(item->key[i].urlspec->proto);
       if(item->key[i].urlspec->host)
          free(item->key[i].urlspec->host);
       if(item->key[i].urlspec->path)
          free(item->key[i].urlspec->path);
       if(item->key[i].urlspec->args && *item->key[i].urlspec->args)
          free(item->key[i].urlspec->args);
       free(item->key[i].urlspec);
      }

    switch(item->entry->val_type)
      {
       /* None or Fixed */

      case Fixed:
      case None:
       break;

       /* Integer */

      case CfgMaxServers:
      case CfgMaxFetchServers:
      case CfgLogLevel:
      case Boolean:
      case PortNumber:
      case AgeDays:
      case TimeSecs:
      case CacheSize:
      case FileSize:
      case Percentage:
      case UserId:
      case GroupId:
      case FileMode:
       break;

       /* String */

      case String:
      case PathName:
      case FileExt:
      case MIMEType:
      case HostOrNone:
      case Host:
      case HostAndPortOrNone:
      case HostAndPort:
      case UserPass:
      case Url:
       if(item->val[i].string)
          free(item->val[i].string);
       break;

       /* Url Specification */

      case UrlSpecification:
       if(item->val[i].urlspec->proto)
          free(item->val[i].urlspec->proto);
       if(item->val[i].urlspec->host)
          free(item->val[i].urlspec->host);
       if(item->val[i].urlspec->path)
          free(item->val[i].urlspec->path);
       if(item->val[i].urlspec->args && *item->val[i].urlspec->args)
          free(item->val[i].urlspec->args);
       free(item->val[i].urlspec);
      }
   }

 if(item->nitems)
   {
    free(item->url);
    free(item->key);
    free(item->val);
   }

 free(item);
}


/*++++++++++++++++++++++++++++++++++++++
  Check if a protocol, host, path and args match a URL-SPECIFICATION in the config file.

  int MatchUrlSpecification Return the matching length if true else 0.

  UrlSpec *spec The URL-SPECIFICATION.

  char *proto The protocol.

  char *host The host.

  char *path The path.

  char *args The arguments.
  ++++++++++++++++++++++++++++++++++++++*/

int MatchUrlSpecification(UrlSpec *spec,char *proto,char *host,char *path,char *args)
{
 char *hoststr,*portstr;
 int match=0;

 SplitHostPort(host,&hoststr,&portstr);

 if((!spec->proto || !strcmp(spec->proto,proto)) &&
    (!spec->host || WildcardMatch(hoststr,spec->host)) &&
    (spec->port==-1 || (!portstr && spec->port==0) || (portstr && atoi(portstr)==spec->port)) &&
    (!spec->path || WildcardMatch(path,spec->path)) &&
    (!spec->args || (args && WildcardMatch(args,spec->args)) || (!args && *spec->args==0)))
   {
    match=(spec->proto?strlen(spec->proto):0)+
          (spec->host?strlen(spec->host):0)+
          (spec->path?strlen(spec->path):0)+
          (spec->args?strlen(spec->args):0)+1;
   }

 RejoinHostPort(host,hoststr,portstr);

 return(match);
}


/*++++++++++++++++++++++++++++++++++++++
  Do a match using a wildcard specified with '*' in it.

  int WildcardMatch returns 1 if there is a match.

  char *string The fixed string that is being matched.

  char *pattern The pattern to match against.
  ++++++++++++++++++++++++++++++++++++++*/

int WildcardMatch(char *string,char *pattern)
{
 int i,nstars=0;

 for(i=0;pattern[i];i++)
    if(pattern[i]=='*')
       nstars++;

 if(nstars)
   {
    int len_pat=strlen(pattern);
    int len_str=strlen(string);

    if((len_pat-nstars)>len_str)
      {
       if(!strcmp(string,"/") && !strcmp(pattern,"/*/"))
          return(1);
       else
          return(0);
      }
    else
      {
       char *str_beg=NULL,*str_mid=NULL,*str_end=NULL;
       int len_beg=0,len_mid=0,len_end=0;

       /* Look for a star at the start. */

       if(pattern[0]!='*')
         {
          str_beg=pattern;
          while(pattern[len_beg]!='*')
             len_beg++;
          if(strncmp(str_beg,string,len_beg))
             return(0);
         }

       /* Look for a star at the end. */

       if(pattern[strlen(pattern)-1]!='*')
         {
          str_end=pattern+len_pat-2;
          while(*str_end!='*')
             str_end--;
          str_end++;
          len_end=pattern+len_pat-str_end;
          if(strncmp(str_end,string+len_str-len_end,len_end))
             return(0);
         }

       /* Check the rest. */

       if(nstars==1)
          return(1);
       else if(nstars==2)
         {
          char *copy=malloc(len_pat+1),*p;
          strcpy(copy,pattern);

          copy[len_beg]=0;
          copy[len_pat-1-len_end]=0;

          str_mid=copy+len_beg+1;
          len_mid=len_pat-len_beg-len_end-2;

          if((p=strstr(string+len_beg,str_mid)) && (p+len_mid)<=(string+len_str-len_end))
            {free(copy);return(1);}
          else
            {free(copy);return(0);}
         }
       else
          return(0);
      }
   }

 return(!strcmp(string,pattern));
}


#if CONFIG_DEBUG_DUMP

/*++++++++++++++++++++++++++++++++++++++
  Remove the old values if the re-read of the file succeeded.
  ++++++++++++++++++++++++++++++++++++++*/

static void DumpConfigFile(ConfigFile* config)
{
 int s,e;

 fprintf(stderr,"CONFIGURATION FILE\n");

 for(s=0;s<config->nsections;s++)
   {
    fprintf(stderr,"  Section %s\n",config->sections[s]->name);
    for(e=0;e<config->sections[s]->nentries;e++)
      {
       fprintf(stderr,"    Item %s\n",config->sections[s]->entries[e].name);
       DumpConfigItem(*config->sections[s]->entries[e].item);
      }
   }
}


/*++++++++++++++++++++++++++++++++++++++
  Dump a ConfigItem list to stderr.

  ConfigItem item The item to dump.
  ++++++++++++++++++++++++++++++++++++++*/

static void DumpConfigItem(ConfigItem item)
{
 int i;

 if(!item)
    return;

 for(i=0;i<item->nitems;i++)
   {
    fprintf(stderr,"      ");

    if(item->url[i])
      {
       if(item->url[i]->args)
          fprintf(stderr,"<%s://%s%s?%s> ",
                  item->url[i]->proto?item->url[i]->proto:"*",
                  item->url[i]->host ?item->url[i]->host :"*",
                  item->url[i]->path ?item->url[i]->path :"/*",
                  *item->url[i]->args?item->url[i]->args :"*");
       else
          fprintf(stderr,"<%s://%s%s> ",
                  item->url[i]->proto?item->url[i]->proto:"*",
                  item->url[i]->host ?item->url[i]->host :"*",
                  item->url[i]->path ?item->url[i]->path :"/*");
      }

    switch(item->entry->key_type)
      {
       /* None or Fixed */

      case Fixed:
      case None:
       fprintf(stderr,"%s",item->key[i].string);
       break;

       /* Integer */

      case CfgMaxServers:
      case CfgMaxFetchServers:
      case CfgLogLevel:
      case Boolean:
      case PortNumber:
      case AgeDays:
      case CacheSize:
      case FileSize:
      case Percentage:
      case UserId:
      case GroupId:
      case FileMode:
       fprintf(stderr,"%d",item->key[i].integer);
       break;

      case TimeSecs:
       fprintf(stderr,"%d",DurationToString(item->key[i].integer));
       break;

       /* String */

      case String:
      case PathName:
      case FileExt:
      case MIMEType:
      case HostOrNone:
      case Host:
      case HostAndPortOrNone:
      case HostAndPort:
      case UserPass:
      case Url:
       fprintf(stderr,"%s",item->key[i].string);
       break;

       /* Url Specification */

      case UrlSpecification:
       fprintf(stderr,"<%s://%s",
               item->key[i].urlspec->proto?item->key[i].urlspec->proto:"*",
               item->key[i].urlspec->host ?item->key[i].urlspec->host :"*");

       if(item->key[i].urlspec->port==-1)
          fprintf(stderr,":*");
       else if(item->key[i].urlspec->port!=0)
          fprintf(stderr,":%d",
                  item->key[i].urlspec->port);

       fprintf(stderr,"%s",
               item->key[i].urlspec->path ?item->key[i].urlspec->path :"/*");

       if(item->key[i].urlspec->args)
          fprintf(stderr,"?%s",
                  *item->key[i].urlspec->args?item->key[i].urlspec->args :"*");

       fprintf(stderr,"> ");
      }

    fprintf(stderr," = ");

    switch(item->entry->val_type)
      {
       /* None or Fixed */

      case Fixed:
      case None:
       fprintf(stderr,"%s",item->val[i].string);
       break;

       /* Integer */

      case CfgMaxServers:
      case CfgMaxFetchServers:
      case CfgLogLevel:
      case Boolean:
      case PortNumber:
      case AgeDays:
      case CacheSize:
      case FileSize:
      case Percentage:
      case UserId:
      case GroupId:
      case FileMode:
       fprintf(stderr,"%d",item->val[i].integer);
       break;

      case TimeSecs:
       fprintf(stderr,"%d",DurationToString(item->key[i].integer));
       break;

       /* String */

      case String:
      case PathName:
      case FileExt:
      case MIMEType:
      case HostOrNone:
      case Host:
      case HostAndPortOrNone:
      case HostAndPort:
      case UserPass:
      case Url:
       fprintf(stderr,"%s",item->val[i].string);
       break;

       /* Url Specification */

      case UrlSpecification:
       fprintf(stderr,"<%s://%s",
               item->val[i].urlspec->proto?item->val[i].urlspec->proto:"*",
               item->val[i].urlspec->host ?item->val[i].urlspec->host :"*");

       if(item->val[i].urlspec->port==-1)
          fprintf(stderr,":*");
       else if(item->val[i].urlspec->port!=0)
          fprintf(stderr,":%d",
                  item->val[i].urlspec->port);

       fprintf(stderr,"%s",
               item->val[i].urlspec->path ?item->val[i].urlspec->path :"/*");

       if(item->val[i].urlspec->args)
          fprintf(stderr,"?%s",
                  *item->val[i].urlspec->args?item->val[i].urlspec->args :"*");

       fprintf(stderr,"> ");
   }

    fprintf(stderr,"\n");
   }
}

#endif
