/***************************************
  $Header: /home/amb/wwwoffle/RCS/connect.c 2.26 2001/03/19 19:29:40 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 2.6b.
  Handle WWWOFFLE connections received by the demon.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1996,97,98,99,2000,01 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <time.h>
#include <signal.h>
#include <unistd.h>

#include "wwwoffle.h"
#include "misc.h"
#include "config.h"
#include "errors.h"
#include "sockets.h"


/*+ The time that the program went online. +*/
extern time_t OnlineTime;

/*+ The server sockets that we listen on +*/
extern int http_fd,             /*+ for the HTTP connections. +*/
           wwwoffle_fd;         /*+ for the WWWOFFLE connections. +*/

/*+ The proxy server that we use. +*/
extern char *proxy;

/*+ The online / offline / autodial status. +*/
extern int online;

/*+ The current number of active servers +*/
extern int n_servers,           /*+ in total. +*/
           n_fetch_servers;     /*+ fetching a previously requested page. +*/

/*+ The maximum number of servers +*/
extern int max_servers,                /*+ in total. +*/
           max_fetch_servers;          /*+ fetching a page. +*/

/*+ The wwwoffle client file descriptor when fetching. +*/
extern int fetch_fd;

/*+ The pids of the servers. +*/
extern int server_pids[MAX_SERVERS];

/*+ The pids of the servers that are fetching. +*/
extern int fetch_pids[MAX_FETCH_SERVERS];

/*+ The current status, fetching or not. +*/
extern int fetching;


/*++++++++++++++++++++++++++++++++++++++
  Parse a request that comes from wwwoffle.

  int client The file descriptor that corresponds to the wwwoffle connection.
  ++++++++++++++++++++++++++++++++++++++*/

void CommandConnect(int client)
{
 char *line=NULL;

 if(!(line=read_line_or_timeout(client,line)))
   {PrintMessage(Warning,"Nothing to read from the wwwoffle control socket [%!s]."); return;}

 if(strncmp(line,"WWWOFFLE ",9))
   {
    PrintMessage(Warning,"WWWOFFLE Not a command."); /* Used in audit-usage.pl */
    return;
   }

 if(ConfigString(PassWord) || !strncmp(&line[9],"PASSWORD ",9))
   {
    char *password;

    if(strlen(line)<18)
       password="";
    else
      {
       int i;

       for(i=18;line[i];i++)
          if(line[i]=='\r' || line[i]=='\n')
             line[i]=0;

       password=&line[18];
      }

    if(strcmp(password,ConfigString(PassWord)))
      {
       write_string(client,"WWWOFFLE Incorrect Password\n");
       PrintMessage(Warning,"WWWOFFLE Incorrect Password."); /* Used in audit-usage.pl */
       return;
      }

    if(!(line=read_line_or_timeout(client,line)))
      {PrintMessage(Warning,"Unexpected end of wwwoffle control command [%!s]."); return;}

    if(strncmp(line,"WWWOFFLE ",9))
      {
       PrintMessage(Warning,"WWWOFFLE Not a command."); /* Used in audit-usage.pl */
       return;
      }
   }

 if(!strncmp(&line[9],"ONLINE",6))
   {
    if(online==1)
       write_string(client,"WWWOFFLE Already Online\n");
    else
      {
       write_string(client,"WWWOFFLE Now Online\n");
       PrintMessage(Important,"WWWOFFLE Online."); /* Used in audit-usage.pl */

       CycleLastTimeSpoolFile();

       ForkRunModeScript(ConfigString(RunOnline),"online",NULL,client);
      }

    OnlineTime=time(NULL);
    online=1;
   }
 else if(!strncmp(&line[9],"AUTODIAL",8))
   {
    if(online==-1)
       write_string(client,"WWWOFFLE Already in Autodial Mode\n");
    else
      {
       write_string(client,"WWWOFFLE Now In Autodial Mode\n");
       PrintMessage(Important,"WWWOFFLE In Autodial Mode."); /* Used in audit-usage.pl */

       ForkRunModeScript(ConfigString(RunAutodial),"autodial",NULL,client);
      }

    OnlineTime=time(NULL);
    online=-1;
   }
 else if(!strncmp(&line[9],"OFFLINE",7))
   {
    if(!online)
       write_string(client,"WWWOFFLE Already Offline\n");
    else
      {
       write_string(client,"WWWOFFLE Now Offline\n");
       PrintMessage(Important,"WWWOFFLE Offline."); /* Used in audit-usage.pl */

       ForkRunModeScript(ConfigString(RunOffline),"offline",NULL,client);
      }

    online=0;
   }
 else if(!strncmp(&line[9],"FETCH",5))
   {
    if(fetch_fd!=-1)
       write_string(client,"WWWOFFLE Already fetching.\n");
    else if(online==0)
       write_string(client,"WWWOFFLE Must be online or autodial to fetch.\n");
    else
      {
       write_string(client,"WWWOFFLE Now Fetching.\n");
       PrintMessage(Important,"WWWOFFLE Fetch."); /* Used in audit-usage.pl */

       RequestMonitoredPages();

       CycleLastOutSpoolFile();

       fetch_fd=client;
       fetching=1;

       ForkRunModeScript(ConfigString(RunFetch),"fetch","start",fetch_fd);
      }
   }
 else if(!strncmp(&line[9],"CONFIG",6))
   {
    write_string(client,"WWWOFFLE Reading Configuration File.\n");
    PrintMessage(Important,"WWWOFFLE Re-reading Configuration File."); /* Used in audit-usage.pl */

    if(ReadConfigurationFile(client))
      {
       PrintMessage(Warning,"Error in configuration file; keeping old values.");
       write_string(client,"WWWOFFLE Error Reading Configuration File.\n");
      }
    else
       write_string(client,"WWWOFFLE Read Configuration File.\n");

    PrintMessage(Important,"WWWOFFLE Finished Re-reading Configuration File.");
   }
 else if(!strncmp(&line[9],"PURGE",5))
   {
    pid_t pid;

    if((pid=fork())==-1)
      {PrintMessage(Warning,"Cannot fork to do a purge [%!s].");return;}
    else if(!pid)
      {
       if(fetch_fd!=-1)
          CloseSocket(fetch_fd);

       CloseSocket(http_fd);
       CloseSocket(wwwoffle_fd);

       write_string(client,"WWWOFFLE Purge Starting.\n");
       PrintMessage(Important,"WWWOFFLE Purge."); /* Used in audit-usage.pl */

       PurgeCache(client);

       write_string(client,"WWWOFFLE Purge Finished.\n");
       PrintMessage(Important,"WWWOFFLE Purge finished.");

       exit(0);
      }
   }
 else if(!strncmp(&line[9],"KILL",4))
   {
    write_string(client,"WWWOFFLE Kill Signalled.\n");
    PrintMessage(Important,"WWWOFFLE Kill."); /* Used in audit-usage.pl */

    kill(getpid(),SIGTERM);
   }
 else
   {
    while(line[strlen(line)-1]=='\r' || line[strlen(line)-1]=='\n')
       line[strlen(line)-1]=0;

    write_formatted(client,"WWWOFFLE Unknown Command '%s'.",line);
    PrintMessage(Warning,"WWWOFFLE Unknown control command '%s'.",line); /* Used in audit-usage.pl */
   }

 if(line)
    free(line);
}


/*++++++++++++++++++++++++++++++++++++++
  Run the associated program when changing mode.

  char *filename The name of the program to run or NULL if none.

  char *mode The new mode to use as the argument.

  char *arg An additional argument (used for fetch mode).

  int client The current client socket to be closed.
  ++++++++++++++++++++++++++++++++++++++*/

void ForkRunModeScript(char *filename,char *mode,char *arg,int client)
{
 pid_t pid;

 if(!filename)
    return;

 if((pid=fork())==-1)
   {PrintMessage(Warning,"Cannot fork to run the run-%s program [%!s].",mode);return;}
 else if(!pid) /* The child */
   {
    if(fetch_fd!=-1)
       CloseSocket(fetch_fd);

    CloseSocket(http_fd);
    CloseSocket(wwwoffle_fd);

    if(client!=fetch_fd)
       CloseSocket(client);

    if(arg)
       execl(filename,filename,mode,arg,NULL);
    else
       execl(filename,filename,mode,NULL);

    PrintMessage(Warning,"Cannot exec the run-%s program '%s' [%!s].",filename);
    exit(1);
   }
}


/*++++++++++++++++++++++++++++++++++++++
  Fork a wwwoffles server.

  int client The file descriptor that the data comes in on.

  int browser Set to true if there is a browser connection.
  ++++++++++++++++++++++++++++++++++++++*/

void ForkServer(int client,int browser)
{
 pid_t pid;
 int i;

 if((pid=fork())==-1)
    PrintMessage(Warning,"Cannot fork a server [%!s].");
 else if(pid) /* The parent */
   {
    for(i=0;i<max_servers;i++)
       if(server_pids[i]==0)
         {server_pids[i]=pid;break;}

    n_servers++;

    if(online!=0 && !browser)
      {
       for(i=0;i<max_fetch_servers;i++)
          if(fetch_pids[i]==0)
            {fetch_pids[i]=pid;break;}

       n_fetch_servers++;
      }

    /* Used in audit-usage.pl */
    PrintMessage(Inform,"Forked wwwoffles -%s (pid=%d).",online==1?browser?"real":"fetch":online==-1?"autodial":"spool",pid);
   }
 else /* The child */
   {
    int status;

    if(fetch_fd!=-1 && fetch_fd!=client)
       CloseSocket(fetch_fd);

    CloseSocket(http_fd);
    CloseSocket(wwwoffle_fd);

    status=wwwoffles(online,browser,client);

    exit(status);
   }
}
