/***************************************
  $Header: /home/amb/wwwoffle/RCS/endian-cache.c 1.4 1998/09/30 18:05:27 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 2.4.
  Correct the the cache for the endian problem.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1998 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <time.h>
#include <utime.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#include "wwwoffle.h"
#include "misc.h"
#include "errors.h"
#include "config.h"

static void UpgradeProto(char *proto);
static void UpgradeHost(char *proto,char *host);
static void UpgradeSpecial(char *dirname);

static int rootp=0;


int main(int argc,char** argv)
{
 long endiancheck=1;
 int i;
 struct stat buf;

 /* Initialise */

 if(*(char*)&endiancheck)
   {fprintf(stderr,"This is a little-endian machine no change needed\n");exit(0);}

 if(argc!=2)
   {fprintf(stderr,"Usage: endian-cache <spool-dir>\n");exit(1);}

 InitErrorHandler("endian-cache",0,1);

 rootp=!geteuid();

 /* Change to the spool directory. */

 if(chdir(argv[1]))
    PrintMessage(Fatal,"Cannot change to the spool directory [%!s]; upgrade failed.");

 /* Create the new spool directory. */

 for(i=0;i<3;i++)
   {
    char *proto,*old;

    if(i==0)
       proto="http",old="http.old";
    else if(i==1)
       proto="ftp",old="ftp.old";
    else
       proto="finger",old="finger.old";

    if(stat(proto,&buf))
       PrintMessage(Inform,"Cannot stat the '%s' directory [%!s] not upgraded.",proto);
    else
      {
       printf("Converting '%s' directory\n",proto);

       if(rename(proto,old))
          PrintMessage(Warning,"Cannot rename the '%s' directory to '%s' [%!s]; upgrade failed.",proto,old);
       else
          if(mkdir(proto,buf.st_mode&07777))
            {
             if(errno==EEXIST)
                PrintMessage(Warning,"The '%s' directory already exists.",proto);
             else
                PrintMessage(Warning,"Cannot create '%s' directory [%!s]; upgrade failed.",proto);
            }
          else
            {
             if(rootp && chown(proto,buf.st_uid,buf.st_gid))
                PrintMessage(Important,"Cannot change owner of '%s' spool directory [%!s]; check it.",proto);

             if(chdir(old))
                PrintMessage(Warning,"Cannot change to the '%s' directory [%!s]; upgrade failed.",old);
             else
               {
                UpgradeProto(proto);

                chdir("..");

                if(rmdir(old))
                   PrintMessage(Warning,"Cannot delete the '%s' directory [%!s]; upgrade failed.",old);
               }
            }
      }
   }

 for(i=0;i<3;i++)
   {
    char *special,*old;

    if(i==0)
       special="outgoing",old="outgoing.old";
    else if(i==1)
       special="lasttime",old="lasttime.old";
    else
       special="monitor",old="monitor.old";

    if(stat(special,&buf))
       PrintMessage(Inform,"Cannot stat the '%s' directory [%!s] not upgraded.",special);
    else
      {
       printf("Converting '%s' directory\n",special);

       if(rename(special,old))
          PrintMessage(Warning,"Cannot rename the '%s' directory to '%s' [%!s]; upgrade failed.",special,old);
       else
          if(mkdir(special,buf.st_mode&07777))
            {
             if(errno==EEXIST)
                PrintMessage(Warning,"The '%s' directory already exists.",special);
             else
                PrintMessage(Warning,"Cannot create '%s' directory [%!s]; upgrade failed.",special);
            }
          else
            {
             if(rootp && chown(special,buf.st_uid,buf.st_gid))
                PrintMessage(Important,"Cannot change owner of '%s' spool directory [%!s]; check it.",special);

             if(chdir(old))
                PrintMessage(Warning,"Cannot change to the '%s' directory [%!s]; upgrade failed.",old);
             else
               {
                UpgradeSpecial(special);

                chdir("..");

                if(rmdir(old))
                   PrintMessage(Warning,"Cannot delete the '%s' directory [%!s]; upgrade failed.",old);
               }
            }
      }
   }

 return(0);
}


/*++++++++++++++++++++++++++++++++++++++
  Upgrade a complete protocol directory.

  char *proto The protocol of the spool directory we are in.
  ++++++++++++++++++++++++++++++++++++++*/

static void UpgradeProto(char *proto)
{
 DIR *dir;
 struct dirent* ent;
 struct stat buf;

 /* Open the spool directory. */

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open spool directory '%s.old' [%!s]; upgrade failed.",proto);return;}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read spool directory '%s.old' [%!s]; upgrade failed.",proto);closedir(dir);return;}
 ent=readdir(dir);  /* skip .. */

 /* Go through each entry. */

 while((ent=readdir(dir)))
   {
    char *newname=(char*)malloc(strlen(proto)+strlen(ent->d_name)+16);

    sprintf(newname,"../%s/%s",proto,ent->d_name);

    if(lstat(ent->d_name,&buf))
       PrintMessage(Warning,"Cannot stat file '%s.old/%s' [%!s] not upgraded.",proto,ent->d_name);
    else if(S_ISDIR(buf.st_mode))
      {
       if(mkdir(newname,buf.st_mode&07777))
         {
          if(errno==EEXIST)
             PrintMessage(Warning,"The '%s/%s' directory already exists.",proto,ent->d_name);
          else
             PrintMessage(Warning,"Cannot create '%s/%s' directory [%!s]; upgrade failed.",proto,ent->d_name);
         }
       else
         {
          if(rootp && chown(newname,buf.st_uid,buf.st_gid))
             PrintMessage(Important,"Cannot change owner of '%s/%s' spool directory [%!s]; check it.",proto,ent->d_name);

          if(chdir(ent->d_name))
             PrintMessage(Warning,"Cannot change to the '%s.old/%s' directory [%!s]; upgrade failed.",proto,ent->d_name);
          else
            {
             UpgradeHost(proto,ent->d_name);

             chdir("..");

             if(rmdir(ent->d_name))
                PrintMessage(Warning,"Cannot delete the '%s.old/%s' directory [%!s]; upgrade failed.",proto,ent->d_name);
            }
         }
      }
    else
       if(rename(ent->d_name,newname))
          PrintMessage(Warning,"Cannot move file '%s.old/%s' [%!s] not upgraded.",proto,ent->d_name);

    free(newname);
   }

 closedir(dir);
}


/*++++++++++++++++++++++++++++++++++++++
  Upgrade a complete host directory.

  char *proto The protocol of the spool directory we are in.

  char *host The hostname of the spool directory we are in.
  ++++++++++++++++++++++++++++++++++++++*/

static void UpgradeHost(char *proto,char *host)
{
 DIR *dir;
 struct dirent* ent;
 struct stat buf;
 char *newname=(char*)malloc(42+strlen(proto)+strlen(host)),*p;

 /* Open the spool directory. */

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open spool directory '%s.old/%s' [%!s]; upgrade failed.",proto,host);return;}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read spool directory '%s.old/%s' [%!s]; upgrade failed.",proto,host);closedir(dir);return;}
 ent=readdir(dir);  /* skip .. */

 /* Go through each entry. */

 sprintf(newname,"../../%s/%s/",proto,host);
 p=newname+strlen(newname);

 while((ent=readdir(dir)))
   {
    if(!lstat(ent->d_name,&buf) && *ent->d_name=='D' && S_ISREG(buf.st_mode))
      {
       char *url=FileNameToURL(ent->d_name);

       if(url)
         {
          URL *Url=SplitURL(url);
          char *name=URLToFileName(Url);
          char *oldname=ent->d_name;

          strcpy(p,name);

          *oldname=*p='U';
          rename(oldname,newname);
          *oldname=*p='D';
          rename(oldname,newname);

          free(name);
          FreeURL(Url);
          free(url);
         }
      }
   }

 closedir(dir);

 free(newname);
}


/*++++++++++++++++++++++++++++++++++++++
  Upgrade a complete special directory.

  char *special The name of the special directory.
  ++++++++++++++++++++++++++++++++++++++*/

static void UpgradeSpecial(char *special)
{
 DIR *dir;
 struct dirent* ent;
 struct stat buf;
 char *newname=(char*)malloc(32+strlen(special)),*p;

 /* Open the spool directory. */

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open spool directory '%s.old' [%!s]; upgrade failed.",special);return;}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read spool directory '%s.old' [%!s]; upgrade failed.",special);closedir(dir);return;}
 ent=readdir(dir);  /* skip .. */

 /* Go through each entry. */

 sprintf(newname,"../%s/",special);
 p=newname+strlen(newname);

 while((ent=readdir(dir)))
   {
    if(!lstat(ent->d_name,&buf) && (*ent->d_name=='O' || *ent->d_name=='D')  && S_ISREG(buf.st_mode))
      {
       char *url=FileNameToURL(ent->d_name);

       if(url)
         {
          URL *Url=SplitURL(url);
          char *name=URLToFileName(Url);
          char *oldname=ent->d_name;

          strcpy(p,name);

          *oldname=*p='U';
          rename(oldname,newname);
          *oldname=*p='D';
          rename(oldname,newname);
          *oldname=*p='O';
          rename(oldname,newname);

          free(name);
          FreeURL(Url);
          free(url);
         }
      }
   }

 closedir(dir);

 free(newname);
}
