/***************************************
  $Header: /home/amb/wwwoffle/RCS/purge.c 2.14 1999/03/17 19:50:28 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 2.4c.
  Purge old files from the cache.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1996,97,98,99 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <time.h>
#include <utime.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/vfs.h>
#include <time.h>
#include <dirent.h>
#include <unistd.h>

#include "wwwoffle.h"
#include "misc.h"
#include "proto.h"
#include "config.h"
#include "errors.h"


/* Local functions */
static int PurgeFiles(char *proto,char *host);

/*+ Set this to 0 for debugging so that nothing is deleted. +*/
#define DO_DELETE 1

/*+ The current time. +*/
static time_t now;

/*+ The number of blocks left of each age. +*/
static int *blocks_by_age;

/*+ The scaling factor for the ages in the second pass. +*/
static double age_scale;

/*+ The pass in the purge (1 or 2). +*/
int pass;


/*++++++++++++++++++++++++++++++++++++++
  Purge files from the cache that meet the age criteria.

  int fd the file descriptor of the wwwoffle client.
  ++++++++++++++++++++++++++++++++++++++*/

void PurgeCache(int fd)
{
 int i,p;
 int blocksize,diskfree;
 struct statfs sbuf;
 struct stat buf;

 now=time(NULL)+600;

 age_scale=-1;

 blocks_by_age=(int*)malloc((DefaultPurgeAge+2)*sizeof(int));

 for(i=0;i<=DefaultPurgeAge+1;i++)
    blocks_by_age[i]=0;

 if(stat(".",&buf))
   {
    PrintMessage(Warning,"Cannot determine the disk block size [%!s]");
    blocksize=1024;
   }
 else
    blocksize=buf.st_size/buf.st_blocks;

 if(statfs(".",&sbuf) || sbuf.f_bsize==-1)
   {
    PrintMessage(Warning,"Cannot determine the disk free space [%!s]");
    diskfree=0;
   }
 else
    diskfree=sbuf.f_bsize*sbuf.f_bavail/blocksize;

 for(pass=1;pass<=2;pass++)
   {
    int total_blocks=0,total_dirs=0;

    write_string(fd,"\n");

    if(PurgeCacheSize)
       if(pass==1)
          write_string(fd,"Pass 1: Checking dates and sizes of files:\n");
       else
          write_string(fd,"Pass 2: Purging files down to specified size:\n");
    else
       write_string(fd,"Checking dates of files:\n");

    if(pass==1)
      {
       if(PurgeUseMTime)
          write_string(fd,"  (Using modification time.)\n");
       else
          write_string(fd,"  (Using last access time.)\n");

       if(PurgeUseURL)
          write_string(fd,"  (Using the full URL.)\n");
       else
          write_string(fd,"  (Using the hostname and protocol only.)\n");
      }

    write_string(fd,"\n");

    for(p=0;p<NProtocols;p++)
      {
       DIR *dir;
       struct dirent* ent;
       struct stat buf;
       char *proto=Protocols[p].name;

       /* Open the spool directory. */

       if(stat(proto,&buf))
         {PrintMessage(Inform,"Cannot stat directory '%s' [%!s]; not purged",proto);continue;}

       total_blocks+=buf.st_blocks;
       blocks_by_age[DefaultPurgeAge+1]+=buf.st_blocks;

       if(chdir(proto))
         {PrintMessage(Warning,"Cannot change to directory '%s' [%!s]; not purged.",proto);continue;}

       dir=opendir(".");
       if(!dir)
         {PrintMessage(Warning,"Cannot open directory '%s' [%!s]; not purged.",proto);chdir("..");continue;}

       ent=readdir(dir);  /* skip .  */
       if(!ent)
         {PrintMessage(Warning,"Cannot read directory '%s' [%!s]; not purged.",proto);closedir(dir);chdir("..");continue;}
       ent=readdir(dir);  /* skip .. */

       /* Search through all of the sub directories. */

       while((ent=readdir(dir)))
         {
          struct stat buf;

          if(lstat(ent->d_name,&buf))
             PrintMessage(Inform,"Cannot stat file '%s/%s' [%!s]; race condition?",proto,ent->d_name);
          else if(S_ISDIR(buf.st_mode))
            {
             int blocks=PurgeFiles(proto,ent->d_name);

             if(blocks==-1)
               {
#if DO_DELETE
                if(rmdir(ent->d_name))
                   PrintMessage(Warning,"Cannot delete what should be an empty directory '%s/%s' [%!s].",proto,ent->d_name);
#else
                PrintMessage(Debug,"rmdir(%s/%s).",proto,ent->d_name);
#endif
               }
             else
               {
                struct utimbuf utbuf;

                utbuf.actime=buf.st_atime;
                utbuf.modtime=buf.st_mtime;
                utime(ent->d_name,&utbuf);

                total_blocks+=buf.st_blocks;
                blocks_by_age[DefaultPurgeAge+1]+=buf.st_blocks;
                total_blocks+=blocks;
                total_dirs++;
               }

             if(PurgeUseURL)
               {
                if(blocks==-1)
                   write_formatted(fd,"Purged %6s://%-32s ; (empty) - deleted\n",proto,ent->d_name);
                else
                   write_formatted(fd,"Purged %6s://%-32s ; %5ld kB\n",proto,ent->d_name,(blocks+buf.st_blocks)*blocksize/1024);
               }
             else
               {
                int age=WhatPurgeAge(proto,ent->d_name,"/");

                if(pass==2 && age>0)
                   age=(int)(age*age_scale);

                if(age<0)
                   write_formatted(fd,"Not Purged       %6s://%-32s ; %5ld kB\n",proto,ent->d_name,(blocks+buf.st_blocks)*blocksize/1024);
                else if(blocks==-1)
                   write_formatted(fd,"Purged (%2d days) %6s://%-32s ; (empty) - deleted\n",age,proto,ent->d_name);
                else
                   write_formatted(fd,"Purged (%2d days) %6s://%-32s ; %5ld kB\n",age,proto,ent->d_name,(blocks+buf.st_blocks)*blocksize/1024);
               }
            }
         }

       closedir(dir);
       chdir("..");
      }

    write_string(fd,"\n");
    write_formatted(fd,"Total of %d directories ; %d kB\n",total_dirs,total_blocks*blocksize/1024);

    if(pass==1)
      {
       int age_for_size=-1,age_for_free=-1;
       int age_blocks_used=blocks_by_age[DefaultPurgeAge+1];
       int age_blocks_free=diskfree+total_blocks-blocks_by_age[DefaultPurgeAge+1];

       write_string(fd,"\n");
       write_string(fd,"Age Profile of cached pages:\n");
       write_string(fd,"  (All ages scaled to the range 0 -> default age.)\n");
       write_string(fd,"\n");

       write_formatted(fd,"Total not purged   ; %5d kB (%6d kB free)\n",
                       age_blocks_used*blocksize/1024,age_blocks_free*blocksize/1024);
       write_string(fd,"\n");

       for(i=0;i<=DefaultPurgeAge;i++)
         {
          age_blocks_used+=blocks_by_age[i];
          age_blocks_free-=blocks_by_age[i];

          if(PurgeCacheSize && age_for_size<0 && (age_blocks_used*blocksize/1024)>(1024*PurgeCacheSize))
            {
             age_for_size=i;

             write_formatted(fd,"Cutoff Age is %2d days for %3d MB cache size\n",age_for_size,PurgeCacheSize);
            }

          if(PurgeDiskFree && age_for_free<0 && (age_blocks_free*blocksize/1024)<(1024*PurgeDiskFree))
            {
             age_for_free=i;

             write_formatted(fd,"Cutoff Age is %2d days for %3d MB disk free\n",age_for_free,PurgeDiskFree);
            }

          if(i==DefaultPurgeAge)
             write_formatted(fd,"Total all ages     ; %5d kB (%6d kB free)\n",
                             (age_blocks_used-blocks_by_age[DefaultPurgeAge+1])*blocksize/1024,age_blocks_free*blocksize/1024);
          else
             write_formatted(fd,"Newer than %2d day%c ; %5d kB (%6d kB free)\n",i+1,i?'s':' ',
                             (age_blocks_used-blocks_by_age[DefaultPurgeAge+1])*blocksize/1024,age_blocks_free*blocksize/1024);
         }

       if(DefaultPurgeAge)
         {
          if(age_for_size!=-1 && (age_for_size<=age_for_free || age_for_free==-1))
             age_scale=(double)age_for_size/(double)DefaultPurgeAge;
          else if(age_for_free!=-1 && (age_for_free<age_for_size || age_for_size==-1))
             age_scale=(double)age_for_free/(double)DefaultPurgeAge;
         }
       else if(age_for_size!=-1 || age_for_free!=-1)
          age_scale=0;
      }

    if(age_scale==-1)
       break;
   }

 write_string(fd,"\n");

 free(blocks_by_age);

 /* Purge the tmp.* files in outgoing. */

 if(chdir("outgoing"))
    PrintMessage(Warning,"Cannot change to directory 'outgoing' [%!s]; not purged.");
 else
   {
    DIR *dir;
    struct dirent* ent;

    dir=opendir(".");
    if(!dir)
      PrintMessage(Warning,"Cannot open directory 'outgoing' [%!s]; not purged.");
    else
      {
       ent=readdir(dir);  /* skip .  */
       if(!ent)
          PrintMessage(Warning,"Cannot read directory 'outgoing' [%!s]; not purged.");
       else
         {
          ent=readdir(dir);  /* skip .. */

          while((ent=readdir(dir)))
             if(!strncmp(ent->d_name,"tmp.",4))
               {
                struct stat buf;

                if(!stat(ent->d_name,&buf) && buf.st_mtime<(now-60))
                  {
#if DO_DELETE
                   if(unlink(ent->d_name))
                      PrintMessage(Warning,"Cannot unlink file 'outgoing/%s' [%!s].",ent->d_name);
#else
                   PrintMessage(Debug,"unlink(outgoing/%s).",ent->d_name);
#endif
                  }
               }
         }

       closedir(dir);
      }
   }

 chdir("..");
}


/*++++++++++++++++++++++++++++++++++++++
  Delete the file in the current directory that are older than the specified age.

  int PurgeFiles Returns the number of blocks in files that are left.

  char *proto The name of the protocol directory to purge.

  char *host The name of the host directory to purge.
  ++++++++++++++++++++++++++++++++++++++*/

static int PurgeFiles(char *proto,char *host)
{
 int blocks_left=-1;
 DIR *dir;
 struct dirent* ent;
 int def_age=WhatPurgeAge(proto,host,"/");

 /* Open the spool subdirectory. */

 if(chdir(host))
   {PrintMessage(Warning,"Cannot change to directory '%s/%s' [%!s]; not purged.",proto,host);return(1);}

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open directory '%s/%s' [%!s]; not purged.",proto,host);chdir("..");return(1);}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read directory '%s/%s' [%!s]; not purged.",proto,host);closedir(dir);chdir("..");return(1);}
 ent=readdir(dir);  /* skip .. */

 /* Check all of the files for age, and delete as needed. */

 while((ent=readdir(dir)))
   {
    struct stat buf,buf2;

    if(stat(ent->d_name,&buf))
       PrintMessage(Inform,"Cannot stat file '%s/%s/%s' [%!s]; race condition?",proto,host,ent->d_name);
    else
      {
       int age=DefaultPurgeAge;
       time_t t=now;

       if(buf.st_mtime>now || buf.st_atime>now)
         {
          PrintMessage(Inform,"Cached file '%s/%s/%s' has a future timestamp; changing timestamp.",proto,host,ent->d_name);
#if DO_DELETE
          utime(ent->d_name,NULL);
#else
          PrintMessage(Debug,"utime(%s/%s/%s).",proto,host,ent->d_name);
#endif
         }

       if(*ent->d_name=='U' || *ent->d_name=='D')
         {
          int s;

          *ent->d_name^='U'^'D';
          s=stat(ent->d_name,&buf2);
          *ent->d_name^='U'^'D';

          if(s)
            {
             PrintMessage(Inform,"Cached file '%s/%s/%s' is not complete (U* and D* files); deleting it.",proto,host,ent->d_name);
             age=0;
            }
          else if(*ent->d_name=='U')
             continue;
          else if(PurgeUseURL)
            {
             char *url=FileNameToURL(ent->d_name);
             if(url)
               {
                URL *Url=SplitURL(url);
                age=WhatPurgeAge(proto,host,Url->path);
                FreeURL(Url);
                free(url);
               }
             else
                age=0;
            }
          else
             age=def_age;

          if(PurgeUseMTime)
             t=buf.st_mtime;
          else
             t=buf.st_atime;
         }
       else
         {
          PrintMessage(Inform,"Cached file '%s/%s/%s' is not valid (U* or D* file); deleting it.",proto,host,ent->d_name);
          age=0;
         }

       if(pass==2 && age>0)
          age=(int)(age*age_scale);

       if(age==-1 || t>(now-age*(24*3600)))
         {
          int size=buf.st_blocks+buf2.st_blocks;

          if(blocks_left==-1)
             blocks_left=0;
          blocks_left+=size;

          if(age>0)
            {
             int days=(now-t)/(24*3600);

             days=days*DefaultPurgeAge/age; /* scale the age to fit into 0 -> DefaultPurgeAge */

             if(days>DefaultPurgeAge)
                days=DefaultPurgeAge;
             blocks_by_age[days]+=size;
            }
          else
             blocks_by_age[DefaultPurgeAge+1]+=size;
         }
       else
         {
#if DO_DELETE
          if(unlink(ent->d_name))
             PrintMessage(Warning,"Cannot unlink file '%s/%s/%s' [%!s].",proto,host,ent->d_name);
#else
          PrintMessage(Debug,"unlink(%s/%s/%s).",proto,host,ent->d_name);
#endif

          if(*ent->d_name=='U' || *ent->d_name=='D')
            {
             *ent->d_name^='U'^'D';

#if DO_DELETE
             if(unlink(ent->d_name))
                PrintMessage(Warning,"Cannot unlink file(2) '%s/%s/%s' [%!s].",proto,host,ent->d_name);
#else
             PrintMessage(Debug,"unlink(%s/%s/%s).",proto,host,ent->d_name);
#endif
            }
         }
      }
   }

 closedir(dir);
 chdir("..");

 return(blocks_left);
}
