/***************************************
  $Header: /home/amb/wwwoffle/RCS/wwwoffles.c 2.109 1998/12/21 20:51:13 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 2.4a.
  A server to fetch the required pages.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1996,97,98 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <signal.h>
#include <unistd.h>
#include <fcntl.h>

#include "wwwoffle.h"
#include "document.h"
#include "misc.h"
#include "proto.h"
#include "config.h"
#include "sockets.h"
#include "errors.h"


static void ssl_tunnel(int client,URL *Url,char *request_head);
static int request_new(URL *Url,char *referer);
static void uninstall_sighandlers(void);


/*+ The mode of operation of the server. +*/
typedef enum _Mode
{
 None,                          /*+ Undecided. +*/
 Real,                          /*+ From server host to cache and client. +*/
 RealNoCache,                   /*+ From server host to client. +*/
 RealRefresh,                   /*+ Refresh the page, forced from index. +*/
 RealPassword,                  /*+ From server host to cache, then again with a password. +*/
 SpoolOrReal,                   /*+ Spool if already cached else Real. +*/
 Fetch,                         /*+ From server host to cache. +*/
 Spool,                         /*+ From cache to client. +*/
 SpoolGet,                      /*+ Not in cache so record request in outgoing. +*/
 SpoolWillGet,                  /*+ Not in cache but is in outgoing. +*/
 SpoolRefresh,                  /*+ Refresh the page, forced from refresh page. +*/
 SpoolPragma                    /*+ Refresh the page, forced from browser by 'Pragma: no-cache'. +*/
}
Mode;


/*++++++++++++++++++++++++++++++++++++++
  The main server program.

  int wwwoffles Returns the exit status.

  int online Whether the demon is online or not.

  int browser Set to true if there is a browser.

  int client The file descriptor of the client.
  ++++++++++++++++++++++++++++++++++++++*/

int wwwoffles(int online,int browser,int client)
{
 int outgoing=-1,spool=-1;
 int fetch_again=0;
 char *aliasProto=NULL,*aliasHost=NULL,*aliasPath=NULL;
 char *request_head,*request_body;
 char *reply_head,*reply_body;
 int reply_status=-1;
 char *url;
 URL *Url,*Urlpw=NULL;
 Mode mode=None;
 int lasttime_exists=0,spool_exists=0,spool_exists_pw=0;
 int offline_request=1;
 char *proxy_auth,*proxy_user;

 /* Initialise things. */

 uninstall_sighandlers();

 InitErrorHandler("wwwoffles",-1,-1); /* change name nothing else */

 if(online==1 && browser)
    mode=Real;
 else if(online==1 && !browser)
    mode=Fetch;
 else if(online==-1 && browser)
    mode=SpoolOrReal;
 else if(!online && browser)
    mode=Spool;
 else
   {
    PrintMessage(Fatal,"Started in a mode that is not allowed (online=%d, browser=%d).",online,browser);
    if(client!=-1)
       HTMLMessage(client,500,"WWWOFFLE Server Error",NULL,"ServerError",
                   "error","Started server in illegal mode.",
                   NULL);
   }

 /* Set up the client file. */

 if(client<0 && mode!=Fetch)
   {PrintMessage(Warning,"Cannot use client file descriptor %d.",client);exit(1);}

 /* Set up the outgoing file. */

 if(mode==Fetch)
   {
    outgoing=OpenOutgoingSpoolFile(1);
    init_buffer(outgoing);

    if(outgoing==-1)
      {PrintMessage(Inform,"No more outgoing requests.");exit(3);}
   }

 /* Get the URL from the request. */

 if(mode==Real || mode==Spool || mode==SpoolOrReal)
    url=ParseRequest(client,&request_head,&request_body);
 else /* mode==Fetch */
    url=ParseRequest(outgoing,&request_head,&request_body);

 PrintMessage(ExtraDebug,"Incoming Request Head (from browser)\n%s",request_head?request_head:"(empty)");

 if(!url)
   {
    PrintMessage(Warning,"Could not parse HTTP request (%s).",request_head?"Parse error":"Empty request");
    if(mode!=Fetch)
       HTMLMessage(client,500,"WWWOFFLE Server Error",NULL,"ServerError",
                   "error",request_head?"Cannot parse the HTTP request":"The HTTP request was empty",
                   NULL);
    exit(1);
   }

 Url=SplitURL(url);

 /* Authenticate the user with the proxy */

 if(!(proxy_user=IsAllowedConnectUser(proxy_auth=GetHTTPHeader(request_head,"Proxy-Authorization:"))))
   {
    PrintMessage(Inform,"HTTP Proxy connection rejected from unauthenticated user."); /* Used in audit-usage.pl */
    HTMLMessageHead(client,407,"WWWOFFLE Proxy Authentication Required",
                    "Proxy-Authenticate","Basic realm=\"wwwoffle-proxy\"",
                    NULL);
    HTMLMessageBody(client,"ProxyAuthFail",
                    NULL);
    exit(1);
   }
 else if(proxy_auth)
    PrintMessage(Inform,"HTTP Proxy connection from user '%s'.",proxy_user); /* Used in audit-usage.pl */

 /* Check the HTTP method that is requested */

 if(!strncasecmp(request_head,"CONNECT",3))
   {
    PrintMessage(Inform,"SSL='%s'.",Url->host); /* Used in audit-usage.pl */

    if(mode==Real || mode==SpoolOrReal)
      {
       if(IsSSLAllowedPort(Url->host))
         {
          ssl_tunnel(client,Url,request_head);
          exit(0);
         }
       else
         {
          PrintMessage(Warning,"A SSL proxy connection for %s was received but is not allowed.",Url->host);
          if(client!=-1)
             HTMLMessage(client,500,"WWWOFFLE Server Error",NULL,"ServerError",
                         "error","SSL proxy connection to specified port is not allowed.",
                         NULL);
         }
      }
    else
      {
       PrintMessage(Warning,"A SSL proxy connection for %s was received but wwwoffles is in wrong mode.",Url->host);
       if(client!=-1)
          HTMLMessage(client,500,"WWWOFFLE Server Error",NULL,"ServerError",
                      "error","SSL proxy connection while offline is not allowed.",
                      NULL);
      }
    exit(1);
   }
 else if(strncasecmp(request_head,"POST",4) && strncasecmp(request_head,"GET",3) && strncasecmp(request_head,"HEAD",4))
   {
    char *method=request_head,*m=method;

    while(*m!=' ')
       m++;
    *m=0;

    PrintMessage(Warning,"The requested HTTP method '%s' is not supported.",method);
    if(client!=-1)
       HTMLMessage(client,501,"WWWOFFLE Method Unsupported",NULL,"MethodUnsupported",
                   "method",method,
                   NULL);
    exit(1);
   }

 PrintMessage(Inform,"URL='%s'%s.",Url->name,Url->user?" (With username/password)":""); /* Used in audit-usage.pl */
 PrintMessage(Debug,"proto='%s'; host='%s'; path='%s'; args='%s'; user:pass='%s:%s'.",
              Url->proto,Url->host,Url->path,Url->args,Url->user,Url->pass);

 /* Check for an alias. */

 if(IsAliased(Url->proto,Url->host,Url->path,&aliasProto,&aliasHost,&aliasPath))
   {
    URL *newUrl;
    char *newurl=(char*)malloc(strlen(aliasProto)+strlen(aliasHost)+strlen(aliasPath)-
                               strlen(Url->path)+strlen(Url->pathp)+8);

    sprintf(newurl,"%s://%s%s%s",aliasProto,aliasHost,aliasPath,Url->pathp+strlen(Url->path));

    newUrl=SplitURL(newurl);

    if(Url->user)
       AddURLPassword(newUrl,Url->user,Url->pass);

    url=newurl;
    FreeURL(Url);
    Url=newUrl;

    PrintMessage(Inform,"Aliased URL='%s'%s.",newUrl->name,newUrl->user?" (With username/password)":"");
    PrintMessage(Debug,"Aliased proto='%s'; host='%s'; path='%s'; args='%s'; user:pass='%s:%s'.",
                 newUrl->proto,newUrl->host,newUrl->path,newUrl->args,newUrl->user,newUrl->pass);
   }

 /* Initial sanity checks. */

 if(IsNotGot(Url->proto,Url->host,Url->path))
   {
    if(!DontGetReplacementURL)
      {
       PrintMessage(Inform,"The server '%s://%s' and/or path '%s' is on the list not to get.",Url->proto,Url->host,Url->path);
       if(mode==Real || mode==Spool || mode==SpoolOrReal)
          HTMLMessage(client,404,"WWWOFFLE Host Not Got",NULL,"HostNotGot",
                      "url",Url->name,
                      NULL);
       exit(0);
      }
    else
      {
       char *newurl=(char*)malloc(strlen(DontGetReplacementURL)+1);
       URL *newUrl=SplitURL(DontGetReplacementURL);

       strcpy(newurl,DontGetReplacementURL);

       url=newurl;
       FreeURL(Url);
       Url=newUrl;

       PrintMessage(Inform,"Replaced URL='%s'%s.",newUrl->name,newUrl->user?" (With username/password)":"");
       PrintMessage(Debug,"Replaced proto='%s'; host='%s'; path='%s'; args='%s'; user:pass='%s:%s'.",
                    newUrl->proto,newUrl->host,newUrl->path,newUrl->args,newUrl->user,newUrl->pass);
      }    
   }

 if(!Url->Protocol)
   {
    PrintMessage(Inform,"The protocol '%s' is not available.",Url->proto);
    if(mode==Real || mode==Spool || mode==SpoolOrReal)
       HTMLMessage(client,404,"WWWOFFLE Illegal Protocol",NULL,"IllegalProtocol",
                   "url",Url->name,
                   "protocol",Url->proto,
                   NULL);
    exit(1);
   }

 /* Change the mode based on the URL as required. */

 /* mode = Spool, Real, SpoolOrReal or Fetch */

 /* A local URL. */

 if(Url->local)
   {
    if(!strncmp("/refresh",Url->path,8))
      {
       int recurse=0;

       char *newurl=RefreshPage(client,Url,request_body,&recurse);

       if(!newurl)
          exit(0);
       else
         {
          URL *newUrl=SplitURL(newurl);

          if(Url->user)
             AddURLPassword(newUrl,Url->user,Url->pass);

          if(newUrl->args && *newUrl->args=='!')
            {
             PrintMessage(Inform,"It is not possible to refresh a URL that was posted.");
             if(mode!=Fetch)
                HTMLMessage(client,404,"WWWOFFLE Cant Refresh Posted",NULL,"CantRefreshPosted",
                            "url",newUrl->name,
                            NULL);
             exit(0);
            }

          if(recurse)
            {
             fetch_again=1;

             if(mode!=Fetch)
               {
                mode=SpoolGet;

                if(!Url->args || *Url->args!='!')
                  {
                   free(newurl);
                   FreeURL(newUrl);
                   newurl=(char*)malloc(strlen(url)+1);
                   strcpy(newurl,url);
                   newUrl=SplitURL(newurl);
                   if(Url->user)
                      AddURLPassword(newUrl,Url->user,Url->pass);
                  }

                free(request_head);
                request_head=RequestURL(newUrl,NULL);
                if(request_body)
                   free(request_body);
                request_body=NULL;
               }
            }
          else
            {
             if(mode==Spool)
                mode=SpoolRefresh;
             else
               {
                DeleteWebpageSpoolFile(newUrl,0);
                mode=RealRefresh;
               }
            }

          url=newurl;
          FreeURL(Url);
          Url=newUrl;

          PrintMessage(Inform,"New URL='%s'%s.",newUrl->name,newUrl->user?" (With username/password)":"");
          PrintMessage(Debug,"New proto='%s'; host='%s'; path='%s'; args='%s'; user:pass='%s:%s'.",
                       newUrl->proto,newUrl->host,newUrl->path,newUrl->args,newUrl->user,newUrl->pass);
         }
      }
    else if(mode==Fetch)
      {
       PrintMessage(Inform,"The request to fetch a page from the local host is ignored.");
       exit(0);
      }
    else if(!strchr(Url->path+1,'/') && !Url->args)
      {
       LocalPage(client,Url->path,request_head);
       exit(0);
      }
    else if(!strncmp("/index/",Url->path,7))
      {
       IndexPage(client,Url);
       exit(0);
      }
    else if(!strncmp("/control/",Url->path,9))
      {
       ControlPage(client,Url,request_head,request_body);
       exit(0);
      }
    else if(!strncmp("/monitor",Url->path,8))
      {
       MonitorPage(client,Url,request_body);
       exit(0);
      }
    else if(!strncmp("/local/",Url->path,7) && !Url->args)
      {
       LocalPage(client,Url->path,request_head);
       exit(0);
      }
    else if(!strncmp("/htdig/",Url->path,7))
      {
       HTDigPage(client,Url,request_head);
       exit(0);
      }
    else
      {
       int i;

       for(i=0;i<NProtocols;i++)
          if(!strncmp(Protocols[i].name,Url->pathp+1,strlen(Protocols[i].name)) &&
             Url->pathp[strlen(Protocols[i].name)+1]=='/')
            {
             url=(char*)malloc(strlen(Url->pathp)+4);
             Url->pathp[strlen(Protocols[i].name)+1]=0;
             sprintf(url,"%s://%s",Url->pathp+1,&Url->pathp[strlen(Protocols[i].name)+2]);
             Url->pathp[strlen(Protocols[i].name)+1]='/';
             break;
            }

       if(i==NProtocols)
         {
          PrintMessage(Inform,"The requested URL '%s' does not exist on the local server.",Url->pathp);
          HTMLMessage(client,404,"WWWOFFLE Page Not Found",NULL,"PageNotFound",
                      "url",Url->name,
                      NULL);
          exit(0);
         }

       FreeURL(Url);
       Url=SplitURL(url);
      }

    /* Repeat initial sanity checks. */

    if(IsNotGot(Url->proto,Url->host,Url->path))
      {
       PrintMessage(Inform,"The server '%s://%s' and/or path '%s' is on the list not to get.",Url->proto,Url->host,Url->path);
       if(mode==Real || mode==Spool || mode==SpoolOrReal)
          HTMLMessage(client,404,"WWWOFFLE Host Not Got",NULL,"HostNotGot",
                      "url",Url->name,
                      NULL);
       exit(0);
      }

    if(!Url->Protocol)
      {
       PrintMessage(Inform,"The protocol '%s' is not available.",Url->proto);
       if(mode==Real || mode==Spool || mode==SpoolOrReal)
          HTMLMessage(client,404,"WWWOFFLE Illegal Protocol",NULL,"IllegalProtocol",
                      "url",Url->name,
                      "protocol",Url->proto,
                      NULL);
       exit(1);
      }
   }
 else if(mode==Fetch)
    ParseRecurseOptions(NULL);

 /* mode = Spool, SpoolGet, SpoolRefresh, Real, RealRefresh, SpoolOrReal or Fetch */

 /* Check for a username / password */

 if(Url->user)
   {
    URL *new=SplitURL(Url->name);

    Urlpw=Url;
    Url=new;

    if(!Urlpw->pass)
      {
       if(mode==Fetch)
         {
          FreeURL(Urlpw);
          Urlpw=NULL;
         }
       else
         {
          HTMLMessage(client,403,"WWWOFFLE Username Needs Password",NULL,"UserNeedsPass",
                      "url",Urlpw->name,
                      "user",Urlpw->user,
                      NULL);
          exit(0);
         }
      }
   }

 /* Check for an existing cached version */

 spool_exists=ExistsWebpageSpoolFile(Url);

 if(Urlpw)
   {
    spool_exists_pw=ExistsWebpageSpoolFile(Urlpw);

    if((mode==Spool || mode==SpoolGet || mode==SpoolOrReal) && spool_exists)
      {
       if(SpooledPageStatus(Url)==401)
         {FreeURL(Url);Url=Urlpw;Urlpw=NULL;
         spool_exists=spool_exists_pw;spool_exists_pw=0;}
       else
         {FreeURL(Urlpw);Urlpw=NULL;
         spool_exists_pw=0;}
      }
    else if(mode==Spool || mode==SpoolGet || mode==SpoolOrReal) /* && ! spool_exists */
      {
       if(spool_exists_pw)
          request_new(Url,NULL);
       FreeURL(Url);Url=Urlpw;Urlpw=NULL;
       spool_exists=spool_exists_pw;spool_exists_pw=0;
      }
    else if(mode==Fetch && spool_exists)
      {FreeURL(Url);Url=Urlpw;Urlpw=NULL;
      spool_exists=spool_exists_pw;spool_exists_pw=0;}
    else if(mode==Real && spool_exists)
      {
       if(SpooledPageStatus(Url)==401)
         {FreeURL(Url);Url=Urlpw;Urlpw=NULL;
         spool_exists=spool_exists_pw;spool_exists_pw=0;}
       else
         {FreeURL(Urlpw);Urlpw=NULL;
         spool_exists_pw=0;}
      }
   }

realpassword:

 /* Check if it needs to be cached. */

 if(IsLocalNetHost(Url->host))
   {
    if(mode==Real || mode==Spool || mode==SpoolOrReal)
       mode=RealNoCache;
    else if(mode==SpoolGet)
       ;
    else
      {
       PrintMessage(Inform,"The request to fetch a page from the local network host '%s' is ignored.",Url->host);
       if(client!=-1)
          write_formatted(client,"Cannot fetch %s [On local network]\n",Url->name);
       exit(0);
      }
   }
 else if((mode==Real || mode==SpoolOrReal) && IsNotCached(Url->proto,Url->host,Url->path))
   {
    mode=RealNoCache;
   }

 if(mode==RealNoCache && Urlpw)
   {FreeURL(Url);Url=Urlpw;Urlpw=NULL;
   spool_exists=spool_exists_pw;spool_exists_pw=0;}

 /* mode = Spool, SpoolGet, SpoolRefresh, Real, RealRefresh, RealNoCache, SpoolOrReal or Fetch */

 /* Check if it was posted. */

 if(!strncasecmp(request_head,"POST",4))
   {
    if(mode==Spool)
       mode=SpoolGet;
   }
 else if(Url->args && *Url->args=='!')
   {
    if(mode==Real || mode==SpoolOrReal)
       mode=Spool;
    else if(mode==Fetch)
      {
       PrintMessage(Inform,"It is not possible to fetch a URL that was posted.");
       if(client!=-1)
          write_formatted(client,"Cannot fetch %s [Reply from a POST]\n",Url->name);
       exit(0);
      }
   }

 /* Check if it is a conditional request. */

 if(mode!=RealNoCache && GetHTTPHeader(request_head,"If-Modified-Since:"))
   {
    char *bol=GetHTTPHeader(request_head,"If-Modified-Since:"),*eol=strchr(bol,'\n');
    *eol++=0;

    if(mode==Spool || mode==SpoolOrReal)
      {
       spool=OpenWebpageSpoolFile(1,Url);

       if(spool!=-1)
         {
          struct stat buf;
          long since;

          buf.st_mtime=time(NULL)+1;
          fstat(spool,&buf);
          since=DateToTimeT(bol+19);

          close(spool);

          if(since>buf.st_mtime)
            {
             HTMLMessageHead(client,304,"WWWOFFLE Not Modified",
                             NULL);
             exit(0);
            }
         }
      }

    while(*eol)
       *bol++=*eol++;
    *bol=0;
   }

 /* Check if a refresh is needed based on pragma, request changes, autodial. */

 if(mode==Real && ExistsLockWebpageSpoolFile(Url))
   {
    mode=Spool;
   }
 else if(mode==Fetch && ExistsLockWebpageSpoolFile(Url))
   {
    PrintMessage(Debug,"Already fetching URL.");
    if(client!=-1)
       write_formatted(client,"Cannot fetch %s [Already fetching]\n",Url->name);
    exit(0);
   }
 else if(PragmaNoCache && GetHTTPHeader(request_head,"Pragma: no-cache"))
   {
    if(mode==Spool || mode==SpoolGet)
      {
       if(spool_exists)
          mode=SpoolPragma;
       else
          if(ExistsOutgoingSpoolFile(Url))
             mode=SpoolWillGet;
          else
             mode=SpoolGet;
      }
    else if(mode==SpoolOrReal)
       mode=Real;

    /* (mode==Fetch || mode==Real) are left unchanged, not modified as below. */
   }
 else if(mode==Spool && !spool_exists)
   {
    if(ExistsOutgoingSpoolFile(Url))
       mode=SpoolWillGet;
    else
       mode=SpoolGet;
   }
 else if(mode==Fetch && spool_exists)
   {
    spool=OpenWebpageSpoolFile(1,Url);
    init_buffer(spool);

    if((RequestChanged>=0 || RequestChangedOnce) && RequestChanges(spool,&request_head)==1)
       TouchWebpageSpoolFile(Url);
    else if(fetch_again)
      {
       lseek(spool,0,SEEK_SET);
       init_buffer(spool);

       if(ParseDocument(spool,Url))
          RecurseFetch(Url,0);

       exit(4);
      }
    else
       exit(0);

    close(spool);
   }
 else if(mode==Real && spool_exists)
   {
    spool=OpenWebpageSpoolFile(1,Url);
    init_buffer(spool);

    if((RequestChanged>=0 || RequestChangedOnce) && RequestChanges(spool,&request_head)==1)
       TouchWebpageSpoolFile(Url);
    else
       mode=Spool;

    close(spool);
   }
 else if(mode==SpoolOrReal)
   {
    if(spool_exists)
       mode=Spool;
    else
       mode=Real;
   }

 /* Don't let htdig request any URLs. */

 if((mode==SpoolGet || mode==Real) && GetHTTPHeader(request_head,"User-Agent: htdig/"))
   {
    PrintMessage(Inform,"URL unavailable to be searched.");
    HTMLMessageHead(client,404,"WWWOFFLE Not Searched",
                    NULL);
    exit(0);
   }

 /* mode = Spool, SpoolGet, SpoolWillGet, SpoolPragma, SpoolRefresh, Real, RealRefresh, RealNoCache or Fetch */

 /* Set up the file descriptor for the spool file. */

 if(mode==Real || mode==Fetch)
   {
    if(spool_exists)
       CreateBackupWebpageSpoolFile(Url);

    spool=OpenWebpageSpoolFile(0,Url);
    CreateLockWebpageSpoolFile(Url);

    if(spool==-1)
      {
       DeleteLockWebpageSpoolFile(Url);
       PrintMessage(Warning,"Cannot open the spooled web page to write.");
       if(mode==Real)
          HTMLMessage(client,500,"WWWOFFLE Server Error",NULL,"ServerError",
                      "error","Cannot open the spooled web page to write.",
                      NULL);
       exit(1);
      }

    lasttime_exists=CreateLastTimeSpoolFile(Url);
   }
 else if(mode==Spool || mode==SpoolPragma)
   {
    spool=OpenWebpageSpoolFile(1,Url);
    init_buffer(spool);

    if(spool==-1)
      {
       PrintMessage(Warning,"Cannot open the spooled web page to read.");
       HTMLMessage(client,500,"WWWOFFLE Server Error",NULL,"ServerError",
                   "error","Cannot open the spooled web page to read.",
                   NULL);
       exit(1);
      }
   }

 /* Set up the outgoing file. */

 if((offline_request=!IsNotRequestedOffline(Url->proto,Url->host,Url->path)) &&
    ((mode==SpoolGet && (!ConfirmRequests || Url->local || (Url->args && *Url->args=='!'))) ||
     mode==SpoolRefresh || mode==SpoolPragma))
   {
    outgoing=OpenOutgoingSpoolFile(0);

    if(outgoing==-1)
      {
       PrintMessage(Warning,"Cannot open the outgoing request to write.");
       HTMLMessage(client,500,"WWWOFFLE Server Error",NULL,"ServerError",
                   "error","Cannot open the outgoing request to write.",
                   NULL);
       exit(1);
      }
   }

 /* Open the connection to the server host. */

 if(mode==Real || mode==RealNoCache || mode==Fetch)
   {
    char *err=(Url->Protocol->open)(Url);

    if(err && ConnectRetry)
      {
       PrintMessage(Inform,"Waiting to try connection again.");
       sleep(10);
       err=(Url->Protocol->open)(Url);
      }

    if(err)
      {
       if(mode==Fetch || mode==Real)
         {
          lseek(spool,0,SEEK_SET);
          ftruncate(spool,0);
          HTMLMessage(spool,503,"WWWOFFLE Remote Host Error",NULL,"RemoteHostError",
                      "url",Url->name,
                      "reason",err,
                      "cache","yes",
                      "backup",spool_exists?"yes":NULL,
                      NULL);
          DeleteLockWebpageSpoolFile(Url);
         }
       if(mode==Real || mode==RealNoCache)
          HTMLMessage(client,503,"WWWOFFLE Remote Host Error",NULL,"RemoteHostError",
                      "url",Url->name,
                      "reason",err,
                      "cache","",
                      "backup",spool_exists?"yes":NULL,
                      NULL);
       exit(1);
      }
   }

 /* Modify the header (Censor / Cannonicalise URL / POST / HTTP-1.1 etc). */

 if(mode==Real || mode==RealNoCache || mode==Fetch || mode==SpoolGet || mode==SpoolRefresh || mode==SpoolPragma)
    request_head=ModifyRequest(Url,request_head);

 /* Write request to remote server or outgoing file. */

 if(mode==Fetch && client!=-1)
    write_formatted(client,"Fetching %s ...\n",Url->name);

 if(mode==Real || mode==RealNoCache || mode==Fetch)
   {
    char *err=(Url->Protocol->request)(Url,request_head,request_body);

    if(err)
      {
       if(mode==Real || mode==Fetch)
         {
          lseek(spool,0,SEEK_SET);
          ftruncate(spool,0);
          HTMLMessage(spool,503,"WWWOFFLE Remote Host Error",NULL,"RemoteHostError",
                      "url",Url->name,
                      "reason",err,
                      "cache","yes",
                      "backup",spool_exists?"yes":NULL,
                      NULL);
          DeleteLockWebpageSpoolFile(Url);
         }
       if(mode==Real)
          HTMLMessage(client,503,"WWWOFFLE Remote Host Error",NULL,"RemoteHostError",
                      "url",Url->name,
                      "reason",err,
                      "cache","",
                      "backup",spool_exists?"yes":NULL,
                      NULL);
       else if(mode==RealNoCache)
          HTMLMessage(client,503,"WWWOFFLE Remote Host Error",NULL,"RemoteHostError",
                      "url",Url->name,
                      "reason",err,
                      "cache",NULL,
                      "backup",NULL,
                      NULL);
       else if(mode==Fetch && client!=-1)
          write_formatted(client,"Fetch Failure %s [Server Connection Error]\n",Url->name);
       exit(1);
      }
   }
 else if(offline_request &&
         ((mode==SpoolGet && (!ConfirmRequests || Url->local || (Url->args && *Url->args=='!'))) ||
          mode==SpoolRefresh || mode==SpoolPragma))
   {
    int err=write_string(outgoing,request_head);

    if(err==-1)
      {
       PrintMessage(Warning,"Cannot write the outgoing request.");
       HTMLMessage(client,500,"WWWOFFLE Server Error",NULL,"ServerError",
                   "error","Cannot write the outgoing request.",
                   NULL);
       exit(1);
      }

    if(request_body)
       write_string(outgoing,request_body);
   }

 /* Parse the reply */

 if(mode==Real || mode==RealNoCache || mode==Fetch)
   {
    reply_status=ParseReply(-1,Url,&reply_head);

    if(!reply_head)
      {
       PrintMessage(Warning,"Timed out reading the reply.");
       if(mode==Real || mode==Fetch)
         {
          lseek(spool,0,SEEK_SET);
          ftruncate(spool,0);
          HTMLMessage(spool,503,"WWWOFFLE Remote Host Error",NULL,"RemoteHostError",
                      "url",Url->name,
                      "reason","TimeoutReply",
                      "cache","yes",
                      "backup",spool_exists?"yes":NULL,
                      NULL);
          DeleteLockWebpageSpoolFile(Url);
         }
       if(mode==Real || mode==RealNoCache)
          HTMLMessage(client,503,"WWWOFFLE Remote Host Error",NULL,"RemoteHostError",
                      "url",Url->name,
                      "reason","TimeoutReply",
                      "cache",NULL,
                      "backup",spool_exists?"yes":NULL,
                      NULL);
       else if(mode==Fetch && client!=-1)
          write_formatted(client,"Fetch Failure %s [Server Reply Error]\n",Url->name);
       exit(1);
      }

    PrintMessage(ExtraDebug,"Incoming Reply Head (from server/proxy)\n%s",reply_head);

    if(mode==Fetch && (reply_status==301 || reply_status==302))
      {
       char *new_url=MovedLocation(Url,reply_head);

       if(client!=-1)
          write_formatted(client,"Fetching More %s [Page Moved]\n",Url->name);

       if(!new_url)
          PrintMessage(Warning,"Cannot parse the reply for the new location.");
       else
          fetch_again+=RecurseFetchRelocation(Url,new_url);
      }
    else if(reply_status==304)
      {
       PrintMessage(Inform,"Server page is not newer than the one in cache.");

       if(mode==Fetch || mode==Real)
         {
          close(spool);
          DeleteLockWebpageSpoolFile(Url);
          RestoreBackupWebpageSpoolFile(Url);
          if(!lasttime_exists)
             DeleteLastTimeSpoolFile(Url);
         }
       if(mode==Fetch)
         {
          if(client!=-1)
             write_formatted(client,"Not fetching %s [Page Unchanged]\n",Url->name);

          if(fetch_again)
            {
             spool=OpenWebpageSpoolFile(1,Url);
             init_buffer(spool);

             if(ParseDocument(spool,Url))
                RecurseFetch(Url,1);

             close(spool);

             exit(4);
            }
          else
             exit(0);
         }
       else if(mode==Real)
         {
          spool=OpenWebpageSpoolFile(1,Url);
          init_buffer(spool);

          mode=Spool;
         }
      }
    else if(mode==Fetch && reply_status==401 && Urlpw)
      {
       if(request_new(Urlpw,Url->name))
          fetch_again++;
       if(client!=-1)
          write_formatted(client,"Fetching More %s [URL with password]\n",Url->name);
      }
    else if(mode==Real && reply_status==401 && Urlpw)
      {
       mode=RealPassword;
      }
   }
 else
    reply_head=NULL;

 if(mode==Spool || mode==SpoolPragma)
    reply_status=ParseReply(spool,Url,&reply_head);

 reply_body=(char*)malloc(257);

 /* mode = Spool, SpoolGet, SpoolWillGet, SpoolPragma, SpoolRefresh, Real, RealRefresh, RealNoCache, RealPassword or Fetch */

 /* Close the outgoing file if any. */

 if(outgoing>=0)
   {
    if(mode==Fetch)
       close(outgoing);
    if(mode==SpoolGet || mode==SpoolRefresh || mode==SpoolPragma)
       CloseOutgoingSpoolFile(outgoing,Url);
   }

 /* The main body of the handling of the data. */

 if(mode==Real || mode==RealNoCache || mode==RealPassword)
   {
    int n=strlen(reply_head),err=0;

    if(mode!=RealNoCache)
       write_data(spool,reply_head,n);
    if(mode!=RealPassword)
       err=write_data(client,reply_head,n);

    while(err!=-1 && (n=(Url->Protocol->readbody)(reply_body,256))>0)
      {
       if(mode!=RealNoCache)
          write_data(spool,reply_body,n);
       if(mode!=RealPassword)
          err=write_data(client,reply_body,n);
      }

    if(mode!=RealNoCache)
      {
       if(err==-1 || n<0)
         {
          lseek(spool,0,SEEK_SET);
          ftruncate(spool,0);
         }

       if(err==-1)
         {
          PrintMessage(Warning,"Error writing to client [%!s]; client disconnected?");
          if(online!=-1) /* !autodial */
             HTMLMessage(spool,503,"WWWOFFLE Remote Host Error",NULL,"RemoteHostError",
                         "url",Url->name,
                         "reason","ClientClose",
                         "cache","yes",
                         "backup",spool_exists?"yes":NULL,
                         NULL);
         }
       else if(n<0)
         {
          PrintMessage(Warning,"Timed out while reading from remote host.");
          if(online!=-1) /* !autodial */
             HTMLMessage(spool,503,"WWWOFFLE Remote Host Error",NULL,"RemoteHostError",
                         "url",Url->name,
                         "reason","TimeoutTransfer",
                         "cache","yes",
                         "backup",spool_exists?"yes":NULL,
                         NULL);
         }
       else
          if(spool_exists)
             DeleteBackupWebpageSpoolFile(Url);

       DeleteLockWebpageSpoolFile(Url);

       if(err==-1 || n<0)
          exit(1);
      }
   }
 else if(mode==Fetch)
   {
    int n=strlen(reply_head);

    write_data(spool,reply_head,n);

    while((n=(Url->Protocol->readbody)(reply_body,256))>0)
       write_data(spool,reply_body,n);

    if(n<0)
      {
       lseek(spool,0,SEEK_SET);
       ftruncate(spool,0);
       PrintMessage(Warning,"Timed out while reading from remote host.");
       HTMLMessage(spool,503,"WWWOFFLE Remote Host Error",NULL,"RemoteHostError",
                   "url",Url->name,
                   "reason","TimeoutTransfer",
                   "cache","yes",
                   "backup",spool_exists?"yes":NULL,
                   NULL);

       if(client!=-1)
          write_formatted(client,"Fetch Failure %s [Timeout]\n",Url->name);
      }
    else
      {
       if(spool_exists)
          DeleteBackupWebpageSpoolFile(Url);
       if(client!=-1)
          write_formatted(client,"Fetch Success %s\n",Url->name);
      }

    DeleteLockWebpageSpoolFile(Url);

    if(n<0)
       exit(1);

    if(reply_status>=200 && reply_status<400)
      {
       lseek(spool,0,SEEK_SET);
       init_buffer(spool);

       if(ParseDocument(spool,Url))
         {
          int links=RecurseFetch(Url,1);

          if(client!=-1 && links)
             write_formatted(client,"Fetching More %s [%d Extra URLs]\n",Url->name,links);

          fetch_again+=links;
         }
      }
   }
 else if(mode==Spool || mode==SpoolPragma)
   {
    struct stat buf;
    char *head="HTTP/1.0 503 WWWOFFLE Remote Host Error\r\n";

    if(ExistsLockWebpageSpoolFile(Url))
      {
       int t=SocketTimeout;

       PrintMessage(Inform,"Waiting for the page to be unlocked.");

       while((t-=5) && ExistsLockWebpageSpoolFile(Url))
          sleep(5);

       if(t==0)
         {
          PrintMessage(Inform,"Timed out waiting for the page to be unlocked.");
          HTMLMessage(spool,503,"WWWOFFLE File Locked",NULL,"FileLocked",
                      "url",Url->name,
                      NULL);
          exit(0);
         }
      }

    if((reply_head && !strncmp(reply_head,head,strlen(head))) || (!fstat(spool,&buf) && buf.st_size==0))
      {
       DeleteWebpageSpoolFile(Url,0);
       RestoreBackupWebpageSpoolFile(Url);
      }

    if(EnableHTMLModifications &&
       reply_status>=200 && reply_status<400 && GetHTTPHeader(reply_head,"Content-Type: text/html") &&
       !GetHTTPHeader(request_head,"Pragma: wwwoffle") && !GetHTTPHeader(request_head,"User-Agent: htdig/"))
      {
       lseek(spool,0,SEEK_SET);
       init_buffer(spool);

       OutputHTMLWithModifications(client,spool,Url);
      }
    else
      {
       int n;

       if(reply_head)
          write_string(client,reply_head);

       while((n=read_data(spool,reply_body,256))>0)
          write_data(client,reply_body,n);
      }
   }
 else if(mode==SpoolGet)
   {
    if(offline_request)
      {
       if(ConfirmRequests && !Url->local && (!Url->args || *Url->args!='!'))
          HTMLMessage(client,404,"WWWOFFLE Confirm Request",NULL,"ConfirmRequest",
                      "url",Url->name,
                      NULL);
       else if(fetch_again)
          HTMLMessage(client,404,"WWWOFFLE Refresh Will Get",NULL,"RefreshWillGet",
                      "url",Url->name,
                      "password",HashOutgoingSpoolFile(Url),
                      NULL);
       else
          HTMLMessage(client,404,"WWWOFFLE Will Get",NULL,"WillGet",
                      "already",NULL,
                      "url",Url->name,
                      "password",HashOutgoingSpoolFile(Url),
                      NULL);
      }
    else
       HTMLMessage(client,404,"WWWOFFLE Refused Request",NULL,"RefusedRequest",
                   "url",Url->name,
                   NULL);
   }
 else if(mode==SpoolWillGet)
   {
    HTMLMessage(client,404,"WWWOFFLE Will Get",NULL,"WillGet",
                "already","yes",
                "url",Url->name,
                "password",HashOutgoingSpoolFile(Url),
                NULL);
   }
 else if(mode==RealRefresh || mode==SpoolRefresh)
   {
    HTMLMessage(client,301,"WWWOFFLE Refresh Redirect",Url->link,"RefreshRedirect",
                "url",Url->name,
                "link",Url->link,
                NULL);
   }

 /* Close down and exit. */

 if((mode==Real || mode==RealPassword) && ExistsOutgoingSpoolFile(Url))
    DeleteOutgoingSpoolFile(Url);

 if(mode==Real || mode==RealNoCache || mode==RealPassword || mode==Fetch)
    (Url->Protocol->close)();

 if(spool>=0)
    close(spool);

 if(reply_head)
    free(reply_head);
 if(reply_body)
    free(reply_body);

 if(mode==RealPassword)
   {
    FreeURL(Url);Url=Urlpw;Urlpw=NULL;
    spool_exists=spool_exists_pw;spool_exists_pw=0;
    reply_head=reply_body=NULL;
    mode=Real;
    goto realpassword;
   }

 if(request_head)
    free(request_head);
 if(request_body)
    free(request_body);

 if(client>=0)
    CloseSocket(client);

 if(fetch_again)
    return(4);
 else
    return(0);
}


/*++++++++++++++++++++++++++++++++++++++
  Make an SSL proxy connection.

  int client The client socket.

  URL *Url The URL to get (used for host only).

  char *request_head The head of the request.
  ++++++++++++++++++++++++++++++++++++++*/

static void ssl_tunnel(int client,URL *Url,char *request_head)
{
 char *err=SSL_Open(Url);

 if(err && ConnectRetry)
   {
    PrintMessage(Inform,"Waiting to try connection again.");
    sleep(10);
    err=SSL_Open(Url);
   }

 if(err)
   {
    HTMLMessage(client,503,"WWWOFFLE Remote Host Error",NULL,"RemoteHostError",
                "url",Url->host,
                "reason",err,
                "cache",NULL,
                "backup",NULL,
                NULL);
    exit(1);
   }

 request_head=ModifyRequest(Url,request_head);

 err=SSL_Request(client,Url,request_head);

 if(err)
   {
    HTMLMessage(client,503,"WWWOFFLE Remote Host Error",NULL,"RemoteHostError",
                "url",Url->host,
                "reason",err,
                "cache",NULL,
                "backup",NULL,
                NULL);
    exit(1);
   }
 else
    HTMLMessageHead(client,200,"WWWOFFLE SSL OK",
                    "Content-Type",NULL,
                    NULL);

 SSL_Transfer(client);

 SSL_Close();
}


/*++++++++++++++++++++++++++++++++++++++
  Make a new request.

  int request_new Returns 1 if successful.

  URL *Url The new URL to get.

  char *referer The refering URL.
  ++++++++++++++++++++++++++++++++++++++*/

static int request_new(URL *Url,char *referer)
{
 char *new_request=RequestURL(Url,referer);
 int new_outgoing=OpenOutgoingSpoolFile(0);
 int retval=0;

 if(new_outgoing==-1)
    PrintMessage(Warning,"Cannot open the new outgoing request to write.");
 else
   {
    write_string(new_outgoing,new_request);
    CloseOutgoingSpoolFile(new_outgoing,Url);
    retval=1;
   }

 free(new_request);

 return(retval);
}


/*++++++++++++++++++++++++++++++++++++++
  Uninstall the signal handlers.
  ++++++++++++++++++++++++++++++++++++++*/

static void uninstall_sighandlers(void)
{
 struct sigaction action;

 /* SIGCHLD */
 action.sa_handler = SIG_DFL;
 sigemptyset (&action.sa_mask);
 action.sa_flags = 0;
 if(sigaction(SIGCHLD, &action, NULL) != 0)
    PrintMessage(Warning, "Cannot uninstall SIGCHLD handler.");

 /* SIGINT, SIGQUIT, SIGTERM */
 action.sa_handler = SIG_DFL;
 sigemptyset(&action.sa_mask);
 action.sa_flags = 0;
 if(sigaction(SIGINT, &action, NULL) != 0)
    PrintMessage(Warning, "Cannot uninstall SIGINT handler.");
 if(sigaction(SIGQUIT, &action, NULL) != 0)
    PrintMessage(Warning, "Cannot uninstall SIGQUIT handler.");
 if(sigaction(SIGTERM, &action, NULL) != 0)
    PrintMessage(Warning, "Cannot uninstall SIGTERM handler.");

 /* SIGHUP */
 action.sa_handler = SIG_DFL;
 sigemptyset(&action.sa_mask);
 action.sa_flags = 0;
 if(sigaction(SIGHUP, &action, NULL) != 0)
    PrintMessage(Warning, "Cannot uninstall SIGHUP handler.");

 /* SIGPIPE */
 action.sa_handler = SIG_IGN;
 sigemptyset (&action.sa_mask);
 action.sa_flags = 0;
 if(sigaction(SIGPIPE, &action, NULL) != 0)
    PrintMessage(Warning, "Cannot ignore SIGPIPE.");
}
