/***************************************
  $Header: /home/amb/wwwoffle/RCS/refresh.c 2.41 1998/12/20 14:36:04 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 2.4a.
  The HTML interactive page to refresh a URL.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1997,98 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <unistd.h>

#include "wwwoffle.h"
#include "document.h"
#include "misc.h"
#include "config.h"
#include "sockets.h"
#include "errors.h"


/*+ The options for recursive or normal fetching. +*/
static int recursive=0;
static int recursive_depth=0,recursive_mode=0,force=0;
static int stylesheets=0,images=0,frames=0,scripts=0,objects=0;

static char *RefreshFormParse(int fd,char *request_body);

static int request_url(URL *Url,char *path,char *referer);


/*++++++++++++++++++++++++++++++++++++++
  Send to the client a page to allow refreshes using HTML.

  char *RefreshPage Returns a modified URLs for a simple refresh.

  int fd The file descriptor of the client.

  URL *Url The URL that was used to request this page.

  char *request_body A pointer to the HTTP request sent by the browser.

  int *recurse Return value set to true if a recursive fetch was asked for.
  ++++++++++++++++++++++++++++++++++++++*/

char *RefreshPage(int fd,URL *Url,char *request_body,int *recurse)
{
 char *newurl=NULL;

 if(!strcmp("/refresh-options/",Url->path))
    HTMLMessage(fd,200,"WWWOFFLE Refresh Form",NULL,"RefreshPage",
                "url",Url->args,
                "stylesheets",FetchStyleSheets?"yes":NULL,
                "images",FetchImages?"yes":NULL,
                "frames",FetchFrames?"yes":NULL,
                "scripts",FetchScripts?"yes":NULL,
                "objects",FetchObjects?"yes":NULL,
                NULL);
 else if(!strcmp("/refresh-request/",Url->path))
   {
    if((newurl=RefreshFormParse(fd,request_body)))
       *recurse=1;
   }
 else if(!strcmp("/refresh/",Url->path))
   {
    newurl=(char*)malloc(strlen(Url->args)+1);
    strcpy(newurl,Url->args);
   }
 else
   {
    newurl=(char*)malloc(strlen(Url->args)+1);
    strcpy(newurl,Url->args);
    *recurse=1;
   }

 if(*recurse)
    ParseRecurseOptions(Url->path+8);

 return(newurl);
}


/*++++++++++++++++++++++++++++++++++++++
  Parse the reply from the form.

  char *RefreshFormParse Returns the first URL to get.

  int fd The file descriptor of the client.

  char *request_body The body of the HTTP request sent by the browser.
  ++++++++++++++++++++++++++++++++++++++*/

static char *RefreshFormParse(int fd,char *request_body)
{
 int i;
 char *copy,*url=NULL,*method=NULL,*force="";
 char *stylesheets="",*images="",*frames="",*scripts="",*objects="";
 URL *Url;
 char *new_url;

 if(!request_body)
   {
    HTMLMessage(fd,404,"WWWOFFLE Refresh Form Error",NULL,"RefreshFormError",
                "body",request_body,
                NULL);
    return(NULL);
   }

 copy=(char*)malloc(strlen(request_body)+1);
 strcpy(copy,request_body);

 for(i=0;copy[i];i++)
   {
    if(i!=0 && copy[i-1]=='&')
       copy[i-1]=0;
    if(i==0 || copy[i-1]==0)
      {
       if(!strncmp("method=",&copy[i],7))
          method=&copy[i+7];
       if(!strncmp("force=",&copy[i],6))
          force=&copy[i+6];
       if(!strncmp("stylesheets=",&copy[i],12))
          stylesheets=&copy[i+12];
       if(!strncmp("images=",&copy[i],7))
          images=&copy[i+7];
       if(!strncmp("frames=",&copy[i],7))
          frames=&copy[i+7];
       if(!strncmp("scripts=",&copy[i],8))
          scripts=&copy[i+8];
       if(!strncmp("objects=",&copy[i],8))
          objects=&copy[i+8];
       if(!strncmp("url=",&copy[i],4))
          url=&copy[i+4];
      }
   }

 if(url==NULL || *url==0 || method==NULL || *method==0)
   {
    HTMLMessage(fd,404,"WWWOFFLE Refresh Form Error",NULL,"RefreshFormError",
                "body",request_body,
                NULL);
    free(copy);
    return(NULL);
   }

 url=URLDecode(url,1);
 Url=SplitURL(url);

 new_url=(char*)malloc(strlen(request_body)+64);
 strcpy(new_url,"/refresh");
 strcat(new_url,method);
 strcat(new_url,force);
 strcat(new_url,stylesheets);
 strcat(new_url,images);
 strcat(new_url,frames);
 strcat(new_url,scripts);
 strcat(new_url,objects);
 strcat(new_url,"/?");
 strcat(new_url,Url->name);

 FreeURL(Url);

 free(copy);

 return(new_url);
}


/*++++++++++++++++++++++++++++++++++++++
  Parse the url method to decide what needs fetching recursively.

  char *method The method to use, encoding the depth and other options.
  ++++++++++++++++++++++++++++++++++++++*/

void ParseRecurseOptions(char *method)
{
 if(method)
   {
    char *copy=(char*)malloc(strlen(method)+1),*dash,*slash;

    strcpy(copy,method);

    if((slash=strchr(copy,'/')))
       *slash=0;

    PrintMessage(Debug,"Refresh method='%s'.",copy);

    if(*copy=='-')
       copy++;

    do
      {
       if((dash=strchr(copy,'-')))
          *dash=0;

       if(!strcmp(copy,"refresh"))
          ;
       else if(!strcmp(copy,"none"))
          ;
       else if(!strcmp(copy,"dir"))
          recursive_mode=1;
       else if(!strcmp(copy,"host"))
          recursive_mode=2;
       else if(!strcmp(copy,"any"))
          recursive_mode=3;
       else if(!strcmp(copy,"force"))
          force=1;
       else if(!strcmp(copy,"stylesheets"))
          stylesheets=1;
       else if(!strcmp(copy,"images"))
          images=1;
       else if(!strcmp(copy,"frames"))
          frames=1;
       else if(!strcmp(copy,"scripts"))
          scripts=1;
       else if(!strcmp(copy,"objects"))
          objects=1;
       else if(atoi(copy))
          recursive_depth=atoi(copy);

       copy=dash+1;
      }
    while(dash);

    recursive=1;
   }
 else
   {
    stylesheets=FetchStyleSheets;
    images=FetchImages;
    frames=FetchFrames;
    scripts=FetchScripts;
    objects=FetchObjects;
   }
}


/*++++++++++++++++++++++++++++++++++++++
  Fetch the images, etc from the just parsed page.

  int RecurseFetch Returns 1 if there are more to be fetched.

  URL *Url The URL that was fetched.

  int new Set to true if the page is new to the cache, else we may be in infinite recursion.
  ++++++++++++++++++++++++++++++++++++++*/

int RecurseFetch(URL *Url,int new)
{
 char **list,*metarefresh;
 int more=0;
 int j;

 /* A Meta-Refesh header. */

 if(new && (metarefresh=MetaRefresh()))
   {
    URL *metarefreshUrl=SplitURL(metarefresh);

    if(!metarefreshUrl->local && metarefreshUrl->Protocol)
      {
       char *refresh;

       if(recursive)
         {
          char *refreshpath=CreateRefreshPath(recursive_depth,recursive_mode,force,
                                              stylesheets,images,frames,scripts,objects);
          refresh=(char*)malloc(strlen(metarefreshUrl->name)+strlen(refreshpath)+8);

          strcpy(refresh,refreshpath);
          strcat(refresh,"/?");
          strcat(refresh,metarefreshUrl->name);
         }
       else
          refresh=metarefreshUrl->name;

       PrintMessage(Debug,"Meta-Refresh=%s",metarefreshUrl->name);
       more+=request_url(metarefreshUrl,refresh,Url->name);

       if(recursive)
          free(refresh);
      }

    FreeURL(metarefreshUrl);
   }

 /* Any style sheets. */

 if(stylesheets && (list=GetReferences(RefStyleSheet)))
    for(j=0;list[j];j++)
      {
       URL *stylesheetUrl=SplitURL(list[j]);

       if(!stylesheetUrl->local && stylesheetUrl->Protocol)
         {
          PrintMessage(Debug,"Style-Sheet=%s",stylesheetUrl->name);
          more+=request_url(stylesheetUrl,stylesheetUrl->name,Url->name);
         }

       FreeURL(stylesheetUrl);
      }

 /* Any images. */

 if(images && (list=GetReferences(RefImage)))
    for(j=0;list[j];j++)
      {
       URL *imageUrl=SplitURL(list[j]);

       if(!imageUrl->local && imageUrl->Protocol)
         {
          PrintMessage(Debug,"Image=%s",imageUrl->name);
          more+=request_url(imageUrl,imageUrl->name,Url->name);
         }

       FreeURL(imageUrl);
      }

 /* Any frames */

 if(new && frames && (list=GetReferences(RefFrame)))
    for(j=0;list[j];j++)
      {
       URL *frameUrl=SplitURL(list[j]);

       if(!frameUrl->local && frameUrl->Protocol)
         {
          char *refresh;

          if(recursive)
            {
             char *refreshpath=CreateRefreshPath(recursive_depth,recursive_mode,force,
                                                 stylesheets,images,frames,scripts,objects);
             refresh=(char*)malloc(strlen(frameUrl->name)+strlen(refreshpath)+8);

             strcpy(refresh,refreshpath);
             strcat(refresh,"/?");
             strcat(refresh,frameUrl->name);
            }
          else
             refresh=frameUrl->name;

          PrintMessage(Debug,"Frame=%s",frameUrl->name);
          more+=request_url(frameUrl,refresh,Url->name);

          if(recursive)
             free(refresh);
         }

       FreeURL(frameUrl);
      }

 /* Any scripts. */

 if(scripts && (list=GetReferences(RefScript)))
    for(j=0;list[j];j++)
      {
       URL *scriptUrl=SplitURL(list[j]);

       if(!scriptUrl->local && scriptUrl->Protocol)
         {
          PrintMessage(Debug,"Script=%s",scriptUrl->name);
          more+=request_url(scriptUrl,scriptUrl->name,Url->name);
         }

       FreeURL(scriptUrl);
      }

 /* Any Objects. */

 if(objects && (list=GetReferences(RefObject)))
    for(j=0;list[j];j++)
      {
       URL *objectUrl=SplitURL(list[j]);

       if(!objectUrl->local && objectUrl->Protocol)
         {
          PrintMessage(Debug,"Object=%s",objectUrl->name);
          more+=request_url(objectUrl,objectUrl->name,Url->name);
         }

       FreeURL(objectUrl);
      }

 if(objects && (list=GetReferences(RefJavaObject)))
    for(j=0;list[j];j++)
      {
       URL *objectUrl=SplitURL(list[j]);

       if(!objectUrl->local && objectUrl->Protocol)
         {
          /* classes have to be fetched recursive */
          char *refreshpath=CreateRefreshPath(recursive_depth,recursive_mode,force,
                                              0,0,0,0,objects);
          char *refresh=(char*)malloc(strlen(objectUrl->name)+strlen(refreshpath)+8);

          strcpy(refresh,refreshpath);
          strcat(refresh,"/?");
          strcat(refresh,objectUrl->name);

          PrintMessage(Debug,"Classes=%s",objectUrl->name);
          more+=request_url(objectUrl,refresh,Url->name);

          free(refresh);
         }

       FreeURL(objectUrl);
      }

 /* Any links */

 if(recursive_depth && (list=GetReferences(RefLink)))
    for(j=0;list[j];j++)
      {
       URL *linkUrl=SplitURL(list[j]);

       if(!linkUrl->local && linkUrl->Protocol)
         {
          int get=1;

          if(recursive_mode!=3)
            {
             if(strcmp(Url->host,linkUrl->host))
                get=0;
             else
                if(recursive_mode!=2)
                  {
                   char *end=Url->path+strlen(Url->path);

                   while(end>Url->path)
                      if(*end=='/')
                         break;
                      else
                         end--;
                   if(*end)
                      *++end=0;
                   if(end!=Url->path && strncmp(Url->path,linkUrl->path,end-Url->path))
                      get=0;
                  }
            }

          if(get)
            {
             char *refreshpath=CreateRefreshPath(recursive_depth-1,recursive_mode,force,
                                                 stylesheets,images,frames,scripts,objects);
             char *refresh=(char*)malloc(strlen(linkUrl->name)+strlen(refreshpath)+8);

             strcpy(refresh,refreshpath);
             strcat(refresh,"/?");
             strcat(refresh,linkUrl->name);

             PrintMessage(Debug,"Link=%s",linkUrl->name);
             more+=request_url(linkUrl,refresh,Url->name);

             free(refresh);
            }
         }

       FreeURL(linkUrl);
      }

 return(more);
}


/*++++++++++++++++++++++++++++++++++++++
  Fetch the relocated page.

  int RecurseFetchRelocation Returns 1 if there are more to be fetched.

  URL *Url The URL that was fetched.

  char *location The new location of the URL.
  ++++++++++++++++++++++++++++++++++++++*/

int RecurseFetchRelocation(URL *Url,char *location)
{
 int more=0;
 URL *locationUrl=SplitURL(location);

 if(!locationUrl->local && locationUrl->Protocol)
   {
    char *refresh;

    if(recursive)
      {
       char *refreshpath=CreateRefreshPath(recursive_depth,recursive_mode,force,
                                           stylesheets,images,frames,scripts,objects);
       refresh=(char*)malloc(strlen(locationUrl->name)+strlen(refreshpath)+8);

       strcpy(refresh,refreshpath);
       strcat(refresh,"/?");
       strcat(refresh,locationUrl->name);
      }
    else
       refresh=locationUrl->name;

    PrintMessage(Debug,"Location=%s",locationUrl->name);
    more+=request_url(locationUrl,refresh,Url->name);

    if(recursive)
       free(refresh);
   }

 return(more);
}


/*++++++++++++++++++++++++++++++++++++++
  Make a request for a URL.

  int request_url Returns 1 on success, else 0.

  URL *Url The URL that was asked for.

  char *path The URL path that is required (with refresh information).

  char *referer The refering URL.
  ++++++++++++++++++++++++++++++++++++++*/

static int request_url(URL *Url,char *path,char *referer)
{
 int retval=0;

 if(recursive && IsNotGotRecursive(Url->proto,Url->host,Url->path))
   {
    PrintMessage(Inform,"The server '%s://%s' and/or path '%s' is on the list not to get recursively.",Url->proto,Url->host,Url->path);
   }
 else
   {
    int new_outgoing=OpenOutgoingSpoolFile(0);

    if(new_outgoing==-1)
       PrintMessage(Warning,"Cannot open the new outgoing request to write.");
    else
      {
       URL *reqUrl=SplitURL(path);
       char *new_request;

       if(Url->pass && !strcmp(Url->host,reqUrl->host))
          AddURLPassword(reqUrl,Url->user,Url->pass);

       new_request=RequestURL(reqUrl,referer);

       if(force)
         {
          char *copy=(char*)malloc(strlen(new_request)+24);
          char *eol=strchr(new_request,'\n');

          *eol=0;eol++;
          strcpy(copy,new_request);
          strcat(copy,"\nPragma: no-cache\r\n");
          strcat(copy,eol);

          free(new_request);
          new_request=copy;
         }

       write_string(new_outgoing,new_request);
       CloseOutgoingSpoolFile(new_outgoing,reqUrl);

       retval=1;

       free(reqUrl);
       free(new_request);
      }
   }

 return(retval);
}


/*++++++++++++++++++++++++++++++++++++++
  Create a Path for doing a refresh with options.

  char *CreateRefreshPath Returns a pointer to a static string.
  ++++++++++++++++++++++++++++++++++++++*/

char *CreateRefreshPath(int recursive_depth,int recursive_mode,int force,
                        int stylesheets,int images,int frames,int scripts,int objects)
{
 static char refresh[64];

 strcpy(refresh,"/refresh");

 if(recursive_depth)
   {
    if(recursive_mode==1)
       strcat(refresh,"-dir");
    else if(recursive_mode==2)
       strcat(refresh,"-host");
    else /* recursive_mode==3 */
       strcat(refresh,"-any");

    sprintf(&refresh[strlen(refresh)],"-%d",recursive_depth);
   }

 if(force)
    strcat(refresh,"-force");

 if(stylesheets)
    strcat(refresh,"-stylesheets");
 if(images)
    strcat(refresh,"-images");
 if(frames)
    strcat(refresh,"-frames");
 if(scripts)
    strcat(refresh,"-scripts");
 if(objects)
    strcat(refresh,"-objects");

 if(!recursive_depth && !force && !stylesheets && !images && !frames && !scripts && !objects)
    strcat(refresh,"-none");

 return(refresh);
}
