/***************************************
  $Header: /home/amb/wwwoffle/RCS/configedit.c 2.17 1998/12/20 14:33:19 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 2.4a.
  Configuration file management via a web-page.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1997,98 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/stat.h>
#include <unistd.h>

#include "wwwoffle.h"
#include "misc.h"
#include "config.h"
#include "errors.h"

#define COMMENT 0
#define SECTION 1
#define CONTENT 2

static void ConfigEditForms(int fd,char ***sections);
static void ConfigEditUpdate(int fd,char *section,char ***sections);

static char ***read_config_file(void);
static int write_config_file(char ***sections);
static void free_sections(char ***sections);


/*++++++++++++++++++++++++++++++++++++++
  The control page that allows editing of the configuration file.

  int fd The file descriptor to write the file to.

  char *args The arguments to the page.

  char *request_body The body of the HTTP request for the page.
  ++++++++++++++++++++++++++++++++++++++*/

void ConfigEditPage(int fd,char *args,char *request_body)
{
 char *newargs=NULL;
 char ***sections;

 if(args)
   {
    if(*args=='!' && strchr(args+1,'!'))
      {
       char *pling;
       newargs=(char*)malloc(strlen(args)+1);
       strcpy(newargs,args+1);
       pling=strchr(newargs,'!');
       *pling=0;
      }
    else if(*args!='!')
      {
       newargs=(char*)malloc(strlen(args)+1);
       strcpy(newargs,args);
      }
   }

 sections=read_config_file();

 if(!sections)
    HTMLMessage(fd,404,"WWWOFFLE Configuration Error",NULL,"ConfigError",
                "section",NULL,
                "reason","ReadError",
                NULL);
 else if(newargs && *newargs)
   {
    int i=0;
    char *section=NULL;

    while(sections[i])
      {
       if(sections[i][SECTION] && !strcmp(sections[i][SECTION],newargs))
         {section=newargs;break;}
       i++;
      }

    if(!section)
       HTMLMessage(fd,404,"WWWOFFLE Configuration Error",NULL,"ConfigError",
                   "section",newargs,
                   "reason","BadSection",
                   NULL);
    else if(!request_body || strncmp(request_body,"value=",6))
       HTMLMessage(fd,404,"WWWOFFLE Configuration Error",NULL,"ConfigError",
                   "section",newargs,
                   "reason","BadBody",
                   NULL);
    else
      {
       char *new=URLDecode(request_body+6,1);

#if !defined(__CYGWIN__)
       char *p,*q;

       for(p=q=new;*p;p++)
          if(*p!='\r')
             *q++=*p;
       *q=0;
#endif

       sections[i][CONTENT]=new;

       ConfigEditUpdate(fd,section,sections);
      }
   }
 else
    ConfigEditForms(fd,sections);

 if(sections)
    free_sections(sections);

 if(newargs)
    free(newargs);
}


/*++++++++++++++++++++++++++++++++++++++
  The page that contains the forms making up the config file.

  int fd The file descriptor to write to.

  char ***sections The sections of the file.
  ++++++++++++++++++++++++++++++++++++++*/

static void ConfigEditForms(int fd,char ***sections)
{
 int i=0;

 HTMLMessageHead(fd,200,"WWWOFFLE Configuration Edit Page",
                 NULL);

 HTMLMessageBody(fd,"ConfigEditPage-Head",
                 NULL);

 while(sections[i])
   {
    char *htmlcomment=NULL,*htmlcontent=NULL;

    if(sections[i][COMMENT])
       htmlcomment=HTMLString(sections[i][COMMENT]);
    if(sections[i][CONTENT])
       htmlcontent=HTMLString(sections[i][CONTENT]);

    HTMLMessageBody(fd,"ConfigEditPage-Body",
                    "section",sections[i][SECTION],
                    "comment",htmlcomment,
                    "content",htmlcontent,
                    NULL);

    if(htmlcomment)
       free(htmlcomment);
    if(htmlcontent)
       free(htmlcontent);

    i++;
   }

 HTMLMessageBody(fd,"ConfigEditPage-Tail",
                 NULL);
}


/*++++++++++++++++++++++++++++++++++++++
  Update the configuration file.

  int fd The file descriptor to write the message to.

  char *section The section that was updated.

  char ***sections The sections including the updated one.
  ++++++++++++++++++++++++++++++++++++++*/

static void ConfigEditUpdate(int fd,char *section,char ***sections)
{
 if(write_config_file(sections))
    HTMLMessage(fd,404,"WWWOFFLE Configuration Error",NULL,"ConfigError",
                "section",section,
                "reason","WriteError",
                NULL);
 else
    HTMLMessage(fd,200,"WWWOFFLE Configuration Update",NULL,"ConfigUpdate",
                "section",section,
                NULL);
}


/*++++++++++++++++++++++++++++++++++++++
  Read in the config file into a set of sections.

  char ***read_config_file Returns the sections of the file as a NULL terminated list.
  ++++++++++++++++++++++++++++++++++++++*/

static char ***read_config_file(void)
{
 int sec_num=0,state=0;
 FILE *conf;
 char ***sections;
 char *line=NULL;
 int line_num=0;

 conf=fopen(ConfigFile,"r");
 if(!conf)
   {PrintMessage(Warning,"Cannot open the config file '%s' for reading.",ConfigFile); return(NULL);}

 sections=(char***)calloc(1,sizeof(char**));

 while((line=fgets_realloc(line,conf)))
   {
    char *l=line;
    char *r=line+strlen(line)-1;

    line_num++;

    while(*l==' ' || *l=='\t' || *l=='\r' || *l=='\n')
       l++;

    if(state==0 && *l=='#')
      {
       state=1;
       sections=(char***)realloc((void*)sections,sizeof(char**)*(sec_num+2));
       sections[sec_num]=(char**)calloc(3,sizeof(char*));
       sections[++sec_num]=NULL;
       sections[sec_num-1][COMMENT]=(char*)malloc(strlen(l)+1);
       strcpy(sections[sec_num-1][COMMENT],l);
      }
    else if((state==1 || state==2) && *l=='#')
      {
       sections[sec_num-1][COMMENT]=(char*)realloc((void*)sections[sec_num-1][COMMENT],strlen(sections[sec_num-1][COMMENT])+strlen(l)+1);
       strcat(sections[sec_num-1][COMMENT],l);
      }
    else if(state==0 && !*l)
      ;
    else if(state==1 && !*l)
       state=0;
    else if((state==0 || state==1) && *l)
      {
       state=2;
       while(r>l && (*r==' ' || *r=='\t' || *r=='\r' || *r=='\n'))
          *r--=0;
       if(sec_num==0 || sections[sec_num-1][SECTION])
         {
          sections=(char***)realloc((void*)sections,sizeof(char**)*(sec_num+2));
          sections[sec_num]=(char**)calloc(3,sizeof(char*));
          sections[++sec_num]=NULL;
         }
       sections[sec_num-1][SECTION]=(char*)malloc(strlen(l)+1);
       strcpy(sections[sec_num-1][SECTION],l);
      }
    else if(state==2 && !*l)
      ;
    else if(state==2 && *l=='{')
      {
       state=3;
       sections[sec_num-1][CONTENT]=(char*)malloc(1);
       strcpy(sections[sec_num-1][CONTENT],"");
      }
    else if(state==3 && *l=='}')
       state=0;
    else if(state==3)
      {
       sections[sec_num-1][CONTENT]=(char*)realloc((void*)sections[sec_num-1][CONTENT],strlen(sections[sec_num-1][CONTENT])+strlen(line)+1);
       strcat(sections[sec_num-1][CONTENT],line);
      }
    else
      {
       line[strlen(line)-1]=0;
       PrintMessage(Warning,"Error parsing config file, line %d = '%s' [state=%d]",line_num,line,state);
       free_sections(sections);
       return(NULL);
      }
   }

 fclose(conf);

 return(sections);
}


/*++++++++++++++++++++++++++++++++++++++
  Write out a set of sections to the config file.

  int write_config_file Returns 1 if in error.

  char ***sections The sections to write out.
  ++++++++++++++++++++++++++++++++++++++*/

static int write_config_file(char ***sections)
{
 char *conf_file_backup=(char*)malloc(strlen(ConfigFile)+5);
 int renamed=0,i=0;
 struct stat buf;
 FILE *conf;

 /* Rename the old file as a backup. */

 strcpy(conf_file_backup,ConfigFile);
 strcat(conf_file_backup,".bak");

 if(rename(ConfigFile,conf_file_backup))
    PrintMessage(Warning,"Cannot rename the config file '%s' to '%s'.",ConfigFile,conf_file_backup);
 else
   {
    renamed=1;
    if(stat(conf_file_backup,&buf))
       PrintMessage(Warning,"Cannot stat the config file '%s'.",ConfigFile);
   }

 conf=fopen(ConfigFile,"w");
 if(!conf)
   {PrintMessage(Warning,"Cannot open the config file '%s' for writing.",ConfigFile); return(1);}

 if(renamed)
   {
    chown(ConfigFile,buf.st_uid,buf.st_gid);
    chmod(ConfigFile,buf.st_mode&(~S_IFMT));
   }

 while(sections[i])
   {
    if(sections[i][COMMENT])
      {
       fprintf(conf,sections[i][COMMENT]);
       fprintf(conf,"\n");
      }
    if(sections[i][SECTION])
      {
       fprintf(conf,"%s\n",sections[i][SECTION]);
       fprintf(conf,"{\n");
       if(sections[i][CONTENT])
          fprintf(conf,sections[i][CONTENT]);
       if(sections[i][CONTENT][strlen(sections[i][CONTENT])-1]!='\n')
          fprintf(conf,"\n");
       fprintf(conf,"}\n\n");
      }
    i++;
   }

 fclose(conf);

 free(conf_file_backup);

 return(0);
}


/*++++++++++++++++++++++++++++++++++++++
  Free up a set of sections.

  char ***sections The sections that are to be freed up.
  ++++++++++++++++++++++++++++++++++++++*/

static void free_sections(char ***sections)
{
 int i=0;

 while(sections[i])
   {
    if(sections[i][COMMENT])
       free(sections[i][COMMENT]);
    if(sections[i][SECTION])
       free(sections[i][SECTION]);
    if(sections[i][CONTENT])
       free(sections[i][CONTENT]);
    free(sections[i]);
    i++;
   }

 free(sections);
}
