/***************************************
  $Header: /home/amb/wwwoffle/RCS/index.c 2.42 1998/06/14 17:56:57 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 2.2.
  Generate an index of the web pages that are cached in the system.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1997,98 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <time.h>
#include <utime.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <fcntl.h>
#include <unistd.h>

#include "wwwoffle.h"
#include "misc.h"
#include "proto.h"
#include "config.h"
#include "errors.h"


/*+ A type to contain the information required to sort the files. +*/
typedef struct _FileIndex
{
 char *name;                    /*+ The URL of the file. +*/
 char *type;                    /*+ The type (file extension) of the URL. +*/
 long time;                     /*+ The time of the file (access or modification as appropriate). +*/
}
FileIndex;

/*+ The method of sorting used. +*/
static char *sorttype[][2]={{"none"  ,"Unsorted"},
                            {"mtime" ,"Modification&nbsp;Time"},
                            {"atime" ,"Access&nbsp;Time"},
                            {"dated" ,"Date&nbsp;Changed"},
                            {"alpha" ,"Alphabetical"},
                            {"type"  ,"File&nbsp;Type"},
                            {"none"  ,"Unsorted"}};

/*+ An enumerated type for the sort modes. +*/
typedef enum _SortMode
{
 None,                          /*+ No sorting, natural order. +*/
 MTime,                         /*+ Sort by modification time +*/
 ATime,                         /*+ Sort by access time. +*/
 Dated,                         /*+ Sort by modification time with separators. +*/
 Alpha,                         /*+ Sort Alphabetically. +*/
 Type,                          /*+ Sort by type (file extension). +*/
 NSortModes                     /*+ The number of sort modes. +*/
}
SortMode;


/*+ The list of files. +*/
static FileIndex **files=NULL;
static int nfiles=0;

/*+ The earliest time that files can appear in the dated list. +*/
static time_t earliest=0;

/*+ The current time. +*/
static time_t now;

static void IndexRoot(int fd);
static void IndexProtocol(int fd,char *proto,SortMode mode);
static void IndexHost(int fd,char *proto,char *host,SortMode mode);
static void IndexOutgoing(int fd,SortMode mode);
static void IndexMonitor(int fd,SortMode mode);
static void IndexLastTime(int fd,SortMode mode,char *name);
static void IndexLatest(int fd,SortMode mode);

static void add_dir(char *name,SortMode mode,char *link);
static void add_file(char *name,SortMode mode);

static void dated_separator(int fd,int file,int *lastdays,int *lasthours);

static int sort_alpha(FileIndex **a,FileIndex **b);
static int sort_type(FileIndex **a,FileIndex **b);
static int sort_time(FileIndex **a,FileIndex **b);


/*++++++++++++++++++++++++++++++++++++++
  Generate an index of the pages that are in the cache.

  int fd The file descriptor to write the output to.

  URL *Url The URL that specifies the path to generate the index for.
  ++++++++++++++++++++++++++++++++++++++*/

void IndexPage(int fd,URL *Url)
{
 char *newpath;
 char *proto=NULL,*host="";
 SortMode sort=NSortModes;
 Protocol *protocol=NULL;
 int outgoing=0,monitor=0,lasttime=0,prevtime=0,latest=0,main=0;
 int i;

 if(!strcmp(Url->path,"/index/url/"))
   {
    URL *indexUrl=SplitURL(Url->args);
    char *localhost=GetLocalHost(1);
    char *relocate=(char*)malloc(strlen(localhost)+strlen(indexUrl->proto)+strlen(indexUrl->host)+24);
    sprintf(relocate,"http://%s/index/%s/%s?%s\r\n",localhost,indexUrl->proto,indexUrl->host,sorttype[Alpha][0]);
    HTMLMessage(fd,301,"WWWOFFLE Index URL Redirect",relocate,"IndexURLRedirect",
                "url",Url->args,
                "link",relocate+7+strlen(localhost),
                NULL);
    FreeURL(indexUrl);
    free(localhost);
    free(relocate);
    return;
   }

 newpath=(char*)malloc(strlen(Url->path)-6);
 strcpy(newpath,Url->path+7);

 if(*newpath && newpath[strlen(newpath)-1]=='/')
    newpath[strlen(newpath)-1]=0;

 for(i=0;newpath[i];i++)
    if(newpath[i]=='/')
      {
       newpath[i]=0;
       host=newpath+i+1;
       break;
      }

 proto=newpath;

 for(i=0;i<NProtocols;i++)
    if(!strcmp(proto,Protocols[i].name))
       protocol=&Protocols[i];

 outgoing=!strcmp(proto,"outgoing");
 monitor =!strcmp(proto,"monitor");
 lasttime=!strcmp(proto,"lasttime");
 prevtime=!strncmp(proto,"prevtime",8) && atoi(proto+8)>=0 && atoi(proto+8)<=NUM_PREVTIME_DIR;
 latest  =!strcmp(proto,"latest");
 main=(!protocol && !outgoing && !monitor && !lasttime && !prevtime && !latest);

 if(Url->args)
    for(sort=0;sort<NSortModes;sort++)
       if(!strcmp(Url->args,sorttype[sort][0]))
          break;

 now=time(NULL);

 if((*host && (strchr(host,'/') || !strcmp(host,"..") || !strcmp(host,"."))) ||
    (main && Url->path[7]) ||
    ((outgoing || monitor || lasttime || prevtime || latest) && *host))
    HTMLMessage(fd,404,"WWWOFFLE Illegal Index Page",NULL,"IndexIllegal",
                "url",Url->pathp,
                NULL);
 else if(sort==NSortModes && (outgoing || monitor || lasttime || prevtime || protocol))
   {
    char *localhost=GetLocalHost(1);
    char *relocate=(char*)malloc(strlen(localhost)+strlen(Url->path)+strlen(sorttype[0][0])+16);
    sprintf(relocate,"http://%s/index/%s?%s\r\n",localhost,Url->path,sorttype[0][0]);
    HTMLMessage(fd,301,"WWWOFFLE Index Redirect",relocate,"IndexRedirect",
                "url",Url->pathp,
                "link",relocate+7+strlen(localhost),
                NULL);
    free(relocate);
    free(localhost);
   }
 else
   {
    HTMLMessageHead(fd,200,"WWWOFFLE Index",
                    NULL);
    HTMLMessageBody(fd,"Index-Head",
                    "type",prevtime?"prevtime":proto,
                    "subtype",prevtime?proto+8:host,
                    "sort",sorttype[sort][0],
                    NULL);

    if(outgoing)
       IndexOutgoing(fd,sort);
    else if(monitor)
       IndexMonitor(fd,sort);
    else if(lasttime || prevtime)
       IndexLastTime(fd,sort,proto);
    else if(latest)
       IndexLatest(fd,sort);
    else if(protocol && !*host)
       IndexProtocol(fd,proto,sort);
    else if(protocol && *host)
       IndexHost(fd,proto,host,sort);
    else
       IndexRoot(fd);

    HTMLMessageBody(fd,"Index-Tail",
                    NULL);
   }

 free(newpath);
}


/*++++++++++++++++++++++++++++++++++++++
  Index the root of the cache.

  int fd The file descriptor to write to.
  ++++++++++++++++++++++++++++++++++++++*/

static void IndexRoot(int fd)
{
 HTMLMessageBody(fd,"IndexRoot-Body",NULL);
}


/*++++++++++++++++++++++++++++++++++++++
  Index the hosts for one protocol in the cache.

  int fd The file descriptor to write to.

  char *proto The protocol to index.

  SortMode mode The sort mode to use.
  ++++++++++++++++++++++++++++++++++++++*/

static void IndexProtocol(int fd,char *proto,SortMode mode)
{
 DIR *dir;
 struct dirent* ent;
 char ***mirrored;
 int i;

 /* Open the spool directory. */

 if(chdir(proto))
   {PrintMessage(Warning,"Cannot change to directory '%s' [%!s]; not indexed.",proto);
   write_formatted(fd,"No %s directory.",proto);return;}

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open spool directory '%s' [%!s]; index failed.",proto);chdir("..");
   write_formatted(fd,"Unable to open directory %s.",proto);return;}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read spool directory '%s' [%!s]; index failed.",proto);closedir(dir);chdir("..");
   write_formatted(fd,"Unable to read directory %s.",proto);return;}
 ent=readdir(dir);  /* skip .. */

 /* Get all of the host sub-directories. */

 while((ent=readdir(dir)))
    add_dir(ent->d_name,mode,NULL);

 closedir(dir);

 /* Get all of the mirrored hosts. */

 if((mirrored=ListMirrors(proto)))
   {
    int i;

    for(i=0;mirrored[i];i++)
      {
       add_dir(mirrored[i][0],mode,mirrored[i][1]);
       free(mirrored[i][0]);
       free(mirrored[i][1]);
       free(mirrored[i]);
      }

    free(mirrored);
   }

 chdir("..");

 /* Sort the files. */

 if(mode==MTime || mode==ATime || mode==Dated)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_time);
 else if(mode==Alpha || mode==Type)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_alpha);

 /* Output the page. */

 write_formatted(fd,"<h2>%s://</h2>\n",proto);

 if(nfiles)
   {
    int lastdays=0,lasthours=0;

    write_formatted(fd,"<b>%d Hosts</b>\n",nfiles);
    write_string(fd,"<ul>\n");

    for(i=0;i<nfiles;i++)
      {
       char *url=(char*)malloc(strlen(proto)+strlen(files[i]->name)+8);
       URL *Url;

       strcpy(url,proto);
       strcat(url,"://");
       strcat(url,files[i]->name);
       Url=SplitURL(url);

       if(mode==Dated)
          dated_separator(fd,i,&lastdays,&lasthours);

       if(ExistsWebpageSpoolFile(Url))
          write_formatted(fd,"<li>[<a href=\"/control/delete-url-all/?%s\">Del All</a>]"
                             "&nbsp;<a href=\"/index/%s/%s?%s\">%s</a>"
                             "&nbsp;[<a href=\"%s\">/</a>]\n",
                          url,
                          proto,files[i]->name,sorttype[mode][0],files[i]->name,
                          Url->link);
       else
          write_formatted(fd,"<li>[<a href=\"/control/delete-url-all/?%s\">Del All</a>]"
                             "&nbsp;<a href=\"/index/%s/%s?%s\">%s</a>\n",
                          url,
                          proto,files[i]->name,sorttype[mode][0],files[i]->name);

       free(files[i]->name);
       free(files[i]);
       FreeURL(Url);
      }

    write_string(fd,"</ul>\n");
   }
 else
    write_string(fd,"<b>No hosts</b>\n");

 if(files)
    free(files);
 files=NULL;
 nfiles=0;
}


/*++++++++++++++++++++++++++++++++++++++
  Create an index of the pages on a host.

  int fd The file descriptor to write to.

  char *proto The protocol to index.

  char *host The name of the subdirectory.

  SortMode mode The sort mode to use.
  ++++++++++++++++++++++++++++++++++++++*/

static void IndexHost(int fd,char *proto,char *host,SortMode mode)
{
 DIR *dir;
 struct dirent* ent;
 int i;
 char *link_proto=NULL,*link_host=NULL;

 /* Write the header. */

 write_formatted(fd,"<h2>%s://%s/</h2>\n",proto,host);

 if(IsMirrored(proto,host,&link_proto,&link_host))
   {
    write_formatted(fd,"Mirror Link =&gt; %s://%s/\n<p>\n",link_proto,link_host);
    proto=link_proto;
    host=link_host;
   }

 /* Open the spool subdirectory. */

 if(chdir(proto))
   {PrintMessage(Warning,"Cannot change to directory '%s' [%!s]; not indexed.",proto);
   write_formatted(fd,"No %s directory.",proto);return;}

 /* Open the spool subdirectory. */

 if(chdir(host))
   {PrintMessage(Warning,"Cannot change to directory '%s/%s' [%!s]; not indexed.",proto,host);chdir("..");
   write_formatted(fd,"No %s/%s directory.",proto,host);return;}

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open directory '%s/%s' [%!s]; not indexed.",proto,host);chdir("../..");
   write_formatted(fd,"Unable to open directory %s/%s.",proto,host);return;}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read directory '%s/%s' [%!s]; not indexed.",proto,host);closedir(dir);chdir("../..");
   write_formatted(fd,"Unable to read directory %s/%s.",proto,host);return;}
 ent=readdir(dir);  /* skip .. */

 /* Add all of the file names. */

 while((ent=readdir(dir)))
    add_file(ent->d_name,mode);

 closedir(dir);

 chdir("../..");

 /* Sort the files. */

 if(mode==MTime || mode==ATime || mode==Dated)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_time);
 else if(mode==Alpha)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_alpha);
 else if(mode==Type)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_type);

 /* Output the page. */

 if(nfiles)
   {
    int lastdays=0,lasthours=0;

    write_formatted(fd,"<b>%d Pages</b>\n",nfiles);
    write_string(fd,"<ul>\n");

    for(i=0;i<nfiles;i++)
      {
       URL *Url=SplitURL(files[i]->name);
       char *decurl=URLDecode(Url->pathp,0);

       if(mode==Dated)
          dated_separator(fd,i,&lastdays,&lasthours);

       if(Url->Protocol && Url->args && *Url->args=='!')
          write_formatted(fd,"<li>[<a href=\"/control/delete-url/?%s\">Del</a>"
                             "|Refresh:"
                             "Opt|"
                             "Mon]"
                             "&nbsp;<a href=\"%s\">%s</a>\n",
                          Url->name,
                          Url->link,decurl);
       else if(Url->Protocol)
          write_formatted(fd,"<li>[<a href=\"/control/delete-url/?%s\">Del</a>"
                             "|<a href=\"/refresh/?%s\">Refresh</a>:"
                             "<a href=\"/refresh-options/?%s\">Opt</a>|"
                             "<a href=\"/monitor-options/?%s\">Mon</a>]"
                             "&nbsp;<a href=\"%s\">%s</a>\n",
                          Url->name,
                          Url->name,
                          Url->name,
                          Url->name,
                          Url->link,decurl);
       else
          write_formatted(fd,"<li>[Del"
                             "|Refresh:"
                             "Opt|"
                             "Mon]"
                             "&nbsp;%s\n",
                          decurl);

       free(files[i]->name);
       free(files[i]);
       FreeURL(Url);
       free(decurl);
      }

    write_string(fd,"</ul>\n");

    write_formatted(fd,"[<a href=\"/control/delete-url-all/?%s://%s/\">Delete ALL Pages</a>]\n",proto,host);
   }
 else
   {
    write_string(fd,"<b>No Pages</b>\n");
    write_string(fd,"<p>\n");
    write_formatted(fd,"[<a href=\"/control/delete-url-all/?%s://%s/\">Delete This Host</a>]\n",proto,host);
   }

 if(files)
    free(files);
 files=NULL;
 nfiles=0;
}


/*++++++++++++++++++++++++++++++++++++++
  Create an index of the requests that are waiting in the outgoing directory.

  int fd The file descriptor to write to.

  SortMode mode The method to use to sort the names.
  ++++++++++++++++++++++++++++++++++++++*/

static void IndexOutgoing(int fd,SortMode mode)
{
 DIR *dir;
 struct dirent* ent;
 int i;

 /* Open the outgoing subdirectory. */

 if(chdir("outgoing"))
   {PrintMessage(Warning,"Cannot change to directory 'outgoing' [%!s]; not indexed.");
   write_formatted(fd,"No outgoing directory.");return;}

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open directory 'outgoing' [%!s]; not indexed.");chdir("..");
   write_formatted(fd,"Unable to open outgoing directory.");return;}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read directory 'outgoing' [%!s]; not indexed.");closedir(dir);chdir("..");
   write_formatted(fd,"Unable to read outgoing directory.");return;}
 ent=readdir(dir);  /* skip .. */

 /* Add all of the file names. */

 while((ent=readdir(dir)))
    add_file(ent->d_name,mode);

 closedir(dir);

 chdir("..");

 /* Sort the files. */

 if(mode==MTime || mode==ATime || mode==Dated)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_time);
 else if(mode==Alpha)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_alpha);
 else if(mode==Type)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_type);

 /* Output the page. */

 write_string(fd,"<h2>Outgoing Requests</h2>\n");

 if(nfiles)
   {
    int lastdays=0,lasthours=0;

    write_formatted(fd,"<b>%d Requests</b>\n",nfiles);
    write_string(fd,"<ul>\n");

    for(i=0;i<nfiles;i++)
      {
       URL *Url=SplitURL(files[i]->name);
       char *decurl=URLDecode(Url->name,0);

       if(mode==Dated)
          dated_separator(fd,i,&lastdays,&lasthours);

       if(Url->Protocol && Url->args && *Url->args=='!')
          write_formatted(fd,"<li>[<a href=\"/control/delete-req/?%s\">Del</a>"
                             "|Refresh:"
                             "Opt|"
                             "Mon]"
                             "&nbsp;<a href=\"%s\">%s</a>\n",
                          Url->name,
                          Url->link,decurl);
       else if(Url->Protocol)
          write_formatted(fd,"<li>[<a href=\"/control/delete-req/?%s\">Del</a>"
                             "|Refresh:"
                             "<a href=\"/refresh-options/?%s\">Opt</a>|"
                             "<a href=\"/monitor-options/?%s\">Mon</a>]"
                             "&nbsp;<a href=\"%s\">%s</a>\n",
                          Url->name,
                          Url->name,
                          Url->name,
                          Url->link,decurl);
       else
          write_formatted(fd,"<li>[<a href=\"/control/delete-req/?%s\">Del</a>"
                             "|Refresh:"
                             "Opt|"
                             "Mon]"
                             "&nbsp;%s\n",
                          Url->name,
                          decurl);

       free(files[i]->name);
       free(files[i]);
       FreeURL(Url);
       free(decurl);
      }

    write_string(fd,"</ul>\n");

    write_string(fd,"[<a href=\"/control/delete-req-all/\">Delete ALL Requests</a>]\n");
   }
 else
    write_string(fd,"<b>No Requests</b>\n");

 write_string(fd,"<p align=center>\n");
 write_string(fd,"[<a href=\"/refresh-options/\">Specify a Page to Be Requested</a>]\n");
 write_string(fd,"</p>\n");

 if(files)
    free(files);
 files=NULL;
 nfiles=0;
}


/*++++++++++++++++++++++++++++++++++++++
  Create an index of the requests that are in the monitor directory.

  int fd The file descriptor to write to.

  SortMode mode The method to use to sort the names.
  ++++++++++++++++++++++++++++++++++++++*/

static void IndexMonitor(int fd,SortMode mode)
{
 DIR *dir;
 struct dirent* ent;
 int i;

 /* Open the monitor subdirectory. */

 if(chdir("monitor"))
   {PrintMessage(Warning,"Cannot change to directory 'monitor' [%!s]; not indexed.");
   write_formatted(fd,"No monitor directory.");return;}

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open directory 'monitor' [%!s]; not indexed.");chdir("..");
   write_formatted(fd,"Unable to open monitor directory.");return;}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read directory 'monitor' [%!s]; not indexed.");closedir(dir);chdir("..");
   write_formatted(fd,"Unable to read monitor directory.");return;}
 ent=readdir(dir);  /* skip .. */

 /* Add all of the file names. */

 while((ent=readdir(dir)))
    add_file(ent->d_name,mode);

 closedir(dir);

 /* Sort the files. */

 if(mode==MTime || mode==ATime || mode==Dated)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_time);
 else if(mode==Alpha)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_alpha);
 else if(mode==Type)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_type);

 /* Output the page. */

 write_string(fd,"<h2>Monitored Pages</h2>\n");
 write_string(fd,"<i>I=Interval, L=Last, N=Next Time; all in days</i>\n");
 write_string(fd,"<p>\n");

 if(nfiles)
   {
    int lastdays=0,lasthours=0;

    write_formatted(fd,"<b>%d Pages</b>\n",nfiles);
    write_string(fd,"<ul>\n");

    for(i=0;i<nfiles;i++)
      {
       URL *Url=SplitURL(files[i]->name);
       char *decurl=URLDecode(Url->name,0);
       int interval,last,next;
       char *file=URLToFileName(Url);

       MonitorTimes(file,&interval,&last,&next);

       if(mode==Dated)
          dated_separator(fd,i,&lastdays,&lasthours);

       if(Url->Protocol && Url->args && *Url->args=='!')
          write_formatted(fd,"<li>[I=%02d:L=%02d:N=%02d]&nbsp;"
                             "[<a href=\"/control/delete-mon/?%s\">Del</a>"
                             "|Refresh:"
                             "Opt|"
                             "Mon]"
                             "&nbsp;<a href=\"%s\">%s</a>\n",
                          interval,last,next,
                          Url->name,
                          Url->link,decurl);
       else if(Url->Protocol)
          write_formatted(fd,"<li>[I=%02d:L=%02d:N=%02d]&nbsp;"
                             "[<a href=\"/control/delete-mon/?%s\">Del</a>"
                             "|<a href=\"/refresh/?%s\">Refresh</a>:"
                             "<a href=\"/refresh-options/?%s\">Opt</a>|"
                             "<a href=\"/monitor-options/?%s\">Mon</a>]"
                             "&nbsp;<a href=\"%s\">%s</a>\n",
                          interval,last,next,
                          Url->name,
                          Url->name,
                          Url->name,
                          Url->name,
                          Url->link,decurl);
       else
          write_formatted(fd,"<li>[I=%02d:L=%02d:N=%02d]&nbsp;"
                             "[<a href=\"/control/delete-mon/?%s\">Del</a>"
                             "|Refresh:"
                             "Opt|"
                             "Mon]"
                             "&nbsp;%s\n",
                          interval,last,next,
                          Url->name,
                          decurl);

       free(file);
       free(files[i]->name);
       free(files[i]);
       FreeURL(Url);
       free(decurl);
      }

    write_string(fd,"</ul>\n");

    write_string(fd,"[<a href=\"/control/delete-mon-all/\">Delete ALL Monitored pages</a>]\n");
   }
 else
    write_string(fd,"<b>No Pages</b>\n");

 write_string(fd,"<p align=center>\n");
 write_string(fd,"[<a href=\"/monitor-options/\">Specify a Page to Be Monitored</a>]\n");
 write_string(fd,"</p>\n");

 chdir("..");

 if(files)
    free(files);
 files=NULL;
 nfiles=0;
}


/*++++++++++++++++++++++++++++++++++++++
  Create an index of the pages that were got last time online.

  int fd The file descriptor to write to.

  SortMode mode The method to use to sort the names.

  char *name The name of the directory.
  ++++++++++++++++++++++++++++++++++++++*/

static void IndexLastTime(int fd,SortMode mode,char *name)
{
 DIR *dir;
 struct dirent* ent;
 int i;
 int num=atoi(name+8);

 /* Open the lasttime subdirectory. */

 if(chdir(name))
   {PrintMessage(Warning,"Cannot change to directory '%s' [%!s]; not indexed.",name);
   write_formatted(fd,"No %s directory.",name);return;}

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open directory '%s' [%!s]; not indexed.",name);chdir("..");
   write_formatted(fd,"Unable to open %s directory.",name);return;}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read directory '%s' [%!s]; not indexed.",name);closedir(dir);chdir("..");
   write_formatted(fd,"Unable to read %s directory.",name);return;}
 ent=readdir(dir);  /* skip .. */

 /* Add all of the file names. */

 while((ent=readdir(dir)))
    add_file(ent->d_name,mode);

 closedir(dir);

 chdir("..");

 /* Sort the files. */

 if(mode==MTime || mode==ATime || mode==Dated)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_time);
 else if(mode==Alpha)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_alpha);
 else if(mode==Type)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_type);

 /* Output the page. */

 if(strcmp(name,"lasttime"))
    write_formatted(fd,"<h2>Previous Time Online Pages (%d)</h2>\n",num);
 else
    write_string(fd,"<h2>Last Time Online Pages</h2>\n");

 if(nfiles)
   {
    int lastdays=0,lasthours=0;

    write_formatted(fd,"<b>%d Pages</b>\n",nfiles);
    write_string(fd,"<ul>\n");

    for(i=0;i<nfiles;i++)
      {
       URL *Url=SplitURL(files[i]->name);
       char *decurl=URLDecode(Url->name,0);

       if(mode==Dated)
          dated_separator(fd,i,&lastdays,&lasthours);

       if(Url->Protocol && Url->args && *Url->args=='!')
          write_formatted(fd,"<li>[<a href=\"/control/delete-url/?%s\">Del</a>"
                             "|Refresh:"
                             "Opt|"
                             "Mon]"
                             "&nbsp;<a href=\"%s\">%s</a>\n",
                          Url->name,
                          Url->link,decurl);
       else if(Url->Protocol)
          write_formatted(fd,"<li>[<a href=\"/control/delete-url/?%s\">Del</a>"
                             "|<a href=\"/refresh/?%s\">Refresh</a>:"
                             "<a href=\"/refresh-options/?%s\">Opt</a>|"
                             "<a href=\"/monitor-options/?%s\">Mon</a>]"
                             "&nbsp;<a href=\"%s\">%s</a>\n",
                          Url->name,
                          Url->name,
                          Url->name,
                          Url->name,
                          Url->link,decurl);
       else
          write_formatted(fd,"<li>[<a href=\"/control/delete-url/?%s\">Del</a>"
                             "|Refresh:"
                             "Opt|"
                             "Mon]"
                             "&nbsp;%s\n",
                          Url->name,
                          decurl);

       free(files[i]->name);
       free(files[i]);
       FreeURL(Url);
       free(decurl);
      }

    write_string(fd,"</ul>\n");
   }
 else
    write_string(fd,"<b>No Pages</b>\n");

 if(num<NUM_PREVTIME_DIR)
   {
    write_string(fd,"<p align=center>\n");
    write_formatted(fd,"[<a href=\"/index/prevtime%d/?%s\">Previous Time (%d)</a>]\n",num+1,sorttype[mode][0],num+1);
    write_string(fd,"</p>\n");
   }

 if(files)
    free(files);
 files=NULL;
 nfiles=0;
}


/*++++++++++++++++++++++++++++++++++++++
  Create an index of the latest pages.

  int fd The file descriptor to write to.

  SortMode mode The method to use to sort the names.
  ++++++++++++++++++++++++++++++++++++++*/

static void IndexLatest(int fd,SortMode mode)
{
 int i,p;

 earliest=now-IndexLatestDays*24*3600;

 /* Add all of the files in the sub directories. */

 for(p=0;p<NProtocols;p++)
   {
    struct stat buf;

    if(!lstat(Protocols[p].name,&buf) && S_ISDIR(buf.st_mode))
      {
       DIR *dir2;
       struct dirent* ent2;

       if(chdir(Protocols[p].name))
         {PrintMessage(Warning,"Cannot change to directory '%s' [%!s]; not indexed.",Protocols[p].name);continue;}

       /* Open the spool directory. */

       dir2=opendir(".");
       if(!dir2)
         {PrintMessage(Warning,"Cannot open directory '%s' [%!s]; not indexed.",Protocols[p].name);chdir("..");continue;}

       ent2=readdir(dir2);  /* skip .  */
       if(!ent2)
         {PrintMessage(Warning,"Cannot read directory '%s' [%!s]; index failed.",Protocols[p].name);closedir(dir2);chdir("..");continue;}
       ent2=readdir(dir2);  /* skip .. */

       /* Print all of the sub directories. */

       while((ent2=readdir(dir2)))
         {
          struct stat buf2;

          if(lstat(ent2->d_name,&buf2))
             PrintMessage(Inform,"Cannot stat file '%s' [%!s]; race condition?",ent2->d_name);
          else
             if(S_ISDIR(buf2.st_mode) && buf2.st_mtime>earliest)
               {
                DIR *dir3;
                struct dirent* ent3;
                struct utimbuf utbuf;

                if(chdir(ent2->d_name))
                  {PrintMessage(Warning,"Cannot change to directory '%s/%s' [%!s]; not indexed.",Protocols[p].name,ent2->d_name);continue;}

                dir3=opendir(".");
                if(!dir3)
                  {PrintMessage(Warning,"Cannot open directory '%s/%s' [%!s]; not indexed.",Protocols[p].name,ent2->d_name);chdir("..");continue;}

                ent3=readdir(dir3);  /* skip .  */
                if(!ent3)
                  {PrintMessage(Warning,"Cannot read directory '%s/%s' [%!s]; not indexed.",Protocols[p].name,ent2->d_name);closedir(dir3);chdir("..");continue;}
                ent3=readdir(dir3);  /* skip .. */

                while((ent3=readdir(dir3)))
                   add_file(ent3->d_name,mode);

                closedir(dir3);
                chdir("..");

                utbuf.actime=buf2.st_atime;
                utbuf.modtime=buf2.st_mtime;
                utime(ent2->d_name,&utbuf);
               }
         }

       closedir(dir2);
       chdir("..");
      }
   }

 /* Sort the files. */

 if(mode==MTime || mode==ATime || mode==Dated)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_time);
 else if(mode==Alpha)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_alpha);
 else if(mode==Type)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_type);

 /* Output the page. */

 write_formatted(fd,"<h2>Latest Pages</h2>\n");
 write_formatted(fd,"From the last %d Days\n<p>\n",IndexLatestDays);

 if(nfiles)
   {
    int lastdays=0,lasthours=0;

    write_formatted(fd,"<b>%d Pages</b>\n",nfiles);
    write_string(fd,"<ul>\n");

    for(i=0;i<nfiles;i++)
      {
       URL *Url=SplitURL(files[i]->name);
       char *decurl=URLDecode(Url->name,0);

       if(mode==Dated)
          dated_separator(fd,i,&lastdays,&lasthours);

       if(Url->Protocol && Url->args && *Url->args=='!')
          write_formatted(fd,"<li>[<a href=\"/control/delete-url/?%s\">Del</a>"
                             "|Refresh:"
                             "Opt|"
                             "Mon]"
                             "&nbsp;<a href=\"%s\">%s</a>\n",
                          Url->name,
                          Url->link,decurl);
       else if(Url->Protocol)
          write_formatted(fd,"<li>[<a href=\"/control/delete-url/?%s\">Del</a>"
                             "|<a href=\"/refresh/?%s\">Refresh</a>:"
                             "<a href=\"/refresh-options/?%s\">Opt</a>|"
                             "<a href=\"/monitor-options/?%s\">Mon</a>]"
                             "&nbsp;<a href=\"%s\">%s</a>\n",
                          Url->name,
                          Url->name,
                          Url->name,
                          Url->name,
                          Url->link,decurl);
       else
          write_formatted(fd,"<li>[<a href=\"/control/delete-url/?%s\">Del</a>"
                             "|Refresh:"
                             "Opt|"
                             "Mon]"
                             "&nbsp;%s\n",
                          Url->name,
                          decurl);

       free(files[i]->name);
       free(files[i]);
       FreeURL(Url);
       free(decurl);
      }

    write_string(fd,"</ul>\n");
   }
 else
    write_string(fd,"<b>No Pages</b>\n");

 if(files)
    free(files);
 files=NULL;
 nfiles=0;
}


/*++++++++++++++++++++++++++++++++++++++
  Add a directory to the list.

  char *name The name of the directory.

  SortMode mode The sort mode.

  char *link The directory that this is linked to if this is a mirror link.
  ++++++++++++++++++++++++++++++++++++++*/

static void add_dir(char *name,SortMode mode,char *link)
{
 struct stat buf;

 if(!link && lstat(name,&buf))
   {PrintMessage(Inform,"Cannot stat directory '%s' [%!s]; race condition?",name);return;}
 else if(link && stat(link,&buf))
   {PrintMessage(Inform,"Cannot stat link '%s' [%!s]; points nowhere?",link);return;}
 else if(S_ISDIR(buf.st_mode))
   {
    if(!(nfiles%16))
      {
       if(!files)
          files=(FileIndex**)malloc(16*sizeof(FileIndex*));
       else
          files=(FileIndex**)realloc(files,(nfiles+16)*sizeof(FileIndex*));
      }
    files[nfiles]=(FileIndex*)malloc(sizeof(FileIndex));

    files[nfiles]->name=(char*)malloc(strlen(name)+1);
    strcpy(files[nfiles]->name,name);

    files[nfiles]->type="";

    if(mode==MTime || mode==Dated)
       files[nfiles]->time=buf.st_mtime;
    else if(mode==ATime)
       files[nfiles]->time=buf.st_atime;

    nfiles++;
   }
}


/*++++++++++++++++++++++++++++++++++++++
  Add a file to the list.

  char *name The name of the file.

  SortMode mode The sort mode.
  ++++++++++++++++++++++++++++++++++++++*/

static void add_file(char *name,SortMode mode)
{
 char *url=NULL;
 struct stat buf;

 url=FileNameToURL(name);

 if(!url)
    return;

 if(stat(name,&buf))
   {PrintMessage(Inform,"Cannot stat file '%s' [%!s]; race condition?",name);return;}
 else if(S_ISREG(buf.st_mode) && (mode!=Dated || buf.st_mtime>earliest))
   {
    if(!(nfiles%16))
      {
       if(!files)
          files=(FileIndex**)malloc(16*sizeof(FileIndex*));
       else
          files=(FileIndex**)realloc(files,(nfiles+16)*sizeof(FileIndex*));
      }
    files[nfiles]=(FileIndex*)malloc(sizeof(FileIndex));

    files[nfiles]->name=url;

    if(mode==Type)
      {
       char *p;

       files[nfiles]->type="";

       p=strchr(files[nfiles]->name,'?');
       if(!p)
          p=files[nfiles]->name+strlen(files[nfiles]->name);

       for(;p>url;p--)
          if(*p=='.')
            {files[nfiles]->type=p;break;}
          else if(*p=='/')
             break;
      }
    else if(mode==MTime || mode==Dated)
       files[nfiles]->time=buf.st_mtime;
    else if(mode==ATime)
       files[nfiles]->time=buf.st_atime;

    nfiles++;
   }
}


/*++++++++++++++++++++++++++++++++++++++
  Write out a date separator for the dated format if needed.

  int fd The file to write to.

  int file The number of the file in the list.

  int *lastdays The age of the previous file in days.

  int *lasthours The age of the previous file in hours.
  ++++++++++++++++++++++++++++++++++++++*/

static void dated_separator(int fd,int file,int *lastdays,int *lasthours)
{
 int days=(now-files[file]->time)/(24*3600),hours=(now-files[file]->time)/3600;

 if(*lastdays<days)
   {
    write_string(fd,"</ul>\n");
    for(;*lastdays<days;(*lastdays)++)
       write_string(fd,"<hr width=\"25%\">\n");
    write_string(fd,"<ul>\n");
   }
 else if(file && *lasthours<(hours-1))
   {
    write_string(fd,"</ul>\n");
    write_string(fd,"<p>\n");
    write_string(fd,"<ul>\n");
   }
 *lasthours=hours;
}

/*++++++++++++++++++++++++++++++++++++++
  Used to sort the files into alphabetical order.

  int sort_alpha Returns the comparison of the pointers to strings.

  FileIndex **a The first FileIndex.

  FileIndex **b The second FileIndex.
  ++++++++++++++++++++++++++++++++++++++*/

static int sort_alpha(FileIndex **a,FileIndex **b)
{
 char *an=(*a)->name;
 char *bn=(*b)->name;

 return(strcmp(an,bn));
}


/*++++++++++++++++++++++++++++++++++++++
  Used to sort the files into type (file extension) order.

  int sort_type Returns the comparison of the pointers to strings.

  FileIndex **a The first FileIndex.

  FileIndex **b The second FileIndex.
  ++++++++++++++++++++++++++++++++++++++*/

static int sort_type(FileIndex **a,FileIndex **b)
{
 char *an=(*a)->name,*at=(*a)->type;
 char *bn=(*b)->name,*bt=(*b)->type;
 int sort1=strcmp(at,bt);

 if(sort1==0)
    return(strcmp(an,bn));
 else
    return(sort1);
}


/*++++++++++++++++++++++++++++++++++++++
  Used to sort the files into alphabetical order.

  int sort_time Returns the comparison of the times.

  FileIndex **a The first FileIndex.

  FileIndex **b The second FileIndex.
  ++++++++++++++++++++++++++++++++++++++*/

static int sort_time(FileIndex **a,FileIndex **b)
{
 long at=(*a)->time;
 long bt=(*b)->time;

 return(bt-at);
}
