/***************************************
  $Header: /home/amb/wwwoffle/RCS/control.c 2.28 1998/06/18 19:43:55 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 2.2a.
  The HTML interactive control pages.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1997,98 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <unistd.h>

#include "wwwoffle.h"
#include "misc.h"
#include "config.h"
#include "sockets.h"
#include "errors.h"


/*+ The action to perform. +*/
typedef enum _Action
{
 None,                          /*+ Undecided. +*/
 Online,                        /*+ Tell the server that we are online. +*/
 Autodial,                      /*+ Tell the server that we are in autodial mode. +*/
 Offline,                       /*+ Tell the server that we are offline. +*/
 Fetch,                         /*+ Tell the server to fetch the requested pages. +*/
 Config,                        /*+ Tell the server to re-read the configuration file. +*/
 Purge,                         /*+ Tell the server to purge pages. +*/
 Delete,                        /*+ Delete a page from the cache or a request from the outgoing directory. +*/
 ConfigEdit,                    /*+ Edit the config file. +*/
}
Action;


static void ActionControlPage(int fd,Action action,char *command);
static void DeleteControlPage(int fd,char *mode,char *args);
static void ControlAuthFail(int fd,char *url);
static int MatchPassword(char *try,char *actual);


/*++++++++++++++++++++++++++++++++++++++
  Send to the client one of the pages to control WWWOFFLE using HTML.

  int fd The file descriptor of the client.

  URL *Url The Url that was requested.

  char *request_head The head of the HTTP request.

  char *request_body The body of the HTTP request.
  ++++++++++++++++++++++++++++++++++++++*/

void ControlPage(int fd,URL *Url,char *request_head,char *request_body)
{
 Action action=None;
 char *newpath=(char*)malloc(strlen(Url->path)-8);
 char *command="";

 strcpy(newpath,Url->path+9);        /* remove the '/control/' */

 if(*newpath && newpath[strlen(newpath)-1]=='/')
    newpath[strlen(newpath)-1]=0;

 /* Determine the action. */

 if(!strcmp(newpath,"online"))
   {action=Online;command="-online";}
 else if(!strcmp(newpath,"autodial"))
   {action=Autodial;command="-autodial";}
 else if(!strcmp(newpath,"offline"))
   {action=Offline;command="-offline";}
 else if(!strcmp(newpath,"fetch"))
   {action=Fetch;command="-fetch";}
 else if(!strcmp(newpath,"config"))
   {action=Config;command="-config";}
 else if(!strcmp(newpath,"purge"))
   {action=Purge;command="-purge";}
 else if(!strncmp(newpath,"delete",6))
    action=Delete;
 else if(!strcmp(newpath,"edit"))
    action=ConfigEdit;

 /* Check the authorisation. */

 if(PassWord)
   {
    char *auth=GetHTTPHeader(request_head,"Authorization:");

    if(!auth && action==Delete && Url->args && strstr(newpath,"password="))
      {
       URL *reqUrl=SplitURL(Url->args);
       char *hash=HashOutgoingSpoolFile(reqUrl);
       char *pswd=strstr(newpath,"password=")+9;

       FreeURL(reqUrl);

       if(!hash || strcmp(pswd,hash))
         {
          PrintMessage(Important,"Interactive control webpage delete with password failed.");
          ControlAuthFail(fd,Url->pathp);
          return;
         }
      }
    else if(!auth)
      {
       ControlAuthFail(fd,Url->pathp);
       return;
      }
    else
      {
       char *eol,*type,*pswd;
       int n;

       auth+=15;
       eol=strchr(auth,'\n');
       type=(char*)malloc(eol-auth);
       pswd=(char*)malloc(eol-auth);
       *eol=0;
       n=sscanf(auth,"%s %s",type,pswd);
       *eol='\n';

       if(n!=2 || strcasecmp(type,"Basic") || !MatchPassword(pswd,PassWord))
         {
          PrintMessage(Important,"Interactive control webpage authorisation failed.");
          ControlAuthFail(fd,Url->pathp);
          return;
         }
      }
   }

 /* Preform the action. */

 if(action==None && Url->path[9])
   {
    HTMLMessage(fd,404,"WWWOFFLE Illegal Control Page",NULL,"ControlIllegal",
                "url",Url->pathp,
                NULL);
    return;
   }

 if(action==None)
    HTMLMessage(fd,200,"WWWOFFLE Control Page",NULL,"ControlPage",
                NULL);
 else if(action==Delete)
    DeleteControlPage(fd,newpath,Url->args);
 else if(action==ConfigEdit)
    ConfigEditPage(fd,Url->args,request_body);
 else
    ActionControlPage(fd,action,command);

 free(newpath);
}


/*++++++++++++++++++++++++++++++++++++++
  The control page that performs an action.

  int fd The file descriptor to write to.

  Action action The action to perform.

  char *command The command line argument that would be used with wwwoffle.
  ++++++++++++++++++++++++++++++++++++++*/

static void ActionControlPage(int fd,Action action,char *command)
{
 int socket=OpenClientSocket("localhost",WWWOFFLE_Port);
 init_buffer(socket);

 if(socket==-1)
   {
    PrintMessage(Warning,"Cannot open connection to wwwoffle server localhost port %d.",WWWOFFLE_Port);
    HTMLMessage(fd,500,"WWWOFFLE Server Error",NULL,"ServerError",
                "error","Cannot open connection to wwwoffle server on localhost",
                NULL);
   }
 else
   {
    char *buffer=NULL;

    HTMLMessage(fd,200,"WWWOFFLE Control Page",NULL,"ControlWWWOFFLE-Head",
                "command",command,
                NULL);

    /* Send the message. */

    if(PassWord)
       write_formatted(socket,"WWWOFFLE PASSWORD %s\r\n",PassWord);

    if(action==Online)
       write_string(socket,"WWWOFFLE ONLINE\r\n");
    else if(action==Autodial)
       write_string(socket,"WWWOFFLE AUTODIAL\r\n");
    else if(action==Offline)
       write_string(socket,"WWWOFFLE OFFLINE\r\n");
    else if(action==Fetch)
       write_string(socket,"WWWOFFLE FETCH\r\n");
    else if(action==Config)
       write_string(socket,"WWWOFFLE CONFIG\r\n");
    else if(action==Purge)
       write_string(socket,"WWWOFFLE PURGE\r\n");

    while((buffer=read_line(socket,buffer)))
       write_string(fd,buffer);

    HTMLMessageBody(fd,"ControlWWWOFFLE-Tail",
                    NULL);
   }
}


/*++++++++++++++++++++++++++++++++++++++
  The control page that deletes a cached page or a request.

  int fd The file descriptor to write to.

  char *mode The mode of deletion that was requested.

  char *args The arguments specified.
  ++++++++++++++++++++++++++++++++++++++*/

static void DeleteControlPage(int fd,char *path,char *args)
{
 char *url=NULL,*req=NULL,*mon=NULL;
 int all=0;

 /* Decide what sort of deletion is required. */

 if(!strncmp(path+10,"-all",4))
    all=1;

 if(!strncmp(path,"delete-url",10))
    url=args;
 else if(!strncmp(path,"delete-mon",10))
   {
    mon=args;

    if(all)
       mon="ALL";
   }
 else if(!strncmp(path,"delete-req",10))
   {
    req=args;

    if(all)
       req="ALL";
   }

 /* Do the required deletion. */

 if(!url && !mon && !req)
   {
    char *controlurl=(char*)malloc(strlen(path)+strlen(args)+24);
    sprintf(controlurl,"/control/%s?%s",path,args);
    PrintMessage(Important,"Invalid interactive delete page requested; path='%s' args='%s'.",path,args);
    HTMLMessage(fd,404,"WWWOFFLE Illegal Control Page",NULL,"ControlIllegal",
                "url",controlurl,
                NULL);
    free(controlurl);
   }
 else if(req && all)
   {
    char *err=DeleteOutgoingSpoolFile(NULL);

    HTMLMessage(fd,err?404:200,"WWWOFFLE Delete Control Page",NULL,"ControlDelete",
                "req",req,
                "all","yes",
                "error",err?err:"",
                NULL);
   }
 else if(req)
   {
    URL *reqUrl=SplitURL(req);

    if(reqUrl->Protocol)
      {
       char *err=DeleteOutgoingSpoolFile(reqUrl);

       HTMLMessage(fd,err?404:200,"WWWOFFLE Delete Control Page",NULL,"ControlDelete",
                   "req",req,
                   "error",err?err:"",
                   NULL);
      }
    else
      {
       char *controlurl=(char*)malloc(16+strlen(path)+strlen(args));
       sprintf(controlurl,"/control/%s?%s",path,args);
       HTMLMessage(fd,404,"WWWOFFLE Illegal Protocol",NULL,"IllegalProtocol",
                   "url",controlurl,
                   "protocol",reqUrl->proto,
                   NULL);
       free(controlurl);
      }

    FreeURL(reqUrl);
   }
 else if(mon && all)
   {
    char *err=DeleteMonitorSpoolFile(NULL);

    HTMLMessage(fd,err?404:200,"WWWOFFLE Delete Control Page",NULL,"ControlDelete",
                "mon",mon,
                "all","yes",
                "error",err?err:"",
                NULL);
   }
 else if(mon)
   {
    URL *monUrl=SplitURL(mon);

    if(monUrl->Protocol)
      {
       char *err=DeleteMonitorSpoolFile(monUrl);

       HTMLMessage(fd,err?404:200,"WWWOFFLE Delete Control Page",NULL,"ControlDelete",
                   "mon",mon,
                   "error",err?err:"",
                   NULL);
      }
    else
      {
       char *controlurl=(char*)malloc(16+strlen(path)+strlen(args));
       sprintf(controlurl,"/control/%s?%s",path,args);
       HTMLMessage(fd,404,"WWWOFFLE Illegal Protocol",NULL,"IllegalProtocol",
                   "url",controlurl,
                   "protocol",monUrl->proto,
                   NULL);
       free(controlurl);
      }

    FreeURL(monUrl);
   }
 else if(url)
   {
    URL *urlUrl=SplitURL(url);

    if(urlUrl->Protocol)
      {
       char *err=DeleteWebpageSpoolFile(urlUrl,all);

       HTMLMessage(fd,err?404:200,"WWWOFFLE Delete Control Page",NULL,"ControlDelete",
                   "url",url,
                   "all",all?"yes":"",
                   "error",err?err:"",
                   NULL);
      }
    else
      {
       char *controlurl=(char*)malloc(16+strlen(path)+strlen(args));
       sprintf(controlurl,"/control/%s?%s",path,args);
       HTMLMessage(fd,404,"WWWOFFLE Illegal Protocol",NULL,"IllegalProtocol",
                   "url",controlurl,
                   "protocol",urlUrl->proto,
                   NULL);
       free(controlurl);
      }

    FreeURL(urlUrl);
   }
}


/*++++++++++++++++++++++++++++++++++++++
  Inform the user that the authorisation failed.

  int fd The file descriptor to write to.

  char *url The specified path.
  ++++++++++++++++++++++++++++++++++++++*/

static void ControlAuthFail(int fd,char *url)
{
 HTMLMessageHead(fd,401,"WWWOFFLE Authorisation Failed",
                 "WWW-Authenticate","Basic realm=\"control\"",
                 NULL);
 HTMLMessageBody(fd,"ControlAuthFail",
                 "url",url,
                 NULL);
}


/*++++++++++++++++++++++++++++++++++++++
  Try to match the authorisation password against the actual one.

  int MatchPassword Returns true if the password is OK.

  char *try The attempted password.

  char *actual The actual password.
  ++++++++++++++++++++++++++++++++++++++*/

static int MatchPassword(char *try,char *actual)
{
 int ok=0,l;
 char *decoded=Base64Decode(try,&l);
 char *colon;

 colon=strchr(decoded,':');

 if(colon && !strcmp(colon+1,actual))
    ok=1;

 free(decoded);

 return(ok);
}
