/***************************************
  $Header: /home/amb/wwwoffle/RCS/wwwoffle.c 2.10 1998/01/03 17:57:45 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 2.0c.
  A user level program to interact with the server.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1996,97,98 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "wwwoffle.h"
#include "misc.h"
#include "config.h"
#include "sockets.h"
#include "errors.h"


static void usage(int verbose);


/*+ The action to perform. +*/
typedef enum _Action
{
 None,                          /*+ Undecided. +*/
 Online,                        /*+ Tell the server that we are online. +*/
 Offline,                       /*+ Tell the server that we are offline. +*/
 Fetch,                         /*+ Tell the server to fetch the requested pages. +*/
 Config,                        /*+ Tell the server to re-read the configuration file. +*/
 Purge,                         /*+ Tell the server to purge pages. +*/
 Kill,                          /*+ Tell the server to exit. +*/
 Get,                           /*+ Tell the server to get pages. +*/
 Output                         /*+ Get a page from the server and output it. +*/
}
Action;


static void add_url_file(char *url_file);

/*+ The list of URLs or files. +*/
char **url_file_list=NULL;
/*+ The number of URLs or files. +*/
int n_url_file_list=0;


/*++++++++++++++++++++++++++++++++++++++
  The main program.
  ++++++++++++++++++++++++++++++++++++++*/

int main(int argc, char** argv)
{
 int i;
 int force=0,images=0,frames=0,recursive_depth=0,recursive_mode=0;

 Action action=None;

 char *host=NULL;
 int port=0;

 /* Parse the command line options */

 ConfigFile=NULL;

 if(argc==1)
    usage(0);

 for(i=1;i<argc;i++)
   {
    if(!strcmp(argv[i],"-h"))
       usage(1);

    if(!strcmp(argv[i],"-o"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Output;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-F"))
      {
       if(action!=None && action!=Get)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Get;
       if(force)
         {fprintf(stderr,"wwwoffle: Only one '-F' argument may be given.\n"); exit(1);}
       force=1;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-i"))
      {
       if(action!=None && action!=Get)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Get;
       if(images)
         {fprintf(stderr,"wwwoffle: Only one '-i' argument may be given.\n"); exit(1);}
       images=1;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-f"))
      {
       if(action!=None && action!=Get)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Get;
       if(frames)
         {fprintf(stderr,"wwwoffle: Only one '-f' argument may be given.\n"); exit(1);}
       frames=1;
       argv[i]=NULL;
       continue;
      }

    if(!strncmp(argv[i],"-R",2) || !strncmp(argv[i],"-r",2) || !strncmp(argv[i],"-d",2))
      {
       if(action!=None && action!=Get)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Get;

       if(recursive_depth)
         {fprintf(stderr,"wwwoffle: Only one '-d', '-r' or '-R' argument may be given.\n"); exit(1);}

       if(argv[i][1]=='d')
          recursive_mode=1;
       else if(argv[i][1]=='r')
          recursive_mode=2;
       else /* argv[i][1]=='R' */
          recursive_mode=3;

       if(argv[i][2])
          recursive_depth=atoi(&argv[i][2]);
       else
          recursive_depth=1;
       if(recursive_depth<=0)
         {fprintf(stderr,"wwwoffle: The '-%c' argument may only be followed by a positive integer.\n",argv[i][1]); exit(1);}

       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-p"))
      {
       char *colon;

       if(++i>=argc)
         {fprintf(stderr,"wwwoffle: The '-p' argument requires a hostname and optionally a port number.\n"); exit(1);}

       if(ConfigFile)
         {fprintf(stderr,"wwwoffle: The '-p' and '-c' options cannot be used together.\n"); exit(1);}

       if((colon=strchr(argv[i],':')))
         {
          *colon++=0;

          port=atoi(colon);

          if(port<0 || port>=65536)
            {fprintf(stderr,"wwwoffle: The port number %d '%s' is invalid.\n",port,argv[i]); exit(1);}
         }

       host=argv[i];

       argv[i-1]=NULL;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-c"))
      {
       if(++i>=argc)
         {fprintf(stderr,"wwwoffle: The '-c' argument requires a configuration file name.\n"); exit(1);}

       if(host)
         {fprintf(stderr,"wwwoffle: The '-p' and '-c' options cannot be used together.\n"); exit(1);}

       ConfigFile=argv[i];

       argv[i-1]=NULL;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-on") || !strcmp(argv[i],"-online"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Online;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-off") || !strcmp(argv[i],"-offline"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Offline;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-fetch"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Fetch;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-config"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Config;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-purge"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Purge;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-kill"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Kill;
       argv[i]=NULL;
       continue;
      }

    if(argv[i][0]=='-' && argv[i][1])
      {
       fprintf(stderr,"wwwoffle: Unknown option '%s'.\n\n",argv[i]);
       usage(0);
      }

    add_url_file(argv[i]);
   }

 if(action==None)
    action=Get;

 /* Initialise things. */

 InitErrorHandler("wwwoffle",0,1);

 if(ConfigFile)
   {
    if(ReadConfigFile(2))
       PrintMessage(Fatal,"Error in configuration file '%s'.",ConfigFile);

    host=GetLocalHost(0);
   }

 /* The connections to the WWWOFFLE server. */

 if(action!=Get && action!=Output)
   {
    int socket;
    char *buffer=NULL;

    socket=OpenClientSocket(host?host:"localhost",port?port:WWWOFFLE_Port);
    init_buffer(socket);

    if(socket==-1)
       PrintMessage(Fatal,"Cannot open connection to wwwoffle server %s port %d.",host?host:"localhost",port?port:WWWOFFLE_Port);

    /* Send the message. */

    if(PassWord)
       write_formatted(socket,"WWWOFFLE PASSWORD %s\r\n",PassWord);

    if(action==Online)
       write_string(socket,"WWWOFFLE ONLINE\r\n");
    else if(action==Offline)
       write_string(socket,"WWWOFFLE OFFLINE\r\n");
    else if(action==Fetch)
       write_string(socket,"WWWOFFLE FETCH\r\n");
    else if(action==Config)
       write_string(socket,"WWWOFFLE CONFIG\r\n");
    else if(action==Purge)
       write_string(socket,"WWWOFFLE PURGE\r\n");
    else if(action==Kill)
       write_string(socket,"WWWOFFLE KILL\r\n");
    else
       write_string(socket,"WWWOFFLE BOGUS\r\n");

    while((buffer=read_line(socket,buffer)))
      {
       fputs(buffer,stdout);
       fflush(stdout);
      }

    CloseSocket(socket);
   }

 /* The connections to the http proxy. */

 else if(action==Get)
   {
    URL *Url;
    struct stat buf;
    char refresh[32];

    strcpy(refresh,"refresh");
    if(images)
       strcat(refresh,"-images");
    if(frames)
       strcat(refresh,"-frames");
    if(force)
       strcat(refresh,"-force");
    if(recursive_depth)
      {
       if(recursive_mode==1)
          strcat(refresh,"-dir");
       else if(recursive_mode==2)
          strcat(refresh,"-host");
       else /* recursive_mode==3 */
          strcat(refresh,"-any");
       sprintf(&refresh[strlen(refresh)],"-%d",recursive_depth);
      }
    if(!images && !frames && !recursive_depth && !force)
       strcat(refresh,"-none");

    for(i=0;i<n_url_file_list;i++)
       if(strcmp(url_file_list[i],"-") && stat(url_file_list[i],&buf))
         {
          int socket;
          char *buffer=NULL;

          socket=OpenClientSocket(host?host:"localhost",port?port:HTTP_Port);
          init_buffer(socket);

          if(socket==-1)
             PrintMessage(Fatal,"Cannot open connection to wwwoffle server %s port %d.",host?host:"localhost",port?port:HTTP_Port);

          Url=SplitURL(url_file_list[i]);

          printf("Getting: %s\n",Url->name);

          write_formatted(socket,"GET /%s/%s/%s HTTP/1.0\r\n"
                                 "Accept: */*\r\n"
                                 "\r\n",
                          refresh,Url->proto,Url->hostp);

          while((buffer=read_line(socket,buffer)))
            {
             fputs(buffer,stdout);
             fflush(stdout);
            }

          CloseSocket(socket);

          FreeURL(Url);
         }
       else if(!strcmp(url_file_list[i],"-") || S_ISREG(buf.st_mode))
         {
          int file;
          char *buffer=(char*)malloc(strlen(url_file_list[i])+256);
          char **links;
          int j;

          if(strcmp(url_file_list[i],"-"))
            {
             file=open(url_file_list[i],O_RDONLY);
             if(file==-1)
               {PrintMessage(Warning,"Cannot open file '%s' for reading.",url_file_list[i]);continue;}
             init_buffer(file);

             printf("Reading: %s\n",url_file_list[i]);
            }
          else
            {
             file=fileno(stdin);

             printf("Reading: <stdin>\n");
            }

          strcpy(buffer,"file://localhost");
          if(*url_file_list[i]=='/')
             strcat(buffer,url_file_list[i]);
          else
            {
             char cwd[256];

             if(!getcwd(cwd,255))
             cwd[0]=0;

             strcat(buffer,cwd);
             strcat(buffer,"/");
             strcat(buffer,url_file_list[i]);
            }

          Url=SplitURL(buffer);
          ParseHTML(file,Url,1);

          if(images && (links=ListImages()))
             for(j=0;links[j];j++)
               {
                URL *linkUrl=SplitURL(links[j]);

                if(strcmp(linkUrl->proto,"file"))
                   add_url_file(linkUrl->name);

                FreeURL(linkUrl);
               }

          if((links=ListLinks()))
             for(j=0;links[j];j++)
               {
                URL *linkUrl=SplitURL(links[j]);

                if(strcmp(linkUrl->proto,"file"))
                   add_url_file(linkUrl->name);

                FreeURL(linkUrl);
               }

          if(frames && (links=ListFrames()))
             for(j=0;links[j];j++)
               {
                URL *linkUrl=SplitURL(links[j]);

                if(strcmp(linkUrl->proto,"file"))
                   add_url_file(linkUrl->name);

                FreeURL(linkUrl);
               }

          FreeURL(Url);
          if(file!=0)
             close(file);
         }
       else
          PrintMessage(Warning,"The file '%s' is not a regular file.",url_file_list[i]);
   }
 else /* action==Output */
   {
    URL *Url;
    int socket;
    char *buffer=NULL,buffer256[257];
    int nbytes;

    if(!n_url_file_list)
       PrintMessage(Fatal,"No URL specified to output");

    socket=OpenClientSocket(host?host:"localhost",port?port:HTTP_Port);
    init_buffer(socket);

    if(socket==-1)
       PrintMessage(Fatal,"Cannot open connection to wwwoffle server %s port %d.",host?host:"localhost",port?port:HTTP_Port);

    Url=SplitURL(url_file_list[0]);

    fprintf(stderr,"Getting: %s\n",Url->name);

    write_formatted(socket,"GET %s HTTP/1.0\r\n"
                           "Accept: */*\r\n"
                           "\r\n",
                    Url->name);

    if((buffer=read_line(socket,buffer)))
      {
       char *willget="HTTP/1.0 404 WWWOFFLE Will Get\r\n"; /* This line must not be changed (see messages.c). */
       int status;

       sscanf(buffer,"%*s %d",&status);

       if(!strcmp(willget,buffer))
          fprintf(stderr,"The URL is not in the cache but has been requested.\n");
       else if(status>=300 && status<400)
          fprintf(stderr,"The URL has been moved, check with a browser.\n");
       else if(status!=200)
          fprintf(stderr,"The URL returns an error message, check with a browser.\n");
       else
         {
          while((buffer=read_line(socket,buffer)) && buffer[0]!='\r' && buffer[0]!='\n')
             ;

          while((nbytes=read_data(socket,buffer256,256))>0)
             fwrite(buffer256,1,nbytes,stdout);
         }
      }
    else
       PrintMessage(Fatal,"Cannot read from wwwoffle server.");

    CloseSocket(socket);

    FreeURL(Url);
   }

 /* exit. */

 return(0);
}


/*++++++++++++++++++++++++++++++++++++++
  Print the program usage in long or short format.

  int verbose True for long format.
  ++++++++++++++++++++++++++++++++++++++*/

static void usage(int verbose)
{
 fputs("\n"
       "WWWOFFLE - World Wide Web Offline Explorer - Version 2.0\n"
       "\n",stderr);

 if(verbose)
    fputs("(c) Andrew M. Bishop 1996,97 [       amb@gedanken.demon.co.uk ]\n"
          "                             [http://www.gedanken.demon.co.uk/]\n"
          "\n",stderr);

 fputs("Usage: wwwoffle -h\n"
       "       wwwoffle -online | -offline | -fetch | -config | -purge | -kill\n"
       "       wwwoffle -o <url>\n"
       "       wwwoffle [-i] [-f] [-F] [-(d|r|R)[<depth>]] <url> ...\n"
       "       wwwoffle [-i] [-f] [-F] [-(d|r|R)[<depth>]] [<file>|-] ...\n"
       "\n"
       "Any of these can also take:  [-p <host>[:<port>] | -c <config-file>]\n"
       "\n",stderr);

 if(verbose)
    fprintf(stderr,
            "wwwoffle -h          : Display this help.\n"
            "\n"
            "wwwoffle -online     : Indicate to the server that the network is active.\n"
            "                       (Proxy requests will be fetched from remote hosts.)\n"
            "\n"
            "wwwoffle -offline    : Indicate to the server that the network is inactive.\n"
            "                       (Proxy requests will be fetched from cache or recorded.)\n"
            "\n"
            "wwwoffle -fetch      : Force the server to fetch the pages that are recorded.\n"
            "\n"
            "wwwoffle -config     : Force the server to re-read the configuration file.\n"
            "\n"
            "wwwoffle -purge      : Force the server to purge pages from the cache.\n"
            "\n"
            "wwwoffle -kill       : Force the server to exit cleanly.\n"
            "\n"
            "wwwoffle <url> ...   : Fetch the specified URLs.\n"
            "\n"
            "wwwoffle <file> ...  : Fetch the URLs that are links in the specified file.\n"
            "\n"
            " -o                  : Fetch the URL and output it on the standard output.\n"
            "\n"
            " -i                  : Fetch the images included in the specified URLs.\n"
            " -f                  : Fetch the frames included in the specified URLs.\n"
            " -F                  : Force the url to be refreshed even if already cached.\n"
            " -(d|r|R)[<depth>]   : Fetch pages linked to the URLs and their links,\n"
            "                       going no more than <depth> steps (default 1).\n"
            "                        (-d => URLs in the same directory or sub-directory)\n"
            "                        (-r => URLs on the same host)\n"
            "                        (-R => URLs on any host)\n"
            "\n"
            " -p <host>[:<port>]  : The host name and port number to talk to the demon on.\n"
            "                       (Defaults to localhost for the server and\n"
            "                        %d for server port, %d for http proxy port).\n"
            "\n"
            " -c <config-file>    : The name of the configuration file with the hostname,\n"
            "                       port number and the password (if any).\n"
            "\n",DEF_WWWOFFLE_PORT,DEF_HTTP_PORT);

 if(verbose)
    exit(0);
 else
    exit(1);
}


/*++++++++++++++++++++++++++++++++++++++
  Add a URL or a file to the list.

  char *url_file The URL or file to add.
  ++++++++++++++++++++++++++++++++++++++*/

static void add_url_file(char *url_file)
{
 if(!(n_url_file_list%16))
   {
    if(n_url_file_list)
       url_file_list=(char**)realloc(url_file_list,(n_url_file_list+16)*sizeof(char*));
    else
       url_file_list=(char**)malloc(16*sizeof(char*));
   }

 url_file_list[n_url_file_list]=(char*)malloc(strlen(url_file)+1);

 strcpy(url_file_list[n_url_file_list],url_file);

 n_url_file_list++;
}
