/***************************************
  $Header: /home/amb/wwwoffle/RCS/spool.c 2.14 1997/12/27 17:08:32 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 2.0a.
  Handle all of the spooling of files in the spool directory.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1996,97 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <time.h>
#include <utime.h>
#include <sys/stat.h>
#include <dirent.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <ctype.h>

#include "wwwoffle.h"
#include "misc.h"
#include "errors.h"
#include "config.h"


/*++++++++++++++++++++++++++++++++++++++
  Open a file in the outgoing directory to write into / read from.

  int OpenOutgoingSpoolFile Returns a file descriptor, or -1 on failure.

  int rw Set to true to read, else false.
  ++++++++++++++++++++++++++++++++++++++*/

int OpenOutgoingSpoolFile(int rw)
{
 struct stat buf;
 int fd=-1;

 /* Create the outgoing directory if needed and change to it */

 if(lstat("outgoing",&buf))
   {
    PrintMessage(Inform,"Directory 'outgoing' does not exist [%!s]; creating one.");
    if(mkdir("outgoing",0755))
      {PrintMessage(Warning,"Cannot create directory 'outgoing' [%!s].");return(-1);}
   }
 else
    if(!S_ISDIR(buf.st_mode))
      {PrintMessage(Warning,"The file 'outgoing' is not a directory.");return(-1);}

 if(chdir("outgoing"))
   {PrintMessage(Warning,"Cannot change to directory 'outgoing' [%!s].");return(-1);}

 /* Open the outgoing file */

 if(rw)
   {
    struct dirent* ent;
    DIR *dir=opendir(".");

    if(!dir)
      {PrintMessage(Warning,"Cannot open current directory 'outgoing' [%!s].");chdir("..");return(-1);}

    ent=readdir(dir);  /* skip .  */
    if(!ent)
      {PrintMessage(Warning,"Cannot read current directory 'outgoing' [%!s].");closedir(dir);chdir("..");return(-1);}
    ent=readdir(dir);  /* skip .. */

    while((ent=readdir(dir)))
       if(*ent->d_name=='O')
         {
          if((fd=open(ent->d_name,O_RDONLY))==-1)
             PrintMessage(Inform,"Cannot open file 'outgoing/%s' [%!s]; race condition?",ent->d_name);
          else
             if(unlink(ent->d_name))
               {
                PrintMessage(Inform,"Cannot unlink file 'outgoing/%s' [%!s]; race condition?",ent->d_name);
                close(fd); fd=-1;
               }
             else
               {
                *ent->d_name='U';
                unlink(ent->d_name);
                break;
               }
         }
    closedir(dir);
   }
 else
   {
    char name[16];

    sprintf(name,"tmp.%d",getpid());

    fd=open(name,O_WRONLY|O_CREAT|O_EXCL,0644);
    if(fd==-1)
       PrintMessage(Inform,"Cannot open file 'outgoing/%s' [%!s]",name);
   }

 chdir("..");

 return(fd);
}


/*++++++++++++++++++++++++++++++++++++++
  Close an outgoing spool file and rename it to the hashed name.

  int fd The file descriptor to close.

  URL *Url The URL to close.
  ++++++++++++++++++++++++++++++++++++++*/

void CloseOutgoingSpoolFile(int fd,URL *Url)
{
 char oldname[16],*newname;
 int ufd;

 close(fd);

 if(chdir("outgoing"))
   {PrintMessage(Warning,"Cannot change to directory 'outgoing' [%!s].");return;}

 sprintf(oldname,"tmp.%d",getpid());

 newname=URLToFileName(Url);
 *newname='U';

 unlink(newname);
 ufd=open(newname,O_WRONLY|O_CREAT|O_EXCL,0644);
 if(ufd!=-1)
   {
    write_string(ufd,Url->name);
    close(ufd);
   }

 *newname='O';
 if(rename(oldname,newname))
   {PrintMessage(Warning,"Cannot rename 'outgoing/%s' to 'outgoing/%s' [%!s].",oldname,newname);unlink(oldname);}

 chdir("..");

 free(newname);
}


/*++++++++++++++++++++++++++++++++++++++
  Check if a specified URL exists in the outgoing directory.

  int ExistsOutgoingSpoolFile Returns a boolean.

  URL *Url The URL to check for.
  ++++++++++++++++++++++++++++++++++++++*/

int ExistsOutgoingSpoolFile(URL *Url)
{
 struct stat buf;
 char *name;
 int exists=0;

 if(chdir("outgoing"))
   {PrintMessage(Warning,"Cannot change to directory 'outgoing' [%!s].");return(0);}

 /* Stat the outgoing file */

 name=URLToFileName(Url);
 *name='O';

 exists=!stat(name,&buf);

 chdir("..");

 free(name);

 return(exists);
}


/*++++++++++++++++++++++++++++++++++++++
  Delete a specified URL request from the outgoing requests.

  char *DeleteOutgoingSpoolFile Returns NULL if OK else error message.

  URL *Url The URL to delete.
  ++++++++++++++++++++++++++++++++++++++*/

char *DeleteOutgoingSpoolFile(URL *Url)
{
 char *err=NULL;

 if(chdir("outgoing"))
   {err=PrintMessage(Warning,"Cannot change to directory 'outgoing' [%!s].");return(err);}

 /* Delete the file for the request or all of them. */

 if(Url)
   {
    char *name;

    name=URLToFileName(Url);
    *name='O';

    if(unlink(name))
       err=PrintMessage(Warning,"Cannot unlink outgoing request '%s' [%!s].",name);

    *name='U';
    unlink(name);

    free(name);
   }
 else
   {
    struct dirent* ent;
    DIR *dir=opendir(".");

    if(!dir)
      {err=PrintMessage(Warning,"Cannot open current directory 'outgoing' [%!s].");chdir("..");return(err);}

    ent=readdir(dir);  /* skip .  */
    if(!ent)
      {err=PrintMessage(Warning,"Cannot read current directory 'outgoing' [%!s].");closedir(dir);chdir("..");return(err);}
    ent=readdir(dir);  /* skip .. */

    while((ent=readdir(dir)))
      {
       if(unlink(ent->d_name))
          err=PrintMessage(Warning,"Cannot unlink outgoing request '%s' [%!s].",ent->d_name);
      }

    closedir(dir);
   }

 chdir("..");

 return(err);
}


/*++++++++++++++++++++++++++++++++++++++
  Open a file in a spool subdirectory to write into / read from.

  int OpenWebpageSpoolFile Returns a file descriptor.

  int rw Set to 1 to read, 0 to write.

  URL *Url The URL to open.
  ++++++++++++++++++++++++++++++++++++++*/

int OpenWebpageSpoolFile(int rw,URL *Url)
{
 struct stat buf;
 char *file;
 int fd=-1;

 /* Create the spool directory if needed and change to it. */

 if(lstat(Url->proto,&buf))
   {
    PrintMessage(Inform,"Directory '%s' does not exist [%!s]; creating one.",Url->proto);
    if(mkdir(Url->proto,0755))
      {PrintMessage(Warning,"Cannot create directory '%s' [%!s].",Url->proto);return(-1);}
   }
 else
    if(!S_ISDIR(buf.st_mode))
      {PrintMessage(Warning,"The file '%s' is not a directory.",Url->proto);return(-1);}

 if(chdir(Url->proto))
   {PrintMessage(Warning,"Cannot change to directory '%s' [%!s].",Url->proto);return(-1);}

 if(lstat(Url->host,&buf))
   {
    PrintMessage(Inform,"Directory '%s/%s' does not exist [%!s]; creating one.",Url->proto,Url->host);
    if(mkdir(Url->host,0755))
      {PrintMessage(Warning,"Cannot create directory '%s/%s' [%!s].",Url->proto,Url->host);chdir("..");return(-1);}
   }
 else
    if(!S_ISDIR(buf.st_mode))
      {PrintMessage(Warning,"The file '%s/%s' is not a directory.",Url->proto,Url->host);chdir("..");return(-1);}

 if(chdir(Url->host))
   {PrintMessage(Warning,"Cannot change to directory '%s/%s' [%!s].",Url->proto,Url->host);chdir("..");return(-1);}

 /* Open the file for the web page. */

 file=URLToFileName(Url);

 *file='D';
 if(rw)
    fd=open(file,O_RDONLY);
 else
    fd=open(file,O_RDWR|O_CREAT,0644);

 if(!rw && fd!=-1)
   {
    int ufd;

    *file='U';
    ufd=open(file,O_WRONLY|O_CREAT|O_TRUNC,0644);

    if(ufd!=-1)
      {
       write_string(ufd,Url->name);
       close(ufd);
      }
    else
       fd=-1;
   }

 /* Change the modification time on the directory. */

 if(!rw && fd!=-1)
    utime(".",NULL);

 chdir("..");

 if(!rw && fd!=-1)
    utime(".",NULL);

 chdir("..");

 free(file);

 return(fd);
}


/*++++++++++++++++++++++++++++++++++++++
  Delete a file in a spool subdirectory.

  char *DeleteWebpageSpoolFile Return NULL if OK else error message.

  URL *Url The URL to delete.

  int all If set then delete all pages from this host.
  ++++++++++++++++++++++++++++++++++++++*/

char *DeleteWebpageSpoolFile(URL *Url,int all)
{
 char *err=NULL;

 if(chdir(Url->proto))
   {err=PrintMessage(Warning,"Cannot change to directory '%s' [%!s].",Url->proto);return(err);}

 if(chdir(Url->host))
   {err=PrintMessage(Warning,"Cannot change to directory '%s/%s' [%!s].",Url->proto,Url->host);chdir("..");return(err);}

 /* Delete the file for the web page. */

 if(all)
   {
    struct dirent* ent;
    DIR *dir=opendir(".");

    if(!dir)
      {err=PrintMessage(Warning,"Cannot open current directory '%s/%s' [%!s].",Url->proto,Url->host);chdir("../..");return(err);}

    ent=readdir(dir);  /* skip .  */
    if(!ent)
      {err=PrintMessage(Warning,"Cannot read current directory '%s/%s' [%!s].",Url->proto,Url->host);closedir(dir);chdir("../..");return(err);}
    ent=readdir(dir);  /* skip .. */

    while((ent=readdir(dir)))
      {
       if(unlink(ent->d_name))
          err=PrintMessage(Warning,"Cannot unlink outgoing request '%s/%s/%s' [%!s].",Url->proto,Url->host,ent->d_name);
      }

    closedir(dir);

    chdir("..");

    if(rmdir(Url->host))
      err=PrintMessage(Warning,"Cannot delete what should be an empty directory '%s/%s' [%!s].",Url->proto,Url->host);

    chdir("..");
   }
 else
   {
    char *file=URLToFileName(Url);
    struct stat buf;
    int didstat=1;

    if(stat(".",&buf))
       PrintMessage(Warning,"Cannot stat directory '%s/%s' [%!s].",Url->proto,Url->host);
    else
       didstat=1;

    *file='D';
    if(unlink(file))
       err=PrintMessage(Warning,"Cannot unlink cached file '%s/%s/%s' [%!s].",Url->proto,Url->host,file);

    *file='U';
    unlink(file);

    free(file);

    if(didstat)
      {
       struct utimbuf utbuf;

       utbuf.actime=time(NULL);
       utbuf.modtime=buf.st_mtime;
       utime(".",&utbuf);
      }

    chdir("../..");
   }

 return(err);
}


/*++++++++++++++++++++++++++++++++++++++
  Touch a file in a spool subdirectory.

  URL *Url The URL to touch.
  ++++++++++++++++++++++++++++++++++++++*/

void TouchWebpageSpoolFile(URL *Url)
{
 char *file;

 if(chdir(Url->proto))
   {PrintMessage(Warning,"Cannot change to directory '%s' [%!s].",Url->proto);return;}

 if(chdir(Url->host))
   {PrintMessage(Warning,"Cannot change to directory '%s/%s' [%!s].",Url->proto,Url->host);chdir("..");return;}

 /* Touch the file for the web page. */

 file=URLToFileName(Url);

 *file='D';
 utime(file,NULL);

 chdir("../..");

 free(file);
}


/*++++++++++++++++++++++++++++++++++++++
  Touch a file in a spool subdirectory.

  int ExistsWebpageSpoolFile Return a true value if the page exists.

  URL *Url The URL to touch.
  ++++++++++++++++++++++++++++++++++++++*/

int ExistsWebpageSpoolFile(URL *Url)
{
 struct stat buf;
 char *file;
 int exists=0;

 if(chdir(Url->proto))
   {PrintMessage(Warning,"Cannot change to directory '%s' [%!s].",Url->proto);return(0);}

 if(chdir(Url->host))
   {PrintMessage(Warning,"Cannot change to directory '%s/%s' [%!s].",Url->proto,Url->host);chdir("..");return(0);}

 /* Stat the file for the web page. */

 file=URLToFileName(Url);
 *file='D';

 exists=!stat(file,&buf);

 chdir("../..");

 free(file);

 return(exists);
}


/*++++++++++++++++++++++++++++++++++++++
  Convert a filename to a URL.

  char *FileNameToURL Returns the URL.

  char *file The file name.
  ++++++++++++++++++++++++++++++++++++++*/

char *FileNameToURL(char *file)
{
 char *path,*copy;
 int ufd,r;

 if(file[0] && file[0]!='D' && file[0]!='O')
    return(NULL);

 copy=(char*)malloc(strlen(file)+1);
 strcpy(copy,file);

 path=(char*)malloc(256+1);

 *copy='U';

 ufd=open(copy,O_RDONLY);

 if(ufd==-1)
   {
    free(copy);
    free(path);
    return(NULL);
   }

 r=read(ufd,path,256);

 if(r==256)
   {
    int rr=0;
    do
      {
       r+=rr;
       path=(char*)realloc(path,r+256+1);
      }
    while((rr=read(ufd,&path[r],256)));
   }
 else if(r==-1)
   {
    close(ufd);
    free(copy);
    free(path);
    return(NULL);
   }

 path[r]=0;

 close(ufd);

 free(copy);

 return(path);
}


/*++++++++++++++++++++++++++++++++++++++
  Convert a URL to a filename

  char *URLToFileName Returns the filename.

  URL *Url The URL to convert to a filename.
  ++++++++++++++++++++++++++++++++++++++*/

char *URLToFileName(URL *Url)
{
 char *hash;
 char *file;

 hash=MakeHash(Url->name);

 file=(char*)malloc(strlen(hash)+2);

 sprintf(file,"X%s",hash);

 return(file);
}
