/***************************************
  $Header: /home/amb/wwwoffle/RCS/parse.c 2.12 1998/01/03 17:59:47 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 2.0c.
  Functions to parse the HTTP requests.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1996,97,98 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>

#include "wwwoffle.h"
#include "misc.h"
#include "proto.h"
#include "errors.h"
#include "config.h"


/*++++++++++++++++++++++++++++++++++++++
  Parse the request to the server.

  char *ParseRequest Returns the request.

  int fd The file descriptor to read the request from.

  char **url Return the url or NULL if it failed.
  ++++++++++++++++++++++++++++++++++++++*/

char *ParseRequest(int fd,char **url)
{
 char *request=NULL,*line=NULL;
 int length=-1;

 *url=NULL;

 while((line=read_line_or_timeout(fd,line)))
   {
    if(!strncmp("Content-length:",line,15) || !strncmp("Content-Length:",line,15))
       length=atoi(&line[15]);

    if(!request)                /* first line */
      {
       request=(char*)malloc(strlen(line)+1);
       strcpy(request,line);

       *url=(char*)malloc(strlen(line));
       if(sscanf(line,"%*s %s",*url)!=1)
         {free(*url);*url=NULL;return(NULL);}
      }
    else
      {
       request=(char*)realloc((void*)request,strlen(line)+strlen(request)+1);
       strcat(request,line);
      }

    if(*line=='\r' || *line=='\n')
       break;
   }

 if(!request)
    return(NULL);

 if(line)
    free(line);

 if(!strncasecmp("POST",request,4))
   {
    int n=strlen(request)+length;

    if(length==-1)
      {free(*url);*url=NULL;return(request);}

    if(length)
      {
       int m,l=length;

       request=(char*)realloc(request,n+1);

       do
         {
          m=read_data_or_timeout(fd,&request[n-l],l);
         }
       while(m>0 && (l-=m));

       if(l)
         {free(*url);*url=NULL;return(request);}

       request[n]=0;
      }

    *url=(char*)realloc((void*)*url,strlen(*url)+32);

    if(strchr(*url,'?'))
      {
       char *from=*url+strlen(*url),*to=from+1;
       while(*from!='?')
          *to--=*from--;
       *to='!';
      }
    else
       strcat(*url,"?");

    sprintf(*url+strlen(*url),"!POST:%s",MakeHash(&request[n-length]));
   }

 return(request);
}


/*++++++++++++++++++++++++++++++++++++++
  Modify the request to ask for changes since the spooled file.

  int RequestChanges Returns 1 if the file needs changes made, 0 if not, or -1 in case of an error.

  int fd The file descriptor of the spooled file.

  char **request The request to modify.
  ++++++++++++++++++++++++++++++++++++++*/

int RequestChanges(int fd,char **request)
{
 struct stat buf;
 char *reply;
 int status=0;
 
 reply=read_line(fd,NULL);

 if(reply)
   {
    sscanf(reply,"%*s %d",&status);
    free(reply);
   }

 if(status==0)
   {
    PrintMessage(Debug,"Empty or no status");
    return(-1);
   }
 else if(status>=200 && status<400 && !fstat(fd,&buf))
   {
    if((time(NULL)-buf.st_mtime)<600)
      {
       PrintMessage(Debug,"Too new");
       return(0);
      }
    else
      {
       char *if_mod=(char*)malloc(64);
       char *copy=(char*)malloc(strlen(*request)+64);
       char *eol=strchr(*request,'\n');

       sprintf(if_mod,"If-Modified-Since: %s",RFC822Date(buf.st_mtime,1));
       if_mod[strlen(if_mod)-1]=0;
       *eol=0; eol++;

       strcpy(copy,*request);
       strcat(copy,"\n");
       strcat(copy,if_mod);
       strcat(copy,"\r\n");
       strcat(copy,eol);

       free(*request);
       free(if_mod);

       *request=copy;
      }
   }

 return(1);
}


/*++++++++++++++++++++++++++++++++++++++
  Return the location that the URL has been moved to.

  char *MovedLocation Returns the new URL.

  URL *Url The original URL.

  char *reply The original reply.
  ++++++++++++++++++++++++++++++++++++++*/

char *MovedLocation(URL *Url,char *reply)
{
 char *location,*eol;
 char *new;

 location=strstr(reply,"\nLocation:");
 if(!location)
    return(NULL);

 location+=11;
 eol=strchr(location,'\n');
 if(eol[-1]=='\r')
    eol--;
 *eol=0;

 new=(char*)malloc(strlen(location)+strlen(Url->name));

 if(!strchr(location,'/'))
   {
    char *newpath=(char*)malloc(strlen(Url->path)+1),*p;
    strcpy(newpath,Url->path);
    p=newpath+strlen(newpath);
    while(p>newpath && *--p!='/')
       *p=0;
    sprintf(new,"%s://%s/%s%s",Url->proto,Url->host,newpath,location);
    free(newpath);
   }
 else if(*location=='/')
   {
    sprintf(new,"%s://%s%s",Url->proto,Url->host,location);
   }
 else
   {
    strcpy(new,location);
   }

 return(new);
}


/*++++++++++++++++++++++++++++++++++++++
  Create a new request for a page.

  char *RequestURL Ask for a page.

  char *url The URL to get.

  char *referer The Refering URL or NULL if none.
  ++++++++++++++++++++++++++++++++++++++*/

char *RequestURL(char *url,char *referer)
{
 char *new=(char*)malloc(strlen(url)+(referer?strlen(referer):0)+64);

 sprintf(new,"GET %s HTTP/1.0\r\n",url);
 if(referer)
    sprintf(&new[strlen(new)],"Referer: %s\r\n",referer);
 strcat(new,"Accept: */*\r\n"
            "\r\n");

 return(new);
}


/*++++++++++++++++++++++++++++++++++++++
  Modify the request taking into account censoring of header and modified URL.

  char *ModifyRequest Return the new request.

  URL *Url The actual URL.

  char *request The original request possibly with a different URL.
  ++++++++++++++++++++++++++++++++++++++*/

char *ModifyRequest(URL *Url,char *request)
{
 char *new;
 char *hostheader=(char*)malloc(strlen(Url->host)+16),*closeheader;
 char *bol,*to,*pling,http[16];

 /* Make up the new headers. */

 sprintf(hostheader,"Host: %s\r\n",Url->host);
 closeheader="Connection: close\r\n";

 new=(char*)malloc(strlen(request)+strlen(closeheader)+strlen(hostheader)+strlen(Url->name));

 /* Parse the old header and create a new one. */

 sscanf(request,"%s %*s %s",new,http);
 strcat(new," ");
 strcat(new,Url->name);

 /* Remove the false arguments from POSTed URLs. */

 if((pling=strchr(new,'!')))
   {
    char *pling2=strchr(pling+1,'!');

    if(pling2)
       for(;pling<pling2;pling++)
          *pling=*(pling+1);

    *(pling-1)=0;
   }

 strcat(new," ");
 strcat(new,http);
 strcat(new,"\r\n");

 strcat(new,hostheader);

 /* Check for HTTP 1.1 and add a Connection header */

 if(!strncmp(http,"HTTP/1.1",8))
    strcat(new,closeheader);

 /* Censor the header */

 to=new+strlen(new);
 bol=strchr(request,'\n')+1;

 while(*bol)
   {
    if(!strncmp("Host:",bol,5) || !strncmp("Connection:",bol,11) || !strncmp("Proxy-Connection:",bol,17))
       bol=strchr(bol,'\n');
    else if(IsCensoredHeader(bol))
      {
       char *eol=strchr(bol,'\n');

       if(*(eol-1)=='\r')
          *(eol-1)=0;
       else
          *eol=0;

       PrintMessage(Debug,"Censored '%s'",bol);

       bol=eol;
      }
    else if(!strncmp("Referer:",bol,8))
      {
       char *eol=strchr(bol,'\n');
       
       if(*(eol-1)=='\r')
          *(eol-1)=0;
       else
          *eol=0;

       if((pling=strchr(bol,'!')))
         {
          char *pling2=strchr(pling+1,'!');

          if(pling2)
             for(;pling<pling2;pling++)
                *pling=*(pling+1);

          *(pling-1)=0;
         }

       while(*bol)
          *to++=*bol++;
       *to++='\r';
       *to++='\n';

       bol=eol;
      }
    else
      {
       while(*bol && *bol!='\n')
          *to++=*bol++;
       *to++=*bol;
      }

    bol++;
   }

 *to=0;

 /* tidy up and exit. */

 free(hostheader);
 free(request);

 return(new);
}


/*++++++++++++++++++++++++++++++++++++++
  Change the request from one to a proxy to a normal one.

  char *MakeRequestNonProxy Return a new request that is suitable for a non-proxy server.

  char *request The buffer containing the request.
  ++++++++++++++++++++++++++++++++++++++*/

char *MakeRequestNonProxy(char *request)
{
 char *new=(char*)malloc(strlen(request)),*r=request,*n=new;

 /* The URL is already in canonical form because of the ModifyRequest() function. */

 while(*r!=' ')                 /* 'GET ' */
    *n++=*r++;
 *n++=*r++;

 while(*r!=':')                 /* 'http://' */
    r++;
 r+=3;

 while(*r!='/')                 /* 'www.host.domain/' */
    r++;

 strcpy(n,r);

 return(new);
}


/*++++++++++++++++++++++++++++++++++++++
  Parse the reply from the server.

  char *ParseReply Return the chunk of data that was read.

  URL *Url The URL that we are reading.

  int *status Return the numeric status code.
  ++++++++++++++++++++++++++++++++++++++*/

char *ParseReply(URL *Url,int *status)
{
 char *reply=NULL,*line=NULL;

 while((line=(Url->Protocol->readline)(line)))
   {
    if(!reply)
      {
       reply=(char*)malloc(strlen(line)+1);
       strcpy(reply,line);
      }
    else
      {
       reply=(char*)realloc((void*)reply,strlen(line)+strlen(reply)+1);
       strcat(reply,line);
      }

    if(*line=='\r' || *line=='\n')
       break;
   }

 if(!line)
   {free(reply);return(NULL);}

 if(sscanf(reply,"%*s %d",status)!=1)
    *status=0;

 return(reply);
}
