/***************************************
  $Header: /home/amb/wwwoffle/RCS/index.c 2.21 1998/01/03 09:56:44 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 2.0c.
  Generate an index of the web pages that are cached in the system.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1997,98 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <time.h>
#include <utime.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <fcntl.h>
#include <unistd.h>

#include "wwwoffle.h"
#include "misc.h"
#include "proto.h"
#include "config.h"
#include "errors.h"


/*+ A type to contain the information required to sort the files. +*/
typedef struct _FileIndex
{
 char *name;                    /*+ The URL of the file. +*/
 char *type;                    /*+ The type (file extension) of the URL. +*/
 long time;                     /*+ The time of the file (access or modification as appropriate). +*/
}
FileIndex;

/*+ The method of sorting used. +*/
static char *sorttype[][2]={{"none"  ,"Unsorted"},
                            {"mtime" ,"Modification&nbsp;Time"},
                            {"atime" ,"Access&nbsp;Time"},
                            {"alpha" ,"Alphabetical"},
                            {"type"  ,"File&nbsp;Type"},
                            {"none"  ,"Unsorted"}};

/*+ An enumerated type for the sort modes. +*/
typedef enum _SortMode
{
 None,                          /*+ No sorting, natural order. +*/
 MTime,                         /*+ Sort by modification time +*/
 ATime,                         /*+ Sort by access time. +*/
 Alpha,                         /*+ Sort Alphabetically. +*/
 Type,                          /*+ Sort by type (file extension). +*/
 NSortModes                     /*+ The number of sort modes. +*/
}
SortMode;


/*+ The list of files. +*/
static FileIndex **files=NULL;
static int nfiles=0;

/*+ The earliest time that files can appear in the latest list. +*/
static time_t earliest;

static void IndexRoot(int fd);
static void IndexProtocol(int fd,char *proto,SortMode mode);
static void IndexHost(int fd,char *proto,char *host,SortMode mode);
static void IndexOutgoing(int fd,SortMode mode);
static void IndexLatest(int fd);

static void CorrectIndex(int fd,char *path,char *args);
static void IllegalIndex(int fd,char *path,char *args);

static void add_dir(char *name,SortMode mode,char *link);
static void add_file(char *name,SortMode mode);

static int sort_alpha(FileIndex **a,FileIndex **b);
static int sort_type(FileIndex **a,FileIndex **b);
static int sort_time(FileIndex **a,FileIndex **b);


/*++++++++++++++++++++++++++++++++++++++
  Generate an index of the pages that are in the cache.

  int fd The file descriptor to write the output to.

  char *url The URL that specifies the path to generate the index for.
  ++++++++++++++++++++++++++++++++++++++*/

void IndexPage(int fd,char *path,char *args)
{
 char *head=
 "HTTP/1.0 200 WWWOFFLE Cache Index\r\n"
 "Content-type: text/html\r\n"
 "\r\n"
 "<HTML>\n"
 "<HEAD>\n"
 "<TITLE>WWWOFFLE Index - %s</TITLE>\n"
 "</HEAD>\n"
 "<BODY>\n"
 "<H1 align=center>WWWOFFLE Cache Index</H1>\n";
 char *tail=
 "</BODY>\n"
 "</HTML>\n";
 char *newpath=(char*)malloc(strlen(path)+1);
 char *proto=newpath,*host=NULL;
 SortMode sort=NSortModes;
 Protocol *protocol=NULL;
 int outgoing=0,latest=0,main=0;
 int i;

 earliest=time(NULL)-IndexLatestDays*24*3600;

 strcpy(newpath,path);

 if(*newpath && newpath[strlen(newpath)-1]=='/')
    newpath[strlen(newpath)-1]=0;

 for(i=0;newpath[i];i++)
    if(newpath[i]=='/')
      {
       newpath[i]=0;
       host=newpath+i+1;
       break;
      }

 for(i=0;i<NProtocols;i++)
    if(!strcmp(proto,Protocols[i].name))
       protocol=&Protocols[i];

 outgoing=!strcmp(proto,"outgoing");
 latest=!strcmp(proto,"latest");
 main=(!protocol && !outgoing && !latest);

 if(args)
    for(sort=0;sort<NSortModes;sort++)
       if(!strcmp(args,sorttype[sort][0]))
          break;

 if(host && (strchr(host,'/') || !strcmp(host,"..") || !strcmp(host,".")))
    IllegalIndex(fd,path,args);
 else if(!outgoing && !latest && !protocol && *newpath)
    IllegalIndex(fd,path,args);
 else if((outgoing || latest) && host)
    IllegalIndex(fd,path,args);
 else if(sort==NSortModes && (outgoing || protocol))
    CorrectIndex(fd,path,args);
 else
   {
    SortMode mode;

    if(main)
       write_formatted(fd,head,"Main Index");
    else if(outgoing)
       write_formatted(fd,head,"Outgoing Requests");
    else if(latest)
       write_formatted(fd,head,"Latest Pages");
    else if(protocol && !host)
       write_formatted(fd,head,proto);
    else if(protocol && host)
      {
       char *protohost=(char*)malloc(strlen(proto)+strlen(host)+4);
       sprintf(protohost,"%s://%s",proto,host);
       write_formatted(fd,head,protohost);
       free(protohost);
      }

    write_string(fd,"<p align=center>\n");

    if(main)
       write_string(fd,"[Main&nbsp;Index]\n");
    else
       write_formatted(fd,"<a href=\"/index/?%s\">[Main&nbsp;Index]</a>\n",sorttype[sort][0]);

    write_string(fd,"<br>\n");

    if(outgoing)
       write_string(fd,"[Outgoing]\n");
    else
       write_formatted(fd,"<a href=\"/index/outgoing/?%s\">[Outgoing]</a>\n",sorttype[sort][0]);

    if(latest)
       write_string(fd,"[Latest]\n");
    else
       write_formatted(fd,"<a href=\"/index/latest/?%s\">[Latest]</a>\n",sorttype[sort][0]);

    for(i=0;i<NProtocols;i++)
       write_formatted(fd,"<a href=\"/index/%s/?%s\">[%s]</a>\n",Protocols[i].name,sorttype[sort][0],Protocols[i].name);

    write_string(fd,"<br>\n");

    for(mode=0;mode<NSortModes;mode++)
       if(mode==sort || latest || main)
          write_formatted(fd,"[%s]\n",sorttype[mode][1]);
       else
          write_formatted(fd,"<a href=\"/index/%s?%s\">[%s]</a>\n",path,sorttype[mode][0],sorttype[mode][1]);

    write_string(fd,"</p>\n");

    write_string(fd,"<hr>\n");

    if(outgoing)
       IndexOutgoing(fd,sort);
    else if(latest)
       IndexLatest(fd);
    else if(protocol && !host)
       IndexProtocol(fd,proto,sort);
    else if(protocol && host)
       IndexHost(fd,proto,host,sort);
    else
       IndexRoot(fd);

    write_string(fd,"<hr>\n");
    write_string(fd,"<p align=center>\n");

    if(outgoing)
       write_formatted(fd,"<a href=\"/index/?%s\">[Back to the Main Index]</a>\n",sorttype[mode][0]);
    else if(latest)
       write_string(fd,"<a href=\"/index/\">[Back to the Main Index]</a>\n");
    else if(protocol && !host)
       write_formatted(fd,"<a href=\"/index/?%s\">[Back to the Main Index]</a>\n",sorttype[mode][0]);
    else if(protocol && host)
       write_formatted(fd,"<a href=\"/index/%s/?%s\">[Back to the Hosts Index]</a>\n",proto,sorttype[mode][0]);
    else
       write_string(fd,"<a href=\"/\">[Back to the Welcome Page]</a>\n");

    write_string(fd,"</p>\n");

    write_string(fd,tail);
   }

 free(newpath);
}


/*++++++++++++++++++++++++++++++++++++++
  Index the root of the cache.

  int fd The file descriptor to write to.
  ++++++++++++++++++++++++++++++++++++++*/

static void IndexRoot(int fd)
{
 char *page=
 "<h2>Main Index</h2>\n"
 "This is the main index page for all of the files in the wwwoffle cache."
 "<h3>New Format Index</h3>\n"
 "The index of the files in the cache has changed in version 2.0 of wwwoffle compared to earlier versions."
 "This is for a number of reasons:\n"
 "<ul>\n"
 "<li>So that the new protocol types can be introduced.\n"
 "<li>To present a consistent interface for all of the index pages (the new buttons bars).\n"
 "<li>To improve the formatting and layout.\n"
 "</ul>\n"
 "It is hoped that this new format is easier to use and makes more sense than the previous version.\n"
 "<h3>First Level Index</h3>\n"
 "This page is the first level index, it is available on the first of the three button bars.\n"
 "<h3>Second Level Indexes</h3>\n"
 "Below the main index there are several second level indexes as shown on the second of the button bars.\n"
 "<br>These indexes are:\n"
 "<dl compact>\n"
 "<dt>Outgoing\n"
 "<dd>The index of the requests that have been made while offline and are waiting to be fetched.\n"
 "<dt>Latest\n"
 "<dd>All of the latest pages from all protocols sorted with the newest at the top and with breaks every hour and a separator 24 hours.\n"
 "<dt>Protocols (http,ftp etc.)\n"
 "<dd>Each of the protocols that wwwoffle supports has an index showing the hosts that have pages stored.\n"
 "</dl>\n"
 "<h3>Third Level Indexes</h3>\n"
 "Below the protocol indexes there are third level indexes (not shown on the button bars).\n"
 "<br>These indexes are the list of pages from a particular host using a particular protocol.\n"
 "<h3>Sorting</h3>\n"
 "All of the second and third level indexes (apart from Latest) can be sorted in several ways using the third button bar.\n"
 "<br>These methods are:\n"
 "<dl compact>\n"
 "<dt>Unsorted\n"
 "<dd>No sorting applied, this uses the default directory order.\n"
 "<dt>Modification Time\n"
 "<dd>The time that the page was last modified (most recent first), this could be when it was fetched or refreshed.\n"
 "<dt>Access Time\n"
 "<dd>The time that the page was last accessed (most recent first), this is the time that the page was last viewed (if later than modification time).\n"
 "<dt>Alphabetical\n"
 "<dd>The pages are sorted into alphabetical order.\n"
 "<dt>File Type\n"
 "<dd>The pages are sorted alphabetically according to the file extension.\n"
 "</dl>\n";

 write_string(fd,page);
}


/*++++++++++++++++++++++++++++++++++++++
  Index the hosts for one protocol in the cache.

  int fd The file descriptor to write to.

  char *proto The protocol to index.

  SortMode mode The sort mode to use.
  ++++++++++++++++++++++++++++++++++++++*/

static void IndexProtocol(int fd,char *proto,SortMode mode)
{
 DIR *dir;
 struct dirent* ent;
 char ***mirrored;
 int i;

 /* Open the spool directory. */

 if(chdir(proto))
   {PrintMessage(Warning,"Cannot change to directory '%s' [%!s]; not indexed.",proto);
   write_formatted(fd,"No %s directory.",proto);return;}

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open spool directory '%s' [%!s]; index failed.",proto);chdir("..");
   write_formatted(fd,"Unable to open directory %s.",proto);return;}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read spool directory '%s' [%!s]; index failed.",proto);closedir(dir);chdir("..");
   write_formatted(fd,"Unable to read directory %s.",proto);return;}
 ent=readdir(dir);  /* skip .. */

 /* Get all of the host sub-directories. */

 while((ent=readdir(dir)))
    add_dir(ent->d_name,mode,NULL);

 closedir(dir);

 /* Get all of the mirrored hosts. */

 if((mirrored=ListMirrors(proto)))
   {
    int i;

    for(i=0;mirrored[i];i++)
      {
       add_dir(mirrored[i][0],mode,mirrored[i][1]);
       free(mirrored[i][0]);
       free(mirrored[i][1]);
       free(mirrored[i]);
      }

    free(mirrored);
   }

 chdir("..");

 /* Sort the files. */

 if(mode==MTime || mode==ATime)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_time);
 else if(mode==Alpha || mode==Type)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_alpha);

 write_formatted(fd,"<h2>%s://</h2>\n",proto);

 if(nfiles)
   {
    write_formatted(fd,"<b>%d Hosts</b>\n",nfiles);

    write_string(fd,"<ul>\n");

    for(i=0;i<nfiles;i++)
      {
       write_formatted(fd,"<li><a href=\"/index/%s/%s?%s\">%s</a>\n",proto,files[i]->name,sorttype[mode][0],files[i]->name);
       free(files[i]->name);
       free(files[i]);
      }

    write_string(fd,"</ul>\n");
   }
 else
    write_string(fd,"<b>No hosts</b>\n");

 if(files)
    free(files);
 files=NULL;
 nfiles=0;
}


/*++++++++++++++++++++++++++++++++++++++
  Create an index of the pages on a host.

  int fd The file descriptor to write to.

  char *proto The protocol to index.

  char *host The name of the subdirectory.

  SortMode mode The sort mode to use.
  ++++++++++++++++++++++++++++++++++++++*/

static void IndexHost(int fd,char *proto,char *host,SortMode mode)
{
 DIR *dir;
 struct dirent* ent;
 int i;
 char *link_proto=NULL,*link_host=NULL;

 /* Write the header. */

 write_formatted(fd,"<h2>%s://%s/</h2>\n",proto,host);

 if(IsMirrored(proto,host,&link_proto,&link_host))
   {
    write_formatted(fd,"Mirror Link =&gt; %s://%s/\n<p>\n",link_proto,link_host);
    proto=link_proto;
    host=link_host;
   }

 /* Open the spool subdirectory. */

 if(chdir(proto))
   {PrintMessage(Warning,"Cannot change to directory '%s' [%!s]; not indexed.",proto);
   write_formatted(fd,"No %s directory.",proto);return;}

 /* Open the spool subdirectory. */

 if(chdir(host))
   {PrintMessage(Warning,"Cannot change to directory '%s/%s' [%!s]; not indexed.",proto,host);chdir("..");
   write_formatted(fd,"No %s/%s directory.",proto,host);return;}

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open directory '%s/%s' [%!s]; not indexed.",proto,host);chdir("../..");
   write_formatted(fd,"Unable to open directory %s/%s.",proto,host);return;}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read directory '%s/%s' [%!s]; not indexed.",proto,host);closedir(dir);chdir("../..");
   write_formatted(fd,"Unable to read directory %s/%s.",proto,host);return;}
 ent=readdir(dir);  /* skip .. */

 /* Print all of the file names. */

 while((ent=readdir(dir)))
    add_file(ent->d_name,mode);

 closedir(dir);

 chdir("../..");

 /* Sort the files. */

 if(mode==MTime || mode==ATime)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_time);
 else if(mode==Alpha)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_alpha);
 else if(mode==Type)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_type);

 if(nfiles)
   {
    write_formatted(fd,"<b>%d Pages</b>\n",nfiles);

    write_string(fd,"<ul>\n");

    for(i=0;i<nfiles;i++)
      {
       URL *Url=SplitURL(files[i]->name);

       if(Url->Protocol && Url->args && *Url->args=='!')
          write_formatted(fd,"<li>[<a href=\"/control/delete?url=%s\">Del</a>"
                             "|Refresh|"
                             "Opt]"
                             "&nbsp;<a href=\"%s\">%s</a>\n",
                          Url->name,
                          Url->name,Url->pathp);
       else if(Url->Protocol)
          write_formatted(fd,"<li>[<a href=\"/control/delete?url=%s\">Del</a>"
                             "|<a href=\"/refresh/%s/%s\">Refresh</a>|"
                             "<a href=\"/refresh/?%s\">Opt</a>]"
                             "&nbsp;<a href=\"%s\">%s</a>\n",
                          Url->name,
                          Url->proto,Url->hostp,
                          Url->name,
                          Url->name,Url->pathp);
       else
          write_formatted(fd,"<li>[Del"
                             "|Refresh|"
                             "Opt]"
                             "&nbsp;%s\n",
                          Url->pathp);

       free(files[i]->name);
       free(files[i]);
       FreeURL(Url);
      }

    write_string(fd,"</ul>\n");

    write_formatted(fd,"[<a href=\"/control/delete?url=%s://%s/&all=yes\">Delete ALL Pages</a>]\n",proto,host);
   }
 else
    write_string(fd,"<b>No Pages</b>\n");

 if(files)
    free(files);
 files=NULL;
 nfiles=0;
}


/*++++++++++++++++++++++++++++++++++++++
  Create an index of the requests that are waiting in the outgoing directory.

  int fd The file descriptor to write to.

  SortMode mode The method to use to sort the names.
  ++++++++++++++++++++++++++++++++++++++*/

static void IndexOutgoing(int fd,SortMode mode)
{
 DIR *dir;
 struct dirent* ent;
 int i;

 /* Open the spool subdirectory. */

 if(chdir("outgoing"))
   {PrintMessage(Warning,"Cannot change to directory 'outgoing' [%!s]; not indexed.");
   write_formatted(fd,"No outgoing directory.");return;}

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open directory 'outgoing' [%!s]; not indexed.");chdir("..");
   write_formatted(fd,"Unable to open outgoing directory.");return;}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read directory 'outgoing' [%!s]; not indexed.");closedir(dir);chdir("..");
   write_formatted(fd,"Unable to read outgoing directory.");return;}
 ent=readdir(dir);  /* skip .. */

 /* Print all of the file names. */

 while((ent=readdir(dir)))
    add_file(ent->d_name,mode);

 closedir(dir);

 chdir("..");

 /* Sort the files. */

 if(mode==MTime || mode==ATime)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_time);
 else if(mode==Alpha)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_alpha);
 else if(mode==Type)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_type);

 write_string(fd,"<h2>Outgoing Requests</h2>\n");

 if(nfiles)
   {
    write_formatted(fd,"<b>%d Requests</b>\n",nfiles);

    write_string(fd,"<ul>\n");

    for(i=0;i<nfiles;i++)
      {
       URL *Url=SplitURL(files[i]->name);

       if(Url->Protocol && Url->args && *Url->args=='!')
          write_formatted(fd,"<li>[<a href=\"/control/delete?req=%s\">Del</a>"
                             "|Refresh|"
                             "Opt]"
                             "&nbsp;<a href=\"%s\">%s</a>\n",
                          Url->name,
                          Url->name,Url->name);
       else if(Url->Protocol)
          write_formatted(fd,"<li>[<a href=\"/control/delete?req=%s\">Del</a>"
                             "|Refresh|"
                             "<a href=\"/refresh/?%s\">Opt</a>]"
                             "&nbsp;<a href=\"%s\">%s</a>\n",
                          Url->name,
                          Url->name,
                          Url->name,Url->name);
       else
          write_formatted(fd,"<li>[<a href=\"/control/delete?req=%s\">Del</a>"
                             "|Refresh|"
                             "Opt]"
                             "&nbsp;%s\n",
                          Url->name,
                          Url->name);

       free(files[i]->name);
       free(files[i]);
       FreeURL(Url);
      }

    write_string(fd,"</ul>\n");

    write_string(fd,"[<a href=\"/control/delete?req=&all=yes\">Delete ALL Requests</a>]\n");
   }
 else
    write_string(fd,"<b>No Requests</b>\n");

 if(files)
    free(files);
 files=NULL;
 nfiles=0;
}


/*++++++++++++++++++++++++++++++++++++++
  Create an index of the latest pages.

  int fd The file descriptor to write to.
  ++++++++++++++++++++++++++++++++++++++*/

static void IndexLatest(int fd)
{
 int i,p;
 int lastdays=0,lasthours=0;
 time_t now=earliest+IndexLatestDays*24*3600;

 /* Print all of the sub directories. */

 for(p=0;p<NProtocols;p++)
   {
    struct stat buf;

    if(!lstat(Protocols[p].name,&buf) && S_ISDIR(buf.st_mode))
      {
       DIR *dir2;
       struct dirent* ent2;

       if(chdir(Protocols[p].name))
         {PrintMessage(Warning,"Cannot change to directory '%s' [%!s]; not indexed.",Protocols[p].name);continue;}

       /* Open the spool directory. */

       dir2=opendir(".");
       if(!dir2)
         {PrintMessage(Warning,"Cannot open directory '%s' [%!s]; not indexed.",Protocols[p].name);chdir("..");continue;}

       ent2=readdir(dir2);  /* skip .  */
       if(!ent2)
         {PrintMessage(Warning,"Cannot read directory '%s' [%!s]; index failed.",Protocols[p].name);closedir(dir2);chdir("..");continue;}
       ent2=readdir(dir2);  /* skip .. */

       /* Print all of the sub directories. */

       while((ent2=readdir(dir2)))
         {
          struct stat buf2;

          if(lstat(ent2->d_name,&buf2))
             PrintMessage(Inform,"Cannot stat file '%s' [%!s]; race condition?",ent2->d_name);
          else
             if(S_ISDIR(buf2.st_mode) && buf2.st_mtime>earliest)
               {
                DIR *dir3;
                struct dirent* ent3;
                struct utimbuf utbuf;

                if(chdir(ent2->d_name))
                  {PrintMessage(Warning,"Cannot change to directory '%s/%s' [%!s]; not indexed.",Protocols[p].name,ent2->d_name);continue;}

                dir3=opendir(".");
                if(!dir3)
                  {PrintMessage(Warning,"Cannot open directory '%s/%s' [%!s]; not indexed.",Protocols[p].name,ent2->d_name);chdir("..");continue;}

                ent3=readdir(dir3);  /* skip .  */
                if(!ent3)
                  {PrintMessage(Warning,"Cannot read directory '%s/%s' [%!s]; not indexed.",Protocols[p].name,ent2->d_name);closedir(dir3);chdir("..");continue;}
                ent3=readdir(dir3);  /* skip .. */

                while((ent3=readdir(dir3)))
                   add_file(ent3->d_name,-1);

                closedir(dir3);
                chdir("..");

                utbuf.actime=buf2.st_atime;
                utbuf.modtime=buf2.st_mtime;
                utime(ent2->d_name,&utbuf);
               }
         }

       closedir(dir2);
       chdir("..");
      }
   }

 /* Sort the files. */

 qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_time);

 write_formatted(fd,"<h2>Latest Pages</h2>\n");
 write_formatted(fd,"From the last %d Days\n<p>\n",IndexLatestDays);

 if(nfiles)
   {
    write_formatted(fd,"<b>%d Pages</b>\n",nfiles);

    write_string(fd,"<ul>\n");

    for(i=0;i<nfiles;i++)
      {
       URL *Url=SplitURL(files[i]->name);
       int days=(now-files[i]->time)/(24*3600),hours=(now-files[i]->time)/3600;

       if(lastdays<days)
         {
          write_string(fd,"</ul>\n");
          for(;lastdays<days;lastdays++)
             write_string(fd,"<hr width=\"25%\">\n");
          write_string(fd,"<ul>\n");
         }
       else if(i && lasthours<hours)
         {
          write_string(fd,"</ul>\n");
          write_string(fd,"<p>\n");
          write_string(fd,"<ul>\n");
         }
       lasthours=hours;

       if(Url->Protocol && Url->args && *Url->args=='!')
          write_formatted(fd,"<li>[<a href=\"/control/delete?url=%s\">Del</a>"
                             "|Refresh|"
                             "Opt]"
                             "&nbsp;<a href=\"%s\">%s</a>\n",
                          Url->name,
                          Url->name,Url->name);
       else if(Url->Protocol)
          write_formatted(fd,"<li>[<a href=\"/control/delete?url=%s\">Del</a>"
                             "|<a href=\"/refresh/%s/%s\">Refresh</a>|"
                             "<a href=\"/refresh/?%s\">Opt</a>]"
                             "&nbsp;<a href=\"%s\">%s</a>\n",
                          Url->name,
                          Url->proto,Url->hostp,
                          Url->name,
                          Url->name,Url->name);
       else
          write_formatted(fd,"<li>[<a href=\"/control/delete?url=%s\">Del</a>"
                             "|Refresh|"
                             "Opt]"
                             "&nbsp;%s\n",
                          Url->name,
                          Url->name);

       free(files[i]->name);
       free(files[i]);
       FreeURL(Url);
      }

    write_string(fd,"</ul>\n");
   }
 else
    write_string(fd,"<b>No Pages</b>\n");

 if(++lastdays<IndexLatestDays)
   {
    for(;lastdays<IndexLatestDays;lastdays++)
       write_string(fd,"<hr width=\"25%\">\n");
   }

 if(files)
    free(files);
 files=NULL;
 nfiles=0;
}


/*++++++++++++++++++++++++++++++++++++++
  Redirect the user to the correct index page in case of a bad argument.

  int fd The file descriptor to write to.

  char *path The specified path.

  char *args The specified args.
  ++++++++++++++++++++++++++++++++++++++*/

static void CorrectIndex(int fd,char *path,char *args)
{
 char *head1=
 "HTTP/1.0 301 WWWOFFLE Index Incorrect\r\n"
 "Content-type: text/html\r\n";
 char *head2=
 "\r\n"
 "<HTML>\n"
 "<HEAD>\n"
 "<TITLE>WWWOFFLE - Index Incorrect</TITLE>\n"
 "</HEAD>\n"
 "<BODY>\n"
 "<H1 align=center>WWWOFFLE Index Incorrect</H1>\n"
 "<p align=center>\n"
 "Your request for the index URL\n"
 "<br><b><tt>\n";
 char *middle=
 "\n"
 "</tt></b><br>\n"
 "was incorrect, select the link below for a correct version.\n"
 "<br>\n";
 char *tail=
 "\n"
 "</BODY>\n"
 "</HTML>\n";

 write_string(fd,head1);
 write_formatted(fd,"Location: http://%s/index/%s?%s\r\n",GetLocalHost(1),path,sorttype[0][0]);
 write_string(fd,head2);
 if(args)
    write_formatted(fd,"/index/%s?%s",path,args);
 else
    write_formatted(fd,"/index/%s",path);
 write_string(fd,middle);
 write_formatted(fd,"<a href=\"/index/%s?%s\">/index/%s?%s</a>",path,sorttype[0][0],path,sorttype[0][0]);
 write_string(fd,tail);
}


/*++++++++++++++++++++++++++++++++++++++
  Inform the user that the specified index page is illegal.

  int fd The file descriptor to write to.

  char *path The specified path.

  char *args The specified args.
  ++++++++++++++++++++++++++++++++++++++*/

static void IllegalIndex(int fd,char *path,char *args)
{
 char *head=
 "HTTP/1.0 404 WWWOFFLE Illegal Index\r\n"
 "Content-type: text/html\r\n"
 "\r\n"
 "<HTML>\n"
 "<HEAD>\n"
 "<TITLE>WWWOFFLE - Illegal Index</TITLE>\n"
 "</HEAD>\n"
 "<BODY>\n"
 "<H1 align=center>WWWOFFLE Illegal Index</H1>\n"
 "<p align=center>\n"
 "Your request for the index URL\n"
 "<br><b><tt>\n";
 char *tail=
 "\n"
 "</tt></b><br>\n"
 "is illegal, select the link below for the main index.\n"
 "<br>\n"
 "<a href=\"/index/\">/index/</a>\n"
 "</BODY>\n"
 "</HTML>\n";

 write_string(fd,head);
 if(args)
    write_formatted(fd,"/index/%s?%s",path,args);
 else
    write_formatted(fd,"/index/%s",path);
 write_string(fd,tail);
}


/*++++++++++++++++++++++++++++++++++++++
  Add a directory to the list.

  char *name The name of the directory.

  SortMode mode The sort mode.

  char *link The directory that this is linked to if this is a mirror link.
  ++++++++++++++++++++++++++++++++++++++*/

static void add_dir(char *name,SortMode mode,char *link)
{
 struct stat buf;

 if(!link && lstat(name,&buf))
   {PrintMessage(Inform,"Cannot stat directory '%s' [%!s]; race condition?",name);return;}
 else if(link && stat(link,&buf))
   {PrintMessage(Inform,"Cannot stat link '%s' [%!s]; points nowhere?",link);return;}
 else if(S_ISDIR(buf.st_mode))
   {
    if(!(nfiles%16))
      {
       if(!files)
          files=(FileIndex**)malloc(16*sizeof(FileIndex*));
       else
          files=(FileIndex**)realloc(files,(nfiles+16)*sizeof(FileIndex*));
      }
    files[nfiles]=(FileIndex*)malloc(sizeof(FileIndex));

    files[nfiles]->name=(char*)malloc(strlen(name)+1);
    strcpy(files[nfiles]->name,name);

    files[nfiles]->type="";

    if(mode==MTime)
       files[nfiles]->time=buf.st_mtime;
    else if(mode==ATime)
       files[nfiles]->time=buf.st_atime;

    nfiles++;
   }
}


/*++++++++++++++++++++++++++++++++++++++
  Add a file to the list.

  char *name The name of the file.

  SortMode mode The sort mode.
  ++++++++++++++++++++++++++++++++++++++*/

static void add_file(char *name,SortMode mode)
{
 char *url=NULL;
 struct stat buf;

 url=FileNameToURL(name);

 if(!url)
    return;

 if(lstat(name,&buf))
   {PrintMessage(Inform,"Cannot stat file '%s' [%!s]; race condition?",name);return;}
 else if(S_ISREG(buf.st_mode) && (mode!=-1 || buf.st_mtime>earliest))
   {
    if(!(nfiles%16))
      {
       if(!files)
          files=(FileIndex**)malloc(16*sizeof(FileIndex*));
       else
          files=(FileIndex**)realloc(files,(nfiles+16)*sizeof(FileIndex*));
      }
    files[nfiles]=(FileIndex*)malloc(sizeof(FileIndex));

    files[nfiles]->name=url;

    if(mode==Type)
      {
       char *p;

       files[nfiles]->type="";

       p=strchr(files[nfiles]->name,'?');
       if(!p)
          p=files[nfiles]->name+strlen(files[nfiles]->name);

       for(;p>url;p--)
          if(*p=='.')
            {files[nfiles]->type=p;break;}
          else if(*p=='/')
             break;
      }
    else if(mode==MTime || mode==-1)
       files[nfiles]->time=buf.st_mtime;
    else if(mode==ATime)
       files[nfiles]->time=buf.st_atime;

    nfiles++;
   }
}


/*++++++++++++++++++++++++++++++++++++++
  Used to sort the files into alphabetical order.

  int sort_alpha Returns the comparison of the pointers to strings.

  FileIndex **a The first FileIndex.

  FileIndex **b The second FileIndex.
  ++++++++++++++++++++++++++++++++++++++*/

static int sort_alpha(FileIndex **a,FileIndex **b)
{
 char *an=(*a)->name;
 char *bn=(*b)->name;

 return(strcmp(an,bn));
}


/*++++++++++++++++++++++++++++++++++++++
  Used to sort the files into type (file extension) order.

  int sort_type Returns the comparison of the pointers to strings.

  FileIndex **a The first FileIndex.

  FileIndex **b The second FileIndex.
  ++++++++++++++++++++++++++++++++++++++*/

static int sort_type(FileIndex **a,FileIndex **b)
{
 char *an=(*a)->name,*at=(*a)->type;
 char *bn=(*b)->name,*bt=(*b)->type;
 int sort1=strcmp(at,bt);

 if(sort1==0)
    return(strcmp(an,bn));
 else
    return(sort1);
}


/*++++++++++++++++++++++++++++++++++++++
  Used to sort the files into alphabetical order.

  int sort_time Returns the comparison of the times.

  FileIndex **a The first FileIndex.

  FileIndex **b The second FileIndex.
  ++++++++++++++++++++++++++++++++++++++*/

static int sort_time(FileIndex **a,FileIndex **b)
{
 long at=(*a)->time;
 long bt=(*b)->time;

 return(bt-at);
}
