/***************************************
  $Header: /home/amb/wwwoffle/RCS/wwwoffles.c 1.53 1997/10/06 18:07:10 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 1.3a.
  A server to fetch the required pages.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1996,97 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <signal.h>
#include <unistd.h>
#include <fcntl.h>

#include "wwwoffle.h"
#include "config.h"
#include "sockets.h"
#include "errors.h"


static void uninstall_sighandlers(void);


/*+ The mode of operation of the server. +*/
typedef enum _Mode
{
 None,                          /*+ Undecided. +*/
 Real,                          /*+ From server host to cache and client. +*/
 RealNoCache,                   /*+ From server host to client. +*/
 RealRefresh,                   /*+ Refresh the page, forced from index. +*/
 Fetch,                         /*+ From server host to cache. +*/
 Spool,                         /*+ From cache to client. +*/
 SpoolGet,                      /*+ Not in cache so record request in outgoing. +*/
 SpoolRefresh,                  /*+ Refresh the page, forced from index. +*/
 SpoolPragma                    /*+ Refresh the page, forced from browser by 'Pragma: no-cache'. +*/
}
Mode;


/*++++++++++++++++++++++++++++++++++++++
  The main server program.

  int wwwoffles Returns the exit status.

  int online Whether the demon is online or not.

  int browser Set to true if there is a browser.

  int client The file descriptor of the client.
  ++++++++++++++++++++++++++++++++++++++*/

int wwwoffles(int online,int browser,int client)
{
 int                     outgoing     =-1  , spool     =-1  ,server=-1;
 FILE *client_file=NULL,*outgoing_file=NULL,*spool_file=NULL;

 char *request=NULL,*reply=NULL,*url=NULL,*outgoing_filename=NULL;
 int replylen=0;

 Mode mode=None;
 int fetch_again=0;

 char *host,*path,*args;
 char *server_host=NULL;
 int server_port=80;

 /* Initialise things. */

 uninstall_sighandlers();

 InitErrorHandler("wwwoffles",-1,-1); /* change name nothing else */

 if(online && browser)
    mode=Real;
 else if(online && !browser)
    mode=Fetch;
 else if(!online && browser)
    mode=Spool;
 else
    PrintMessage(Fatal,"Started in a mode that is not allowed.");

 /* Set up the client file. */

 if(client<0 && mode!=Fetch)
   {PrintMessage(Warning,"Cannot use client file descriptor %d.",client);exit(1);}

 if(client>=0)
   {
    client_file=fdopen(client,"r");
    if(!client_file)
      {PrintMessage(Warning,"Cannot use fdopen on the client file descriptor.");exit(1);}
   }

 /* Set up the outgoing file. */

 if(mode==Fetch)
   {
    outgoing=OpenOutgoingSpoolFile(1);

    if(outgoing==-1)
      {PrintMessage(Inform,"No more outgoing requests.");exit(3);}

    outgoing_file=fdopen(outgoing,"r");
    if(!outgoing_file)
      {PrintMessage(Warning,"Cannot use fdopen on the outgoing file descriptor.");exit(1);}
   }

 /* Get the URL from the request. */

 if(mode==Real || mode==Spool)
    request=ParseRequest(client_file,&url);
 else /* mode==Fetch */
    request=ParseRequest(outgoing_file,&url);

 if(!url)
   {
    if(mode!=Fetch)
       ServerError(client,request?"Cannot parse the HTTP request":"The HTTP request was empty");
    PrintMessage(Warning,"Could not parse HTTP request (%s).",request?"Parse error":"Empty request");
    exit(1);
   }

 PrintMessage(Inform,"URL = %s",url);

 SplitURL(url,&host,&path,&args);

 /* Change the mode based on the URL as required. */
 /* (At this point mode is Spool, Real or Fetch only.) */

 if(IsNotGot(host,path))
   {
    char* colon=strchr(host,':');
    if(colon) *colon=0;
    if(mode==Real || mode==Spool)
       RemoteHostError(client,url,"is on the list of hosts and paths not to get.");
    PrintMessage(Inform,"The host '%s' and/or path '%s' is on the list not to get.",host,path);
    exit(0);
   }

 if(!*host)
   {
    int newurl=0;

    if(!strncmp("refresh",path,7))
      {
       url=RefreshPage(client,path,args,url,&request,online);

       if(url==(char*)1)
          newurl=-1;
       else if(url)
         {
          SplitURL(url,&host,&path,&args);

          if(args && *args=='!')
            {
             PrintMessage(Inform,"It is not possible to refresh a URL that was posted.");
             if(mode!=Fetch)
                CantRefreshPosted(client,url);
             exit(0);
            }
          else if(!*host || IsLocalNetHost(host))
             mode=RealNoCache;
          else if(mode==Real)
            {
             mode=RealRefresh;
             OpenWebpageSpoolFile(-1,&host,path,args);
            }
          else if(mode==Spool)
            {
             spool=OpenWebpageSpoolFile(1,&host,path,args);
             if(spool!=-1)
               {mode=SpoolRefresh;close(spool);}
             else
                mode=SpoolGet;
            }
          newurl=1;
         }
      }
    else if(mode==Fetch)
       PrintMessage(Inform,"The request to fetch a page from the local host is ignored.");
    else if(!*path && !args)
       WelcomePage(client,PassWord?1:0);
    else if(!strncmp("index/",path,6))
       IndexPage(client,&path[6],args);
    else if(!strncmp("control/",path,8))
       ControlPage(client,&path[8],args,request);
    else
       IllegalPage(client,path,args);

    if(!newurl)
       exit(0);
    else if(newurl==-1)
       exit(4);
   }

 if(IsLocalNetHost(host))
   {
    if(mode==Real || mode==Spool)
       mode=RealNoCache;
    else
      {
       PrintMessage(Inform,"The request to fetch a page from the local network host '%s' is ignored.",host);
       exit(0);
      }
   }
 else if(mode==Real && IsNotCached(host,path))
   {
    mode=RealNoCache;
   }

 if(!strncasecmp(request,"POST",4))
   {
    if(mode==Spool)
       mode=SpoolGet;
   }
 else if(args && *args=='!')
   {
    if(mode==Real)
       mode=Spool;
    else if(mode==Fetch)
      {
       PrintMessage(Inform,"It is not possible to fetch a URL that was posted.");
       exit(0);
      }
   }

 if(strstr(request,"\nIf-Modified-Since:"))
   {
    char *bol=strstr(request,"\nIf-Modified-Since:")+1,*eol=strchr(bol,'\n');
    *eol++=0;

    if(mode==Spool)
      {
       spool=OpenWebpageSpoolFile(1,&host,path,args);

       if(spool!=-1)
         {
          struct stat buf;
          long since;

          buf.st_mtime=time(NULL)+1;
          fstat(spool,&buf);
          since=DateToTimeT(bol+19);

          close(spool);

          if(DateToTimeT(bol+19)>buf.st_mtime)
            {NotModified(client);exit(0);}
         }
      }

    while(*eol)
       *bol++=*eol++;
    *bol=0;
   }

 if(PragmaNoCache && strstr(request,"\nPragma: no-cache"))
   {
    if(mode==Spool || mode==SpoolGet)
      {
       spool=OpenWebpageSpoolFile(1,&host,path,args);

       if(spool==-1)
          mode=SpoolGet;
       else
         {
          mode=SpoolPragma;
          close(spool);
         }
      }
    /* (mode==Fetch || mode==Real) are left unchanged, not modified as below. */
   }
 else
   {
    spool=OpenWebpageSpoolFile(1,&host,path,args);

    if(mode==Spool && spool==-1)
       mode=SpoolGet;
    else if(mode==Fetch && spool!=-1)
      {
       if(RequestChanges(spool,&request)==1)
          OpenWebpageSpoolFile(-2,&host,path,args);
       else
          exit(0);
      }
    else if(mode==Real && spool!=-1)
      {
       if(RequestChanged && RequestChanges(spool,&request))
          OpenWebpageSpoolFile(-2,&host,path,args);
       else
          mode=Spool;
      }

    if(spool!=-1)
       close(spool);
   }

 /* Set up the file descriptor for the spool file. */

 if(mode==Real || mode==Fetch || mode==SpoolGet)
   {
    spool=OpenWebpageSpoolFile(0,&host,path,args);

    if(spool==-1)
      {
       if(mode==Real || mode==SpoolGet)
          ServerError(client,"Cannot open the spooled web page to write.");
       PrintMessage(Warning,"Cannot open the spooled web page to write.");
       exit(1);
      }
   }
 else if(mode==Spool || mode==SpoolPragma)
   {
    spool=OpenWebpageSpoolFile(1,&host,path,args);

    if(spool==-1)
      {
       ServerError(client,"Cannot open the spooled web page to read.");
       PrintMessage(Warning,"Cannot open the spooled web page to read.");
       exit(1);
      }
   }

 /* Set up the outgoing file. */

 if(mode==SpoolGet || mode==SpoolRefresh || mode==SpoolPragma)
   {
    outgoing=OpenOutgoingSpoolFile(0);

    if(outgoing==-1)
      {
       ServerError(client,"Cannot open the outgoing request to write.");
       PrintMessage(Warning,"Cannot open the outgoing request to write.");
       exit(1);
      }
   }

 /* Set up the file descriptor for the server host. */

 if(mode==Real || mode==RealNoCache || mode==Fetch)
   {
    char *colon;
    char *proxy=WhichProxy(host);

    if(proxy && !IsLocalNetHost(host))
      {
       server_host=(char*)malloc(strlen(proxy)+1);
       strcpy(server_host,proxy);
      }
    else
      {
       server_host=(char*)malloc(strlen(host)+1);
       strcpy(server_host,host);
       MakeRequestNonProxy(url,path,args,&request);
      }

    if((colon=strchr(server_host,':')))
      {
       *colon++=0;
       if(*colon)
          server_port=atoi(colon);
      }

    server=OpenClientSocket(server_host,server_port);

    if(server==-1)
      {
       if(mode==Fetch || mode==Real)
         {
          lseek(spool,0,SEEK_SET);
          ftruncate(spool,0);
          RemoteHostError(spool,url,"the server could not be connected.");
         }
       if(mode==Real || mode==RealNoCache)
          RemoteHostError(client,url,"the server could not be connected.");
       PrintMessage(Warning,"Cannot open a socket to the remote host %s port %d.",server_host,server_port);

       exit(1);
      }
   }

 /* Censor the header. */

 if(mode==Real || mode==RealNoCache || mode==SpoolGet || mode==SpoolRefresh || mode==SpoolPragma)
   {
    char *hostheader=(char*)malloc(strlen(host)+16);
    char *copy=(char*)malloc(strlen(request)+strlen(host)+16);
    char *bol,*lastbol;

    sprintf(hostheader,"Host: %s\r\n",host);
    strcpy(copy,request);
    bol=strchr(copy,'\n')+1; *bol=0;
    strcat(copy,hostheader);
    bol=strchr(request,'\n')+1;
    strcat(copy,bol);
    free(request);
    free(hostheader);
    request=copy;

    bol=lastbol=strchr(request,'\n')+1;
    while((bol=strchr(bol,'\n')) && (bol-lastbol)>2)
      {
       bol++;

       while(!strncmp("Host:",bol,5) || IsCensoredHeader(bol))
         {
          char *from=strchr(bol,'\n')+1,*to=bol;

          *(from-2)=0;
          if(strncmp("Host:",bol,5))
             PrintMessage(Debug,"Censored '%s'",bol);

          while(*from)
             *to++=*from++;
          *to=0;
          bol-=4;
         }

       lastbol=bol;
      }
   }

 /* Write request to remote server or outgoing file. */

 if(mode==Real || mode==RealNoCache || mode==Fetch)
   {
    int err=write(server,request,strlen(request));

    if(err==-1)
      {
       if(mode==Real || mode==Fetch)
         {
          lseek(spool,0,SEEK_SET);
          ftruncate(spool,0);
          RemoteHostError(spool,url,"the server could not be written to.");
         }
       if(mode==Real || mode==RealNoCache)
          RemoteHostError(client,url,"the server could not be written to.");
       PrintMessage(Warning,"Failed to write to remote host %s.",server_host);
       exit(1);
      }
    else if(mode==Fetch && client!=-1)
      {
       char *buffer=(char*)malloc(strlen(url)+16);
       sprintf(buffer,"Fetching %s\n",url);
       write(client,buffer,strlen(buffer));
       free(buffer);
      }
   }
 else if(mode==SpoolGet || mode==SpoolRefresh || mode==SpoolPragma)
   {
    int err=write(outgoing,request,strlen(request));

    if(err==-1)
      {
       ServerError(client,"Cannot write the outgoing request.");
       PrintMessage(Warning,"Cannot write the outgoing request.");
       exit(1);
      }
   }

 /* Parse the reply */

 if(mode==Real || mode==RealNoCache || mode==Fetch)
   {
    int status=0;

    reply=ParseReply(server,&status,&replylen);

    if(!reply)
      {
       if(mode==Real || mode==Fetch)
         {
          lseek(spool,0,SEEK_SET);
          ftruncate(spool,0);
          RemoteHostError(spool,url,"the server timed out before sending the reply.");
         }
       if(mode==Real || mode==RealNoCache)
          RemoteHostError(client,url,"the server timed out before sending the reply.");
       PrintMessage(Warning,"Timed out reading the reply.");
       exit(1);
      }

    if(mode==Fetch && (status==301 || status==302))
      {
       char *new_request;

       PrintMessage(Inform,"Server page has been moved.");

       new_request=RequestMoved(request,reply,host,path);
       if(!new_request)
          PrintMessage(Warning,"Cannot parse the reply for the new location.");
       else
         {
          int new_outgoing=OpenOutgoingSpoolFile(0);
          if(new_outgoing==-1)
             PrintMessage(Warning,"Cannot open the new outgoing request to write.");
          else
            {
             write(new_outgoing,new_request,strlen(new_request));
             CloseOutgoingSpoolFile(new_outgoing);
             free(new_request);
             fetch_again++;
            }
         }
      }
    else if(status==304)
      {
       PrintMessage(Inform,"Server page is not newer than the one in cache.");
       if(mode==Fetch)
          exit(0);
       if(mode==Real)
          mode=Spool;
      }

    if(replylen<=256)
       reply=(char*)realloc(reply,257);
   }
 else
    reply=(char*)malloc(257);

 /* Close the outgoing file if any. */

 if(outgoing_file)
    fclose(outgoing_file);
 else if(outgoing>=0)
    outgoing_filename=CloseOutgoingSpoolFile(outgoing);

 /* Read reply and process it. */

 if(mode==Real || mode==RealNoCache)
   {
    int n=replylen,err=0;

    do
      {
       if(mode==Real)
          write(spool,reply,n);
       err=write(client,reply,n);
      }
    while(err!=-1 && (n=ReadOrTimeout(server,reply,256))>0);

    if(mode==Real)
       if(err==-1 || n<0)
         {
          lseek(spool,0,SEEK_SET);
          ftruncate(spool,0);
          if(err==-1)
            {
             RemoteHostError(spool,url,"the connection was closed by the client during the transfer.");
             PrintMessage(Warning,"Error writing to client [%!s]; client disconnected?.");
             exit(1);
            }
          else
            {
             RemoteHostError(spool,url,"the server timed out during the transfer.");
             PrintMessage(Warning,"Timed out while reading from remote host.");
             exit(1);
            }
         }
       else
         {
          int tell=lseek(spool,0,SEEK_CUR);
          if(tell!=-1)
             ftruncate(spool,tell);
         }
   }
 else if(mode==Fetch)
   {
    int n=replylen;

    do
      {
       write(spool,reply,n);
      }
    while((n=ReadOrTimeout(server,reply,256))>0);

    if(n<0)
      {
       lseek(spool,0,SEEK_SET);
       ftruncate(spool,0);
       RemoteHostError(spool,url,"the server timed out during the transfer.");
       PrintMessage(Warning,"Timed out while reading from remote host.");
       exit(1);
      }
    else
      {
       int tell=lseek(spool,0,SEEK_CUR);
       if(tell!=-1)
          ftruncate(spool,tell);
      }

    if(FetchImages || FetchFrames)
      {
       lseek(spool,0,SEEK_SET);
       spool_file=fdopen(spool,"r");
       if(!spool_file)
         {PrintMessage(Warning,"Cannot use fdopen on the spool file descriptor.");exit(1);}

       if(ParseHTML(spool_file,host,path))
         {
          char **list;

          if(FetchImages && (list=ListImages()))
             for(n=0;list[n];n++)
               {
                char *new_request=RequestURL(list[n],url);
                int new_outgoing=OpenOutgoingSpoolFile(0);

                PrintMessage(Debug,"Image=%s",list[n]);

                if(new_outgoing==-1)
                   PrintMessage(Warning,"Cannot open the new outgoing request to write.");
                else
                  {
                   write(new_outgoing,new_request,strlen(new_request));
                   CloseOutgoingSpoolFile(new_outgoing);
                   fetch_again++;
                  }

                free(new_request);
               }

          if(FetchFrames && (list=ListFrames()))
             for(n=0;list[n];n++)
               {
                char *new_request=RequestURL(list[n],url);
                int new_outgoing=OpenOutgoingSpoolFile(0);

                PrintMessage(Debug,"Frame=%s",list[n]);

                if(new_outgoing==-1)
                   PrintMessage(Warning,"Cannot open the new outgoing request to write.");
                else
                  {
                   write(new_outgoing,new_request,strlen(new_request));
                   CloseOutgoingSpoolFile(new_outgoing);
                   fetch_again++;
                  }

                free(new_request);
               }
         }
      }
   }
 else if(mode==Spool || mode==SpoolPragma)
   {
    struct stat buf;
    char *head="HTTP/1.0 503 WWWOFFLE Remote Host Error\r\n"; /* This line must not be changed (see messages.c). */
    int n=read(spool,reply,64);

    if(!strncmp(reply,head,strlen(head)) || (!fstat(spool,&buf) && buf.st_size==0))
       OpenWebpageSpoolFile(-1,&host,path,args);

    if(AddInfoRefresh)
      {
       struct stat buf;
       int status=0;

       sscanf(reply,"%*s %d",&status);

       lseek(spool,0,SEEK_SET);
       spool_file=fdopen(spool,"r");
       if(!spool_file)
         {PrintMessage(Warning,"Cannot use fdopen on the spool file descriptor.");exit(1);}

       if((status>=200 && status<400) && ParseHTML(spool_file,host,path) && !fstat(spool,&buf))
         {
          time_t t_ago=time(NULL)-buf.st_mtime;
          int position=GetHTMLEnd();
          char *date=RFC822Date(buf.st_mtime,0),ago[16],*encurl=UrlEncode(url),*localhost=GetLocalHost();
          char *refresh=(char*)malloc(strlen(url)+3*strlen(localhost)+2*strlen(encurl)+256);
          char *msg=(char*)malloc(strlen(date)+strlen(url)+3*strlen(localhost)+2*strlen(encurl)+256);

          if(t_ago<0)
             sprintf(ago,"in the future");
          else if(t_ago<3600)
             sprintf(ago,"%ld min%s ago",t_ago/60,t_ago/60==1?"":"s");
          else if(t_ago<(24*3600))
             sprintf(ago,"%ld hour%s ago",t_ago/3600,t_ago/3600==1?"":"s");
          else if(t_ago<(14*24*3600))
             sprintf(ago,"%ld day%s ago",t_ago/(24*3600),t_ago/(24*3600)==1?"":"s");
          else if(t_ago<(30*24*3600))
             sprintf(ago,"%ld week%s ago",t_ago/(7*24*3600),t_ago/(7*24*3600)==1?"":"s");
          else
             sprintf(ago,"%ld month%s ago",t_ago/(30*24*3600),t_ago/(30*24*3600)==1?"":"s");

          if(!args)
             sprintf(refresh,"[<a href=\"http://%s:%d/control/delete?url=%s\">Delete</a>"
                             "|<a href=\"http://%s:%d/refresh/%s/%s\">Refresh</a>|"
                             "<a href=\"http://%s:%d/refresh/?%s\">Options</a>]",
                     localhost,HTTP_Port,encurl,
                     localhost,HTTP_Port,host,path[0]=='/'?"":path,
                     localhost,HTTP_Port,encurl);
          else if(*args=='!')
             sprintf(refresh,"[<a href=\"http://%s:%d/control/delete?url=%s\">Delete</a>"
                             "|Refresh|"
                             "Opt]",
                     localhost,HTTP_Port,encurl);
          else
             sprintf(refresh,"[<a href=\"http://%s:%d/control/delete?url=%s\">Delete</a>"
                             "|<a href=\"http://%s:%d/refresh/%s/%s?%s\">Refresh</a>|"
                             "<a href=\"http://%s:%d/refresh/?%s\">Options</a>]",
                     localhost,HTTP_Port,encurl,
                     localhost,HTTP_Port,host,path[0]=='/'?"":path,args,
                     localhost,HTTP_Port,encurl);

          sprintf(msg,"\n"
                      "<hr>\n"
                      "<p align=center>\n"
                      "WWWOFFLE - %s (%s) - %s - WWWOFFLE\n"
                      "</p>\n"
                      "<hr>\n",
                  date,ago,refresh);

          fseek(spool_file,0,SEEK_SET);
          while((reply=fgets_realloc(reply,spool_file)))
            {
             if(!strncmp(reply,"Content-Length:",15))
                sprintf(reply,"Content-Length: %d\r\n",atoi(&reply[16])+strlen(msg));
             write(client,reply,strlen(reply));
             if(reply[0]=='\r' || reply[0]=='\n')
                break;
            }
          reply=realloc(reply,257);

          while(position>0 && (n=fread(reply,1,position>256?256:position,spool_file))>0)
            {
             write(client,reply,n);
             position-=n;
            }

          write(client,msg,strlen(msg));

          while((n=fread(reply,1,256,spool_file))>0)
            {
             write(client,reply,n);
            }

          free(refresh);
          free(encurl);
          free(msg);
         }
       else
         {
          fseek(spool_file,0,SEEK_SET);
          while((n=fread(reply,1,256,spool_file))>0)
            {
             write(client,reply,n);
            }
         }
      }
    else
       do
         {
          write(client,reply,n);
         }
       while((n=read(spool,reply,256))>0);
   }
 else if(mode==SpoolGet)
   {
    int tell;

    WillGetURL(client,url,outgoing_filename,0);
    WillGetURL(spool,url,outgoing_filename,1);

    tell=lseek(spool,0,SEEK_CUR);
    if(tell!=-1)
       ftruncate(spool,tell);
   }
 else if(mode==RealRefresh || mode==SpoolRefresh)
   {
    RefreshRedirect(client,url);
   }

 /* Close down and exit. */

 if(server>=0)
    CloseSocket(server);

 if(spool_file)
    fclose(spool_file);
 else if(spool>=0)
    close(spool);

 if(request)
    free(request);

 if(reply)
    free(reply);

 if(client_file)
    fclose(client_file);
 else if(client>=0)
    close(client);

 if(fetch_again)
    return(4);
 else
    return(0);
}


/*++++++++++++++++++++++++++++++++++++++
  Uninstall the signal handlers.
  ++++++++++++++++++++++++++++++++++++++*/

static void uninstall_sighandlers(void)
{
 struct sigaction action;

 /* SIGCHLD */
 action.sa_handler = SIG_DFL;
 sigemptyset (&action.sa_mask);
 action.sa_flags = 0;
 if(sigaction(SIGCHLD, &action, NULL) != 0)
    PrintMessage(Warning, "Cannot uninstall SIGCHLD handler.");

 /* SIGINT, SIGQUIT, SIGTERM */
 action.sa_handler = SIG_DFL;
 sigemptyset(&action.sa_mask);
 action.sa_flags = 0;
 if(sigaction(SIGINT, &action, NULL) != 0)
    PrintMessage(Warning, "Cannot uninstall SIGINT handler.");
 if(sigaction(SIGQUIT, &action, NULL) != 0)
    PrintMessage(Warning, "Cannot uninstall SIGQUIT handler.");
 if(sigaction(SIGTERM, &action, NULL) != 0)
    PrintMessage(Warning, "Cannot uninstall SIGTERM handler.");

 /* SIGHUP */
 action.sa_handler = SIG_DFL;
 sigemptyset(&action.sa_mask);
 action.sa_flags = 0;
 if(sigaction(SIGHUP, &action, NULL) != 0)
    PrintMessage(Warning, "Cannot uninstall SIGHUP handler.");

 /* SIGPIPE */
 action.sa_handler = SIG_IGN;
 sigemptyset (&action.sa_mask);
 action.sa_flags = 0;
 if(sigaction(SIGPIPE, &action, NULL) != 0)
    PrintMessage(Warning, "Cannot ignore SIGPIPE.");
}
