/***************************************
  $Header: /home/amb/wwwoffle/RCS/spool.c 1.16 1997/09/06 19:58:41 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 1.3.
  Handle all of the spooling of files in the spool directory.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1996,97 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <time.h>
#include <utime.h>
#include <sys/stat.h>
#include <dirent.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <ctype.h>

#include "wwwoffle.h"
#include "errors.h"
#include "config.h"



/*++++++++++++++++++++++++++++++++++++++
  Open a file in the outgoing directory to write into / read from.

  int OpenOutgoingSpoolFile Returns a file descriptor, or -1 on failure.

  int rw Set to true to read, else false.
  ++++++++++++++++++++++++++++++++++++++*/

int OpenOutgoingSpoolFile(int rw)
{
 int fd=-1;

 /* Open the first file for read, or a temporary file for write */
 if (rw)
   {
    /* Open the outgoing directory */
    struct dirent* ent;
    DIR *dir=opendir("outgoing");

    if(!dir)
      {
       PrintMessage(Inform,"Cannot open outgoing directory [%!s]; creating one.");
       if(mkdir("outgoing",0755) || !(dir=opendir("outgoing")))
         {PrintMessage(Warning,"Cannot create or access outgoing directory [%!s].");return(-1);}
      }

    /* Walk through the directory and try to open/unlink one entry after the other.
     * Only use files with names that start with a digit.
     */
    while((ent=readdir(dir))!=0)
      {
       char name [30];

       /* Check for illegal names */
       if(!isdigit(ent->d_name[0])||strlen(ent->d_name)>15)
          continue;
       /* Open the file (don't bother to stat it) */
       strcpy(name,"outgoing/");
       strcat(name,ent->d_name);
       if((fd=open(name,O_RDONLY))==-1)
          PrintMessage(Inform,"Cannot open file '%s' [%!s]; race condition?",name);
       else
          if(unlink(name))
            {
             PrintMessage(Inform,"Cannot unlink file '%s' [%!s]; race condition?",name);
             close(fd); fd=-1;
            }
          else
             break;
      }
      closedir(dir);
   }
 else
   {
    char name[24];

    sprintf(name,"outgoing/tmp.%d",getpid());

    fd=open(name,O_WRONLY|O_CREAT,0644);
    if(fd==-1)
       PrintMessage(Inform,"Cannot open file '%s' [%!s]",name);
   }

 return(fd);
}


/*++++++++++++++++++++++++++++++++++++++
  Close the temporary outgoing spool file and rename it.

  char *CloseOutgoingSpoolFile Return the name of the file in the outgoing directory.

  int fd The file descriptor that was being written to.
  ++++++++++++++++++++++++++++++++++++++*/

char *CloseOutgoingSpoolFile(int fd)
{
 static int number=1;
 int i;
 int pid=getpid();
 char oldname[40];
 static char newname[40];

 /* Close the file */
 close(fd);

 /* Make the current name of the file. */
 sprintf(oldname,"outgoing/tmp.%d",pid);

 i=MaxServers*4;
 while(i-->0)
   {
    /* Make the new name of the file. To avoid clashes and long file
     * searches, filenames have the form pid.number, where pid is the
     * process id of the current process, and number is some serial number.
     */
    sprintf(newname,"outgoing/%d.%d",pid,number++);

    /* Link the file to it's new name. This checks if the new name exists
     * while avoiding any race conditions. If the link succeeds, remove the
     * old name.
     */
    if(link(oldname,newname)==0)
      {
       if(remove(oldname)!=0)
          PrintMessage(Warning,"Cannot remove file '%s' [%!s].",oldname);
       break;
      }

    /* We had an error trying to link the file. If this is not the error, we
     * expected, something is wrong.
     */
    if(errno!=EEXIST)
      {
       PrintMessage(Warning,"Could not link '%s' to '%s' [%!s].",oldname,newname);
      }
   }

 /* Check if we could rename the file */
 if(i<=0)
   {
    /* Print a warning and delete the old file. */
    PrintMessage(Warning,"Unable to find temporary name for %s",oldname);
    if(unlink(oldname))
       PrintMessage(Warning,"Cannot unlink outgoing request '%s' [%!s].",oldname);
    return(NULL);
   }
 else
   /* Return the new name */
   return(&newname[9]);
}



/*++++++++++++++++++++++++++++++++++++++
  Delete a specified file from the cache.

  int DeleteOutgoingSpoolFile Return 1 if the file from the cache was also deleted.

  char *name The name of the file to delete.

  char **url The name of the page that the file had requested.
  ++++++++++++++++++++++++++++++++++++++*/

int DeleteOutgoingSpoolFile(char *name,char **url)
{
 char *path=(char*)malloc(strlen(name)+16);
 FILE *file=NULL;
 int ret=0;

 strcpy(path,"outgoing/");
 strcat(path,name);

 file=fopen(path,"r");
 if(file)
   {
    char *req;

    req=ParseRequest(file,url);

    if(url)
      {
       char *host,*path,*args;
       int spool;

       SplitURL(*url,&host,&path,&args);
       spool=OpenWebpageSpoolFile(1,&host,path,args);

       if(spool!=-1)
         {
          char reply[65],*head="HTTP/1.0 404 WWWOFFLE Will Get\r\n"; /* This line must not be changed (see messages.c). */

          read(spool,reply,64);
          close(spool);

          if(!strncmp(reply,head,strlen(head)))
            {
             ret=1;
             OpenWebpageSpoolFile(-1,&host,path,args);
            }
         }
      }

    if(req)
       free(req);

    fclose(file);
   }

 if(unlink(path))
   PrintMessage(Warning,"Cannot unlink outgoing request '%s' [%!s].",name);

 return(ret);
}


/*++++++++++++++++++++++++++++++++++++++
  Open a file in a spool subdirectory to write into / read from / delete.

  int OpenWebpageSpoolFile Returns a file descriptor.

  int rw Set to 1 to read, 0 to write, -1 to delete, -2 to touch.

  char **host The hostname from the URL.

  char *path The pathname from the URL.

  char *args The arguments to the URL.
  ++++++++++++++++++++++++++++++++++++++*/

int OpenWebpageSpoolFile(int rw,char **host,char *path,char *args)
{
 struct stat buf;
 int fd=-1;
 char *file,*newpath=(char*)malloc(strlen(path)+2);
 unsigned long hash=0;
 int i;

 /* Modify the path and args. */

 if(!*path)
   {
    newpath[0]=PATH_SEP;
    newpath[1]=0;
   }
 else
   {
    for(i=0;path[i];i++)
       if(path[i]=='/')
          newpath[i]=PATH_SEP;
       else
          newpath[i]=path[i];
    newpath[i]=0;
   }

 if(!strcmp(path,"."))
   {newpath[0]=PATH_SEP;newpath[1]='.';newpath[2]=0;}

 if(args)
    hash=MakeHashFromArgs(args);

 /* Check if the host is a link. */

 while(!lstat(*host,&buf) && S_ISLNK(buf.st_mode))
   {
    char link[257];
    int n=readlink(*host,link,256);
    if(n==-1)
      {PrintMessage(Warning,"The file '%s' is a link that cannot be read.",*host);return(-1);}
    link[n]=0;
    *host=(char*)malloc(n+1);
    strcpy(*host,link);
   }

 /* Create or check for the directory for the spool file. */

 if(lstat(*host,&buf))
   {
    PrintMessage(Inform,"Directory '%s' does not exist [%!s]; creating one.",*host);
    if(mkdir(*host,0755))
      {PrintMessage(Warning,"Cannot create directory '%s' [%!s].",*host);return(-1);}
   }
 else
    if(!S_ISDIR(buf.st_mode))
      {PrintMessage(Warning,"The file '%s' is not a directory.",*host);return(-1);}

 /* Create the file for the web page. */

 file=(char*)malloc(strlen(*host)+strlen(path)+16);

 if(args)
    sprintf(file,"%s/%s%c%c%08lx",*host,newpath,PATH_SEP,PATH_SEP,hash);
 else
    sprintf(file,"%s/%s",*host,newpath);

 if(rw==0)
    fd=open(file,O_RDWR|O_CREAT,0644);
 else if(rw==1)
    fd=open(file,O_RDONLY);
 else if(rw==-1)
    unlink(file);
 else if(rw==-2)
    utime(file,NULL);

 if(args && (rw<0 || fd!=-1))
   {
    sprintf(file,"%s/%c%c%08lx",*host,PATH_SEP,PATH_SEP,hash);

    if(rw==0)
      {
       int afd=open(file,O_WRONLY|O_CREAT|O_TRUNC,0644);
       if(afd!=-1)
         {
          write(afd,args,strlen(args));
          close(afd);
         }
       else
         {
          close(fd);
          fd=-1;
         }
      }
    else if(rw==1)
      {
       int afd=open(file,O_RDONLY);
       if(afd!=-1)
          close(afd);
       else
         {
          close(fd);
          fd=-1;
         }
      }
    else if(rw==-1)
       unlink(file);
    else if(rw==-2)
       utime(file,NULL);
   }

 free(file);
 free(newpath);

 /* Change the modification time on the directory. */

 if((rw==0 && fd!=-1) || rw==-1 || rw==-2)
    utime(*host,NULL);

 return(fd);
}
