/***************************************
  $Header: /home/amb/wwwoffle/RCS/index.c 1.27 1997/10/06 17:57:02 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 1.3a.
  Generate an index of the web pages that are cached in the system.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1997 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <time.h>
#include <utime.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <fcntl.h>
#include <unistd.h>

#include "wwwoffle.h"
#include "config.h"
#include "errors.h"


/*+ A type to contain the information required to sort the files. +*/
typedef struct _FileIndex
{
 char *name;                    /*+ The URL of the file. +*/
 char *file;                    /*+ The name of the file. +*/
 char *type;                    /*+ The type (file extension) of the URL. +*/
 long time;                     /*+ The time of the file (access or modification as appropriate). +*/
}
FileIndex;

/*+ The method of sorting used. +*/
static char *sorttype[][2]={{"none"  ,"Unsorted"},
                            {"mtime" ,"Modification&nbsp;Time"},
                            {"atime" ,"Access&nbsp;Time"},
                            {"alpha" ,"Alphabetical"},
                            {"type"  ,"File&nbsp;Type"},
                            {"latest","Latest"}};

/*+ An enumerated type for the sort modes. +*/
typedef enum _SortMode
{
 None,                          /*+ No sorting, natural order. +*/
 MTime,                         /*+ Sort by modification time +*/
 ATime,                         /*+ Sort by access time. +*/
 Alpha,                         /*+ Sort Alphabetically. +*/
 Type,                          /*+ Sort by type (file extension). +*/
 Latest,                        /*+ Sort all files in one list by modification time. +*/
 NSortModes                     /*+ The number of sort modes. +*/
}
SortMode;


/*+ The list of files. +*/
static FileIndex **files=NULL;
int nfiles=0;


static void IndexRoot(int fd,char *args,SortMode mode);
static void IndexDir(int fd,char *dirname,SortMode mode);
static void IndexOutgoing(int fd,SortMode mode);
static void IndexLatest(int fd);

static void CorrectIndex(int fd,char *path,char *args);
static void IllegalIndex(int fd,char *path,char *args);

static void add_dir(char *name,struct stat *buf,SortMode mode);
static void add_file(char *name,struct stat *buf,SortMode mode,char *dir,char *outgoing);

static int sort_alpha(FileIndex **a,FileIndex **b);
static int sort_type(FileIndex **a,FileIndex **b);
static int sort_time(FileIndex **a,FileIndex **b);


/*++++++++++++++++++++++++++++++++++++++
  Generate an index of the pages that are in the cache.

  int fd The file descriptor to write the output to.

  char *url The URL that specifies the path to generate the index for.
  ++++++++++++++++++++++++++++++++++++++*/

void IndexPage(int fd,char *path,char *args)
{
 char *head=
 "HTTP/1.0 200 WWWOFFLE Index\r\n"
 "Content-type: text/html\r\n"
 "\r\n"
 "<HTML>\n"
 "<HEAD>\n"
 "<TITLE>\n"
 "WWWOFFLE - Index\n"
 "</TITLE>\n"
 "</HEAD>\n"
 "<BODY>\n"
 "<H1 align=center>WWWOFFLE Cache Index</H1>\n"
 "<p align=center>\n";
 char *tail=
 "</BODY>\n"
 "</HTML>\n";
 SortMode sort=NSortModes;
 char *newpath=(char*)malloc(strlen(path)+1);
 char *buffer=(char*)malloc(strlen(path)+80);
 struct stat buf;

 strcpy(newpath,path);

 if(*newpath && newpath[strlen(newpath)-1]=='/')
    newpath[strlen(newpath)-1]=0;

 if(args)
    for(sort=0;sort<NSortModes;sort++)
       if(!strcmp(args,sorttype[sort][0]))
          break;

 if(strchr(newpath,'/') || !strcmp(newpath,"..") || !strcmp(newpath,".") ||
    (*newpath && (stat(newpath,&buf) || !S_ISDIR(buf.st_mode))))
    IllegalIndex(fd,newpath,args);
 else if(sort==NSortModes)
    CorrectIndex(fd,newpath,args);
 else
   {
    SortMode mode;

    write(fd,head,strlen(head));

    for(mode=0;mode<NSortModes;mode++)
      {
       if(*newpath && mode==Latest)
          continue;
       if(mode==sort)
          sprintf(buffer,"[%s]\n",sorttype[mode][1]);
       else
          sprintf(buffer,"<a href=\"/index/%s?%s\">[%s]</a>\n",newpath,sorttype[mode][0],sorttype[mode][1]);
       write(fd,buffer,strlen(buffer));
      }

    if(!*newpath)
      {
       if(sort==Latest)
          IndexLatest(fd);
       else
          IndexRoot(fd,args,sort);
      }
    else if(!strcmp(newpath,"outgoing"))
       IndexOutgoing(fd,sort);
    else
       IndexDir(fd,newpath,sort);

    write(fd,tail,strlen(tail));
   }

 free(newpath);
 free(buffer);
}


/*++++++++++++++++++++++++++++++++++++++
  Index the root of the cache.

  int fd The file descriptor to write to.

  char *args The argument to put in the index for sorting.

  SortMode mode The sort mode to use.
  ++++++++++++++++++++++++++++++++++++++*/

static void IndexRoot(int fd,char *args,SortMode mode)
{
 DIR *dir;
 struct dirent* ent;
 int i;
 char buffer[80];

 /* Open the spool directory. */

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open spool directory [%!s]; index failed.");return;}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read spool directory [%!s]; index failed.");closedir(dir);return;}
 ent=readdir(dir);  /* skip .. */

 /* Print all of the sub directories. */

 while((ent=readdir(dir)))
   {
    struct stat buf;

    if(stat(ent->d_name,&buf))
       PrintMessage(Inform,"Cannot stat file '%s' [%!s]; race condition?",ent->d_name);
    else if(S_ISDIR(buf.st_mode) && strcmp(ent->d_name,"outgoing"))
       add_dir(ent->d_name,&buf,mode);
   }

 closedir(dir);

 /* Sort the files. */

 if(mode==MTime || mode==ATime)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_time);
 else if(mode==Alpha || mode==Type)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_alpha);

 sprintf(buffer,"<p>\n<b>Server Hosts</b>\n");
 write(fd,buffer,strlen(buffer));

 if(nfiles)
   {
    sprintf(buffer,"<p>\n%d Hosts\n",nfiles);
    write(fd,buffer,strlen(buffer));

    write(fd,"<ul>\n",5);

    for(i=0;i<nfiles;i++)
      {
       char *buffer=(char*)malloc(2*strlen(files[i]->name)+strlen(args)+40);

       sprintf(buffer,"<li><a href=\"/index/%s?%s\">%s</a>\n",files[i]->name,args,files[i]->name);
       write(fd,buffer,strlen(buffer));

       free(buffer);
       free(files[i]->name);
       free(files[i]);
      }

    write(fd,"</ul>\n",6);
   }
 else
    write(fd,"<p>\nNo hosts.\n",14);

 sprintf(buffer,"<p align=center>\n<a href=\"/index/outgoing?%s\">[Outgoing&nbsp;Requests]</a>\n",sorttype[mode][0]);
 write(fd,buffer,strlen(buffer));

 free(files);
 files=NULL;
 nfiles=0;
}


/*++++++++++++++++++++++++++++++++++++++
  Create an index of a subdirectory.

  int fd The file descriptor to write to.

  char *dirname The name of the subdirectory.

  SortMode mode The sort mode to use.
  ++++++++++++++++++++++++++++++++++++++*/

static void IndexDir(int fd,char *dirname,SortMode mode)
{
 DIR *dir;
 struct dirent* ent;
 int i;
 char *buffer;
 char name[257],link[257];
 struct stat buf;

 /* Check if this is a link. */

 strcpy(name,dirname);
 link[0]=0;

 while(!lstat(name,&buf) && S_ISLNK(buf.st_mode))
   {
    int n;

    if((n=readlink(name,link,256))==-1)
      {PrintMessage(Warning,"Cannot read link '%s' [%!s].",name);*link=0;break;}

    link[n]=0;
    strcpy(name,link);
   }

 /* Open the spool subdirectory. */

 if(chdir(dirname))
   {PrintMessage(Warning,"Cannot change to directory '%s' [%!s]; not indexed.",dirname);return;}

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open directory '%s' [%!s]; not indexed.",dirname);chdir("..");return;}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read directory '%s' [%!s]; not indexed.",dirname);closedir(dir);chdir("..");return;}
 ent=readdir(dir);  /* skip .. */

 /* Print all of the file names. */

 while((ent=readdir(dir)))
   {
    struct stat buf;

    if(stat(ent->d_name,&buf))
       PrintMessage(Inform,"Cannot stat file '%s/%s' [%!s]; race condition?",dirname,ent->d_name);
    else if(S_ISREG(buf.st_mode))
       add_file(ent->d_name,&buf,mode,NULL,NULL);
   }

 closedir(dir);

 chdir("..");

 /* Sort the files. */

 if(mode==MTime || mode==ATime)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_time);
 else if(mode==Alpha)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_alpha);
 else if(mode==Type)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_type);

 buffer=(char*)malloc(80+strlen(dirname));
 sprintf(buffer,"<p>\n<b>http://%s/</b>\n",dirname);
 write(fd,buffer,strlen(buffer));
 free(buffer);

 if(*link)
   {
    buffer=(char*)malloc(80+strlen(link));
    sprintf(buffer,"=&gt; <i>http://%s/</i>\n",link);
    write(fd,buffer,strlen(buffer));
    free(buffer);
   }

 if(nfiles)
   {
    buffer=(char*)malloc(24);
    sprintf(buffer,"<p>\n%d Pages\n",nfiles);
    write(fd,buffer,strlen(buffer));
    free(buffer);

    write(fd,"<ul>\n",5);

    for(i=0;i<nfiles;i++)
      {
       char *url=(char*)malloc(strlen(files[i]->name)+strlen(dirname)+8),*encurl,*buffer;

       strcpy(url,dirname);
       strcat(url,"/");
       if(files[i]->name[0]=='/')
          strcat(url,&files[i]->name[1]);
       else
          strcat(url,files[i]->name);
       encurl=UrlEncode(url);

       buffer=(char*)malloc(3*strlen(files[i]->name)+2*strlen(dirname)+2*strlen(encurl)+256);
       if(strstr(files[i]->name,"?!"))
          sprintf(buffer,"<li>[Del"
                         "|Refresh|"
                         "Opt]"
                         "&nbsp;<a href=\"http://%s/%s\">%s</a>\n",
                  dirname,files[i]->name[0]=='/'?&files[i]->name[1]:files[i]->name,
                  files[i]->name);
       else
          sprintf(buffer,"<li>[<a href=\"/control/delete?url=%s\">Del</a>"
                         "|<a href=\"/refresh/%s/%s\">Refresh</a>|"
                         "<a href=\"/refresh/?%s\">Opt</a>]"
                         "&nbsp;<a href=\"http://%s/%s\">%s</a>\n",
                  encurl,
                  dirname,files[i]->name[0]=='/'?&files[i]->name[1]:files[i]->name,
                  encurl,
                  dirname,files[i]->name[0]=='/'?&files[i]->name[1]:files[i]->name,files[i]->name);
       write(fd,buffer,strlen(buffer));

       free(url);
       free(encurl);
       free(buffer);
       free(files[i]->name);
       free(files[i]);
      }

    write(fd,"</ul>\n",6);
   }
 else
    write(fd,"<p>\nNo pages on this host.\n",27);

 buffer=(char*)malloc(80);
 sprintf(buffer,"<p align=center><a href=\"/index/?%s\">[Back to the Hosts Index]</a></p>\n",sorttype[mode][0]);
 write(fd,buffer,strlen(buffer));
 free(buffer);

 free(files);
 files=NULL;
 nfiles=0;
}


/*++++++++++++++++++++++++++++++++++++++
  Create an index of the requests that are waiting in the outgoing directory.

  int fd The file descriptor to write to.

  SortMode mode The method to use to sort the names.
  ++++++++++++++++++++++++++++++++++++++*/

static void IndexOutgoing(int fd,SortMode mode)
{
 DIR *dir;
 struct dirent* ent;
 int i;
 char buffer[80];

 /* Open the spool subdirectory. */

 if(chdir("outgoing"))
   {PrintMessage(Warning,"Cannot change to directory 'outgoing' [%!s]; not indexed.");return;}

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open directory 'outgoing' [%!s]; not indexed.");chdir("..");return;}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read directory 'outgoing' [%!s]; not indexed.");closedir(dir);chdir("..");return;}
 ent=readdir(dir);  /* skip .. */

 /* Print all of the file names. */

 while((ent=readdir(dir)))
   {
    struct stat buf;

    if(stat(ent->d_name,&buf))
       PrintMessage(Inform,"Cannot stat file 'outgoing/%s' [%!s]; race condition?",ent->d_name);
    else if(S_ISREG(buf.st_mode) && strncmp(ent->d_name,"tmp",3))
      {
       FILE *file=fopen(ent->d_name,"r");

       if(file)
         {
          struct utimbuf utbuf;
          char *url,*req;

          req=ParseRequest(file,&url);
          if(url)
            {
             char *host,*path,*args;

             SplitURL(url,&host,&path,&args);
             if(args)
                sprintf(url,"%s/%s?%s",host,*path=='/'?&path[1]:path,args);
             else
                sprintf(url,"%s/%s",host,*path=='/'?&path[1]:path);

             add_file(url,&buf,mode,NULL,ent->d_name);
             free(url);
            }

          fclose(file);
          if(req)
             free(req);

          utbuf.actime=buf.st_atime;
          utbuf.modtime=buf.st_mtime;
          utime(ent->d_name,&utbuf);
         }
      }
   }

 closedir(dir);

 chdir("..");

 /* Sort the files. */

 if(mode==MTime || mode==ATime)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_time);
 else if(mode==Alpha)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_alpha);
 else if(mode==Type)
    qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_type);

 sprintf(buffer,"<p>\n<b>Outgoing Requests</b>\n");
 write(fd,buffer,strlen(buffer));

 if(nfiles)
   {
    sprintf(buffer,"<p>\n%d Requests\n",nfiles);
    write(fd,buffer,strlen(buffer));

    write(fd,"<ul>\n",5);

    for(i=0;i<nfiles;i++)
      {
       char *buffer,*encurl;

       encurl=UrlEncode(files[i]->name);

       buffer=(char*)malloc(2*strlen(files[i]->name)+strlen(files[i]->file)+strlen(encurl)+256);
       if(strstr(files[i]->name,"?!"))
          sprintf(buffer,"<li>[<a href=\"/control/delete?req=%s\">Del</a>"
                         "|Refresh|"
                         "Opt]"
                         "&nbsp;<a href=\"http://%s\">%s</a>\n",
                  files[i]->file,
                  files[i]->name,files[i]->name);
       else
          sprintf(buffer,"<li>[<a href=\"/control/delete?req=%s\">Del</a>"
                         "|Refresh|"
                         "<a href=\"/refresh/?%s\">Opt</a>]"
                         "&nbsp;<a href=\"http://%s\">%s</a>\n",
                  files[i]->file,
                  encurl,
                  files[i]->name,files[i]->name);
       write(fd,buffer,strlen(buffer));

       free(encurl);
       free(buffer);
       free(files[i]->name);
       free(files[i]->file);
       free(files[i]);
      }

    write(fd,"</ul>\n",6);
   }
 else
    write(fd,"<p>\nNo requests in outgoing directory.\n",39);

 sprintf(buffer,"<p align=center><a href=\"/index/?%s\">[Back to the Hosts Index]</a></p>\n",sorttype[mode][0]);
 write(fd,buffer,strlen(buffer));

 free(files);
 files=NULL;
 nfiles=0;
}


/*++++++++++++++++++++++++++++++++++++++
  Create an index of the latest pages.

  int fd The file descriptor to write to.
  ++++++++++++++++++++++++++++++++++++++*/

static void IndexLatest(int fd)
{
 DIR *dir;
 struct dirent* ent;
 int i;
 time_t now=time(NULL);
 int lastdays=0,lasthours=0;
 char buffer[80];

 /* Open the spool directory. */

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open spool directory [%!s]; index failed.");return;}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read spool directory [%!s]; index failed.");closedir(dir);return;}
 ent=readdir(dir);  /* skip .. */

 /* Print all of the sub directories. */

 while((ent=readdir(dir)))
   {
    struct stat buf;

    if(lstat(ent->d_name,&buf))
       PrintMessage(Inform,"Cannot stat file '%s' [%!s]; race condition?",ent->d_name);
    else
       if(S_ISDIR(buf.st_mode) && strcmp(ent->d_name,"outgoing") && (now-buf.st_mtime)<=IndexLatestDays*(24*3600))
         {
          DIR *dir2;
          struct dirent* ent2;
          struct utimbuf utbuf;

          if(chdir(ent->d_name))
            {PrintMessage(Warning,"Cannot change to directory '%s' [%!s]; not indexed.",ent->d_name);continue;}

          dir2=opendir(".");
          if(!dir2)
            {PrintMessage(Warning,"Cannot open directory '%s' [%!s]; not indexed.",ent->d_name);chdir("..");continue;}

          ent2=readdir(dir2);  /* skip .  */
          if(!ent2)
            {PrintMessage(Warning,"Cannot read directory '%s' [%!s]; not indexed.",ent->d_name);closedir(dir2);chdir("..");continue;}
          ent2=readdir(dir2);  /* skip .. */

          while((ent2=readdir(dir2)))
            {
             struct stat buf;

             if(stat(ent2->d_name,&buf))
                PrintMessage(Inform,"Cannot stat file '%s/%s' [%!s]; race condition?",ent->d_name,ent2->d_name);
             else if(S_ISREG(buf.st_mode) && (now-buf.st_mtime)<=IndexLatestDays*(24*3600))
                add_file(ent2->d_name,&buf,Latest,ent->d_name,NULL);
            }

          closedir(dir2);
          chdir("..");

          utbuf.actime=buf.st_atime;
          utbuf.modtime=buf.st_mtime;
          utime(ent->d_name,&utbuf);
         }
   }

 closedir(dir);

 /* Sort the files. */

 qsort(files,nfiles,sizeof(FileIndex*),(int (*)(const void*,const void*))sort_time);

 sprintf(buffer,"<p>\n<b>Latest Pages</b>\n<ul>\n");
 write(fd,buffer,strlen(buffer));

 for(i=0;i<nfiles;i++)
   {
    char *encurl=UrlEncode(files[i]->name);
    char *buffer=(char*)malloc(3*strlen(files[i]->name)+2*strlen(encurl)+256);
    int days=(now-files[i]->time)/(24*3600),hours=(now-files[i]->time)/3600;

    if(lastdays<days)
      {
       write(fd,"</ul>\n",6);
       for(;lastdays<days;lastdays++)
          write(fd,"<hr width=\"50%\">\n",17);
       write(fd,"<ul>\n",5);
      }
    else if(i && lasthours<hours)
      {
       write(fd,"</ul>\n",6);
       write(fd,"<p>\n",4);
       write(fd,"<ul>\n",5);
      }
    lasthours=hours;

    if(strstr(files[i]->name,"?!"))
       sprintf(buffer,"<li>[<a href=\"/control/delete?url=%s\">Del</a>"
                      "|Refresh|"
                      "Opt]"
                      "&nbsp;<a href=\"http://%s\">%s</a>\n",
               encurl,
               files[i]->name,files[i]->name);
    else
       sprintf(buffer,"<li>[<a href=\"/control/delete?url=%s\">Del</a>"
                      "|<a href=\"/refresh/%s\">Refresh</a>|"
                      "<a href=\"/refresh/?%s\">Opt</a>]"
                      "&nbsp;<a href=\"http://%s\">%s</a>\n",
               encurl,
               files[i]->name,
               encurl,
               files[i]->name,files[i]->name);
    write(fd,buffer,strlen(buffer));

    free(encurl);
    free(buffer);
    free(files[i]->name);
    free(files[i]);
   }

 if(++lastdays<IndexLatestDays)
   {
    write(fd,"</ul>\n",6);
    for(;lastdays<IndexLatestDays;lastdays++)
       write(fd,"<hr width=\"50%\">\n",17);
    write(fd,"<ul>\n",5);
   }

 write(fd,"</ul>\n",6);

 sprintf(buffer,"<p align=center><a href=\"/index/?%s\">[Back to the Hosts Index]</a></p>\n","none");
 write(fd,buffer,strlen(buffer));

 free(files);
 files=NULL;
 nfiles=0;
}


/*++++++++++++++++++++++++++++++++++++++
  Redirect the user to the correct index page in case of a bad argument.

  int fd The file descriptor to write to.

  char *path The specified path.

  char *args The specified args.
  ++++++++++++++++++++++++++++++++++++++*/

static void CorrectIndex(int fd,char *path,char *args)
{
 char *string=(char*)malloc(2*strlen(sorttype[0][0])+2*strlen(path)+(args?strlen(args):0)+40);
 char *head=
 "HTTP/1.0 301 WWWOFFLE Index Incorrect\r\n"
 "Content-type: text/html\r\n";
 char *head2=
 "\r\n"
 "<HTML>\n"
 "<HEAD>\n"
 "<TITLE>\n"
 "WWWOFFLE - Index Incorrect\n"
 "</TITLE>\n"
 "</HEAD>\n"
 "<BODY>\n"
 "<H1 align=center>WWWOFFLE Index Incorrect</H1>\n"
 "<p align=center>\n"
 "Your request for the index URL\n"
 "<br><b><tt>\n";
 char *middle=
 "\n"
 "</tt></b><br>\n"
 "was incorrect, select the link below for a correct version.\n"
 "<br>\n";
 char *tail=
 "\n"
 "</BODY>\n"
 "</HTML>\n";

 write(fd,head,strlen(head));
 sprintf(string,"Location: /index/%s?%s\r\n",path,sorttype[0][0]);
 write(fd,string,strlen(string));
 write(fd,head2,strlen(head2));
 if(args)
    sprintf(string,"/index/%s?%s",path,args);
 else
    sprintf(string,"/index/%s",path);
 write(fd,string,strlen(string));
 write(fd,middle,strlen(middle));
 sprintf(string,"<a href=\"/index/%s?%s\">/index/%s?%s</a>",path,sorttype[0][0],path,sorttype[0][0]);
 write(fd,string,strlen(string));
 write(fd,tail,strlen(tail));

 free(string);
}


/*++++++++++++++++++++++++++++++++++++++
  Inform the user that the specified index page is illegal.

  int fd The file descriptor to write to.

  char *path The specified path.

  char *args The specified args.
  ++++++++++++++++++++++++++++++++++++++*/

static void IllegalIndex(int fd,char *path,char *args)
{
 char *string=(char*)malloc(2*strlen(sorttype[0][0])+2*strlen(path)+(args?strlen(args):0)+40);
 char *head=
 "HTTP/1.0 404 WWWOFFLE Illegal Index\r\n"
 "Content-type: text/html\r\n"
 "\r\n"
 "<HTML>\n"
 "<HEAD>\n"
 "<TITLE>\n"
 "WWWOFFLE - Illegal Index\n"
 "</TITLE>\n"
 "</HEAD>\n"
 "<BODY>\n"
 "<H1 align=center>WWWOFFLE Illegal Index</H1>\n"
 "<p align=center>\n"
 "Your request for the index URL\n"
 "<br><b><tt>\n";
 char *middle=
 "\n"
 "</tt></b><br>\n"
 "is illegal, select the link below for the main index.\n"
 "<br>\n";
 char *tail=
 "\n"
 "</BODY>\n"
 "</HTML>\n";

 write(fd,head,strlen(head));
 if(args)
    sprintf(string,"/index/%s?%s",path,args);
 else
    sprintf(string,"/index/%s",path);
 write(fd,string,strlen(string));
 write(fd,middle,strlen(middle));
 sprintf(string,"<a href=\"/index/?%s\">/index/?%s</a>",sorttype[0][0],sorttype[0][0]);
 write(fd,string,strlen(string));
 write(fd,tail,strlen(tail));

 free(string);
}


/*++++++++++++++++++++++++++++++++++++++
  Add a directory to the list.

  char *name The name of the directory.

  struct stat *buf The stat result on the file.

  SortMode mode The sort mode.
  ++++++++++++++++++++++++++++++++++++++*/

static void add_dir(char *name,struct stat *buf,SortMode mode)
{
 if(!(nfiles%16))
   {
    if(!files)
       files=(FileIndex**)malloc(16*sizeof(FileIndex*));
    else
       files=(FileIndex**)realloc(files,(nfiles+16)*sizeof(FileIndex*));
   }
 files[nfiles]=(FileIndex*)malloc(sizeof(FileIndex));

 files[nfiles]->name=(char*)malloc(strlen(name)+1);
 strcpy(files[nfiles]->name,name);

 files[nfiles]->type="";

 if(mode==MTime)
    files[nfiles]->time=buf->st_mtime;
 else if(mode==ATime)
    files[nfiles]->time=buf->st_atime;

 nfiles++;
}


/*++++++++++++++++++++++++++++++++++++++
  Add a file to the list.

  char *name The name of the file or the URL if in outgoing.

  struct stat *buf The stat result on the file.

  SortMode mode The sort mode.

  char *dir The directory that the file is in for the 'Latest' mode.

  char *outgoing The name of the file in the outgoing directory (if any).
  ++++++++++++++++++++++++++++++++++++++*/

static void add_file(char *name,struct stat *buf,SortMode mode,char *dir,char *outgoing)
{
 if(name[0] && name[0]==PATH_SEP &&
    name[1] && name[1]==PATH_SEP)
    return;

 if(!(nfiles%16))
   {
    if(!files)
       files=(FileIndex**)malloc(16*sizeof(FileIndex*));
    else
       files=(FileIndex**)realloc(files,(nfiles+16)*sizeof(FileIndex*));
   }
 files[nfiles]=(FileIndex*)malloc(sizeof(FileIndex));

 if(!outgoing)
   {
    char *args="";
    int i;

    for(i=0;name[i];i++)
       if(name[i]==PATH_SEP)
         {
          if(name[i+1]!=PATH_SEP)
             name[i]='/';
          else
             if(name[i+1] && name[i+2]!=PATH_SEP)
               {args=&name[i];break;}
             else
               {name[i]='/';args=&name[i+1];break;}
         }

    if(*args)
      {
       char *old_args=args;
       int afd=open(old_args,O_RDONLY);

       if(afd!=-1)
         {
          int rr,r=0;
          struct utimbuf utbuf;

          args=(char*)malloc(64+1);

          while((rr=read(afd,&args[r],64)))
            {
             r+=rr;
             args=(char*)realloc(args,r+64+1);
            }

          args[r]=0;
          close(afd);

          utbuf.actime=buf->st_atime;
          utbuf.modtime=buf->st_mtime;
          utime(old_args,&utbuf);
         }

       *old_args=0;
      }

    files[nfiles]->name=(char*)malloc((dir?strlen(dir)+2:0)+strlen(name)+strlen(args)+2);
    if(*args)
      {
       if(dir)
          sprintf(files[nfiles]->name,"%s/%s?%s",dir,*name=='/'?name+1:name,args);
       else
          sprintf(files[nfiles]->name,"%s?%s",name,args);

       free(args);
      }
    else
      {
       if(dir)
          sprintf(files[nfiles]->name,"%s/%s",dir,*name=='/'?name+1:name);
       else
          strcpy(files[nfiles]->name,name);
      }
   }
 else
   {
    files[nfiles]->name=(char*)malloc(strlen(name)+1);
    strcpy(files[nfiles]->name,name);

    files[nfiles]->file=(char*)malloc(strlen(outgoing)+1);
    strcpy(files[nfiles]->file,outgoing);
   }

 if(mode==Type)
   {
    int i;
    files[nfiles]->type="";
    for(i=strlen(name);i>0;i--)
       if(files[nfiles]->name[i]=='.')
          files[nfiles]->type=&files[nfiles]->name[i];
       else if(files[nfiles]->name[i]=='/')
          break;
   }
 else if(mode==MTime || mode==Latest)
    files[nfiles]->time=buf->st_mtime;
 else if(mode==ATime)
    files[nfiles]->time=buf->st_atime;

 nfiles++;
}


/*++++++++++++++++++++++++++++++++++++++
  Used to sort the files into alphabetical order.

  int sort_alpha Returns the comparison of the pointers to strings.

  FileIndex **a The first FileIndex.

  FileIndex **b The second FileIndex.
  ++++++++++++++++++++++++++++++++++++++*/

static int sort_alpha(FileIndex **a,FileIndex **b)
{
 char *an=(*a)->name;
 char *bn=(*b)->name;

 return(strcmp(an,bn));
}


/*++++++++++++++++++++++++++++++++++++++
  Used to sort the files into type (file extension) order.

  int sort_type Returns the comparison of the pointers to strings.

  FileIndex **a The first FileIndex.

  FileIndex **b The second FileIndex.
  ++++++++++++++++++++++++++++++++++++++*/

static int sort_type(FileIndex **a,FileIndex **b)
{
 char *an=(*a)->name,*at=(*a)->type;
 char *bn=(*b)->name,*bt=(*b)->type;
 int sort1=strcmp(at,bt);

 if(sort1==0)
    return(strcmp(an,bn));
 else
    return(sort1);
}


/*++++++++++++++++++++++++++++++++++++++
  Used to sort the files into alphabetical order.

  int sort_time Returns the comparison of the times.

  FileIndex **a The first FileIndex.

  FileIndex **b The second FileIndex.
  ++++++++++++++++++++++++++++++++++++++*/

static int sort_time(FileIndex **a,FileIndex **b)
{
 long at=(*a)->time;
 long bt=(*b)->time;

 return(bt-at);
}
