/***************************************
  $Header: /home/amb/wwwoffle/RCS/control.c 1.6 1997/08/03 10:46:14 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 1.2d.
  The HTML interactive control pages.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1997 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <unistd.h>

#include "wwwoffle.h"
#include "config.h"
#include "sockets.h"
#include "errors.h"


/*+ The action to perform. +*/
typedef enum _Action
{
 None,                          /*+ Undecided. +*/
 Online,                        /*+ Tell the server that we are online. +*/
 Offline,                       /*+ Tell the server that we are offline. +*/
 Fetch,                         /*+ Tell the server to fetch the requested pages. +*/
 Config,                        /*+ Tell the server to re-read the configuration file. +*/
 Purge,                         /*+ Tell the server to purge pages. +*/
 Delete,                        /*+ Delete a page from the cache or a request from the outgoing directory. +*/
}
Action;


static void MainControlPage(int fd);
static void ActionControlPage(int fd,Action action,char *command);
static void DeleteControlPage(int fd,char *file);
static void IllegalControlPage(int fd,char *path,char *args);
static void ControlAuthFail(int fd,char *path);
static int MatchPassword(char *try,char *actual);
static char *UnEncodeBase64(char *encoded);


/*++++++++++++++++++++++++++++++++++++++
  Send to the client one of the pages to control WWWOFFLE using HTML.

  int fd The file descriptor of the client.

  char *path The path that was specified.

  char *args The arguments that were passed.

  char *request The HTTP request.
  ++++++++++++++++++++++++++++++++++++++*/

void ControlPage(int fd,char *path,char *args,char *request)
{
 Action action=None;
 char *newpath=(char*)malloc(strlen(path)+1);
 char *command="";

 strcpy(newpath,path);

 if(*newpath && newpath[strlen(newpath)-1]=='/')
    newpath[strlen(newpath)-1]=0;

 if(!strcmp(newpath,"online"))
   {action=Online;command="-online";}
 else if(!strcmp(newpath,"offline"))
   {action=Offline;command="-offline";}
 else if(!strcmp(newpath,"fetch"))
   {action=Fetch;command="-fetch";}
 else if(!strcmp(newpath,"config"))
   {action=Config;command="-config";}
 else if(!strcmp(newpath,"purge"))
   {action=Purge;command="-purge";}
 else if(!strcmp(newpath,"delete"))
    action=Delete;

 if(PassWord)
   {
    char *auth=strstr(request,"\nAuthorization:");
    char *eol,*type,*pswd;
    int n;

    if(!auth)
      {
       ControlAuthFail(fd,newpath);
       return;
      }

    auth+=15;
    eol=strchr(auth,'\n');
    type=(char*)malloc(eol-auth);
    pswd=(char*)malloc(eol-auth);
    *eol=0;
    n=sscanf(auth,"%s %s",type,pswd);
    *eol='\n';

    if(n!=2 || strcasecmp(type,"Basic") || !MatchPassword(pswd,PassWord))
      {
       ControlAuthFail(fd,newpath);
       PrintMessage(Important,"Interactive control webpage authorisation failed.");
       return;
      }
   }

 if(action==None && *newpath)
   {
    IllegalControlPage(fd,newpath,NULL);
    return;
   }

 if(action==None)
    MainControlPage(fd);
 else if(action==Delete)
    DeleteControlPage(fd,args);
 else
    ActionControlPage(fd,action,command);
}


/*++++++++++++++++++++++++++++++++++++++
  The main control page with all of the buttons.

  int fd The file descriptor to write to.
  ++++++++++++++++++++++++++++++++++++++*/

static void MainControlPage(int fd)
{
 char *page=
 "HTTP/1.0 200 WWWOFFLE Control Page\r\n"
 "Content-type: text/html\r\n"
 "\r\n"
 "<HTML>\n"
 "<HEAD>\n"
 "<TITLE>\n"
 "WWWOFFLE - Interactive Control Page\n"
 "</TITLE>\n"
 "</HEAD>\n"
 "<BODY>\n"
 "<H1 align=center>WWWOFFLE Interactive Control Page</H1>\n"
 "The <b>wwwoffled</b> demon program can be controlled from here.\n"
 "<h2>Command Line Replacements</h2>\n"
 "All of the control actions available in the <b>wwwoffle</b> program are here.\n"
 "<p>\n"
 "<form action=\"/control/online\" method=post>\n"
 "<input type=\"hidden\" name=action value=\"online\">\n"
 "<input type=\"submit\" value=\"Online\"> Put the <b>wwwoffled</b> program online.\n"
 "</form>\n"
 "<form action=\"/control/offline\" method=post>\n"
 "<input type=\"hidden\" name=action value=\"offline\">\n"
 "<input type=\"submit\" value=\"Offline\"> Put the <b>wwwoffled</b> program offline.\n"
 "</form>\n"
 "<form action=\"/control/fetch\" method=post>\n"
 "<input type=\"hidden\" name=action value=\"fetch\">\n"
 "<input type=\"submit\" value=\"Fetch\"> Fetch the pages that <b>wwwoffled</b> has pending.\n"
 "</form>\n"
 "<form action=\"/control/config\" method=post>\n"
 "<input type=\"hidden\" name=action value=\"config\">\n"
 "<input type=\"submit\" value=\"Config\"> Force <b>wwwoffled</b> to re-read the configuration file.\n"
 "</form>\n"
 "<form action=\"/control/purge\" method=post>\n"
 "<input type=\"hidden\" name=action value=\"purge\">\n"
 "<input type=\"submit\" value=\"Purge\"> Force the <b>wwwoffled</b> cache to be purged.\n"
 "</form>\n"
 "<h2>Delete Pages or Outgoing Requests</h2>\n"
 "You can also delete a specified cached URL or a request from the outgoing directory.\n"
 "<form action=\"/control/delete\" method=get>\n"
 "<input type=\"submit\" value=\"Delete URL From Cache\">.\n"
 "<input type=\"text\" name=url value=\"\">\n"
 "</form>\n"
 "<form action=\"/control/delete\" method=get>\n"
 "<input type=\"submit\" value=\"Delete File From Outgoing\">.\n"
 "<input type=\"text\" name=req value=\"\">\n"
 "</form>\n"
 "</BODY>\n"
 "</HTML>\n";

 write(fd,page,strlen(page));
}


/*++++++++++++++++++++++++++++++++++++++
  The control page that performs an action.

  int fd The file descriptor to write to.

  Action action The action to perform.

  char *command The command line argument that would be used with wwwoffle.
  ++++++++++++++++++++++++++++++++++++++*/

static void ActionControlPage(int fd,Action action,char *command)
{
 int socket=OpenClientSocket("localhost",WWWOFFLE_Port);

 if(socket==-1)
   {
    ServerError(fd,"Cannot open connection to wwwoffle server on localhost");
    PrintMessage(Warning,"Cannot open connection to wwwoffle server localhost port %d.",WWWOFFLE_Port);
   }
 else
   {
    FILE *server_r,*server_w;
    char *buffer=NULL;
    char *string=(char*)malloc(32+strlen(command)+strlen(ConfigFile));
    char *head=
    "HTTP/1.0 200 WWWOFFLE Control Page\r\n"
    "Content-type: text/html\r\n"
    "\r\n"
    "<HTML>\n"
    "<HEAD>\n"
    "<TITLE>\n"
    "WWWOFFLE - Interactive Control Page\n"
    "</TITLE>\n"
    "</HEAD>\n"
    "<BODY>\n"
    "<H1 align=center>WWWOFFLE Interactive Control Page</H1>\n"
    "This page is equivalent to using <b>wwwoffle</b> on the command line\n"
    "<pre>\n";
    char *middle=
    "</pre>\n"
    "The output of this command is:\n"
    "<pre>\n";
    char *tail=
    "</pre>\n"
    "<p align=center><a href=\"/control/\">[Return to the control page]</a>"
    "</BODY>\n"
    "</HTML>\n";

    write(fd,head,strlen(head));

    sprintf(string,"wwwoffle %s -c %s\n",command,ConfigFile);
    write(fd,string,strlen(string));

    server_r=fdopen(socket,"r");
    server_w=fdopen(socket,"w");

    /* Send the message. */

    if(PassWord)
       fprintf(server_w,"WWWOFFLE PASSWORD %s\r\n",PassWord);

    if(action==Online)
       fprintf(server_w,"WWWOFFLE ONLINE\r\n");
    else if(action==Offline)
       fprintf(server_w,"WWWOFFLE OFFLINE\r\n");
    else if(action==Fetch)
       fprintf(server_w,"WWWOFFLE FETCH\r\n");
    else if(action==Config)
       fprintf(server_w,"WWWOFFLE CONFIG\r\n");
    else if(action==Purge)
       fprintf(server_w,"WWWOFFLE PURGE\r\n");

    fprintf(server_w,"\r\n");
    fflush(server_w);

    write(fd,middle,strlen(middle));

    while((buffer=fgets_realloc(buffer,server_r)))
       write(fd,buffer,strlen(buffer));

    fclose(server_w);
    fclose(server_r);

    write(fd,tail,strlen(tail));
   }
}


/*++++++++++++++++++++++++++++++++++++++
  The control page that deletes a cached page or a request.

  int fd The file descriptor to write to.

  char *args The arguments specified.
  ++++++++++++++++++++++++++++++++++++++*/

static void DeleteControlPage(int fd,char *args)
{
 char *head=
 "HTTP/1.0 200 WWWOFFLE Control Page\r\n"
 "Content-type: text/html\r\n"
 "\r\n"
 "<HTML>\n"
 "<HEAD>\n"
 "<TITLE>\n"
 "WWWOFFLE - Interactive Delete Page\n"
 "</TITLE>\n"
 "</HEAD>\n"
 "<BODY>\n"
 "<H1 align=center>WWWOFFLE Interactive Delete Page</H1>\n"
 "The requested file\n"
 "<br><b><tt>\n";
 char *middle1=
 "</tt></b><br>\n"
 "has been removed from the outgoing directory\n";
 char *middle1_2=
 "and"
 "<br><b><tt>\n";
 char *middle2=
 "</tt></b><br>\n"
 "has been removed from the cache\n";
 char *tail=
 "<p align=center><a href=\"/control/\">[Go to the control page]</a>"
 "</BODY>\n"
 "</HTML>\n";
 char *url=NULL,*req=NULL;
 char *copy=(char*)malloc(strlen(args)+1);
 int i;

 strcpy(copy,args);
 for(i=0;copy[i];i++)
   {
    if(i!=0 && copy[i-1]=='&')
       copy[i-1]=0;
    if(i==0 || copy[i-1]==0)
      {
       if(!strncmp("url=",&copy[i],4))
          url=&copy[i+4];
       if(!strncmp("req=",&copy[i],4))
          req=&copy[i+4];
      }
   }

 if(!(!!url ^ !!req))
   {
    IllegalControlPage(fd,"delete",args);
    PrintMessage(Important,"Invalid interactive delete page requested; args='%s'.",args);
   }
 else
   {
    write(fd,head,strlen(head));

    if(req)
      {
       int delcached=DeleteOutgoingSpoolFile(req,&url);

       write(fd,req,strlen(req));
       write(fd,middle1,strlen(middle1));
       if(delcached)
         {
          write(fd,middle1_2,strlen(middle1_2));
          write(fd,url,strlen(url));
          write(fd,middle2,strlen(middle2));
         }
      }
    else
      {
       char *host,*path,*args;

       url=UrlDecode(url);
       SplitURL(url,&host,&path,&args);
       OpenWebpageSpoolFile(-1,host,path,args);

       write(fd,url,strlen(url));
       write(fd,middle2,strlen(middle2));
      }

    write(fd,tail,strlen(tail));
   }

 free(copy);
}


/*++++++++++++++++++++++++++++++++++++++
  Inform the user that the specified control page is illegal.

  int fd The file descriptor to write to.

  char *path The specified path.

  char *args The specified arguments.
  ++++++++++++++++++++++++++++++++++++++*/

static void IllegalControlPage(int fd,char *path,char *args)
{
 char *string=(char*)malloc(strlen(path)+args?strlen(args):0+16);
 char *head=
 "HTTP/1.0 404 WWWOFFLE Illegal Control Page\r\n"
 "Content-type: text/html\r\n"
 "\r\n"
 "<HTML>\n"
 "<HEAD>\n"
 "<TITLE>\n"
 "WWWOFFLE - Illegal Interactive Control Page\n"
 "</TITLE>\n"
 "</HEAD>\n"
 "<BODY>\n"
 "<H1 align=center>WWWOFFLE Illegal Interactive Control Page</H1>\n"
 "<p align=center>\n"
 "Your request for the control URL\n"
 "<br><b><tt>\n";
 char *tail=
 "\n"
 "</tt></b><br>\n"
 "is illegal, select the link below for the main interactive control page.\n"
 "<br>\n"
 "<a href=\"/control/\">/control/</a>\n"
 "</BODY>\n"
 "</HTML>\n";

 write(fd,head,strlen(head));
 if(args)
    sprintf(string,"/control/%s?%s",path,args);
 else
    sprintf(string,"/control/%s",path);
 write(fd,string,strlen(string));
 write(fd,tail,strlen(tail));

 free(string);
}


/*++++++++++++++++++++++++++++++++++++++
  Inform the user that the authorisation failed.

  int fd The file descriptor to write to.

  char *path The specified path.
  ++++++++++++++++++++++++++++++++++++++*/

static void ControlAuthFail(int fd,char *path)
{
 char *string=(char*)malloc(strlen(path)+16);
 char *head=
 "HTTP/1.0 401 WWWOFFLE Authorisation Failed\r\n"
 "WWW-Authenticate: Basic realm=\"control\"\r\n"
 "Content-type: text/html\r\n"
 "\r\n"
 "<HTML>\n"
 "<HEAD>\n"
 "<TITLE>\n"
 "WWWOFFLE - Authorisation Failed\n"
 "</TITLE>\n"
 "</HEAD>\n"
 "<BODY>\n"
 "<H1 align=center>WWWOFFLE Authorisation Failed</H1>\n"
 "<p align=center>\n"
 "Your request for the interactive control URL\n"
 "<br><b><tt>\n";
 char *tail=
 "\n"
 "</tt></b><br>\n"
 "requires a password and you have failed to be authorised.\n"
 "</BODY>\n"
 "</HTML>\n";

 write(fd,head,strlen(head));
 sprintf(string,"/control/%s",path);
 write(fd,string,strlen(string));
 write(fd,tail,strlen(tail));
}


/*++++++++++++++++++++++++++++++++++++++
  Try to match the authorisation password against the actual one.

  int MatchPassword Returns true if the password is OK.

  char *try The attempted password.

  char *actual The actual password.
  ++++++++++++++++++++++++++++++++++++++*/

static int MatchPassword(char *try,char *actual)
{
 char *decoded=UnEncodeBase64(try);
 char *colon;
 int ok=0;

 colon=strchr(decoded,':');

 if(!strcmp(colon+1,actual))
    ok=1;

 free(decoded);

 return(ok);
}


static char base64[64]={'A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P',
                        'Q','R','S','T','U','V','W','X','Y','Z','a','b','c','d','e','f',
                        'g','h','i','j','k','l','m','n','o','p','q','r','s','t','u','v',
                        'w','x','y','z','0','1','2','3','4','5','6','7','8','9','+','/'};

/*++++++++++++++++++++++++++++++++++++++
  Decode a base 64 string.

  char *UnEncodeBase64 Return a malloced string containing the decoded version.

  char *encoded The encoded string.
  ++++++++++++++++++++++++++++++++++++++*/

static char *UnEncodeBase64(char *encoded)
{
 int l=strlen(encoded);
 char *decoded=(char*)malloc(l+8);
 int i,j;

 for(i=0;i<(l+8);i++)
   {
    if(i<l)
       for(j=0;j<64;j++)
          if(base64[j]==encoded[i])
            {decoded[i]=j;break;}
    if(i>=l || j==64)
       decoded[i]=0;
   }

 for(i=0;i<(l+3);i+=4)
   {
    unsigned long l=0;

    for(j=0;j<4;j++)
       l|=((unsigned long)decoded[i+3-j])<<(j*8);

    l=((l>>2)&0xffffffc0)|(l&0x0000003f);
    l=((l>>2)&0xfffff000)|(l&0x00000fff);
    l=((l>>2)&0xfffc0000)|(l&0x0003ffff);

    for(j=0;j<4;j++)
       decoded[i+3-j]=(char)((l>>(j*8))&0xff);
   }

 for(i=0,j=0;i<(l+3);i++)
   {
    if(!(i%4)) continue;
    decoded[j++]=decoded[i];
   }

 return(decoded);
}
