/***************************************
  $Header: /home/amb/wwwoffle/RCS/config.c 1.25 1997/08/25 15:08:26 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 1.2e.
  Configuration file management functions.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1997 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <unistd.h>
#include <pwd.h>
#include <grp.h>

#include "config.h"
#include "errors.h"
#include "wwwoffle.h"


#ifndef SPOOL_DIR
#define SPOOL_DIR DEF_SPOOL
#endif

/* Type definitions */

/*+ The type of value to expect for a value. +*/
typedef enum _ConfigType
{
 Fixed,                         /*+ When the left hand side is fixed. +*/
 None,                          /*+ When there is no right hand side. +*/

 Boolean,                       /*+ A boolean response (yes/no 1/0 true/false). +*/

 PortNumber,                    /*+ For port numbers (>0). +*/
 AgeDays,                       /*+ An age in days (can be -ve). +*/
 FileSize,                      /*+ A file size (must be >=0). +*/
 CfgMaxServers,                 /*+ Max number of servers to fork (>0, <MAX_SERVERS). +*/
 CfgMaxFetchServers,            /*+ Max number of servers for fetching pages (>0, <MAX_FETCH_SERVERS). +*/

 UserId,                        /*+ For user IDs, (numeric or string). +*/
 GroupId,                       /*+ For group IDs, (numeric or string). +*/

 String,                        /*+ For an arbitrary string (no whitespace). +*/
 DirName,                       /*+ For directory name values (string starting with '/'). +*/
 FileExt,                       /*+ A file extension (string). +*/
 Host,                          /*+ For host names (string). +*/
 HostAndPort,                   /*+ For host name and port numbers (string[:port]). +*/
 CfgLogLevel                    /*+ A log level (debug, info, important, warning or fatal). +*/
}
ConfigType;

/*+ A description of an entry in a section of the config file. +*/
typedef struct _Entry
{
 char *name;                    /*+ The name of the entry. +*/
 void *pointer;                 /*+ A pointer to the value of the entry. +*/
 char list;                     /*+ Set to true if it is a list of KeyPairs. +*/
 ConfigType left_type;          /*+ The type of the left side of the equals sign. +*/
 ConfigType right_type;         /*+ The type of the right side of the equals sign. +*/
}
Entry;

/*+ A description of a section in the config file. +*/
typedef struct _Section
{
 char *name;                    /*+ The name of the section. +*/
 Entry *entries;                /*+ The entries in the section (NULL terminated). +*/
}
Section;

/*+ A keyed entry for a wildcard left hand side. +*/
typedef struct _KeyPair
{
 char *name;                    /*+ The key name of the entry. +*/
 union
 {
  int   integer;                /*+ An integer value. +*/
  char *string;                 /*+ A string value. +*/
 }
 value;
}
KeyPair;

/*+ The last of a KeyPair list. +*/
static KeyPair KeyPairEnd={NULL};


/* StartUp section */

/*+ The name of the configuration file. +*/
char *ConfigFile=SPOOL_DIR "/wwwoffle.conf";

/*+ The port number to use for the HTTP proxy port. +*/
int HTTP_Port=DEF_HTTP_PORT;

/*+ The port number to use for the wwwoffle port. +*/
int WWWOFFLE_Port=DEF_WWWOFFLE_PORT;

/*+ The spool directory. +*/
char *SpoolDir=SPOOL_DIR;

/*+ The user id for wwwoffled or -1 for none. +*/
int WWWOFFLE_Uid=-1;

/*+ The group id for wwwoffled or -1 for none. +*/
int WWWOFFLE_Gid=-1;

/*+ Whether to use the syslog facility or not. +*/
int UseSyslog=1;

/*+ The password required for demon configuration. +*/
char *PassWord=NULL;

/*+ Maximum number of servers  +*/
int MaxServers=DEF_MAX_SERVERS; /*+ in total. +*/
int MaxFetchServers=DEF_MAX_FETCH_SERVERS; /*+ for fetching. +*/

/*+ The entries in the StartUp section. +*/
static Entry startup_entries[]={{"http-port"         ,(void*)&HTTP_Port       ,0,Fixed,PortNumber        },
                                {"wwwoffle-port"     ,(void*)&WWWOFFLE_Port   ,0,Fixed,PortNumber        },
                                {"spool-dir"         ,(void*)&SpoolDir        ,0,Fixed,DirName           },
                                {"run-uid"           ,(void*)&WWWOFFLE_Uid    ,0,Fixed,UserId            },
                                {"run-gid"           ,(void*)&WWWOFFLE_Gid    ,0,Fixed,GroupId           },
                                {"use-syslog"        ,(void*)&UseSyslog       ,0,Fixed,Boolean           },
                                {"password"          ,(void*)&PassWord        ,0,Fixed,String            },
                                {"max-servers"       ,(void*)&MaxServers      ,0,Fixed,CfgMaxServers     },
                                {"max-fetch-servers" ,(void*)&MaxFetchServers ,0,Fixed,CfgMaxFetchServers},
                                {NULL                ,NULL                    ,0,   -1,                -1}};

/*+ The StartUp section. +*/
static Section startup_section={"StartUp",startup_entries};


/* Options Section */

/*+ The level of error logging, see ErrorLevel in errors.h +*/
int LogLevel=Important;

/*+ The option to also fetch images. +*/
int FetchImages=0;

/*+ The option to also fetch frames. +*/
int FetchFrames=0;

/*+ The number of days to display in the index of the latest pages. +*/
int IndexLatestDays=7;

/*+ The option of a tag that can be added to the bottom of the spooled pages with the date and a refresh button. +*/
int AddInfoRefresh=0;

/*+ The option to always request the server for changes when it is the cache and online. +*/
int RequestChanged=1;

/*+ The entries in the Options section. +*/
static Entry options_entries[]={{"loglevel"         ,(void*)&LogLevel       ,0,Fixed,CfgLogLevel},
                                {"fetch-images"     ,(void*)&FetchImages    ,0,Fixed,Boolean    },
                                {"fetch-frames"     ,(void*)&FetchFrames    ,0,Fixed,Boolean    },
                                {"index-latest-days",(void*)&IndexLatestDays,0,Fixed,AgeDays    },
                                {"add-info-refresh" ,(void*)&AddInfoRefresh ,0,Fixed,Boolean    },
                                {"request-changed"  ,(void*)&RequestChanged ,0,Fixed,Boolean    },
                                {NULL               ,NULL                   ,0,   -1,         -1}};

/*+ The Options section. +*/
static Section options_section={"Options",options_entries};


/* LocalHost section */

/*+ The list of localhost hostnames. +*/
static KeyPair **LocalHost=NULL;

/*+ The entries in the LocalHost section. +*/
static Entry localhost_entries[]={{""  ,(void*)&LocalHost,1,Host,None},
                                  {NULL,NULL             ,0,  -1,  -1}};

/*+ The LocalHost section. +*/
static Section localhost_section={"LocalHost",localhost_entries};


/* LocalNet section */

/*+ The list of local network hostnames. +*/
static KeyPair **LocalNet=NULL;

/*+ The entries in the LocalNet section. +*/
static Entry localnet_entries[]={{""  ,(void*)&LocalNet,1,Host,None},
                                 {NULL,NULL            ,0,  -1,  -1}};

/*+ The LocalNet section. +*/
static Section localnet_section={"LocalNet",localnet_entries};


/* AllowedConnect section */

/*+ The list of allowed hostnames. +*/
static KeyPair **AllowedConnect=NULL;

/*+ The entries in the AllowedConnect section. +*/
static Entry allowedconnect_entries[]={{""  ,(void*)&AllowedConnect,1,Host,None},
                                       {NULL,NULL                  ,0,  -1,  -1}};

/*+ The AllowedConnect section. +*/
static Section allowedconnect_section={"AllowedConnect",allowedconnect_entries};


/* DontCache section */

/*+ The list of hosts not to cache. +*/
static KeyPair **DontCacheHosts=NULL;

/*+ The list of file extensions not to cache. +*/
static KeyPair **DontCacheExts=NULL;

/*+ The list of non-cached hostnames + pathnames. +*/
static KeyPair **DontCacheHostPath=NULL;

/*+ The entries in the DontCache section. +*/
static Entry dontcache_entries[]={{"host"    ,(void*)&DontCacheHosts   ,1,Fixed,Host   },
                                  {"file-ext",(void*)&DontCacheExts    ,1,Fixed,FileExt},
                                  {""        ,(void*)&DontCacheHostPath,1,Host ,DirName},
                                  {NULL      ,NULL                     ,0,   -1,     -1}};

/*+ The DontCache section. +*/
static Section dontcache_section={"DontCache",dontcache_entries};


/* DontGet section */

/*+ The list of hosts not to get. +*/
static KeyPair **DontGetHosts=NULL;

/*+ The list of file extensions not to get. +*/
static KeyPair **DontGetExts=NULL;

/*+ The list of non-got hostnames + pathnames. +*/
static KeyPair **DontGetHostPath=NULL;

/*+ The entries in the DontGet section. +*/
static Entry dontget_entries[]={{"host"    ,(void*)&DontGetHosts   ,1,Fixed,Host   },
                                {"file-ext",(void*)&DontGetExts    ,1,Fixed,FileExt},
                                {""        ,(void*)&DontGetHostPath,1,Host ,DirName},
                                {NULL      ,NULL                   ,0,   -1,     -1}};

/*+ The DontGet section. +*/
static Section dontget_section={"DontGet",dontget_entries};


/* CensorHeader section */

/*+ The list of censored headers. +*/
static KeyPair **CensorHeader=NULL;

/*+ The entries in the censor headers section. +*/
static Entry censorheader_entries[]={{""  ,(void*)&CensorHeader,1,String,None},
                                     {NULL,NULL                ,0,    -1,  -1}};

/*+ The CensorHeader section. +*/
static Section censorheader_section={"CensorHeader",censorheader_entries};


/* Proxy section */

/*+ The default proxy server to use. +*/
static char *DefaultProxy=NULL;

/*+ The list of hostnames and proxies. +*/
static KeyPair **Proxies=NULL;

/*+ The entries in the Proxy section. +*/
static Entry proxy_entries[]={{"default",(void*)&DefaultProxy,0,Fixed,HostAndPort},
                              {""       ,(void*)&Proxies     ,1,Host ,HostAndPort},
                              {NULL     ,NULL                ,0,   -1,         -1}};

/*+ The Proxy section. +*/
static Section proxy_section={"Proxy",proxy_entries};


/* Purge section */

/*+ If true then use modification time instead of access time. +*/
int PurgeUseMTime=0;

/*+ The default age for purging files. +*/
int DefaultPurgeAge=DEF_PURGE_AGE;

/*+ The maximum allowed size of the cache. +*/
int PurgeCacheSize=0;

/*+ The list of hostnames and purge ages. +*/
static KeyPair **PurgeAges=NULL;

/*+ The entries in the Purge section. +*/
static Entry purge_entries[]={{"use-mtime",(void*)&PurgeUseMTime  ,0,Fixed,Boolean },
                              {"default"  ,(void*)&DefaultPurgeAge,0,Fixed,AgeDays },
                              {"max-size" ,(void*)&PurgeCacheSize ,0,Fixed,FileSize},
                              {""         ,(void*)&PurgeAges      ,1,Host ,AgeDays },
                              {NULL       ,NULL                   ,0,   -1,      -1}};

/*+ The Purge section. +*/
static Section purge_section={"Purge",purge_entries};


/* All sections */

/*+ The list of sections (NULL terminated). +*/
static Section *sections[]={&startup_section,
                            &options_section,
                            &localhost_section,
                            &localnet_section,
                            &allowedconnect_section,
                            &dontcache_section,
                            &dontget_section,
                            &censorheader_section,
                            &proxy_section,
                            &purge_section,
                            NULL};


/* Local functions */
static int NotCachedOrGot(char *host,char *path,KeyPair **Hosts,KeyPair **HostPath,KeyPair **Exts);

static int ReadFile(FILE *config);
static int ReadSection(FILE *config,Section *section);
static int ParseEntry(Entry *entry,char *left,char *right);
static int ParseValue(char *text,ConfigType type,void *value,int rhs);

static void SaveOldValues(void);
static void RestoreOldValues(void);
static void RemoveOldValues(void);
static void FreeKeyPairList(KeyPair **list,int andvalue);


/*+ Only the first time that the config file is read can the StartUp section be used. +*/
static int first_time=1;

/*+ The error message. +*/
static char *errmsg;


/*++++++++++++++++++++++++++++++++++++++
  Read in the configuration file.

  int fd The file descriptor to write the error message to.
  ++++++++++++++++++++++++++++++++++++++*/

int ReadConfigFile(int fd)
{
 FILE* config;
 int line_no;

 errmsg=NULL;

 config=fopen(ConfigFile,"r");
 if(!config)
   {
    char *msg=(char*)malloc(40+strlen(ConfigFile));
    sprintf(msg,"Cannot open the configuration file '%s'\n",ConfigFile);
    write(fd,msg,strlen(msg));
    free(msg);
    return(1);
   }

 SaveOldValues();

 line_no=ReadFile(config);

 if(errmsg)
    RestoreOldValues();
 else
    RemoveOldValues();

 fclose(config);

 if(errmsg)
   {
    char *msg=(char*)malloc(40+strlen(errmsg)+strlen(ConfigFile));
    sprintf(msg,"Syntax error at line %d in '%s'; %s\n",line_no,ConfigFile,errmsg);
    write(fd,msg,strlen(msg));
    free(msg);
    free(errmsg);
   }

 if(first_time)
    first_time=0;

 if(errmsg)
    return(1);
 else
    return(0);
}


/*++++++++++++++++++++++++++++++++++++++
  Get the first specified name of the server host.

  char *GetLocalHost Returns the first named localhost.
  ++++++++++++++++++++++++++++++++++++++*/

char *GetLocalHost(void)
{
 if(LocalHost)
    return((*LocalHost)->name);
 else
    return("localhost");
}


/*++++++++++++++++++++++++++++++++++++++
  Check if the specified hostname is the server (localhost).

  int IsLocalHost Return true if it is the host the server is on.

  char *host The name of the host (and port number) to be checked.
  ++++++++++++++++++++++++++++++++++++++*/

int IsLocalHost(char *host)
{
 KeyPair **p;
 char *colon=strchr(host,':');
 int port=80;
 int isit=0;

 if(colon)
   {
    *colon=0;
    port=atoi(colon+1);
   }

 if(LocalHost)
    for(p=LocalHost;(*p)->name;p++)
       if(!strcmp((*p)->name,host) && port==HTTP_Port)
         {isit=1;break;}

 if(colon)
    *colon=':';

 return(isit);
}


/*++++++++++++++++++++++++++++++++++++++
  Check if the specified hostname is to be cached.

  int IsLocalNetHost Return true if the host is on the local network.

  char *host The name of the host (and port number) to be checked.
  ++++++++++++++++++++++++++++++++++++++*/

int IsLocalNetHost(char *host)
{
 KeyPair **p;
 char *colon=strchr(host,':');
 int isip=isdigit(*host);
 int isit=0;

 if(colon)
    *colon=0;

 if(LocalHost)
    for(p=LocalHost;(*p)->name;p++)
       if(!strcmp((*p)->name,host))
         {isit=1;break;}

 if(LocalNet && !isit)
    for(p=LocalNet;(*p)->name;p++)
       if(strlen((*p)->name)<=strlen(host) &&
          ((!isip && !strcmp ((*p)->name,host+strlen(host)-strlen((*p)->name))) ||
           ( isip && !strncmp((*p)->name,host,strlen((*p)->name)))))
         {isit=1;break;}

 if(colon)
    *colon=':';

 return(isit);
}


/*++++++++++++++++++++++++++++++++++++++
  Check if the specified hostname is allowed to connect.

  int IsAllowedConnect Return true if it is allowed to connect.

  char *host The name of the host (and port number) to be checked.
  ++++++++++++++++++++++++++++++++++++++*/

int IsAllowedConnect(char *host)
{
 KeyPair **p;
 int isip=isdigit(*host);
 int isit=0;

 if(LocalHost)
    for(p=LocalHost;(*p)->name;p++)
       if(!strcmp((*p)->name,host))
         {isit=1;break;}

 if(AllowedConnect && !isit)
    for(p=AllowedConnect;(*p)->name;p++)
       if(strlen((*p)->name)<=strlen(host) &&
          ((!isip && !strcmp ((*p)->name,host+strlen(host)-strlen((*p)->name))) ||
           ( isip && !strncmp((*p)->name,host,strlen((*p)->name)))))
         {isit=1;break;}

 return(isit);
}


/*++++++++++++++++++++++++++++++++++++++
  Check if the specified URL is to be cached.

  int IsNotCached Return true if it is not to be cached.

  char *host The name of the host to be checked.

  char *path The path on the host to be checked.
  ++++++++++++++++++++++++++++++++++++++*/

int IsNotCached(char *host,char *path)
{
 return(NotCachedOrGot(host,path,DontCacheHosts,DontCacheHostPath,DontCacheExts));
}


/*++++++++++++++++++++++++++++++++++++++
  Check if the specified URL is to be got.

  int IsNotGot Return true if it is not to be got.

  char *host The name of the host (and port number) to be checked.

  char *path The path on the host to be checked.
  ++++++++++++++++++++++++++++++++++++++*/

int IsNotGot(char *host,char *path)
{
 return(NotCachedOrGot(host,path,DontGetHosts,DontGetHostPath,DontGetExts));
}


/*++++++++++++++++++++++++++++++++++++++
  Check if the URL is not to be cached or got based on the keypair lists.

  int NotCachedOrGot Return true if it is not to be cached or got.

  char *host The host name to check.

  char *path The path to check.

  KeyPair **Hosts The list of hostss not to cache or get.

  KeyPair **HostPath The list of hosts and paths not to cache or get.

  KeyPair **Exts The list of file extensions not to cache or get.
  ++++++++++++++++++++++++++++++++++++++*/

static int NotCachedOrGot(char *host,char *path,KeyPair **Hosts,KeyPair **HostPath,KeyPair **Exts)
{
 KeyPair **p;
 int isit=0;

 if(Hosts)
   {
    char *colon=strchr(host,':');
    int isip=isdigit(*host);

    if(colon)
       *colon=0;

    for(p=Hosts;(*p)->name;p++)
       if(strlen((*p)->value.string)<=strlen(host) &&
          ((!isip && !strcmp ((*p)->value.string,host+strlen(host)-strlen((*p)->value.string))) ||
           ( isip && !strncmp((*p)->value.string,host,strlen((*p)->value.string)))))
         {isit=1;break;}

    if(colon)
       *colon=':';
   }

 if(HostPath && !isit)
   {
    char *colon=strchr(host,':');
    int isip=isdigit(*host);

    if(colon)
       *colon=0;

    for(p=HostPath;(*p)->name;p++)
       if(strlen((*p)->name)<=strlen(host) &&
          ((!isip && !strcmp ((*p)->name,host+strlen(host)-strlen((*p)->name))) ||
           ( isip && !strncmp((*p)->name,host,strlen((*p)->name)))))
          if(!strncmp((*p)->value.string+1,path,strlen((*p)->value.string)-1))
            {isit=1;break;}

    if(colon)
       *colon=':';
   }

 if(Exts && !isit)
   {
    char *ext=NULL;
    int i;

    for(i=strlen(path);i>=0;i--)
       if(path[i]=='/')
          break;
       else if(path[i]=='.')
         {ext=&path[i+1];break;}

    if(!ext)
       isit=0;
    else
       for(p=Exts;(*p)->value.string;p++)
          if(!strcmp((*p)->value.string,ext))
            {isit=1;break;}
   }

 return(isit);
}


/*++++++++++++++++++++++++++++++++++++++
  Decide if the header line is to be sent to the server.

  int IsCensoredheader Returns true if the header line is to be deleted.

  char *line The line to check.
  ++++++++++++++++++++++++++++++++++++++*/

int IsCensoredHeader(char *line)
{
 KeyPair **p;
 int isit=0;

 if(CensorHeader && !isit)
    for(p=CensorHeader;(*p)->name;p++)
       if(line[strlen((*p)->name)]==':' &&
          !strncmp((*p)->name,line,strlen((*p)->name)))
         {isit=1;break;}

 return(isit);
}


/*++++++++++++++++++++++++++++++++++++++
  Determine the proxy to use for a specified host.

  char *WhichProxy Return a pointer to the proxy.

  char *host The name of the host to be contacted.
  ++++++++++++++++++++++++++++++++++++++*/

char *WhichProxy(char *host)
{
 KeyPair **p;
 char *colon=strchr(host,':');
 int isip=isdigit(*host);
 char *proxy=DefaultProxy;
 int matchlen=0;

 if(colon)
    *colon=0;

 if(Proxies)
    for(p=Proxies;(*p)->name;p++)
       if(strlen((*p)->name)<=strlen(host) &&
          strlen((*p)->name)>matchlen &&
          ((!isip && !strcmp ((*p)->name,host+strlen(host)-strlen((*p)->name))) ||
           ( isip && !strncmp((*p)->name,host,strlen((*p)->name)))))
         {
          matchlen=strlen((*p)->name);
          proxy=(*p)->value.string;
         }

 if(colon)
    *colon=':';

 return(proxy);
}


/*++++++++++++++++++++++++++++++++++++++
  Determine the age to use when purging for a specified host.

  int WhatPurgeAge Return the age in days.

  char *host The name of the host to be purged.
  ++++++++++++++++++++++++++++++++++++++*/

int WhatPurgeAge(char *host)
{
 KeyPair **p;
 char *colon=strchr(host,':');
 int isip=isdigit(*host);
 int age=DefaultPurgeAge;
 int matchlen=0;

 if(colon)
    *colon=0;

 if(PurgeAges)
    for(p=PurgeAges;(*p)->name;p++)
       if(strlen((*p)->name)<=strlen(host) &&
          strlen((*p)->name)>matchlen &&
          ((!isip && !strcmp ((*p)->name,host+strlen(host)-strlen((*p)->name))) ||
           ( isip && !strncmp((*p)->name,host,strlen((*p)->name)))))
         {
          matchlen=strlen((*p)->name);
          age=(*p)->value.integer;
         }

 if(colon)
    *colon=':';

 return(age);
}


/*++++++++++++++++++++++++++++++++++++++
  Read the data from the file.

  int ReadFile Returns the number of lines read.

  FILE *config The file to read from.
  ++++++++++++++++++++++++++++++++++++++*/

static int ReadFile(FILE *config)
{
 char *line=NULL;
 int line_no=0;

 while((line=fgets_realloc(line,config)))
   {
    int i;
    char *l=line;
    char *r=line+strlen(line)-1;

    line_no++;

    while(*l==' ' || *l=='\t' || *l=='\r' || *l=='\n')
       l++;

    if(!*l || *l=='#')
       continue;

    while(r>l && (*r==' ' || *r=='\t' || *r=='\r' || *r=='\n'))
       *r--=0;

    for(i=0;sections[i];i++)
       if(!strcmp(sections[i]->name,l))
          break;

    if(!sections[i])
      {errmsg=(char*)malloc(32+strlen(l));sprintf(errmsg,"Unrecognised section label '%s'.",l);break;}
    else
       line_no+=ReadSection(config,sections[i]);

    if(errmsg)
       break;
   }

 if(line)
    free(line);

 return(line_no);
}


/*++++++++++++++++++++++++++++++++++++++
  Read all the data from a section.

  int ReadSection Returns the number of lines read.

  FILE *config The file to read from.

  Section *section The section to read.
  ++++++++++++++++++++++++++++++++++++++*/

static int ReadSection(FILE *config,Section *section)
{
 char *line=NULL;
 int line_no=0;
 int open_brace=0;

 while((line=fgets_realloc(line,config)))
   {
    int i;
    char *l=line;
    char *r=line+strlen(line)-1;

    line_no++;

    while(*l==' ' || *l=='\t' || *l=='\r' || *l=='\n')
       l++;

    if(!*l || *l=='#')
       continue;

    while(r>l && (*r==' ' || *r=='\t' || *r=='\r' || *r=='\n'))
       *r--=0;

    if(*l=='{')
      {
       if(*++l)
         {errmsg=(char*)malloc(32);strcpy(errmsg,"Open brace has trailing junk.");break;}
       else
         {
          if(open_brace)
            {errmsg=(char*)malloc(32);strcpy(errmsg,"Open brace repeated.");break;}

          open_brace=1;
         }
      }
    else if(*l=='}')
      {
       if(*++l)
         {errmsg=(char*)malloc(32);strcpy(errmsg,"Close brace has trailing junk.");break;}
       else
         {
          if(!open_brace)
            {errmsg=(char*)malloc(32);strcpy(errmsg,"No open brace seen.");break;}

          break;
         }
      }
    else
      {
       if(!open_brace)
         {errmsg=(char*)malloc(32);strcpy(errmsg,"Open brace missing.");break;}

       if(!first_time && section==*sections)
          continue;

       for(i=0;section->entries[i].name;i++)
          if(!*section->entries[i].name || !strncmp(section->entries[i].name,l,strlen(section->entries[i].name)))
            {
             char *ll,*rr;

             if(*section->entries[i].name)
               {
                ll=l+strlen(section->entries[i].name);

                if(*ll && *ll!='=' && *ll!=' ' && *ll!='\t' && *ll!='\r' && *ll!='\n')
                   continue;
               }
             else
               {
                ll=l;
                while(*ll && *ll!='=' && *ll!=' ' && *ll!='\t' && *ll!='\r' && *ll!='\n')
                   ll++;
               }

             if(section->entries[i].right_type==None)
               {
                *ll=0;
                r=NULL;
               }
             else
               {
                r=strchr(line,'=');
                if(!r)
                  {errmsg=(char*)malloc(32);strcpy(errmsg,"No equal sign for entry.");break;}

                *ll=0;
                if(!*l)
                  {errmsg=(char*)malloc(48);strcpy(errmsg,"Nothing to the left of the equal sign.");break;}

                r++;
                while(*r==' ' || *r=='\t' || *r=='\r' || *r=='\n')
                   r++;

                rr=r;
                while(*rr && *rr!=' ' && *rr!='\t' && *rr!='\r' && *rr!='\n')
                   rr++;

                *rr=0;
               }
             break;
            }

       if(errmsg)
          break;

       if(!section->entries[i].name)
         {errmsg=(char*)malloc(32+strlen(l));sprintf(errmsg,"Unrecognised entry label '%s'.",l);break;}

       ParseEntry(&section->entries[i],l,r);

       if(errmsg)
          break;
      }
   }

 if(line)
    free(line);

 return(line_no);
}


/*++++++++++++++++++++++++++++++++++++++
  Parse an entry from the file.

  int ParseEntry Returns non-zero if there was an error.

  Entry *entries The list of matching entry in the section.

  char *left The string to the left of the equals sign.

  char *right The string to the right of the equals sign.
  ++++++++++++++++++++++++++++++++++++++*/

static int ParseEntry(Entry *entry,char *left,char *right)
{
 if(entry->list)
   {
    KeyPair **p,*new=(KeyPair*)malloc(sizeof(KeyPair));
    int i;

    if(entry->left_type==Fixed)
       new->name=entry->name;
    else
       if(ParseValue(left,entry->left_type,(void*)&new->name,0))
         {free(new);return(1);}

    if(entry->right_type==None)
       new->value.string=NULL;
    else
       if(ParseValue(right,entry->right_type,(void*)&new->value,1))
         {free(new);return(1);}

    if(!*(KeyPair***)entry->pointer)
      {
       *(KeyPair***)entry->pointer=(KeyPair**)malloc(8*sizeof(KeyPair*));
       **(KeyPair***)entry->pointer=&KeyPairEnd;
      }

    p=*(KeyPair***)entry->pointer;
    for(i=0;(*p)->name;p++,i++);

    if(i%8==7)
       *(KeyPair***)entry->pointer=(KeyPair**)realloc(*(void**)entry->pointer,(i+9)*sizeof(KeyPair*));

    for(p=*(KeyPair***)entry->pointer;(*p)->name;p++);

    *p=new;
    p++;
    *p=&KeyPairEnd;
   }
 else
    if(ParseValue(right,entry->right_type,entry->pointer,1))
       return(1);

 return(0);
}


/*++++++++++++++++++++++++++++++++++++++
  Parse the text and put a value into the location.

  int ParseValue Returns non-zero in case of error.

  char *text The text string to parse.

  ConfigType type The type we are looking for.

  void *value The location to store the value.

  int rhs Set to true if this is the right hand side.
  ++++++++++++++++++++++++++++++++++++++*/

static int ParseValue(char *text,ConfigType type,void *value,int rhs)
{
 switch(type)
   {
   case Fixed:
   case None:
    break;

   case Boolean:
    if(!*text)
       *(int*)value=0;
    else
       if(!strcasecmp(text,"yes") || !strcasecmp(text,"true") || !strcmp(text,"1"))
          *(int*)value=1;
       else
          *(int*)value=0;
    break;

   case PortNumber:
    if(!*text)
      {errmsg=(char*)malloc(48);strcpy(errmsg,"Expecting a port number, got nothing.");break;}
    *(int*)value=atoi(text);
    if(*(int*)value<=0 || *(int*)value>65535)
      {errmsg=(char*)malloc(32);sprintf(errmsg,"Invalid port number %d.",*(int*)value);break;}
    break;

   case AgeDays:
    if(!*text)
      {errmsg=(char*)malloc(48);strcpy(errmsg,"Expecting an age in days, got nothing.");break;}
    *(int*)value=atoi(text);
    break;

   case FileSize:
    if(!*text)
      {errmsg=(char*)malloc(48);strcpy(errmsg,"Expecting a file size in MB, got nothing.");break;}
    *(int*)value=atoi(text);
    if(*(int*)value<0)
      {errmsg=(char*)malloc(48);sprintf(errmsg,"Invalid file size %d.",*(int*)value);break;}
    break;

   case CfgMaxServers:
    if(!*text)
      {errmsg=(char*)malloc(60);strcpy(errmsg,"Expecting a maximum server count, got nothing.");break;}
    *(int*)value=atoi(text);
    if(*(int*)value<=0 || *(int*)value>MAX_SERVERS)
      {errmsg=(char*)malloc(48);sprintf(errmsg,"Invalid maximum server count: %d.",*(int*)value);break;}
    break;

   case CfgMaxFetchServers:
    if(!*text)
      {errmsg=(char*)malloc(60);strcpy(errmsg,"Expecting a maximum fetch server count, got nothing.");break;}
    *(int*)value=atoi(text);
    if(*(int*)value<=0 || *(int*)value>MAX_FETCH_SERVERS)
      {errmsg=(char*)malloc(48);sprintf(errmsg,"Invalid maximum fetch server count: %d.",*(int*)value);break;}
    break;

   case UserId:
    if(!*text)
      {errmsg=(char*)malloc(48);strcpy(errmsg,"Expecting a username or uid, got nothing.");break;}
    else
      {
       int uid;
       struct passwd *userInfo = getpwnam(text);
       if(userInfo)
          uid = userInfo->pw_uid;
       else
         {
          if(sscanf(text, "%d", &uid) != 1)
            {errmsg=(char*)malloc(32+strlen(text));sprintf(errmsg,"Invalid user %s.",text);break;}
          if(!getpwuid(uid))
            {errmsg=(char*)malloc(32);sprintf(errmsg,"Unknown user id %d.",uid);break;}
         }
       *(int*)value=uid;
      }
    break;

   case GroupId:
    if(!*text)
      {errmsg=(char*)malloc(48);strcpy(errmsg,"Expecting a group name or gid, got nothing.");break;}
    else
      {
       int gid;
       struct group *groupInfo = getgrnam(text);
       if(groupInfo)
          gid = groupInfo->gr_gid;
       else
         {
          if(sscanf(text, "%d", &gid) != 1)
            {errmsg=(char*)malloc(32+strlen(text));sprintf(errmsg,"Invalid group %s.",text);break;}
          if(!getgrgid(gid))
            {errmsg=(char*)malloc(32);sprintf(errmsg,"Unknown group id %d.",gid);break;}
         }
       *(int*)value=gid;
      }
    break;

   case String:
    if(!*text || !strcasecmp(text,"none"))
       *(char**)value=NULL;
    else
      {
       *(char**)value=(char*)malloc(strlen(text)+1);
       strcpy(*(char**)value,text);
      }
    break;

   case DirName:
    if(!*text)
      {errmsg=(char*)malloc(48);strcpy(errmsg,"Expecting a directory name, got nothing.");break;}
    if(*text!='/')
      {errmsg=(char*)malloc(48+strlen(text));sprintf(errmsg,"Expecting an absolute directory name '%s'.",text);break;}
    *(char**)value=(char*)malloc(strlen(text)+1);
    strcpy(*(char**)value,text);
    break;

   case FileExt:
    *(char**)value=(char*)malloc(strlen(text)+1);
    strcpy(*(char**)value,text);
    break;

   case Host:
    if(rhs && (!*text || !strcasecmp(text,"none")))
       *(char**)value=NULL;
    else
      {
       char *colon=strchr(text,':');
       if(!*text)
         {errmsg=(char*)malloc(40);strcpy(errmsg,"Expecting a hostname, got nothing.");break;}
       if(colon)
         {errmsg=(char*)malloc(48+strlen(text));sprintf(errmsg,"Not expecting a port number, got '%s'.",text);break;}
       *(char**)value=(char*)malloc(strlen(text)+1);
       strcpy(*(char**)value,text);
      }
    break;

   case HostAndPort:
    if(rhs && (!*text || !strcasecmp(text,"none")))
       *(char**)value=NULL;
    else
      {
       char *colon=strchr(text,':');
       if(!*text)
         {errmsg=(char*)malloc(48);strcpy(errmsg,"Expecting a hostname (and port), got nothing.");break;}
       if(*text==':')
         {errmsg=(char*)malloc(48+strlen(text));sprintf(errmsg,"Expecting a hostname before the ':', got '%s'.",text);break;}
       if(colon && (atoi(colon+1)<=0 || atoi(colon+1)>65535))
         {errmsg=(char*)malloc(32);sprintf(errmsg,"Invalid port number %d.",atoi(colon+1));break;}
       *(char**)value=(char*)malloc(strlen(text)+1);
       strcpy(*(char**)value,text);
      }
    break;

   case CfgLogLevel:
    if(!*text)
      {errmsg=(char*)malloc(48);strcpy(errmsg,"Expecting a log level, got nothing.");break;}
    if(strcasecmp(text,"debug")==0)
       {*(int*)value=Debug;break;}
    if(strcasecmp(text,"info")==0)
       {*(int*)value=Inform;break;}
    if(strcasecmp(text,"important")==0)
       {*(int*)value=Important;break;}
    if(strcasecmp(text,"warning")==0)
       {*(int*)value=Warning;break;}
    if(strcasecmp(text,"fatal")==0)
       {*(int*)value=Fatal;break;}
    /* Unrecognized value */
    errmsg=(char*)malloc(48);
    strcpy(errmsg,"Invalid value for log level.");
    break;
   }

 if(errmsg)
    return(1);
 else
    return(0);
}


/*+ The old value of the level of error logging, see ErrorLevel in errors.h +*/
static int old_LogLevel=Important;

/*+ The old value of the option to also fetch images. +*/
static int old_FetchImages=0;

/*+ The old value of the option to also fetch frames. +*/
static int old_FetchFrames=0;

/*+ The old value of the number of days to display in the index of the latest pages. +*/
static int old_IndexLatestDays=0;

/*+ The old value of the option for a tag that can be added to the bottom of the spooled pages with the date and a refresh button. +*/
static int old_AddInfoRefresh=0;

/*+ The old value of the option to always request the server for changes when it is the cache and online. +*/
static int old_RequestChanged=1;

/*+ The old values of the list of localhost hostnames. +*/
static KeyPair **old_LocalHost=NULL;

/*+ The old value of the list of local network hostnames. +*/
static KeyPair **old_LocalNet=NULL;

/*+ The old values of the list of allowed hostnames. +*/
static KeyPair **old_AllowedConnect=NULL;

/*+ The old value of the list of hosts not to cache. +*/
static KeyPair **old_DontCacheHosts=NULL;

/*+ The old value of the list of file extensions not to cache. +*/
static KeyPair **old_DontCacheExts=NULL;

/*+ The old value of the list of non-cached hostnames. +*/
static KeyPair **old_DontCacheHostPath=NULL;

/*+ The old value of the list of hosts not to get. +*/
static KeyPair **old_DontGetHosts=NULL;

/*+ The old value of the list of file extensions not to get. +*/
static KeyPair **old_DontGetExts=NULL;

/*+ The old value of the list of non-got hostnames. +*/
static KeyPair **old_DontGetHostPath=NULL;

/*+ The old value of the default proxy server to use. +*/
static char *old_DefaultProxy=NULL;

/*+ The old value of the list of hostnames and proxies. +*/
static KeyPair **old_Proxies=NULL;

/*+ The old value of the flag that if true then use modification time instead of access time. +*/
static int old_PurgeUseMTime=0;

/*+ The old value of the default age for purging files. +*/
static int old_DefaultPurgeAge=0;

/*+ The old value of the maximum allowed size of the cache. +*/
static int old_PurgeCacheSize=0;

/*+ The old value of the list of hostnames and purge ages. +*/
static KeyPair **old_PurgeAges=NULL;


/*++++++++++++++++++++++++++++++++++++++
  Save the old values in case the re-read of the file fails.
  ++++++++++++++++++++++++++++++++++++++*/

static void SaveOldValues(void)
{
 old_LogLevel=LogLevel;
 LogLevel=Important;
 old_FetchImages=FetchImages;
 FetchImages=0;
 old_FetchFrames=FetchFrames;
 FetchFrames=0;
 old_IndexLatestDays=IndexLatestDays;
 IndexLatestDays=7;
 old_AddInfoRefresh=AddInfoRefresh;
 AddInfoRefresh=0;
 old_RequestChanged=RequestChanged;
 RequestChanged=1;

 old_LocalHost=LocalHost;
 LocalHost=NULL;

 old_LocalNet=LocalNet;
 LocalNet=NULL;

 old_AllowedConnect=AllowedConnect;
 AllowedConnect=NULL;

 old_DontCacheHosts=DontCacheHosts;
 DontCacheHosts=NULL;
 old_DontCacheExts=DontCacheExts;
 DontCacheExts=NULL;
 old_DontCacheHostPath=DontCacheHostPath;
 DontCacheHostPath=NULL;

 old_DontGetHosts=DontGetHosts;
 DontGetHosts=NULL;
 old_DontGetExts=DontGetExts;
 DontGetExts=NULL;
 old_DontGetHostPath=DontGetHostPath;
 DontGetHostPath=NULL;

 old_DefaultProxy=DefaultProxy;
 DefaultProxy=NULL;
 old_Proxies=Proxies;
 Proxies=NULL;

 old_PurgeUseMTime=PurgeUseMTime;
 PurgeUseMTime=0;
 old_DefaultPurgeAge=DefaultPurgeAge;
 DefaultPurgeAge=DEF_PURGE_AGE;
 old_PurgeCacheSize=PurgeCacheSize;
 PurgeCacheSize=0;
 old_PurgeAges=PurgeAges;
 PurgeAges=NULL;
}


/*++++++++++++++++++++++++++++++++++++++
  Restore the old values if the re-read failed.
  ++++++++++++++++++++++++++++++++++++++*/

static void RestoreOldValues(void)
{
 LogLevel=old_LogLevel;
 FetchImages=old_FetchImages;
 FetchFrames=old_FetchFrames;
 IndexLatestDays=old_IndexLatestDays;
 AddInfoRefresh=old_AddInfoRefresh;
 RequestChanged=old_RequestChanged;

 if(LocalHost)
    FreeKeyPairList(LocalHost,1);
 LocalHost=old_LocalHost;

 if(LocalNet)
    FreeKeyPairList(LocalNet,1);
 LocalNet=old_LocalNet;

 if(AllowedConnect)
    FreeKeyPairList(AllowedConnect,1);
 AllowedConnect=old_AllowedConnect;

 if(DontCacheHosts)
    FreeKeyPairList(DontCacheHosts,2);
 DontCacheHosts=old_DontCacheHosts;
 if(DontCacheExts)
    FreeKeyPairList(DontCacheExts,2);
 DontCacheExts=old_DontCacheExts;
 if(DontCacheHostPath)
    FreeKeyPairList(DontCacheHostPath,3);
 DontCacheHostPath=old_DontCacheHostPath;

 if(DontGetHosts)
    FreeKeyPairList(DontGetHosts,2);
 DontGetHosts=old_DontGetHosts;
 if(DontGetExts)
    FreeKeyPairList(DontGetExts,2);
 DontGetExts=old_DontGetExts;
 if(DontGetHostPath)
    FreeKeyPairList(DontGetHostPath,3);
 DontGetHostPath=old_DontGetHostPath;

 if(DefaultProxy)
    free(DefaultProxy);
 DefaultProxy=old_DefaultProxy;
 if(Proxies)
    FreeKeyPairList(Proxies,3);
 Proxies=old_Proxies;

 PurgeUseMTime=old_PurgeUseMTime;
 DefaultPurgeAge=old_DefaultPurgeAge;
 PurgeCacheSize=old_PurgeCacheSize;
 if(PurgeAges)
    FreeKeyPairList(PurgeAges,1);
 PurgeAges=old_PurgeAges;
}


/*++++++++++++++++++++++++++++++++++++++
  Remove the old values if the re-read succeeded.
  ++++++++++++++++++++++++++++++++++++++*/

static void RemoveOldValues(void)
{
 if(old_LocalHost)
    FreeKeyPairList(old_LocalHost,1);

 if(old_LocalNet)
    FreeKeyPairList(old_LocalNet,1);

 if(old_AllowedConnect)
    FreeKeyPairList(old_AllowedConnect,1);

 if(old_DontCacheHosts)
    FreeKeyPairList(old_DontCacheHosts,2);
 if(old_DontCacheExts)
    FreeKeyPairList(old_DontCacheExts,2);
 if(old_DontCacheHostPath)
    FreeKeyPairList(old_DontCacheHostPath,3);

 if(old_DontGetHosts)
    FreeKeyPairList(old_DontGetHosts,2);
 if(old_DontGetExts)
    FreeKeyPairList(old_DontGetExts,2);
 if(old_DontGetHostPath)
    FreeKeyPairList(old_DontGetHostPath,3);

 if(old_DefaultProxy)
    free(old_DefaultProxy);
 if(old_Proxies)
    FreeKeyPairList(old_Proxies,3);

 if(old_PurgeAges)
    FreeKeyPairList(old_PurgeAges,1);
}


/*++++++++++++++++++++++++++++++++++++++
  Free a KeyPair list.

  KeyPair **list The list to free.

  int freewhat If 1 then free the name, if 2 then free the value, if 3 free both, if 0 free neither.
  ++++++++++++++++++++++++++++++++++++++*/

static void FreeKeyPairList(KeyPair **list,int freewhat)
{
 KeyPair **p;

 for(p=list;(*p)->name;p++)
   {
    if(freewhat&1)
       free((*p)->name);
    if(freewhat&2 && (*p)->value.string)
       free((*p)->value.string);
    free(*p);
   }
 free(list);
}
