/***************************************
  $Header: /home/amb/wwwoffle/RCS/wwwoffle.c 1.19 1997/07/06 16:22:10 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 1.2c.
  A user level program to interact with the server.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1996,97 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <unistd.h>

#include "wwwoffle.h"
#include "config.h"
#include "sockets.h"
#include "errors.h"


static void usage(int verbose);


/*+ The action to perform. +*/
typedef enum _Action
{
 None,                          /*+ Undecided. +*/
 Online,                        /*+ Tell the server that we are online. +*/
 Offline,                       /*+ Tell the server that we are offline. +*/
 Fetch,                         /*+ Tell the server to fetch the requested pages. +*/
 Config,                        /*+ Tell the server to re-read the configuration file. +*/
 Purge,                         /*+ Tell the server to purge pages. +*/
 Get,                           /*+ Tell the server to get pages. +*/
 Output                         /*+ Get a page from the server and output it. +*/
}
Action;


/*++++++++++++++++++++++++++++++++++++++
  The main program.
  ++++++++++++++++++++++++++++++++++++++*/

int main(int argc, char** argv)
{
 int i;
 int force=0,images=0,frames=0,recursive=0,recursive_mode=0;

 Action action=None;

 char *host=NULL;
 int port=0;

 /* Parse the command line options */

 ConfigFile=NULL;

 if(argc==1)
    usage(0);

 for(i=1;i<argc;i++)
   {
    if(!strcmp(argv[i],"-h"))
       usage(1);

    if(!strcmp(argv[i],"-o"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Output;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-F"))
      {
       if(action!=None && action!=Get)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Get;
       if(force)
         {fprintf(stderr,"wwwoffle: Only one '-F' argument may be given.\n"); exit(1);}
       force=1;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-i"))
      {
       if(action!=None && action!=Get)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Get;
       if(images)
         {fprintf(stderr,"wwwoffle: Only one '-i' argument may be given.\n"); exit(1);}
       images=1;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-f"))
      {
       if(action!=None && action!=Get)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Get;
       if(frames)
         {fprintf(stderr,"wwwoffle: Only one '-f' argument may be given.\n"); exit(1);}
       frames=1;
       argv[i]=NULL;
       continue;
      }

    if(!strncmp(argv[i],"-R",2) || !strncmp(argv[i],"-r",2) || !strncmp(argv[i],"-d",2))
      {
       if(action!=None && action!=Get)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Get;

       if(recursive)
         {fprintf(stderr,"wwwoffle: Only one '-R', '-r' or '-d' argument may be given.\n"); exit(1);}

       if(argv[i][1]=='R')
          recursive_mode=1;
       else if(argv[i][1]=='d')
          recursive_mode=-1;

       if(argv[i][2])
          recursive=atoi(&argv[i][2]);
       else
          recursive=1;
       if(recursive<=0)
         {fprintf(stderr,"wwwoffle: The '-%c' argument may only be followed by a positive integer.\n",argv[i][1]); exit(1);}

       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-p"))
      {
       char *colon;

       if(++i>=argc)
         {fprintf(stderr,"wwwoffle: The '-p' argument requires a hostname and optionally a port number.\n"); exit(1);}

       if(ConfigFile)
         {fprintf(stderr,"wwwoffle: The '-p' and '-c' options cannot be used together.\n"); exit(1);}

       if((colon=strchr(argv[i],':')))
         {
          *colon++=0;

          port=atoi(colon);

          if(port<0 || port>=65536)
            {fprintf(stderr,"wwwoffle: The port number %d '%s' is invalid.\n",port,argv[i]); exit(1);}
         }

       host=argv[i];

       argv[i-1]=NULL;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-c"))
      {
       if(++i>=argc)
         {fprintf(stderr,"wwwoffle: The '-c' argument requires a configuration file name.\n"); exit(1);}

       if(host)
         {fprintf(stderr,"wwwoffle: The '-p' and '-c' options cannot be used together.\n"); exit(1);}

       ConfigFile=argv[i];

       argv[i-1]=NULL;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-on") || !strcmp(argv[i],"-online"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Online;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-off") || !strcmp(argv[i],"-offline"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Offline;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-fetch"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Fetch;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-config"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Config;
       argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-purge"))
      {
       if(action!=None)
         {fprintf(stderr,"wwwoffle: Only one command at a time.\n\n");usage(0);}
       action=Purge;
       argv[i]=NULL;
       continue;
      }

    if(argv[i][0]=='-')
      {
       fprintf(stderr,"wwwoffle: Unknown option '%s'.\n\n",argv[i]);
       usage(0);
      }
   }

 if(action==None)
    action=Get;

 /* Initialise things. */

 InitErrorHandler("wwwoffle",0,1);

 if(ConfigFile)
   {
    if(ReadConfigFile(2))
       PrintMessage(Fatal,"Error in configuration file '%s'.",ConfigFile);

    host=GetLocalHost();
   }

 /* The connections to the WWWOFFLE server. */

 if(action!=Get && action!=Output)
   {
    int socket;
    FILE *server_r,*server_w;
    char *buffer=NULL;

    socket=OpenClientSocket(host?host:"localhost",port?port:WWWOFFLE_Port);

    if(socket==-1)
       PrintMessage(Fatal,"Cannot open connection to wwwoffle server %s port %d.",host?host:"localhost",port?port:WWWOFFLE_Port);

    server_r=fdopen(socket,"r");
    server_w=fdopen(socket,"w");

    /* Send the message. */

    if(PassWord)
       fprintf(server_w,"WWWOFFLE PASSWORD %s\r\n",PassWord);

    if(action==Online)
       fprintf(server_w,"WWWOFFLE ONLINE\r\n");
    else if(action==Offline)
       fprintf(server_w,"WWWOFFLE OFFLINE\r\n");
    else if(action==Fetch)
       fprintf(server_w,"WWWOFFLE FETCH\r\n");
    else if(action==Config)
       fprintf(server_w,"WWWOFFLE CONFIG\r\n");
    else if(action==Purge)
       fprintf(server_w,"WWWOFFLE PURGE\r\n");

    fprintf(server_w,"\r\n");
    fflush(server_w);

    while((buffer=fgets_realloc(buffer,server_r)))
       fputs(buffer,stdout);

    fclose(server_w);
    fclose(server_r);
   }

 /* The connections to the http proxy. */

 else if(action==Get)
   {
    char refresh[32];

    strcpy(refresh,"refresh");
    if(images)
       strcat(refresh,"-images");
    if(frames)
       strcat(refresh,"-frames");
    if(force)
       strcat(refresh,"-force");
    if(recursive)
      {
       if(recursive_mode==1)
          strcat(refresh,"-any");
       else if(recursive_mode==-1)
          strcat(refresh,"-dir");
       else
          strcat(refresh,"-host");
       sprintf(&refresh[strlen(refresh)],"-%d",recursive);
      }
    if(!images && !frames && !recursive && !force)
       strcat(refresh,"-none");

    for(i=1;i<argc;i++)
       if(argv[i])
         {
          int socket;
          FILE *server_r,*server_w;
          char *buffer=NULL;

          socket=OpenClientSocket(host?host:"localhost",port?port:HTTP_Port);

          if(socket==-1)
             PrintMessage(Fatal,"Cannot open connection to wwwoffle server %s port %d.",host?host:"localhost",port?port:HTTP_Port);

          server_r=fdopen(socket,"r");
          server_w=fdopen(socket,"w");

          if(!strncmp(argv[i],"http:",5))
            {
             argv[i]+=5;
             if(*argv[i]=='/') argv[i]++;
             if(*argv[i]=='/') argv[i]++;
            }

          printf("Getting: http://%s\n",argv[i]);

          fprintf(server_w,"GET /%s/%s HTTP/1.0\r\n"
                           "Accept: */*\r\n"
                           "\r\n",
                  refresh,argv[i]);
          fflush(server_w);

          while((buffer=fgets_realloc(buffer,server_r)))
             fputs(buffer,stdout);

          fclose(server_w);
          fclose(server_r);
         }
   }
 else /* action==Output */
   {
    for(i=1;i<argc;i++)
       if(argv[i])
         {
          int socket;
          FILE *server_r,*server_w;
          char *buffer=NULL;
          char buffer256[256];
          int nbytes;

          socket=OpenClientSocket(host?host:"localhost",port?port:HTTP_Port);

          if(socket==-1)
             PrintMessage(Fatal,"Cannot open connection to wwwoffle server %s port %d.",host?host:"localhost",port?port:HTTP_Port);

          server_r=fdopen(socket,"r");
          server_w=fdopen(socket,"w");

          if(!strncmp(argv[i],"http:",5))
            {
             argv[i]+=5;
             if(*argv[i]=='/') argv[i]++;
             if(*argv[i]=='/') argv[i]++;
            }

          fprintf(stderr,"Getting: http://%s\n",argv[i]);

          fprintf(server_w,"GET http://%s HTTP/1.0\r\n"
                           "Accept: */*\r\n"
                           "\r\n",
                  argv[i]);
          fflush(server_w);

          buffer=fgets_realloc(buffer,server_r);
          if(buffer)
            {
             char *willget="HTTP/1.0 404 WWWOFFLE Will Get\r\n"; /* This line must not be changed (see messages.c). */
             int status;

             sscanf(buffer,"%*s %d",&status);

             if(!strcmp(willget,buffer))
               {
                fprintf(stderr,"The URL is not in the cache but has been requested.\n");
                break;
               }
             else if(status>=300 && status<400)
               {
                fprintf(stderr,"The URL has been moved, check with a browser.\n");
                break;
               }
             else if(status!=200)
               {
                fprintf(stderr,"The URL returns an error message, check with a browser.\n");
                break;
               }
            }

          while((buffer=fgets_realloc(buffer,server_r)) && buffer[0]!='\r' && buffer[0]!='\n')
             ;

          while((nbytes=fread(buffer256,1,256,server_r)))
             fwrite(buffer256,1,nbytes,stdout);

          fclose(server_w);
          fclose(server_r);
          break;
         }
   }

 /* exit. */

 return(0);
}


/*++++++++++++++++++++++++++++++++++++++
  Print the program usage in long or short format.

  int verbose True for long format.
  ++++++++++++++++++++++++++++++++++++++*/

static void usage(int verbose)
{
 fputs("\n"
       "WWWOFFLE - World Wide Web Offline Explorer - Version 1.2c\n"
       "\n",stderr);

 if(verbose)
    fputs("(c) Andrew M. Bishop 1996,97 [       amb@gedanken.demon.co.uk ]\n"
          "                             [http://www.gedanken.demon.co.uk/]\n"
          "\n",stderr);

 fputs("Usage: wwwoffle -h\n"
       "       wwwoffle -online | -offline | -fetch | -config | -purge\n"
       "       wwwoffle -o <url>\n"
       "       wwwoffle [-i] [-f] [-F] [-(R|r|d)[<depth>]] <url> ...\n"
       "\n"
       "Any of these can also take:  [-p <host>[:<port>] | -c <config-file>]\n"
       "\n",stderr);

 if(verbose)
    fprintf(stderr,
            "wwwoffle -h          : Display this help.\n"
            "\n"
            "wwwoffle -online     : Indicate to the server that the network is active.\n"
            "                       (Proxy requests will be fetched from remote hosts.)\n"
            "\n"
            "wwwoffle -offline    : Indicate to the server that the network is inactive.\n"
            "                       (Proxy requests will be fetched from cache or recorded.)\n"
            "\n"
            "wwwoffle -fetch      : Force the server to fetch the pages that are recorded.\n"
            "\n"
            "wwwoffle -config     : Force the server to re-read the configuration file.\n"
            "\n"
            "wwwoffle -purge      : Force the server to purge pages from the cache.\n"
            "\n"
            "wwwoffle <url> ...   : Fetch the specified URLs (or record for fetching).\n"
            "\n"
            " -o                  : Fetch the url and output it on the standard output.\n"
            "\n"
            " -i                  : Fetch the images included in the specified URLs.\n"
            " -f                  : Fetch the frames included in the specified URLs.\n"
            " -F                  : Force the url to be refreshed even if already cached.\n"
            " -(R|r|d)[<depth>]   : Fetch pages linked to the URLs and their links,\n"
            "                       going no more than <depth> steps (default 1).\n"
            "                        (-R => URLs on any host)\n"
            "                        (-r => URLs on the same host)\n"
            "                        (-d => URLs in the same directory or sub-directory)\n"
            "\n"
            " -p <host>[:<port>]  : The host name and port number to talk to the demon on.\n"
            "                       (Defaults to localhost for the server and\n"
            "                        %d for server port, %d for http proxy port).\n"
            "\n"
            " -c <config-file>    : The name of the configuration file with the hostname,\n"
            "                       port number and the password (if any).\n"
            "\n",DEF_WWWOFFLE_PORT,DEF_HTTP_PORT);

 if(verbose)
    exit(0);
 else
    exit(1);
}
